﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

#include "hid_SixAxisSensorMathUtility.h"
#include "hid_Accelerometer.h"

namespace {

const nn::util::Float3 GravitationalAcceleration = {{{ 0.0f, 0.0f, -1.0f }}}; //!< 重力加速度値[G]

const nn::util::Float3 DefaultZeroCount = {{{ 0.f, 0.f, 0.f }}}; //!< ゼロ点のデフォルト [count]
const nn::util::Float3 DefaultSensitivityCount = {{{ 16384.f, 16384.f, 16384.f }}}; //!< 感度のデフォルト [count]

float CalculateAccelerationLoose(float currentAcc,
                                 float lastAcc,
                                 float playRadius,
                                 float sensitivity)
{
    const float Delta = currentAcc - lastAcc; //----- 目標値への足りない差分

    float  f1 = ( Delta < 0.0f ) ? -Delta : Delta;

    //----- 遊び内外での引き寄せ率計算
    if ( f1 >= playRadius )
    {
        //----- 遊び外であれば追従性を100%適用
        f1 = 1.0f ;
    }
    else
    {
        //----- 遊び内であれば目的に近いほど追従性を弱くする
        f1 /= playRadius ;
        f1 *= f1 ;      // ２乗
        f1 *= f1 ;      // ４乗
    }

    f1 *= sensitivity ;

    //----- 引き寄せ
    return static_cast<float>(lastAcc + f1 * Delta);
}


float CalculateAccelerationTight(float currentAcc,
                                 float lastAcc,
                                 float playRadius,
                                 float sensitivity)
{
    const float Delta = currentAcc - lastAcc; //----- 目標値への足りない差分
    float  calculatedAcceleration = lastAcc;

    // 遊び範囲外の場合は感度に応じた補正を掛ける
    if ( Delta < -playRadius )
    {
        calculatedAcceleration += ( Delta + playRadius ) * sensitivity;
    }
    else if ( Delta > playRadius )
    {
        calculatedAcceleration += ( Delta - playRadius ) * sensitivity;
    }

    return calculatedAcceleration;
}

}

namespace nn { namespace hid { namespace detail {

Accelerometer::Accelerometer() NN_NOEXCEPT
    : m_ZeroCount(DefaultZeroCount)
    , m_SensitivityCount(DefaultSensitivityCount)
    , m_Scale()
    , m_LastAcceleration(GravitationalAcceleration)
    , m_pAccelerometerSetting(nullptr)
{
    // 何もしない
}

void Accelerometer::SetAccelerometerSetting(const AccelerometerSetting* const pSetting) NN_NOEXCEPT
{
    m_pAccelerometerSetting = pSetting;
}

void Accelerometer::GetState(nn::util::Float3* pOutState, const nn::util::Float3& accelerationCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutState);

    nn::util::Float3 calculatedAcceleration;

    ConvertToAcceleration(&calculatedAcceleration, accelerationCount);
    ReviseAccelerationByPlayMode(pOutState, calculatedAcceleration);
    UpdateLastAcceleration(*pOutState);
}

void Accelerometer::SetScaleValue() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAccelerometerSetting);

    const float Delta = 1.0f - 0.0f;   // 1Gスケール - ゼロ点オフセット

    m_Scale.x = Delta / m_SensitivityCount.x;
    m_Scale.y = Delta / m_SensitivityCount.y;
    m_Scale.z = Delta / m_SensitivityCount.z;
}

void Accelerometer::ConvertToAcceleration( nn::util::Float3* pOut, const nn::util::Float3& count ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);

    pOut->x = ( count.x - m_ZeroCount.x ) * m_Scale.x;
    pOut->y = ( count.y - m_ZeroCount.y ) * m_Scale.y;
    pOut->z = ( count.z - m_ZeroCount.z ) * m_Scale.z;
}

void Accelerometer::ReviseAccelerationByPlayMode( nn::util::Float3* pOut, const nn::util::Float3& acceleration ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);

    pOut->x = CalcAcceleration( acceleration.x, m_LastAcceleration.x );
    pOut->y = CalcAcceleration( acceleration.y, m_LastAcceleration.y );
    pOut->z = CalcAcceleration( acceleration.z, m_LastAcceleration.z );
}

void Accelerometer::UpdateLastAcceleration( const nn::util::Float3& acceleration ) NN_NOEXCEPT
{
    m_LastAcceleration = acceleration;
}

float Accelerometer::CalcAcceleration(const float& currentAcc, const float& lastAcc ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAccelerometerSetting);

    switch(m_pAccelerometerSetting->playMode)
    {
    case AccelerometerPlayMode::AccelerometerPlayMode_Loose:
        return CalculateAccelerationLoose(currentAcc,
                                          lastAcc,
                                          m_pAccelerometerSetting->playRadius,
                                          m_pAccelerometerSetting->sensitivity);
    case AccelerometerPlayMode::AccelerometerPlayMode_Tight:
        return CalculateAccelerationTight(currentAcc,
                                          lastAcc,
                                          m_pAccelerometerSetting->playRadius,
                                          m_pAccelerometerSetting->sensitivity);
    default: NN_UNEXPECTED_DEFAULT;
    }
}

float Accelerometer::UpdateDirection( nn::hid::DirectionState* pOutDirection, const nn::util::Float3& acceleration ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutDirection);
    NN_SDK_REQUIRES_NOT_NULL(m_pAccelerometerSetting);

    if (!m_pAccelerometerSetting->isReviseEnabled)
    {
        return (0.0f);
    }

    // __VPADReviseDirAcc を参照
    nn::util::Float3          v1;
    nn::util::Float3          v2;
    nn::util::Float3          vec;
    nn::hid::DirectionState   d1;
    nn::hid::DirectionState   d2;
    float                     f1;
    float                     level;

    f1 = acceleration.x * acceleration.x + acceleration.y * acceleration.y + acceleration.z * acceleration.z ;
    if ( f1 == 0.0f )
    {
        return ( 0.0f ) ;
    }

    f1 = Sqrt( f1 ) ;
    if ( f1 < 1.0f )
    {
        if ( f1 <= (1.0f - m_pAccelerometerSetting->reviseRange) )
        {
            return ( 0.0f ) ;
        }
        level = ( f1 - (1.0f - m_pAccelerometerSetting->reviseRange) )
            * (1.0f / m_pAccelerometerSetting->reviseRange) ;
    }
    else
    {
        if ( f1 >= (1.0f + m_pAccelerometerSetting->reviseRange) )
        {
            return ( 0.0f ) ;
        }
        level = ( f1 - (1.0f + m_pAccelerometerSetting->reviseRange) )
            * (-1.0f / m_pAccelerometerSetting->reviseRange) ;
    }
    level *= level ;
    level *= m_pAccelerometerSetting->revisePower ;
    //level = 1.0f; // 重力成分により完全に補正する

    f1 = 1.0f / f1 ;
    vec.x = f1 * acceleration.x ;
    vec.y = f1 * acceleration.y ;
    vec.z = f1 * acceleration.z ;

    v1.x = vec.x * pOutDirection->x.x + vec.y * pOutDirection->y.x + vec.z * pOutDirection->z.x ;
    v1.y = vec.x * pOutDirection->x.y + vec.y * pOutDirection->y.y + vec.z * pOutDirection->z.y ;
    v1.z = vec.x * pOutDirection->x.z + vec.y * pOutDirection->y.z + vec.z * pOutDirection->z.z ;

    v2.x = ( GravitationalAcceleration.x - v1.x ) * level + v1.x ;
    v2.y = ( GravitationalAcceleration.y - v1.y ) * level + v1.y ;
    v2.z = ( GravitationalAcceleration.z - v1.z ) * level + v1.z ;

    if ( nn::hid::detail::Normalize( &v2 ) == 0.0f ) {
        return ( 0.0f ) ;
    }

    nn::hid::detail::MakeVecDir( &d1, v1, v2 );                        // v1 が v2 に向く行列 d1 を作る
    nn::hid::detail::MultiplyDirection( &d2, *pOutDirection, d1 ) ;    // m_Direction を d1 で変換
    *pOutDirection = d2 ;

    nn::hid::detail::NormalizeDirectionState( pOutDirection, 2.999f ) ;

    //----- 補正の強さ計算
    v1.x -= v2.x ;
    v1.y -= v2.y ;
    v1.z -= v2.z ;
    f1 = Sqrt( v1.x * v1.x + v1.y * v1.y + v1.z * v1.z ) ;

    return ( f1 ) ;
}

void Accelerometer::SetCalibrationValue( const nn::xcd::SensorState& accelerometerOrigin,
                                         const nn::xcd::SensorState& accelerometerSensitivity ) NN_NOEXCEPT
{
    m_ZeroCount.x = static_cast<float>(accelerometerOrigin.x);
    m_ZeroCount.y = static_cast<float>(accelerometerOrigin.y);
    m_ZeroCount.z = static_cast<float>(accelerometerOrigin.z);

    if (accelerometerSensitivity.x == 0 ||
        accelerometerSensitivity.y == 0 ||
        accelerometerSensitivity.z == 0)
    {
        m_SensitivityCount = DefaultSensitivityCount;
    }
    else
    {
        m_SensitivityCount.x = static_cast<float>(accelerometerSensitivity.x);
        m_SensitivityCount.y = static_cast<float>(accelerometerSensitivity.y);
        m_SensitivityCount.z = static_cast<float>(accelerometerSensitivity.z);
    }

    ShiftCalibrationValue(m_pAccelerometerSetting->shiftOrigin,
                          m_pAccelerometerSetting->shiftSensitivity);

    SetScaleValue();
}

void Accelerometer::ShiftCalibrationValue(const float& origin,
                                          const float& sensitivity) NN_NOEXCEPT
{
    m_SensitivityCount.x *= sensitivity;
    m_SensitivityCount.y *= sensitivity;
    m_SensitivityCount.z *= sensitivity;

    const float Delta = 1.0f - 0.0f;   // 1Gスケール - ゼロ点オフセット

    m_ZeroCount.x -= origin * static_cast<float>(m_SensitivityCount.x / Delta);
    m_ZeroCount.y -= origin * static_cast<float>(m_SensitivityCount.y / Delta);
    m_ZeroCount.z -= origin * static_cast<float>(m_SensitivityCount.z / Delta);

}

}}} // namespace nn::hid::detail
