﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/xcd/xcd.h>

#include <nn/nn_SdkLog.h>

#include "hid_AbstractedPalmaPadManager.h"

namespace nn { namespace hid { namespace detail {

AbstractedPalmaPadManager::AbstractedPalmaPadManager() NN_NOEXCEPT
    : m_ActivationCount()
{
}

AbstractedPalmaPadManager::~AbstractedPalmaPadManager() NN_NOEXCEPT
{
    // 何もしない
}

void AbstractedPalmaPadManager::SetAbstractedPadIdPublisher(AbstractedPadIdPublisher* pPublisher) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPublisher);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_pIdPublisher = pPublisher;
}

void AbstractedPalmaPadManager::SetAbstractedPalmaPads(AbstractedPalmaPad* pPads, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPads);
    NN_SDK_REQUIRES_GREATER(count, 0);
    NN_SDK_REQUIRES_LESS_EQUAL(count, AbstractedPalmaPadCountMax);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    for (int padIndex = 0; padIndex < count; padIndex++)
    {
        m_pPads[padIndex] = &pPads[padIndex];
    }
    m_PadCount = count;
}

::nn::Result AbstractedPalmaPadManager::Activate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pPads);
    NN_SDK_REQUIRES_GREATER(m_PadCount, 0);
    NN_SDK_REQUIRES_NOT_NULL(m_pIdPublisher);

    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(),
                           ResultGamePadDriverActivationUpperLimitOver());

    if (m_ActivationCount.IsZero())
    {
        // 何もしない
    }

    // アクティブ化した回数をインクリメント
    ++m_ActivationCount;

    NN_RESULT_SUCCESS;

}

::nn::Result AbstractedPalmaPadManager::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultGamePadDriverDeactivationLowerLimitOver());

    // アクティブ化した回数をデクリメント
    --m_ActivationCount;

    if(m_ActivationCount.IsZero())
    {
        // 何もしない
    }

    NN_RESULT_SUCCESS;
}

void AbstractedPalmaPadManager::UpdateDevices() NN_NOEXCEPT
{
    if (m_ActivationCount.IsZero())
    {
        return;
    }

    ::nn::xcd::BleDeviceList deviceList;

    if(::nn::xcd::ListDevices(&deviceList).IsFailure())
    {
        deviceList.deviceCount = 0;
    }

    // 切断されたデバイスを探索
    for (int i = 0; i < m_PadCount; i++)
    {
        CheckForDeviceDetach(m_pPads[i], &deviceList);
    }

    // 接続済みのデバイスを処理
    for(int i = 0; i < deviceList.deviceCount; i++)
    {
        // 新規で接続されたデバイスか確認
        if (IsConnectedXcdDevice(deviceList.handleList[i]) == false)
        {
            ::nn::xcd::BleDeviceInfo info;
            if (::nn::xcd::GetDeviceInfo(&info, deviceList.handleList[i]).IsFailure())
            {
                // デバイス情報が正しく読み出せないので切断
                nn::xcd::Detach(deviceList.handleList[i]);
                continue;
            }

            if ( info.deviceType != ::nn::xcd::BleDeviceType_Palma
               )
            {
                // Palma 以外は現時点では非サポート
                nn::xcd::Detach(deviceList.handleList[i]);
                continue;
            }

            AddXcdDevice(deviceList.handleList[i], info);
        }
    }
}

bool AbstractedPalmaPadManager::IsConnectedXcdDevice(::nn::xcd::BleConnectionHandle handle) NN_NOEXCEPT
{
    // NpadManager を探索して接続済みかどうかをチェック
    for (int padIndex = 0; padIndex < m_PadCount; padIndex++)
    {
        if (m_pPads[padIndex]->GetXcdDeviceHandle() == handle)
        {
            // 接続済みのデバイス
            return true;
        }
    }

    // 未接続のデバイス
    return false;
}

void AbstractedPalmaPadManager::CheckForDeviceDetach(AbstractedPalmaPad* pPad, ::nn::xcd::BleDeviceList* pList) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pList);

    // AbstractedPad に対してもともとデバイスが何も割り当てされていない場合は何もしない
    if (pPad->IsAttached() == false)
    {
        return;
    }

    // まだデバイスが接続されているかどうか探索
    for (int i = 0; i < pList->deviceCount; i++)
    {
        if (pPad->GetXcdDeviceHandle() == pList->handleList[i])
        {
            // 接続中なので切断しない
            return;
        }
    }
    // ここまで到達した場合は切断
    pPad->RemoveDevice();
}

void AbstractedPalmaPadManager::AddXcdDevice(::nn::xcd::BleConnectionHandle handle, ::nn::xcd::BleDeviceInfo deviceInfo) NN_NOEXCEPT
{
    NN_UNUSED(deviceInfo);

    for (int padIndex = 0; padIndex < m_PadCount; padIndex++)
    {
        if (m_pPads[padIndex]->IsAttached() == false)
        {
            // あいている Pad に追加 & 新規に Id 発行
            m_pPads[padIndex]->AttachDevice(handle, m_pIdPublisher->PublishId());
            return;
        }
    }

    // 割り当てできる Pad が見つからなかった場合は Xcd から Detach
    ::nn::xcd::Detach(handle);
}

}}} // namespace nn::hid::detail
