﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/xcd/xcd.h>

#include "hid_AbstractedPadXcd.h"
#include "hid_CommonStateUtility.h"

namespace nn { namespace hid { namespace detail {

namespace {

// DeviceType と InterfaceType から FeatureSet を取得する
void GetFeatureSetFromDeviceType(AbstractedPadFeatureSet* pOutFeatureSet, ::nn::xcd::DeviceType deviceType, ::nn::xcd::InterfaceType interfaceType) NN_NOEXCEPT;

} // namespace

AbstractedPadXcd::AbstractedPadXcd() NN_NOEXCEPT
    : m_IsConnected(false)
{
    // 何もしない
}

AbstractedPadXcd::~AbstractedPadXcd() NN_NOEXCEPT
{
    // 何もしない
}

AbstractedPadType AbstractedPadXcd::GetType() NN_NOEXCEPT
{
    return s_Type;
}

bool AbstractedPadXcd::IsConnected() NN_NOEXCEPT
{
    return m_IsConnected;
}

system::PowerInfo AbstractedPadXcd::GetPowerInfo() NN_NOEXCEPT
{
    return m_PowerInfo;
}

AbstractedPadState AbstractedPadXcd::GetPadState() NN_NOEXCEPT
{
    return m_PadState;
}

bool AbstractedPadXcd::GetButtonTriggerElapsedTime(nn::os::Tick* pOutTick, AbstractedPadButtonSet button) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(button.CountPopulation(), 1);
    if (m_IsConnected == false)
    {
        return false;
    }

    // TODO : 変換用のユーティリティを別途設ける
    if (button.Test<AbstractedPadButton::SL>())
    {
        return nn::xcd::GetButtonTriggerElapsedTime(pOutTick, ::nn::xcd::PadButton::SL::Index, m_XcdHandle).IsSuccess();
    }
    if (button.Test<AbstractedPadButton::SR>())
    {
        return nn::xcd::GetButtonTriggerElapsedTime(pOutTick, ::nn::xcd::PadButton::SR::Index, m_XcdHandle).IsSuccess();
    }

    return false;
}

system::ControllerColor AbstractedPadXcd::GetColor() NN_NOEXCEPT
{
    if (m_IsConnected == false)
    {
        return system::ControllerColor();
    }
    return m_Color;
}

bool AbstractedPadXcd::GetGripColor(nn::util::Color4u8Type* pOutLeftGrip, nn::util::Color4u8Type* pOutRightGrip) NN_NOEXCEPT
{
    if (m_DeviceType.Test<system::DeviceType::SwitchProController>() == true)
    {
        *pOutLeftGrip = m_LeftGripColor;
        *pOutRightGrip = m_RightGripColor;
        return true;
    }
    return false;
}

int AbstractedPadXcd::GetSensorStates(nn::xcd::SixAxisSensorState* pOutValue, int count) NN_NOEXCEPT
{
    int gotCount = 0;
    if (nn::xcd::GetSensorStates(&gotCount, pOutValue, count, m_XcdHandle).IsFailure())
    {
        return 0;
    }
    return gotCount;
}

bool AbstractedPadXcd::GetSensorCalibrationValue(nn::xcd::SensorCalibrationValue* pOutValue) NN_NOEXCEPT
{
    if (m_IsConnected == false)
    {
        return false;
    }
    *pOutValue = m_SensorCalibrationValue;
    return true;
}

nn::TimeSpan AbstractedPadXcd::GetSensorSamplingInterval() NN_NOEXCEPT
{
    return nn::TimeSpan::FromMilliSeconds(5);
}

void AbstractedPadXcd::Connect() NN_NOEXCEPT
{
    if (m_IsAttached == false)
    {
        return;
    }

    // 接続状態に変更
    if (m_IsConnected == false)
    {
        m_IsConnected = true;
    }
}

void AbstractedPadXcd::Detach() NN_NOEXCEPT
{
    if (m_IsAttached)
    {
        // Usb の場合は、内部の接続状態のみ変更
        if (m_InterfaceType == system::InterfaceType_Usb)
        {
            // Led を消す
            this->SetIndicator(0, 0);
        }

        m_IsConnected = false;
        m_IsOnDelayedDetach = false;

        // Bluetooth の場合は Detach コマンドを送信
        if (m_InterfaceType == system::InterfaceType_Bluetooth)
        {
            ::nn::xcd::Detach(m_XcdHandle);
        }

        ResetBatteryLowInternalState();
    }
}

void AbstractedPadXcd::Reboot(bool reconnect) NN_NOEXCEPT
{
    if (m_IsConnected)
    {
        ::nn::xcd::Reboot(m_XcdHandle, reconnect, true);
    }
}

void AbstractedPadXcd::SetIndicator(uint8_t pattern, bool blink) NN_NOEXCEPT
{
    if (m_IsConnected)
    {
        if (blink == true)
        {
            pattern = (pattern << 4) & 0xf0;
        }
        nn::xcd::SetPlayerIndicatorPattern(pattern, m_XcdHandle);
    }
}

uint8_t AbstractedPadXcd::GetIndicator() NN_NOEXCEPT
{
    if (m_IsConnected)
    {
        nn::Bit8 pattern;
        if (nn::xcd::GetPlayerIndicatorPattern(&pattern, m_XcdHandle).IsSuccess())
        {
            // 点滅パターンの場合はシフトする
            if (pattern > 0x0f)
            {
                pattern = (pattern >> 4) & 0x0f;
            }
            return static_cast<uint8_t>(pattern);
        }
    }

    return 0;
}

void AbstractedPadXcd::VibrationOnConnect() NN_NOEXCEPT
{
    if (m_IsConnected)
    {
        nn::xcd::StartVibrationOnConnect(m_XcdHandle);
    }
}

bool AbstractedPadXcd::IsWired() NN_NOEXCEPT
{
    if (m_IsConnected == false)
    {
        return false;
    }
    return m_IsWired;
}


void AbstractedPadXcd::ResetInternalDeviceState() NN_NOEXCEPT
{
    if (m_DeviceType.Test<system::DeviceType::FullKeyController>())
    {
        nn::xcd::McuResume(m_XcdHandle, true);
    }
    if (m_DeviceType.Test<system::DeviceType::HandheldJoyRight>() ||
        m_DeviceType.Test<system::DeviceType::JoyConRight>())
    {
        nn::xcd::McuResume(m_XcdHandle, false);
    }
}

bool AbstractedPadXcd::IsUsbConnected() NN_NOEXCEPT
{
    bool outIsConnceted = false;
    if (m_IsConnected == false)
    {
        return false;
    }
    nn::xcd::GetUsbConnected(&outIsConnceted, m_XcdHandle);
    return outIsConnceted;
}

bool AbstractedPadXcd::HasBattery() NN_NOEXCEPT
{
    return m_HasBattery;
}

bool AbstractedPadXcd::SetDeviceInfoOnPlayReportControllerUsage(system::PlayReportControllerUsage* pOutValue) NN_NOEXCEPT
{
    // デバイスタイプの変換
    if (m_DeviceType.Test<system::DeviceType::HandheldJoyLeft>() ||
        m_DeviceType.Test<system::DeviceType::JoyConLeft>())
    {
        pOutValue->deviceType = system::PlayReportDeviceType_JoyConLeft;
    }
    else if (m_DeviceType.Test<system::DeviceType::HandheldJoyRight>() ||
        m_DeviceType.Test<system::DeviceType::JoyConRight>())
    {
        pOutValue->deviceType = system::PlayReportDeviceType_JoyConRight;
    }
    else if (m_DeviceType.Test<system::DeviceType::FullKeyController>())
    {
        pOutValue->deviceType = system::PlayReportDeviceType_SwitchProController;
    }
    else
    {
        pOutValue->deviceType = system::PlayReportDeviceType_Unknown;
    }
    pOutValue->nxControllerInfo.interfaceType = m_InterfaceType;
    pOutValue->nxControllerInfo.subType = 0;  // サブタイプは現状設定しない

    nn::xcd::BtFirmwareVersion btVersion;
    nn::xcd::GetBtFirmwareVersion(&btVersion, m_XcdHandle);
    pOutValue->nxControllerInfo.version[0] = btVersion.major;
    pOutValue->nxControllerInfo.version[1] = btVersion.minor;

    if (m_DeviceType.Test<system::DeviceType::HandheldJoyRight>() ||
        m_DeviceType.Test<system::DeviceType::JoyConRight>() ||
        m_DeviceType.Test<system::DeviceType::SwitchProController>())
    {
        nn::xcd::McuVersionData mcuVersion;
        NN_RESULT_TRY(nn::xcd::GetMcuVersion(&mcuVersion, m_XcdHandle))
            NN_RESULT_CATCH(nn::xcd::ResultMcuVersionNotAvailable)
        {
            ::nn::xcd::RequestMcuVersion(m_XcdHandle);
            return false;
        }
        NN_RESULT_CATCH_ALL
        {
            // Tarragon のように Switch Pro Controller なのに Mcu がない場合はここ
            pOutValue->nxControllerInfo.version[2] = 0x00;
            pOutValue->nxControllerInfo.version[3] = 0x00;
        }
        NN_RESULT_END_TRY;

        if (mcuVersion.isCorrupted || mcuVersion.isIapCorrupted)
        {
            // 壊れの場合は 0x00
            pOutValue->nxControllerInfo.version[2] = 0x00;
            pOutValue->nxControllerInfo.version[3] = 0x00;
        }
        else
        {
            pOutValue->nxControllerInfo.version[2] = static_cast<uint8_t>(mcuVersion.major);
            pOutValue->nxControllerInfo.version[3] = static_cast<uint8_t>(mcuVersion.minor);
        }
    }
    else
    {
        pOutValue->nxControllerInfo.version[2] = 0x00;
        pOutValue->nxControllerInfo.version[3] = 0x00;
    }
    return true;
}

void AbstractedPadXcd::GetPadDriverState(PadDriverState* pOutState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutState);

    nn::xcd::PadState padState;
    if (nn::xcd::GetPadState(&padState, m_XcdHandle).IsFailure())
    {
        // エラーの場合は無入力を返す
        pOutState->buttons.Reset();
        pOutState->analogStickL = AnalogStickDriverState();
        pOutState->analogStickR = AnalogStickDriverState();
        return;
    }

    pOutState->sampleNumber = padState.sampleNumber;

    //!< Button の計算
    pOutState->buttons.Reset();
    pOutState->buttons.Set<PadButton::A>(padState.buttons.Test<nn::xcd::PadButton::A>());
    pOutState->buttons.Set<PadButton::B>(padState.buttons.Test<nn::xcd::PadButton::B>());
    pOutState->buttons.Set<PadButton::X>(padState.buttons.Test<nn::xcd::PadButton::X>());
    pOutState->buttons.Set<PadButton::Y>(padState.buttons.Test<nn::xcd::PadButton::Y>());
    pOutState->buttons.Set<PadButton::StickL>(padState.buttons.Test<nn::xcd::PadButton::StickL>());
    pOutState->buttons.Set<PadButton::StickR>(padState.buttons.Test<nn::xcd::PadButton::StickR>());
    pOutState->buttons.Set<PadButton::L>(padState.buttons.Test<nn::xcd::PadButton::L>());
    pOutState->buttons.Set<PadButton::R>(padState.buttons.Test<nn::xcd::PadButton::R>());
    pOutState->buttons.Set<PadButton::ZL>(padState.buttons.Test<nn::xcd::PadButton::ZL>());
    pOutState->buttons.Set<PadButton::ZR>(padState.buttons.Test<nn::xcd::PadButton::ZR>());
    pOutState->buttons.Set<PadButton::Start>(padState.buttons.Test<nn::xcd::PadButton::Start>());
    pOutState->buttons.Set<PadButton::Select>(padState.buttons.Test<nn::xcd::PadButton::Select>());
    pOutState->buttons.Set<PadButton::Left>(padState.buttons.Test<nn::xcd::PadButton::Left>());
    pOutState->buttons.Set<PadButton::Up>(padState.buttons.Test<nn::xcd::PadButton::Up>());
    pOutState->buttons.Set<PadButton::Right>(padState.buttons.Test<nn::xcd::PadButton::Right>());
    pOutState->buttons.Set<PadButton::Down>(padState.buttons.Test<nn::xcd::PadButton::Down>());
    pOutState->buttons.Set<PadButton::SL>(padState.buttons.Test<nn::xcd::PadButton::SL>());
    pOutState->buttons.Set<PadButton::SR>(padState.buttons.Test<nn::xcd::PadButton::SR>());
    pOutState->buttons.Set<PadButton::Home>(padState.buttons.Test<nn::xcd::PadButton::Home>());
    pOutState->buttons.Set<PadButton::Shot>(padState.buttons.Test<nn::xcd::PadButton::Shot>());

    //!< Analog Stick の計算
    pOutState->analogStickR.x = padState.analogStickR.x;
    pOutState->analogStickR.y = padState.analogStickR.y;
    pOutState->analogStickL.x = padState.analogStickL.x;
    pOutState->analogStickL.y = padState.analogStickL.y;
}

int AbstractedPadXcd::GetSixAxisSensorDriverStates(SixAxisSensorDriverState* pOutStates,
                                                   int count) NN_NOEXCEPT
{
    int gotCount = 0;
    nn::xcd::SixAxisSensorState states[nn::xcd::SixAxisSensorSampleCountMax];
    if (nn::xcd::GetSensorStates(&gotCount, states, NN_ARRAY_SIZE(states), m_XcdHandle).IsFailure())
    {
        return 0;
    }
    gotCount = std::min(gotCount, count);
    for (int i = 0; i < gotCount; ++i)
    {
        pOutStates[i].accelerometer.x = static_cast<int32_t>(states[i].accelerometer.x);
        pOutStates[i].accelerometer.y = static_cast<int32_t>(states[i].accelerometer.y);
        pOutStates[i].accelerometer.z = static_cast<int32_t>(states[i].accelerometer.z);

        pOutStates[i].gyroscope.x = static_cast<int32_t>(states[i].gyroscope.x);
        pOutStates[i].gyroscope.y = static_cast<int32_t>(states[i].gyroscope.y);
        pOutStates[i].gyroscope.z = static_cast<int32_t>(states[i].gyroscope.z);

        pOutStates[i].sampleNumber = states[i].sampleNumber;
    }
    return gotCount;
}

RxPacketHistory AbstractedPadXcd::GetRxPacketHistory() NN_NOEXCEPT
{
    if (m_IsConnected == false)
    {
        return RxPacketHistory();
    }
    return nn::xcd::GetRxPacketHistory(m_XcdHandle);
}

void AbstractedPadXcd::AttachDevice(::nn::xcd::DeviceHandle handle, AbstractedPadId id) NN_NOEXCEPT
{
    nn::xcd::DeviceInfo deviceInfo;
    nn::xcd::DeviceStatus deviceStatus;
    nn::xcd::DeviceColor color;

    if (nn::xcd::GetDeviceInfo(&deviceInfo, handle).IsFailure() ||
        nn::xcd::GetDeviceStatus(&deviceStatus, handle).IsFailure() ||
        nn::xcd::GetDeviceColor(&color, handle).IsFailure() ||
        nn::xcd::GetSensorCalibrationValue(&m_SensorCalibrationValue, handle).IsFailure())
    {
        // デバイスが切断状態にあるため追加を行わない
        return;
    }
    m_XcdHandle = handle;
    m_Id = id;
    m_IsOnDelayedDetach = false;
    m_LastSampleNumber = std::numeric_limits<int64_t>::min();
    m_IsNewSampleReceived = false;

    m_Color.main = color.color[0];
    m_Color.sub = color.color[1];
    m_LeftGripColor = color.color[2];
    m_RightGripColor = color.color[3];

    // インタフェースの設定
    switch (deviceStatus.interfaceType)
    {
    case nn::xcd::InterfaceType_Bluetooth:
        m_InterfaceType = system::InterfaceType_Bluetooth;
        break;
    case nn::xcd::InterfaceType_Uart:
        m_InterfaceType = system::InterfaceType_Rail;
        break;
    case nn::xcd::InterfaceType_Usb:
        m_InterfaceType = system::InterfaceType_Usb;
        break;
    default:
        m_InterfaceType = system::InterfaceType_Unknown;
        break;
    }

    // デバイスタイプの取得
    switch (deviceInfo.deviceType)
    {
    case nn::xcd::DeviceType_FullKey:
        m_DeviceType = system::DeviceType::SwitchProController::Mask;
        m_HasBattery = true;
        break;
    case nn::xcd::DeviceType_Left:
        m_DeviceType = (m_InterfaceType == system::InterfaceType_Rail) ?
            system::DeviceType::HandheldJoyLeft::Mask :
            system::DeviceType::JoyConLeft::Mask;
        m_HasBattery = true;
        break;
    case nn::xcd::DeviceType_Right:
        m_DeviceType = (m_InterfaceType == system::InterfaceType_Rail) ?
            system::DeviceType::HandheldJoyRight::Mask :
            system::DeviceType::JoyConRight::Mask;
        m_HasBattery = true;
        break;
    case nn::xcd::DeviceType_MiyabiLeft:
        m_DeviceType =system::DeviceType::JoyConLeft::Mask;
        m_HasBattery = false;
        break;
    case nn::xcd::DeviceType_MiyabiRight:
        m_DeviceType = system::DeviceType::JoyConRight::Mask;
        m_HasBattery = false;
        break;
    case nn::xcd::DeviceType_Tarragon:
        m_DeviceType = system::DeviceType::SwitchProController::Mask;
        m_HasBattery = false;
        break;
    default:
        m_DeviceType = system::DeviceType::Unknown::Mask;
        m_HasBattery = false;
    }

    m_Address = deviceInfo.address;

    // Attach されたときにセンサーは常時 ON
    nn::xcd::SleepSensor(false, m_XcdHandle);

    GetFeatureSetFromDeviceType(&m_FeatureSet, deviceInfo.deviceType, deviceStatus.interfaceType);

    m_IsAttached = true;

    m_ButtonMask.Reset();

#ifdef NN_BUILD_CONFIG_OS_HORIZON
    // Horizon 環境では Usb 以外はデバイス割り当て時に自動的に接続状態にする
    if (m_InterfaceType != system::InterfaceType_Usb)
    {
        m_IsConnected = true;
    }
#endif
}

void AbstractedPadXcd::RemoveDevice() NN_NOEXCEPT
{
    m_XcdHandle = nn::xcd::DeviceHandle::GetInvalidHandle();
    m_Address = nn::bluetooth::Address();
    m_Id = AbstractedPadId();
    m_IsWired = false;
    m_IsConnected = false;
    m_IsAttached = false;
    ResetBatteryLowInternalState();
}

::nn::xcd::DeviceHandle AbstractedPadXcd::GetXcdDeviceHandle() NN_NOEXCEPT
{
    return m_XcdHandle;
}

void AbstractedPadXcd::Update() NN_NOEXCEPT
{
    if (m_IsAttached == false)
    {
        return;
    }
    // ボタンのトリガー検出用
    auto oldButtons = m_PadState.buttons;

    // 入力状態の更新
    UpdatePadState();

    // 無効化
    m_PadState.buttons &= m_ButtonMask;

    // 何かボタンが押されたら接続状態へと変更する
    if (m_IsConnected == false &&
        ((oldButtons ^ m_PadState.buttons) & m_PadState.buttons).IsAnyOn())
    {
        m_IsConnected = true;
        // 入力無効化するボタンを追加
        m_ButtonMask &= ~m_PadState.buttons;
    }

    // PowerInfo の更新
    UpdatePowerInfo();

    // 電池残量低下通知のチェック
    this->UpdateBatteryLowNotify();
}

::nn::bluetooth::Address AbstractedPadXcd::GetAddress() NN_NOEXCEPT
{
    return m_Address;
}

void AbstractedPadXcd::SetDelayedDetachTick(const ::nn::os::Tick& tick) NN_NOEXCEPT
{
    m_DelayedDetachTick = tick;
    m_IsOnDelayedDetach = true;
}

bool AbstractedPadXcd::IsOnDelayedDetach() NN_NOEXCEPT
{
    return m_IsOnDelayedDetach;
}

::nn::os::Tick AbstractedPadXcd::GetDelayedDetachTick() NN_NOEXCEPT
{
    return m_DelayedDetachTick;
}

void AbstractedPadXcd::UpdatePadState() NN_NOEXCEPT
{
    nn::xcd::PadState padState;
    if (nn::xcd::GetPadState(&padState, m_XcdHandle).IsFailure())
    {
        // 入力状態を更新しない
        return;
    }

    // 新しいサンプルを受信したときのみ更新
    if (m_LastSampleNumber == padState.sampleNumber)
    {
        return;
    }
    m_LastSampleNumber = padState.sampleNumber;
    m_IsNewSampleReceived = true;

    //!< Button の計算
    m_PadState.buttons.Reset();
    m_PadState.buttons.Set<AbstractedPadButton::A>(padState.buttons.Test<nn::xcd::PadButton::A>());
    m_PadState.buttons.Set<AbstractedPadButton::B>(padState.buttons.Test<nn::xcd::PadButton::B>());
    m_PadState.buttons.Set<AbstractedPadButton::X>(padState.buttons.Test<nn::xcd::PadButton::X>());
    m_PadState.buttons.Set<AbstractedPadButton::Y>(padState.buttons.Test<nn::xcd::PadButton::Y>());
    m_PadState.buttons.Set<AbstractedPadButton::StickL>(padState.buttons.Test<nn::xcd::PadButton::StickL>());
    m_PadState.buttons.Set<AbstractedPadButton::StickR>(padState.buttons.Test<nn::xcd::PadButton::StickR>());
    m_PadState.buttons.Set<AbstractedPadButton::L>(padState.buttons.Test<nn::xcd::PadButton::L>());
    m_PadState.buttons.Set<AbstractedPadButton::R>(padState.buttons.Test<nn::xcd::PadButton::R>());
    m_PadState.buttons.Set<AbstractedPadButton::ZL>(padState.buttons.Test<nn::xcd::PadButton::ZL>());
    m_PadState.buttons.Set<AbstractedPadButton::ZR>(padState.buttons.Test<nn::xcd::PadButton::ZR>());
    m_PadState.buttons.Set<AbstractedPadButton::Start>(padState.buttons.Test<nn::xcd::PadButton::Start>());
    m_PadState.buttons.Set<AbstractedPadButton::Select>(padState.buttons.Test<nn::xcd::PadButton::Select>());
    m_PadState.buttons.Set<AbstractedPadButton::Left>(padState.buttons.Test<nn::xcd::PadButton::Left>());
    m_PadState.buttons.Set<AbstractedPadButton::Up>(padState.buttons.Test<nn::xcd::PadButton::Up>());
    m_PadState.buttons.Set<AbstractedPadButton::Right>(padState.buttons.Test<nn::xcd::PadButton::Right>());
    m_PadState.buttons.Set<AbstractedPadButton::Down>(padState.buttons.Test<nn::xcd::PadButton::Down>());
    m_PadState.buttons.Set<AbstractedPadButton::SL>(padState.buttons.Test<nn::xcd::PadButton::SL>());
    m_PadState.buttons.Set<AbstractedPadButton::SR>(padState.buttons.Test<nn::xcd::PadButton::SR>());
    m_PadState.buttons.Set<AbstractedPadButton::Home>(padState.buttons.Test<nn::xcd::PadButton::Home>());
    m_PadState.buttons.Set<AbstractedPadButton::Shot>(padState.buttons.Test<nn::xcd::PadButton::Shot>());

    //!< Analog Stick の計算
    ::nn::xcd::AnalogStickValidRange range;
    ::nn::xcd::GetRightAnalogStickValidRange(&range, m_XcdHandle);
    m_PadState.analogStickR = ConvertXcdAnalogStickState(padState.analogStickR, range);

    ::nn::xcd::GetLeftAnalogStickValidRange(&range, m_XcdHandle);
    m_PadState.analogStickL = ConvertXcdAnalogStickState(padState.analogStickL, range);

    SetCrossStickEmulationButtonsOnAbstractedPad(&m_PadState);

    // 離されたボタンは無効化を解除
    m_ButtonMask |= ~m_PadState.buttons;
}

void AbstractedPadXcd::UpdatePowerInfo() NN_NOEXCEPT
{
    nn::xcd::DeviceStatus status;
    if (nn::xcd::GetDeviceStatus(&status, m_XcdHandle).IsFailure())
    {
        // 電源状態を更新しない
        return;
    }

    m_PowerInfo.batteryLevel = static_cast<system::BatteryLevel>(status.batteryLevel);
    m_PowerInfo.isCharging = status.charged;
    m_PowerInfo.isPowered = status.powered;

    // ケーブル接続状態もあわせて更新
    m_IsWired = status.cablePlugged;
}

namespace
{

void GetFeatureSetFromDeviceType(AbstractedPadFeatureSet* pOutFeatureSet, ::nn::xcd::DeviceType deviceType, ::nn::xcd::InterfaceType interfaceType) NN_NOEXCEPT
{
    pOutFeatureSet->Reset();
    switch (deviceType)
    {
    case nn::xcd::DeviceType_FullKey:
        pOutFeatureSet->Set<AbstractedPadFeature::AnalogStick>();
        pOutFeatureSet->Set<AbstractedPadFeature::SixAxisSensorCenter>();
        pOutFeatureSet->Set<AbstractedPadFeature::AnalogStickUserCal>();
        pOutFeatureSet->Set<AbstractedPadFeature::SixAxisSensorUserCal>();
        pOutFeatureSet->Set<AbstractedPadFeature::LraRight>();
        pOutFeatureSet->Set<AbstractedPadFeature::LraLeft>();
        if (interfaceType == nn::xcd::InterfaceType_Bluetooth)
        {
            pOutFeatureSet->Set<AbstractedPadFeature::Nfc>();
        }

        break;
    case nn::xcd::DeviceType_Left:
        pOutFeatureSet->Set<AbstractedPadFeature::AnalogStick>();
        pOutFeatureSet->Set<AbstractedPadFeature::SixAxisSensorLeft>();
        pOutFeatureSet->Set<AbstractedPadFeature::AnalogStickUserCal>();
        pOutFeatureSet->Set<AbstractedPadFeature::SixAxisSensorUserCal>();
        pOutFeatureSet->Set<AbstractedPadFeature::LraLeft>();
        if (interfaceType == nn::xcd::InterfaceType_Bluetooth)
        {
            pOutFeatureSet->Set<AbstractedPadFeature::RailAttachmentLeft>();
        }
        break;
    case nn::xcd::DeviceType_Right:
        pOutFeatureSet->Set<AbstractedPadFeature::AnalogStick>();
        pOutFeatureSet->Set<AbstractedPadFeature::SixAxisSensorRight>();
        pOutFeatureSet->Set<AbstractedPadFeature::SixAxisSensorCenter>();
        pOutFeatureSet->Set<AbstractedPadFeature::AnalogStickUserCal>();
        pOutFeatureSet->Set<AbstractedPadFeature::SixAxisSensorUserCal>();
        pOutFeatureSet->Set<AbstractedPadFeature::LraRight>();
        if (interfaceType == nn::xcd::InterfaceType_Bluetooth ||
            interfaceType == nn::xcd::InterfaceType_Uart)
        {
            pOutFeatureSet->Set<AbstractedPadFeature::Nfc>();
            pOutFeatureSet->Set<AbstractedPadFeature::IrSensor>();
        }
        if (interfaceType == nn::xcd::InterfaceType_Bluetooth)
        {
            pOutFeatureSet->Set<AbstractedPadFeature::RailAttachmentRight>();
        }
        break;
    case nn::xcd::DeviceType_MiyabiLeft:
        pOutFeatureSet->Set<AbstractedPadFeature::AnalogStick>();
        pOutFeatureSet->Set<AbstractedPadFeature::SixAxisSensorLeft>();
        pOutFeatureSet->Set<AbstractedPadFeature::AnalogStickUserCal>();
        pOutFeatureSet->Set<AbstractedPadFeature::SixAxisSensorUserCal>();
        pOutFeatureSet->Set<AbstractedPadFeature::LraLeft>();
        break;
    case nn::xcd::DeviceType_MiyabiRight:
        pOutFeatureSet->Set<AbstractedPadFeature::AnalogStick>();
        pOutFeatureSet->Set<AbstractedPadFeature::SixAxisSensorRight>();
        pOutFeatureSet->Set<AbstractedPadFeature::SixAxisSensorCenter>();
        pOutFeatureSet->Set<AbstractedPadFeature::AnalogStickUserCal>();
        pOutFeatureSet->Set<AbstractedPadFeature::SixAxisSensorUserCal>();
        pOutFeatureSet->Set<AbstractedPadFeature::LraRight>();
        break;
    case nn::xcd::DeviceType_Tarragon:
        pOutFeatureSet->Set<AbstractedPadFeature::AnalogStick>();
        pOutFeatureSet->Set<AbstractedPadFeature::SixAxisSensorCenter>();
        break;
    default:
        ;
    }
}

} // namespace

}}} // namespace nn::hid::detail
