﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/result/result_HandlingUtility.h>

#include "hid_AbstractedAutoPilotVirtualPad.h"
#include "hid_CommonStateUtility.h"

namespace nn { namespace hid { namespace detail {

AbstractedAutoPilotVirtualPad::AbstractedAutoPilotVirtualPad() NN_NOEXCEPT
    : m_IsConnected(false)
    , m_PadState()
    , m_Pattern(0)
    , m_IsEnableSixAxisSensor(false)
{
    // Usb 接続
    m_InterfaceType = system::InterfaceType_Bluetooth;

    // Switch Pro Controller 互換
    m_DeviceType = system::DeviceType::SwitchProController::Mask;

    // デバイス色
    m_Color = system::ControllerColor();

    // FeatureSet
    m_FeatureSet.Reset();
    m_FeatureSet.Set<AbstractedPadFeature::AnalogStick>();
}

AbstractedAutoPilotVirtualPad::~AbstractedAutoPilotVirtualPad() NN_NOEXCEPT
{
    // 何もしない
}

AbstractedPadType AbstractedAutoPilotVirtualPad::GetType() NN_NOEXCEPT
{
    return s_Type;
}

bool AbstractedAutoPilotVirtualPad::IsConnected() NN_NOEXCEPT
{
    return m_IsConnected;
}

system::PowerInfo AbstractedAutoPilotVirtualPad::GetPowerInfo() NN_NOEXCEPT
{
    return m_PowerInfo;
}

AbstractedPadState AbstractedAutoPilotVirtualPad::GetPadState() NN_NOEXCEPT
{
    return m_PadState;
}

bool AbstractedAutoPilotVirtualPad::GetButtonTriggerElapsedTime(nn::os::Tick* pOutTick, AbstractedPadButtonSet button) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(button.CountPopulation(), 1);
    NN_UNUSED(pOutTick);
    NN_UNUSED(button);

    // ボタン押しをさかのぼる機能は提供しない
    return false;
}

system::ControllerColor AbstractedAutoPilotVirtualPad::GetColor() NN_NOEXCEPT
{
    if (m_IsConnected == false)
    {
        return system::ControllerColor();
    }
    return m_Color;
}

int AbstractedAutoPilotVirtualPad::GetSensorStates(nn::xcd::SixAxisSensorState* pOutValue, int count) NN_NOEXCEPT
{
    NN_UNUSED(pOutValue);
    NN_UNUSED(count);
    return 0;
}

void AbstractedAutoPilotVirtualPad::SetIndicator(uint8_t pattern, bool blink) NN_NOEXCEPT
{
    NN_UNUSED(blink);

    if (m_IsConnected)
    {
        m_Pattern = pattern;
    }
}

uint8_t AbstractedAutoPilotVirtualPad::GetIndicator() NN_NOEXCEPT
{
    if (m_IsConnected)
    {
        return m_Pattern;
    }

    return 0;
}

bool AbstractedAutoPilotVirtualPad::IsWired() NN_NOEXCEPT
{
    return (m_InterfaceType == system::InterfaceType_Usb);
}

bool AbstractedAutoPilotVirtualPad::IsUsbConnected() NN_NOEXCEPT
{
    if (m_IsConnected == false)
    {
        return false;
    }

    return (m_InterfaceType == system::InterfaceType_Usb);
}

void AbstractedAutoPilotVirtualPad::Connect() NN_NOEXCEPT
{
    // 接続状態に変更
    if (m_IsConnected == false)
    {
        m_IsConnected = true;
    }
}

void AbstractedAutoPilotVirtualPad::Detach() NN_NOEXCEPT
{
    m_IsConnected = false;
}

bool AbstractedAutoPilotVirtualPad::HasBattery() NN_NOEXCEPT
{
    if (m_IsConnected == false)
    {
        return false;
    }

    if (m_DeviceType.Test<system::DeviceType::SwitchProController>() ||
        m_DeviceType.Test<system::DeviceType::HandheldJoyLeft>() ||
        m_DeviceType.Test<system::DeviceType::HandheldJoyRight>() ||
        m_DeviceType.Test<system::DeviceType::JoyConLeft>() ||
        m_DeviceType.Test<system::DeviceType::JoyConRight>())
    {
        return true;
    }

    return false;
}

bool AbstractedAutoPilotVirtualPad::SetDeviceInfoOnPlayReportControllerUsage(system::PlayReportControllerUsage* pOutValue) NN_NOEXCEPT
{
    if (m_DeviceType.Test <system::DeviceType::JoyConLeft>() || m_DeviceType.Test <system::DeviceType::HandheldJoyLeft>() )
    {
        pOutValue->deviceType = system::PlayReportDeviceType_JoyConLeft;
    }
    else if (m_DeviceType.Test <system::DeviceType::JoyConRight>() || m_DeviceType.Test <system::DeviceType::HandheldJoyRight>())
    {
        pOutValue->deviceType = system::PlayReportDeviceType_JoyConRight;
    }
    else if (m_DeviceType.Test <system::DeviceType::SwitchProController>())
    {
        pOutValue->deviceType = system::PlayReportDeviceType_SwitchProController;
    }
    else
    {
        pOutValue->deviceType = system::PlayReportDeviceType_Unknown;
    }

    for (auto& byte : pOutValue->raw)
    {
        byte = 0;
    }
    return true;
}

void AbstractedAutoPilotVirtualPad::SetAbstractedPadIdPublisher(AbstractedPadIdPublisher* pPublisher) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPublisher);
    m_pIdPublisher = pPublisher;
}


::nn::Result AbstractedAutoPilotVirtualPad::SetAutoPilotState(
    const ::nn::hid::debug::AbstractedPadState& value) NN_NOEXCEPT
{
    {
        if (value.attributes.Test<nn::hid::debug::AbstractedPadAttribute::IsConnected>())
        {
            // 接続インターフェース
            m_InterfaceType = value.interfaceType;

            // デバイスタイプ
            m_DeviceType    = value.deviceType;

            // 電源状態
            m_PowerInfo     = value.powerInfo;

            // デバイス色
            m_Color = value.color;

            if (m_IsConnected == false)
            {
                m_Id = m_pIdPublisher->PublishId();
                m_IsConnected = true;

                // 6 軸センサー
                m_IsEnableSixAxisSensor = value.attributes.Test<nn::hid::debug::AbstractedPadAttribute::IsSixAxisSensorEnabled>();

                // 対応機能更新
                UpdateFeatureSet();
            }

            m_PadState.buttons.Reset();
            if (m_DeviceType.Test <system::DeviceType::JoyConLeft>()
                    || m_DeviceType.Test <system::DeviceType::HandheldJoyLeft>()
                    || m_DeviceType.Test<system::DeviceType::SwitchProController>())
            {
                // 左 Joy-Con
                m_PadState.buttons.Set<AbstractedPadButton::Left>(value.buttons.Test<AbstractedPadButton::Left>());
                m_PadState.buttons.Set<AbstractedPadButton::Up>(value.buttons.Test<AbstractedPadButton::Up>());
                m_PadState.buttons.Set<AbstractedPadButton::Right>(value.buttons.Test<AbstractedPadButton::Right>());
                m_PadState.buttons.Set<AbstractedPadButton::Down>(value.buttons.Test<AbstractedPadButton::Down>());
                m_PadState.buttons.Set<AbstractedPadButton::Select>(value.buttons.Test<AbstractedPadButton::Select>());
                m_PadState.buttons.Set<AbstractedPadButton::StickL>(value.buttons.Test<AbstractedPadButton::StickL>());
                m_PadState.buttons.Set<AbstractedPadButton::Shot>(value.buttons.Test<AbstractedPadButton::Shot>());
                m_PadState.buttons.Set<AbstractedPadButton::L>(value.buttons.Test<AbstractedPadButton::L>());
                m_PadState.buttons.Set<AbstractedPadButton::ZL>(value.buttons.Test<AbstractedPadButton::ZL>());

                if (m_DeviceType.Test <system::DeviceType::JoyConLeft>())
                {
                    m_PadState.buttons.Set<AbstractedPadButton::SL>(value.buttons.Test<AbstractedPadButton::SL>());
                    m_PadState.buttons.Set<AbstractedPadButton::SR>(value.buttons.Test<AbstractedPadButton::SR>());
                }

                m_PadState.analogStickL = LimitAnalogStick(
                    value.analogStickL, 0, AnalogStickMax, AnalogStickMax);
            }
            if (m_DeviceType.Test <system::DeviceType::JoyConRight>()
                    || m_DeviceType.Test <system::DeviceType::HandheldJoyRight>()
                    || m_DeviceType.Test<system::DeviceType::SwitchProController>())
            {
                // 右 Joy-Con
                m_PadState.buttons.Set<AbstractedPadButton::A>(value.buttons.Test<AbstractedPadButton::A>());
                m_PadState.buttons.Set<AbstractedPadButton::B>(value.buttons.Test<AbstractedPadButton::B>());
                m_PadState.buttons.Set<AbstractedPadButton::X>(value.buttons.Test<AbstractedPadButton::X>());
                m_PadState.buttons.Set<AbstractedPadButton::Y>(value.buttons.Test<AbstractedPadButton::Y>());
                m_PadState.buttons.Set<AbstractedPadButton::Start>(value.buttons.Test<AbstractedPadButton::Start>());
                m_PadState.buttons.Set<AbstractedPadButton::StickR>(value.buttons.Test<AbstractedPadButton::StickR>());
                m_PadState.buttons.Set<AbstractedPadButton::Home>(value.buttons.Test<AbstractedPadButton::Home>());
                m_PadState.buttons.Set<AbstractedPadButton::R>(value.buttons.Test<AbstractedPadButton::R>());
                m_PadState.buttons.Set<AbstractedPadButton::ZR>(value.buttons.Test<AbstractedPadButton::ZR>());

                if (m_DeviceType.Test <system::DeviceType::JoyConRight>())
                {
                    m_PadState.buttons.Set<AbstractedPadButton::SL>(value.buttons.Test<AbstractedPadButton::SL>());
                    m_PadState.buttons.Set<AbstractedPadButton::SR>(value.buttons.Test<AbstractedPadButton::SR>());
                }

                m_PadState.analogStickR = LimitAnalogStick(
                    value.analogStickR, 0, AnalogStickMax, AnalogStickMax);
            }
            // アナログスティックの十字ボタンエミュレーション
            SetCrossStickEmulationButtonsOnAbstractedPad(&m_PadState);

            // サンプル受信
            m_IsNewSampleReceived = true;
        }
        else
        {
            m_IsConnected = false;
            m_IsNewSampleReceived = false;
            m_PadState = AbstractedPadState();
        }
    }

    NN_RESULT_SUCCESS;
}

::nn::Result AbstractedAutoPilotVirtualPad::UnsetAutoPilotState() NN_NOEXCEPT
{
    m_IsConnected = false;
    m_IsNewSampleReceived = false;
    m_PadState = AbstractedPadState();
    NN_RESULT_SUCCESS;
}

void AbstractedAutoPilotVirtualPad::UpdateFeatureSet() NN_NOEXCEPT
{
    m_FeatureSet.Reset();
    if (m_DeviceType.Test<system::DeviceType::Unknown>() == false)
    {
        m_FeatureSet.Set<AbstractedPadFeature::AnalogStick>();
        if (m_DeviceType.Test<system::DeviceType::HandheldJoyRight>() ||
            m_DeviceType.Test<system::DeviceType::JoyConRight>() ||
            m_DeviceType.Test<system::DeviceType::FullKeyController>())
        {
            if (m_IsEnableSixAxisSensor)
            {
                m_FeatureSet.Set<AbstractedPadFeature::SixAxisSensorRight>();
                m_FeatureSet.Set<AbstractedPadFeature::SixAxisSensorCenter>();
            }
            m_FeatureSet.Set<AbstractedPadFeature::LraRight>();
        }
        if (m_DeviceType.Test<system::DeviceType::HandheldJoyLeft>() ||
            m_DeviceType.Test<system::DeviceType::JoyConLeft>() ||
            m_DeviceType.Test<system::DeviceType::FullKeyController>())
        {
            if (m_IsEnableSixAxisSensor)
            {
                m_FeatureSet.Set<AbstractedPadFeature::SixAxisSensorLeft>();

            }
            m_FeatureSet.Set<AbstractedPadFeature::LraLeft>();
        }

        m_FeatureSet.Set<AbstractedPadFeature::Nfc>(false);
        m_FeatureSet.Set<AbstractedPadFeature::IrSensor>(false);
        if (m_DeviceType.Test<system::DeviceType::JoyConRight>())
        {
            if (m_InterfaceType == system::InterfaceType_Bluetooth)
            {
                m_FeatureSet.Set<AbstractedPadFeature::RailAttachmentRight>();
            }
        }
        if (m_DeviceType.Test<system::DeviceType::JoyConLeft>())
        {
            if (m_InterfaceType == system::InterfaceType_Bluetooth)
            {
                m_FeatureSet.Set<AbstractedPadFeature::RailAttachmentLeft>();
            }
        }
        m_FeatureSet.Set<AbstractedPadFeature::AnalogStickUserCal>();
        if (m_IsEnableSixAxisSensor)
        {
            m_FeatureSet.Set<AbstractedPadFeature::SixAxisSensorUserCal>();
        }
    }
}

}}} // namespace nn::hid::detail
