﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "grcsrv_MovieTrimmerImpl.h"
#include <nn/grcsrv/trimming/grcsrv_MovieTrimmer.h>

#include <nn/nn_Common.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/grc/grc_Result.h>
#include <nn/nn_SdkLog.h>
#include <new>
#include <cstdlib>
#include <mm_MemoryManagement.h>

#include "grcsrv_AndroidUtility.h"

#define GRC_TRIM_LOG_ENABLED
#ifdef GRC_TRIM_LOG_ENABLED
    #define GRC_TRIM_LOG(...) NN_SDK_LOG("[grc:trim] " __VA_ARGS__)
#else
    #define GRC_TRIM_LOG(...) (void)0
#endif

namespace nn { namespace grcsrv { namespace trimming {

namespace detail {

Result ToResult(movie::Status movieStatus) NN_NOEXCEPT
{
    switch (movieStatus)
    {
        case movie::Status_Success: return ResultSuccess();
        case movie::Status_Malformed: return grc::ResultInvalidSource();
        default:
        {
            GRC_TRIM_LOG("movieStatus = 0x%04x\n", static_cast<int>(movieStatus));
            return grc::ResultInvalidState(); // TODO:(Result)
        }
    }
}

Result ToResult(AndroidStatus androidStatus) NN_NOEXCEPT
{
    switch (androidStatus.GetStatus())
    {
        case 0: return ResultSuccess();
        default:
        {
            auto movieStatus = AndroidStatusToMovieStatus(androidStatus.GetStatus());
            GRC_TRIM_LOG("movieStatus = 0x%04x(android=%d)\n", static_cast<int>(movieStatus), static_cast<int>(androidStatus.GetStatus()));
            return ToResult(movieStatus);
        }
    }
}

Result FileStreamReader::Initialize(nn::fs::FileHandle handle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(!m_Initialized);
    int64_t size;
    NN_RESULT_DO(nn::fs::GetFileSize(&size, handle));
    this->m_Initialized = true;
    this->m_Handle = handle;
    this->m_FileSize = size;
    this->m_LastResult = nn::ResultSuccess();
    NN_RESULT_SUCCESS;
}

Result FileStreamReader::GetLastResult() const NN_NOEXCEPT
{
    return m_LastResult;
}

bool FileStreamReader::Open(const char*, void** pOut)
{
    NN_ABORT_UNLESS(m_Initialized);
    NN_ABORT_UNLESS(!m_Opened);
    this->m_Opened = true;
    *pOut = this;
    return true;
}

size_t FileStreamReader::ReadAt(void* this_, int64_t offset, void* buffer, size_t size)
{
    NN_SDK_ASSERT(this_ == this);
    NN_UNUSED(this_);
    NN_ABORT_UNLESS(m_Opened);
    NN_ABORT_UNLESS(offset <= m_FileSize);
    if (!(size <= static_cast<uint64_t>(m_FileSize - offset)))
    {
        size = m_FileSize - offset;
    }

    size_t ret;
    auto result = nn::fs::ReadFile(&ret, m_Handle, offset, buffer, size);
    if (!result.IsSuccess())
    {
        this->m_LastResult = result;
        return 0;
    }
    return ret;
}

bool FileStreamReader::Close(void* this_)
{
    NN_SDK_ASSERT(this_ == this);
    NN_UNUSED(this_);
    NN_ABORT_UNLESS(m_Opened);
    this->m_Opened = false;
    return true;
}

void FileStreamReader::GetSize(void* this_, int64_t* pOut)
{
    NN_SDK_ASSERT(this_ == this);
    NN_UNUSED(this_);
    *pOut = this->m_FileSize;
}

class FileStreamWriter final : public android::MPEG4WriterStream
{
private:

    fs::FileHandle m_Handle;
    int64_t m_Offset = 0;
    Result m_Result = ResultSuccess();

public:

    explicit FileStreamWriter(fs::FileHandle handle) NN_NOEXCEPT
        : m_Handle(handle)
    {
    }

    ~FileStreamWriter() NN_NOEXCEPT
    {
        fs::FlushFile(m_Handle);
    }

    virtual void SetPosition(int64_t offset) NN_OVERRIDE
    {
        this->m_Offset = offset;
    }

    virtual void Write(const void* buffer, uint32_t size) NN_OVERRIDE
    {
        if (!m_Result.IsSuccess())
        {
            return;
        }
        this->m_Result = fs::WriteFile(m_Handle, m_Offset, buffer, size, fs::WriteOption::MakeValue(0));
        this->m_Offset += size;
    }

};

bool IsKeyFrame(const void* buffer, size_t bufferSize)
{
    auto p = static_cast<const uint8_t*>(buffer);
    if (bufferSize < 4)
    {
        return false;
    }
    for (size_t i = 0; i < bufferSize - 4; ++i)
    {
        if (p[i + 0] == 0x00 && p[i + 1] == 0x00 && p[i + 2] == 0x01)
        {
            const uint8_t ExpectedNalType = 5;
            if ((p[i + 3] & 0x1F) == ExpectedNalType)
            {
                return true;
            }
        }
    }
    return false;
}

} // namespace detail

#define NN_GRCSRV_TRIM_DO(s) NN_RESULT_DO(::nn::grcsrv::trimming::detail::ToResult(s))

Result MovieTrimmerImpl::InitializeImpl(fs::FileHandle handle, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    auto success = false;

    // TODO: 常時録画のほうの設定と競合するため、排他するか、より適切な場所に移すべき
    nv::mm::SetAllocator(
        [](size_t size, size_t alignment, void*)
        {
            return ::aligned_alloc(alignment, size);
        },
        [](void* p, void*)
        {
            ::free(p);
        },
        [](void* p, size_t size, void*)
        {
            return ::realloc(p, size);
        },
        nullptr
    );

    NN_RESULT_DO(m_Reader.Initialize(handle));

    m_pDemuxer.emplace(movie::ContainerType_Mpeg4, movie::CacheSize_5MB);
    NN_UTIL_SCOPE_EXIT
    {
        if (!success)
        {
            m_pDemuxer = util::nullopt;
        }
    };

    NN_GRCSRV_TRIM_DO(m_pDemuxer->SetDataSource(&m_Reader, ""));

    success = true;
    this->m_WorkBuffer = buffer;
    this->m_WorkBufferSize = bufferSize;
    NN_RESULT_SUCCESS;
}

Result MovieTrimmerImpl::ReadHeaderImpl() NN_NOEXCEPT
{
    int32_t trackCount;
    NN_GRCSRV_TRIM_DO(m_pDemuxer->GetTrackCount(&trackCount));
    GRC_TRIM_LOG("Retrieving track information:\n");
    for (auto i = 0; i < trackCount; ++i)
    {
        NN_RESULT_DO(ReadTrack(i));
    }
    NN_RESULT_THROW_UNLESS(m_VideoTrackIndex >= 0, grc::ResultInvalidSource());
    NN_RESULT_THROW_UNLESS(m_AudioTrackIndex >= 0, grc::ResultInvalidSource());
    NN_RESULT_SUCCESS;
}

Result MovieTrimmerImpl::ReadTrack(int index) NN_NOEXCEPT
{
    movie::MediaData trackFormat;
    NN_GRCSRV_TRIM_DO(m_pDemuxer->GetTrackConfiguration(index, &trackFormat));
    NN_UTIL_SCOPE_EXIT
    {
        trackFormat.Clear();
    };

    const char* mime;
    if (!trackFormat.FindString("mime", &mime))
    {
        NN_RESULT_SUCCESS;
    }

    if (strncasecmp("video/", mime, 6) == 0)
    {
        if (!(strncasecmp("video/avc", mime, 9) == 0))
        {
            GRC_TRIM_LOG("  └─ CRITICAL ERROR : the mp4 video track doesn't contain a h264 video stream (current = %s)\n", mime);
            NN_RESULT_THROW(grc::ResultInvalidSource());
        }
        android::sp<android::AMessage> formatHeader;
        NN_RESULT_DO(ReadVideoTrack(&formatHeader, trackFormat));
        NN_GRCSRV_TRIM_DO(m_pDemuxer->SelectTrack(index));
        this->m_VideoTrackIndex = index;
        this->m_VideoFormatHeader = formatHeader;
        NN_RESULT_SUCCESS;
    }
    if (strncasecmp("audio/", mime, 6) == 0)
    {
        if (strncasecmp("audio/mp4a-latm", mime, 15))
        {
            GRC_TRIM_LOG("  └─ CRITICAL ERROR : the mp4 video track doesn't contain an AAC audio stream (current = %s)\n", mime);
            NN_RESULT_THROW(grc::ResultInvalidSource());
        }
        GRC_TRIM_LOG("  ├─ Audio format     : %s\n", mime);
        android::sp<android::AMessage> formatHeader;
        NN_RESULT_DO(ReadAudioTrack(&formatHeader, trackFormat));
        NN_GRCSRV_TRIM_DO(m_pDemuxer->SelectTrack(index));
        this->m_AudioTrackIndex = index;
        this->m_AudioFormatHeader = formatHeader;
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_SUCCESS;
}

android::sp<android::AMessage> CreateVideoFormatMP4(void* bufferSPS, size_t sizeSPS, void* bufferPPS, size_t sizePPS)
{
    android::sp<android::ABuffer> csd0 = new android::ABuffer(sizeSPS);
    android::sp<android::ABuffer> csd1 = new android::ABuffer(sizePPS);
    std::memcpy(csd0->data(), bufferSPS, sizeSPS);
    std::memcpy(csd1->data(), bufferPPS, sizePPS);

    android::sp<android::AMessage> videoFormat = new android::AMessage();
    videoFormat->setString("mime", "video/avc");
    videoFormat->setInt32 ("width", 1280);
    videoFormat->setInt32 ("height", 720);
    videoFormat->setInt32 ("what", 0x6F757443);
    videoFormat->setBuffer("csd-0", csd0);
    videoFormat->setBuffer("csd-1", csd1);
    return videoFormat;
}

Result MovieTrimmerImpl::ReadVideoTrack(android::sp<android::AMessage>* pOut, const movie::MediaData& trackFormat) NN_NOEXCEPT
{
    movie::Buffer* origHeaderAvcc;
    if (!trackFormat.FindBuffer("avcc", &origHeaderAvcc))
    {
        GRC_TRIM_LOG("  └─ CRITICAL ERROR : the mp4 video track doesn't contain any AVCC header !\n");
        NN_RESULT_THROW(grc::ResultInvalidSource());
    }

    android::sp<android::AMessage> pOriginalMessageHeader;
    NN_GRCSRV_TRIM_DO(detail::AndroidUtils::ConvertMediaDataToMessage(&trackFormat, &pOriginalMessageHeader));
    GRC_TRIM_LOG("------- originalMessageHeader:\n");

    android::sp<android::ABuffer> valCSD0;
    android::sp<android::ABuffer> valCSD1;
    if (!pOriginalMessageHeader->findBuffer("csd-0", &valCSD0))
    {
        GRC_TRIM_LOG("  └─ CRITICAL ERROR : missing csd-0 part in AVCC header !\n");
        NN_RESULT_THROW(grc::ResultInvalidSource());
    }
    if (!pOriginalMessageHeader->findBuffer("csd-1", &valCSD1))
    {
        GRC_TRIM_LOG("  └─ CRITICAL ERROR : missing csd-1 part in AVCC header !\n");
        NN_RESULT_THROW(grc::ResultInvalidSource());
    }

    *pOut = CreateVideoFormatMP4(valCSD0->data(), valCSD0->size(), valCSD1->data(), valCSD1->size());
    GRC_TRIM_LOG("------- m_videoFormatHeader:\n");
    NN_RESULT_SUCCESS;
}

android::sp<android::AMessage> CreateAudioFormatAAC(const int audioChannels, const int audioFrequency)
{
    static uint8_t csdData[] = {0x11,0x90}; // Important to keep it STATIC due to usage outside of this function.
    android::sp<android::AMessage> audioFormat = new android::AMessage();
    audioFormat->setString("mime", "audio/mp4a-latm");
    audioFormat->setInt32("channel-count", audioChannels);
    audioFormat->setInt32("sample-rate", audioFrequency);
    audioFormat->setInt32("what", 0x6F757443);
    audioFormat->setBuffer("csd-0", new android::ABuffer(csdData, sizeof(csdData)));
    return audioFormat;
}

Result MovieTrimmerImpl::ReadAudioTrack(android::sp<android::AMessage>* pOut, const movie::MediaData& trackFormat) NN_NOEXCEPT
{
    int32_t audioSampleRate;
    NN_RESULT_THROW_UNLESS(trackFormat.FindInt32("sample-rate", &audioSampleRate), grc::ResultInvalidSource());
    GRC_TRIM_LOG("  ├─ Audio samplerate : %d Hz\n", audioSampleRate);

    int32_t audioChannels;
    NN_RESULT_THROW_UNLESS(trackFormat.FindInt32("channel-count", &audioChannels), grc::ResultInvalidSource());
    GRC_TRIM_LOG("  ├─ Audio channels   : %d\n", audioChannels);

    // Build a clean target message
    *pOut = CreateAudioFormatAAC(audioChannels, audioSampleRate);
    NN_RESULT_SUCCESS;
}

Result MovieTrimmerImpl::TrimImpl(int* pFrameCount, fs::FileHandle handle, int beginIndex, int endIndex, const capsrv::ScreenShotAttribute& attribute) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(beginIndex >= 0);
    NN_SDK_REQUIRES(beginIndex <= endIndex);

    //NN_GRCSRV_TRIM_DO(m_pDemuxer->SeekTo(0));

    detail::FileStreamWriter writer{handle};
    android::MediaMuxer muxer(&writer, android::MediaMuxer::OUTPUT_FORMAT_MPEG_4);

    // Add Video track
    GRC_TRIM_LOG("---- Adding VIDEO track with format:\n");
    auto videoTrackIndex = muxer.addTrack(m_VideoFormatHeader);
    if (!(videoTrackIndex >= 0))
    {
        GRC_TRIM_LOG("ERROR: Failed to add video track to muxer\n");
        NN_RESULT_THROW(grc::ResultInvalidSource());
    }
    {
        int videoOrientation = (360 - attribute.orientation * 90) % 360;
        muxer.setOrientationHint(videoOrientation);
    }

    // Add Audio track
    GRC_TRIM_LOG("---- Adding AUDIO track with format:\n");
    auto audioTrackIndex = muxer.addTrack(m_AudioFormatHeader);
    if (!(audioTrackIndex >= 0))
    {
        GRC_TRIM_LOG("ERROR: Failed to add audio track to muxer\n");
        NN_RESULT_THROW(grc::ResultInvalidSource());
    }

    NN_GRCSRV_TRIM_DO(muxer.start());
    auto muxerStopped = false;
    NN_UTIL_SCOPE_EXIT
    {
        if (!muxerStopped)
        {
            muxer.stop();
        }
    };

    NN_GRCSRV_TRIM_DO(m_pDemuxer->SeekTo(0));

    auto frameCount = 0;
    auto currentKeyFrameIndex = -1;
    util::optional<int64_t> firstTimeStamp;
    while (currentKeyFrameIndex < endIndex)
    {
        size_t currentTrackIndex;
        {
            auto movieStatus = m_pDemuxer->GetTrackIndexForAvailableData(&currentTrackIndex);
            if (movieStatus == movie::Status_EndOfStream)
            {
                break;
            }
            NN_GRCSRV_TRIM_DO(movieStatus);
        }

        movie::Buffer movieBuffer(m_WorkBuffer, m_WorkBufferSize);
        {
            auto movieStatus = m_pDemuxer->Read(&movieBuffer);
            if (movieStatus == movie::Status_EndOfStream)
            {
                break;
            }
            NN_GRCSRV_TRIM_DO(movieStatus);
        }

        int64_t inputTimeStamp;
        {
            auto movieStatus = m_pDemuxer->GetSampleTime(&inputTimeStamp);
            if (movieStatus == movie::Status_EndOfStream)
            {
                break;
            }
            NN_GRCSRV_TRIM_DO(movieStatus);
        }

        m_pDemuxer->Advance();

        auto isKeyFrame = currentTrackIndex == m_VideoTrackIndex && detail::IsKeyFrame(movieBuffer.Base(), movieBuffer.Size());
        if (isKeyFrame)
        {
            ++currentKeyFrameIndex;
            if (!(currentKeyFrameIndex < endIndex))
            {
                // 最終キーフレームを付加するのであれば、ここの break をコメントアウトする
                break;
            }
        }

        if (beginIndex <= currentKeyFrameIndex)
        {
            if (!firstTimeStamp)
            {
                firstTimeStamp = inputTimeStamp;
            }

            int trackIndex;
            if (currentTrackIndex == m_VideoTrackIndex)
            {
                trackIndex = videoTrackIndex;
            }
            else if (currentTrackIndex == m_AudioTrackIndex)
            {
                trackIndex = audioTrackIndex;
            }
            else
            {
                NN_RESULT_THROW(grc::ResultInvalidSource());
            }

            auto aBuffer = new android::ABuffer(movieBuffer.Base(), movieBuffer.Size());
            auto timeStamp = inputTimeStamp - *firstTimeStamp;
            auto flags = isKeyFrame ? android::MediaCodec::BUFFER_FLAG_SYNCFRAME : 0;
            NN_GRCSRV_TRIM_DO(muxer.writeSampleData(aBuffer, trackIndex, timeStamp, flags));
            if (currentTrackIndex == m_VideoTrackIndex)
            {
                ++frameCount;
            }
        }
    }

    muxerStopped = true;
    NN_GRCSRV_TRIM_DO(muxer.stop());

    *pFrameCount = frameCount;
    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

Result MovieTrimmerImpl::Initialize(fs::FileHandle handle, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    auto result = InitializeImpl(handle, buffer, bufferSize);
    NN_RESULT_DO(m_Reader.GetLastResult());
    return result;
}

Result MovieTrimmerImpl::ReadHeader() NN_NOEXCEPT
{
    auto result = ReadHeaderImpl();
    NN_RESULT_DO(m_Reader.GetLastResult());
    return result;
}

Result MovieTrimmerImpl::Trim(int* pFrameCount, fs::FileHandle handle, int beginIndex, int endIndex, const capsrv::ScreenShotAttribute& attribute) NN_NOEXCEPT
{
    auto result = TrimImpl(pFrameCount, handle, beginIndex, endIndex, attribute);
    NN_RESULT_DO(m_Reader.GetLastResult());
    return result;
}

void InitializeTrimmingStatic() NN_NOEXCEPT
{
    movie::SetCoreMask(1 << 3);
    movie::SetAllocator(
        [](size_t size, size_t alignment, void*)
        {
            return ::aligned_alloc(alignment, size);
        },
        [](void* p, void*)
        {
            ::free(p);
        },
        [](void* p, size_t size, void*)
        {
            return ::realloc(p, size);
        },
        nullptr
    );
};

MovieTrimmer::MovieTrimmer() NN_NOEXCEPT
{
    static_assert(sizeof(m_ImplBuffer) >= sizeof(MovieTrimmerImpl), "");
    this->m_pImpl = new (m_ImplBuffer) MovieTrimmerImpl();
}

MovieTrimmer::~MovieTrimmer() NN_NOEXCEPT
{
    m_pImpl->~MovieTrimmerImpl();
}

Result MovieTrimmer::Initialize(fs::FileHandle handle, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    return m_pImpl->Initialize(handle, buffer, bufferSize);
}

Result MovieTrimmer::ReadHeader() NN_NOEXCEPT
{
    return m_pImpl->ReadHeader();
}

Result MovieTrimmer::Trim(int* pFrameCount, fs::FileHandle handle, int beginIndex, int endIndex, const capsrv::ScreenShotAttribute& attribute) NN_NOEXCEPT
{
    return m_pImpl->Trim(pFrameCount, handle, beginIndex, endIndex, attribute);
}

}}}
