﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "grcsrv_AndroidUtility.h"

#include <algorithm>
#include <cstring>
#include <strings.h>
#include <nn/util/util_StringUtil.h>

namespace nn { namespace grcsrv { namespace trimming { namespace detail {

movie::Status AndroidStatusToMovieStatus(android::status_t androidResult) NN_NOEXCEPT
{
    using namespace android;
    switch (androidResult)
    {
        case OK: return movie::Status_Success;
        case -EAGAIN: return movie::Status_NotEnoughData;
        case android::ERROR_END_OF_STREAM: return movie::Status_EndOfStream;
        case INFO_DISCONTINUITY: return movie::Status_Discontinuity;
        case ERROR_ALREADY_CONNECTED: return movie::Status_AlreadyConnected;
        case ERROR_NOT_CONNECTED: return movie::Status_NotConnected;
        case ERROR_UNKNOWN_HOST: return movie::Status_UnknownHost;
        case ERROR_CANNOT_CONNECT: return movie::Status_CanNotConnect;
        case ERROR_IO: return movie::Status_IoError;
        case ERROR_CONNECTION_LOST: return movie::Status_ConnectionLost;
        case ERROR_MALFORMED: return movie::Status_Malformed;
        case ERROR_OUT_OF_RANGE: return movie::Status_OutOfRange;
        case ERROR_BUFFER_TOO_SMALL: return movie::Status_BufferTooSmall;
        case ERROR_UNSUPPORTED: return movie::Status_NotSupported;
        case INFO_FORMAT_CHANGED: return movie::Status_FormatChanged;
        case INFO_OUTPUT_BUFFERS_CHANGED: return movie::Status_OutputBuffersChanged;
        case ERROR_OUT_OF_MEMORY: return movie::Status_OutOfMemory;
        default: return movie::Status_UnknownError;
    }
}

namespace {

inline uint16_t ReadAsUint16(const uint8_t *ptr) NN_NOEXCEPT
{
    return ptr[0] << 8 | ptr[1];
}

}

AndroidStatus AndroidUtils::ConvertMediaDataToMessage(const movie::MediaData* meta, android::sp<android::AMessage> *format) NN_NOEXCEPT
{
    using namespace android;
    using namespace movie;

    format->clear();

    sp<AMessage> msg = new AMessage;
    const char *mime;

    if (meta->FindString("mime", &mime))
    {
        AString str(mime, util::Strnlen(mime, 64));
        msg->setString("mime", str);
    }

    {
        int64_t durationUs;
        if (meta->FindInt64("durationUs", &durationUs))
        {
            msg->setInt64("durationUs", durationUs);
        }
    }

    {
        int avgBitRate;
        if (meta->FindInt32("bit-rate", &avgBitRate))
        {
            msg->setInt32("bit-rate", avgBitRate);
        }
    }

    {
        int32_t isSync;
        if (meta->FindInt32("is-sync-frame", &isSync) && isSync != 0)
        {
            msg->setInt32("is-sync-frame", 1);
        }
    }

    if (!strncasecmp("video/", mime, 6))
    {
        {
            int32_t width;
            if (meta->FindInt32("width", &width))
            {
                msg->setInt32("width", width);
            }
        }
        {
            int32_t height;
            if (meta->FindInt32("height", &height))
            {
                msg->setInt32("height", height);
            }
        }
        {
            int32_t sarWidth, sarHeight;
            if (meta->FindInt32("sar-width", &sarWidth) && meta->FindInt32("sar-height", &sarHeight))
            {
                msg->setInt32("sar-width", sarWidth);
                msg->setInt32("sar-height", sarHeight);
            }
        }
        {
            int32_t colorFormat;
            if (meta->FindInt32("color-format", &colorFormat))
            {
                msg->setInt32("color-format", colorFormat);
            }
        }
        {
            int32_t cropLeft, cropTop, cropRight, cropBottom;
            if (meta->FindRect("crop", &cropLeft, &cropTop, &cropRight, &cropBottom))
            {
                msg->setRect("crop", cropLeft, cropTop, cropRight, cropBottom);
            }
        }
        {
            int32_t rotationDegrees;
            if (meta->FindInt32("rotation-degrees", &rotationDegrees))
            {
                msg->setInt32("rotation-degrees", rotationDegrees);
            }
        }
        {
            int32_t videoFrameRate;
            if (meta->FindInt32("frame-rate", &videoFrameRate))
            {
                msg->setInt32("frame-rate", videoFrameRate);
            }
        }
    }
    else if (!strncasecmp("audio/", mime, 6))
    {
        {
            int32_t numChannels;
            if (meta->FindInt32("channel-count", &numChannels))
            {
                msg->setInt32("channel-count", numChannels);
            }
        }
        {
            int32_t sampleRate;
            if (meta->FindInt32("sample-rate", &sampleRate))
            {
                msg->setInt32("sample-rate", sampleRate);
            }
        }
        {
            int32_t channelMask;
            if (meta->FindInt32("channel-mask", &channelMask))
            {
                msg->setInt32("channel-mask", channelMask);
            }
        }
        {
            int32_t delay = 0;
            if (meta->FindInt32("encoder-delay", &delay))
            {
                msg->setInt32("encoder-delay", delay);
            }
        }
        {
            int32_t padding = 0;
            if (meta->FindInt32("encoder-padding", &padding))
            {
                msg->setInt32("encoder-padding", padding);
            }
        }

        {
            int32_t isADTS;
            if (meta->FindInt32("is-adts", &isADTS))
            {
                msg->setInt32("is-adts", true);
            }
        }

        {
            int32_t aacProfile = -1;
            if (meta->FindInt32("aac-profile", &aacProfile))
            {
                msg->setInt32("aac-profile", aacProfile);
            }
        }
    }

    {
        int32_t maxInputSize;
        if (meta->FindInt32("max-input-size", &maxInputSize))
        {
            msg->setInt32("max-input-size", maxInputSize);
        }
    }
    {
        int32_t rotationDegrees;
        if (meta->FindInt32("rotation-degrees", &rotationDegrees))
        {
            msg->setInt32("rotation-degrees", rotationDegrees);
        }
    }

    Buffer *temp;
    if (meta->FindBuffer("avcc", &temp))
    {
        // Parse the AVCDecoderConfigurationRecord

        auto ptr = temp->Base();
        auto size = temp->Capacity();
        if (!(size >= 7))
        {
            return ERROR_MALFORMED;
        }
        if (!(ptr[0] == 1)) // configurationVersion == 1
        {
            return ERROR_MALFORMED;
        }
        auto profile = ptr[1];
        NN_UNUSED(profile);
        auto level = ptr[3];
        NN_UNUSED(level);

        // There is decodable content out there that fails the following
        // assertion, let's be lenient for now...
        // CHECK((ptr[4] >> 2) == 0x3f);  // reserved

        size_t lengthSize = 1 + (ptr[4] & 3);

        // commented out check below as H264_QVGA_500_NO_AUDIO.3gp
        // violates it...
        // CHECK((ptr[5] >> 5) == 7);  // reserved

        size_t numSeqParameterSets = ptr[5] & 31;

        ptr += 6;
        size -= 6;

        const uint32_t bufferSize = 1024;
        sp<ABuffer> buffer = new ABuffer(bufferSize);
        buffer->setRange(0, 0);

        for (size_t i = 0; i < numSeqParameterSets; ++i)
        {
            if (!(size >= 2))
            {
                return ERROR_MALFORMED;
            }
            size_t length = ReadAsUint16(ptr);

            ptr += 2;
            size -= 2;

            if (!(size >= length && length + 4 <= bufferSize))
            {
                return ERROR_MALFORMED;
            }

            memcpy(buffer->data() + buffer->size(), "\x00\x00\x00\x01", 4);
            memcpy(buffer->data() + buffer->size() + 4, ptr, length);
            buffer->setRange(0, buffer->size() + 4 + length);

            ptr += length;
            size -= length;
        }

        buffer->meta()->setInt32("csd", true);
        buffer->meta()->setInt64("timeUs", 0);

        msg->setBuffer("csd-0", buffer);

        buffer = new ABuffer(1024);
        buffer->setRange(0, 0);

        if (!(size >= 1))
        {
            return ERROR_MALFORMED;
        }
        size_t numPictureParameterSets = *ptr;
        ++ptr;
        --size;

        for (size_t i = 0; i < numPictureParameterSets; ++i)
        {
            if (!(size >= 2))
            {
                return ERROR_MALFORMED;
            }
            size_t length = ReadAsUint16(ptr);

            ptr += 2;
            size -= 2;

            if (!(size >= length && length + 4 <= bufferSize))
            {
                return ERROR_MALFORMED;
            }

            memcpy(buffer->data() + buffer->size(), "\x00\x00\x00\x01", 4);
            memcpy(buffer->data() + buffer->size() + 4, ptr, length);
            buffer->setRange(0, buffer->size() + 4 + length);

            ptr += length;
            size -= length;
        }

        buffer->meta()->setInt32("csd", true);
        buffer->meta()->setInt64("timeUs", 0);
        msg->setBuffer("csd-1", buffer);
    }
    else
    {
        return ERROR_MALFORMED;
    }
#if 0
    else if (meta->FindBuffer("hvcc", &temp))
    {
        const uint8_t *ptr = (const uint8_t *)temp->Base();
        auto size = temp->Capacity();
        //CHECK(size >= 7);
        //CHECK_EQ((unsigned)ptr[0], 1u);  // configurationVersion == 1
        uint8_t profile = ptr[1] & 31;
        uint8_t level = ptr[12];
        ptr += 22;
        size -= 22;


        size_t numofArrays = (char)ptr[0];
        ptr += 1;
        size -= 1;
        size_t j = 0, i = 0;

        sp<ABuffer> buffer = new ABuffer(1024);
        buffer->setRange(0, 0);

        for (i = 0; i < numofArrays; i++)
        {
            ptr += 1;
            size -= 1;

            //Num of nals
            size_t numofNals = ReadAsUint16(ptr);

            ptr += 2;
            size -= 2;

            for (j = 0; j < numofNals; j++)
            {
                //CHECK(size >= 2);
                size_t length = ReadAsUint16(ptr);

                ptr += 2;
                size -= 2;

                //CHECK(size >= length);

                memcpy(buffer->data() + buffer->size(), "\x00\x00\x00\x01", 4);
                memcpy(buffer->data() + buffer->size() + 4, ptr, length);
                buffer->setRange(0, buffer->size() + 4 + length);

                ptr += length;
                size -= length;
            }
        }
        buffer->meta()->setInt32("csd", true);
        buffer->meta()->setInt64("timeUs", 0);
        msg->setBuffer("csd-0", buffer);

    }
    else if (meta->FindBuffer("esds", &temp))
    {
        NN_SDK_ASSERT(false);
        //data = (const void *)temp->Base();
        //size = temp->Capacity();

        //ESDS esds((const char *)data, size);
        //esds.InitCheck();

        //const void *codec_specific_data;
        //size_t codec_specific_data_size;
        //esds.getCodecSpecificInfo(&codec_specific_data, &codec_specific_data_size);

        //sp<ABuffer> buffer = new ABuffer(codec_specific_data_size);

        //memcpy(buffer->data(), codec_specific_data, codec_specific_data_size);

        //buffer->meta()->setInt32("csd", true);
        //buffer->meta()->setInt64("timeUs", 0);
        //msg->setBuffer("csd-0", buffer);
    }
    else if (meta->FindBuffer("ohdr", &temp))
    {
        data = (const void *)temp->Base();
        auto size = temp->Capacity();

        sp<ABuffer> buffer = new ABuffer(size);
        memcpy(buffer->data(), data, size);

        buffer->meta()->setInt32("csd", true);
        buffer->meta()->setInt64("timeUs", 0);
        msg->setBuffer("csd-0", buffer);
    }
    else
    {
        bool Oggcsd0 = false;
        bool Oggcsd1 = false;
        const void *dataBcsd0;
        Buffer *tempBcsd0;
        size_t sizeBcsd0;
        const void *dataBcsd1;
        Buffer *tempBcsd1;
        size_t sizeBcsd1;

        if (meta->FindBuffer("csd-0", &tempBcsd0))
        {
            Oggcsd0 = true;
            dataBcsd0 = (const void *)tempBcsd0->Base();
            sizeBcsd0 = tempBcsd0->Capacity();
        }

        if (meta->FindBuffer("csd-1", &tempBcsd1))
        {
            Oggcsd1 = true;
            dataBcsd1 = (const void *)tempBcsd1->Base();
            sizeBcsd1 = tempBcsd1->Capacity();
        }
        if( Oggcsd0 && Oggcsd1 )
        {
            sp<ABuffer> buffer0 = new ABuffer(sizeBcsd0);
            buffer0->meta()->setInt32("csd", true);
            buffer0->meta()->setInt64("timeUs", 0);
            memcpy(buffer0->data(), dataBcsd0, sizeBcsd0);
            msg->setBuffer("csd-0", buffer0);

            sp<ABuffer> buffer1 = new ABuffer(sizeBcsd1);
            buffer1->meta()->setInt32("csd", true);
            buffer1->meta()->setInt64("timeUs", 0);
            memcpy(buffer1->data(), dataBcsd1, sizeBcsd1);
            msg->setBuffer("csd-1", buffer1);
        }
    }
#endif

    *format = msg;

    return OK;
} // NOLINT(impl/function_size)

}}}}
