﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Result.h>
#include <nn/os.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/vi/vi_Result.h>

namespace nn{ namespace grcsrv{ namespace offscreen{ namespace util{

    template<typename T, int TCapacity, bool TThreadSafe>
    class TQueue;

    template<typename T, int TCapacity>
    class TQueue<T, TCapacity, false>
    {
    public:
        static const int Capacity = TCapacity;

        typedef T ValueType;

    protected:
        static const int InternalSize = Capacity + 1;

    public:
        TQueue() NN_NOEXCEPT
            : m_EntryList()
            , m_HeadIndex(0)
            , m_TailIndex(0)
        {
        }

        void Initialize() NN_NOEXCEPT
        {
            m_HeadIndex = 0;
            m_TailIndex = 0;
            for(int i = 0; i < InternalSize; i++)
            {
                m_EntryList[i] = T();
            }
        }

        void Finalize() NN_NOEXCEPT
        {
            Clear();
        }

        bool IsFull() const NN_NOEXCEPT
        {
            return ((m_TailIndex + 1) % InternalSize) == m_HeadIndex;
        }

        bool IsEmpty() const NN_NOEXCEPT
        {
            return m_TailIndex == m_HeadIndex;
        }

        int GetCount() const NN_NOEXCEPT
        {
            int tail = m_TailIndex;
            if(m_TailIndex < m_HeadIndex)
            {
                tail += InternalSize;
            }
            NN_SDK_ASSERT_GREATER_EQUAL(tail, m_HeadIndex);
            return tail - m_HeadIndex;
        }

        void Clear() NN_NOEXCEPT
        {
            m_HeadIndex = 0;
            m_TailIndex = 0;
            for(int i = 0; i < InternalSize; i++)
            {
                m_EntryList[i] = T();
            }
        }

        nn::Result TryEnqueue(T entry) NN_NOEXCEPT
        {
            int nextTail = (m_TailIndex + 1) % InternalSize;
            NN_RESULT_THROW_UNLESS(nextTail != m_HeadIndex, nn::vi::ResultNotReady());

            m_EntryList[m_TailIndex] = entry;
            m_TailIndex = nextTail;
            NN_RESULT_SUCCESS;
        }

        nn::Result TryDequeue(T* pOutEntry) NN_NOEXCEPT
        {
            NN_RESULT_THROW_UNLESS(m_HeadIndex != m_TailIndex, nn::vi::ResultNotReady());

            *pOutEntry = m_EntryList[m_HeadIndex];
            m_EntryList[m_HeadIndex] = T();
            m_HeadIndex = (m_HeadIndex + 1) % InternalSize;
            NN_RESULT_SUCCESS;
        }

        //// 次にデキューされる要素のポインタを取得する。この関数ではデキューされない。
        //// キューが空の場合 nullptr が返る。
        //T* GetHead() NN_NOEXCEPT
        //{
        //    if(IsEmpty())
        //    {
        //        return nullptr;
        //    }
        //    return &m_EntryList[m_HeadIndex];
        //}

        // 次にデキューされる要素のポインタを取得する。この関数ではデキューされない。
        // キューが空の場合 nullptr が返る。
        const T* GetHead() const NN_NOEXCEPT
        {
            if(IsEmpty())
            {
                return nullptr;
            }
            return &m_EntryList[m_HeadIndex];
        }

        //// 最後にエンキューされた要素のポインタを取得する。
        //// キューが空の場合 nullptr が返る。
        //T* GetTail() NN_NOEXCEPT
        //{
        //    if(IsEmpty())
        //    {
        //        return nullptr;
        //    }

        //    int tail = m_TailIndex - 1;
        //    if(tail < 0)
        //    {
        //        tail = InternalSize - 1;
        //    }
        //    return &m_EntryList[tail];
        //}

        // 最後にエンキューされた要素のポインタを取得する。
        // キューが空の場合 nullptr が返る。
        const T* GetTail() const NN_NOEXCEPT
        {
            if(IsEmpty())
            {
                return nullptr;
            }

            int tail = m_TailIndex - 1;
            if(tail < 0)
            {
                tail = InternalSize - 1;
            }
            return &m_EntryList[tail];
        }

        // 全要素に対して func を呼び出す。
        // @tparam[in] void (*F)(T)
        template<typename F>
        void Foreach(F func) const NN_NOEXCEPT
        {
            if(IsEmpty())
            {
                return;
            }

            int pos = m_HeadIndex;
            while(pos != m_TailIndex)
            {
                func(m_EntryList[pos]);
                pos = (pos + 1) % InternalSize;
            }
        }

    private:
        T m_EntryList[InternalSize];
        int m_HeadIndex;
        int m_TailIndex;
    };

    template<typename T, int TCapacity>
    class TQueue<T, TCapacity, true>
        : private TQueue<T, TCapacity, false>
    {
    private:
        typedef TQueue<T, TCapacity, false> BaseType;

    public:
        static const int Capacity     = BaseType::Capacity;
        typedef typename BaseType::ValueType ValueType;

    protected:
        static const int InternalSize = BaseType::InternalSize;

    public:
        TQueue() NN_NOEXCEPT
            : BaseType()
            , m_Mutex()
            , m_EnqueueSemaphore()
            , m_DequeueSemaphore()
        {
        }

        void Initialize() NN_NOEXCEPT
        {
            BaseType::Initialize();
            nn::os::InitializeMutex(&m_Mutex, false, 0);
            nn::os::InitializeSemaphore(&m_EnqueueSemaphore, Capacity, Capacity);
            nn::os::InitializeSemaphore(&m_DequeueSemaphore,        0, Capacity);
        }

        void Finalize() NN_NOEXCEPT
        {
            nn::os::FinalizeSemaphore(&m_DequeueSemaphore);
            nn::os::FinalizeSemaphore(&m_EnqueueSemaphore);
            nn::os::FinalizeMutex(&m_Mutex);
            BaseType::Finalize();
        }

        bool IsEmpty() NN_NOEXCEPT
        {
            nn::os::LockMutex(&m_Mutex);
            NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };

            return BaseType::IsEmpty();
        }

        bool IsFull() NN_NOEXCEPT
        {
            nn::os::LockMutex(&m_Mutex);
            NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };

            return BaseType::IsFull();
        }

        // キューが空になるまでデキューを繰り返します。
        //
        // スレッド安全性：
        //   基本的に他のスレッドがエンキュー・デキュー操作を行わないことを保証した上で呼び出してください。
        //
        //   この関数の呼出中に他のスレッドがエンキューし続ける場合、この関数が終わらなかったり、
        //   戻った際にキューが空になっていることが保証されません。
        //
        //   他のスレッドが同時にデキューする場合、関数が戻った際にキューが完全に空になっていない場合があります。
        void Clear() NN_NOEXCEPT
        {
            while(nn::os::TryAcquireSemaphore(&m_DequeueSemaphore))
            {
                nn::os::LockMutex(&m_Mutex);
                NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };

                T v;
                NN_ABORT_UNLESS_RESULT_SUCCESS(BaseType::TryDequeue(&v));

                nn::os::ReleaseSemaphore(&m_EnqueueSemaphore);
            }
        }

        // キューが空くまでブロックします。
        void Enqueue(T entry) NN_NOEXCEPT
        {
            nn::os::AcquireSemaphore(&m_EnqueueSemaphore);
            nn::os::LockMutex(&m_Mutex);
            NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };

            NN_ABORT_UNLESS_RESULT_SUCCESS(BaseType::TryEnqueue(entry));

            nn::os::ReleaseSemaphore(&m_DequeueSemaphore);
        }

        // エンキューされるまでブロックします。
        void Dequeue(T* pOutEntry) NN_NOEXCEPT
        {
            nn::os::AcquireSemaphore(&m_DequeueSemaphore);
            nn::os::LockMutex(&m_Mutex);
            NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };

            NN_ABORT_UNLESS_RESULT_SUCCESS(BaseType::TryDequeue(pOutEntry));

            nn::os::ReleaseSemaphore(&m_EnqueueSemaphore);
        }

        // デキュー可能になった際にシグナルする MultiWaitHolder を初期化します。
        void InitializeDequeueableMultiWaitHolder(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pHolder);
            nn::os::InitializeMultiWaitHolder(pHolder, &m_DequeueSemaphore);
        }

        // デキュー可能になった際にシグナルする MultiWaitHolder を破棄します。
        void FinalizeDequeueableMultiWaitHolder(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pHolder);
            nn::os::FinalizeMultiWaitHolder(pHolder);
        }

    private:
        nn::os::MutexType m_Mutex;
        nn::os::SemaphoreType m_EnqueueSemaphore;
        nn::os::SemaphoreType m_DequeueSemaphore;
    };

}}}}
