﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "grcsrv_ParcelIo.h"

#include <nn/util/util_StringUtil.h>
#include <nn/vi/vi_Result.h>
#include "../grcsrvOffscreen_Config.h"

namespace nn{ namespace grcsrv{ namespace offscreen{ namespace native{

    namespace {

        void CloseParcelBuffer(
            android::Parcel* pParcel,
            const uint8_t* data,
            size_t /*dataSize*/,
            const binder_size_t* /*objects*/,
            size_t /*objectsSize*/,
            void* /*cookie*/
            )
        {
            if(data == NULL)
            {
                return;
            }
            if (pParcel != NULL)
            {
                pParcel->closeFileDescriptors();
            }
        }

        android::status_t InitializeParcel(android::Parcel* pParcel, size_t* pOutDataPosition, const uint8_t *src, size_t size) NN_NOEXCEPT
        {
            if(size < sizeof(hosbinder_transaction_t))
            {
                return android::BAD_VALUE;
            }

            hosbinder_transaction_t *tr = (hosbinder_transaction_t *) src;

            size_t required = sizeof(*tr) + tr->data_size + tr->obj_size;
            if (size < required)
            {
                return android::BAD_VALUE;
            }

            pParcel->ipcSetDataReference(
                src + tr->data_offset,
                tr->data_size,
                reinterpret_cast<const binder_size_t*>(src + tr->obj_offset),
                tr->obj_size / sizeof(binder_size_t),
                CloseParcelBuffer,
                nullptr
                );
            *pOutDataPosition = static_cast<size_t>(tr->data_offset);

            return android::NO_ERROR;
        }
        void FinalizeParcel(android::Parcel* pParcel) NN_NOEXCEPT
        {
            pParcel->freeData();
        }

        size_t SaveParcel(const android::Parcel* pParcel, uint8_t *dst, size_t space)
        {
            if (pParcel->errorCheck() != android::NO_ERROR)
                return 0;

            hosbinder_transaction_t *tr = (hosbinder_transaction_t *) dst;

            size_t required = sizeof(*tr)
                            + pParcel->ipcDataSize()
                            + pParcel->ipcObjectsCount() * sizeof(binder_size_t);
            if (space < required) {
                return 0;
            }

            memset(tr, 0, sizeof(*tr));
            tr->data_size   = pParcel->ipcDataSize();
            tr->data_offset = sizeof(*tr);
            tr->obj_size    = pParcel->ipcObjectsCount() * sizeof(binder_size_t);
            tr->obj_offset  = tr->data_offset + tr->data_size;

            memcpy(dst + tr->data_offset, (void *) pParcel->ipcData(),    tr->data_size);
            memcpy(dst + tr->obj_offset,  (void *) pParcel->ipcObjects(), tr->obj_size);

            size_t num_obj = tr->obj_size / sizeof(binder_size_t);
            binder_size_t *objs = (binder_size_t *) (dst + tr->obj_offset);
            for (size_t i=0; i<num_obj; ++i) {
                flat_binder_object *flat;
                flat = (flat_binder_object *) (dst + tr->data_offset + objs[i]);
                if (flat->type == BINDER_TYPE_BINDER ||
                    flat->type == BINDER_TYPE_WEAK_BINDER)
                {
                    return 0;
                }
            }

            return required;
        }

    }

    android::status_t ParcelIo::OpenParcel(android::Parcel* pOutParcel, size_t* pOutDataPosition, const void* pSrcData, size_t srcDataSize) NN_NOEXCEPT
    {
        return InitializeParcel(pOutParcel, pOutDataPosition, reinterpret_cast<const uint8_t*>(pSrcData), srcDataSize);
    }

    void ParcelIo::CloseParcel(android::Parcel* pParcel) NN_NOEXCEPT
    {
        FinalizeParcel(pParcel);
    }

    void ParcelIo::WriteParcel(size_t* pOutSize, void* pBuffer, size_t bufferSize, const android::Parcel* pParcel) NN_NOEXCEPT
    {
        *pOutSize = SaveParcel(pParcel, reinterpret_cast<uint8_t*>(pBuffer), bufferSize);
    }

    namespace
    {
        struct ExtractBinderHandleListArg
        {
            nn::Result result;
            int outCount;
            int32_t* buf;
            int length;
            size_t dataPosition;
        };

        void ExtractBinderHandleListImpl(flat_binder_object* pObject, void* userPtr) NN_NOEXCEPT
        {
            auto pArg = reinterpret_cast<ExtractBinderHandleListArg*>(userPtr);

            // 既に失敗していたら処理しない
            if(pArg->result.IsFailure())
            {
                return;
            }

            // 満杯の状態で更に Binder を見つけたらエラー
            if(pArg->outCount == pArg->length)
            {
                pArg->result = nn::vi::ResultResourceLimit();
                return;
            }

            pArg->buf[pArg->outCount] = pObject->handle;
            pArg->outCount++;
        }
    }

    nn::Result ParcelIo::ExtractBinderHandleList(int* pOutCount, int32_t* buf, int length, android::Parcel* pParcel) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutCount);
        NN_SDK_REQUIRES_NOT_NULL(buf);
        NN_SDK_REQUIRES_NOT_NULL(pParcel);

        ExtractBinderHandleListArg arg = {};
        arg.buf = buf;
        arg.length = length;

        pParcel->foreachObject(ExtractBinderHandleListImpl, &arg);

        NN_RESULT_DO(arg.result);
        *pOutCount = arg.outCount;
        NN_RESULT_SUCCESS;
    }

    namespace {

        struct ReplaceBinderServiceNameArg
        {
            char name[8];
        };

        void ReplaceBinderServiceNameImpl(flat_binder_object* pObject, void* userPtr) NN_NOEXCEPT
        {
            auto pArg = reinterpret_cast<ReplaceBinderServiceNameArg*>(userPtr);
            NN_GRCSRV_OFFSCRN_LOG_DEV("[parcel](h=%d)replacing '%s' -> '%s'\n", pObject->handle, pObject->service_name, pArg->name);
            std::memset(pObject->service_name, 0, sizeof(pObject->service_name));
            std::memcpy(pObject->service_name, pArg->name, sizeof(pArg->name));
        }

    }

    void ParcelIo::ReplaceBinderServiceName(android::Parcel* pParcel, const char* name) NN_NOEXCEPT
    {
        ReplaceBinderServiceNameArg arg = {};
        nn::util::Strlcpy(arg.name, name, sizeof(arg.name));
        pParcel->foreachObject(ReplaceBinderServiceNameImpl, &arg);
    }

    void ParcelIo::ReplaceBinderServiceName(void* buffer, size_t size, const char* name) NN_NOEXCEPT
    {
        android::Parcel parcel = {};
        size_t pos = 0;
        NN_ABORT_UNLESS_EQUAL(OpenParcel(&parcel, &pos, buffer, size), android::NO_ERROR);
        ReplaceBinderServiceName(&parcel, name);
        CloseParcel(&parcel);
    }

}}}}
