﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "grcsrvOffscreen_MultiWaitHandler.h"
#include <nn/grcsrv/grcsrv_Offscreen.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_Optional.h>

namespace nn{ namespace grcsrv{ namespace offscreen{

    namespace {
        static const int MultiWaitSlotCount = RequiredMultiWaitTagCount;

        uintptr_t g_TagBegin = 0;
        uintptr_t g_TagEnd = 0;
        void (*g_LinkMultiWaitFunction)(nn::os::MultiWaitHolderType*, void*) = nullptr;
        void (*g_UnlinkMultiWaitFunction)(nn::os::MultiWaitHolderType*, void*) = nullptr;
        void *g_MultiWaitLinkUserPtr = nullptr;

        struct MultiWaitEntry
        {
            nn::os::MultiWaitHolderType* pHolder;
            void (*onSignalFunction)(nn::os::MultiWaitHolderType*, bool, void*);
            void* userPtr;
        };

        nn::util::optional<MultiWaitEntry> g_EntryList[MultiWaitSlotCount];

    }

    void SetupMultiWait(
        uintptr_t multiWaitTagBegin,
        uintptr_t multiWaitTagCount,
        void (*linkMultiWait)(nn::os::MultiWaitHolderType* pHolder, void* userPtr),
        void (*unlinkMultiWait)(nn::os::MultiWaitHolderType* pHolder, void* userPtr),
        void* userPtr
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER_EQUAL(multiWaitTagCount, RequiredMultiWaitTagCount);
        g_TagBegin = multiWaitTagBegin;
        g_TagEnd = multiWaitTagBegin + MultiWaitSlotCount;
        g_LinkMultiWaitFunction = linkMultiWait;
        g_UnlinkMultiWaitFunction = unlinkMultiWait;
        g_MultiWaitLinkUserPtr = userPtr;
    }

    void NotifyMultiWaitSignal(nn::os::MultiWaitHolderType* pHolder, bool isUnlinked) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pHolder);
        uintptr_t tag = pHolder->userData;
        NN_SDK_REQUIRES_RANGE(tag, g_TagBegin, g_TagEnd);

        int index = static_cast<int>(tag - g_TagBegin);
        NN_SDK_ASSERT(g_EntryList[index] != nn::util::nullopt);
        NN_SDK_ASSERT_EQUAL(g_EntryList[index]->pHolder, pHolder);

        auto callback = g_EntryList[index]->onSignalFunction;
        auto userPtr = g_EntryList[index]->userPtr;

        if(isUnlinked)
        {
            g_EntryList[index] = nn::util::nullopt;
        }

        callback(pHolder, isUnlinked, userPtr);
    }

    void LinkMultiWait(
        nn::os::MultiWaitHolderType* pHolder,
        void (callback)(nn::os::MultiWaitHolderType*, bool, void*),
        void* userPtr
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(g_LinkMultiWaitFunction);

        int index = -1;
        for(int i = 0; i < MultiWaitSlotCount; i++)
        {
            if(g_EntryList[i] == nn::util::nullopt)
            {
                index = i;
                break;
            }
        }
        NN_ABORT_UNLESS_RANGE(index, 0, MultiWaitSlotCount);

        MultiWaitEntry e = {};
        e.pHolder = pHolder;
        e.onSignalFunction = callback;
        e.userPtr = userPtr;
        g_EntryList[index] = e;

        pHolder->userData = g_TagBegin + index;
        g_LinkMultiWaitFunction(pHolder, g_MultiWaitLinkUserPtr);
    }

    void UnlinkMultiWait(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
    {
        auto index = static_cast<int>(pHolder->userData - g_TagBegin);
        NN_SDK_REQUIRES_NOT_NULL(g_UnlinkMultiWaitFunction);
        NN_SDK_REQUIRES_RANGE(index, 0, MultiWaitSlotCount);
        NN_SDK_REQUIRES(g_EntryList[index] != nn::util::nullopt);
        NN_SDK_REQUIRES_EQUAL(g_EntryList[index]->pHolder, pHolder);

        g_UnlinkMultiWaitFunction(pHolder, g_MultiWaitLinkUserPtr);
        g_EntryList[index] = nn::util::nullopt;
    }

    //--------------------------------------------

    MultiWaitHandler::MultiWaitHandler() NN_NOEXCEPT
    {
        m_Callback = nullptr;
        m_UserPtr = nullptr;
    }

    nn::os::MultiWaitHolderType* MultiWaitHandler::GetHolder() NN_NOEXCEPT
    {
        return &m_Holder;
    }

    bool MultiWaitHandler::IsLinked() const NN_NOEXCEPT
    {
        return m_Callback != nullptr;
    }

    void MultiWaitHandler::Link(void (*onSignal)(MultiWaitHandler*, void*), void* userPtr) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(onSignal);
        if(onSignal == nullptr)
        {
            return;
        }

        if(m_Callback)
        {
            Unlink();
        }

        m_Callback = onSignal;
        m_UserPtr = userPtr;
        LinkMultiWait(&m_Holder, CallbackImpl, this);
    }

    void MultiWaitHandler::Unlink() NN_NOEXCEPT
    {
        if(!m_Callback)
        {
            return;
        }

        UnlinkMultiWait(&m_Holder);
        m_Callback = nullptr;
        m_UserPtr = nullptr;
    }

    void MultiWaitHandler::CallbackImpl(nn::os::MultiWaitHolderType* pHolder, bool isUnlinked, void* userPtr) NN_NOEXCEPT
    {
        auto pSelf = reinterpret_cast<MultiWaitHandler*>(userPtr);
        NN_SDK_ASSERT_EQUAL(pHolder, &pSelf->m_Holder);
        NN_SDK_ASSERT(pSelf->m_Callback);

        auto callback = pSelf->m_Callback;
        auto arg  = pSelf->m_UserPtr;

        if(isUnlinked)
        {
            pSelf->m_Callback = nullptr;
            pSelf->m_UserPtr = nullptr;
        }

        callback(pSelf, arg);
    }

}}}
