﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "grcsrvOffscreen_Layer.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/vi/vi_Result.h>
#include <nn/audio/audio_SampleFormat.h>
#include <nn/grc/grc_Result.h>
#include <nn/grc/grc_ResultPrivate.h>
#include "grcsrvOffscreen_Macro.h"
#include "grcsrvOffscreen_ResourceId.h"
#include "grcsrvOffscreen_MultiWaitHandler.h"
#include "grcsrvOffscreen_ConvertCapsrvError.h"

NN_PRAGMA_PUSH_WARNINGS
NN_GRCSRV_SUPPRESS_MOVIE_WARNINGS
#include <media/ICrypto.h>
#include <media/stagefright/MediaErrors.h>
#include <../services/nvnflinger/nvnflinger_nullalloc.h>
NN_PRAGMA_POP_WARNINGS

namespace nn{ namespace grcsrv{ namespace offscreen{

    Layer::Layer() NN_NOEXCEPT
        : m_State(LayerState::Idle)
        , m_Handle(0)
    {
    }

    nn::Result Layer::Initialize(
        uint64_t* pOutHandle,
        nn::applet::AppletResourceUserId rendererAruid,
        nn::capsrv::capture::CaptureModule* pCaptureModule
    ) NN_NOEXCEPT
    {
        NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("initializing layer...\n");
        NN_GRCSRV_PROCESS_START();

        uint64_t handle = GetNewResourceId();

        NN_RESULT_DO(nn::os::CreateSystemEvent(&m_FinishReadyEvent, nn::os::EventClearMode_ManualClear, true));
        NN_GRCSRV_PROCESS_ROLLBACK(nn::os::DestroySystemEvent(&m_FinishReadyEvent));

        m_VideoRendererToVideoCaptureQueue.Initialize();
        NN_GRCSRV_PROCESS_ROLLBACK(m_VideoRendererToVideoCaptureQueue.Finalize());
        m_VideoCaptureToVideoRendererQueue.Initialize();
        NN_GRCSRV_PROCESS_ROLLBACK(m_VideoCaptureToVideoRendererQueue.Finalize());
        m_VideoEncoderToVideoCaptureQueue.Initialize();
        NN_GRCSRV_PROCESS_ROLLBACK(m_VideoEncoderToVideoCaptureQueue.Finalize());
        m_VideoCaptureToVideoEncoderQueue.Initialize();
        NN_GRCSRV_PROCESS_ROLLBACK(m_VideoCaptureToVideoEncoderQueue.Finalize());
        m_AudioRendererToAudioEncoderQueue.Initialize();
        NN_GRCSRV_PROCESS_ROLLBACK(m_AudioRendererToAudioEncoderQueue.Finalize());
        m_AudioEncoderToAudioRendererQueue.Initialize();
        NN_GRCSRV_PROCESS_ROLLBACK(m_AudioEncoderToAudioRendererQueue.Finalize());

        NN_RESULT_DO(InitializeRendererObject(rendererAruid));
        NN_GRCSRV_PROCESS_ROLLBACK(FinalizeRendererObject());

        NN_RESULT_DO(m_VideoRenderer.Initialize(
            m_Renderer.m_pConsumer,
            &m_Renderer.m_AcquirableEvent,
            &m_VideoCaptureToVideoRendererQueue,
            &m_VideoRendererToVideoCaptureQueue
        ));
        NN_GRCSRV_PROCESS_ROLLBACK(m_VideoRenderer.Finalize());

        NN_RESULT_DO(m_VideoCapture.Initialize(
            pCaptureModule,
            &m_VideoRendererToVideoCaptureQueue,
            &m_VideoCaptureToVideoRendererQueue,
            &m_VideoEncoderToVideoCaptureQueue,
            &m_VideoCaptureToVideoEncoderQueue
        ));
        NN_GRCSRV_PROCESS_ROLLBACK(m_VideoCapture.Finalize());

        NN_RESULT_DO(m_VideoEncoder.Initialize());
        NN_GRCSRV_PROCESS_ROLLBACK(m_VideoEncoder.Finalize());

        NN_RESULT_DO(m_AudioRenderer.Initialize(
            &m_AudioEncoderToAudioRendererQueue,
            &m_AudioRendererToAudioEncoderQueue
        ));
        NN_GRCSRV_PROCESS_ROLLBACK(m_AudioRenderer.Finalize());

        NN_RESULT_DO(m_AudioEncoder.Initialize());
        NN_GRCSRV_PROCESS_ROLLBACK(m_AudioEncoder.Finalize());

        NN_RESULT_DO(m_Muxer.Initialize());
        NN_GRCSRV_PROCESS_ROLLBACK(m_Muxer.Finalize());

        m_EncodingBufferList.Initialize();
        NN_GRCSRV_PROCESS_ROLLBACK(m_EncodingBufferList.Finalize());

        NN_GRCSRV_PROCESS_SUCCESS();
        NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("initialized layer\n");
        m_State = LayerState::Idle;
        m_Handle = handle;
        m_ErrorResult = nn::util::nullopt;
        m_FileHandle = {};
        m_Parameter = {};
        m_pVideoEncodeThread = nullptr;
        m_pVideoMuxerThread  = nullptr;
        m_pAudioEncodeThread = nullptr;
        m_pAudioMuxerThread  = nullptr;
        *pOutHandle = handle;
        NN_RESULT_SUCCESS;
    }

    void Layer::Finalize() NN_NOEXCEPT
    {
        NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("finalizing layer...\n");
        if(IsStarted())
        {
            Discard();
        }
        if(IsRendererBound())
        {
            UnbindRenderer();
        }

        m_EncodingBufferList.Finalize();
        m_Muxer.Finalize();
        m_AudioEncoder.Finalize();
        m_AudioRenderer.Finalize();
        m_VideoEncoder.Finalize();
        m_VideoCapture.Finalize();
        m_VideoRenderer.Finalize();
        FinalizeRendererObject();

        m_AudioEncoderToAudioRendererQueue.Finalize();
        m_AudioRendererToAudioEncoderQueue.Finalize();
        m_VideoCaptureToVideoEncoderQueue.Finalize();
        m_VideoEncoderToVideoCaptureQueue.Finalize();
        m_VideoCaptureToVideoRendererQueue.Finalize();
        m_VideoRendererToVideoCaptureQueue.Finalize();

        nn::os::DestroySystemEvent(&m_FinishReadyEvent);

        NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("finalized layer\n");
        m_Handle = {};
    }

    nn::Result Layer::InitializeRendererObject(nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
    {
        auto& obj = m_Renderer;

        NN_GRCSRV_PROCESS_START();
        android::sp<android::IGraphicBufferAlloc> pBufferAllocator(new NULLGraphicBufferAlloc());
        NN_RESULT_THROW_UNLESS(pBufferAllocator != nullptr, nn::grc::ResultOutOfMemory());

        // Renderer オブジェクトを作成
        android::BufferQueue::createBufferQueue(&obj.m_pProducer, &obj.m_pConsumer, pBufferAllocator);
        NN_RESULT_THROW_UNLESS(obj.m_pProducer != nullptr, nn::grc::ResultOutOfMemory());
        NN_RESULT_THROW_UNLESS(obj.m_pConsumer != nullptr, nn::grc::ResultOutOfMemory());
        NN_GRCSRV_PROCESS_ROLLBACK(obj.m_pProducer.clear());
        NN_GRCSRV_PROCESS_ROLLBACK(obj.m_pConsumer.clear());

        // connect to consumer
        {
            android::sp<android::BufferQueue::ProxyConsumerListener> pConsumerListener(new android::BufferQueue::ProxyConsumerListener(nullptr));
            NN_RESULT_THROW_UNLESS(pConsumerListener != nullptr, nn::grc::ResultOutOfMemory());
            NN_RESULT_THROW_UNLESS(obj.m_pConsumer->consumerConnect(pConsumerListener, true) == android::NO_ERROR, nn::grc::ResultInvalidState());
            obj.m_pConsumer->setConsumerUsageBits(android::GraphicBuffer::USAGE_HW_COMPOSER);
        }

        // setup acquirable event
        {
            nn::os::NativeHandle h = {};
            bool isManaged = false;
            NN_RESULT_THROW_UNLESS(
                obj.m_pConsumer->getFrameAvailableEventHandle(h, isManaged) == android::NO_ERROR,
                nn::grc::ResultInvalidState()
            );
            nn::os::AttachReadableHandleToSystemEvent(&obj.m_AcquirableEvent, h, isManaged, nn::os::EventClearMode_ManualClear);
        }
        NN_GRCSRV_PROCESS_ROLLBACK(nn::os::DestroySystemEvent(&obj.m_AcquirableEvent));

        NN_GRCSRV_PROCESS_SUCCESS();
        obj.m_Aruid = aruid;
        obj.m_IsBound = false;
        NN_RESULT_SUCCESS;
    }

    void Layer::FinalizeRendererObject() NN_NOEXCEPT
    {
        auto& obj = m_Renderer;

        nn::os::DestroySystemEvent(&obj.m_AcquirableEvent);
        obj.m_pProducer.clear();
        obj.m_pConsumer.clear();
        obj.m_Aruid = {};
        obj.m_IsBound = false;
    }

    nn::Result Layer::InitializeVideoBufferObject(int count, android::sp<android::GraphicBuffer>* list) NN_NOEXCEPT
    {
        auto& obj = m_VideoBuffer;
        obj.m_Count = count;
        obj.m_List = list;
        NN_RESULT_SUCCESS;
    }

    void Layer::FinalizeVideoBufferObject() NN_NOEXCEPT
    {
        auto& obj = m_VideoBuffer;
        obj.m_Count = 0;
        obj.m_List = nullptr;
    }

    //------------------------------

    uint64_t Layer::GetHandle() const NN_NOEXCEPT
    {
        return m_Handle;
    }

    nn::Result Layer::GetErrorResult() const NN_NOEXCEPT
    {
        if(m_ErrorResult == nn::util::nullopt)
        {
            // エラーなし
            NN_RESULT_SUCCESS;
        }

        // ErrorResult は成功ではないはず
        NN_GRCSRV_ASSERT_THROW(m_ErrorResult->IsFailure(), nn::grc::ResultInternalOffscreenLayerError());
        return *m_ErrorResult;
    }

    nn::fs::FileHandle Layer::GetFileHandle() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsStarted());
        return m_FileHandle;
    }

    nn::applet::AppletResourceUserId Layer::GetRendererAruid() const NN_NOEXCEPT
    {
        return m_Renderer.m_Aruid;
    }

    bool Layer::IsInitialized() const NN_NOEXCEPT
    {
        return m_Handle != 0;
    }

    bool Layer::IsRendererBound() const NN_NOEXCEPT
    {
        return m_Renderer.m_IsBound;
    }

    bool Layer::IsStarted() const NN_NOEXCEPT
    {
        return m_State != LayerState::Idle;
    }

    int Layer::GetVideoFrameCount() const NN_NOEXCEPT
    {
        return m_VideoCapture.GetCapturedFrameCount();
    }

    int Layer::GetLastCaptureBufferIndex() const NN_NOEXCEPT
    {
        return m_VideoCapture.GetLastDestinationBufferIndex();
    }

    nn::grc::OffscreenRecordingParameter Layer::GetRunningParameter() const NN_NOEXCEPT
    {
        return m_Parameter;
    }

    android::sp<android::IGraphicBufferProducer> Layer::GetRendererProducer() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_Renderer.m_IsBound);
        return m_Renderer.m_pProducer;
    }

    void Layer::GetFinishReadySystemEventHandle(nn::sf::NativeHandle& outHandle) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        outHandle = nn::sf::NativeHandle(nn::os::GetReadableHandleOfSystemEvent(&m_FinishReadyEvent), false);
    }

    void Layer::GetAudioEncodeReadySystemEventHandle(nn::sf::NativeHandle& outHandle) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        return m_AudioRenderer.GetEncodeReadyEvent(outHandle);
    }

    //-----------------------------

    nn::Result Layer::BindRenderer(nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(IsInitialized(), nn::grc::ResultInvalidCall());
        NN_RESULT_THROW_UNLESS(!m_Renderer.m_IsBound, nn::grc::ResultInvalidCall());
        NN_RESULT_THROW_UNLESS(m_Renderer.m_Aruid == aruid, nn::grc::ResultInvalidCall());

        NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("renderer bound\n");
        m_Renderer.m_IsBound = true;
        NN_RESULT_SUCCESS;
    }

    void Layer::UnbindRenderer() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(IsInitialized());
        if(IsStarted())
        {
            Discard();
        }

        NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("renderer unbound\n");
        m_Renderer.m_IsBound = false;
    }

    //------------------------------

    nn::Result Layer::Start(
        nn::fs::FileHandle hFile,
        const nn::grc::OffscreenRecordingParameter& param,
        int videoBufferCount,
        android::sp<android::GraphicBuffer>* videoBufferList,
        nn::os::ThreadType* pVideoEncodeThread,
        nn::os::ThreadType* pVideoMuxerThread,
        nn::os::ThreadType* pAudioEncodeThread,
        nn::os::ThreadType* pAudioMuxerThread
    ) NN_NOEXCEPT
    {
        NN_GRCSRV_PROCESS_START();
        NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("layer starting file...\n");
        NN_GRCSRV_PROCESS_ROLLBACK( NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("layer failed to start file\n") );

        NN_RESULT_THROW_UNLESS(IsInitialized(), nn::grc::ResultInvalidCall());
        NN_RESULT_THROW_UNLESS(!IsStarted(), nn::grc::ResultInvalidCall());
        NN_RESULT_THROW_UNLESS(IsRendererBound(), nn::grc::ResultInvalidCall());
        m_EncodingBufferList.Clear();
        m_VideoRendererToVideoCaptureQueue.Clear();
        m_VideoCaptureToVideoRendererQueue.Clear();
        m_VideoCaptureToVideoEncoderQueue.Clear();
        m_VideoEncoderToVideoCaptureQueue.Clear();
        m_AudioRendererToAudioEncoderQueue.Clear();
        m_AudioEncoderToAudioRendererQueue.Clear();

        NN_RESULT_DO(InitializeVideoBufferObject(videoBufferCount, videoBufferList));
        NN_GRCSRV_PROCESS_ROLLBACK(FinalizeVideoBufferObject());

        for(int i = 0; i < videoBufferCount; i++)
        {
            detail::VideoEncoderToVideoCaptureValue e = {};
            e.index = i;
            e.buffer = videoBufferList[i];
            m_VideoEncoderToVideoCaptureQueue.Enqueue(e);
        }

        NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("  starting muxer\n");
        NN_RESULT_DO(m_Muxer.Start(hFile));
        NN_GRCSRV_PROCESS_ROLLBACK(m_Muxer.Stop());

        NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("  starting audio-encoder\n");
        NN_RESULT_DO(m_AudioEncoder.Start(param));
        NN_GRCSRV_PROCESS_ROLLBACK(m_AudioEncoder.Stop());

        NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("  starting audio-renderer\n");
        NN_RESULT_DO(m_AudioRenderer.Start(param));
        NN_GRCSRV_PROCESS_ROLLBACK(m_AudioRenderer.Abort());

        NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("  starting video-encoder\n");
        NN_RESULT_DO(m_VideoEncoder.Start(param));
        NN_GRCSRV_PROCESS_ROLLBACK(m_VideoEncoder.Stop());

        NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("  starting video-capture\n");
        NN_RESULT_DO(m_VideoCapture.Start());
        NN_GRCSRV_PROCESS_ROLLBACK(m_VideoCapture.Abort());

        NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("  starting video-renderer\n");
        NN_RESULT_DO(m_VideoRenderer.Start(param.videoFrameRate));
        NN_GRCSRV_PROCESS_ROLLBACK(m_VideoRenderer.Abort());

        m_Muxer.SetVideoOrientation(param.videoImageOrientation);

        NN_GRCSRV_PROCESS_SUCCESS(); // 以降失敗しない

        // 終了待ちを初期化
        m_VideoRenderer.InitializeExitWaitHolder(m_VideoRendererExitWaitHandler.GetHolder());
        m_VideoCapture.InitializeExitWaitHolder(m_VideoCaptureExitWaitHandler.GetHolder());
        m_AudioRenderer.InitializeExitWaitHolder(m_AudioRendererExitWaitHandler.GetHolder());
        nn::os::InitializeMultiWaitHolder(m_VideoEncodeThreadExitWaitHandler.GetHolder(), pVideoEncodeThread);
        nn::os::InitializeMultiWaitHolder(m_VideoMuxerThreadExitWaitHandler .GetHolder(), pVideoMuxerThread );
        nn::os::InitializeMultiWaitHolder(m_AudioEncodeThreadExitWaitHandler.GetHolder(), pAudioEncodeThread);
        nn::os::InitializeMultiWaitHolder(m_AudioMuxerThreadExitWaitHandler .GetHolder(), pAudioMuxerThread );

        // エラー監視を設定
        LinkExitObserverAsErrorDetectorImpl();

        // ワーカースレッドを開始。
        // 必ず各 Handler の Start() を呼び出してからスレッドを開始すること。
        NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("  starting worker threads\n");
        m_VideoEncodeThreadResult = nn::util::nullopt;
        m_VideoMuxerThreadResult  = nn::util::nullopt;
        m_AudioEncodeThreadResult = nn::util::nullopt;
        m_AudioMuxerThreadResult  = nn::util::nullopt;
        nn::os::StartThread(pVideoEncodeThread);
        nn::os::StartThread(pVideoMuxerThread);
        nn::os::StartThread(pAudioEncodeThread);
        nn::os::StartThread(pAudioMuxerThread);

        NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("layer started file\n");
        m_State = LayerState::Running;
        m_ErrorResult = nn::util::nullopt;
        m_FileHandle = hFile;
        m_Parameter = param;
        m_pVideoEncodeThread = pVideoEncodeThread;
        m_pVideoMuxerThread  = pVideoMuxerThread;
        m_pAudioEncodeThread = pAudioEncodeThread;
        m_pAudioMuxerThread  = pAudioMuxerThread;
        NN_RESULT_SUCCESS;
    }

    void Layer::Discard() NN_NOEXCEPT
    {
        NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("layer discarding file...\n");
        NN_GRCSRV_ASSERT_ABORT(IsInitialized());
        NN_GRCSRV_ASSERT_ABORT(IsStarted());

        // エラー監視を解除
        UnlinkExitObserverImpl();

        // abort 要求をエンキュー
        NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("  queueing abort request to video encoder\n");
        {
            detail::VideoCaptureToVideoEncoderValue e = {};
            e.isAbortRequested = true;
            m_VideoCaptureToVideoEncoderQueue.Enqueue(e);
        }
        NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("  queueing abort request to audio encoder\n");
        {
            detail::AudioRendererToAudioEncoderValue e = {};
            e.isAbortRequested = true;
            m_AudioRendererToAudioEncoderQueue.Enqueue(e);
        }

        NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("  aborting video-renderer\n");
        m_VideoRenderer.Abort();
        NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("  aborting video-capture\n");
        m_VideoCapture.Abort();
        NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("  aborting video-encoder\n");
        m_VideoEncoder.Stop();
        NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("  aborting audio-renderer\n");
        m_AudioRenderer.Abort();
        NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("  aborting audio-encoder\n");
        m_AudioEncoder.Stop();
        NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("  aborting muxer\n");
        m_Muxer.Stop();

        NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("  waiting for video-encode-thread\n");
        nn::os::WaitThread(m_pVideoEncodeThread);
        NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("  waiting for video-muxer-thread\n");
        nn::os::WaitThread(m_pVideoMuxerThread);
        NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("  waiting for audio-encode-thread\n");
        nn::os::WaitThread(m_pAudioEncodeThread);
        NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("  waiting for audio-muxer-thread\n");
        nn::os::WaitThread(m_pAudioMuxerThread);

        CleanupAndChangeStateIdleImpl();
        NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("layer discarded file\n");
    }

    nn::Result Layer::RequestFinishReady() NN_NOEXCEPT
    {
        if(m_State == LayerState::Error)
        {
            return GetErrorResult();
        }
        NN_RESULT_THROW_UNLESS(m_State == LayerState::Running, nn::grc::ResultInvalidCall());
        NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("layer requesting finish...\n");

        nn::os::ClearSystemEvent(&m_FinishReadyEvent);

        NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("  requesting finish to video-capture\n");
        m_VideoRenderer.RequestFinish();
        m_AudioRenderer.RequestFinish();

        // 完了待ちのためのコールバックを再設定
        LinkExitObserverAsFinishWaiterImpl();

        NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("layer requested finish...\n");
        m_State = LayerState::Finishing;
        NN_RESULT_SUCCESS;
    }

    nn::Result Layer::CheckFinishReady() const NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_State == LayerState::FinishReady, nn::grc::ResultInvalidCall());
        NN_RESULT_SUCCESS;
    }

    nn::Result Layer::Finish() NN_NOEXCEPT
    {
        if(m_State == LayerState::Error)
        {
            return GetErrorResult();
        }
        NN_RESULT_THROW_UNLESS(m_State == LayerState::FinishReady, nn::grc::ResultInvalidCall());
        NN_SDK_ASSERT(!m_VideoRendererExitWaitHandler.IsLinked());
        NN_SDK_ASSERT(!m_VideoCaptureExitWaitHandler.IsLinked());
        NN_SDK_ASSERT(!m_AudioRendererExitWaitHandler.IsLinked());
        NN_SDK_ASSERT(!m_VideoEncodeThreadExitWaitHandler.IsLinked());
        NN_SDK_ASSERT(!m_VideoMuxerThreadExitWaitHandler.IsLinked());
        NN_SDK_ASSERT(!m_AudioEncodeThreadExitWaitHandler.IsLinked());
        NN_SDK_ASSERT(!m_AudioMuxerThreadExitWaitHandler.IsLinked());

        NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("layer finishing file\n");
        // 停止処理
        m_VideoRenderer.Reset();
        m_VideoCapture.Reset();
        m_VideoEncoder.Stop();
        m_AudioRenderer.Reset();
        m_AudioEncoder.Stop();
        m_Muxer.Stop();

        CleanupAndChangeStateIdleImpl();
        NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("layer finished file\n");
        NN_RESULT_SUCCESS;
    }

    void Layer::CleanupAndChangeStateIdleImpl() NN_NOEXCEPT
    {
        NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("layer cleanup\n");
        m_EncodingBufferList.Clear();
        m_VideoRendererToVideoCaptureQueue.Clear();
        m_VideoCaptureToVideoRendererQueue.Clear();
        m_VideoCaptureToVideoEncoderQueue.Clear();
        m_VideoEncoderToVideoCaptureQueue.Clear();
        m_AudioRendererToAudioEncoderQueue.Clear();
        m_AudioEncoderToAudioRendererQueue.Clear();
        FinalizeVideoBufferObject();

        // Idle に戻るときに破棄
        m_VideoRenderer.FinalizeExitWaitHolder(m_VideoRendererExitWaitHandler.GetHolder());
        m_VideoCapture.FinalizeExitWaitHolder(m_VideoCaptureExitWaitHandler.GetHolder());
        m_AudioRenderer.FinalizeExitWaitHolder(m_AudioRendererExitWaitHandler.GetHolder());
        nn::os::FinalizeMultiWaitHolder(m_VideoEncodeThreadExitWaitHandler.GetHolder());
        nn::os::FinalizeMultiWaitHolder(m_VideoMuxerThreadExitWaitHandler.GetHolder());
        nn::os::FinalizeMultiWaitHolder(m_AudioEncodeThreadExitWaitHandler.GetHolder());
        nn::os::FinalizeMultiWaitHolder(m_AudioMuxerThreadExitWaitHandler.GetHolder());

        m_ErrorResult = nn::util::nullopt;
        m_FileHandle = {};
        m_Parameter = {};
        m_pVideoEncodeThread = nullptr;
        m_pVideoMuxerThread  = nullptr;
        m_pAudioEncodeThread = nullptr;
        m_pAudioMuxerThread  = nullptr;
        m_VideoEncodeThreadResult = nn::util::nullopt;
        m_VideoMuxerThreadResult  = nn::util::nullopt;
        m_AudioEncodeThreadResult = nn::util::nullopt;
        m_AudioMuxerThreadResult  = nn::util::nullopt;

        m_State = LayerState::Idle;
    }

    //---------------------------------------

    nn::Result Layer::EncodeAudioSample(size_t* pOutSize, const void* buffer, size_t size) NN_NOEXCEPT
    {
        return m_AudioRenderer.EncodeAudio(pOutSize, buffer, size);
    }

    //---------------------------------------

    void Layer::UnlinkExitObserverImpl() NN_NOEXCEPT
    {
        NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("unlinking exit observer\n");
        m_VideoRendererExitWaitHandler.Unlink();
        m_VideoCaptureExitWaitHandler.Unlink();
        m_AudioRendererExitWaitHandler.Unlink();
        m_VideoEncodeThreadExitWaitHandler.Unlink();
        m_VideoMuxerThreadExitWaitHandler.Unlink();
        m_AudioEncodeThreadExitWaitHandler.Unlink();
        m_AudioMuxerThreadExitWaitHandler.Unlink();
    }

    void Layer::ErrorDetectorExitWaiterImpl(nn::util::optional<nn::Result> exitResult) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(exitResult != nn::util::nullopt);
        m_State       = LayerState::Error;
        m_ErrorResult = exitResult;
        m_VideoRenderer.NotifyError();
        m_AudioRenderer.NotifyError();
    }

    void Layer::LinkExitObserverAsErrorDetectorImpl() NN_NOEXCEPT
    {
        NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("linking exit observer as error detector\n");
        // 各サブモジュールでエラーが起こった際に以下を行う。
        //
        // - 発生したエラーを Layer のエラーとして記録
        // - Layer を LayerState::Error に変更
        // - VideoRenderer/AudioRenderer にエラーを通知してアプリのバッファを空回りさせる。
        //   VideoRenderer/AudioRenderer 自身のエラーを通知した場合無視される。
        //
        // 1 つめのエラーを検出した時点で ExitObserver はすべて Unlink されるため
        // 2 つめ以降のエラーは無視される。

        m_VideoRendererExitWaitHandler.Link(
            [](MultiWaitHandler*, void* p)->void
            {
                NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("error: video-renderer exit\n");
                auto pSelf = reinterpret_cast<Layer*>(p);
                pSelf->UnlinkExitObserverImpl();
                pSelf->ErrorDetectorExitWaiterImpl(pSelf->m_VideoRenderer.GetExitResult());
            },
            this
        );

        m_VideoCaptureExitWaitHandler.Link(
            [](MultiWaitHandler*, void* p)->void
            {
                NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("error: video-capture exit\n");
                auto pSelf = reinterpret_cast<Layer*>(p);
                pSelf->UnlinkExitObserverImpl();
                pSelf->ErrorDetectorExitWaiterImpl(pSelf->m_VideoCapture.GetExitResult());
            },
            this
        );

        m_AudioRendererExitWaitHandler.Link(
            [](MultiWaitHandler*, void* p)->void
            {
                NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("error: audio-renderer exit\n");
                auto pSelf = reinterpret_cast<Layer*>(p);
                pSelf->UnlinkExitObserverImpl();
                pSelf->ErrorDetectorExitWaiterImpl(pSelf->m_AudioRenderer.GetExitResult());
            },
            this
        );

        m_VideoEncodeThreadExitWaitHandler.Link(
            [](MultiWaitHandler*, void* p)->void
            {
                NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("error: video-encode-thread exit\n");
                auto pSelf = reinterpret_cast<Layer*>(p);
                pSelf->UnlinkExitObserverImpl();
                pSelf->ErrorDetectorExitWaiterImpl(pSelf->m_VideoEncodeThreadResult);
            },
            this
        );

        m_VideoMuxerThreadExitWaitHandler.Link(
            [](MultiWaitHandler* pHandler, void* p)->void
            {
                NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("error: video-muxer-thread exit\n");
                auto pSelf = reinterpret_cast<Layer*>(p);
                pSelf->UnlinkExitObserverImpl();
                pSelf->ErrorDetectorExitWaiterImpl(pSelf->m_VideoMuxerThreadResult);
            },
            this
        );

        m_AudioEncodeThreadExitWaitHandler.Link(
            [](MultiWaitHandler* pHandler, void* p)->void
            {
                NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("error: audio-encode-thread exit\n");
                auto pSelf = reinterpret_cast<Layer*>(p);
                pSelf->UnlinkExitObserverImpl();
                pSelf->ErrorDetectorExitWaiterImpl(pSelf->m_AudioEncodeThreadResult);
            },
            this
        );

        m_AudioMuxerThreadExitWaitHandler.Link(
            [](MultiWaitHandler* pHandler, void* p)->void
            {
                NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("error: audio-muxer-thread exit\n");
                auto pSelf = reinterpret_cast<Layer*>(p);
                pSelf->UnlinkExitObserverImpl();
                pSelf->ErrorDetectorExitWaiterImpl(pSelf->m_AudioMuxerThreadResult);
            },
            this
        );
    }

    void Layer::LinkExitObserverAsFinishWaiterImpl() NN_NOEXCEPT
    {
        NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("linking exit observer as finish waiter\n");

        m_VideoRendererExitWaitHandler.Link(
            [](MultiWaitHandler* pHandler, void* p)->void
            {
                NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("video-renderer exit\n");
                pHandler->Unlink();
                auto pSelf = reinterpret_cast<Layer*>(p);
                pSelf->ExitObserverFinishWaiterImpl(pSelf->m_VideoRenderer.GetExitResult());
            },
            this
        );

        m_VideoCaptureExitWaitHandler.Link(
            [](MultiWaitHandler* pHandler, void* p)->void
            {
                NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("video-capture exit\n");
                pHandler->Unlink();
                auto pSelf = reinterpret_cast<Layer*>(p);
                pSelf->ExitObserverFinishWaiterImpl(pSelf->m_VideoCapture.GetExitResult());
            },
            this
        );

        m_AudioRendererExitWaitHandler.Link(
            [](MultiWaitHandler* pHandler, void* p)->void
            {
                NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("audio-renderer exit\n");
                pHandler->Unlink();
                auto pSelf = reinterpret_cast<Layer*>(p);
                pSelf->ExitObserverFinishWaiterImpl(pSelf->m_AudioRenderer.GetExitResult());
            },
            this
        );

        m_VideoEncodeThreadExitWaitHandler.Link(
            [](MultiWaitHandler* pHandler, void* p)->void
            {
                NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("video-encode-thread exit\n");
                pHandler->Unlink();
                auto pSelf = reinterpret_cast<Layer*>(p);
                pSelf->ExitObserverFinishWaiterImpl(pSelf->m_VideoEncodeThreadResult);
            },
            this
        );

        m_VideoMuxerThreadExitWaitHandler.Link(
            [](MultiWaitHandler* pHandler, void* p)->void
            {
                NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("video-muxer-thread exit\n");
                pHandler->Unlink();
                auto pSelf = reinterpret_cast<Layer*>(p);
                pSelf->ExitObserverFinishWaiterImpl(pSelf->m_VideoMuxerThreadResult);
            },
            this
        );

        m_AudioEncodeThreadExitWaitHandler.Link(
            [](MultiWaitHandler* pHandler, void* p)->void
            {
                NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("audio-encode-thread exit\n");
                pHandler->Unlink();
                auto pSelf = reinterpret_cast<Layer*>(p);
                pSelf->ExitObserverFinishWaiterImpl(pSelf->m_AudioEncodeThreadResult);
            },
            this
        );

        m_AudioMuxerThreadExitWaitHandler.Link(
            [](MultiWaitHandler* pHandler, void* p)->void
            {
                NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("audio-muxer-thread exit\n");
                pHandler->Unlink();
                auto pSelf = reinterpret_cast<Layer*>(p);
                pSelf->ExitObserverFinishWaiterImpl(pSelf->m_AudioMuxerThreadResult);
            },
            this
        );
    }

    void Layer::ExitObserverFinishWaiterImpl(nn::util::optional<nn::Result> exitResult) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(exitResult != nn::util::nullopt);
        if(exitResult->IsSuccess())
        {
            CheckFinishReadyCallbackImpl();
        }
        else
        {
            UnlinkExitObserverImpl();
            m_State = LayerState::Error;
            m_ErrorResult = exitResult;
            m_VideoRenderer.NotifyError();
            nn::os::SignalSystemEvent(&m_FinishReadyEvent);
        }
    }
    void Layer::CheckFinishReadyCallbackImpl() NN_NOEXCEPT
    {
        NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("checking finish ready...\n");
        bool isComplete = true;

        if(m_VideoRendererExitWaitHandler.IsLinked())
        {
            NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("  video-renderer still running\n");
            isComplete = false;
        }

        if(m_VideoCaptureExitWaitHandler.IsLinked())
        {
            NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("  video-capture still running\n");
            isComplete = false;
        }

        if(m_AudioRendererExitWaitHandler.IsLinked())
        {
            NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("  audio-renderer still running\n");
            isComplete = false;
        }

        if(m_VideoEncodeThreadExitWaitHandler.IsLinked())
        {
            NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("  video-ecode-thread still running\n");
            isComplete = false;
        }

        if(m_VideoMuxerThreadExitWaitHandler.IsLinked())
        {
            NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("  video-muxer-thread still running\n");
            isComplete = false;
        }

        if(m_AudioEncodeThreadExitWaitHandler.IsLinked())
        {
            NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("  audio-ecode-thread still running\n");
            isComplete = false;
        }

        if(m_AudioMuxerThreadExitWaitHandler.IsLinked())
        {
            NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("  audio-muxer-thread still running\n");
            isComplete = false;
        }

        if(!isComplete)
        {
            NN_GRCSRV_OFFSCRN_LOG_DEV_LAYER("checked finish ready. not ready.\n");
            // wait for next signal
            return;
        }

        m_State = LayerState::FinishReady;
        nn::os::SignalSystemEvent(&m_FinishReadyEvent);
    }

    //---------------------------------------

#define NN_GRCSRV_VENCT_CHECK_RESULT(result, message)   \
    {                                                   \
        nn::Result r = (result);                        \
        if(r.IsFailure())                               \
        {                                               \
            NN_GRCSRV_OFFSCRN_LOG_DEV_VENCT_ERR(message "(result=%d-%d)\n", r.GetModule(), r.GetDescription());   \
            exitResult = nn::grc::ResultInternalOffscreenVideoEncoderError();   \
            break;                                      \
        }                                               \
    }

#define NN_GRCSRV_VENCT_CHECK_STATUS(status, message)   \
    {                                                   \
        int s = (status);                               \
        if(s != android::NO_ERROR)                      \
        {                                               \
            NN_GRCSRV_OFFSCRN_LOG_DEV_VENCT_ERR(message "(status=%d)\n", s);   \
            exitResult = nn::grc::ResultInternalOffscreenVideoEncoderError();   \
            break;                                      \
        }                                               \
    }

    void Layer::VideoEncodeThreadFunction(void* arg) NN_NOEXCEPT
    {
        auto pLayer = reinterpret_cast<Layer*>(arg);
        pLayer->VideoEncodeThreadFunctionImpl();
    }

    void Layer::VideoEncodeThreadFunctionImpl() NN_NOEXCEPT
    {
        // 終了条件：
        //   ・captureBuffer から停止要求が来る
        //   ・エラーが発生する
        // Abort による中断の場合、別スレッドにより VideoEncoder が停止され、各種関数がエラー（状態が不正）を返すため、終了する。
        NN_GRCSRV_OFFSCRN_LOG_DEV_VENCT("start\n");

        nn::Result exitResult;

        for(;;)
        {
            nn::Result result = nn::ResultSuccess();

            detail::VideoCaptureToVideoEncoderValue capturedBuffer = {};
            NN_GRCSRV_OFFSCRN_LOG_DEV_VENCT("waiting for video-capture...\n");
            m_VideoCaptureToVideoEncoderQueue.Dequeue(&capturedBuffer);
            NN_GRCSRV_OFFSCRN_LOG_DEV_VENCT("acquired from video-capture\n");

            if(capturedBuffer.isAbortRequested)
            {
                NN_GRCSRV_OFFSCRN_LOG_DEV_VENCT_ERR("abort requested\n");
                exitResult = nn::grc::ResultInternalOffscreenAbortRequested();
                break;
            }

            detail::VideoEncoderInput encoderInput = {};
            result = m_VideoEncoder.AcquireInputBuffer(&encoderInput);
            NN_GRCSRV_VENCT_CHECK_RESULT(result,              "encoder error at acqiuiring input buffer");
            NN_GRCSRV_VENCT_CHECK_STATUS(encoderInput.status, "encoder error at acqiuiring input buffer");
            NN_GRCSRV_OFFSCRN_LOG_DEV_VENCT("acquired from video-encoder\n");

            if(!capturedBuffer.isFinishRequested)
            {
                // エンコーダの入力にハンドルをセット
                buffer_handle_t bufferHandle = static_cast<buffer_handle_t>(capturedBuffer.buffer->handle);
                const uint32_t type = 1;
                std::memcpy(encoderInput.buffer->data(), &type, sizeof(type));
                std::memcpy(encoderInput.buffer->data() + sizeof(type), &bufferHandle, sizeof(buffer_handle_t));
                encoderInput.buffer->setRange(0, sizeof(type) + sizeof(buffer_handle_t));

                // エンコード完了後に CaptureBuffer を timestamp で引けるように登録
                {
                    detail::LayerEncodingBuffer e = {};
                    e.captureBufferIndex = capturedBuffer.index;
                    e.captureBuffer      = capturedBuffer.buffer;
                    e.timestamp = capturedBuffer.timestampUs;
                    NN_ABORT_UNLESS_RESULT_SUCCESS(m_EncodingBufferList.Register(e));
                    NN_GRCSRV_OFFSCRN_LOG_DEV_VENCT("remember encoding buffer(idx=%d, timestamp=%lld)\n", e.captureBufferIndex, e.timestamp);
                }

                // Encoder に送る
                int32_t queueingStatus = 0;
                result = m_VideoEncoder.QueueInputBuffer(
                    &queueingStatus, encoderInput.index, encoderInput.buffer->offset(), encoderInput.buffer->size(), capturedBuffer.timestampUs, 0 /* flags */
                );
                NN_GRCSRV_VENCT_CHECK_RESULT(result,         "encoder error at queueing input buffer");
                NN_GRCSRV_VENCT_CHECK_STATUS(queueingStatus, "encoder error at queueing input buffer");
                NN_GRCSRV_OFFSCRN_LOG_DEV_VENCT("queued buffer to video encoder\n");
            }
            else // 終了要求
            {
                int32_t queueingStatus = 0;
                result = m_VideoEncoder.QueueInputBuffer(&queueingStatus, encoderInput.index, 0, 0, 0, android::MediaCodec::BUFFER_FLAG_EOS);
                NN_GRCSRV_VENCT_CHECK_RESULT(result,         "encoder error at queueing eos marker");
                NN_GRCSRV_VENCT_CHECK_STATUS(queueingStatus, "encoder error at queueing eos marker");
                NN_GRCSRV_OFFSCRN_LOG_DEV_VENCT("queued eos to video encoder\n");

                // NOTE: VideoCapture からの終了要求はバッファが紐づいていないので返却しなくて良い

                exitResult = nn::ResultSuccess();
                break;
            }
        }

        if(exitResult.IsSuccess())
        {
            NN_GRCSRV_OFFSCRN_LOG_DEV_VENCT("exit successfully\n");
            m_VideoEncodeThreadResult = nn::util::make_optional<nn::Result>(nn::ResultSuccess());
        }
        else
        {
            NN_GRCSRV_OFFSCRN_LOG_DEV_VENCT("exit by error\n");
            m_VideoEncodeThreadResult = nn::util::make_optional<nn::Result>(std::move(exitResult));
        }

    }

    //---------------------------------------

#define NN_GRCSRV_VMUXT_CHECK_RESULT(result, message)    \
    {                                                   \
        nn::Result r = (result);                        \
        if(r.IsFailure())                               \
        {                                               \
            NN_GRCSRV_OFFSCRN_LOG_DEV_VMUXT_ERR(message "(result=%d-%d)\n", r.GetModule(), r.GetDescription());   \
            exitResult = nn::grc::ResultInternalOffscreenVideoMuxerError();  \
            break;                                      \
        }                                               \
    }

#define NN_GRCSRV_VMUXT_CHECK_STATUS(status, message)    \
    {                                                   \
        int s = (status);                               \
        if(s != android::NO_ERROR)                      \
        {                                               \
            NN_GRCSRV_OFFSCRN_LOG_DEV_VMUXT_ERR(message "(status=%d)\n", s);   \
            exitResult = nn::grc::ResultInternalOffscreenVideoMuxerError();  \
            break;                                      \
        }                                               \
    }



    void Layer::VideoMuxerThreadFunction(void* arg) NN_NOEXCEPT
    {
        auto pLayer = reinterpret_cast<Layer*>(arg);
        pLayer->VideoMuxerThreadFunctionImpl();
    }

    void Layer::VideoMuxerThreadFunctionImpl() NN_NOEXCEPT
    {
        // 終了条件：
        //   ・最終バッファを書き出す
        //   ・エラーが発生する
        // Abort による中断の場合、別スレッドにより VideoEncoder や Muxer が停止され、各種関数がエラー（状態が不正）を返すため、終了する。
        //
        // 動画の開始のたびにスレッドを起動しなおす必要がある。
        NN_GRCSRV_OFFSCRN_LOG_DEV_VMUXT("start\n");

        nn::Result exitResult;

        for(;;)
        {
            nn::Result result = nn::ResultSuccess();
            detail::VideoEncoderOutput encoderOutput = {};
            result = m_VideoEncoder.AcquireOutputBuffer(&encoderOutput);
            NN_GRCSRV_VMUXT_CHECK_RESULT(result, "encoder error at acquiring output buffer");
            NN_GRCSRV_OFFSCRN_LOG_DEV_VMUXT("acquired from video-encoder(status=%d)\n", encoderOutput.status);

            if(encoderOutput.status == android::NO_ERROR)
            {
                // フレームのエンコードが完了したらキャプチャバッファは用済みなので返却
                // NOTE: 取りまわしを簡略化するため、最終バッファは返却しない
                // NOTE: エンコーダは 1 フレームの入力に対して複数回の出力を行うことがある
                if(
                    (encoderOutput.flags & android::MediaCodec::BUFFER_FLAG_EOS) == 0 &&
                    (encoderOutput.flags & android::MediaCodec::BUFFER_FLAG_CODECCONFIG) == 0
                )
                {
                    NN_GRCSRV_OFFSCRN_LOG_DEV_VMUXT("recalling encoding buffer(timestamp=%lld)\n", encoderOutput.timestamp);
                    detail::LayerEncodingBuffer e;
                    result = m_EncodingBufferList.Find(
                        &e,
                        [&](const detail::LayerEncodingBuffer& v)->bool
                        {
                            //NN_GRCSRV_OFFSCRN_LOG_DEV_MUXT("  %lld vs. %lld\n", v.timestamp, encoderOutput.timestamp);
                            return v.timestamp == encoderOutput.timestamp;
                        }
                    );
                    NN_GRCSRV_VMUXT_CHECK_RESULT(result, "find encoding buffer failed");

                    detail::VideoEncoderToVideoCaptureValue captureInput = {};
                    captureInput.index  = e.captureBufferIndex;
                    captureInput.buffer = e.captureBuffer;
                    m_VideoEncoderToVideoCaptureQueue.Enqueue(captureInput);
                    NN_ABORT_UNLESS_RESULT_SUCCESS(m_EncodingBufferList.Unregister(e));
                    NN_GRCSRV_OFFSCRN_LOG_DEV_VMUXT("released to video-capture\n");
                }

                // Muxer で書き出し
                result = m_Muxer.StoreVideoFrame(encoderOutput.buffer, encoderOutput.timestamp, encoderOutput.flags);
                if(TryConvertCapsrvError(&result, result))
                {
                    NN_GRCSRV_OFFSCRN_LOG_DEV_VMUXT_ERR("muxer failed to store video frame by album error (result=%d-%d)\n", result.GetModule(), result.GetDescription());
                    exitResult = result;
                    break;
                }
                NN_GRCSRV_VMUXT_CHECK_RESULT(result, "muxer failed to store video frame");
                NN_GRCSRV_OFFSCRN_LOG_DEV_VMUXT("stored video frame by muxer\n");

                // Encoder に返却
                {
                    int32_t status;
                    result = m_VideoEncoder.ReleaseOutputBuffer(&status, encoderOutput.index);
                    NN_GRCSRV_VMUXT_CHECK_RESULT(result, "video encoder error at releasing output buffer");
                    NN_GRCSRV_VMUXT_CHECK_STATUS(status, "video encoder error at releasing output buffer");
                    NN_GRCSRV_OFFSCRN_LOG_DEV_VMUXT("released to video-encoder\n");
                }

                // EOS であれば終わる
                if((encoderOutput.flags & android::MediaCodec::BUFFER_FLAG_EOS) != 0)
                {
                    exitResult = nn::ResultSuccess();
                    break;
                }
            }
            else if(encoderOutput.status == android::INFO_FORMAT_CHANGED)
            {
                // エンコードの開始時に 1 回だけ呼ばれる。
                NN_GRCSRV_OFFSCRN_LOG_DEV_VMUXT("video-format changed\n");

                // 新しいフォーマットを取得
                int32_t status;
                android::sp<android::AMessage> format;
                result = m_VideoEncoder.GetOutputFormat(&status, &format);
                NN_GRCSRV_VMUXT_CHECK_RESULT(result, "video encoder error at getting output format");
                NN_GRCSRV_VMUXT_CHECK_STATUS(status, "video encoder error at getting output format");
                if(format == nullptr)
                {
                    NN_GRCSRV_OFFSCRN_LOG_DEV_VMUXT_ERR("video encoder error at getting output format(format null)\n");
                    exitResult = nn::grc::ResultInternalOffscreenVideoMuxerError();
                    break;
                }

                // Muxer に設定。もし 2 回目が呼ばれた場合、 ResultInvalidCall で失敗する。
                result = m_Muxer.SetVideoFormat(format);
                NN_GRCSRV_VMUXT_CHECK_RESULT(result, "muxer failed to set video format");
                NN_GRCSRV_OFFSCRN_LOG_DEV_VMUXT("set new format to muxer\n");

                // Muxer の準備ができるまで待機。
                // このスレッドは SetVideoFormat() だけ呼べばいいのでここで待つ。
                // 待機中に Stop() が呼び出されると失敗が返る。
                result = m_Muxer.WaitForWriting();
                NN_GRCSRV_VMUXT_CHECK_RESULT(result, "abort requested while waiting for muxer ready");
            }
            else
            {
                NN_GRCSRV_VMUXT_CHECK_STATUS(encoderOutput.status, "video encoder error at aquiring output buffer");
            }

        }
        NN_GRCSRV_OFFSCRN_LOG_DEV_VMUXT("exit\n");

        if(exitResult.IsSuccess())
        {
            NN_GRCSRV_OFFSCRN_LOG_DEV_VMUXT("exit successfully\n");
            m_VideoMuxerThreadResult = nn::util::make_optional<nn::Result>(nn::ResultSuccess());
        }
        else
        {
            NN_GRCSRV_OFFSCRN_LOG_DEV_VMUXT("exit by error\n");
            m_VideoMuxerThreadResult = nn::util::make_optional<nn::Result>(std::move(exitResult));
        }
    }

    //---------------------------------------

#define NN_GRCSRV_AENCT_CHECK_RESULT(result, message)   \
    {                                                   \
        nn::Result r = (result);                        \
        if(r.IsFailure())                               \
        {                                               \
            NN_GRCSRV_OFFSCRN_LOG_DEV_AENCT_ERR(message "(result=%d-%d)\n", r.GetModule(), r.GetDescription());   \
            exitResult = nn::grc::ResultInternalOffscreenAudioEncoderError();   \
            break;                                      \
        }                                               \
    }

#define NN_GRCSRV_AENCT_CHECK_STATUS(status, message)   \
    {                                                   \
        int s = (status);                               \
        if(s != android::NO_ERROR)                      \
        {                                               \
            NN_GRCSRV_OFFSCRN_LOG_DEV_AENCT_ERR(message "(status=%d)\n", s);   \
            exitResult = nn::grc::ResultInternalOffscreenAudioEncoderError();   \
            break;                                      \
        }                                               \
    }

    void Layer::AudioEncodeThreadFunction(void* arg) NN_NOEXCEPT
    {
        auto pLayer = reinterpret_cast<Layer*>(arg);
        pLayer->AudioEncodeThreadFunctionImpl();
    }

    void Layer::AudioEncodeThreadFunctionImpl() NN_NOEXCEPT
    {
        // 終了条件：
        //   ・captureBuffer から停止要求が来る
        //   ・エラーが発生する
        // Abort による中断の場合、別スレッドにより VideoEncoder が停止され、各種関数がエラー（状態が不正）を返すため、終了する。
        NN_GRCSRV_OFFSCRN_LOG_DEV_VENCT("start\n");

        nn::Result exitResult = nn::ResultSuccess();

        for(;;)
        {
            nn::Result result = nn::ResultSuccess();
            int32_t status = android::NO_ERROR;

            detail::AudioRendererToAudioEncoderValue inputBuffer = {};
            NN_GRCSRV_OFFSCRN_LOG_DEV_AENCT("waiting for audio-renderer...\n");
            m_AudioRendererToAudioEncoderQueue.Dequeue(&inputBuffer);
            NN_GRCSRV_OFFSCRN_LOG_DEV_AENCT("acquired from audio-renderer\n");

            if(inputBuffer.isAbortRequested)
            {
                NN_GRCSRV_OFFSCRN_LOG_DEV_AENCT_ERR("abort requested\n");
                exitResult = nn::grc::ResultInternalOffscreenAbortRequested();
                break;
            }

            if(!inputBuffer.isFinishRequested)
            {
                // Encoder に送る
                result = m_AudioEncoder.QueueInputBuffer(
                    &status, inputBuffer.index, inputBuffer.buffer->offset(), inputBuffer.buffer->size(), inputBuffer.timestampUs, 0 /*flags*/
                );
                NN_GRCSRV_AENCT_CHECK_RESULT(result, "encoder error at queueing input buffer");
                NN_GRCSRV_AENCT_CHECK_STATUS(status, "encoder error at queueing input buffer");
                NN_GRCSRV_OFFSCRN_LOG_DEV_AENCT("queued buffer to audio encoder\n");
            }
            else // 終了要求
            {
                result = m_AudioEncoder.QueueInputBuffer(&status, inputBuffer.index, 0, 0, inputBuffer.timestampUs, android::MediaCodec::BUFFER_FLAG_EOS);
                NN_GRCSRV_AENCT_CHECK_RESULT(result, "encoder error at queueing eos marker");
                NN_GRCSRV_AENCT_CHECK_STATUS(status, "encoder error at queueing eos marker");
                NN_GRCSRV_OFFSCRN_LOG_DEV_AENCT("queued eos to audio encoder\n");

                exitResult = nn::ResultSuccess();
                break;
            }
        }

        if(exitResult.IsSuccess())
        {
            NN_GRCSRV_OFFSCRN_LOG_DEV_AENCT("exit successfully\n");
            m_AudioEncodeThreadResult = nn::util::make_optional<nn::Result>(nn::ResultSuccess());
        }
        else
        {
            NN_GRCSRV_OFFSCRN_LOG_DEV_AENCT("exit by error\n");
            m_AudioEncodeThreadResult = nn::util::make_optional<nn::Result>(std::move(exitResult));
        }

    }

    //---------------------------------------

#define NN_GRCSRV_AMUXT_CHECK_RESULT(result, message)    \
    {                                                   \
        nn::Result r = (result);                        \
        if(r.IsFailure())                               \
        {                                               \
            NN_GRCSRV_OFFSCRN_LOG_DEV_AMUXT_ERR(message "(result=%d-%d)\n", r.GetModule(), r.GetDescription());   \
            exitResult = nn::grc::ResultInternalOffscreenAudioMuxerError();  \
            break;                                      \
        }                                               \
    }

#define NN_GRCSRV_AMUXT_CHECK_STATUS(status, message)    \
    {                                                   \
        int s = (status);                               \
        if(s != android::NO_ERROR)                      \
        {                                               \
            NN_GRCSRV_OFFSCRN_LOG_DEV_AMUXT_ERR(message "(status=%d)\n", s);   \
            exitResult = nn::grc::ResultInternalOffscreenAudioMuxerError();  \
            break;                                      \
        }                                               \
    }

    void Layer::AudioMuxerThreadFunction(void* arg) NN_NOEXCEPT
    {
        auto pLayer = reinterpret_cast<Layer*>(arg);
        pLayer->AudioMuxerThreadFunctionImpl();
    }

    void Layer::AudioMuxerThreadFunctionImpl() NN_NOEXCEPT
    {
        // 動画の開始のたびにスレッドを起動しなおす必要がある。
        NN_GRCSRV_OFFSCRN_LOG_DEV_AMUXT("start\n");

        nn::Result exitResult = nn::ResultSuccess();

        // Encoder から Renderer にバッファを移す。
        // Encoder の持っているバッファがなくなるか、 Renderer への Queue がいっぱいになったら終わり。
        // 正常終了した場合 true を返す。
        // エラーが発生した場合 false を返す。このとき exitResult にエラーが記録されるが、基本的に後から上書きする。
        auto moveBuffersFromEncoderToRenderer = [&]()->bool
        {
            NN_GRCSRV_OFFSCRN_LOG_DEV_AMUXT("moving encoder input buffer to renderer...\n");
            for(;;)
            {
                nn::Result result = nn::ResultSuccess();

                // Renderer 側のキューがいっぱいなら終わり
                if(m_AudioEncoderToAudioRendererQueue.IsFull())
                {
                    NN_GRCSRV_OFFSCRN_LOG_DEV_AMUXT("moved encoder input buffer to renderer(destination full)\n");
                    return true;
                }

                detail::AudioEncoderInput e;
                result = m_AudioEncoder.TryAcquireInputBuffer(&e);
                if(nn::grc::ResultInternalOffscreenAudioEncoderNoInputBufferAvailable::Includes(result))
                {
                    NN_GRCSRV_OFFSCRN_LOG_DEV_AMUXT("moved encoder input buffer to renderer(source empty)\n");
                    return true;
                }
                NN_GRCSRV_AMUXT_CHECK_RESULT(result  , "encoder error at acquiring input buffer");
                NN_GRCSRV_AMUXT_CHECK_STATUS(e.status, "encoder error at acquiring input buffer");

                NN_GRCSRV_OFFSCRN_LOG_DEV_AMUXT("  moving buffer(idx=%lld)\n", e.index);
                detail::AudioEncoderToAudioRendererValue v = {};
                v.index = e.index;
                v.buffer = e.buffer;
                m_AudioEncoderToAudioRendererQueue.Enqueue(v);
            }
            return false;
        };

        // 最初に InputBuffer を可能な限り Encoder から Renderer に渡しておく
        if(!moveBuffersFromEncoderToRenderer())
        {
            exitResult = nn::grc::ResultInternalOffscreenAudioMuxerError();
            goto EXIT;
        }

        for(;;)
        {
            nn::Result result = nn::ResultSuccess();

            detail::AudioEncoderOutput encoderOutput = {};
            result = m_AudioEncoder.AcquireOutputBuffer(&encoderOutput);
            NN_GRCSRV_AMUXT_CHECK_RESULT(result, "encoder error at acquiring output buffer");
            NN_GRCSRV_OFFSCRN_LOG_DEV_AMUXT("acquired from audio-encoder(status=%d)\n", encoderOutput.status);

            // ところで OutputBuffer が取れたということはエンコードが進んでいるはずなので
            // InputBuffer を可能な限り Encoder から Renderer に渡しておく
            if(!moveBuffersFromEncoderToRenderer())
            {
                exitResult = nn::grc::ResultInternalOffscreenAudioMuxerError();
                break;
            }

            if(encoderOutput.status == android::NO_ERROR)
            {
                // Muxer で書き出し
                result = m_Muxer.StoreAudioFrame(encoderOutput.buffer, encoderOutput.timestamp, encoderOutput.flags);
                if(TryConvertCapsrvError(&result, result))
                {
                    NN_GRCSRV_OFFSCRN_LOG_DEV_AMUXT_ERR("muxer failed to store audio frame by album error (result=%d-%d)\n", result.GetModule(), result.GetDescription());
                    exitResult = result;
                    break;
                }
                NN_GRCSRV_AMUXT_CHECK_RESULT(result, "muxer failed to store audio frame");
                NN_GRCSRV_OFFSCRN_LOG_DEV_AMUXT("stored audio frame by muxer\n");

                // Encoder に返却
                {
                    int32_t status;
                    result = m_AudioEncoder.ReleaseOutputBuffer(&status, encoderOutput.index);
                    NN_GRCSRV_AMUXT_CHECK_RESULT(result, "audio encoder error at releasing output buffer");
                    NN_GRCSRV_AMUXT_CHECK_STATUS(status, "audio encoder error at releasing output buffer");
                    NN_GRCSRV_OFFSCRN_LOG_DEV_AMUXT("released to audio-encoder\n");
                }

                // EOS であれば終わる
                if((encoderOutput.flags & android::MediaCodec::BUFFER_FLAG_EOS) != 0)
                {
                    exitResult = nn::ResultSuccess();
                    break;
                }
            }
            else if(encoderOutput.status == android::INFO_FORMAT_CHANGED)
            {
                NN_GRCSRV_OFFSCRN_LOG_DEV_AMUXT("audio-format changed\n");

                // 新しいフォーマットを取得
                int32_t status;
                android::sp<android::AMessage> format;
                result = m_AudioEncoder.GetOutputFormat(&status, &format);
                NN_GRCSRV_AMUXT_CHECK_RESULT(result, "audio encoder error at getting output format");
                NN_GRCSRV_AMUXT_CHECK_STATUS(status, "audio encoder error at getting output format");
                if(format == nullptr)
                {
                    NN_GRCSRV_OFFSCRN_LOG_DEV_AMUXT_ERR("audio encoder error at getting output format(format null)\n");
                    exitResult = nn::grc::ResultInternalOffscreenAudioMuxerError();
                    break;
                }

                // Muxer に設定。もし 2 回目が呼ばれた場合、 ResultInvalidCall で失敗する。
                result = m_Muxer.SetAudioFormat(format);
                NN_GRCSRV_AMUXT_CHECK_RESULT(result, "muxer failed to set audio format");
                NN_GRCSRV_OFFSCRN_LOG_DEV_AMUXT("set new format to muxer\n");

                // Muxer の準備ができるまで待機。
                // このスレッドは SetAudioFormat() だけ呼べばいいのでここで待つ。
                // 待機中に Stop() が呼び出されると失敗が返る。
                result = m_Muxer.WaitForWriting();
                NN_GRCSRV_AMUXT_CHECK_RESULT(result, "abort requested while waiting for muxer ready");
            }
            else
            {
                NN_GRCSRV_AMUXT_CHECK_STATUS(encoderOutput.status, "audio encoder error at aquiring output buffer");
            }

        }
        NN_GRCSRV_OFFSCRN_LOG_DEV_AMUXT("exit\n");

    EXIT:
        if(exitResult.IsSuccess())
        {
            NN_GRCSRV_OFFSCRN_LOG_DEV_AMUXT("exit successfully\n");
            m_AudioMuxerThreadResult = nn::util::make_optional<nn::Result>(nn::ResultSuccess());
        }
        else
        {
            NN_GRCSRV_OFFSCRN_LOG_DEV_AMUXT("exit by error\n");
            m_AudioMuxerThreadResult = nn::util::make_optional<nn::Result>(std::move(exitResult));
        }
    }// NOLINT(impl/function_size)
}}}

