﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "grcsrvOffscreen_CaptureImageBuffer.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/grc/grc_Result.h>
#include "grcsrvOffscreen_Macro.h"
#include "../../capsrv/capture/capsrv_ImageBuffer-module.nvnflinger.h"
#include "../../capsrv/capture/capsrv_CaptureModule-module.nvnflinger.h"

namespace nn{ namespace grcsrv{ namespace offscreen{

    const int CaptureImageBuffer::BufferCountMax;

    CaptureImageBuffer::CaptureImageBuffer() NN_NOEXCEPT
        : m_Memory(nullptr)
        , m_MemorySize(0)
        , m_pModule(nullptr)
        , m_WidthMax(0)
        , m_HeightMax(0)
    {
        std::memset(m_Nv12YBufferOffsetList, 0, sizeof(m_Nv12YBufferOffsetList));
        std::memset(m_Nv12UvBufferOffsetList, 0, sizeof(m_Nv12UvBufferOffsetList));

        std::memset(m_WidthList, 0, sizeof(m_WidthList));
        std::memset(m_HeightList, 0, sizeof(m_HeightList));
    }

    nn::Result CaptureImageBuffer::InitializeMemoryPool(
        nn::capsrv::capture::CaptureModule* pModule,
        void* memory,
        size_t size,
        int widthMax,
        int heightMax
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_ALIGNED(memory, GetRequiredMemoryAlignment());
        NN_SDK_REQUIRES_GREATER_EQUAL(size, GetRequiredMemorySize(widthMax, heightMax, BufferCountMax));

        NN_GRCSRV_PROCESS_START();

        nn::capsrv::capture::ImageBufferInfo infoY = {};
        infoY.width  = widthMax;
        infoY.height = heightMax;
        infoY.format = nn::capsrv::capture::ImageFormat_Y_NV12;

        nn::capsrv::capture::ImageBufferInfo infoUv = {};
        infoUv.width  = widthMax  / 2;
        infoUv.height = heightMax / 2;
        infoUv.format = nn::capsrv::capture::ImageFormat_Uv_NV12;

        size_t reqSizeY  = GetRequiredBufferMemorySizeNv12YImpl(widthMax, heightMax);
        size_t reqSizeUv = GetRequiredBufferMemorySizeNv12UvImpl(widthMax, heightMax);
        size_t reqAlign  = GetRequiredMemoryAlignment();

        NN_GRCSRV_ASSERT_ABORT(reqSizeY >= nn::capsrv::capture::ImageBuffer::GetRequiredMemorySize(infoY));
        NN_GRCSRV_ASSERT_ABORT(reqSizeUv >= nn::capsrv::capture::ImageBuffer::GetRequiredMemorySize(infoUv));
        NN_GRCSRV_ASSERT_ABORT(reqAlign >= nn::capsrv::capture::ImageBuffer::GetRequiredMemoryAlignment(pModule, infoY));
        NN_GRCSRV_ASSERT_ABORT(reqAlign >= nn::capsrv::capture::ImageBuffer::GetRequiredMemoryAlignment(pModule, infoUv));

        size_t reqPoolSize = GetRequiredMemorySize(widthMax, heightMax, BufferCountMax);
        NN_GRCSRV_OFFSCRN_LOG_DEV_IMGBUF("required memory pool size = %llu, align = %llu\n", reqPoolSize, reqAlign);

        NN_RESULT_DO(m_MemoryPool.Initialize(pModule, memory, reqPoolSize));
        NN_GRCSRV_PROCESS_ROLLBACK(m_MemoryPool.Finalize());

        // 先にオフセット計算だけしておく
        ptrdiff_t offset = 0;
        for(int i = 0; i < BufferCountMax; i++)
        {
            m_Nv12YBufferOffsetList[i] = offset;
            offset += reqSizeY;
            m_Nv12UvBufferOffsetList[i] = offset;
            offset += reqSizeUv;
        }

        NN_GRCSRV_PROCESS_SUCCESS();
        m_WidthMax = widthMax;
        m_HeightMax = heightMax;
        m_Memory = reinterpret_cast<char*>(memory);
        m_MemorySize = size;
        m_pModule = pModule;
        NN_RESULT_SUCCESS;
    }

    void CaptureImageBuffer::FinalizeMemoryPool() NN_NOEXCEPT
    {
        for(int i = 0; i < BufferCountMax; i++)
        {
            if(IsBufferInitialized(i))
            {
                NN_GRCSRV_OFFSCRN_LOG_ERR("[imgbuf] index=%d is still initialized\n", i);
            }
            NN_SDK_REQUIRES(!IsBufferInitialized(i));
        }

        std::memset(m_Nv12YBufferOffsetList, 0, sizeof(m_Nv12YBufferOffsetList));
        std::memset(m_Nv12UvBufferOffsetList, 0, sizeof(m_Nv12UvBufferOffsetList));
        m_MemoryPool.Finalize();

        m_WidthMax = 0;
        m_HeightMax = 0;
        m_Memory = nullptr;
        m_MemorySize = 0;
        m_pModule = nullptr;
    }

    //-----------------------------------

    nn::Result CaptureImageBuffer::InitializeBuffer(int index, int width, int height) NN_NOEXCEPT
    {
        NN_GRCSRV_OFFSCRN_LOG_DEV_IMGBUF("initialize index=%d w=%d h=%d\n", index, width, height);
        NN_SDK_REQUIRES_NOT_NULL(m_Memory);
        NN_RESULT_THROW_UNLESS(index >= 0 && index < BufferCountMax, nn::grc::ResultPreConditionViolation());
        NN_RESULT_THROW_UNLESS(!IsBufferInitialized(index), nn::grc::ResultPreConditionViolation());
        NN_RESULT_THROW_UNLESS(width > 0 && width <= m_WidthMax, nn::grc::ResultPreConditionViolation());
        NN_RESULT_THROW_UNLESS(height > 0 && height <= m_HeightMax, nn::grc::ResultPreConditionViolation());

        NN_GRCSRV_PROCESS_START();

        nn::capsrv::capture::ImageBufferInfo infoY = {};
        infoY.width  = width;
        infoY.height = height;
        infoY.format = nn::capsrv::capture::ImageFormat_Y_NV12;

        nn::capsrv::capture::ImageBufferInfo infoUv = {};
        infoUv.width  = width  / 2;
        infoUv.height = height / 2;
        infoUv.format = nn::capsrv::capture::ImageFormat_Uv_NV12;

        size_t reqSizeY  = GetRequiredBufferMemorySizeNv12YImpl(width, height);
        size_t reqSizeUv = GetRequiredBufferMemorySizeNv12UvImpl(width, height);
        size_t reqAlign  = GetRequiredMemoryAlignment();

        NN_GRCSRV_ASSERT_ABORT(reqSizeY >= nn::capsrv::capture::ImageBuffer::GetRequiredMemorySize(infoY));
        NN_GRCSRV_ASSERT_ABORT(reqSizeUv >= nn::capsrv::capture::ImageBuffer::GetRequiredMemorySize(infoUv));
        NN_GRCSRV_ASSERT_ABORT(reqAlign >= nn::capsrv::capture::ImageBuffer::GetRequiredMemoryAlignment(m_pModule, infoY));
        NN_GRCSRV_ASSERT_ABORT(reqAlign >= nn::capsrv::capture::ImageBuffer::GetRequiredMemoryAlignment(m_pModule, infoUv));

        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(m_Nv12PlaneList[2 * index + 0].Initialize(m_pModule, infoY , &m_MemoryPool, m_Nv12YBufferOffsetList [index], reqSizeY));
            NN_ABORT_UNLESS_RESULT_SUCCESS(m_Nv12PlaneList[2 * index + 1].Initialize(m_pModule, infoUv, &m_MemoryPool, m_Nv12UvBufferOffsetList[index], reqSizeUv));
            m_SurfaceList[2 * index + 0] = *m_Nv12PlaneList[2 * index + 0].GetData()->GetSurface();
            m_SurfaceList[2 * index + 1] = *m_Nv12PlaneList[2 * index + 1].GetData()->GetSurface();
        }

        {
            buffer_handle_t bufferHandle = {};
            auto importResult = nvgr_import_nvrmsurface(&m_SurfaceList[2 * index], 2, &bufferHandle);
            NN_ABORT_UNLESS_EQUAL(importResult, 0);
            m_BufferHandleList[index] = bufferHandle;
            m_GraphicBufferList[index] = new android::GraphicBuffer(
                width,
                height,
                NVGR_PIXEL_FORMAT_NV12 | NVGR_PIXEL_FORMAT_709,
                GRALLOC_USAGE_HW_COMPOSER | GRALLOC_USAGE_HW_VIDEO_ENCODER,
                static_cast<uint32_t>(nvgr_get_stride(bufferHandle)),
                const_cast<native_handle*>(bufferHandle),
                false
            );
            NN_ABORT_UNLESS_NOT_NULL(m_GraphicBufferList[index]);
        }


        FlushDataCache();
        NN_GRCSRV_PROCESS_SUCCESS();
        m_WidthList[index]  = width;
        m_HeightList[index] = height;
        NN_RESULT_SUCCESS;
    }

    void CaptureImageBuffer::FinalizeBuffer(int index) NN_NOEXCEPT
    {
        NN_GRCSRV_OFFSCRN_LOG_DEV_IMGBUF("finalize index=%d\n", index);
        NN_SDK_REQUIRES_NOT_NULL(m_Memory);
        NN_SDK_REQUIRES(IsBufferInitialized(index));

        auto pGrAlloc = m_pModule->GetGralloc();

        {
            m_GraphicBufferList[index].clear();
            pGrAlloc->free(pGrAlloc, m_BufferHandleList[index]);
            m_BufferHandleList[index] = nullptr;
        }

        {
            m_SurfaceList[2 * index + 0] = {};
            m_SurfaceList[2 * index + 1] = {};
            m_Nv12PlaneList[2 * index + 0].Finalize();
            m_Nv12PlaneList[2 * index + 1].Finalize();
        }

        m_WidthList[index] = 0;
        m_HeightList[index] = 0;
    }

    //----------------------------------------------

    bool CaptureImageBuffer::IsBufferInitialized(int index) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(index, 0, BufferCountMax);
        return m_GraphicBufferList[index] != nullptr;
    }

    int CaptureImageBuffer::GetBufferWidth(int index) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(index, 0, BufferCountMax);
        return m_WidthList[index];
    }

    int CaptureImageBuffer::GetBufferHeight(int index) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(index, 0, BufferCountMax);
        return m_HeightList[index];
    }

    void* CaptureImageBuffer::GetBufferMemoryNv12Y(int index) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(index, 0, BufferCountMax);
        return m_Memory + m_Nv12YBufferOffsetList[index];
    }

    void* CaptureImageBuffer::GetBufferMemoryNv12Uv(int index) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(index, 0, BufferCountMax);
        return m_Memory + m_Nv12UvBufferOffsetList[index];
    }

    size_t CaptureImageBuffer::GetBufferMemorySizeNv12Y(int index) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(index, 0, BufferCountMax);
        return GetRequiredBufferMemorySizeNv12YImpl(m_WidthList[index], m_HeightList[index]);
    }

    size_t CaptureImageBuffer::GetBufferMemorySizeNv12Uv(int index) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(index, 0, BufferCountMax);
        return GetRequiredBufferMemorySizeNv12UvImpl(m_WidthList[index], m_HeightList[index]);
    }

    android::sp<android::GraphicBuffer>* CaptureImageBuffer::GetGraphicBufferListRef() NN_NOEXCEPT
    {
        return m_GraphicBufferList;
    }

    void CaptureImageBuffer::FlushDataCache() NN_NOEXCEPT
    {
        nn::os::FlushDataCache(m_Memory, m_MemorySize);
    }

}}}

