﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "grcsrvOffscreen_AudioRendererHandler.h"

#include <algorithm>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/grc/grc_Result.h>
#include <nn/grc/grc_ResultPrivate.h>

namespace nn{ namespace grcsrv{ namespace offscreen{ namespace detail{

    AudioRendererHandler::AudioRendererHandler() NN_NOEXCEPT
        : m_State(State::NotInitialized)
        , m_pAcquirableOutputBufferQueue(nullptr)
        , m_pQueuedOutputBufferQueue(nullptr)
    {
    }

    bool AudioRendererHandler::IsInitialized() const NN_NOEXCEPT
    {
        return m_State != State::NotInitialized;
    }

    nn::util::optional<nn::Result> AudioRendererHandler::GetExitResult() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        return m_ExitResult;
    }

    nn::Result AudioRendererHandler::Initialize(
        AcquirableOutputBufferQueue* pAcquirableOutputBufferQueue,
        QueuedOutputBufferQueue* pQueuedOutputBufferQueue
    ) NN_NOEXCEPT
    {
        NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIORENDERER("initializing audio-renderer...\n");
        NN_SDK_REQUIRES(!IsInitialized());

        NN_RESULT_DO(nn::os::CreateSystemEvent(&m_EncodeReadyEvent, nn::os::EventClearMode_ManualClear, true));

        // 以降失敗しない
        nn::os::InitializeEvent(&m_ExitEvent, false, nn::os::EventClearMode_ManualClear);
        pAcquirableOutputBufferQueue->InitializeDequeueableMultiWaitHolder(m_OutputDequeueableWaitHandler.GetHolder());

        m_ExitResult = nn::util::nullopt;
        m_pAcquirableOutputBufferQueue = pAcquirableOutputBufferQueue;
        m_pQueuedOutputBufferQueue = pQueuedOutputBufferQueue;
        m_SampleRate = 0;
        m_ChannelCount = 0;
        m_ChannelSize = 0;
        m_TotalByteCount = 0;
        m_State = State::Initializing;
        ChangeStateIdleImpl();
        NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIORENDERER("initialized audio-renderer\n");
        NN_RESULT_SUCCESS;
    }

    void AudioRendererHandler::Finalize() NN_NOEXCEPT
    {
        NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIORENDERER("finalizing audio-renderer...\n");
        NN_SDK_REQUIRES(IsInitialized());

        ChangeStateFinalizingImpl();

        m_pAcquirableOutputBufferQueue->FinalizeDequeueableMultiWaitHolder(m_OutputDequeueableWaitHandler.GetHolder());
        nn::os::FinalizeEvent(&m_ExitEvent);
        nn::os::DestroySystemEvent(&m_EncodeReadyEvent);

        m_ExitResult = nn::util::nullopt;
        m_pAcquirableOutputBufferQueue = nullptr;
        m_pQueuedOutputBufferQueue = nullptr;
        m_SampleRate = 0;
        m_ChannelCount = 0;
        m_ChannelSize = 0;
        m_TotalByteCount = 0;
        m_State = State::NotInitialized;
        NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIORENDERER("finalized audio-renderer\n");
    }

    //-----------------------------------------------------------

    nn::Result AudioRendererHandler::Start(const nn::grc::OffscreenRecordingParameter& param) NN_NOEXCEPT
    {
        NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIORENDERER("starting audio-renderer...\n");
        NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIORENDERER("  sampleRate  : %llu\n", param.audioSampleRate);
        NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIORENDERER("  channelCount: %llu\n", param.audioChannelCount);
        NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIORENDERER("  channelSize : %llu\n", 2);
        NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIORENDERER("  bitRate     : %llu\n", param.audioBitRate);
        NN_SDK_REQUIRES(IsInitialized());

        NN_RESULT_THROW_UNLESS(m_State == State::Idle, nn::grc::ResultInvalidState());

        m_ExitResult   = nn::util::nullopt;
        m_SampleRate   = param.audioSampleRate;
        m_ChannelCount = param.audioChannelCount;
        m_ChannelSize  = 2; // TODO:
        m_TotalByteCount = 0;
        ChangeStateEncodingImpl();
        NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIORENDERER("started audio-renderer\n");
        NN_RESULT_SUCCESS;
    }

    void AudioRendererHandler::Abort() NN_NOEXCEPT
    {
        NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIORENDERER("aborting audio-renderer...\n");
        NN_SDK_REQUIRES(IsInitialized());

        ChangeStateIdleImpl();
        m_ExitResult = nn::util::nullopt;
        m_SampleRate   = 0;
        m_ChannelCount = 0;
        m_ChannelSize  = 0;
        m_TotalByteCount = 0;
        NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIORENDERER("aborted audio-renderer\n");
    }

    void AudioRendererHandler::Reset() NN_NOEXCEPT
    {
        NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIORENDERER("resetting audio-renderer...\n");
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_EQUAL(m_State, State::Exited);

        ChangeStateIdleImpl();
        m_ExitResult = nn::util::nullopt;
        m_SampleRate   = 0;
        m_ChannelCount = 0;
        m_ChannelSize  = 0;
        m_TotalByteCount = 0;
        NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIORENDERER("reset audio-renderer\n");
    }

    void AudioRendererHandler::RequestFinish() NN_NOEXCEPT
    {
        NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIORENDERER("requesting finish audio-renderer...\n");
        NN_SDK_REQUIRES(IsInitialized());

        NN_GRCSRV_ASSERT_ABORT(m_State == State::Encoding);

        ChangeStateFinishingImpl();
        NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIORENDERER("requested finish audio-renderer\n");
    }

    void AudioRendererHandler::NotifyError() NN_NOEXCEPT
    {
        NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIORENDERER("notifying error..\n");
        NN_SDK_REQUIRES(IsInitialized());
        if(m_State == State::Encoding)
        {
            ChangeStateExitedImpl(nn::grc::ResultInternalOffscreenErrorReported());
        }
        NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIORENDERER("notifyed error..\n");
    }

    void AudioRendererHandler::InitializeExitWaitHolder(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        nn::os::InitializeMultiWaitHolder(pHolder, &m_ExitEvent);
    }

    void AudioRendererHandler::FinalizeExitWaitHolder(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
    {
        nn::os::FinalizeMultiWaitHolder(pHolder);
    }

    //--------------------------------------------------------------------

    void AudioRendererHandler::GetEncodeReadyEvent(nn::sf::NativeHandle& outHandle) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        auto h = nn::os::GetReadableHandleOfSystemEvent(&m_EncodeReadyEvent);
        outHandle = nn::sf::NativeHandle(h, false);
    }

    nn::Result AudioRendererHandler::EncodeAudio(size_t* pOutEncodedSize, const void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_RESULT_THROW_UNLESS(m_State == State::Encoding, nn::grc::ResultInvalidCall());

        if(size == 0)
        {
            NN_RESULT_SUCCESS;
        }
        NN_RESULT_THROW_UNLESS(!m_pAcquirableOutputBufferQueue->IsEmpty(), nn::grc::ResultInvalidState());

        AcquirableOutputBufferQueue::ValueType output = {};
        m_pAcquirableOutputBufferQueue->Dequeue(&output);

        size_t sizeToEncode = std::min(size, output.buffer->capacity());
        size_t totalSampleCount = (m_TotalByteCount + sizeToEncode) / (m_ChannelCount * m_ChannelSize);
        size_t totalTimeUs = totalSampleCount * 1000000 / m_SampleRate;
        NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIORENDERER("encoding input (%lld/%lld bytes) total %lldsamples(%lldus)\n",
            sizeToEncode,
            size,
            totalSampleCount,
            totalTimeUs
        );
        NN_UNUSED(totalSampleCount);
        NN_UNUSED(totalTimeUs);

        // output.buffer->capacity() == 0 の場合失敗するが…そんなことは起こらないはず
        NN_SDK_ASSERT_GREATER(sizeToEncode, 0);

        std::memcpy(output.buffer->base(), buffer, sizeToEncode);
        output.buffer->setRange(0, sizeToEncode);

        QueuedOutputBufferQueue::ValueType q = {};
        q.index = output.index;
        q.buffer = output.buffer;
        q.timestampUs = (m_TotalByteCount * 1000000) / (m_SampleRate * m_ChannelCount * m_ChannelSize);
        m_pQueuedOutputBufferQueue->Enqueue(q);

        CheckEncodeReadyImpl();

        m_TotalByteCount += sizeToEncode;
        *pOutEncodedSize = sizeToEncode;
        NN_RESULT_SUCCESS;
    }

    void AudioRendererHandler::SubmitFinishRequestImpl() NN_NOEXCEPT
    {
        // イベント待ちをしてからシグナルすること。
        NN_SDK_REQUIRES(!m_pAcquirableOutputBufferQueue->IsEmpty());
        // バッファに十分な容量を持たせておくこと。
        NN_SDK_REQUIRES(!m_pQueuedOutputBufferQueue->IsFull());

        AcquirableOutputBufferQueue::ValueType buf = {};
        m_pAcquirableOutputBufferQueue->Dequeue(&buf);

        buf.buffer->setRange(0,0);

        QueuedOutputBufferQueue::ValueType q = {};
        q.index = buf.index;
        q.buffer = buf.buffer;
        q.timestampUs = (m_TotalByteCount * 1000000) / (m_SampleRate * m_ChannelCount * m_ChannelSize);
        q.isFinishRequested = true;
        m_pQueuedOutputBufferQueue->Enqueue(q);
    }

    //--------------------------------------------------------------------

    void AudioRendererHandler::ChangeStateIdleImpl() NN_NOEXCEPT
    {
        NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIORENDERER("changing state to idle\n");

        ExitCurrentStateImpl();
        m_State = State::Idle;
        nn::os::ClearEvent(&m_ExitEvent);
        CheckEncodeReadyImpl();
    }

    void AudioRendererHandler::ChangeStateEncodingImpl() NN_NOEXCEPT
    {
        NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIORENDERER("changing state to encoding\n");

        ExitCurrentStateImpl();
        m_State = State::Encoding;
        nn::os::ClearEvent(&m_ExitEvent);
        CheckEncodeReadyImpl();
    }

    void AudioRendererHandler::ChangeStateFinishingImpl() NN_NOEXCEPT
    {
        NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIORENDERER("changing state to encoding\n");

        ExitCurrentStateImpl();
        m_State = State::Finishing;
        nn::os::ClearEvent(&m_ExitEvent);
        CheckEncodeReadyImpl();
    }

    void AudioRendererHandler::ChangeStateExitedImpl(nn::Result exitResult) NN_NOEXCEPT
    {
        NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIORENDERER("changing state to exited(%d-%d)\n", exitResult.GetModule(), exitResult.GetDescription());

        ExitCurrentStateImpl();
        m_State = State::Exited;
        m_ExitResult = exitResult;
        nn::os::SignalEvent(&m_ExitEvent);
        CheckEncodeReadyImpl();
    }

    void AudioRendererHandler::ChangeStateFinalizingImpl() NN_NOEXCEPT
    {
        NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIORENDERER("changing state to finalizing\n");

        ExitCurrentStateImpl();
        m_State = State::Finalizing;
        nn::os::ClearEvent(&m_ExitEvent);
        CheckEncodeReadyImpl();
    }

    void AudioRendererHandler::ExitCurrentStateImpl() NN_NOEXCEPT
    {
    }

    void AudioRendererHandler::CheckEncodeReadyImpl() NN_NOEXCEPT
    {
        if(m_State == State::Finishing)
        {
            NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIORENDERER("finishing\n");
            m_OutputDequeueableWaitHandler.Link(
                [](nn::grcsrv::offscreen::MultiWaitHandler*, void* p)->void
                {
                    auto pSelf = reinterpret_cast<AudioRendererHandler*>(p);
                    pSelf->SubmitFinishRequestImpl();
                    pSelf->ChangeStateExitedImpl(nn::ResultSuccess());
                },
                this
            );
            nn::os::SignalSystemEvent(&m_EncodeReadyEvent);
        }
        else if(m_State == State::Encoding && m_pAcquirableOutputBufferQueue->IsEmpty())
        {
            NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIORENDERER("encode not-ready\n");
            m_OutputDequeueableWaitHandler.Link(
                [](nn::grcsrv::offscreen::MultiWaitHandler*, void* p)->void
                {
                    auto pSelf = reinterpret_cast<AudioRendererHandler*>(p);
                    pSelf->CheckEncodeReadyImpl();
                },
                this
            );
            nn::os::ClearSystemEvent(&m_EncodeReadyEvent);
        }
        else
        {
            if(m_State == State::Encoding)
            {
                NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIORENDERER("encode ready\n");
            }
            m_OutputDequeueableWaitHandler.Unlink();
            nn::os::SignalSystemEvent(&m_EncodeReadyEvent);
        }
    }

    //-----------------------------------------------------------------------------------------

}}}}
