﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "grcsrvOffscreen_AudioEncoderHandler.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/grc/grc_Result.h>
#include <nn/grc/grc_ResultPrivate.h>
#include "../grcsrvOffscreen_Config.h"
#include "../grcsrvOffscreen_Macro.h"

NN_PRAGMA_PUSH_WARNINGS
NN_GRCSRV_SUPPRESS_MOVIE_WARNINGS
#include <media/ICrypto.h>
#include <media/stagefright/MediaErrors.h>
NN_PRAGMA_POP_WARNINGS

namespace nn{ namespace grcsrv{ namespace offscreen{ namespace detail{

    namespace {
        static const char MimeType[] = { "audio/mp4a-latm" };
    }

    AudioEncoderHandler::AudioEncoderHandler() NN_NOEXCEPT
        : m_State(State::NotInitialized)
    {
    }

    bool AudioEncoderHandler::IsInitialized() const NN_NOEXCEPT
    {
        return m_State != State::NotInitialized;
    }

    nn::Result AudioEncoderHandler::Initialize() NN_NOEXCEPT
    {
        NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIOENCODER("initializing audio-encoder...\n");
        NN_SDK_REQUIRES(!IsInitialized());

        nn::os::InitializeMutex(&m_Mutex, false, 0);

        m_BitRate      = 0;
        m_SampleRate   = 0;
        m_ChannelCount = 0;
        m_SampleFormat = 0;
        m_pLooper.clear();
        m_pFormat.clear();
        m_pEncoder.clear();
        m_EncoderInputBufferList.clear();
        m_EncoderOutputBufferList.clear();
        m_State = State::Stop;
        NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIOENCODER("initialized audio-encoder\n");
        NN_RESULT_SUCCESS;
    }

    void AudioEncoderHandler::Finalize() NN_NOEXCEPT
    {
        NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIOENCODER("finalizing audio-encoder...\n");
        NN_SDK_REQUIRES(IsInitialized());
        NN_GRCSRV_ASSERT_ABORT(m_State == State::Stop);

        nn::os::FinalizeMutex(&m_Mutex);

        m_BitRate      = 0;
        m_SampleRate   = 0;
        m_ChannelCount = 0;
        m_SampleFormat = 0;
        m_EncoderOutputBufferList.clear();
        m_EncoderInputBufferList.clear();
        m_pEncoder.clear();
        m_pFormat.clear();
        m_pLooper.clear();
        m_State = State::NotInitialized;
        NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIOENCODER("finalized audio-encoder\n");
    }

    nn::Result AudioEncoderHandler::Start(const nn::grc::OffscreenRecordingParameter& param) NN_NOEXCEPT
    {
        nn::os::LockMutex(&m_Mutex);
        NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };

        NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIOENCODER("starting audio-encoder...\n");
        NN_GRCSRV_PROCESS_START();
        NN_GRCSRV_ASSERT_THROW(m_State == State::Stop, nn::grc::ResultInvalidCall());

        NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIOENCODER("  creating looper\n");
        android::sp<android::ALooper> pLooper = new android::ALooper();
        NN_RESULT_THROW_UNLESS(pLooper != nullptr, nn::grc::ResultOutOfMemory());
        pLooper->setName("AudioEnc");

        NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIOENCODER("  starting looper\n");
        NN_RESULT_THROW_UNLESS(android::NO_ERROR == pLooper->start(), nn::grc::ResultInvalidState());

        NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIOENCODER("  creating format\n");
        // see AudioRecorder.cpp in Externals/scoopvideo
        android::sp<android::AMessage> pFormat = new android::AMessage();
        NN_RESULT_THROW_UNLESS(pFormat != nullptr, nn::grc::ResultOutOfMemory());
        pFormat->setString("mime",          MimeType);
        pFormat->setInt32 ("aac-profile",   2); // ProfileLevel: ObjectLC = 2, ObjectHE = 5, ObjectELD = 39;);
        pFormat->setInt32 ("encoder",       true);
        pFormat->setInt32 ("bitrate",       param.audioBitRate);
        pFormat->setInt32 ("sample-rate",   param.audioSampleRate);
        pFormat->setInt32 ("channel-count", param.audioChannelCount);
        pFormat->setString("bitrate-mode",  "CBR");

        NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIOENCODER("  creating encoder\n");
        android::sp<android::MediaCodec> pEncoder = android::MediaCodec::CreateByType(pLooper, MimeType, true);
        NN_RESULT_THROW_UNLESS(pEncoder != nullptr, nn::grc::ResultOutOfMemory());

        NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIOENCODER("  configuring encoder\n");
        NN_RESULT_THROW_UNLESS(android::NO_ERROR == pEncoder->configure(pFormat, NULL, NULL, android::MediaCodec::CONFIGURE_FLAG_ENCODE), nn::grc::ResultInvalidState());

        NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIOENCODER("  starting encoder\n");
        NN_RESULT_THROW_UNLESS(android::NO_ERROR == pEncoder->start(), nn::grc::ResultInvalidState());

        NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIOENCODER("  get encoder buffers\n");
        // 多分失敗するとしたらメモリ不足
        NN_RESULT_THROW_UNLESS(android::NO_ERROR == pEncoder->getInputBuffers(&m_EncoderInputBufferList), nn::grc::ResultOutOfMemory());
        NN_GRCSRV_PROCESS_ROLLBACK(m_EncoderInputBufferList.clear());

        NN_RESULT_THROW_UNLESS(android::NO_ERROR == pEncoder->getOutputBuffers(&m_EncoderOutputBufferList), nn::grc::ResultOutOfMemory());
        NN_GRCSRV_PROCESS_ROLLBACK(m_EncoderOutputBufferList.clear());

        NN_GRCSRV_PROCESS_SUCCESS();
        m_BitRate      = param.audioBitRate;
        m_SampleRate   = param.audioSampleRate;
        m_ChannelCount = param.audioChannelCount;
        m_SampleFormat = param.audioFormat;
        m_pLooper = pLooper;
        m_pFormat = pFormat;
        m_pEncoder = pEncoder;
        m_State = State::Running;
        NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIOENCODER("started audio-encoder\n");
        NN_RESULT_SUCCESS;
    }

    void AudioEncoderHandler::Stop() NN_NOEXCEPT
    {
        NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIOENCODER("stopping audio-encoder...\n");
        NN_GRCSRV_ASSERT_ABORT(m_State == State::Running);

        // これはきっとスレッド安全
        NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIOENCODER("  encoder.stop()\n");
        NN_ABORT_UNLESS_EQUAL(android::NO_ERROR, m_pEncoder->stop());
        NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIOENCODER("  encoder.release()\n");
        NN_ABORT_UNLESS_EQUAL(android::NO_ERROR, m_pEncoder->release());

        // m_State とアトミックに書き換えるために Mutex を取る
        {
            NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIOENCODER("  locking mutex...\n");
            nn::os::LockMutex(&m_Mutex);
            NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };
            NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIOENCODER("  locked mutex\n");

            m_BitRate      = 0;
            m_SampleRate   = 0;
            m_ChannelCount = 0;
            m_SampleFormat = 0;
            m_pEncoder.clear();
            m_pFormat.clear();
            m_pLooper.clear();
            m_EncoderInputBufferList.clear();
            m_EncoderOutputBufferList.clear();
            m_EncoderInputBufferList.clear();
            m_State = State::Stop;
            NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIOENCODER("  unlocked mutex\n");
        }
        NN_GRCSRV_OFFSCRN_LOG_DEV_AUDIOENCODER("stopped audio-encoder\n");
    }

    nn::Result AudioEncoderHandler::AcquireInputBuffer(AudioEncoderInput* pOutValue) NN_NOEXCEPT
    {
        android::sp<android::MediaCodec> pEncoder = m_pEncoder;
        NN_RESULT_THROW_UNLESS(pEncoder != nullptr, nn::grc::ResultInvalidState());

        size_t index = -1;
        // ブロックするので Mutex を取った状態にしないこと
        auto status = pEncoder->dequeueInputBuffer(&index, -1);

        {
            nn::os::LockMutex(&m_Mutex);
            NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };
            NN_RESULT_THROW_UNLESS(m_State == State::Running, nn::grc::ResultInvalidState());

            AudioEncoderInput input = {};
            input.status = status;
            input.index = index;
            if(status == android::NO_ERROR && /* index >= 0 && */ index < m_EncoderInputBufferList.size())
            {
                input.buffer = m_EncoderInputBufferList[index];
            }
            *pOutValue = input;
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result AudioEncoderHandler::TryAcquireInputBuffer(AudioEncoderInput* pOutValue) NN_NOEXCEPT
    {
        android::sp<android::MediaCodec> pEncoder = m_pEncoder;
        NN_RESULT_THROW_UNLESS(pEncoder != nullptr, nn::grc::ResultInvalidState());

        size_t index = -1;
        // ブロックするので Mutex を取った状態にしないこと
        auto status = pEncoder->dequeueInputBuffer(&index, 0);

        NN_RESULT_THROW_UNLESS(index != -1, nn::grc::ResultInternalOffscreenAudioEncoderNoInputBufferAvailable());

        {
            nn::os::LockMutex(&m_Mutex);
            NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };
            NN_RESULT_THROW_UNLESS(m_State == State::Running, nn::grc::ResultInvalidState());

            AudioEncoderInput input = {};
            input.status = status;
            input.index = index;
            if(status == android::NO_ERROR && /* index >= 0 && */ index < m_EncoderInputBufferList.size())
            {
                input.buffer = m_EncoderInputBufferList[index];
            }
            *pOutValue = input;
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result AudioEncoderHandler::QueueInputBuffer(int32_t* pOutStatus, size_t index, size_t offset, size_t size, int64_t timestampUs, uint32_t flags) NN_NOEXCEPT
    {
        android::sp<android::MediaCodec> pEncoder = m_pEncoder;
        NN_RESULT_THROW_UNLESS(pEncoder != nullptr, nn::grc::ResultInvalidState());

        // ブロックするので Mutex を取った状態にしないこと
        auto status = pEncoder->queueInputBuffer(index, offset, size, timestampUs, flags);

        *pOutStatus = status;
        NN_RESULT_SUCCESS;
    }

    nn::Result AudioEncoderHandler::AcquireOutputBuffer(AudioEncoderOutput* pOutValue) NN_NOEXCEPT
    {
        android::sp<android::MediaCodec> pEncoder = m_pEncoder;
        NN_RESULT_THROW_UNLESS(pEncoder != nullptr, nn::grc::ResultInvalidState());

        size_t index;
        size_t offset;
        size_t size;
        int64_t timestamp;
        uint32_t flags;
        // ブロックするので Mutex を取った状態にしないこと
        auto status = pEncoder->dequeueOutputBuffer(&index, &offset, &size, &timestamp, &flags, -1);

        {
            nn::os::LockMutex(&m_Mutex);
            NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };
            NN_RESULT_THROW_UNLESS(m_State == State::Running, nn::grc::ResultInvalidState());

            AudioEncoderOutput output = {};
            output.status = status;
            output.index = index;
            output.offset = offset;
            output.size = size;
            output.timestamp = timestamp;
            output.flags = flags;
            if(status == android::NO_ERROR && index <= m_EncoderOutputBufferList.size())
            {
                output.buffer = m_EncoderOutputBufferList[index];
            }
            *pOutValue = output;
        }
        NN_RESULT_SUCCESS;
    }

    nn::Result AudioEncoderHandler::ReleaseOutputBuffer(int32_t* pOutStatus, size_t index) NN_NOEXCEPT
    {
        android::sp<android::MediaCodec> pEncoder = m_pEncoder;
        NN_RESULT_THROW_UNLESS(pEncoder != nullptr, nn::grc::ResultInvalidState());

        *pOutStatus = pEncoder->releaseOutputBuffer(index);

        NN_RESULT_SUCCESS;
    }

    nn::Result AudioEncoderHandler::GetOutputFormat(int32_t* pOutStatus, android::sp<android::AMessage>* pOutValue) NN_NOEXCEPT
    {
        nn::os::LockMutex(&m_Mutex);
        NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };

        NN_RESULT_THROW_UNLESS(m_State == State::Running, nn::grc::ResultInvalidState());

        *pOutStatus = m_pEncoder->getOutputFormat(pOutValue);
        NN_RESULT_SUCCESS;
    }

}}}}
