﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "grcsrv_TransferMemoryHolder.h"

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_Exchange.h>
#include <memory>
#include <nn/util/util_Optional.h>
#include <nn/grc/grc_Result.h>

namespace nn { namespace grcsrv { namespace detail {

namespace {

void ClearMemory8Aligned(void* p, size_t size) NN_NOEXCEPT
{
    typedef volatile Bit64 Unit;
    NN_ABORT_UNLESS_ALIGNED(p, sizeof(Unit));
    NN_ABORT_UNLESS_ALIGNED(size, sizeof(Unit));
    auto begin = static_cast<Unit*>(p);
    auto end = static_cast<Unit*>(p) + size / sizeof(Unit);
    for (; begin != end; ++begin)
    {
        *begin = 0;
    }
}

}

Result TransferMemoryHolder::Initialize(sf::NativeHandle&& workMemoryHandle, uint64_t workMemorySize) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_Initialized, grc::ResultInvalidState());

    auto success = false;

    void* workMemory;
    os::AttachTransferMemory(&m_WorkMemoryTransferMemory, workMemorySize, workMemoryHandle.GetOsHandle(), false);
    NN_UTIL_SCOPE_EXIT
    {
        if (!success)
        {
            os::DetachTransferMemory(&m_WorkMemoryTransferMemory);
        }
    };

    NN_RESULT_DO(os::MapTransferMemory(&workMemory, &m_WorkMemoryTransferMemory, os::MemoryPermission_None));
    NN_UTIL_SCOPE_EXIT
    {
        if (!success)
        {
            ClearMemory8Aligned(workMemory, workMemorySize);
            os::UnmapTransferMemory(&m_WorkMemoryTransferMemory);
        }
    };

    success = true;
    this->m_Initialized = true;
    this->m_WorkMemoryHandle = std::move(workMemoryHandle);
    this->m_WorkMemory = workMemory;
    this->m_WorkMemorySize = static_cast<size_t>(workMemorySize);
    this->m_AllocateHead = m_WorkMemory;
    this->m_AllocateRest = m_WorkMemorySize;
    NN_RESULT_SUCCESS;
}

void TransferMemoryHolder::Finalize() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Initialized);
    ClearMemory8Aligned(m_WorkMemory, m_WorkMemorySize);
    os::UnmapTransferMemory(&m_WorkMemoryTransferMemory);
    os::DetachTransferMemory(&m_WorkMemoryTransferMemory);
    this->m_Initialized = false;
}

TransferMemoryHolder::~TransferMemoryHolder() NN_NOEXCEPT
{
    if (m_Initialized)
    {
        Finalize();
    }
    NN_SDK_ASSERT(!m_Initialized);
}

void* TransferMemoryHolder::AlignedAllocate(size_t alignment, size_t size) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Initialized);
    auto ret = std::align(alignment, size, this->m_AllocateHead, this->m_AllocateRest);
    if (!ret)
    {
        return nullptr;
    }
    this->m_AllocateHead = reinterpret_cast<void*>(reinterpret_cast<uintptr_t>(m_AllocateHead) + size);
    this->m_AllocateRest -= size;
    return ret;
}

}}}
