﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/grcsrv/grcsrv_Service.h>
#include <nn/grcsrv/grcsrv_GrcServices.sfdl.h>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <utility>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/sf_LmemUtility.h>
#include <nn/grc/grc_Result.h>
#include <nn/grcsrv/grcsrv_MovieFileWriter.h>
#include <nn/grcsrv/grcsrv_RecordingNotificator.h>
#include <nn/grcsrv/grcsrv_OvlnRecordingNotifier.h>
#include <nn/grcsrv/grcsrv_Offscreen.h>

#include "grcsrv_GameMovieTrimmerImpl.h"
#include "grcsrv_ContinuousRecorderImpl.h"
#include "grcsrv_OffscreenRecorderImpl.h"
#include "grcsrv_CapsrvMovieFileWriter.h"
#include "grcsrv_CapsrvMovieFileReader.h"

namespace nn { namespace grcsrv {

namespace {

int g_FlushThreadPriority = os::DefaultThreadPriority;
const char* g_FlushThreadName = "";

int g_TrimThreadPriority = os::DefaultThreadPriority;
const char* g_TrimThreadName = "";

CapsrvMovieFileWriter g_CapsrvMovieFileWriter;
CapsrvMovieFileReader g_CapsrvMovieFileReader;
OvlnRecordingNotifier g_RecordingNotificator;

MovieFileWriter* g_pMovieFileWriter = &g_CapsrvMovieFileWriter;
MovieFileReader* g_pMovieFileReader = &g_CapsrvMovieFileReader;
RecordingNotificator* g_pRecordingNotificator = &g_RecordingNotificator;

MemoryResource* GetGrcObjectMemoryResource() NN_NOEXCEPT
{
    static Bit64 g_HeapMemory[1024];
    NN_FUNCTION_LOCAL_STATIC(lmem::HeapHandle, g_HeapHandle, = lmem::CreateExpHeap(g_HeapMemory, sizeof(g_HeapMemory), lmem::CreationOption_ThreadSafe));
    NN_FUNCTION_LOCAL_STATIC(sf::ExpHeapMemoryResource, g_MemoryResource, {g_HeapHandle});
    return &g_MemoryResource;
}

DeactivatableHolder g_DeactivatableHolder;

}

class GrcServiceImpl
{
public:

    Result CreateContinuousRecorder(sf::Out<sf::SharedPointer<grcsrv::IContinuousRecorder>> pOut, sf::NativeHandle&& workMemoryHandle, uint64_t workMemorySize, const grcsrv::ContinuousRecordingParameter& paramter) NN_NOEXCEPT
    {
        sf::ScopedCurrentMemoryResourceSetter sc(GetGrcObjectMemoryResource());
        auto p = sf::CreateSharedObjectEmplaced<IContinuousRecorder, ContinuousRecorderImpl2>(g_pMovieFileWriter, g_pRecordingNotificator, &g_DeactivatableHolder);
        NN_RESULT_THROW_UNLESS(p, grc::ResultOutOfMemory());
        NN_RESULT_DO(p.GetImpl().Initialize(std::move(workMemoryHandle), workMemorySize, paramter, g_FlushThreadPriority, g_FlushThreadName));
        *pOut = std::move(p);
        NN_RESULT_SUCCESS;
    }

    Result CreateGameMovieTrimmer(sf::Out<sf::SharedPointer<IGameMovieTrimmer>> pOut, sf::NativeHandle&& workMemoryHandle, uint64_t workMemorySize) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(g_DeactivatableHolder.Deactivate(), grc::ResultGrcBusy());
        sf::ScopedCurrentMemoryResourceSetter sc(GetGrcObjectMemoryResource());
        auto p = sf::CreateSharedObjectEmplaced<IGameMovieTrimmer, GameMovieTrimmerImpl>(g_pMovieFileReader, g_pMovieFileWriter);
        NN_RESULT_THROW_UNLESS(p, grc::ResultOutOfMemory());
        NN_RESULT_DO(p.GetImpl().Initialize(std::move(workMemoryHandle), workMemorySize, g_TrimThreadPriority, g_TrimThreadName));
        *pOut = std::move(p);
        NN_RESULT_SUCCESS;
    }

    Result CreateOffscreenRecorder(sf::Out<sf::SharedPointer<IOffscreenRecorder>> pOut, sf::NativeHandle&& workMemoryHandle, uint64_t workMemorySize) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(g_DeactivatableHolder.Deactivate(), grc::ResultGrcBusy());
        sf::ScopedCurrentMemoryResourceSetter sc(GetGrcObjectMemoryResource());
        auto p = sf::CreateSharedObjectEmplaced<IOffscreenRecorder, OffscreenRecorderImpl>();
        NN_RESULT_THROW_UNLESS(p, nn::grc::ResultOutOfMemory());
        NN_RESULT_DO(p.GetImpl().Initialize(std::move(workMemoryHandle), workMemorySize));
        *pOut = std::move(p);
        NN_RESULT_SUCCESS;
    }

    Result CreateMovieMakerForApplication(nn::sf::Out<nn::sf::SharedPointer<nn::grcsrv::IMovieMaker>> outProxy, nn::applet::AppletResourceUserId aruid, ncm::ApplicationId applicationId) NN_NOEXCEPT
    {
        return GetMovieMakerService()->CreateMovieMaker(outProxy, aruid, applicationId);
    }

    Result SetOffscreenRecorderCheckPointForTesting(std::uint64_t value) NN_NOEXCEPT
    {
        offscreen::SetCheckPointForTesting(value);
        NN_RESULT_SUCCESS;
    }

};

void SetFlushThreadInfo(int priority, const char* name) NN_NOEXCEPT
{
    g_FlushThreadPriority = priority;
    g_FlushThreadName = name;
}

void SetMovieFileWriterForDebug(MovieFileWriter* pMovieFileWriter) NN_NOEXCEPT
{
    g_pMovieFileWriter = pMovieFileWriter;
}

sf::UnmanagedServiceObject<IGrcService, GrcServiceImpl> g_GrcServiceImpl;

sf::SharedPointer<IGrcService> GetGrcService() NN_NOEXCEPT
{
    return g_GrcServiceImpl.GetShared();
}

}}
