﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/grcsrv/grcsrv_OvlnRecordingNotifier.h>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/ovln/ovln_SenderForOverlay.h>
#include <nn/ovln/format/ovln_ContinuousRecordingMessage.h>
#include <nn/capsrv/capsrv_Result.h>

namespace nn { namespace grcsrv {

namespace {

class OvlnRecordingNotifierImpl
{
private:

    ovln::SenderForOverlayType m_Sender;

    void SendImpl(const ovln::Message& m) NN_NOEXCEPT
    {
        auto ret = ovln::Send(&m_Sender, m);
        NN_UNUSED(ret);
    }

    void Send(uint32_t dataTag) NN_NOEXCEPT
    {
        ovln::Message m = {};
        m.tag = dataTag;
        SendImpl(m);
    }

    ovln::format::ContinuousRecordingResult ConvertToContinuousRecordingResult(Result result) NN_NOEXCEPT
    {
        NN_RESULT_TRY(result)
            NN_RESULT_CATCH(capsrv::ResultAlbumIsFull)
            {
                return ovln::format::ContinuousRecordingResult::AlbumIsFull;
            }
            NN_RESULT_CATCH(capsrv::ResultAlbumFileCountLimit)
            {
                return ovln::format::ContinuousRecordingResult::AlbumFileCountLimit;
            }
            NN_RESULT_CATCH_ALL
            {
                return ovln::format::ContinuousRecordingResult::AlbumFileSystemError;
            }
        NN_RESULT_END_TRY
        return ovln::format::ContinuousRecordingResult::Success;
    }

    void SendWithData(uint32_t dataTag, ovln::format::ContinuousRecordingResult result, const capsrv::AlbumFileId& albumFileId) NN_NOEXCEPT
    {
        ovln::format::ContinuousRecordingData data = {};
        data.result = result;
        static_assert(sizeof(data._entry) == sizeof(albumFileId), "");
        std::memcpy(&data._entry, &albumFileId, sizeof(albumFileId));
        ovln::Message m = {};
        m.tag = dataTag;
        std::memcpy(&m.data, &data, sizeof(data));
        m.dataSize = sizeof(data);
        SendImpl(m);
    }

public:

    void Initialize() NN_NOEXCEPT
    {
        NN_RESULT_ABORTING_BLOCK
        {
            NN_RESULT_DO(ovln::InitializeSenderLibraryForOverlay());
            NN_RESULT_DO(ovln::InitializeSenderForValue(&m_Sender));
            NN_RESULT_SUCCESS;
        };
    }

    void OnDataError() NN_NOEXCEPT
    {
        Send(ovln::format::ContinuousRecordingErrorDataTag);
    }

    void BeforeFlush(const capsrv::AlbumFileId& albumFileId) NN_NOEXCEPT
    {
        SendWithData(ovln::format::ContinuousRecordingStartedDataTag, ovln::format::ContinuousRecordingResult::Success, albumFileId);
    }

    void AfterFlush(Result result, const capsrv::AlbumFileId& albumFileId) NN_NOEXCEPT
    {
        auto dataTag = result.IsSuccess()
            ? ovln::format::ContinuousRecordingFlushSuccessDataTag
            : ovln::format::ContinuousRecordingFlushFailedDataTag;
        SendWithData(dataTag, ConvertToContinuousRecordingResult(result), albumFileId);
    }

};

OvlnRecordingNotifierImpl g_OvlnRecordingNotifierImpl;

}

void OvlnRecordingNotifier::InitializeStatic() NN_NOEXCEPT
{
    g_OvlnRecordingNotifierImpl.Initialize();
}

void OvlnRecordingNotifier::OnDataError() NN_NOEXCEPT
{
    g_OvlnRecordingNotifierImpl.OnDataError();
}

void OvlnRecordingNotifier::BeforeFlush(const capsrv::AlbumFileId& albumFileId) NN_NOEXCEPT
{
    g_OvlnRecordingNotifierImpl.BeforeFlush(albumFileId);
}

void OvlnRecordingNotifier::AfterFlush(Result result, const capsrv::AlbumFileId& albumFileId) NN_NOEXCEPT
{
    g_OvlnRecordingNotifierImpl.AfterFlush(result, albumFileId);
}

}}
