﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "grcsrv_GameMovieTrimmerImpl.h"

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_Exchange.h>
#include <memory>
#include <mutex>
#include <utility>
#include <nn/grc/grc_CommonTypes.h>
#include <nn/grc/grc_Result.h>
#include <nn/os/os_TransferMemoryApi.h>
#include <nn/grcsrv/grcsrv_MovieFileWriter.h>
#include <nn/util/util_Optional.h>
#include <nn/grcsrv/grcsrv_SetAllocatorForMalloc.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nn/capsrv/capsrv_AlbumControl.h>
#include <nn/capsrv/capsrv_ScreenShotControl.h>
#include <nn/capsrv/capsrv_AlbumAccess.h>
#include <nn/grcsrv/trimming/grcsrv_MovieTrimmer.h>
#include <nerd/scoop/video.h>

namespace nn { namespace grcsrv {

namespace {

std::atomic_bool g_IsUsed{false};
util::optional<mem::StandardAllocator> g_pAllocator;
NN_OS_ALIGNAS_THREAD_STACK char g_TrimThreadStack[1024 * 16];

}

GameMovieTrimmerImpl::GameMovieTrimmerImpl(MovieFileReader* pMovieFileReader, MovieFileWriter* pMovieFileWriter) NN_NOEXCEPT
    : m_pMovieFileReader(pMovieFileReader)
    , m_pMovieFileWriter(pMovieFileWriter)
{
}

Result GameMovieTrimmerImpl::Initialize(sf::NativeHandle&& workMemoryHandle, uint64_t workMemorySize, int trimThreadPriority, const char* trimThreadName) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_Initialized, grc::ResultInvalidState());

    auto success = false;

    NN_RESULT_THROW_UNLESS(!g_IsUsed.exchange(true), grc::ResultOutOfContinuousRecorder());
    NN_UTIL_SCOPE_EXIT
    {
        if (!success)
        {
            g_IsUsed = false;
        }
    };

    nerd::scoop::InitializeResourcesOnce();

    NN_RESULT_DO(m_MemoryHolder.Initialize(std::move(workMemoryHandle), workMemorySize));
    NN_UTIL_SCOPE_EXIT
    {
        if (!success)
        {
            m_MemoryHolder.Finalize();
        }
    };

    const size_t Mi = 1024 * 1024;

    const size_t mallocBufferSize = 24 * Mi;
    auto mallocBuffer = m_MemoryHolder.AlignedAllocate(2 * Mi, mallocBufferSize);
    NN_RESULT_THROW_UNLESS(mallocBuffer, grc::ResultOutOfMemory());
    g_pAllocator.emplace(mallocBuffer, mallocBufferSize);
    SetStandardAllocatorForMalloc(&*g_pAllocator);
    NN_UTIL_SCOPE_EXIT
    {
        if (!success)
        {
            SetStandardAllocatorForMalloc(nullptr);
        }
    };

    const size_t trimBufferSize = 512 * 1024;
    auto trimBuffer = m_MemoryHolder.AlignedAllocate(4096, trimBufferSize);
    NN_RESULT_THROW_UNLESS(trimBuffer, grc::ResultOutOfMemory());

    const size_t metaBufferSize = 512 * 1024;
    auto metaBuffer = m_MemoryHolder.AlignedAllocate(4096, metaBufferSize);
    NN_RESULT_THROW_UNLESS(metaBuffer, grc::ResultOutOfMemory());

    const size_t imageBufferSize = 1280 * 720 * 4;
    auto imageBuffer = m_MemoryHolder.AlignedAllocate(128 * 1024, imageBufferSize);

    NN_RESULT_DO(SubTaskThread::StartThread(g_TrimThreadStack, sizeof(g_TrimThreadStack), trimThreadPriority, trimThreadName));

    success = true;
    this->m_Initialized = true;
    this->m_TrimBuffer = trimBuffer;
    this->m_TrimBufferSize = trimBufferSize;
    this->m_MetaBuffer = metaBuffer;
    this->m_MetaBufferSize = metaBufferSize;
    this->m_ImageBuffer = imageBuffer;
    this->m_ImageBufferSize = imageBufferSize;
    std::memset(imageBuffer, 0, imageBufferSize);
    NN_RESULT_SUCCESS;
}

void GameMovieTrimmerImpl::FinalizeImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Initialized);

    SubTaskThread::StopThread();

    SetStandardAllocatorForMalloc(nullptr);
    g_pAllocator->WalkAllocatedBlocks([](void*, size_t, void*) -> int
    {
        #if !defined(NN_SDK_BUILD_RELEASE)
        g_pAllocator->Dump();
        #endif
        NN_ABORT();
    }, nullptr);
    g_pAllocator = util::nullopt;

    m_MemoryHolder.Finalize();
    g_IsUsed = false;
}

GameMovieTrimmerImpl::~GameMovieTrimmerImpl() NN_NOEXCEPT
{
    if (m_Initialized)
    {
        FinalizeImpl();
    }
}

Result GameMovieTrimmerImpl::DoTaskImpl() NN_NOEXCEPT
{
    capsrv::ScreenShotAttribute attribute;
    {
        capsrv::AlbumMovieReadStreamHandle streamHandle;
        NN_RESULT_DO(capsrv::OpenAlbumMovieReadStream(&streamHandle, m_InFileId));
        NN_UTIL_SCOPE_EXIT
        {
            capsrv::CloseAlbumMovieReadStream(streamHandle);
        };
        NN_RESULT_DO(capsrv::ReadAttributeFromAlbumMovieReadStream(&attribute, streamHandle));
    }
    std::unique_ptr<MovieFileReadableHandle> readableHandle;
    NN_RESULT_DO(m_pMovieFileReader->Create(&readableHandle, m_InFileId));

    capsrv::AlbumFileId outFileId;
    NN_ABORT_UNLESS_RESULT_SUCCESS(capsrv::GenerateCurrentAlbumFileId(&outFileId, m_InFileId.applicationId, capsrv::AlbumFileContents_Movie));

    capsrv::movie::MovieMetaDataBuilder metaBuilder;
    metaBuilder.Initialize(nn::capsrv::movie::MovieMetaDataVersion_1, m_MetaBuffer, m_MetaBufferSize, outFileId);
    NN_UTIL_SCOPE_EXIT
    {
        metaBuilder.Finalize();
    };

    metaBuilder.SetDescription(capsrv::AlbumFileDescription_MovieTrimmed);
    metaBuilder.SetImageDataRgba(m_ImageBuffer, m_ImageBufferSize, 1280, 720);

    auto dataWriter = MakeFunctorDataWriter([&](fs::FileHandle handle) -> Result
    {
        trimming::MovieTrimmer trimmer;
        NN_RESULT_DO(trimmer.Initialize(readableHandle->GetFileHandle(), m_TrimBuffer, m_TrimBufferSize));
        NN_RESULT_DO(trimmer.ReadHeader());
        int frameCount;
        NN_RESULT_DO(trimmer.Trim(&frameCount, handle, m_BeginIndex, m_EndIndex, attribute));
        NN_RESULT_DO(fs::FlushFile(handle));
        auto timeMs = static_cast<int>(int64_t{frameCount} * 1000 * attribute.frameRateNumerator / attribute.frameRateDenominator);
        metaBuilder.SetAttribute(frameCount, attribute.frameRateNumerator, attribute.frameRateDenominator, timeMs, attribute.keyFrameInterval, attribute.IsCopyrightImageComposited(), capsrv::ScreenShotSize_1280x720, attribute.orientation);
        NN_RESULT_SUCCESS;
    });
    NN_RESULT_DO(m_pMovieFileWriter->Write(outFileId, &dataWriter, &metaBuilder));

    this->m_OutFileId = outFileId;
    NN_RESULT_SUCCESS;
}

void GameMovieTrimmerImpl::DoTask() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Initialized);
    this->m_Result = DoTaskImpl();
}

void GameMovieTrimmerImpl::DoCancel() NN_NOEXCEPT
{
    // nop
}

Result GameMovieTrimmerImpl::GetNotTrimmingEvent(sf::Out<sf::NativeHandle> pOutSystemEventHandle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Initialized);
    return SubTaskThread::GetNotRunningEvent(pOutSystemEventHandle);
}

Result GameMovieTrimmerImpl::BeginTrim(const capsrv::AlbumFileId& id, int beginIndex, int endIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Initialized);
    NN_RESULT_THROW_UNLESS(0 <= beginIndex, grc::ResultInvalidCall());
    NN_RESULT_THROW_UNLESS(beginIndex <= endIndex, grc::ResultInvalidCall());
    this->m_InFileId = id;
    this->m_BeginIndex = beginIndex;
    this->m_EndIndex = endIndex;
    this->m_Result = grc::ResultInvalidCall();
    return SubTaskThread::Start();
}

Result GameMovieTrimmerImpl::BeginTrim(const GameMovieId& id, int beginIndex, int endIndex) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Initialized);
    capsrv::AlbumFileId albumFileId;
    static_assert(sizeof(albumFileId) <= sizeof(id), "");
    std::memcpy(&albumFileId, &id, sizeof(albumFileId));
    return BeginTrim(albumFileId, beginIndex, endIndex);
}

Result GameMovieTrimmerImpl::EndTrim(sf::Out<GameMovieId> pOut) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Initialized);
    NN_RESULT_THROW_UNLESS(!SubTaskThread::IsRunning(), grc::ResultInvalidCall());
    NN_RESULT_DO(m_Result);
    std::memcpy(pOut.GetPointer(), &m_OutFileId, sizeof(m_OutFileId));
    NN_RESULT_SUCCESS;
}

Result GameMovieTrimmerImpl::SetThumbnailRgba(const sf::InBuffer& imageData, int width, int height) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(width == 1280, grc::ResultInvalidCall());
    NN_RESULT_THROW_UNLESS(height == 720, grc::ResultInvalidCall());
    NN_RESULT_THROW_UNLESS(imageData.GetSize() >= 1280 * 720 * 4, grc::ResultInvalidCall());
    std::memcpy(m_ImageBuffer, imageData.GetPointerUnsafe(), 1280 * 720 * 4);
    NN_RESULT_SUCCESS;
}

}}
