﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/grc/grc_Application.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_Optional.h>
#include <nn/sf/sf_ShimLibraryUtility.h>
#include <nn/applet/applet_Apis.h>
#include <nn/grcsrv/grcsrv_GrcServices.sfdl.h>
#include <nn/grc/grc_Result.h>
#include <nn/capsrv/capsrv_ApplicationData.h>

#include <binder/HOSServiceManager.h>
#include <utils/String8.h>
#include <nvnflinger_service.h>

namespace nn{ namespace grc{

    namespace {
        nn::sf::SharedPointer<nn::grcsrv::IMovieMaker> g_pMovieMakerProxy;

        struct OffscreenLayer
        {
            uint64_t layerHandle;
            android::sp<android::IGraphicBufferProducer> pProducer;
            android::sp<android::Surface> pSurface;
        };
        nn::util::optional<OffscreenLayer> g_BoundOffscreenLayer;
    }

    namespace dev{

        nn::Result InitializeMovieMakerProxy(nn::sf::SharedPointer<nn::grcsrv::IMovieMaker>&& pProxy) NN_NOEXCEPT
        {
            g_pMovieMakerProxy = std::move(pProxy);
            NN_RESULT_SUCCESS;
        }

        void FinalizeMovieMakerProxy() NN_NOEXCEPT
        {
            g_pMovieMakerProxy.Reset();
        }

        nn::Result InitializeMovieVideoProxy() NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(g_pMovieMakerProxy);
            nn::sf::SharedPointer<nns::hosbinder::IHOSBinderDriver> pProxy;
            NN_RESULT_DO(g_pMovieMakerProxy->CreateVideoProxy(&pProxy));
            android::IPCThreadState::self();
            NN_RESULT_THROW_UNLESS(
                android::defaultHOSServiceManager()->addServiceProxy(android::String8("grc"), pProxy) == android::NO_ERROR,
                nn::grc::ResultInvalidState()
            );
            NN_RESULT_SUCCESS;
        }

        void FinalizeMovieVideoProxy() NN_NOEXCEPT
        {
            NN_ABORT_UNLESS_GREATER_EQUAL(android::defaultHOSServiceManager()->removeServiceProxy(android::String8("grc")), 0);
        }
    }

    nn::Result InitializeForApplication(nn::sf::SharedPointer<nn::grcsrv::IMovieMaker>&& pMovieMakerProxy) NN_NOEXCEPT
    {
        bool isSuccess = false;

        NN_RESULT_DO(dev::InitializeMovieMakerProxy(std::move(pMovieMakerProxy)));
        NN_UTIL_SCOPE_EXIT
        {
            if(!isSuccess)
            {
                dev::FinalizeMovieMakerProxy();
            }
        };

        NN_RESULT_DO(dev::InitializeMovieVideoProxy());
        NN_UTIL_SCOPE_EXIT
        {
            if(!isSuccess)
            {
                dev::FinalizeMovieVideoProxy();
            }
        };

        isSuccess = true;
        NN_RESULT_SUCCESS;
    }

    void FinalizeForApplication() NN_NOEXCEPT
    {
        dev::CloseAllMovieLayers();

        dev::FinalizeMovieVideoProxy();
        dev::FinalizeMovieMakerProxy();
    }

    //-------------------------------------------------------------------------------

    namespace dev{

        nn::Result OpenOffscreenLayer(int32_t* pOutProducerHandle, uint64_t layerHandle) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(g_pMovieMakerProxy);
            return g_pMovieMakerProxy->OpenOffscreenLayer(pOutProducerHandle, layerHandle);
        }

        void CloseOffscreenLayer(uint64_t layerHandle) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(g_pMovieMakerProxy);
            (void)g_pMovieMakerProxy->CloseOffscreenLayer(layerHandle);
        }

        nn::Result RegisterOffscreenLayer(uint64_t layerHandle, int32_t producerHandle, int width, int height) NN_NOEXCEPT
        {
            NN_RESULT_THROW_UNLESS(g_BoundOffscreenLayer == nn::util::nullopt, nn::grc::ResultOutOfResource());

            android::sp<android::IBinder> pBinder;
            {
                // 直接 IBinder を作れないので Parcel 経由で作る。
                flat_binder_object obj = {};
                obj.type = BINDER_TYPE_HANDLE;
                obj.flags = 0x7f | FLAT_BINDER_FLAG_ACCEPTS_FDS;
                obj.handle = producerHandle;
                obj.cookie = 0;
                std::strncpy(obj.service_name, "grc", sizeof(obj.service_name));

                binder_size_t table = {};
                table = 0;

                android::Parcel parcel;
                parcel.ipcSetDataReference(
                    reinterpret_cast<uint8_t*>(&obj),
                    sizeof(obj),
                    &table,
                    1,
                    [](android::Parcel*, const uint8_t*, size_t, const binder_size_t*, size_t, void*){},
                    nullptr
                );

                pBinder = parcel.readStrongBinder();
                parcel.freeData();
            }
            auto pProducer = android::interface_cast<android::IGraphicBufferProducer>(pBinder);
            auto pSurface = android::SurfaceControl::getSurface(pProducer);
            NN_RESULT_THROW_UNLESS(pProducer != nullptr && pSurface != nullptr, nn::grc::ResultOutOfResource());

            {
                auto pNativeWindow = static_cast<ANativeWindow*>(pSurface.get());
                native_window_set_buffers_user_dimensions(pNativeWindow, width, height);
            }

            OffscreenLayer value = {};
            value.layerHandle = layerHandle;
            value.pProducer = pProducer;
            value.pSurface  = pSurface;
            g_BoundOffscreenLayer = value;
            NN_RESULT_SUCCESS;
        }

        void UnregisterOffscreenLayer(uint64_t layerHandle) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS(g_BoundOffscreenLayer != nn::util::nullopt);
            NN_ABORT_UNLESS(g_BoundOffscreenLayer->layerHandle == layerHandle);

            g_BoundOffscreenLayer = nn::util::nullopt;
        }

        void CloseAllMovieLayers() NN_NOEXCEPT
        {
            if(g_BoundOffscreenLayer != nn::util::nullopt)
            {
                CloseMovieLayer(reinterpret_cast<MovieLayer*>(&*g_BoundOffscreenLayer));
            }
        }

    }

    nn::Result OpenMovieLayer(MovieLayer** pOutLayer, uint64_t layerHandle, int width, int height) NN_NOEXCEPT
    {
        bool isSuccess = false;

        int32_t producerHandle = 0;
        NN_RESULT_DO(dev::OpenOffscreenLayer(&producerHandle, layerHandle));
        NN_UTIL_SCOPE_EXIT
        {
            if(!isSuccess)
            {
                dev::CloseOffscreenLayer(layerHandle);
            }
        };

        NN_RESULT_DO(dev::RegisterOffscreenLayer(layerHandle, producerHandle, width, height));
        NN_UTIL_SCOPE_EXIT
        {
            if(!isSuccess)
            {
                dev::UnregisterOffscreenLayer(layerHandle);
            }
        };

        isSuccess = true;
        *pOutLayer = reinterpret_cast<MovieLayer*>(&*g_BoundOffscreenLayer);
        NN_RESULT_SUCCESS;
    }

    void CloseMovieLayer(MovieLayer* pLayer) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(g_BoundOffscreenLayer != nn::util::nullopt);
        NN_ABORT_UNLESS_EQUAL(pLayer, reinterpret_cast<MovieLayer*>(&*g_BoundOffscreenLayer));

        auto layerHandle = g_BoundOffscreenLayer->layerHandle;
        dev::UnregisterOffscreenLayer(layerHandle);
        dev::CloseOffscreenLayer(layerHandle);
    }

    nn::Result GetVideoNativeWindow(nn::vi::NativeWindowHandle* pOutNativeWindow, MovieLayer* pLayer) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(g_BoundOffscreenLayer != nn::util::nullopt, nn::grc::ResultInvalidCall());
        NN_RESULT_THROW_UNLESS(pLayer == reinterpret_cast<MovieLayer*>(&*g_BoundOffscreenLayer), nn::grc::ResultInvalidCall());

        *pOutNativeWindow = reinterpret_cast<nn::vi::NativeWindowHandle>(static_cast<ANativeWindow*>(g_BoundOffscreenLayer->pSurface.get()));
        NN_RESULT_SUCCESS;
    }

    nn::Result StartMovieFile(MovieLayer* pLayer, const MovieMakerMovieParameter& param) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pMovieMakerProxy);
        //NN_SDK_REQUIRES(param.CheckParameterValid());
        NN_RESULT_THROW_UNLESS(g_BoundOffscreenLayer != nn::util::nullopt, nn::grc::ResultInvalidCall());
        NN_RESULT_THROW_UNLESS(pLayer == reinterpret_cast<MovieLayer*>(&*g_BoundOffscreenLayer), nn::grc::ResultInvalidCall());

        nn::grcsrv::OffscreenRecordingParameter paramData = {};
        NN_STATIC_ASSERT(sizeof(paramData) == sizeof(param));
        std::memcpy(&paramData, &param, sizeof(param));

        auto layerHandle = g_BoundOffscreenLayer->layerHandle;
        return g_pMovieMakerProxy->StartOffscreenRecordingEx(layerHandle, paramData);
    }

    nn::Result RequestFinishMovieFile(MovieLayer* pLayer) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pMovieMakerProxy);
        NN_RESULT_THROW_UNLESS(g_BoundOffscreenLayer != nn::util::nullopt, nn::grc::ResultInvalidCall());
        NN_RESULT_THROW_UNLESS(pLayer == reinterpret_cast<MovieLayer*>(&*g_BoundOffscreenLayer), nn::grc::ResultInvalidCall());

        auto layerHandle = g_BoundOffscreenLayer->layerHandle;
        return g_pMovieMakerProxy->RequestOffscreenRecordingFinishReady(layerHandle);
    }

    nn::Result CompleteFinishMovieFile(MovieLayer* pLayer, const sf::InBuffer& userData, const sf::InBuffer& thumbnailImage, album::ImageSize imageSize) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pMovieMakerProxy);
        NN_SDK_REQUIRES(imageSize == album::ImageSize_1280x720);
        NN_RESULT_THROW_UNLESS(g_BoundOffscreenLayer != nn::util::nullopt, nn::grc::ResultInvalidCall());
        NN_RESULT_THROW_UNLESS(pLayer == reinterpret_cast<MovieLayer*>(&*g_BoundOffscreenLayer), nn::grc::ResultInvalidCall());

        auto layerHandle = g_BoundOffscreenLayer->layerHandle;
        return g_pMovieMakerProxy->CompleteOffscreenRecordingFinishEx0(layerHandle, userData, thumbnailImage, album::GetImageWidth(imageSize), album::GetImageHeight(imageSize));
    }

    nn::Result AbortMovieFile(MovieLayer* pLayer) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pMovieMakerProxy);
        NN_RESULT_THROW_UNLESS(g_BoundOffscreenLayer != nn::util::nullopt, nn::grc::ResultInvalidCall());
        NN_RESULT_THROW_UNLESS(pLayer == reinterpret_cast<MovieLayer*>(&*g_BoundOffscreenLayer), nn::grc::ResultInvalidCall());

        auto layerHandle = g_BoundOffscreenLayer->layerHandle;
        return g_pMovieMakerProxy->AbortOffscreenRecording(layerHandle);
    }

    nn::Result CheckMovieFileError(MovieLayer* pLayer) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pMovieMakerProxy);
        NN_RESULT_THROW_UNLESS(g_BoundOffscreenLayer != nn::util::nullopt, nn::grc::ResultInvalidCall());
        NN_RESULT_THROW_UNLESS(pLayer == reinterpret_cast<MovieLayer*>(&*g_BoundOffscreenLayer), nn::grc::ResultInvalidCall());

        auto layerHandle = g_BoundOffscreenLayer->layerHandle;
        return g_pMovieMakerProxy->GetOffscreenLayerError(layerHandle);
    }

    nn::Result EncodeMovieAudioSample(size_t* pOutEncodedSize, MovieLayer* pLayer, const void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pMovieMakerProxy);
        NN_RESULT_THROW_UNLESS(g_BoundOffscreenLayer != nn::util::nullopt, nn::grc::ResultInvalidCall());
        NN_RESULT_THROW_UNLESS(pLayer == reinterpret_cast<MovieLayer*>(&*g_BoundOffscreenLayer), nn::grc::ResultInvalidCall());

        auto layerHandle = g_BoundOffscreenLayer->layerHandle;
        uint64_t outSize = 0;
        NN_RESULT_DO(g_pMovieMakerProxy->EncodeOffscreenLayerAudioSample(&outSize, layerHandle, nn::sf::InBuffer(reinterpret_cast<const char*>(buffer), size)));
        *pOutEncodedSize = static_cast<size_t>(outSize);
        NN_RESULT_SUCCESS;
    }

    nn::Result InitializeFinishMovieFileReadyEvent(nn::os::SystemEventType* pEvent, MovieLayer* pLayer) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pMovieMakerProxy);
        NN_RESULT_THROW_UNLESS(g_BoundOffscreenLayer != nn::util::nullopt, nn::grc::ResultInvalidCall());
        NN_RESULT_THROW_UNLESS(pLayer == reinterpret_cast<MovieLayer*>(&*g_BoundOffscreenLayer), nn::grc::ResultInvalidCall());

        auto layerHandle = g_BoundOffscreenLayer->layerHandle;
        nn::sf::NativeHandle h;
        NN_RESULT_DO(g_pMovieMakerProxy->GetOffscreenLayerRecordingFinishReadyEvent(&h, layerHandle));

        nn::os::AttachReadableHandleToSystemEvent(pEvent, h.GetOsHandle(), h.IsManaged(), nn::os::EventClearMode_ManualClear);
        h.Detach();

        NN_RESULT_SUCCESS;
    }

    void FinalizeFinishMovieFileReadyEvent(nn::os::SystemEventType* pEvent, MovieLayer* pLayer) NN_NOEXCEPT
    {
        NN_UNUSED(pLayer);
        nn::os::DestroySystemEvent(pEvent);
    }

    nn::Result InitializeMovieAudioEncodeReadyEvent(nn::os::SystemEventType* pEvent, MovieLayer* pLayer) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pMovieMakerProxy);
        NN_RESULT_THROW_UNLESS(g_BoundOffscreenLayer != nn::util::nullopt, nn::grc::ResultInvalidCall());
        NN_RESULT_THROW_UNLESS(pLayer == reinterpret_cast<MovieLayer*>(&*g_BoundOffscreenLayer), nn::grc::ResultInvalidCall());

        auto layerHandle = g_BoundOffscreenLayer->layerHandle;
        nn::sf::NativeHandle h;
        NN_RESULT_DO(g_pMovieMakerProxy->GetOffscreenLayerAudioEncodeReadyEvent(&h, layerHandle));

        nn::os::AttachReadableHandleToSystemEvent(pEvent, h.GetOsHandle(), h.IsManaged(), nn::os::EventClearMode_ManualClear);
        h.Detach();

        NN_RESULT_SUCCESS;
    }

    void FinalizeMovieAudioEncodeReadyEvent(nn::os::SystemEventType* pEvent, MovieLayer* pLayer) NN_NOEXCEPT
    {
        NN_UNUSED(pLayer);
        nn::os::DestroySystemEvent(pEvent);
    }

}}

