﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/gfx/util/gfx_TextureCompressor.h>

namespace nn {
namespace gfx {
namespace util {

#if NN_GFX_IS_TARGET_GL

namespace {

    const int BlockSize = 4;
    const int InvalidIndex = -1;

    nn::gfx::Texture            workTexture[TextureCompressor::SupportedFormatCount];
    nn::gfx::Buffer             workBuffer[TextureCompressor::SupportedFormatCount];
    nn::gfx::ColorTargetView    workColorTargetView[TextureCompressor::SupportedFormatCount];

    int workTextureWidth[TextureCompressor::SupportedFormatCount];
    int workTextureHeight[TextureCompressor::SupportedFormatCount];

    enum SupportedFormat
    {
        SupportedFormat_Bc1_Unorm,
        SupportedFormat_Bc1_UnormSrgb,
        SupportedFormat_Bc2_Unorm,
        SupportedFormat_Bc2_UnormSrgb,
        SupportedFormat_Bc3_Unorm,
        SupportedFormat_Bc3_UnormSrgb,
        SupportedFormat_Bc4_Unorm,
        SupportedFormat_Bc4_Snorm,
        SupportedFormat_Bc5_Unorm,
        SupportedFormat_Bc5_Snorm,
        SupportedFormat_End
    };
    NN_STATIC_ASSERT(SupportedFormat::SupportedFormat_End <= TextureCompressor::SupportedFormatCount);

    int GetSupportedFormatIndex(nn::gfx::ImageFormat compressionFormat)
    {
        switch (compressionFormat)
        {
        case nn::gfx::ImageFormat_Bc1_Unorm:
            return SupportedFormat_Bc1_Unorm;
        case nn::gfx::ImageFormat_Bc1_UnormSrgb:
            return SupportedFormat_Bc1_UnormSrgb;
        case nn::gfx::ImageFormat_Bc2_Unorm:
            return SupportedFormat_Bc2_Unorm;
        case nn::gfx::ImageFormat_Bc2_UnormSrgb:
            return SupportedFormat_Bc2_UnormSrgb;
        case nn::gfx::ImageFormat_Bc3_Unorm:
            return SupportedFormat_Bc3_Unorm;
        case nn::gfx::ImageFormat_Bc3_UnormSrgb:
            return SupportedFormat_Bc3_UnormSrgb;
        case nn::gfx::ImageFormat_Bc4_Unorm:
            return SupportedFormat_Bc4_Unorm;
        case nn::gfx::ImageFormat_Bc4_Snorm:
            return SupportedFormat_Bc4_Snorm;
        case nn::gfx::ImageFormat_Bc5_Unorm:
            return SupportedFormat_Bc5_Unorm;
        case nn::gfx::ImageFormat_Bc5_Snorm:
            return SupportedFormat_Bc5_Snorm;
        default:
            return InvalidIndex;
        }
    }

    nn::gfx::ImageFormat GetWorkImageFormat(nn::gfx::ImageFormat compressionFormat)
    {
        NN_SDK_ASSERT(TextureCompressor::IsSupportedFormat(compressionFormat));

        switch (compressionFormat)
        {
        case nn::gfx::ImageFormat_Bc1_Unorm:
        case nn::gfx::ImageFormat_Bc1_UnormSrgb:
        case nn::gfx::ImageFormat_Bc4_Unorm:
        case nn::gfx::ImageFormat_Bc4_Snorm:
            return nn::gfx::ImageFormat_R16_G16_B16_A16_Uint;

        case nn::gfx::ImageFormat_Bc2_Unorm:
        case nn::gfx::ImageFormat_Bc2_UnormSrgb:
        case nn::gfx::ImageFormat_Bc3_Unorm:
        case nn::gfx::ImageFormat_Bc3_UnormSrgb:
        case nn::gfx::ImageFormat_Bc5_Unorm:
        case nn::gfx::ImageFormat_Bc5_Snorm:
            return nn::gfx::ImageFormat_R32_G32_B32_A32_Uint;

        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    int GetWorkBufferTexelSize(nn::gfx::ImageFormat workImageFormat)
    {
        switch (workImageFormat)
        {
        case nn::gfx::ImageFormat_R16_G16_B16_A16_Uint:
            return 8;

        case nn::gfx::ImageFormat_R32_G32_B32_A32_Uint:
            return 16;

        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    void InitializeWorkTexture(nn::gfx::Device* pDevice, const int workWidth, const int workHeight, nn::gfx::ImageFormat compressionFormat)
    {
        NN_SDK_ASSERT_NOT_NULL(pDevice);
        NN_SDK_ASSERT(nn::gfx::IsInitialized(*pDevice));

        int index = GetSupportedFormatIndex(compressionFormat);
        nn::gfx::ImageFormat workSpaceFormat = GetWorkImageFormat(compressionFormat);
        {
            nn::gfx::Texture::InfoType info;
            info.SetDefault();
            info.SetGpuAccessFlags(nn::gfx::GpuAccess_Texture);
            info.SetImageFormat(workSpaceFormat);
            info.SetWidth(workWidth);
            info.SetHeight(workHeight);

            NN_SDK_ASSERT(!nn::gfx::Texture::IsMemoryPoolRequired);
            workTexture[index].Initialize(pDevice, info, NULL, 0, 0);
        }

        {
            nn::gfx::ColorTargetView::InfoType info;
            info.SetDefault();
            info.SetImageFormat(workSpaceFormat);
            info.SetTexturePtr(&workTexture[index]);
            workColorTargetView[index].Initialize(pDevice, info);
        }
    }

    void InitializeWorkBuffer(nn::gfx::Device* pDevice, int workWidth, int workHeight, nn::gfx::ImageFormat compressionFormat)
    {
        int index = GetSupportedFormatIndex(compressionFormat);
        nn::gfx::ImageFormat workSpaceFormat = GetWorkImageFormat(compressionFormat);
        size_t size = GetWorkBufferTexelSize(workSpaceFormat);

        nn::gfx::Buffer::InfoType info;
        info.SetDefault();
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_Read | nn::gfx::GpuAccess_Write);
        info.SetSize(workWidth * workHeight * size);

        NN_SDK_ASSERT(!nn::gfx::Buffer::IsMemoryPoolRequired);
        workBuffer[index].Initialize(pDevice, info, NULL, 0, 0);
    }
}

TextureCompressor::TextureCompressor() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!IsInitialized());
}

TextureCompressor::~TextureCompressor() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!IsInitialized());
}

void TextureCompressor::Initialize(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!IsInitialized());
    NN_SDK_REQUIRES_NOT_NULL(pDevice);
    NN_SDK_REQUIRES(nn::gfx::IsInitialized(*pDevice));

    memset(m_pShaders, 0, sizeof(m_pShaders));

    m_CommonState.Initialize(pDevice);
    NN_SDK_ASSERT(IsInitialized());

    const nn::gfx::ImageFormat compressibleFormats[] = {
        nn::gfx::ImageFormat_Bc1_Unorm,
        nn::gfx::ImageFormat_Bc1_UnormSrgb,
        nn::gfx::ImageFormat_Bc2_Unorm,
        nn::gfx::ImageFormat_Bc2_UnormSrgb,
        nn::gfx::ImageFormat_Bc3_Unorm,
        nn::gfx::ImageFormat_Bc3_UnormSrgb,
        nn::gfx::ImageFormat_Bc4_Unorm,
        nn::gfx::ImageFormat_Bc4_Snorm,
        nn::gfx::ImageFormat_Bc5_Unorm,
        nn::gfx::ImageFormat_Bc5_Snorm,
    };

    for (int i = 0; i < SupportedFormatCount; i++)
    {
        NN_SDK_ASSERT(IsSupportedFormat(compressibleFormats[i]));
        if (!nn::gfx::IsInitialized(workTexture[i]))
        {
            workTextureWidth[i] = 1024 / BlockSize;
            workTextureHeight[i] = 1024 / BlockSize;
            InitializeWorkTexture(pDevice, workTextureWidth[i], workTextureHeight[i], compressibleFormats[i]);
            InitializeWorkBuffer(pDevice, workTextureWidth[i], workTextureHeight[i], compressibleFormats[i]);
        }
    }
}

void TextureCompressor::Finalize() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());

    for (int i = 0; i < SupportedFormatCount; i++)
    {
        if (nn::gfx::IsInitialized(workTexture[i]))
        {
            workColorTargetView[i].Finalize(m_CommonState.GetDevice());
            workBuffer[i].Finalize(m_CommonState.GetDevice());
            workTexture[i].Finalize(m_CommonState.GetDevice());
        }
    }

    m_CommonState.Finalize();
}

bool TextureCompressor::IsInitialized() const NN_NOEXCEPT
{
    return m_CommonState.IsInitialized();
}

void TextureCompressor::RegisterCompressionShader(nn::gfx::ImageFormat targetFormat, const nn::gfx::Shader* pShader) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(IsSupportedFormat(targetFormat));
    NN_SDK_REQUIRES_NOT_NULL(pShader);
    NN_SDK_REQUIRES(nn::gfx::IsInitialized(*pShader));

    int index = GetSupportedFormatIndex(targetFormat);
    m_pShaders[index] = pShader;
}

void TextureCompressor::UnregisterCompressionShader(nn::gfx::ImageFormat targetFormat) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(IsSupportedFormat(targetFormat));
    int index = GetSupportedFormatIndex(targetFormat);
    m_pShaders[index] = NULL;
}

bool TextureCompressor::IsSupportedFormat(nn::gfx::ImageFormat format) NN_NOEXCEPT
{
    return GetSupportedFormatIndex(format) != InvalidIndex;
}

const nn::gfx::Shader* TextureCompressor::GetShader(nn::gfx::ImageFormat imageFormat) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_CommonState.IsInitialized());
    NN_SDK_REQUIRES(IsSupportedFormat(imageFormat));

    int index = GetSupportedFormatIndex(imageFormat);
    return m_pShaders[index];
}

void TextureCompressor::MakeCommand(
    nn::gfx::CommandBuffer* pCommandBuffer,
    const nn::gfx::DescriptorSlot& srcTextureDescriptorSlot,
    const nn::gfx::DescriptorSlot& samplerDescriptorSlot,
    const TextureCompressorTargetInfo& targetTextureInfo
) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES_NOT_NULL(pCommandBuffer);
    NN_SDK_REQUIRES(nn::gfx::IsInitialized(*pCommandBuffer));
    NN_SDK_REQUIRES_NOT_NULL(targetTextureInfo.GetTexture());

    const nn::gfx::Texture::InfoType& textureInfo = targetTextureInfo.GetTextureInfo();
    NN_SDK_REQUIRES(IsSupportedFormat(textureInfo.GetImageFormat()));
    NN_SDK_REQUIRES_RANGE(textureInfo.GetArrayLength(), 0, 0x10000);
    NN_SDK_REQUIRES_RANGE(textureInfo.GetMipCount(), 0, 0x10);
    NN_SDK_REQUIRES_NOT_NULL(GetShader(textureInfo.GetImageFormat()));

    //各種ステートを書き込む
    m_CommonState.MakeCommand(pCommandBuffer, srcTextureDescriptorSlot, samplerDescriptorSlot);

    const nn::gfx::Shader* pShader = GetShader(textureInfo.GetImageFormat());
    pCommandBuffer->SetShader(pShader, nn::gfx::ShaderStageBit_All);
    nn::gfx::Device* pDevice = m_CommonState.GetDevice();


    const int mipCount = textureInfo.GetMipCount();
    const int arraySize = (textureInfo.GetArrayLength() == 0) ? 1 : textureInfo.GetArrayLength();

    int requiredTextureWidth = textureInfo.GetWidth();
    requiredTextureWidth = std::max((requiredTextureWidth + BlockSize - 1) / BlockSize, 1);

    int requiredTextureHeight = textureInfo.GetHeight();
    requiredTextureHeight = std::max((requiredTextureHeight + BlockSize - 1) / BlockSize, 1);


    // ワーク用 テクスチャとバッファの再生成を行う
    int index = GetSupportedFormatIndex(textureInfo.GetImageFormat());
    if (requiredTextureWidth > workTextureWidth[index] || requiredTextureHeight > workTextureHeight[index])
    {
        workColorTargetView[index].Finalize(pDevice);
        workBuffer[index].Finalize(pDevice);
        workTexture[index].Finalize(pDevice);

        InitializeWorkTexture(pDevice, requiredTextureWidth, requiredTextureHeight, textureInfo.GetImageFormat());
        InitializeWorkBuffer(pDevice, requiredTextureWidth, requiredTextureHeight, textureInfo.GetImageFormat());

        //ワークのサイズを更新.
        workTextureWidth[index] = requiredTextureWidth;
        workTextureHeight[index] = requiredTextureHeight;
    }


    //レンダーターゲットを切り替え
    nn::gfx::ColorTargetView* pColorTargetView = &workColorTargetView[index];
    pCommandBuffer->SetRenderTargets(1, &pColorTargetView, NULL);


    for (int layer = 0; layer < arraySize; layer++)
    {
        for (int miplevel = 0; miplevel < mipCount; miplevel++)
        {
            // mip に応じたサイズを算出
            int width = std::max(textureInfo.GetWidth() >> miplevel, 1);
            int height = std::max(textureInfo.GetHeight() >> miplevel, 1);

            // BC1 のブロックサイズにあわせて調節
            int compressedwidth = std::max((width + BlockSize - 1) / BlockSize, 1);
            int compressedheight = std::max((height + BlockSize - 1) / BlockSize, 1);


            // コピー用リージョンを初期化
            nn::gfx::TextureCopyRegion bufferCopyRegion;
            nn::gfx::TextureCopyRegion destTextureCopyRegion;
            {
                bufferCopyRegion.SetDefault();
                bufferCopyRegion.SetWidth(compressedwidth);
                bufferCopyRegion.SetHeight(compressedheight);

                destTextureCopyRegion.SetDefault();
                destTextureCopyRegion.EditSubresource().SetMipLevel(miplevel);
                destTextureCopyRegion.EditSubresource().SetArrayIndex(layer);
                destTextureCopyRegion.SetWidth(width);
                destTextureCopyRegion.SetHeight(height);
            }


            // ビューポートシザーステートを作成
            nn::gfx::ViewportScissorState viewportScissorState;
            {
                nn::gfx::ViewportScissorState::InfoType info;

                // Viewport State
                nn::gfx::ViewportStateInfo viewportInfo;
                {
                    viewportInfo.SetDefault();
                    viewportInfo.SetWidth(static_cast<float>(compressedwidth));
                    viewportInfo.SetHeight(static_cast<float>(compressedheight));
                }

                // Scissor State
                nn::gfx::ScissorStateInfo scissorInfo;
                {
                    scissorInfo.SetDefault();
                    scissorInfo.SetWidth(compressedwidth);
                    scissorInfo.SetHeight(compressedheight);
                }

                {
                    info.SetDefault();
                    info.SetScissorEnabled(true);
                    info.SetViewportStateInfoArray(&viewportInfo, 1);
                    info.SetScissorStateInfoArray(&scissorInfo, 1);
                }

                viewportScissorState.Initialize(pDevice, info);
            }

            pCommandBuffer->SetViewportScissorState(&viewportScissorState);

            // miplevel: 4bit layer: 16bit
            int texInfo = layer & 0xFFFF;
            texInfo |= ((miplevel & 0xF) << 16);
            texInfo <<= 2;

            //描画
            pCommandBuffer->Draw(nn::gfx::PrimitiveTopology_TriangleStrip, 4, texInfo, 1, 0);

            //ワークからコピー
            pCommandBuffer->CopyImageToBuffer(&workBuffer[index], 0, &workTexture[index], bufferCopyRegion);
            pCommandBuffer->CopyBufferToImage(targetTextureInfo.GetTexture(), destTextureCopyRegion, &workBuffer[index], 0);

            // gfxオブジェクトを破棄
            viewportScissorState.Finalize(pDevice);
        }
    }
} // NOLINT(impl/function_size)

#endif // NN_GFX_IS_TARGET_GL

}
}
}
