﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/gfx/util/gfx_PrimitiveShape.h>

namespace nn {
namespace gfx {
namespace util {

PrimitiveShape::PrimitiveShape( PrimitiveShapeFormat       vertexFormat,
                                nn::gfx::PrimitiveTopology primitiveTopology )
    : m_pIndexBuffer(nullptr)
    , m_pVertexBuffer(nullptr)
    , m_VertexFormat(vertexFormat)
    , m_IndexBufferFormat(nn::gfx::IndexFormat_Uint32)
    , m_PrimitiveTopology(primitiveTopology)
    , m_VertexCount(0)
    , m_IndexCount(0)
    , m_VertexBufferSize(0)
    , m_IndexBufferSize(0)
{
}

PrimitiveShape::~PrimitiveShape()
{
}

void* PrimitiveShape::GetIndexBuffer() const
{
    NN_SDK_REQUIRES_NOT_NULL(m_pIndexBuffer);
    return m_pIndexBuffer;
}

void* PrimitiveShape::GetVertexBuffer() const
{
    NN_SDK_REQUIRES_NOT_NULL(m_pVertexBuffer);
    return m_pVertexBuffer;
}

size_t PrimitiveShape::GetStride() const
{
    size_t stride = 0;

    if ( m_VertexFormat & PrimitiveShapeFormat_Pos )
    {
        stride+=sizeof(nn::util::Float3);
    }

    if ( m_VertexFormat & PrimitiveShapeFormat_Normal )
    {
        stride+=sizeof(nn::util::Float3);
    }

    if ( m_VertexFormat & PrimitiveShapeFormat_Uv )
    {
        stride+=sizeof(nn::util::Float2);
    }

    return stride;
}

size_t PrimitiveShape::GetVertexBufferSize() const
{
    return m_VertexBufferSize;
}

size_t PrimitiveShape::GetIndexBufferSize() const
{
    return m_IndexBufferSize;
}

nn::gfx::PrimitiveTopology PrimitiveShape::GetPrimitiveTopology() const
{
    return m_PrimitiveTopology;
}

PrimitiveShapeFormat PrimitiveShape::GetVertexFormat() const
{
    return m_VertexFormat;
}

IndexFormat PrimitiveShape::GetIndexBufferFormat() const
{
    return m_IndexBufferFormat;
}

int PrimitiveShape::GetVertexCount() const
{
    return m_VertexCount;
}

int PrimitiveShape::GetIndexCount() const
{
    return m_IndexCount;
}

void PrimitiveShape::Calculate(void* pVertexMemory, size_t vertexSize, void* pIndexMemory, size_t indexSize)
{
    CalculateImpl(pVertexMemory, vertexSize, pIndexMemory, indexSize);
}

void PrimitiveShape::SetVertexBuffer(void* pVertexBuffer)
{
    m_pVertexBuffer = pVertexBuffer;
}

void PrimitiveShape::SetIndexBuffer(void* pIndexBuffer)
{
    m_pIndexBuffer = pIndexBuffer;
}

void PrimitiveShape::SetVertexBufferSize( size_t vertexBufferSize )
{
    m_VertexBufferSize = vertexBufferSize;
}

void PrimitiveShape::SetIndexBufferSize( size_t indexBufferSize )
{
    m_IndexBufferSize = indexBufferSize;
}

void  PrimitiveShape::SetVertexCount( int vertexCount )
{
    m_VertexCount = vertexCount;
}

void PrimitiveShape::SetIndexCount( int indexCount )
{
    #if NN_GFX_IS_TARGET_VK
    // Vulkan では Uint8 型のインデックスは未サポートです。
    if (indexCount <= 0xFFFF)
    {
        m_IndexBufferFormat = nn::gfx::IndexFormat_Uint16;
    }
    else
    {
        m_IndexBufferFormat = nn::gfx::IndexFormat_Uint32;
    }
    #else
    if (indexCount <= 0xFF)
    {
        m_IndexBufferFormat = nn::gfx::IndexFormat_Uint8;
    }
    else if (indexCount <= 0xFFFF)
    {
        m_IndexBufferFormat = nn::gfx::IndexFormat_Uint16;
    }
    else
    {
        m_IndexBufferFormat = nn::gfx::IndexFormat_Uint32;
    }
    #endif

    m_IndexCount = indexCount;
}

SphereShape::SphereShape( PrimitiveShapeFormat       vertexFormat,
                          nn::gfx::PrimitiveTopology primitiveTopology,
                          int                        sliceCount,
                          int                        stackCount         ) : PrimitiveShape(vertexFormat, primitiveTopology),
                                                                            m_SliceCount(sliceCount),
                                                                            m_StackCount(stackCount)
{
    // プリミティブトポロジーが対応しているものかチェック
    NN_SDK_REQUIRES( GetPrimitiveTopology() == nn::gfx::PrimitiveTopology::PrimitiveTopology_LineList ||
                     GetPrimitiveTopology() == nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleList );

    // 頂点数を計算する
    SetVertexCount( CalculateVertexCount() );

    // インデックス数を計算する
    SetIndexCount( CalculateIndexCount() );

    const size_t stride = GetStride();

    // 頂点バッファのサイズを計算する。
    SetVertexBufferSize( stride * GetVertexCount() );

    // インデックスバッファのサイズを計算する。
    SetIndexBufferSize( sizeof(uint32_t) * GetIndexCount() );
}

SphereShape::~SphereShape()
{
}

int SphereShape::CalculateVertexCount()
{
    NN_SDK_ASSERT((2 <= m_StackCount) || ( 3 <= m_SliceCount), "m_StackCount (%d) < 2, m_SliceCount (%d) < 3\n", m_StackCount, m_SliceCount);

    int numVertices = 0;
    numVertices = (m_StackCount + 1) * (m_SliceCount + 1);

    return numVertices;
}

int SphereShape::CalculateIndexCount()
{
    NN_SDK_ASSERT((2 <= m_StackCount) || ( 3 <= m_SliceCount), "m_StackCount (%d) < 2, m_SliceCount (%d) < 3\n", m_StackCount, m_SliceCount);

    int numIndices = 0;

    const nn::gfx::PrimitiveTopology primitiveTopology = GetPrimitiveTopology();
    switch( primitiveTopology )
    {
    // ワイヤフレーム
    case nn::gfx::PrimitiveTopology::PrimitiveTopology_LineList:
        {
            numIndices  = (m_StackCount * 2 * m_SliceCount) +      // 垂直
                          (m_SliceCount * 2 * (m_StackCount - 1)); // 水平
        }
        break;
    // ソリッド
    case nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleList:
        {
            int numFaces = (m_StackCount - 2) * (m_SliceCount * 2);

            // 北極と南極
            numFaces += m_SliceCount + m_SliceCount;
            numIndices = numFaces * 3;
        }
        break;
    default:
        break;
    }

    return numIndices;
}

void* SphereShape::CalculateVertexBuffer()
{
    float* pVertexBuffer = static_cast<float*>(GetVertexBuffer());
    // 頂点バッファ
    for( int idxStack = 0; idxStack < m_StackCount + 1; idxStack++ )
    {
        float divStack = static_cast<float>(idxStack) / static_cast<float>(m_StackCount);
        float heightAngleRadian = divStack * (nn::util::detail::FloatPi);
        float cosHeight = static_cast<float>(nn::util::CosEst(heightAngleRadian));
        float sinHeight = static_cast<float>(nn::util::SinEst(heightAngleRadian));

        int slicesCount = m_SliceCount + 1;

        // 高さ方向の角度∈[0, 180]
        for( int idxSlice = 0; idxSlice < slicesCount; idxSlice++ )
        {
            float divSlice = static_cast<float>(idxSlice) / static_cast<float>(m_SliceCount);
            // XZ平面の角度∈[0, 360]
            float angleOnXZPlane = -divSlice * (nn::util::detail::Float2Pi);
            if( idxSlice == m_SliceCount )
            {
                angleOnXZPlane = 0.0f;
            }

            float cosXZ = static_cast<float>(nn::util::CosEst(angleOnXZPlane));
            float sinXZ = static_cast<float>(nn::util::SinEst(angleOnXZPlane));

            // 座標
            if ( GetVertexFormat() & PrimitiveShapeFormat_Pos )
            {
                pVertexBuffer[0] = cosXZ * sinHeight;
                pVertexBuffer[1] = cosHeight;
                pVertexBuffer[2] = sinXZ * sinHeight;
                pVertexBuffer+=3;
            }

            // 法線
            if ( GetVertexFormat() & PrimitiveShapeFormat_Normal )
            {
                pVertexBuffer[0] = cosXZ * sinHeight;
                pVertexBuffer[1] = cosHeight;
                pVertexBuffer[2] = sinXZ * sinHeight;
                pVertexBuffer+=3;
            }

            // UV
            if ( GetVertexFormat() & PrimitiveShapeFormat_Uv )
            {
                float divSliceUv = static_cast<float>(idxSlice) / static_cast<float>(m_SliceCount);
                pVertexBuffer[0] = divSliceUv;
                pVertexBuffer[1] = divStack;
                pVertexBuffer+=2;
            }
        }
    }

    // 頂点バッファの末端アドレスを返す。
    return pVertexBuffer;
}

template <typename T>
void SphereShape::CalculateIndexBuffer()
{
    T* pIndexData = static_cast<T*>(GetIndexBuffer());
    int idx  = 0;

    const nn::gfx::PrimitiveTopology primitiveTopology = GetPrimitiveTopology();
    switch( primitiveTopology )
    {
    // ワイヤフレーム
    case nn::gfx::PrimitiveTopology::PrimitiveTopology_LineList:
        {
            // 垂直
            uint32_t numVertsPerStack = static_cast<uint32_t>(m_SliceCount + 1);

            for( int idxSlice = 0U; idxSlice < m_SliceCount; idxSlice++ )
            {
                uint32_t sliceStartVtxIdx = idxSlice;
                for( int idxStack = 0U; idxStack < m_StackCount; idxStack++ )
                {
                    pIndexData[idx] = static_cast<T>(sliceStartVtxIdx + idxStack * numVertsPerStack);
                    idx++;

                    pIndexData[idx] = static_cast<T>(sliceStartVtxIdx + (idxStack + 1U) * numVertsPerStack);
                    idx++;
                }
            }

            // 水平
            for( int idxStack = 1; idxStack < m_StackCount; idxStack++ )
            {
                uint32_t startStackVtxIdx = static_cast<uint32_t>((m_SliceCount + 1) * idxStack);
                for( int idxSlice = 0; idxSlice < m_SliceCount; idxSlice++ )
                {
                    pIndexData[idx] = static_cast<T>(startStackVtxIdx + idxSlice);
                    idx++;

                    pIndexData[idx] = static_cast<T>(startStackVtxIdx + idxSlice + 1);
                    idx++;
                }
            }
        }
        break;
    // ソリッド
    case nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleList:
        {
            // 北極
            int idxFace = 0;

            for( int idxSlice = 0; idxSlice < m_SliceCount; idxSlice++ )
            {
                pIndexData[idxFace * 3 + 0] = static_cast<T>(idxSlice);
                pIndexData[idxFace * 3 + 1] = static_cast<T>(idxSlice + m_SliceCount + 1);
                pIndexData[idxFace * 3 + 2] = static_cast<T>(idxSlice + m_SliceCount + 1 + 1);
                idxFace++;
            }

            int idxStack = 0;

            for( idxStack=1; idxStack<m_StackCount - 1; idxStack++ )
            {
                for( int idxSlice=0; idxSlice<m_SliceCount; idxSlice++ )
                {
                    pIndexData[idxFace * 3 + 0] = static_cast<T>(idxSlice + idxStack * (m_SliceCount + 1));
                    pIndexData[idxFace * 3 + 1] = static_cast<T>(idxSlice + idxStack * (m_SliceCount + 1) + (m_SliceCount + 1) + 1);
                    pIndexData[idxFace * 3 + 2] = static_cast<T>(idxSlice + idxStack * (m_SliceCount + 1) + 1);
                    idxFace++;

                    pIndexData[idxFace * 3 + 0] = static_cast<T>(idxSlice + idxStack * (m_SliceCount + 1));
                    pIndexData[idxFace * 3 + 1] = static_cast<T>(idxSlice + idxStack * (m_SliceCount + 1) + (m_SliceCount + 1));
                    pIndexData[idxFace * 3 + 2] = static_cast<T>(idxSlice + idxStack * (m_SliceCount + 1) + (m_SliceCount + 1) + 1);
                    idxFace++;

                }
            }

            // 南極
            for( int idxSlice=0; idxSlice<m_SliceCount; idxSlice++ )
            {
                pIndexData[idxFace * 3 + 0] = static_cast<T>(idxSlice + idxStack * (m_SliceCount + 1));
                pIndexData[idxFace * 3 + 1] = static_cast<T>(idxSlice + idxStack * (m_SliceCount + 1) + (m_SliceCount + 1));
                pIndexData[idxFace * 3 + 2] = static_cast<T>(idxSlice + idxStack * (m_SliceCount + 1) + 1);
                idxFace++;
            }
        }
        break;
    default:
        break;
    }
}

void SphereShape::CalculateImpl(void* pVertexMemory, size_t vertexSize, void* pIndexMemory, size_t indexSize)
{
    NN_UNUSED(vertexSize);
    NN_UNUSED(indexSize);

    // バッファサイズが足りているかチェックします。
    NN_SDK_ASSERT( ( vertexSize >= GetVertexBufferSize() ) ||  ( indexSize >= GetIndexBufferSize() ));

    // 頂点バッファのセット
    SetVertexBuffer(pVertexMemory);

    // 頂点バッファの計算
    CalculateVertexBuffer();

    // インデックスバッファのセット
    SetIndexBuffer(pIndexMemory);

    // インデックスバッファの計算
    switch (GetIndexBufferFormat())
    {
    case nn::gfx::IndexFormat_Uint8:
        CalculateIndexBuffer<uint8_t>();
        break;
    case nn::gfx::IndexFormat_Uint16:
        CalculateIndexBuffer<uint16_t>();
        break;
    case nn::gfx::IndexFormat_Uint32:
        CalculateIndexBuffer<uint32_t>();
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }
}

CircleShape::CircleShape( PrimitiveShapeFormat       vertexFormat,
                          nn::gfx::PrimitiveTopology primitiveTopology,
                          int                        sliceCount           ) : PrimitiveShape(vertexFormat, primitiveTopology),
                                                                              m_SliceCount(sliceCount)
{
    // プリミティブトポロジーが対応しているものかチェック
    NN_SDK_REQUIRES( GetPrimitiveTopology() == nn::gfx::PrimitiveTopology::PrimitiveTopology_LineStrip ||
                     GetPrimitiveTopology() == nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleList );

    // 頂点数を計算する
    SetVertexCount( CalculateVertexCount() );

    // インデックス数を計算する
    SetIndexCount( CalculateIndexCount() );

    const size_t stride = GetStride();

    // 頂点バッファのサイズを計算する。
    SetVertexBufferSize( stride * GetVertexCount() );

    // インデックスバッファのサイズを計算する。
    SetIndexBufferSize( sizeof(uint32_t) * GetIndexCount() );
}

CircleShape::~CircleShape()
{
}

int CircleShape::CalculateVertexCount()
{
    NN_SDK_ASSERT( 3 <= m_SliceCount, "m_SliceCount (%d) < 3\n", m_SliceCount );

    int numVertices = 0;
    numVertices = m_SliceCount + 1;

    return numVertices;
}

int CircleShape::CalculateIndexCount()
{
    NN_SDK_ASSERT( 3 <= m_SliceCount, "m_SliceCount (%d) < 3\n", m_SliceCount );

    int numIndices = 0;

    const nn::gfx::PrimitiveTopology primitiveTopology = GetPrimitiveTopology();
    switch( primitiveTopology )
    {
    // ワイヤフレーム
    case nn::gfx::PrimitiveTopology::PrimitiveTopology_LineStrip:
        {
            numIndices = m_SliceCount + 1;
        }
        break;
    // ソリッド
    case nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleList:
        {
            numIndices = m_SliceCount * 3;
        }
        break;
    default:
        break;
    }

    return numIndices;
}

void* CircleShape::CalculateVertexBuffer()
{
    float* pVertexBuffer = static_cast<float*>(GetVertexBuffer());
    // 頂点バッファ
    for( int idxSlice = 0; idxSlice < m_SliceCount; idxSlice++ )
    {
        float rad = nn::util::DegreeToRadian( 360.f ) * idxSlice / m_SliceCount;
        float cosXY = nn::util::CosTable( nn::util::RadianToAngleIndex( rad ) );
        float sinXY = nn::util::SinTable( nn::util::RadianToAngleIndex( rad ) );

        // 座標
        if ( GetVertexFormat() & PrimitiveShapeFormat_Pos )
        {
            pVertexBuffer[0] = cosXY;
            pVertexBuffer[1] = sinXY;
            pVertexBuffer[2] = 0.f;
            pVertexBuffer+=3;
        }

        // 法線
        if ( GetVertexFormat() & PrimitiveShapeFormat_Normal )
        {
            pVertexBuffer[0] = 0.0f;
            pVertexBuffer[1] = 0.0f;
            pVertexBuffer[2] = 1.f;
            pVertexBuffer+=3;
        }

        // UV
        if ( GetVertexFormat() & PrimitiveShapeFormat_Uv )
        {
            pVertexBuffer[0] = 0.5f + cosXY * 0.5f;
            pVertexBuffer[1] = 1.0f - (0.5f + sinXY * 0.5f);
            pVertexBuffer+=2;
        }
    }

    // 中心
    if ( GetVertexFormat() & PrimitiveShapeFormat_Pos )
    {
        pVertexBuffer[0] = 0.0f;
        pVertexBuffer[1] = 0.0f;
        pVertexBuffer[2] = 0.0f;
        pVertexBuffer+=3;
    }

    // 法線
    if ( GetVertexFormat() & PrimitiveShapeFormat_Normal )
    {
        pVertexBuffer[0] = 0.0f;
        pVertexBuffer[1] = 0.0f;
        pVertexBuffer[2] = 1.0f;
        pVertexBuffer+=3;
    }

    // UV
    if ( GetVertexFormat() & PrimitiveShapeFormat_Uv )
    {
        pVertexBuffer[0] = 0.5f;
        pVertexBuffer[1] = 0.5f;
        pVertexBuffer+=2;
    }

    // 頂点バッファの末端アドレスを返す。
    return pVertexBuffer;
}

template <typename T>
void CircleShape::CalculateIndexBuffer()
{
    T* pIndexData = static_cast<T*>(GetIndexBuffer());

    const nn::gfx::PrimitiveTopology primitiveTopology = GetPrimitiveTopology();
    switch( primitiveTopology )
    {
    // ワイヤフレーム
    case nn::gfx::PrimitiveTopology::PrimitiveTopology_LineStrip:
        {
            for( int idx = 0; idx < GetIndexCount(); ++idx )
            {
                pIndexData[idx] = static_cast<T>( idx % m_SliceCount );
            }
        }
        break;
    // ソリッド
    case nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleList:
        {
            for( int idx = 0; idx < m_SliceCount; ++idx )
            {
                pIndexData[ idx * 3 + 0 ] = static_cast<T>( idx );
                pIndexData[ idx * 3 + 1 ] = static_cast<T>( ( idx + 1 ) % m_SliceCount );
                pIndexData[ idx * 3 + 2 ] = static_cast<T>( m_SliceCount );
            }
        }
        break;
    default:
        break;
    }
}

void CircleShape::CalculateImpl(void* pVertexMemory, size_t vertexSize, void* pIndexMemory, size_t indexSize)
{
    NN_UNUSED(vertexSize);
    NN_UNUSED(indexSize);

    // バッファサイズが足りているかチェックします。
    NN_SDK_ASSERT( ( vertexSize >= GetVertexBufferSize() ) ||  ( indexSize >= GetIndexBufferSize() ));

    // 頂点バッファのセット
    SetVertexBuffer(pVertexMemory);

    // 頂点バッファの計算
    CalculateVertexBuffer();

    // インデックスバッファのセット
    SetIndexBuffer(pIndexMemory);

    // インデックスバッファの計算
    switch (GetIndexBufferFormat())
    {
    case nn::gfx::IndexFormat_Uint8:
        CalculateIndexBuffer<uint8_t>();
        break;
    case nn::gfx::IndexFormat_Uint16:
        CalculateIndexBuffer<uint16_t>();
        break;
    case nn::gfx::IndexFormat_Uint32:
        CalculateIndexBuffer<uint32_t>();
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }
}

CubeShape::CubeShape( PrimitiveShapeFormat       vertexFormat,
                      nn::gfx::PrimitiveTopology primitiveTopology ) : PrimitiveShape(vertexFormat, primitiveTopology)
{
    // プリミティブトポロジーが対応しているものかチェック
    NN_SDK_REQUIRES( GetPrimitiveTopology() == nn::gfx::PrimitiveTopology::PrimitiveTopology_LineList ||
                     GetPrimitiveTopology() == nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleList );

    // 頂点数を計算する
    SetVertexCount( CalculateVertexCount() );

    // インデックス数を計算する
    SetIndexCount( CalculateIndexCount() );

    const size_t stride = GetStride();

    // 頂点バッファのサイズを計算する。
    SetVertexBufferSize( stride * GetVertexCount() );

    // インデックスバッファのサイズを計算する。
    SetIndexBufferSize( sizeof(uint32_t) * GetIndexCount() );
}

CubeShape::~CubeShape()
{
}

int CubeShape::CalculateVertexCount()
{
    int numVertices = 0;

    const nn::gfx::PrimitiveTopology primitiveTopology = GetPrimitiveTopology();
    switch( primitiveTopology )
    {
    // ワイヤフレーム
    case nn::gfx::PrimitiveTopology::PrimitiveTopology_LineList:
        {
            numVertices = CubeVertexCount_Wired;
        }
        break;
    // ソリッド
    case nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleList:
        {
            numVertices = CubeVertexCount_Solid;
        }
        break;
    default:
        break;
    }
    return numVertices;
}

int CubeShape::CalculateIndexCount()
{
    int numIndices = 0;

    const nn::gfx::PrimitiveTopology primitiveTopology = GetPrimitiveTopology();
    switch( primitiveTopology )
    {
    // ワイヤフレーム
    case nn::gfx::PrimitiveTopology::PrimitiveTopology_LineList:
        {
            numIndices = CubeIndexCount_Wired;
        }
        break;
    // ソリッド
    case nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleList:
        {
            numIndices = CubeIndexCount_Solid;
        }
        break;
    default:
        break;
    }

    return numIndices;
}

void* CubeShape::CalculateVertexBuffer()
{
    // 頂点バッファを設定します。
    static const nn::util::Float3 VertexPos[CubeVertexCount_Wired] =
    {
        NN_UTIL_FLOAT_3_INITIALIZER(  1.f,  1.f, -1.f ),
        NN_UTIL_FLOAT_3_INITIALIZER(  1.f, -1.f, -1.f ),
        NN_UTIL_FLOAT_3_INITIALIZER( -1.f, -1.f, -1.f ),
        NN_UTIL_FLOAT_3_INITIALIZER( -1.f,  1.f, -1.f ),
        NN_UTIL_FLOAT_3_INITIALIZER(  1.f,  1.f,  1.f ),
        NN_UTIL_FLOAT_3_INITIALIZER(  1.f, -1.f,  1.f ),
        NN_UTIL_FLOAT_3_INITIALIZER( -1.f, -1.f,  1.f ),
        NN_UTIL_FLOAT_3_INITIALIZER( -1.f,  1.f,  1.f )
    };

    static const nn::util::Float3 VertexNormal[6] =
    {
        NN_UTIL_FLOAT_3_INITIALIZER(  0.0f,  0.0f, -1.0f ),
        NN_UTIL_FLOAT_3_INITIALIZER( -1.0f,  0.0f,  0.0f ),
        NN_UTIL_FLOAT_3_INITIALIZER(  0.0f,  0.0f,  1.0f ),
        NN_UTIL_FLOAT_3_INITIALIZER(  1.0f,  0.0f,  0.0f ),
        NN_UTIL_FLOAT_3_INITIALIZER(  0.0f, -1.0f,  0.0f ),
        NN_UTIL_FLOAT_3_INITIALIZER(  0.0f,  1.0f,  0.0f )
    };

    static const nn::util::Float2 VertexUv[4] =
    {
        NN_UTIL_FLOAT_2_INITIALIZER(  0.0f,  0.0f ),
        NN_UTIL_FLOAT_2_INITIALIZER(  0.0f,  1.0f ),
        NN_UTIL_FLOAT_2_INITIALIZER(  1.0f,  1.0f ),
        NN_UTIL_FLOAT_2_INITIALIZER(  1.0f,  0.0f )
    };

    // インデックスバッファを設定します。
    static const uint32_t FaceTable[6][4] =
    {
        { 0, 1, 2, 3 }, // Front
        { 3, 2, 6, 7 }, // Right
        { 7, 6, 5, 4 }, // Back
        { 4, 5, 1, 0 }, // Left
        { 1, 5, 6, 2 }, // Bottom
        { 4, 0, 3, 7 }, // Top
    };

    float* pVertexBuffer = static_cast<float*>(GetVertexBuffer());
    // 頂点バッファ
    const nn::gfx::PrimitiveTopology primitiveTopology = GetPrimitiveTopology();

    if ( primitiveTopology == nn::gfx::PrimitiveTopology::PrimitiveTopology_LineList )
    {
        for( int idx = 0; idx < CubeVertexCount_Wired; idx++ )
        {
            // 位置
            pVertexBuffer[0] = VertexPos[idx].x * 0.5f;
            pVertexBuffer[1] = VertexPos[idx].y * 0.5f;
            pVertexBuffer[2] = VertexPos[idx].z * 0.5f;
            pVertexBuffer += 3;
        }
    }
    else
    {
        // ソリッドメッシュです。
        for( int idxFace = 0; idxFace < 6; idxFace++ )
        {
            for( int idxCorner = 0; idxCorner < 4; idxCorner++ )
            {
                int idx =  FaceTable[idxFace][idxCorner];

                // 座標
                if ( GetVertexFormat() & PrimitiveShapeFormat_Pos )
                {
                    pVertexBuffer[0] = VertexPos[idx].x * 0.5f;
                    pVertexBuffer[1] = VertexPos[idx].y * 0.5f;
                    pVertexBuffer[2] = VertexPos[idx].z * 0.5f;
                    pVertexBuffer += 3;
                }

                // 法線
                if ( GetVertexFormat() & PrimitiveShapeFormat_Normal )
                {
                    pVertexBuffer[0] = VertexNormal[idxFace].x;
                    pVertexBuffer[1] = VertexNormal[idxFace].y;
                    pVertexBuffer[2] = VertexNormal[idxFace].z;
                    pVertexBuffer += 3;
                }

                // UV
                if ( GetVertexFormat() & PrimitiveShapeFormat_Uv )
                {
                    pVertexBuffer[0] = VertexUv[idxCorner].x;
                    pVertexBuffer[1] = VertexUv[idxCorner].y;
                    pVertexBuffer += 2;
                }
            }
        }
    }

    // 頂点バッファの末端アドレスを返す。
    return pVertexBuffer;
}

template <typename T>
void CubeShape::CalculateIndexBuffer()
{
    // インデックスバッファを設定します。
    static const T FaceTable[6][4] =
    {
        { 0, 1, 2, 3 }, // Front
        { 3, 2, 6, 7 }, // Right
        { 7, 6, 5, 4 }, // Back
        { 4, 5, 1, 0 }, // Left
        { 1, 5, 6, 2 }, // Bottom
        { 4, 0, 3, 7 }, // Top
    };

    T* pIndexData = static_cast<T*>(GetIndexBuffer());

    size_t idx = 0;
    const nn::gfx::PrimitiveTopology primitiveTopology = GetPrimitiveTopology();
    switch( primitiveTopology )
    {
    // ワイヤフレーム
    case nn::gfx::PrimitiveTopology::PrimitiveTopology_LineList:
        {
            for( int idxFace = 0; idxFace < 6; idxFace++ )
            {
                pIndexData[idx++] = FaceTable[idxFace][0];
                pIndexData[idx++] = FaceTable[idxFace][1];

                pIndexData[idx++] = FaceTable[idxFace][1];
                pIndexData[idx++] = FaceTable[idxFace][2];

                pIndexData[idx++] = FaceTable[idxFace][2];
                pIndexData[idx++] = FaceTable[idxFace][3];

                pIndexData[idx++] = FaceTable[idxFace][3];
                pIndexData[idx++] = FaceTable[idxFace][0];
            }
        }
        break;
    // ソリッド
    case nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleList:
        {
            for( int idxFace = 0; idxFace < 6; idxFace++ )
            {
                pIndexData[idx++] = static_cast<T>(idxFace * 4 + 0);
                pIndexData[idx++] = static_cast<T>(idxFace * 4 + 1);
                pIndexData[idx++] = static_cast<T>(idxFace * 4 + 2);

                pIndexData[idx++] = static_cast<T>(idxFace * 4 + 0);
                pIndexData[idx++] = static_cast<T>(idxFace * 4 + 2);
                pIndexData[idx++] = static_cast<T>(idxFace * 4 + 3);
            }
        }
        break;
    default:
        break;
    }
}

void CubeShape::CalculateImpl(void* pVertexMemory, size_t vertexSize, void* pIndexMemory, size_t indexSize)
{
    NN_UNUSED(vertexSize);
    NN_UNUSED(indexSize);

    // バッファサイズが足りているかチェックします。
    NN_SDK_ASSERT( ( vertexSize >= GetVertexBufferSize() ) ||  ( indexSize >= GetIndexBufferSize() ));

    // 頂点バッファのセット
    SetVertexBuffer(pVertexMemory);

    // 頂点バッファの計算
    CalculateVertexBuffer();

    // インデックスバッファのセット
    SetIndexBuffer(pIndexMemory);

    // インデックスバッファの計算
    switch (GetIndexBufferFormat())
    {
    case nn::gfx::IndexFormat_Uint8:
        CalculateIndexBuffer<uint8_t>();
        break;
    case nn::gfx::IndexFormat_Uint16:
        CalculateIndexBuffer<uint16_t>();
        break;
    case nn::gfx::IndexFormat_Uint32:
        CalculateIndexBuffer<uint32_t>();
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }

}

QuadShape::QuadShape( PrimitiveShapeFormat       vertexFormat,
                      nn::gfx::PrimitiveTopology primitiveTopology ) : PrimitiveShape(vertexFormat, primitiveTopology)
{
    // プリミティブトポロジーが対応しているものかチェック
    NN_SDK_REQUIRES( GetPrimitiveTopology() == nn::gfx::PrimitiveTopology::PrimitiveTopology_LineStrip ||
                     GetPrimitiveTopology() == nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleList );

    // 頂点数を計算する
    SetVertexCount( CalculateVertexCount() );

    // インデックス数を計算する
    SetIndexCount( CalculateIndexCount() );

    const size_t stride = GetStride();

    // 頂点バッファのサイズを計算する。
    SetVertexBufferSize( stride * GetVertexCount() );

    // インデックスバッファのサイズを計算する。
    SetIndexBufferSize( sizeof(uint32_t) * GetIndexCount() );
}

QuadShape::~QuadShape()
{
}

int QuadShape::CalculateVertexCount()
{
    return QuadVertexCount;
}

int QuadShape::CalculateIndexCount()
{
    int numIndices = 0;

    const nn::gfx::PrimitiveTopology primitiveTopology = GetPrimitiveTopology();
    switch( primitiveTopology )
    {
    // ワイヤフレーム
    case nn::gfx::PrimitiveTopology::PrimitiveTopology_LineStrip:
        {
            numIndices = QuadIndexCount_Wired;
        }
        break;
    // ソリッド
    case nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleList:
        {
            numIndices = QuadIndexCountt_Solid;
        }
        break;
    default:
        break;
    }

    return numIndices;
}

void* QuadShape::CalculateVertexBuffer()
{
    // 頂点バッファを設定します。
    const nn::util::Float3 VertexPos[ QuadVertexCount ] =
    {
        NN_UTIL_FLOAT_3_INITIALIZER( -1.f,  1.f, 0.f ),
        NN_UTIL_FLOAT_3_INITIALIZER(  1.f,  1.f, 0.f ),
        NN_UTIL_FLOAT_3_INITIALIZER( -1.f, -1.f, 0.f ),
        NN_UTIL_FLOAT_3_INITIALIZER(  1.f, -1.f, 0.f )
    };

    const nn::util::Float2 VertexUv[ QuadVertexCount ] =
    {
        NN_UTIL_FLOAT_2_INITIALIZER( 0.f, 0.f ),
        NN_UTIL_FLOAT_2_INITIALIZER( 1.f, 0.f ),
        NN_UTIL_FLOAT_2_INITIALIZER( 0.f, 1.f ),
        NN_UTIL_FLOAT_2_INITIALIZER( 1.f, 1.f )
    };

    float* pVertexBuffer = static_cast<float*>(GetVertexBuffer());
    // 頂点バッファ
    const nn::gfx::PrimitiveTopology primitiveTopology = GetPrimitiveTopology();

    if ( primitiveTopology == nn::gfx::PrimitiveTopology::PrimitiveTopology_LineStrip )
    {
        for( int idx = 0; idx < QuadVertexCount; idx++ )
        {
            // 位置
            pVertexBuffer[0] = VertexPos[idx].x;
            pVertexBuffer[1] = VertexPos[idx].y;
            pVertexBuffer[2] = VertexPos[idx].z;
            pVertexBuffer += 3;
        }
    }
    else
    {
        // ソリッドメッシュです。
        for( int idx = 0; idx < QuadVertexCount; idx++ )
        {

            // 座標
            if ( GetVertexFormat() & PrimitiveShapeFormat_Pos )
            {
                pVertexBuffer[0] = VertexPos[idx].x;
                pVertexBuffer[1] = VertexPos[idx].y;
                pVertexBuffer[2] = VertexPos[idx].z;
                pVertexBuffer += 3;
            }

            // 法線
            if ( GetVertexFormat() & PrimitiveShapeFormat_Normal )
            {
                pVertexBuffer[0] = 0.0f;
                pVertexBuffer[1] = 0.0f;
                pVertexBuffer[2] = 1.0f;
                pVertexBuffer += 3;
            }

            // UV
            if ( GetVertexFormat() & PrimitiveShapeFormat_Uv )
            {
                pVertexBuffer[0] = VertexUv[idx].x;
                pVertexBuffer[1] = VertexUv[idx].y;
                pVertexBuffer += 2;
            }
        }
    }

    // 頂点バッファの末端アドレスを返す。
    return pVertexBuffer;
}

template<typename T>
void QuadShape::CalculateIndexBuffer()
{
    static const T VertexIndicesWired[ 2 * 4 ] =
    {
        0, 1, 3, 2, 0
    };

    static const T VertexIndices[ 6 ] =
    {
        0, 2, 1,
        1, 2, 3
    };

    T* pIndexData = static_cast<T*>(GetIndexBuffer());

    const nn::gfx::PrimitiveTopology primitiveTopology = GetPrimitiveTopology();
    switch( primitiveTopology )
    {
    // ワイヤフレーム
    case nn::gfx::PrimitiveTopology::PrimitiveTopology_LineStrip:
        {
            pIndexData[0] = VertexIndicesWired[0];
            pIndexData[1] = VertexIndicesWired[1];
            pIndexData[2] = VertexIndicesWired[2];
            pIndexData[3] = VertexIndicesWired[3];
            pIndexData[4] = VertexIndicesWired[4];
        }
        break;
    // ソリッド
    case nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleList:
        {
            pIndexData[0] = VertexIndices[0];
            pIndexData[1] = VertexIndices[1];
            pIndexData[2] = VertexIndices[2];

            pIndexData[3] = VertexIndices[3];
            pIndexData[4] = VertexIndices[4];
            pIndexData[5] = VertexIndices[5];

        }
        break;
    default:
        break;
    }
}

void QuadShape::CalculateImpl(void* pVertexMemory, size_t vertexSize, void* pIndexMemory, size_t indexSize)
{
    NN_UNUSED(vertexSize);
    NN_UNUSED(indexSize);

    // バッファサイズが足りているかチェックします。
    NN_SDK_ASSERT( ( vertexSize >= GetVertexBufferSize() ) ||  ( indexSize >= GetIndexBufferSize() ));

    // 頂点バッファのセット
    SetVertexBuffer(pVertexMemory);

    // 頂点バッファの計算
    CalculateVertexBuffer();

    // インデックスバッファのセット
    SetIndexBuffer(pIndexMemory);

    // インデックスバッファの計算
    switch (GetIndexBufferFormat())
    {
    case nn::gfx::IndexFormat_Uint8:
        CalculateIndexBuffer<uint8_t>();
        break;
    case nn::gfx::IndexFormat_Uint16:
        CalculateIndexBuffer<uint16_t>();
        break;
    case nn::gfx::IndexFormat_Uint32:
        CalculateIndexBuffer<uint32_t>();
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }

}

HemiSphereShape::HemiSphereShape( PrimitiveShapeFormat       vertexFormat,
                                  nn::gfx::PrimitiveTopology primitiveTopology,
                                  int                        sliceCount         ) : PrimitiveShape(vertexFormat, primitiveTopology),
                                                                                    m_SliceCount(sliceCount)
{
    // プリミティブトポロジーが対応しているものかチェック
    NN_SDK_REQUIRES( GetPrimitiveTopology() == nn::gfx::PrimitiveTopology::PrimitiveTopology_LineList ||
                     GetPrimitiveTopology() == nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleList );

    // 頂点数を計算する
    SetVertexCount( CalculateVertexCount() );

    // インデックス数を計算する
    SetIndexCount( CalculateIndexCount() );

    const size_t stride = GetStride();

    // 頂点バッファのサイズを計算する。
    SetVertexBufferSize( stride * GetVertexCount() );

    // インデックスバッファのサイズを計算する。
    SetIndexBufferSize( sizeof(uint32_t) * GetIndexCount() );
}

HemiSphereShape::~HemiSphereShape()
{
}

int HemiSphereShape::CalculateVertexCount()
{
    NN_SDK_ASSERT( 3 <= m_SliceCount , "m_SliceCount (%d) < 3\n", m_SliceCount );

    return m_SliceCount * ( m_SliceCount / 2 ) + 1;
}

int HemiSphereShape::CalculateIndexCount()
{
    NN_SDK_ASSERT( 3 <= m_SliceCount , "m_SliceCount (%d) < 3\n", m_SliceCount );

    return m_SliceCount * 3 + ( ( m_SliceCount / 2 ) - 1 ) * m_SliceCount * 6;
}

void* HemiSphereShape::CalculateVertexBuffer()
{
    float* pVertexBuffer = static_cast<float*>(GetVertexBuffer());

    // 天頂.
    {
        // 座標
        if ( GetVertexFormat() & PrimitiveShapeFormat_Pos )
        {
            pVertexBuffer[0] = 0.0f;
            pVertexBuffer[1] = 1.0f;
            pVertexBuffer[2] = 0.0f;
            pVertexBuffer += 3;
        }

        // 法線
        if ( GetVertexFormat() & PrimitiveShapeFormat_Normal )
        {
            pVertexBuffer[0] = 0.0f;
            pVertexBuffer[1] = 1.0f;
            pVertexBuffer[2] = 0.0f;
            pVertexBuffer += 3;
        }

        // UV
        if ( GetVertexFormat() & PrimitiveShapeFormat_Uv )
        {
            pVertexBuffer[0] = 0.0f;
            pVertexBuffer[1] = 1.0f;
            pVertexBuffer += 2;
        }
    }

    // 縦方向にループ
    int stackCount = m_SliceCount / 2;
    float halfPi      = nn::util::detail::FloatPi / 2;

    for ( int j = 0; j < stackCount; j++ )
    {
        // 横方向にループ
        for ( int i = 0; i < m_SliceCount; i++ )
        {
            float theta1 = ( i + 1 ) * 2 * nn::util::FloatPi / m_SliceCount;
            float theta2 = ( j + 1 ) * halfPi / stackCount + halfPi;

            float sine, cosine;
            nn::util::SinCosTable( &sine, &cosine, nn::util::RadianToAngleIndex(theta1) );

            float height = nn::util::SinTable( nn::util::RadianToAngleIndex(theta2) );
            float xzRadius = nn::util::SinEst( (j + 1) * halfPi / stackCount );

            float v0 = xzRadius * sine;
            float v1 = height;
            float v2 = xzRadius * cosine;

            // 座標
            if ( GetVertexFormat() & PrimitiveShapeFormat_Pos )
            {
                pVertexBuffer[0] = v0;
                pVertexBuffer[1] = v1;
                pVertexBuffer[2] = v2;
                pVertexBuffer += 3;
            }

            // 法線
            if ( GetVertexFormat() & PrimitiveShapeFormat_Normal )
            {
                pVertexBuffer[0] = v0;
                pVertexBuffer[1] = height;
                pVertexBuffer[2] = v2;
                pVertexBuffer += 3;
            }

            // UV
            if ( GetVertexFormat() & PrimitiveShapeFormat_Uv )
            {
                pVertexBuffer[0] = static_cast<float>(i) / static_cast<float>(m_SliceCount);
                pVertexBuffer[1] = static_cast<float>(j) / static_cast<float>(stackCount - 1);
                pVertexBuffer += 2;
            }
        }
    }

    // 頂点バッファの末端アドレスを返す。
    return pVertexBuffer;
}

template <typename T>
void HemiSphereShape::CalculateIndexBuffer()
{
    T* pIndexData = static_cast<T*>(GetIndexBuffer());
    int totalIndex = 0;

    // 天頂とつながる箇所は三角形になる
    for( int i = 0; i < m_SliceCount; ++i )
    {
        pIndexData[totalIndex + 0] = static_cast<T>(((i + 2) == m_SliceCount ) ? m_SliceCount : (i + 2) % m_SliceCount);
        pIndexData[totalIndex + 1] = static_cast<T>(0);
        pIndexData[totalIndex + 2] = static_cast<T>((i % m_SliceCount) + 1);

        totalIndex += 3;
    }

    const nn::gfx::PrimitiveTopology primitiveTopology = GetPrimitiveTopology();
    int stackCount = m_SliceCount / 2;
    // 天頂を除いた半分までのインデックスの処理.
    for ( int j = 0; j < stackCount - 1; j++ )
    {
        for ( int i = 0; i < m_SliceCount; i++ )
        {
            // 上側.
            int index0 = j * m_SliceCount + i + 1;
            int index1 = j * m_SliceCount + (i + 1) % m_SliceCount + 1;

            // 下側.
            int index2 = (j + 1) * m_SliceCount + i + 1;
            int index3 = (j + 1) * m_SliceCount + (i + 1) % m_SliceCount + 1;

            switch( primitiveTopology )
            {
            // ワイヤフレーム
            case nn::gfx::PrimitiveTopology::PrimitiveTopology_LineList:
                {
                    pIndexData[totalIndex + 0] = static_cast<T>(index0);
                    pIndexData[totalIndex + 1] = static_cast<T>(index1);
                    pIndexData[totalIndex + 2] = static_cast<T>(index1);
                    pIndexData[totalIndex + 3] = static_cast<T>(index3);
                    pIndexData[totalIndex + 4] = static_cast<T>(index2);
                    pIndexData[totalIndex + 5] = static_cast<T>(index3);
                }
                break;
            // ソリッド
            case nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleList:
                {
                    pIndexData[totalIndex + 0] = static_cast<T>(index0);
                    pIndexData[totalIndex + 1] = static_cast<T>(index2);
                    pIndexData[totalIndex + 2] = static_cast<T>(index1);

                    pIndexData[totalIndex + 3] = static_cast<T>(index1);
                    pIndexData[totalIndex + 4] = static_cast<T>(index2);
                    pIndexData[totalIndex + 5] = static_cast<T>(index3);
                }
                break;
           default:
               break;
           }
           totalIndex += 6;
        }
    }
}

void HemiSphereShape::CalculateImpl(void* pVertexMemory, size_t vertexSize, void* pIndexMemory, size_t indexSize)
{
    NN_UNUSED(vertexSize);
    NN_UNUSED(indexSize);

    // バッファサイズが足りているかチェックします。
    NN_SDK_ASSERT( ( vertexSize >= GetVertexBufferSize() ) ||  ( indexSize >= GetIndexBufferSize() ));

    // 頂点バッファのセット
    SetVertexBuffer(pVertexMemory);

    // 頂点バッファの計算
    CalculateVertexBuffer();

    // インデックスバッファのセット
    SetIndexBuffer(pIndexMemory);

    // インデックスバッファの計算
    switch (GetIndexBufferFormat())
    {
    case nn::gfx::IndexFormat_Uint8:
        CalculateIndexBuffer<uint8_t>();
        break;
    case nn::gfx::IndexFormat_Uint16:
        CalculateIndexBuffer<uint16_t>();
        break;
    case nn::gfx::IndexFormat_Uint32:
        CalculateIndexBuffer<uint32_t>();
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }

}

PipeShape::PipeShape( PrimitiveShapeFormat       vertexFormat,
                      nn::gfx::PrimitiveTopology primitiveTopology,
                      int                        sliceCount           ) : PrimitiveShape(vertexFormat, primitiveTopology),
                                                                          m_SliceCount(sliceCount)
{
    // プリミティブトポロジーが対応しているものかチェック
    NN_SDK_REQUIRES( GetPrimitiveTopology() == nn::gfx::PrimitiveTopology::PrimitiveTopology_LineList ||
                     GetPrimitiveTopology() == nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleList );

    // 頂点数を計算する
    SetVertexCount( CalculateVertexCount() );

    // インデックス数を計算する
    SetIndexCount( CalculateIndexCount() );

    const size_t stride = GetStride();

    // 頂点バッファのサイズを計算する。
    SetVertexBufferSize( stride * GetVertexCount() );

    // インデックスバッファのサイズを計算する。
    SetIndexBufferSize( sizeof(uint32_t) * GetIndexCount() );
}

PipeShape::~PipeShape()
{
}

int PipeShape::CalculateVertexCount()
{
    NN_SDK_ASSERT( 3 <= m_SliceCount , "m_SliceCount (%d) < 3\n", m_SliceCount );

    return m_SliceCount * 2;  // 上下で2倍になる
}

int PipeShape::CalculateIndexCount()
{
    NN_SDK_ASSERT( 3 <= m_SliceCount , "m_SliceCount (%d) < 3\n", m_SliceCount );

    return m_SliceCount * 6;  // 1面につき3, 3で6個それが各面ごとにある
}

void* PipeShape::CalculateVertexBuffer()
{
    float* pVertexBuffer = static_cast<float*>(GetVertexBuffer());

    // 頂点バッファ
    nn::util::Vector3fType v1;
    int idx = 0;
    int elementCount = 0;

    // 頂点の要素数を計算
    if ( GetVertexFormat() & PrimitiveShapeFormat_Pos )
    {
        elementCount += 3;
    }

    if ( GetVertexFormat() & PrimitiveShapeFormat_Normal )
    {
        elementCount += 3;
    }

    if ( GetVertexFormat() & PrimitiveShapeFormat_Uv )
    {
        elementCount += 2;
    }

    for ( int i = 0; i < m_SliceCount; ++i )
    {
        float rad = ( i + 1 ) * 2 * nn::util::FloatPi / m_SliceCount;
        float sinXZ, cosXZ;
        nn::util::SinCosTable(&sinXZ, &cosXZ, nn::util::RadianToAngleIndex(rad));

        nn::util::VectorSet( &v1, sinXZ, 0.f, cosXZ );

        // 座標
        if ( GetVertexFormat() & PrimitiveShapeFormat_Pos )
        {
            pVertexBuffer[idx + 0] = nn::util::VectorGetX( v1 );
            pVertexBuffer[idx + 1] = 0.0f;
            pVertexBuffer[idx + 2] = nn::util::VectorGetZ( v1 );
            idx += 3;
        }

        // 法線
        if ( GetVertexFormat() & PrimitiveShapeFormat_Normal )
        {
            pVertexBuffer[idx + 0] = nn::util::VectorGetX( v1 );
            pVertexBuffer[idx + 1] = 0.f;
            pVertexBuffer[idx + 2] = nn::util::VectorGetZ( v1 );
            idx += 3;
        }

        // UV
        if ( GetVertexFormat() & PrimitiveShapeFormat_Uv )
        {
            pVertexBuffer[idx + 0] = 0.5f + cosXZ * 0.5f;
            pVertexBuffer[idx + 1] = 1.0f;
            idx += 2;
        }

        // 座標
        int bottomIdx = (i + m_SliceCount) * elementCount;

        if ( GetVertexFormat() & PrimitiveShapeFormat_Pos )
        {
            pVertexBuffer[bottomIdx + 0] = nn::util::VectorGetX( v1 );
            pVertexBuffer[bottomIdx + 1] = 1.0f;
            pVertexBuffer[bottomIdx + 2] = nn::util::VectorGetZ( v1 );
            bottomIdx += 3;
        }
        // 法線
        if ( GetVertexFormat() & PrimitiveShapeFormat_Normal )
        {
            pVertexBuffer[bottomIdx + 0] = nn::util::VectorGetX( v1 );
            pVertexBuffer[bottomIdx + 1] = 0.0f;
            pVertexBuffer[bottomIdx + 2] = nn::util::VectorGetZ( v1 );
            bottomIdx += 3;
        }

        // UV
        if ( GetVertexFormat() & PrimitiveShapeFormat_Uv )
        {
            pVertexBuffer[bottomIdx + 0] = 0.5f + cosXZ * 0.5f;
            pVertexBuffer[bottomIdx + 1] = 0.0f;
            bottomIdx += 2;
        }
    }

    // 頂点バッファの末端アドレスを返す。
    return pVertexBuffer + GetVertexBufferSize();
}

template <typename T>
void PipeShape::CalculateIndexBuffer()
{
    T* pIndexData = static_cast<T*>(GetIndexBuffer());

    const nn::gfx::PrimitiveTopology primitiveTopology = GetPrimitiveTopology();

    for( uint32_t i = 0; i<static_cast<uint32_t>( m_SliceCount ); ++i )
    {
        uint32_t index0 = i;
        uint32_t index1 = (i + 1) % m_SliceCount;
        uint32_t index2 = (i + m_SliceCount) % GetVertexCount();
        uint32_t index3 = (i + 1) % m_SliceCount + m_SliceCount;

        switch( primitiveTopology )
        {
        // ワイヤフレーム
        case nn::gfx::PrimitiveTopology::PrimitiveTopology_LineList:
            {
                pIndexData[i * 6 + 0] = static_cast<T>(index0);
                pIndexData[i * 6 + 1] = static_cast<T>(index1);
                pIndexData[i * 6 + 2] = static_cast<T>(index2);

                pIndexData[i * 6 + 3] = static_cast<T>(index3);
                pIndexData[i * 6 + 4] = static_cast<T>(index2);
                pIndexData[i * 6 + 5] = static_cast<T>(index0);
            }
            break;
        // ソリッド
        case nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleList:
            {
                pIndexData[i * 6 + 0] = static_cast<T>(index0);
                pIndexData[i * 6 + 1] = static_cast<T>(index2);
                pIndexData[i * 6 + 2] = static_cast<T>(index1);

                pIndexData[i * 6 + 3] = static_cast<T>(index1);
                pIndexData[i * 6 + 4] = static_cast<T>(index2);
                pIndexData[i * 6 + 5] = static_cast<T>(index3);
            }
            break;
        default:
            break;
        }
    }
}

void PipeShape::CalculateImpl(void* pVertexMemory, size_t vertexSize, void* pIndexMemory, size_t indexSize)
{
    NN_UNUSED(vertexSize);
    NN_UNUSED(indexSize);

    // バッファサイズが足りているかチェックします。
    NN_SDK_ASSERT( ( vertexSize >= GetVertexBufferSize() ) ||  ( indexSize >= GetIndexBufferSize() ));

    // 頂点バッファのセット
    SetVertexBuffer(pVertexMemory);

    // 頂点バッファの計算
    CalculateVertexBuffer();

    // インデックスバッファのセット
    SetIndexBuffer(pIndexMemory);

    // インデックスバッファの計算
    switch (GetIndexBufferFormat())
    {
    case nn::gfx::IndexFormat_Uint8:
        CalculateIndexBuffer<uint8_t>();
        break;
    case nn::gfx::IndexFormat_Uint16:
        CalculateIndexBuffer<uint16_t>();
        break;
    case nn::gfx::IndexFormat_Uint32:
        CalculateIndexBuffer<uint32_t>();
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }

}

CylinderShape::CylinderShape( PrimitiveShapeFormat       vertexFormat,
                      nn::gfx::PrimitiveTopology primitiveTopology,
                      int                        sliceCount           ) : PrimitiveShape(vertexFormat, primitiveTopology),
                                                                          m_SliceCount(sliceCount)
{
    // プリミティブトポロジーが対応しているものかチェック
    NN_SDK_REQUIRES( GetPrimitiveTopology() == nn::gfx::PrimitiveTopology::PrimitiveTopology_LineList ||
                     GetPrimitiveTopology() == nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleList );

    // 頂点数を計算する
    SetVertexCount( CalculateVertexCount() );

    // インデックス数を計算する
    SetIndexCount( CalculateIndexCount() );

    const size_t stride = GetStride();

    // 頂点バッファのサイズを計算する。
    SetVertexBufferSize( stride * GetVertexCount() );

    // インデックスバッファのサイズを計算する。
    SetIndexBufferSize( sizeof(uint32_t) * GetIndexCount() );
}

CylinderShape::~CylinderShape()
{
}

int CylinderShape::CalculateVertexCount()
{
    NN_SDK_ASSERT( 3 <= m_SliceCount , "m_SliceCount (%d) < 3\n", m_SliceCount );

    return m_SliceCount * 4 + 2;  // 側面の上下で2倍と円の上下面と上下の頂点
}

int CylinderShape::CalculateIndexCount()
{
    NN_SDK_ASSERT( 3 <= m_SliceCount , "m_SliceCount (%d) < 3\n", m_SliceCount );

    const nn::gfx::PrimitiveTopology primitiveTopology = GetPrimitiveTopology();
    int numIndices = 0;
    switch( primitiveTopology )
    {
    // ワイヤフレーム
     case nn::gfx::PrimitiveTopology::PrimitiveTopology_LineList:
         {
             numIndices = m_SliceCount * (6 + 6 + 6);  // 1面につき3, 3で6個それが各面ごとにあり、上下の円それぞれ６個ある
         }
        break;
    // ソリッド
     case nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleList:
         {
             numIndices = m_SliceCount * (6 + 3 + 3);  // 1面につき3, 3で6個それが各面ごとにあり、上下の円それぞれ３個ある
         }
         break;
    default:
        break;
    }
    return numIndices;
}

void* CylinderShape::CalculateVertexBuffer()
{
    float* pVertexBuffer = static_cast<float*>(GetVertexBuffer());

    // 頂点バッファ
    nn::util::Vector3fType v1;
    int idx = 0;
    int elementCount = 0;

    // 頂点の要素数を計算
    if ( GetVertexFormat() & PrimitiveShapeFormat_Pos )
    {
        elementCount += 3;
    }

    if ( GetVertexFormat() & PrimitiveShapeFormat_Normal )
    {
        elementCount += 3;
    }

    if ( GetVertexFormat() & PrimitiveShapeFormat_Uv )
    {
        elementCount += 2;
    }

    // 上下面と側面の頂点を次のよう作成。
    // 頂点バッファの構造
    // 0                - m_SliceCount-1   -> 側面の底面用円周上頂点
    // m_SliceCount     - m_SliceCount*2-1 -> 側面の上面用円周上頂点
    // m_SliceCount*2   - m_SliceCount*3-1 -> 底面用円頂点
    // m_SliceCount*3                      -> 底面中心頂点
    // m_SliceCount*3+1 - m_CliceCount*4   -> 側面用円周上頂点
    // m_CliceCount*4+1                    -> 上面中心頂点

    //　側面
    for ( int i = 0; i < m_SliceCount; ++i )
    {
        float rad = ( i + 1 ) * 2 * nn::util::FloatPi / m_SliceCount;
        float sinXZ, cosXZ;
        nn::util::SinCosTable(&sinXZ, &cosXZ, nn::util::RadianToAngleIndex(rad));

        nn::util::VectorSet( &v1, sinXZ, 0.f, cosXZ );

        // 座標
        if ( GetVertexFormat() & PrimitiveShapeFormat_Pos )
        {
            pVertexBuffer[idx + 0] = nn::util::VectorGetX( v1 );
            pVertexBuffer[idx + 1] = 0.0f;
            pVertexBuffer[idx + 2] = nn::util::VectorGetZ( v1 );
            idx += 3;
        }

        // 法線
        if ( GetVertexFormat() & PrimitiveShapeFormat_Normal )
        {
            pVertexBuffer[idx + 0] = nn::util::VectorGetX( v1 );
            pVertexBuffer[idx + 1] = 0.f;
            pVertexBuffer[idx + 2] = nn::util::VectorGetZ( v1 );
            idx += 3;
        }

        // UV
        if ( GetVertexFormat() & PrimitiveShapeFormat_Uv )
        {
            pVertexBuffer[idx + 0] = 0.5f + cosXZ * 0.5f;
            pVertexBuffer[idx + 1] = 1.0f;
            idx += 2;
        }

        // 座標
        int bottomIdx = (i + m_SliceCount) * elementCount;

        if ( GetVertexFormat() & PrimitiveShapeFormat_Pos )
        {
            pVertexBuffer[bottomIdx + 0] = nn::util::VectorGetX( v1 );
            pVertexBuffer[bottomIdx + 1] = 1.0f;
            pVertexBuffer[bottomIdx + 2] = nn::util::VectorGetZ( v1 );
            bottomIdx += 3;
        }
        // 法線
        if ( GetVertexFormat() & PrimitiveShapeFormat_Normal )
        {
            pVertexBuffer[bottomIdx + 0] = nn::util::VectorGetX( v1 );
            pVertexBuffer[bottomIdx + 1] = 0.0f;
            pVertexBuffer[bottomIdx + 2] = nn::util::VectorGetZ( v1 );
            bottomIdx += 3;
        }

        // UV
        if ( GetVertexFormat() & PrimitiveShapeFormat_Uv )
        {
            pVertexBuffer[bottomIdx + 0] = 0.5f + cosXZ * 0.5f;
            pVertexBuffer[bottomIdx + 1] = 0.0f;
            bottomIdx += 2;
        }
    }

    //　上下円面
    int bottomCircleIdx =(m_SliceCount + m_SliceCount) * elementCount;
    int CircleIdx =(m_SliceCount * 3 + 1 ) * elementCount;

    for ( int i = 0; i < m_SliceCount; ++i )
    {
        float rad = ( i + 1 ) * 2 * nn::util::FloatPi / m_SliceCount;
        float sinXZ, cosXZ;
        nn::util::SinCosTable(&sinXZ, &cosXZ, nn::util::RadianToAngleIndex(rad));

        nn::util::VectorSet( &v1, sinXZ, 0.f, cosXZ );

        // 下面座標
        if ( GetVertexFormat() & PrimitiveShapeFormat_Pos )
        {
            pVertexBuffer[bottomCircleIdx + 0] = nn::util::VectorGetX( v1 );
            pVertexBuffer[bottomCircleIdx + 1] = 0.0f;
            pVertexBuffer[bottomCircleIdx + 2] = nn::util::VectorGetZ( v1 );
            bottomCircleIdx += 3;
        }

        // 下面法線
        if ( GetVertexFormat() & PrimitiveShapeFormat_Normal )
        {
            pVertexBuffer[bottomCircleIdx + 0] = nn::util::VectorGetX( v1 );
            pVertexBuffer[bottomCircleIdx + 1] = -1.f;
            pVertexBuffer[bottomCircleIdx + 2] = nn::util::VectorGetZ( v1 );
            bottomCircleIdx += 3;
        }

        // 下面UV
        if ( GetVertexFormat() & PrimitiveShapeFormat_Uv )
        {
            pVertexBuffer[bottomCircleIdx + 0] = 0.5f + cosXZ * 0.5f;
            pVertexBuffer[bottomCircleIdx + 1] = 1.0f - (0.5f + sinXZ * 0.5f);
            bottomCircleIdx += 2;
        }

        // 上面座標
        if ( GetVertexFormat() & PrimitiveShapeFormat_Pos )
        {
            pVertexBuffer[CircleIdx + 0] = nn::util::VectorGetX( v1 );
            pVertexBuffer[CircleIdx + 1] = 1.0f;
            pVertexBuffer[CircleIdx + 2] = nn::util::VectorGetZ( v1 );
            CircleIdx += 3;
        }

        // 上面法線
        if ( GetVertexFormat() & PrimitiveShapeFormat_Normal )
        {
            pVertexBuffer[CircleIdx + 0] = nn::util::VectorGetX( v1 );
            pVertexBuffer[CircleIdx + 1] = 0.0f;
            pVertexBuffer[CircleIdx + 2] = nn::util::VectorGetZ( v1 );
            CircleIdx += 3;
        }

        // 上面UV
        if ( GetVertexFormat() & PrimitiveShapeFormat_Uv )
        {
            pVertexBuffer[CircleIdx + 0] = 0.5f + cosXZ * 0.5f;
            pVertexBuffer[CircleIdx + 1] = 1.0f - (0.5f + sinXZ * 0.5f);
            CircleIdx += 2;
        }
    }

    // 下部中心
    // 座標
    if ( GetVertexFormat() & PrimitiveShapeFormat_Pos )
    {
        pVertexBuffer[bottomCircleIdx + 0] = 0.0f;
        pVertexBuffer[bottomCircleIdx + 1] = 0.0f;
        pVertexBuffer[bottomCircleIdx + 2] = 0.0f;
        bottomCircleIdx += 3;
    }

    // 法線
    if ( GetVertexFormat() & PrimitiveShapeFormat_Normal )
    {
        pVertexBuffer[bottomCircleIdx + 0] =  0.0f;
        pVertexBuffer[bottomCircleIdx + 1] = -1.0f;
        pVertexBuffer[bottomCircleIdx + 2] =  0.0f;
        bottomCircleIdx += 3;
    }

    // UV
    if ( GetVertexFormat() & PrimitiveShapeFormat_Uv )
    {
        pVertexBuffer[bottomCircleIdx + 0] = 0.5f;
        pVertexBuffer[bottomCircleIdx + 1] = 0.5f;
        bottomCircleIdx += 2;
    }

    // 上部中心
    // 座標
    if ( GetVertexFormat() & PrimitiveShapeFormat_Pos )
    {
        pVertexBuffer[CircleIdx + 0] = 0.0f;
        pVertexBuffer[CircleIdx + 1] = 1.0f;
        pVertexBuffer[CircleIdx + 2] = 0.0f;
        CircleIdx += 3;
    }

    // 法線
    if ( GetVertexFormat() & PrimitiveShapeFormat_Normal )
    {
        pVertexBuffer[CircleIdx + 0] = 0.0f;
        pVertexBuffer[CircleIdx + 1] = 1.0f;
        pVertexBuffer[CircleIdx + 2] = 0.0f;
        CircleIdx += 3;
    }

    // UV
    if ( GetVertexFormat() & PrimitiveShapeFormat_Uv )
    {
        pVertexBuffer[CircleIdx + 0] = 0.5f;
        pVertexBuffer[CircleIdx + 1] = 0.5f;
        CircleIdx += 2;
    }

    // 頂点バッファの末端アドレスを返す。
    return pVertexBuffer + GetVertexBufferSize();
} //NOLINT(impl/function_size)

template <typename T>
void CylinderShape::CalculateIndexBuffer()
{
    T* pIndexData = static_cast<T*>(GetIndexBuffer());

    const nn::gfx::PrimitiveTopology primitiveTopology = GetPrimitiveTopology();

    // 側面
    int sideVertexCount = m_SliceCount * m_SliceCount;
    for( uint32_t i = 0; i<static_cast<uint32_t>( m_SliceCount ); ++i )
    {
        uint32_t index0 = i;
        uint32_t index1 = (i + 1) % m_SliceCount;
        uint32_t index2 = (i + m_SliceCount) % sideVertexCount;
        uint32_t index3 = (i + 1) % m_SliceCount + m_SliceCount;

        switch( primitiveTopology )
        {
        // ワイヤフレーム
        case nn::gfx::PrimitiveTopology::PrimitiveTopology_LineList:
            {
                pIndexData[i * 6 + 0] = static_cast<T>(index0);
                pIndexData[i * 6 + 1] = static_cast<T>(index1);
                pIndexData[i * 6 + 2] = static_cast<T>(index2);

                pIndexData[i * 6 + 3] = static_cast<T>(index3);
                pIndexData[i * 6 + 4] = static_cast<T>(index2);
                pIndexData[i * 6 + 5] = static_cast<T>(index0);
            }
            break;
        // ソリッド
        case nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleList:
            {
                pIndexData[i * 6 + 0] = static_cast<T>(index0);
                pIndexData[i * 6 + 1] = static_cast<T>(index2);
                pIndexData[i * 6 + 2] = static_cast<T>(index1);

                pIndexData[i * 6 + 3] = static_cast<T>(index1);
                pIndexData[i * 6 + 4] = static_cast<T>(index2);
                pIndexData[i * 6 + 5] = static_cast<T>(index3);
            }
            break;
        default:
            break;
        }
    }

    // 下部面
    int bottomCircleIdx = m_SliceCount + m_SliceCount;
    for( uint32_t i = 0; i<static_cast<uint32_t>( m_SliceCount ); ++i )
    {
        uint32_t index0 = i + bottomCircleIdx;
        uint32_t index1 = ( (i + 1) % m_SliceCount ) + bottomCircleIdx;
        uint32_t centerIndex = bottomCircleIdx + m_SliceCount;
        switch( primitiveTopology )
        {
        // ワイヤフレーム
        case nn::gfx::PrimitiveTopology::PrimitiveTopology_LineList:
            {
                pIndexData[m_SliceCount * 6 + i * 6 + 0] = static_cast<T>(centerIndex);
                pIndexData[m_SliceCount * 6 + i * 6 + 1] = static_cast<T>(index0);
                pIndexData[m_SliceCount * 6 + i * 6 + 2] = static_cast<T>(centerIndex);

                pIndexData[m_SliceCount * 6 + i * 6 + 3] = static_cast<T>(index1);
                pIndexData[m_SliceCount * 6 + i * 6 + 4] = static_cast<T>(index0);
                pIndexData[m_SliceCount * 6 + i * 6 + 5] = static_cast<T>(index1);
            }
            break;
        // ソリッド
        case nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleList:
            {
                pIndexData[m_SliceCount * 6 + i * 3 + 0] = static_cast<T>(centerIndex);
                pIndexData[m_SliceCount * 6 + i * 3 + 1] = static_cast<T>(index0);
                pIndexData[m_SliceCount * 6 + i * 3 + 2] = static_cast<T>(index1);
            }
            break;
        default:
            break;
        }
    }

    // 上部面
    int circleIdx = bottomCircleIdx + m_SliceCount + 1;
    for( uint32_t i = 0; i<static_cast<uint32_t>( m_SliceCount ); ++i )
    {
        uint32_t index0 = i + circleIdx;
        uint32_t index1 = ( (i + 1) % m_SliceCount ) + circleIdx;
        uint32_t centerIndex = circleIdx + m_SliceCount;

        switch( primitiveTopology )
        {
        // ワイヤフレーム
        case nn::gfx::PrimitiveTopology::PrimitiveTopology_LineList:
            {
                pIndexData[m_SliceCount * (6 + 6) + i * 6 + 0] = static_cast<T>(centerIndex);
                pIndexData[m_SliceCount * (6 + 6) + i * 6 + 1] = static_cast<T>(index0);
                pIndexData[m_SliceCount * (6 + 6) + i * 6 + 2] = static_cast<T>(centerIndex);

                pIndexData[m_SliceCount * (6 + 6) + i * 6 + 3] = static_cast<T>(index1);
                pIndexData[m_SliceCount * (6 + 6) + i * 6 + 4] = static_cast<T>(index0);
                pIndexData[m_SliceCount * (6 + 6) + i * 6 + 5] = static_cast<T>(index1);
            }
            break;
        // ソリッド
        case nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleList:
            {
                pIndexData[m_SliceCount * (6 + 3) + i * 3 + 0] = static_cast<T>(index1);
                pIndexData[m_SliceCount * (6 + 3) + i * 3 + 1] = static_cast<T>(index0);
                pIndexData[m_SliceCount * (6 + 3) + i * 3 + 2] = static_cast<T>(centerIndex);
            }
            break;
        default:
            break;
        }
    }
} //NOLINT(impl/function_size)

void CylinderShape::CalculateImpl(void* pVertexMemory, size_t vertexSize, void* pIndexMemory, size_t indexSize)
{
    NN_UNUSED(vertexSize);
    NN_UNUSED(indexSize);

    // バッファサイズが足りているかチェックします。
    NN_SDK_ASSERT( ( vertexSize >= GetVertexBufferSize() ) ||  ( indexSize >= GetIndexBufferSize() ));

    // 頂点バッファのセット
    SetVertexBuffer(pVertexMemory);

    // 頂点バッファの計算
    CalculateVertexBuffer();

    // インデックスバッファのセット
    SetIndexBuffer(pIndexMemory);

    // インデックスバッファの計算
    switch (GetIndexBufferFormat())
    {
    case nn::gfx::IndexFormat_Uint8:
        CalculateIndexBuffer<uint8_t>();
        break;
    case nn::gfx::IndexFormat_Uint16:
        CalculateIndexBuffer<uint16_t>();
        break;
    case nn::gfx::IndexFormat_Uint32:
        CalculateIndexBuffer<uint32_t>();
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }
}

ConeShape::ConeShape( PrimitiveShapeFormat       vertexFormat,
                      nn::gfx::PrimitiveTopology primitiveTopology,
                      int                        sliceCount          ) : PrimitiveShape(vertexFormat, primitiveTopology),
                                                                       m_SliceCount(sliceCount)
{
    // プリミティブトポロジーが対応しているものかチェック
    NN_SDK_REQUIRES( GetPrimitiveTopology() == nn::gfx::PrimitiveTopology::PrimitiveTopology_LineList ||
                     GetPrimitiveTopology() == nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleList );

    // 頂点数を計算する
    SetVertexCount( CalculateVertexCount() );

    // インデックス数を計算する
    SetIndexCount( CalculateIndexCount() );

    const size_t stride = GetStride();

    // 頂点バッファのサイズを計算する。
    SetVertexBufferSize( stride * GetVertexCount() );

    // インデックスバッファのサイズを計算する。
    SetIndexBufferSize( sizeof(uint32_t) * GetIndexCount() );
}

ConeShape::~ConeShape()
{
}

int ConeShape::CalculateVertexCount()
{
    NN_SDK_ASSERT( 3 <= m_SliceCount , "m_SliceCount (%d) < 3\n", m_SliceCount );

    return m_SliceCount * 2 + 2;
}

int ConeShape::CalculateIndexCount()
{
    NN_SDK_ASSERT( 3 <= m_SliceCount , "m_SliceCount (%d) < 3\n", m_SliceCount );

    int numIndices = 0;

    const nn::gfx::PrimitiveTopology primitiveTopology = GetPrimitiveTopology();
    switch( primitiveTopology )
    {
    // ワイヤフレーム
    case nn::gfx::PrimitiveTopology::PrimitiveTopology_LineList:
        {
            numIndices = m_SliceCount * 4;
        }
        break;
    // ソリッド
    case nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleList:
        {
            numIndices = m_SliceCount * 6;
        }
        break;
    default:
        break;
    }

    return numIndices;
}

void* ConeShape::CalculateVertexBuffer()
{
    // 底面と側面で法線を変えるため底面の枠の頂点を2重に作成する。
    // 頂点バッファの構造
    // 0 - m_SliceCount-1   -> 底面用円周上頂点
    // m_SliceCount         -> 底面中心頂点
    // m_SliceCount+1 - m_CliceCount*2  -> 側面用円周上頂点
    // m_SliceCount*2+1     -> 頂点

    float* pVertexBuffer = static_cast<float*>(GetVertexBuffer());

    // 底面の頂点データ.
    for ( int i = 0; i < m_SliceCount; ++i)
    {
        float rad = ( nn::util::DegreeToRadian(360.0f) / m_SliceCount ) * i;
        float cosXZ = nn::util::CosTable( nn::util::RadianToAngleIndex(rad) );
        float sinXZ = nn::util::SinTable( nn::util::RadianToAngleIndex(rad) );
        // 座標
        if ( GetVertexFormat() & PrimitiveShapeFormat_Pos )
        {
            pVertexBuffer[0] = cosXZ;
            pVertexBuffer[1] = 0.0f;
            pVertexBuffer[2] = sinXZ;
            pVertexBuffer += 3;
        }

        // 法線
        // 底面側のため法線は下向き
        if ( GetVertexFormat() & PrimitiveShapeFormat_Normal )
        {
            pVertexBuffer[0] =  0.0f;
            pVertexBuffer[1] = -1.0f;
            pVertexBuffer[2] =  0.0f;
            pVertexBuffer += 3;
        }

        // UV
        if ( GetVertexFormat() & PrimitiveShapeFormat_Uv )
        {
            pVertexBuffer[0] = 0.5f + cosXZ * 0.5f;
            pVertexBuffer[1] = 1.0f;
            pVertexBuffer += 2;
        }
    }

    nn::util::Float3 centerPos = NN_UTIL_FLOAT_3_INITIALIZER( 0.0f, 0.0f, 0.0f );
    // 座標
    if ( GetVertexFormat() & PrimitiveShapeFormat_Pos )
    {
        pVertexBuffer[0] = centerPos.x;
        pVertexBuffer[1] = centerPos.y;
        pVertexBuffer[2] = centerPos.z;
        pVertexBuffer += 3;
    }

    // 法線
    if ( GetVertexFormat() & PrimitiveShapeFormat_Normal )
    {
        pVertexBuffer[0] =  0.0f;
        pVertexBuffer[1] = -1.0f;
        pVertexBuffer[2] =  0.0f;
        pVertexBuffer += 3;
    }

    // UV
    if ( GetVertexFormat() & PrimitiveShapeFormat_Uv )
    {
        pVertexBuffer[0] = 0.5f;
        pVertexBuffer[1] = 1.0f;
        pVertexBuffer += 2;
    }

    // 側面の頂点データ
    for ( int i = 0; i < m_SliceCount; ++i)
    {
        float rad = ( nn::util::DegreeToRadian(360.0f) / m_SliceCount ) * i;
        float cosXZ = nn::util::CosTable( nn::util::RadianToAngleIndex(rad) );
        float sinXZ = nn::util::SinTable( nn::util::RadianToAngleIndex(rad) );
        // 座標
        if ( GetVertexFormat() & PrimitiveShapeFormat_Pos )
        {
            pVertexBuffer[0] = cosXZ;
            pVertexBuffer[1] = 0.0f;
            pVertexBuffer[2] = sinXZ;
            pVertexBuffer += 3;
        }

        // 法線
        if ( GetVertexFormat() & PrimitiveShapeFormat_Normal )
        {
            pVertexBuffer[0] = cosXZ;
            pVertexBuffer[1] = 0.0f;
            pVertexBuffer[2] = sinXZ;
            pVertexBuffer += 3;
        }

        // UV
        if ( GetVertexFormat() & PrimitiveShapeFormat_Uv )
        {
            pVertexBuffer[0] = 0.5f + cosXZ * 0.5f;
            pVertexBuffer[1] = 1.0f;
            pVertexBuffer += 2;
        }
    }

    centerPos.y += 1.0f;

    // 座標
    if ( GetVertexFormat() & PrimitiveShapeFormat_Pos )
    {
        pVertexBuffer[0] = centerPos.x;
        pVertexBuffer[1] = centerPos.y;
        pVertexBuffer[2] = centerPos.z;
        pVertexBuffer += 3;
    }

    // 法線
    if ( GetVertexFormat() & PrimitiveShapeFormat_Normal )
    {
        pVertexBuffer[0] = centerPos.x;
        pVertexBuffer[1] = 1.0f;
        pVertexBuffer[2] = centerPos.z;
        pVertexBuffer += 3;
    }

    // UV
    if ( GetVertexFormat() & PrimitiveShapeFormat_Uv )
    {
        pVertexBuffer[0] = 0.5f;
        pVertexBuffer[1] = 0.0f;
        pVertexBuffer += 2;
    }

    // 頂点バッファの末端アドレスを返す。
    return pVertexBuffer;
}

template <typename T>
void ConeShape::CalculateIndexBuffer()
{
    T* pIndexData = static_cast<T*>(GetIndexBuffer());

    const nn::gfx::PrimitiveTopology primitiveTopology = GetPrimitiveTopology();

    // 底面のインデックス.
    for( int i = 0; i < m_SliceCount; ++i )
    {
        switch( primitiveTopology )
        {
        // ワイヤフレーム
        case nn::gfx::PrimitiveTopology::PrimitiveTopology_LineList:
            {
                pIndexData[i * 2    ] = static_cast<T>(i);
                pIndexData[i * 2 + 1] = static_cast<T>((i + 1) % m_SliceCount);
            }
            break;
        // ソリッド
        case nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleList:
            {
                pIndexData[i * 3 + 0] = static_cast<T>((i + 1) % m_SliceCount);
                pIndexData[i * 3 + 1] = static_cast<T>(m_SliceCount);
                pIndexData[i * 3 + 2] = static_cast<T>(i);
            }
            break;
        default:
            break;
        }
    }

    // 側面のインデックス.
    for(int i=0; i< m_SliceCount; ++i)
    {
        switch( primitiveTopology )
        {
        // ワイヤフレーム
        case nn::gfx::PrimitiveTopology::PrimitiveTopology_LineList:
            {
                pIndexData[m_SliceCount * 2 + i * 2    ] = static_cast<T>(i);
                pIndexData[m_SliceCount * 2 + i * 2 + 1] = static_cast<T>(m_SliceCount * 2 + 1);
            }
            break;
        // ソリッド
        case nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleList:
            {
                int indexBaseOffset = m_SliceCount + 1;
                pIndexData[m_SliceCount * 3 + i * 3    ] = static_cast<T>(indexBaseOffset + i);
                pIndexData[m_SliceCount * 3 + i * 3 + 1] = static_cast<T>(m_SliceCount * 2 + 1);
                pIndexData[m_SliceCount * 3 + i * 3 + 2] = static_cast<T>(indexBaseOffset + ((i + 1) % m_SliceCount));
           }
            break;
        default:
            break;
        }
    }
}

void ConeShape::CalculateImpl(void* pVertexMemory, size_t vertexSize, void* pIndexMemory, size_t indexSize)
{
    NN_UNUSED(vertexSize);
    NN_UNUSED(indexSize);

    // バッファサイズが足りているかチェックします。
    NN_SDK_ASSERT( ( vertexSize >= GetVertexBufferSize() ) ||  ( indexSize >= GetIndexBufferSize() ));

    // 頂点バッファのセット
    SetVertexBuffer(pVertexMemory);

    // 頂点バッファの計算
    CalculateVertexBuffer();

    // インデックスバッファのセット
    SetIndexBuffer(pIndexMemory);

    // インデックスバッファの計算
    switch (GetIndexBufferFormat())
    {
    case nn::gfx::IndexFormat_Uint8:
        CalculateIndexBuffer<uint8_t>();
        break;
    case nn::gfx::IndexFormat_Uint16:
        CalculateIndexBuffer<uint16_t>();
        break;
    case nn::gfx::IndexFormat_Uint32:
        CalculateIndexBuffer<uint32_t>();
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }

}

} // namespace util
} // namespace gfx
} // namespace nn
