﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/gfx/util/gfx_MemoryPoolAllocator.h>
#include <nn/util/util_BitUtil.h>
#include <nn/nn_SdkAssert.h>


namespace nn {
namespace gfx {
namespace util {


MemoryPoolAllocator::MemoryPoolAllocator() NN_NOEXCEPT :
m_Impl(),
m_pMemoryPool(nullptr),
m_BaseOffset(0),
m_Size(0),
m_AllocatableAlignmentMax(0),
m_IsInitialized(false)
{}

void MemoryPoolAllocator::Initialize(
    nn::mem::MallocCallback pAllocateFunction,
    void* pAllocateFunctionUserData,
    nn::mem::FreeCallback pFreeFunction,
    void* pFreeFunctionUserData,
    nn::gfx::MemoryPool* pMemoryPool,
    ptrdiff_t baseOffset,
    size_t size,
    size_t allocatableAlignmentMax,
    bool isThreadSafe
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!m_IsInitialized);
    NN_SDK_REQUIRES_NOT_NULL(pAllocateFunction);
    NN_SDK_REQUIRES_NOT_NULL(pFreeFunction);
    NN_SDK_REQUIRES_NOT_NULL(pMemoryPool);
    NN_SDK_REQUIRES_GREATER_EQUAL(baseOffset, 0);
    NN_SDK_REQUIRES((baseOffset % allocatableAlignmentMax) == 0); //baseOffset は allocatableAlignmentMax でアラインされている
    NN_SDK_REQUIRES_MINMAX(size, static_cast<size_t>(1), static_cast<size_t>(SizeMax));
    NN_SDK_REQUIRES_MINMAX(allocatableAlignmentMax, static_cast<size_t>(1), static_cast<size_t>(AlignmentMax));
    NN_SDK_REQUIRES((allocatableAlignmentMax & (allocatableAlignmentMax - 1)) == 0);    //alignmentMax は 2 の累乗

    m_pMemoryPool = pMemoryPool;
    m_BaseOffset = baseOffset;
    m_Size = size;
    m_AllocatableAlignmentMax = allocatableAlignmentMax;

    size_t unitCount = size / AllocatorUnitSize;    //半端な分は切り捨てる
    //unitCount == 0 だった場合は、Allocate() が常に失敗する。

    m_Impl.Initialize(pAllocateFunction, pAllocateFunctionUserData, pFreeFunction, pFreeFunctionUserData, isThreadSafe);
    if(unitCount > 0){
        m_Impl.AddRange(0, static_cast<int>(unitCount));
    }
    m_IsInitialized = true;
}

void MemoryPoolAllocator::Finalize() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsInitialized);

    m_Impl.Finalize();
    m_pMemoryPool = nullptr;
    m_IsInitialized = false;
}

bool MemoryPoolAllocator::IsInitialized() const NN_NOEXCEPT
{
    return m_IsInitialized;
}

ptrdiff_t MemoryPoolAllocator::Allocate(size_t size, size_t alignment) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsInitialized);
    NN_SDK_REQUIRES((alignment & (alignment - 1)) == 0);    //alignment は 2 の累乗
    NN_SDK_REQUIRES_MINMAX(alignment, static_cast<size_t>(1), m_AllocatableAlignmentMax);

    size_t unitCount = nn::util::align_up(size, AllocatorUnitSize) / AllocatorUnitSize;
    //AllocatorUnitSize 以下のアライメント要求ならそのまま Alloc できる。
    //それより大きいアライメント要求なら、AlignedAlloc する必要がある。
    size_t alignmentUnit = alignment / AllocatorUnitSize;
    int indexUnit;
    if(alignmentUnit > 1)
    {
        if(m_Impl.Allocate(&indexUnit, static_cast<int>(unitCount), static_cast<int>(alignmentUnit)) != true){
            return InvalidOffset;
        }
    }
    else
    {
        if(m_Impl.Allocate(&indexUnit, static_cast<int>(unitCount)) != true){
            return InvalidOffset;
        }
    }
    return m_BaseOffset + ( AllocatorUnitSize * static_cast<ptrdiff_t>(indexUnit) );
}

void MemoryPoolAllocator::Free(ptrdiff_t offset) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsInitialized);

    if (offset == InvalidOffset) {
        return;
    }

    NN_SDK_REQUIRES(((offset - m_BaseOffset) % AllocatorUnitSize) == 0);

    int indexUnit = static_cast<int>( (offset - m_BaseOffset) / AllocatorUnitSize );
    m_Impl.Free(indexUnit);
}

nn::gfx::MemoryPool* MemoryPoolAllocator::GetMemoryPool() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsInitialized);

    return m_pMemoryPool;
}

ptrdiff_t MemoryPoolAllocator::GetBaseOffset() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsInitialized);

    return m_BaseOffset;
}

size_t MemoryPoolAllocator::GetSize() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsInitialized);

    return m_Size;
}

size_t MemoryPoolAllocator::GetAllocatableAlignmentMax() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsInitialized);

    return m_AllocatableAlignmentMax;
}

size_t MemoryPoolAllocator::GetSizeOf(ptrdiff_t offset) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsInitialized);

    if (offset == InvalidOffset) {
        return 0;
    }

    NN_SDK_REQUIRES(((offset - m_BaseOffset) % AllocatorUnitSize) == 0);

    int indexUnit = static_cast<int>((offset - m_BaseOffset) / AllocatorUnitSize);
    size_t unitCount = static_cast<size_t>(m_Impl.GetSizeOf(indexUnit));
    return unitCount * AllocatorUnitSize;
}

size_t MemoryPoolAllocator::GetTotalFreeSize() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsInitialized);

    size_t unitCount = static_cast<size_t>(m_Impl.GetTotalFreeSize());
    return unitCount * AllocatorUnitSize;
}

size_t MemoryPoolAllocator::GetAllocatableSize() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsInitialized);

    size_t unitCount = static_cast<size_t>(m_Impl.GetAllocatableSize());
    return unitCount * AllocatorUnitSize;
}

void MemoryPoolAllocator::Dump() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsInitialized);

    m_Impl.Dump(m_BaseOffset, AllocatorUnitSize);
}


} // namespace util
} // namespace gfx
} // namespace nn
