﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/gfx/util/detail/gfx_IndexRingBuffer.h>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>

namespace nn{ namespace gfx{ namespace util{ namespace detail{

    IndexRingBuffer::IndexRingBuffer() NN_NOEXCEPT
    {
        m_BaseIndex = 0;
        m_IndexCount = 0;
        m_HeadLocalIndex.store(0, std::memory_order_relaxed);
        m_TailLocalIndex.store(0, std::memory_order_relaxed);
        m_LastLocalIndex = 0;
        m_IsInitialized = false;
        m_IsRecording.store(false, std::memory_order_relaxed);
    }

    void IndexRingBuffer::Initialize(
        int base,
        int count
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!m_IsInitialized);
        NN_SDK_REQUIRES_GREATER_EQUAL(base, 0);
        NN_SDK_REQUIRES_GREATER_EQUAL(count, 1);
        NN_SDK_REQUIRES_LESS_EQUAL(count, static_cast<int>(IndexCountMax));
        NN_SDK_REQUIRES_LESS_EQUAL(base, INT_MAX - count + 1);

        m_BaseIndex   = base;
        m_IndexCount  = count;

        m_HeadLocalIndex.store(0, std::memory_order_relaxed);
        m_TailLocalIndex.store(count - 1, std::memory_order_relaxed);
        m_LastLocalIndex = 0;

        m_IsInitialized = true;
        m_IsRecording = false;
    }

    void IndexRingBuffer::Finalize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        m_IsInitialized = false;
    }

    bool IndexRingBuffer::IsInitialized() const NN_NOEXCEPT
    {
        return m_IsInitialized;
    }

    bool IndexRingBuffer::IsRecording() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        return m_IsRecording;
    }

    int IndexRingBuffer::GetBaseIndex() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        return m_BaseIndex;
    }
    int IndexRingBuffer::GetIndexCount() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        return m_IndexCount;
    }

    void IndexRingBuffer::Begin() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        NN_SDK_REQUIRES(!m_IsRecording);

        NN_SDK_ASSERT_RANGE(m_HeadLocalIndex, 0, m_IndexCount);
        NN_SDK_ASSERT_RANGE(m_TailLocalIndex, m_HeadLocalIndex, m_HeadLocalIndex + m_IndexCount);
        NN_SDK_ASSERT_EQUAL(m_HeadLocalIndex, m_LastLocalIndex);

        m_IsRecording.store(true);
    }

    void IndexRingBuffer::End(IndexRange* pOutRange) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        NN_SDK_REQUIRES(m_IsRecording);
        NN_SDK_REQUIRES_NOT_NULL(pOutRange);

        m_IsRecording.store(false);

        int head = m_HeadLocalIndex.load(std::memory_order_relaxed);
        int tail = m_TailLocalIndex.load(std::memory_order_relaxed);
        int last = m_LastLocalIndex;

        pOutRange->base  = last;
        pOutRange->count = head - last;

        NN_SDK_REQUIRES_RANGE(pOutRange->base, 0, m_IndexCount);
        NN_SDK_REQUIRES_RANGE(pOutRange->count, 0, m_IndexCount);

        // 1 周していたら戻す
        if(head >= m_IndexCount)
        {
            head -= m_IndexCount;
            tail -= m_IndexCount;
            m_HeadLocalIndex.store(head, std::memory_order_relaxed);
            m_TailLocalIndex.store(tail, std::memory_order_relaxed);
        }

        NN_SDK_ASSERT_RANGE(head, 0, m_IndexCount);
        NN_SDK_ASSERT_RANGE(tail, head, head + m_IndexCount);

        m_LastLocalIndex = head;
    }

    void IndexRingBuffer::ReleaseIndexRange(const IndexRange* pRange) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        NN_SDK_REQUIRES_NOT_NULL(pRange);

        // ゼロ初期化の値の場合は何もしない
        if(pRange->base == 0 && pRange->count == 0)
        {
            return;
        }

        NN_SDK_REQUIRES_RANGE(pRange->base , 0, m_IndexCount);
        NN_SDK_REQUIRES_RANGE(pRange->count, 0, m_IndexCount);
        // 順番通りに解放していることを確認
        NN_SDK_REQUIRES_EQUAL((m_TailLocalIndex + 1) % m_IndexCount, pRange->base);
        // Last の位置を追い越さないことを確認
        NN_SDK_REQUIRES_LESS(m_TailLocalIndex + pRange->count, m_LastLocalIndex + m_IndexCount);

        m_TailLocalIndex.fetch_add(pRange->count, std::memory_order_release);
    }

    int IndexRingBuffer::AcquireIndex() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsRecording);
        NN_SDK_REQUIRES(m_IsInitialized);

        int head = m_HeadLocalIndex.load(std::memory_order_acquire);
        int tail;
        do{
            tail = m_TailLocalIndex.load(std::memory_order_acquire);
            if(head == tail)
            {
                return InvalidIndex;
            }
        }while(!m_HeadLocalIndex.compare_exchange_weak(head, head + 1));

        NN_SDK_ASSERT_LESS(head, tail);

        int result = m_BaseIndex + (head % m_IndexCount);
        return result;
    }

    int IndexRingBuffer::AcquireIndexRange(int count) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsRecording);
        NN_SDK_REQUIRES(m_IsInitialized);
        NN_SDK_REQUIRES_GREATER_EQUAL(count, 0);

        // * 確保する数 は 1 以上。
        // * 空の状態でも (m_IndexCount - 1) スロットしか確保できない。
        if(count == 0 || count >= m_IndexCount)
        {
            return InvalidIndex;
        }

        int head = m_HeadLocalIndex.load(std::memory_order_acquire);
        int tail;
        int newHead;
        int resultLocal;
        do{
            tail = m_TailLocalIndex.load(std::memory_order_acquire);
            // * 前提として { 0 <= head <= tail }。
            //   仮に { head == 0 } としても { count > tail } なら確保できない。
            // * { head + count > tail } なら確保できない。
            if(count > tail || head > tail - count)
            {
                return InvalidIndex;
            }
            newHead = head + count;
            resultLocal = head;
            // 取得した範囲がリングバッファの終端を跨いでいたら 2 周目の先頭から取る
            //   * 最初から 2 周目から確保していれば head >= m_IndexCount。
            //   * 1 周目の最後までに収まったら newHead <= m_IndexCount。
            if(head < m_IndexCount && newHead > m_IndexCount)
            {
                // * {m_IndexCount * 2 < INT_MAX} で、 {count < m_IndexCount} なので {m_IndexCount + count < INT_MAX}。
                newHead = m_IndexCount + count;
                resultLocal = m_IndexCount;
                if(newHead > tail)
                {
                    return InvalidIndex;
                }
            }
        }while(!m_HeadLocalIndex.compare_exchange_weak(head, newHead));

        NN_SDK_ASSERT_LESS(head, tail);

        int result = m_BaseIndex + (resultLocal % m_IndexCount);
        return result;
    }

}}}}
