﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/gfx/gfx_Interop-api.nvn.h>

#include <nn/gfx/gfx_Queue.h>
#include <nn/gfx/gfx_MemoryPool.h>
#include <nn/gfx/gfx_DescriptorPool.h>
#include <nn/gfx/gfx_Buffer.h>
#include <nn/gfx/gfx_CommandBuffer.h>
#include <nn/gfx/gfx_Sampler.h>
#include <nn/gfx/gfx_Shader.h>
#include <nn/gfx/gfx_Texture.h>
#include <nn/gfx/gfx_Device.h>

#include <nn/gfx/detail/gfx_Queue-api.nvn.8.h>
#include <nn/gfx/detail/gfx_MemoryPool-api.nvn.8.h>
#include <nn/gfx/detail/gfx_DescriptorPool-api.nvn.8.h>
#include <nn/gfx/detail/gfx_Buffer-api.nvn.8.h>
#include <nn/gfx/detail/gfx_CommandBuffer-api.nvn.8.h>
#include <nn/gfx/detail/gfx_Sampler-api.nvn.8.h>
#include <nn/gfx/detail/gfx_Shader-api.nvn.8.h>
#include <nn/gfx/detail/gfx_Texture-api.nvn.8.h>
#include <nn/gfx/detail/gfx_Device-api.nvn.8.h>

#include "detail/gfx_CommonHelper.h"
#include "detail/gfx_NvnHelper.h"

namespace nn {
namespace gfx {

typedef ApiVariation< ApiTypeNvn > Target;

void TInteroperation< Target >::ConvertToGfxDevice(
    TDevice< ApiVariationNvn8 >* pOutGfxDevice, NVNdevice* pNvnDevice ) NN_NOEXCEPT
{
    detail::UseMiddleWare();

    NN_SDK_REQUIRES_NOT_NULL( pOutGfxDevice );
    NN_SDK_REQUIRES_NOT_NULL( pNvnDevice );
    NN_SDK_REQUIRES( !IsInitialized( *pOutGfxDevice ) ||
        pOutGfxDevice->ToData()->flags.GetBit( TDevice< Target >::DataType::Flag_Shared ) );

    TDevice< Target >::DataType& obj = pOutGfxDevice->ToData();
    obj.state = TDevice< Target >::DataType::State_Initialized;
    obj.flags.SetBit( TDevice< Target >::DataType::Flag_Shared, true );
    obj.pNvnDevice = pNvnDevice;
    obj.supportedFeatures = detail::Nvn::GetDeviceFeature( pNvnDevice );
}

void TInteroperation< Target >::ConvertToGfxQueue( TQueue< Target >* pOutGfxQueue,
    NVNqueue* pNvnQueue, TDevice< Target >* pGfxDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pOutGfxQueue );
    NN_SDK_REQUIRES_NOT_NULL( pNvnQueue );
    NN_SDK_REQUIRES( !IsInitialized( *pOutGfxQueue ) ||
        pOutGfxQueue->ToData()->flags.GetBit( TQueue< Target >::DataType::Flag_Shared ) );
    NN_SDK_REQUIRES( pGfxDevice == NULL || IsInitialized( *pGfxDevice ) );

    TQueue< Target >::DataType& obj = pOutGfxQueue->ToData();
    obj.state = TQueue< Target >::DataType::State_Initialized;
    obj.flags.SetBit( TQueue< Target >::DataType::Flag_Shared, true );
    obj.pNvnQueue = pNvnQueue;
    obj.pNnDevice = pGfxDevice;
    obj.pImpl = NULL;
}

void TInteroperation< Target >::ConvertToGfxMemoryPool( TMemoryPool< Target >* pOutGfxMemoryPool,
    NVNmemoryPool* pNvnMemoryPool, void* pMemory ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pOutGfxMemoryPool );
    NN_SDK_REQUIRES_NOT_NULL( pNvnMemoryPool );
    NN_SDK_REQUIRES( !IsInitialized( *pOutGfxMemoryPool ) ||
        pOutGfxMemoryPool->ToData()->flags.GetBit( TMemoryPool< Target >::DataType::Flag_Shared ) );

    TMemoryPool< Target >::DataType& obj = pOutGfxMemoryPool->ToData();
    obj.state = TMemoryPool< Target >::DataType::State_Initialized;
    obj.flags.SetBit( TMemoryPool< Target >::DataType::Flag_Shared, true );
    obj.pNvnMemoryPool = pNvnMemoryPool;
    obj.pMemory = pMemory;
}

void TInteroperation< Target >::ConvertToGfxDescriptorPool(
    TDescriptorPool< Target >* pOutGfxDescriptorPool, NVNtexturePool* pNvnTexturePool ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pOutGfxDescriptorPool );
    NN_SDK_REQUIRES_NOT_NULL( pNvnTexturePool );
    NN_SDK_REQUIRES( !IsInitialized( *pOutGfxDescriptorPool ) );

    TDescriptorPool< Target >::DataType& obj = pOutGfxDescriptorPool->ToData();
    obj.descriptorPoolType = DescriptorPoolType_TextureView;
    NN_GFX_CALL_NVN_FUNCTION( obj.slotCount = nvnTexturePoolGetSize( pNvnTexturePool ) );
    obj.pDescriptorPool = pNvnTexturePool;
    obj.reservedSlots = 256; // TODO
    obj.flags.SetBit( TDescriptorPool< Target >::DataType::Flag_Shared, true );
    obj.state = TDescriptorPool< Target >::DataType::State_Initialized;
}

void TInteroperation< Target >::ConvertToGfxDescriptorPool(
    TDescriptorPool< Target >* pOutGfxDescriptorPool, NVNsamplerPool* pNvnSamplerPool ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pOutGfxDescriptorPool );
    NN_SDK_REQUIRES_NOT_NULL( pNvnSamplerPool );
    NN_SDK_REQUIRES( !IsInitialized( *pOutGfxDescriptorPool ) );

    TDescriptorPool< Target >::DataType& obj = pOutGfxDescriptorPool->ToData();
    obj.descriptorPoolType = DescriptorPoolType_Sampler;
    NN_GFX_CALL_NVN_FUNCTION( obj.slotCount = nvnSamplerPoolGetSize( pNvnSamplerPool ) );
    obj.pDescriptorPool = pNvnSamplerPool;
    obj.reservedSlots = 256; // TODO
    obj.flags.SetBit( TDescriptorPool< Target >::DataType::Flag_Shared, true );
    obj.state = TDescriptorPool< Target >::DataType::State_Initialized;
}

void TInteroperation< Target >::ConvertToGfxBuffer( TBuffer< Target >* pOutGfxBuffer,
    NVNbuffer* pNvnBuffer ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pOutGfxBuffer );
    NN_SDK_REQUIRES_NOT_NULL( pNvnBuffer );
    NN_SDK_REQUIRES( !IsInitialized( *pOutGfxBuffer ) ||
        pOutGfxBuffer->ToData()->flags.GetBit( TBuffer< Target >::DataType::Flag_Shared ) );

    NVNmemoryPool* pNvnMemoryPool = NULL;
    NVNmemoryPoolFlags nvnMemoryPoolFlags = NVNmemoryPoolFlags();
    NN_GFX_CALL_NVN_FUNCTION( pNvnMemoryPool = nvnBufferGetMemoryPool( pNvnBuffer ) );
    NN_SDK_ASSERT_NOT_NULL( pNvnMemoryPool );
    NN_GFX_CALL_NVN_FUNCTION( nvnMemoryPoolFlags = nvnMemoryPoolGetFlags( pNvnMemoryPool ) );

    TBuffer< Target >::DataType& obj = pOutGfxBuffer->ToData();
    obj.state = TBuffer< Target >::DataType::State_Initialized;
    obj.flags.SetBit( TBuffer< Target >::DataType::Flag_CpuCached,
        ( nvnMemoryPoolFlags & NVN_MEMORY_POOL_FLAGS_CPU_CACHED_BIT ) != 0 );
    obj.flags.SetBit( TBuffer< Target >::DataType::Flag_Shared, true );
    obj.pNvnBuffer = pNvnBuffer;
}

void TInteroperation< Target >::ConvertToGfxCommandBuffer(
    TCommandBuffer< Target >* pOutGfxCommandBuffer,
    TDevice< Target >* pDevice, NVNcommandBuffer* pNvnCommandBuffer ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pOutGfxCommandBuffer );
    NN_SDK_REQUIRES_NOT_NULL( pNvnCommandBuffer );
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( !IsInitialized( *pOutGfxCommandBuffer ) ||
        pOutGfxCommandBuffer->ToData()->flags.GetBit(
        TCommandBuffer< Target >::DataType::Flag_Shared ) );

    TCommandBuffer< Target >::DataType& obj = pOutGfxCommandBuffer->ToData();
    obj.state = TCommandBuffer< Target >::DataType::State_Initialized;
    obj.flags.SetBit( TCommandBuffer< Target >::DataType::Flag_Shared, true );
    obj.pNnDevice = pDevice;
    obj.pNvnCommandBuffer = pNvnCommandBuffer;
}

void TInteroperation< Target >::ConvertToGfxSampler(
    TSampler< Target >* pOutGfxSampler, NVNsampler* pNvnSampler ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pOutGfxSampler );
    NN_SDK_REQUIRES_NOT_NULL( pNvnSampler );
    NN_SDK_REQUIRES( !IsInitialized( *pOutGfxSampler ) ||
        pOutGfxSampler->ToData()->flags.GetBit( TSampler< Target >::DataType::Flag_Shared ) );

    TSampler< Target >::DataType& obj = pOutGfxSampler->ToData();
    obj.state = TSampler< Target >::DataType::State_Initialized;
    obj.flags.SetBit( TSampler< Target >::DataType::Flag_Shared, true );
    obj.pNvnSampler = pNvnSampler;
}

void TInteroperation< Target >::ConvertToGfxShader( TShader< Target >* pOutGfxShader,
    NVNprogram* pNvnProgram, bool isSeparable, int stageBits ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pOutGfxShader );
    NN_SDK_REQUIRES_NOT_NULL( pNvnProgram );
    NN_SDK_REQUIRES( !IsInitialized( *pOutGfxShader ) ||
        pOutGfxShader->ToData()->flags.GetBit( TShader< Target >::DataType::Flag_Shared ) );

    TShader< Target >::DataType& obj = pOutGfxShader->ToData();
    obj.state = TShader< Target >::DataType::State_Initialized;
    obj.flags.SetBit( TShader< Target >::DataType::Flag_Shared, true );
    obj.flags.SetBit( TShader< Target >::DataType::Flag_SeparationEnable, isSeparable );
    obj.nvnShaderStageBits = ( stageBits & ShaderStageBit_Compute ) ?
        NVN_SHADER_STAGE_COMPUTE_BIT : NVN_SHADER_STAGE_ALL_GRAPHICS_BITS;
    obj.pNvnProgram = pNvnProgram;
    obj.pReflection = NULL;
}

void TInteroperation< Target >::ConvertToGfxTexture(
    TTexture< Target >* pOutGfxTexture, NVNtexture* pNvnTexture ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pOutGfxTexture );
    NN_SDK_REQUIRES_NOT_NULL( pNvnTexture );
    NN_SDK_REQUIRES( !IsInitialized( *pOutGfxTexture ) ||
        pOutGfxTexture->ToData()->flags.GetBit( TTexture< Target >::DataType::Flag_Shared ) );

    TTexture< Target >::DataType& obj = pOutGfxTexture->ToData();
    obj.state = TTexture< Target >::DataType::State_Initialized;
    obj.flags.SetBit( TTexture< Target >::DataType::Flag_Shared, true );
    obj.pNvnTexture = pNvnTexture;
}

void TInteroperation< Target >::ConvertToGfxTextureView( TTextureView< Target >* pOutGfxTextureView,
    NVNtexture* pNvnTexture, NVNtextureView* pNvnTextureView ) NN_NOEXCEPT
{
    // テクスチャービューの NULL は許可する
    NN_SDK_REQUIRES_NOT_NULL( pOutGfxTextureView );
    NN_SDK_REQUIRES_NOT_NULL( pNvnTexture );
    NN_SDK_REQUIRES( !IsInitialized( *pOutGfxTextureView ) ||
        pOutGfxTextureView->ToData()->flags.GetBit( TTextureView< Target >::DataType::Flag_Shared ) );

    TTextureView< Target >::DataType& obj = pOutGfxTextureView->ToData();
    obj.state = TTextureView< Target >::DataType::State_Initialized;
    obj.flags.SetBit( TTextureView< Target >::DataType::Flag_Shared, true );
    obj.pNvnTexture = pNvnTexture;
    obj.pNvnTextureView = pNvnTextureView;
}

void TInteroperation< Target >::ConvertToGfxColorTargetView( TColorTargetView< Target >* pOutGfxTextureView,
    NVNtexture* pNvnTexture, NVNtextureView* pNvnTextureView ) NN_NOEXCEPT
{
    // テクスチャービューの NULL は許可する
    NN_SDK_REQUIRES_NOT_NULL( pOutGfxTextureView );
    NN_SDK_REQUIRES_NOT_NULL( pNvnTexture );
    NN_SDK_REQUIRES( !IsInitialized( *pOutGfxTextureView ) ||
        pOutGfxTextureView->ToData()->flags.GetBit( TTextureView< Target >::DataType::Flag_Shared ) );

    TColorTargetView< Target >::DataType& obj = pOutGfxTextureView->ToData();
    obj.state = TColorTargetView< Target >::DataType::State_Initialized;
    obj.flags.SetBit( TColorTargetView< Target >::DataType::Flag_Shared, true );
    obj.pNvnTexture = pNvnTexture;
    obj.pNvnTextureView = pNvnTextureView;
}

void TInteroperation< Target >::ConvertToGfxDepthStencilView( TDepthStencilView< Target >* pOutGfxDepthStencilView,
    NVNtexture* pNvnTexture, NVNtextureView* pNvnTextureView ) NN_NOEXCEPT
{
    // テクスチャービューの NULL は許可する
    NN_SDK_REQUIRES_NOT_NULL( pOutGfxDepthStencilView );
    NN_SDK_REQUIRES_NOT_NULL( pNvnTexture );
    NN_SDK_REQUIRES( !IsInitialized( *pOutGfxDepthStencilView ) ||
        pOutGfxDepthStencilView->ToData()->flags.GetBit( TTextureView< Target >::DataType::Flag_Shared ) );

    TDepthStencilView< Target >::DataType& obj = pOutGfxDepthStencilView->ToData();
    obj.state = TDepthStencilView< Target >::DataType::State_Initialized;
    obj.flags.SetBit( TDepthStencilView< Target >::DataType::Flag_Shared, true );
    obj.pNvnTexture = pNvnTexture;
    obj.pNvnTextureView = pNvnTextureView;
}

NVNformat TInteroperation< Target >::ConvertToNvnFormat( ImageFormat gfxImageFormat ) NN_NOEXCEPT
{
    return detail::Nvn::GetImageFormat( gfxImageFormat );
}

NVNformat TInteroperation< Target >::ConvertToNvnFormat( AttributeFormat gfxAttributeFormat ) NN_NOEXCEPT
{
    return detail::Nvn::GetAttributeFormat( gfxAttributeFormat );
}

}
}
