﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/gfx/gfx_Interop-api.gl.4.h>

#include <nn/gfx/gfx_Device.h>
#include <nn/gfx/gfx_Buffer.h>
#include <nn/gfx/gfx_Sampler.h>
#include <nn/gfx/gfx_Shader.h>
#include <nn/gfx/gfx_Texture.h>

#include <nn/gfx/detail/gfx_Device-api.gl.4.h>
#include <nn/gfx/detail/gfx_Buffer-api.gl.4.h>
#include <nn/gfx/detail/gfx_Sampler-api.gl.4.h>
#include <nn/gfx/detail/gfx_Shader-api.gl.4.h>
#include <nn/gfx/detail/gfx_Texture-api.gl.4.h>

#include "detail/gfx_GlHelper.h"

namespace nn {
namespace gfx {

typedef ApiVariationGl4 Target;

void TInteroperation< Target >::ConvertToGfxBuffer( TBuffer< Target >* pOutGfxBuffer,
    TDevice< Target >* pDevice, GlHandleType hGl4Buffer ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pOutGfxBuffer );
    NN_SDK_REQUIRES( !IsInitialized( *pOutGfxBuffer ) ||
        pOutGfxBuffer->ToData()->flags.GetBit( TBuffer< Target >::DataType::Flag_Shared ) );
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( detail::IsValid( hGl4Buffer ) );

    TBuffer< Target >::DataType& obj = pOutGfxBuffer->ToData();
    obj.state = TBuffer< Target >::DataType::State_Initialized;
    obj.flags.SetBit( TBuffer< Target >::DataType::Flag_Shared, true );
    obj.hBuffer = hGl4Buffer;
    obj.pGfxDevice = static_cast< detail::DeviceImpl< Target >* >( static_cast< void* >( pDevice ) );

    detail::GlDeviceActivator activator( pDevice );

    // マップ対応はとりあえず PERSISTENT ビットがついている場合のみ
    obj.pMapped = NULL;
    GLint isMapped;
    NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glGetNamedBufferParameteriv )(
        obj.hBuffer, GL_BUFFER_MAPPED, &isMapped ) );
    const int checkFlags = GL_MAP_READ_BIT | GL_MAP_WRITE_BIT | GL_MAP_PERSISTENT_BIT;
    if( isMapped == GL_TRUE )
    {
        GLint mapFlags;
        NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glGetNamedBufferParameteriv )(
            obj.hBuffer, GL_BUFFER_ACCESS_FLAGS, &mapFlags ) );
        if( ( mapFlags & checkFlags ) == checkFlags )
        {
            NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glGetNamedBufferPointerv )(
                obj.hBuffer, GL_BUFFER_MAP_POINTER, &obj.pMapped ) );
            obj.flags.SetBit( TBuffer< Target >::DataType::Flag_FlushExplicit,
                ( mapFlags & GL_MAP_FLUSH_EXPLICIT_BIT ) != 0 );
        }
    }
    else
    {
        GLint storageFlags;
        NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glGetNamedBufferParameteriv )(
            obj.hBuffer, GL_BUFFER_STORAGE_FLAGS, &storageFlags ) );
        if( ( storageFlags & checkFlags ) == checkFlags )
        {
            GLint size;
            NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glGetNamedBufferParameteriv )(
                obj.hBuffer, GL_BUFFER_SIZE, &size ) );
            obj.pMapped = NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glMapNamedBufferRange )(
                obj.hBuffer, 0, size, checkFlags | ( ( storageFlags & GL_MAP_COHERENT_BIT )
                ? 0 : GL_MAP_FLUSH_EXPLICIT_BIT ) ) );
            obj.flags.SetBit( TBuffer< Target >::DataType::Flag_FlushExplicit,
                ( storageFlags & GL_MAP_COHERENT_BIT ) == 0 );
        }
    }

    NN_GFX_GL_ASSERT();
}

void TInteroperation< Target >::ConvertToGfxSampler( TSampler<
    Target >* pOutGfxSampler, GlHandleType hGl4Sampler ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pOutGfxSampler );
    NN_SDK_REQUIRES( !IsInitialized( *pOutGfxSampler ) ||
        pOutGfxSampler->ToData()->flags.GetBit( TSampler< Target >::DataType::Flag_Shared ) );
    NN_SDK_REQUIRES( detail::IsValid( hGl4Sampler ) );

    TSampler< Target >::DataType& obj = pOutGfxSampler->ToData();
    obj.state = TSampler< Target >::DataType::State_Initialized;
    obj.flags.SetBit( TSampler< Target >::DataType::Flag_Shared, true );
    obj.hSampler = hGl4Sampler;
}

void TInteroperation< Target >::ConvertToGfxShader( TShader< ApiVariationGl4 >* pOutGfxShader,
    TDevice< Target >* pDevice, GlHandleType hGl4Shader ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pOutGfxShader );
    NN_SDK_REQUIRES( !IsInitialized( *pOutGfxShader ) ||
        pOutGfxShader->ToData()->flags.GetBit( TShader< ApiVariationGl4 >::DataType::Flag_Shared ) );
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( detail::IsValid( hGl4Shader ) );

    TShader< Target >::DataType& obj = pOutGfxShader->ToData();
    obj.state = TShader< Target >::DataType::State_Initialized;
    obj.flags.SetBit( TShader< Target >::DataType::Flag_Shared, true );
    obj.flags.SetBit( TShader< Target >::DataType::Flag_SeparationEnable, false );
    obj.hCombinedProgram = hGl4Shader;
    obj.pGfxDevice = static_cast< detail::DeviceImpl< Target >* >( static_cast< void* >( pDevice ) );
}

void TInteroperation< Target >::ConvertToGfxTexture( TTexture< Target >* pOutGfxTexture,
    GlHandleType hGl4Texture, GlEnumType gl4BindTarget, ImageFormat imageFormat ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pOutGfxTexture );
    NN_SDK_REQUIRES( !IsInitialized( *pOutGfxTexture ) ||
        pOutGfxTexture->ToData()->flags.GetBit( TTexture< Target >::DataType::Flag_Shared ) );
    NN_SDK_REQUIRES( detail::IsValid( hGl4Texture ) );

    TTexture< Target >::DataType& obj = pOutGfxTexture->ToData();
    obj.state = TTexture< Target >::DataType::State_Initialized;
    obj.flags.SetBit( TTexture< Target >::DataType::Flag_Shared, true );
    obj.hTexture = hGl4Texture;
    obj.target = gl4BindTarget;
    obj.imageFormat = static_cast< Bit32 >( imageFormat );
    // GL 4.5
    /*if( ::glGetTextureParameteriv )
    {
        GlDeviceActivator activator( pDevice );

        GLint target;
        NN_GFX_CALL_GL_FUNCTION( ::glGetTextureParameteriv(
            *pNativeObject, GL_TEXTURE_TARGET, &target ) );
        obj.target = static_cast< uint32_t >( target );

        NN_GFX_GL_ASSERT();
    }*/
}

void TInteroperation< Target >::ConvertToGfxTextureView( TTextureView< Target >* pOutGfxTextureView,
    GlHandleType hGl4Texture, GlEnumType gl4BindTarget ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pOutGfxTextureView );
    NN_SDK_REQUIRES( !IsInitialized( *pOutGfxTextureView ) ||
        pOutGfxTextureView->ToData()->flags.GetBit( TTextureView< Target >::DataType::Flag_Shared ) );
    NN_SDK_REQUIRES( detail::IsValid( hGl4Texture ) );

    TTextureView< Target >::DataType& obj = pOutGfxTextureView->ToData();
    obj.state = TTextureView< Target >::DataType::State_Initialized;
    obj.flags.SetBit( TTextureView< Target >::DataType::Flag_Shared, true );
    obj.hTexture = hGl4Texture;
    obj.target = gl4BindTarget;
}

void TInteroperation< Target >::ConvertToGfxColorTargetView( TColorTargetView<
    Target >* pOutGfxColorTargetView, GlHandleType hGl4Texture, GlEnumType gl4BindTarget ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pOutGfxColorTargetView );
    NN_SDK_REQUIRES( !IsInitialized( *pOutGfxColorTargetView ) ||
        pOutGfxColorTargetView->ToData()->flags.GetBit(
        TColorTargetView< Target >::DataType::Flag_Shared ) );

    TColorTargetView< Target >::DataType& obj = pOutGfxColorTargetView->ToData();
    obj.state = TColorTargetView< Target >::DataType::State_Initialized;
    obj.flags.SetBit( TColorTargetView< Target >::DataType::Flag_Shared, true );
    obj.hTexture = hGl4Texture;
    obj.target = gl4BindTarget;
}

void TInteroperation< Target >::ConvertToGfxDepthStencilView(
    TDepthStencilView< Target >* pOutGfxDepthStencilView, GlHandleType hGl4Texture,
    GlEnumType gl4BindTarget, GlEnumType internalFormat ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pOutGfxDepthStencilView );
    NN_SDK_REQUIRES( !IsInitialized( *pOutGfxDepthStencilView ) ||
        pOutGfxDepthStencilView->ToData()->flags.GetBit(
            TDepthStencilView< Target >::DataType::Flag_Shared ) );

    TDepthStencilView< Target >::DataType& obj = pOutGfxDepthStencilView->ToData();
    obj.state = TDepthStencilView< Target >::DataType::State_Initialized;
    obj.flags.SetBit( TDepthStencilView< Target >::DataType::Flag_Shared, true );
    obj.hTexture = hGl4Texture;
    obj.target = gl4BindTarget;
    obj.internalFormat = internalFormat;
}

}
}
