﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <nn/os.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>

#include <nn/util/util_BitPack.h>

#include <nn/gfx/gfx_Enum.h>
#include <nn/gfx/gfx_SwapChainInfo.h>
#include <nn/gfx/gfx_TextureInfo.h>
#include <nn/gfx/gfx_GpuAddress.h>
#include <nn/gfx/gfx_CommandBufferInfo.h>
#include <nn/gfx/gfx_StateData-api.vk.1.h>

#include <nn/gfx/detail/gfx_Common-api.vk.h>
#include <nn/gfx/detail/gfx_Device-api.vk.1.h>
#include <nn/gfx/detail/gfx_Declare.h>
#include <nn/gfx/detail/gfx_Misc.h>
#if defined(NN_BUILD_CONFIG_OS_WIN32)
#define VK_USE_PLATFORM_WIN32_KHR
#endif
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
#define VK_USE_PLATFORM_VI_NN
#endif
#include <vulkan/vulkan.h>

#define NN_GFX_CALL_VK_FUNCTION(func) (func)

namespace nn {
namespace gfx {
namespace detail {

class GfxVkImageLayoutState;
static const uint32_t ImageLayoutStateHashSize = 256;
typedef HashList< GfxVkImageLayoutState, VkImage, ImageLayoutStateHashSize > ImageLayoutStateCacheList;

class GfxVkImageLayoutState
{
public:
    NN_IMPLICIT GfxVkImageLayoutState() :
        image( VK_NULL_HANDLE ),
        pInitialLayoutInfo( NULL ),
        pFinalLayoutInfo( NULL ),
        layerCount( 0 ),
        mipCount( 0 ),
        lastLayout( VK_IMAGE_LAYOUT_UNDEFINED ),
        lastBaseLayer( 0 ),
        lastLayerCount( 0 ),
        lastBaseMip( 0 ),
        lastMipCount( 0 ),
        pManager( NULL ) {}

    void SetImageHandle( VkImage handle )
    {
        image = handle;
    }

    size_t GetLayoutInfoSize( uint32_t layers, uint32_t mips )
    {
        return 2 * layers * mips * sizeof( VkImageLayout );
    }

    void SetLayoutInfo( uint32_t layers, uint32_t mips, TextureManageData* mgr, void* pData )
    {
        pInitialLayoutInfo = reinterpret_cast< VkImageLayout* >( pData );
        pFinalLayoutInfo = reinterpret_cast< VkImageLayout* >( pData ) + layers * mips;
        layerCount = layers;
        mipCount = mips;
        for ( uint32_t index = 0; index < layers * mips; index++ )
        {
            pInitialLayoutInfo[ index ] = VK_IMAGE_LAYOUT_UNDEFINED;
            pFinalLayoutInfo[ index ] = VK_IMAGE_LAYOUT_UNDEFINED;
        }
        pManager = mgr;
    }

    void ApplyLayoutTransition( VkCommandBuffer commandBuffer ) NN_NOEXCEPT
    {
        const uint32_t MaxBarriers = 32;
        VkImageMemoryBarrier barriers[ MaxBarriers ];
        uint32_t barrierCount = 0;

        for ( uint32_t layer = 0; layer < layerCount; layer++ )
        {
            VkPipelineStageFlags srcStageFlags = 0;
            VkPipelineStageFlags dstStageFlags = 0;
            for ( uint32_t mip = 0; mip < mipCount; mip++ )
            {
                // Make sure the image is transitioned to the first layout that is used
                VkImageLayout initialLayout = GetInitialImageLayout( layer, mip );
                if ( initialLayout != VK_IMAGE_LAYOUT_UNDEFINED )
                {
                    pManager->PrepareLayoutTransition( initialLayout, &barrierCount, &barriers[ 0 ], MaxBarriers,
                        layer, 1, mip, 1, &srcStageFlags, &dstStageFlags, 0 );
                }

                // Track the last layout that the image was in after this command buffer completes
                VkImageLayout finalLayout = GetFinalImageLayout( layer, mip );
                if ( finalLayout != VK_IMAGE_LAYOUT_UNDEFINED )
                {
                    pManager->SetImageLayout( finalLayout, layer, mip );
                }
            }

            if ( barrierCount > 0 )
            {
                NN_GFX_CALL_VK_FUNCTION( vkCmdPipelineBarrier( commandBuffer, srcStageFlags, dstStageFlags, 0, 0, nullptr, 0, nullptr,
                    barrierCount, &barriers[ 0 ] ) );
                barrierCount = 0;
            }
        }
    }

    void PrepareLayoutTransition( VkImageLayout requiredLayout, uint32_t* pLayoutCount, VkImageMemoryBarrier* pImageBarriers,
        uint32_t maxLayout, uint32_t baseLayer, uint32_t layers, uint32_t baseMip, uint32_t mips,
        VkPipelineStageFlags* pSrcStages, VkPipelineStageFlags* pDstStages, VkImageAspectFlags imageAspectFlags ) NN_NOEXCEPT
    {
        NN_UNUSED( maxLayout );
        uint32_t barrierCount = *pLayoutCount;

        // Make sure parameters are in-bounds
        layers = ( baseLayer + layers ) <= layerCount ? layers : ( layerCount - baseLayer + 1 );
        mips = ( baseMip + mips ) <= mipCount ? mips : ( mipCount - baseMip + 1 );

        if ( lastLayout != requiredLayout ||
            lastBaseLayer != baseLayer ||
            lastLayerCount != layerCount ||
            lastBaseMip != baseMip ||
            lastMipCount != mipCount )
        {
            VkImageMemoryBarrier imageMemoryBarrier;
            imageMemoryBarrier.sType = VK_STRUCTURE_TYPE_IMAGE_MEMORY_BARRIER;
            imageMemoryBarrier.pNext = NULL;
            imageMemoryBarrier.oldLayout = VK_IMAGE_LAYOUT_UNDEFINED;
            imageMemoryBarrier.newLayout = requiredLayout;
            imageMemoryBarrier.srcAccessMask = 0;
            imageMemoryBarrier.dstAccessMask = Vk::GetLayoutAccessFlags( requiredLayout );
            imageMemoryBarrier.srcQueueFamilyIndex = VK_QUEUE_FAMILY_IGNORED;
            imageMemoryBarrier.dstQueueFamilyIndex = VK_QUEUE_FAMILY_IGNORED;
            imageMemoryBarrier.image = image;
            imageMemoryBarrier.subresourceRange.aspectMask = imageAspectFlags;
            imageMemoryBarrier.subresourceRange.baseArrayLayer = 0;
            imageMemoryBarrier.subresourceRange.layerCount = 0;
            imageMemoryBarrier.subresourceRange.baseMipLevel = 0;
            imageMemoryBarrier.subresourceRange.levelCount = 0;

            bool isDirty = false;
            for ( uint32_t layer = baseLayer; layer < baseLayer + layers; layer++ )
            {
                for ( uint32_t mip = baseMip; mip < baseMip + mips; mip++ )
                {
                    // Initially the state is unknown so just accept the application's first use.
                    // Later in ApplyLayoutTransition this will be compared against the TextureManageData.
                    if ( GetFinalImageLayout( layer, mip ) == VK_IMAGE_LAYOUT_UNDEFINED )
                    {
                        pInitialLayoutInfo[ layer * mipCount + mip ] = requiredLayout;
                        pFinalLayoutInfo[ layer * mipCount + mip ] = requiredLayout;
                    }
                    else
                    {
                        VkImageLayout oldLayout = GetFinalImageLayout( layer, mip );
                        if ( isDirty )
                        {
                            if ( imageMemoryBarrier.oldLayout != oldLayout )
                            {
                                // The old layout changed so the barrier must be flush.
                                NN_SDK_ASSERT_LESS( barrierCount, maxLayout );
                                imageMemoryBarrier.subresourceRange.layerCount++;
                                pImageBarriers[ barrierCount ] = imageMemoryBarrier;
                                barrierCount++;
                                isDirty = false;
                            }
                            else
                            {
                                // Coalesce transitions in the same level
                                imageMemoryBarrier.subresourceRange.levelCount++;

                                // Update tracking information
                                pFinalLayoutInfo[ layer * mipCount + mip ] = requiredLayout;
                            }
                        }

                        if ( !isDirty && oldLayout != requiredLayout )
                        {
                            imageMemoryBarrier.subresourceRange.baseMipLevel = mip;
                            imageMemoryBarrier.subresourceRange.levelCount = 1;
                            imageMemoryBarrier.subresourceRange.baseArrayLayer = layer;
                            imageMemoryBarrier.subresourceRange.layerCount = 0;
                            imageMemoryBarrier.oldLayout = oldLayout;
                            imageMemoryBarrier.srcAccessMask = Vk::GetLayoutAccessFlags( oldLayout );

                            // Update tracking information
                            pFinalLayoutInfo[ layer * mipCount + mip ] = requiredLayout;

                            isDirty = true;
                        }
                    }
                }

                // Perform each layer separately.
                // TODO: This could be optimized further to do multiple levels.
                if ( isDirty )
                {
                    // Flush
                    NN_SDK_ASSERT_LESS( barrierCount, maxLayout );
                    imageMemoryBarrier.subresourceRange.layerCount++;
                    pImageBarriers[ barrierCount ] = imageMemoryBarrier;
                    barrierCount++;
                    isDirty = false;
                }
            }

            // Optimization
            lastLayout = requiredLayout;
            lastBaseLayer = baseLayer;
            lastLayerCount = layerCount;
            lastBaseMip = baseMip;
            lastMipCount = mipCount;
        }

        // Update used slots
        *pLayoutCount = barrierCount;

        // FIXME: This should be made to be more accurate based on actual usage.
        *pSrcStages |= VK_PIPELINE_STAGE_ALL_COMMANDS_BIT;
        *pDstStages |= VK_PIPELINE_STAGE_ALL_COMMANDS_BIT;
    }

    void SetImageLayout( VkImageLayout layout, const VkImageSubresourceRange& range ) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_LESS_EQUAL( range.baseArrayLayer + range.layerCount, layerCount );
        NN_SDK_ASSERT_LESS_EQUAL( range.baseMipLevel + range.levelCount, mipCount );
        for ( uint32_t layer = range.baseArrayLayer; layer < range.baseArrayLayer + range.layerCount; layer++ )
        {
            for ( uint32_t mip = range.baseMipLevel; mip < range.baseMipLevel + range.levelCount; mip++ )
            {
                // NOTE: Leaving pInitialLayoutInfo as VK_IMAGE_LAYOUT_UNDEFINED
                // allows for implementing fast clears.
                pFinalLayoutInfo[ layer * mipCount + mip ] = layout;
            }
        }

        // Optimization
        lastLayout = layout;
        lastBaseLayer = range.baseArrayLayer;
        lastLayerCount = range.layerCount;
        lastBaseMip = range.baseMipLevel;
        lastMipCount = range.levelCount;
    }

    static ImageLayoutStateCacheList::HashValue CreateHashValue( const VkImage& imageHandle ) NN_NOEXCEPT;
    static int GetHashTableIndexFromHashValue( ImageLayoutStateCacheList::HashValue hashValue, int hashTableSize ) NN_NOEXCEPT;
    static bool Compare( const GfxVkImageLayoutState& cachedObject, const VkImage& imageHandle ) NN_NOEXCEPT;

private:
    VkImageLayout GetInitialImageLayout( uint32_t layer, uint32_t mip ) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_LESS_EQUAL( layer, layerCount );
        NN_SDK_ASSERT_LESS_EQUAL( mip, mipCount );
        return pInitialLayoutInfo[ layer * mipCount + mip ];
    }

    VkImageLayout GetFinalImageLayout( uint32_t layer, uint32_t mip ) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_LESS_EQUAL( layer, layerCount );
        NN_SDK_ASSERT_LESS_EQUAL( mip, mipCount );
        return pFinalLayoutInfo[ layer * mipCount + mip ];
    }

    VkImage image;
    VkImageLayout* pInitialLayoutInfo;
    VkImageLayout* pFinalLayoutInfo;
    uint32_t layerCount;
    uint32_t mipCount;
    VkImageLayout lastLayout;
    uint32_t lastBaseLayer;
    uint32_t lastLayerCount;
    uint32_t lastBaseMip;
    uint32_t lastMipCount;

    TextureManageData* pManager;
};

struct GfxVkGpuState
{
    static const uint32_t maxColorAttachments = 8; //!< 最大カラーアタッチメント数です
    static const uint32_t maxDescriptorSlot = 64; //!< 最大DescriptorSlot数です
    static const uint32_t maxDynamicDescriptorSlot = 1;
    static const uint32_t maxViewports = 16; //!< 最大ビューポート数です
    static const uint32_t dynamicStateItemCount = 9;

    struct ConstSizedPipelineState
    {
        struct ShaderState
        {
            detail::VkHandle shaderModule[ ShaderStage_End ]; //!< シェーダモジュールです
            uint64_t shaderBindingMask[ ShaderStage_End ];
        } shaderState;

        struct VertexInputState
        {
            uint32_t bufferStateCount; //!< bufferStateの有効数です
            uint32_t attributeStateCount; //!< attributeStateの有効数です
        } vertexInputState;

        struct InputAssemblyState
        {
            detail::VkEnum primitiveTopology; //!< プリミティブトポロジーです
        } inputAssemblyState;

        struct TessellationState
        {
            uint32_t patchControlPoints; //!< パッチの制御点の数です
        } tessellationState;

        struct ViewportState
        {
            uint32_t viewportCount; //!< viewportの有効数です
        } viewportState;

        struct RasterizationState
        {
            enum Flag
            {
                Flag_RasterDisable = RasterizerStateImplData< ApiVariationVk1 >::Flag_RasterDisable,
                Flag_MultisampleEnable = RasterizerStateImplData< ApiVariationVk1 >::Flag_MultisampleEnable,
                Flag_AlphaToCoverageEnable = RasterizerStateImplData< ApiVariationVk1 >::Flag_AlphaToCoverageEnable,
                Flag_DepthClampEnable = RasterizerStateImplData< ApiVariationVk1 >::Flag_DepthClampEnable,
                Flag_PolygonOffsetEnable = RasterizerStateImplData< ApiVariationVk1 >::Flag_PolygonOffsetEnable,
                Flag_ConservativeRasterEnable = RasterizerStateImplData< ApiVariationVk1 >::Flag_ConservativeRasterEnable,
                Flag_AlphaToOneEnable //!< アルファを1で置き換えます
            };
            nn::util::BitPack8 flag; //!< フラグです

            detail::VkEnum polygonMode; //!< ポリゴンモードです
            detail::VkEnum frontFace; //!< 前面です
            detail::VkEnum cullFace; //!< カリングモードです
            uint32_t sampleMask; //!< サンプルマスクです
            int32_t sampleCount; //!< サンプル数です
        } rasterizationState;

        struct DepthStencilState
        {
            enum Flag
            {
                Flag_DepthTestEnable = DepthStencilStateImplData< ApiVariationVk1 >::Flag_DepthTestEnable,
                Flag_DepthWriteEnable = DepthStencilStateImplData< ApiVariationVk1 >::Flag_DepthWriteEnable,
                Flag_StencilTestEnable = DepthStencilStateImplData< ApiVariationVk1 >::Flag_StencilTestEnable,
                Flag_DepthBoundsTestEnable = DepthStencilStateImplData< ApiVariationVk1 >::Flag_DepthBoundsTestEnable
            };
            nn::util::BitPack16 flag; //!< フラグです

            struct StencilState
            {
                detail::VkEnum sfail; //!< ステンシルテスト失敗時のオペレーションです
                detail::VkEnum dpfail; //!< 深度テスト失敗時のオペレーションです
                detail::VkEnum dppass; //!< 深度テスト成功時のオペレーションです
                detail::VkEnum func; //!< ステンシルテストの比較関数です
            };
            detail::VkEnum depthFunc; //!< 深度テストの比較関数です
            StencilState frontStencil; //!< 前面のステンシルステートです
            StencilState backStencil; //!< 背面のステンシルステートです
        } depthStencilState;

        struct BlendState
        {
            enum Flag
            {
                Flag_AlphaToCoverageEnable = BlendStateImplData< ApiVariationVk1 >::Flag_AlphaToCoverageEnable,
                Flag_IndependentBlendEnable = BlendStateImplData< ApiVariationVk1 >::Flag_IndependentBlendEnable,
                Flag_LogicOpEnable = BlendStateImplData< ApiVariationVk1 >::Flag_LogicOpEnable
            };

            nn::util::BitPack16 flag; //!< フラグです
            detail::VkEnum logicOp; //!< 論理オペレーションです
            uint32_t blendTargetCount; //!< ブレンドターゲットの数です
        } blendState;

    } constSizedPipelineState;

    struct VariableSizedPipelineState
    {
        static const uint32_t maxVertexInputBindings = 32; //!< 最大頂点バッファバインド数です
        static const uint32_t maxVertexInputAttributes = 32; //!< 最大頂点属性数です

        struct VertexBufferState
        {
            uint32_t divisor; //!< インスタンスの除数です
            uint32_t stride; //!< ストライドです
        } vertexBufferState[ maxVertexInputBindings ];

        struct VertexAttributeState
        {
            uint32_t location; //!< シェーダーにおけるスロット番号です
            uint32_t bindingIndex; //!< バッファーインデックスです
            detail::VkEnum format; //!< 頂点属性フォーマットです
            uint32_t offset; //!< バイトでのバッファーにおけるオフセットです
        } vertexAttributeState[ maxVertexInputAttributes ];

        struct BlendTargetState
        {
            enum Flag
            {
                Flag_BlendEnable = BlendStateImplData< ApiVariationVk1 >::BlendTargetState::Flag_BlendEnable
            };

            nn::util::BitPack8 flag; //!< フラグです
            Bit8 colorMask; //!< カラーマスクです
            detail::VkEnum srcRGB; //!< ソースのカラー要素のブレンドファクターです
            detail::VkEnum dstRGB; //!< デスティネーションのカラー要素のブレンドファクターです
            detail::VkEnum srcAlpha; //!< ソースのアルファ要素のブレンドファクターです
            detail::VkEnum dstAlpha; //!< デスティネーションのアルファ要素のブレンドファクターです
            detail::VkEnum modeRGB; //!< カラー要素のブレンド関数です
            detail::VkEnum modeAlpha; //!< アルファ要素のブレンド関数です
        } blendTargetState[ maxColorAttachments ];

    } variableSizedPipelineState;

    struct RenderTargetState
    {
        // RenderPassとFramebufferに必要なステート
        enum Flag
        {
            Flag_DepthAttached //!< DepthStencilバッファがアタッチされていることを示します
        };
        nn::util::BitPack8 flag; //!< フラグです
        int32_t colorAttachmentCount; //!< カラーバッファの面数です

        // RenderPassに必要なステート
        struct AttachmentState
        {
            detail::VkEnum format; //!< VkImageViewのフォーマットです
            int32_t samples; //!< VkImageViewのサンプル数です
            uint32_t layer;
            uint32_t mip;
            uint32_t layerCount; //!< The finalLayout transitions the entire image subresource view. mipCount is always 1.
            detail::VkEnum loadOp;
        };

        AttachmentState colorAttachment[ maxColorAttachments ]; //!< カラーアタッチメントの情報です
        AttachmentState depthStencilAttachment; //!< DepthStencilアタッチメントの情報です

        // Framebufferに必要なステート
        uint32_t width; //!< 幅ピクセル数です
        uint32_t height; //!< 高さピクセル数です
        uint32_t layers; //!< レイヤー数です
        detail::VkHandle colorTargetView[ maxColorAttachments ]; //!< ColorTarget用のVkImageViewオブジェクトです
        detail::VkHandle depthStencilTargetView; //!< DepthTarget用のVkImageViewオブジェクト
        detail::TextureManageData* pColorTextureManagerData[ maxColorAttachments ]; //! TextureManageData for current attachment.
        detail::TextureManageData* pDepthStencilTextureManagerData; //! TextureManageData for current attachment.
    } renderTargetState;

    struct RenderPassState
    {
        float clearColors[ maxColorAttachments ][ 4 ];
        float clearDepth;
        uint32_t clearStencil;
    } renderPassState;

    struct DescriptorSetLayoutState
    {
        // DescriptorSetで共通で必要なステート
        uint32_t availableDescriptorCount; //!< 有効Descriptorの設定数です
        uint32_t maxDescriptorIndex; //!< 現状設定されているDescriptor Slotの最大インデックスです

        // DescriptorSetLayoutに必要なステート
        struct SlotState
        {
            enum Flag
            {
                Flag_IsSet
            };
            nn::util::BitPack8 flag; //!< フラグです
            detail::VkEnum type; //!< Descriptorのタイプです
            detail::VkEnum shaderStage; //!< 設定対象のシェーダです
        } slotState[ maxDescriptorSlot ]; //!< DescriptorSlotの情報です

        uint64_t validSlotMask; //!< Tracks valid descriptor slots for the given shader.
    } descriptorSetLayoutState;

    struct DescriptorResourceState
    {
        // DescriptorSetのUpdateに必要な情報
        union ResourceState
        {
            VkDescriptorBufferInfo vkBufferInfo; //!< Bufferの情報です
            VkDescriptorImageInfo vkImageInfo; //!< Imageの情報です
            VkBufferView vkBufferViewInfo; //!< VkBufferViewの情報です
        } resourceState[ maxDescriptorSlot ]; //!< DescriptorSlotの情報です
    } descriptorResourceState;

    #if defined( NN_SDK_BUILD_DEBUG )
    struct DynamicState
    {
        struct ViewportScissorState
        {
            float viewport[ maxViewports ][ 6 ]; //!< ビューポート設定です
            int32_t scissor[ maxViewports ][ 4 ]; //!< シザ設定です
        } viewportScissorState;

        struct RasterizationState
        {
            float slopeScaledDepthBias; //!< 最大深度でスケールされる、深度値に加算される値です
            float depthBias; //!< 深度値に加算される値です
            float depthBiasClamp; //!< 深度値に加算される値の最大値です
            float lineWidth; //!< ライン幅です
        } rasterizationState;

        struct DepthStencilState
        {
            struct StencilState
            {
                uint32_t ref; //!< ステンシル参照値です
            };
            StencilState frontStencil; //!< 前面のステンシルステートです
            StencilState backStencil; //!< 背面のステンシルステートです
            uint32_t stencilWriteMask; //!< ステンシル値の書き込みマスクです
            uint32_t stencilReadMask; //!< ステンシル値の読み込みマスクです
            float minDepthBounds; //!< 最小深度境界です
            float maxDepthBounds; //!< 最大深度境界です
        } depthStencilState;

        struct BlendState
        {
            float blendColor[ 4 ]; //!< ブレンド定数です
        } blendState;

    } dynamicState;
    #endif

    struct QueryState
    {
        enum QueryTargetFlag
        {
            QueryTargetFlag_GraphicsPipelineStatistics = QueryTarget_End
        };
        nn::util::BitPack16 queryStartedFlag;
    } queryState;
};

class DescriptorSetCache;

class PipelineCache
{
    NN_DISALLOW_COPY( PipelineCache );

public:
    PipelineCache() NN_NOEXCEPT;
    virtual ~PipelineCache() NN_NOEXCEPT;

    virtual void Initialize( const GfxVkGpuState& gpuState, VkDescriptorSetLayout descriptorSetLayout ) NN_NOEXCEPT = 0;
    virtual void Finalize() NN_NOEXCEPT = 0;
    virtual VkPipeline GetPipeline() const NN_NOEXCEPT = 0;
    virtual VkPipelineLayout GetPipelineLayout() const NN_NOEXCEPT = 0;
    virtual VkRenderPass GetRenderPass() const NN_NOEXCEPT = 0;
    virtual DescriptorSetCache* GetDescriptorSetCache() const NN_NOEXCEPT = 0;
    virtual void SetDescriptorSetCache( DescriptorSetCache* pInDescriptorSetCache ) NN_NOEXCEPT = 0;
    virtual uint64_t GetValidSlotMask() const NN_NOEXCEPT = 0;

    virtual GfxVkGpuState::ConstSizedPipelineState* GetConstSizedPipelineState() NN_NOEXCEPT = 0;
    virtual GfxVkGpuState::VariableSizedPipelineState* GetVariableSizedPipelineState() NN_NOEXCEPT = 0;
    virtual GfxVkGpuState::DescriptorSetLayoutState* GetDescriptorSetLayoutState() NN_NOEXCEPT = 0;
    virtual GfxVkGpuState::RenderTargetState* GetRenderTargetState() NN_NOEXCEPT = 0;
};

class GraphicsPipelineCache;
const int GraphicsPipelineCacheHashSize = 256;
typedef HashList< GraphicsPipelineCache, GfxVkGpuState, GraphicsPipelineCacheHashSize > GraphicsPipelineCacheList;

class GraphicsPipelineCache : public PipelineCache
{
    NN_DISALLOW_COPY( GraphicsPipelineCache );

public:
    GraphicsPipelineCache() NN_NOEXCEPT;
    virtual ~GraphicsPipelineCache() NN_NOEXCEPT;

    virtual void Initialize( const GfxVkGpuState& gpuState, VkDescriptorSetLayout descriptorSetLayout ) NN_NOEXCEPT;
    virtual void Finalize() NN_NOEXCEPT;
    virtual VkPipeline GetPipeline() const NN_NOEXCEPT { return m_hPipeline; }
    virtual VkPipelineLayout GetPipelineLayout() const NN_NOEXCEPT { return m_hPipelineLayout; }
    virtual VkRenderPass GetRenderPass() const NN_NOEXCEPT { return m_hRenderPass; }
    virtual DescriptorSetCache* GetDescriptorSetCache() const NN_NOEXCEPT { return m_pDescriptorSetCache; }
    virtual void SetDescriptorSetCache( DescriptorSetCache* pInDescriptorSetCache ) NN_NOEXCEPT { this->m_pDescriptorSetCache = pInDescriptorSetCache; }
    virtual uint64_t GetValidSlotMask() const NN_NOEXCEPT { return this->m_DescriptorSetLayoutState.validSlotMask; }

    virtual GfxVkGpuState::ConstSizedPipelineState* GetConstSizedPipelineState() NN_NOEXCEPT { return &m_ConstSizedPipelineState; }
    virtual GfxVkGpuState::VariableSizedPipelineState* GetVariableSizedPipelineState() NN_NOEXCEPT { return &m_VariableSizedPipelineState; }
    virtual GfxVkGpuState::DescriptorSetLayoutState* GetDescriptorSetLayoutState() NN_NOEXCEPT { return &m_DescriptorSetLayoutState; }
    virtual GfxVkGpuState::RenderTargetState* GetRenderTargetState() NN_NOEXCEPT { return &m_RenderTargetState; }

    static GraphicsPipelineCacheList::HashValue CreateHashValue( const GfxVkGpuState& gpuState ) NN_NOEXCEPT;
    static int GetHashTableIndexFromHashValue( GraphicsPipelineCacheList::HashValue hashValue, int hashTableSize ) NN_NOEXCEPT;
    static bool Compare( const GraphicsPipelineCache& cachedObject, const GfxVkGpuState& gpuState ) NN_NOEXCEPT;

private:

    VkPipeline                                  m_hPipeline;                //!< VkPipelineオブジェクトです
    VkPipelineLayout                            m_hPipelineLayout;          //!< VkPipelineLayoutオブジェクトです
    VkRenderPass                                m_hRenderPass;              //!< VkRenderPassオブジェクトです

    GfxVkGpuState::ConstSizedPipelineState      m_ConstSizedPipelineState;      //!< Vulkanオブジェクトを生成したときのGPUステートスナップショット
    GfxVkGpuState::VariableSizedPipelineState   m_VariableSizedPipelineState;   //!< Vulkanオブジェクトを生成したときのGPUステートスナップショット
    GfxVkGpuState::DescriptorSetLayoutState     m_DescriptorSetLayoutState;     //!< Vulkanオブジェクトを生成したときのGPUステートスナップショット
    GfxVkGpuState::RenderTargetState            m_RenderTargetState;            //!< Vulkanオブジェクトを生成したときのGPUステートスナップショット。

    DescriptorSetCache*                         m_pDescriptorSetCache;      //!< DescriptorSetCacheのポインタを保持します
};

class ComputePipelineCache;
const int ComputePipelineCacheHashSize = 256;
typedef HashList< ComputePipelineCache, GfxVkGpuState, ComputePipelineCacheHashSize > ComputePipelineCacheList;

class ComputePipelineCache : public PipelineCache
{
    NN_DISALLOW_COPY( ComputePipelineCache );

public:
    ComputePipelineCache() NN_NOEXCEPT;
    virtual ~ComputePipelineCache() NN_NOEXCEPT;

    virtual void Initialize( const GfxVkGpuState& gpuState, VkDescriptorSetLayout descriptorSetLayout ) NN_NOEXCEPT;
    virtual void Finalize() NN_NOEXCEPT;
    virtual VkPipeline GetPipeline() const NN_NOEXCEPT { return m_hPipeline; }
    virtual VkPipelineLayout GetPipelineLayout() const NN_NOEXCEPT { return m_hPipelineLayout; }
    virtual VkRenderPass GetRenderPass() const NN_NOEXCEPT { return VK_NULL_HANDLE; }
    virtual DescriptorSetCache* GetDescriptorSetCache() const NN_NOEXCEPT { return m_pDescriptorSetCache; }
    virtual void SetDescriptorSetCache( DescriptorSetCache* pInDescriptorSetCache ) NN_NOEXCEPT { this->m_pDescriptorSetCache = pInDescriptorSetCache; }
    virtual uint64_t GetValidSlotMask() const NN_NOEXCEPT { return this->m_DescriptorSetLayoutState.validSlotMask; }

    virtual GfxVkGpuState::ConstSizedPipelineState* GetConstSizedPipelineState() NN_NOEXCEPT { return &m_ConstSizedPipelineState; }
    virtual GfxVkGpuState::VariableSizedPipelineState* GetVariableSizedPipelineState() NN_NOEXCEPT { return &m_VariableSizedPipelineState; }
    virtual GfxVkGpuState::DescriptorSetLayoutState* GetDescriptorSetLayoutState() NN_NOEXCEPT { return &m_DescriptorSetLayoutState; }
    virtual GfxVkGpuState::RenderTargetState* GetRenderTargetState() NN_NOEXCEPT { return NULL; }

    static ComputePipelineCacheList::HashValue CreateHashValue( const GfxVkGpuState& gpuState ) NN_NOEXCEPT;
    static int GetHashTableIndexFromHashValue( ComputePipelineCacheList::HashValue hashValue, int hashTableSize ) NN_NOEXCEPT;
    static bool Compare( const ComputePipelineCache& cachedObject, const GfxVkGpuState& gpuState ) NN_NOEXCEPT;

private:
    VkPipeline                                  m_hPipeline;            //!< VkPipelineオブジェクトです
    VkPipelineLayout                            m_hPipelineLayout;      //!< VkPipelineLayoutオブジェクトです

    GfxVkGpuState::ConstSizedPipelineState      m_ConstSizedPipelineState;      //!< Vulkanオブジェクトを生成したときのGPUステートスナップショット
    GfxVkGpuState::VariableSizedPipelineState   m_VariableSizedPipelineState;   //!< Vulkanオブジェクトを生成したときのGPUステートスナップショット
    GfxVkGpuState::DescriptorSetLayoutState     m_DescriptorSetLayoutState;     //!< Vulkanオブジェクトを生成したときのGPUステートスナップショット

    DescriptorSetCache*                         m_pDescriptorSetCache;  //!< DescriptorSetCacheのポインタを保持します
};

class FramebufferCache;
const int FramebufferCacheHashSize = 256;
typedef HashList< FramebufferCache, GfxVkGpuState, FramebufferCacheHashSize > FramebufferCacheList;

class FramebufferCache
{
    NN_DISALLOW_COPY( FramebufferCache );

public:
    FramebufferCache() NN_NOEXCEPT;
    virtual ~FramebufferCache() NN_NOEXCEPT;

    void Initialize( const GfxVkGpuState& gpuState, VkRenderPass renderPass ) NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;
    VkFramebuffer GetFramebuffer() const NN_NOEXCEPT { return m_hFramebuffer; }
    GfxVkGpuState::RenderTargetState* GetRenderTargetState() NN_NOEXCEPT { return &m_RenderTargetState; }

    static FramebufferCacheList::HashValue CreateHashValue( const GfxVkGpuState& gpuState ) NN_NOEXCEPT;
    static int GetHashTableIndexFromHashValue( FramebufferCacheList::HashValue hashValue, int hashTableSize ) NN_NOEXCEPT;
    static bool Compare( const FramebufferCache& cachedObject, const GfxVkGpuState& gpuState ) NN_NOEXCEPT;

private:

    VkFramebuffer                       m_hFramebuffer;         //!< VkFramebufferオブジェクトです
    GfxVkGpuState::RenderTargetState    m_RenderTargetState;    //!< Vulkanオブジェクトを生成したときのGPUステートスナップショット。
};

class DescriptorSetCache;
const int DescriptorSetCacheHashSize = 256;
typedef HashList< DescriptorSetCache, GfxVkGpuState, DescriptorSetCacheHashSize > DescriptorSetCacheList;

class DescriptorSetCache
{
    NN_DISALLOW_COPY( DescriptorSetCache );

public:
    DescriptorSetCache() NN_NOEXCEPT;
    virtual ~DescriptorSetCache() NN_NOEXCEPT;

    class Node
    {
    public:
        Node() NN_NOEXCEPT;
        virtual ~Node() NN_NOEXCEPT;

        void Initialize( const GfxVkGpuState& gpuState, VkDescriptorSetLayout descriptorSetLayout ) NN_NOEXCEPT;
        void Finalize() NN_NOEXCEPT;
        VkDescriptorPool GetDescriptorPool() const NN_NOEXCEPT { return m_hDescriptorPool; }
        VkDescriptorSet GetDescriptorSet() const NN_NOEXCEPT { return m_hDescriptorSet; }
        VkWriteDescriptorSet* GetWriteDescriptorSet() const NN_NOEXCEPT { return m_pWriteDescriptorSet; }

    private:
        VkDescriptorPool        m_hDescriptorPool;      //!< VkDescriptorPoolオブジェクトです
        VkDescriptorSet         m_hDescriptorSet;       //!< VkDescriptorSetオブジェクトです
        VkWriteDescriptorSet*   m_pWriteDescriptorSet;  //!< DescriptorSetのUpdateに使用します。
    };

    void Initialize( const GfxVkGpuState& gpuState ) NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;
    List< Node >& GetNodeList() NN_NOEXCEPT { return m_NodeList; }
    List< Node >::Iterator& GetCurrentNodeIterator() NN_NOEXCEPT { return m_CurrentNodeIterator; }
    void SetCurrentNodeIteratorBegin() NN_NOEXCEPT { m_CurrentNodeIterator = m_NodeList.Begin(); }
    void SetCurrentNodeIteratorNext() NN_NOEXCEPT { ++m_CurrentNodeIterator; }
    void SetCurrentNodeIteratorBeforeEnd() NN_NOEXCEPT { m_CurrentNodeIterator = m_NodeList.BeforeEnd(); }
    bool IsCurrentNodeIteratorEnd() NN_NOEXCEPT { return m_NodeList.IsEnd( m_CurrentNodeIterator ); }
    VkDescriptorSetLayout GetDescriptorSetLayout() const NN_NOEXCEPT { return m_hDescriptorSetLayout; }
    GfxVkGpuState::DescriptorSetLayoutState* GetDescriptorSetLayoutState() NN_NOEXCEPT { return &m_DescriptorSetLayoutState; }

    static DescriptorSetCacheList::HashValue CreateHashValue( const GfxVkGpuState& gpuState ) NN_NOEXCEPT;
    static int GetHashTableIndexFromHashValue( DescriptorSetCacheList::HashValue hashValue, int hashTableSize ) NN_NOEXCEPT;
    static bool Compare( const DescriptorSetCache& cachedObject, const GfxVkGpuState& gpuState ) NN_NOEXCEPT;

private:
    List< Node >                            m_NodeList;                 //!< Nodeクラスのリストです
    List< Node >::Iterator                  m_CurrentNodeIterator;      //!< Nodeクラスリストの最後に使用したノードを指すIteratorです
    VkDescriptorSetLayout                   m_hDescriptorSetLayout;     //!< VkDescriptorSetLayoutオブジェクトです。
    GfxVkGpuState::DescriptorSetLayoutState m_DescriptorSetLayoutState; //!< Vulkanオブジェクトを生成したときのGPUステートスナップショット。
};

// GfxVkCommandInfoはDrawまたはDispatchコマンドごとに生成されます。
struct GfxVkCommandInfo
{
    enum UpdatedStateFlag
    {
        UpdatedStateFlag_SetRenderTarget,              //!< RenderTargetのStateが変更されたことを示すフラグです
        UpdatedStateFlag_SetDescriptorPool,            //!< DescriptorPoolのStateが変更されたことを示すフラグです
        UpdatedStateFlag_SetShader,                    //!< ShaderのStateが変更されたことを示すフラグです
        UpdatedStateFlag_SetRasterizerState,           //!< RasterizerのStateが変更されたことを示すフラグです
        UpdatedStateFlag_SetBlendState,                //!< BlendのStateが変更されたことを示すフラグです
        UpdatedStateFlag_SetDepthStencilState,         //!< DepthStencilのStateが変更されたことを示すフラグです
        UpdatedStateFlag_SetVertexState,               //!< VertexのStateが変更されたことを示すフラグです
        UpdatedStateFlag_SetTessellationState,         //!< TessellationのStateが変更されたことを示すフラグです
        UpdatedStateFlag_SetInputAssemblyState,        //!< InputAssemblyのStateが変更されたことを示すフラグです
        UpdatedStateFlag_SetUniform,                   //!< Set when the dynamic uniform buffer has been updated
        UpdatedStateFlag_OutsideRenderPass,            //!< Set when renderpass ends
        UpdatedStateFlag_StateEnd
    };

    enum CommandInfoFlag
    {
        CommandInfoFlag_IsDrawCommand,                 //!< trueの場合ドローコマンドです。falseの場合Dispatchコマンドです
        CommandInfoFlag_IsStateUpdated,                //!< 直前のドローからPipelineオブジェクトの切り替えが必要です
        CommandInfoFlag_ForceNewRenderPass
    };

    nn::util::BitPack16         updatedState;               //!< 直前のDraw or Dispatchコマンドから更新されたステートを示します
    nn::util::BitPack8          info;                       //!< コマンドの情報です

    PipelineCache*              pPipelineCache;             //!< Pipelineオブジェクトと関連オブジェクトのキャッシュです
    FramebufferCache*           pFramebufferCache;          //!< Framebufferオブジェクトのキャッシュです
    DescriptorSetCache*         pDescriptorSetCache;        //!< DescriptorSetオブジェクトと関連オブジェクトのキャッシュです
    DescriptorSetCache::Node*   pDescriptorSetCacheNode;    //!< DescriptorSetオブジェクトの実体です

    nn::os::LightEventType      lightEvent;                 //!< VulkanCallerContextがStateUpdaterContextを待つための軽量イベントです

    uint32_t                    dynamicOffsets[ GfxVkGpuState::maxDynamicDescriptorSlot ]; //!< Offsets for VK_DESCRIPTOR_TYPE_UNIFORM_BUFFER_DYNAMIC and VK_DESCRIPTOR_TYPE_STORAGE_BUFFER_DYNAMIC
};

class VkManager
{
    NN_DISALLOW_COPY( VkManager );

public:
    explicit VkManager( CommandBufferImpl< ApiVariationVk1 >* pCommandBuffer ) NN_NOEXCEPT;
    virtual ~VkManager() NN_NOEXCEPT;

    void Reset() NN_NOEXCEPT;
    void Begin( int infoCount ) NN_NOEXCEPT;
    void End() NN_NOEXCEPT;
    GfxVkCommandInfo* GetCommandInfo( int infoIndex ) NN_NOEXCEPT;
    GfxVkCommandInfo* GetUpdatedCommandInfo( int infoIndex ) NN_NOEXCEPT;
    template< bool isDraw > GfxVkCommandInfo* UpdateCommandInfo( const GfxVkGpuState& gpuState, int infoIndex, bool force ) NN_NOEXCEPT;
    bool ValidateUpdatedState( const GfxVkGpuState& gpuState, int infoIndex ) NN_NOEXCEPT;
    void SetUniformBlockSize( uint32_t size ) NN_NOEXCEPT;
    void SetUniformBufferBinding( int slot ) NN_NOEXCEPT;
    int32_t GetUniformBufferBinding() NN_NOEXCEPT;
    uint32_t GetUniformBufferOffset() NN_NOEXCEPT;
    void* GetUniformBufferPtr( uint32_t offset ) NN_NOEXCEPT;
    void AdvanceUniformBufferOffset() NN_NOEXCEPT;
    bool IsVulkanObjectCached() NN_NOEXCEPT { return m_StateFlag.GetBit( StateFlag_Cached ); }
    const CommandBufferImpl< ApiVariationVk1 >* GetCommandBuffer() NN_NOEXCEPT { return m_pGfxCommandBuffer; }
    void RegisterDestroyingShaderList( DestroyingShaderObjectInfo* pDestroyingShaderObjectInfo ) NN_NOEXCEPT;
    void RegisterDestroyingColorTargetViewList( DestroyingImageViewObjectInfo* pDestroyingImageViewObjectInfo ) NN_NOEXCEPT;
    void RegisterDestroyingDepthTargetViewList( DestroyingImageViewObjectInfo* pDestroyingImageViewObjectInfo ) NN_NOEXCEPT;

private:
    void ClearCache() NN_NOEXCEPT;
    void DestroyShaderList() NN_NOEXCEPT;
    void DestroyImageViewList( List< DestroyingImageViewObjectInfo* >* pDestroyList ) NN_NOEXCEPT;
    static bool IsShaderObjectUsed( const GraphicsPipelineCacheList::Node& graphicsPipelineCache, void* pVkManager ) NN_NOEXCEPT;
    static bool IsShaderObjectUsed( const ComputePipelineCacheList::Node& computePipelineCache, void* pVkManager ) NN_NOEXCEPT;
    static bool IsImageViewObjectUsed( const FramebufferCacheList::Node& framebufferCache, void* pVkManager ) NN_NOEXCEPT;

    static const uint32_t uniformBufferAlignment = 256; //!< Worst case, the largest value of minUniformBufferOffsetAlignment permitted by the Vulkan specification

    CommandBufferImpl< ApiVariationVk1 >*   m_pGfxCommandBuffer;        //!< gfx::CommandBufferへのポインターです

    int32_t                                 m_CommandInfoCount;         //!< 現在処理中のコマンド情報の個数です
    int32_t                                 m_CommandInfoStorageSize;   //!< pCommandInfoArray内のストレージ数です
    GfxVkCommandInfo*                       m_pCommandInfoArray;        //!< コマンド情報の配列へのポインターです

    GraphicsPipelineCacheList               m_GraphicsPipelineCache;    //!< グラフィックスパイプラインキャッシュのハッシュリストです
    ComputePipelineCacheList                m_ComputePipelineCache;     //!< コンピュートパイプラインキャッシュのハッシュリストです
    FramebufferCacheList                    m_FramebufferCache;         //!< フレームバッファキャッシュのハッシュリストです
    DescriptorSetCacheList                  m_DescriptorSetCache;       //!< デスクリプターセットキャッシュのハッシュリストです

    void*                                   m_pUniformBuffer;           //!< CPU mapping of the nn::gfx::CommandBuffer's uniform buffer
    int32_t                                 m_UniformBufferBinding;     //!< Slot index the uniform buffer is presently bound to (or -1 if none)
    uint32_t                                m_UniformBlockSize;         //!< Total size of the currently bound shader's register uniforms
    uint32_t                                m_UniformBufferOffset;      //!< Offset of the register uniform block used by the next draw call

    enum StateFlag
    {
        StateFlag_Cached    //!< Any pipeline object or frambuffer object is cached.
    };
    nn::util::BitPack8                      m_StateFlag;                //!< Flag for state
    List< DestroyingShaderObjectInfo* >          m_DestroyingShaderList;      //!< Shader module to be destroyed after removing related cache.
    List< DestroyingImageViewObjectInfo* >       m_DestroyingColorTargetViewList; //!< Image and ImageView handle for color target to be destroyed after removing related cache.
    List< DestroyingImageViewObjectInfo* >       m_DestroyingDepthTargetViewList; //!< Image and ImageView handle for depth target to be destroyed after removing related cache.
};

}
}
}
