﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <algorithm>
#include <memory>
#include <cstring>
#include <cstdlib>

#include <nn/os.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_StaticAssert.h>

#include <nn/util/util_BytePtr.h>

#include <nn/gfx/gfx_Common.h>

#include <nn/gfx/gfx_GpuAddress.h>
#include <nn/gfx/gfx_CommandBufferInfo.h>

#include <nn/gfx/detail/gfx_MemoryPool-api.vk.1.h>
#include <nn/gfx/detail/gfx_Device-api.vk.1.h>
#include <nn/gfx/detail/gfx_CommandBuffer-api.vk.1.h>
#include <nn/gfx/detail/gfx_Log.h>

#include "gfx_VkHelper.h"
#include "gfx_VkManager.h"
#include "gfx_CommonHelper.h"

namespace nn {
namespace gfx {
namespace detail {

const uint32_t GfxVkGpuState::dynamicStateItemCount;
const uint32_t VkManager::uniformBufferAlignment;

#if defined(NN_SDK_BUILD_DEBUG)
#define USE_DEBUG_COUNTER
#endif

#ifdef USE_DEBUG_COUNTER
bool g_DebugInfoInitialized = false;

struct DebugInformation
{
    void Initialize() NN_NOEXCEPT
    {
        if ( !g_DebugInfoInitialized )
        {
            for ( int idxCounter = 0; idxCounter < DebugInformation::Counter_Last; ++idxCounter )
            {
                objectTotalCreatedCount[ idxCounter ] = 0;
            }
            descriptorSetObjectNodeTotalCreatedCount = 0;
            g_DebugInfoInitialized = true;
        }
    }

    void Reset() NN_NOEXCEPT
    {
        drawCount = 0;
        dispatchCount = 0;
        for ( int idxCounter = 0; idxCounter < Counter_Last; ++idxCounter )
        {
            objectFrameCreatedCount[ idxCounter ] = 0;
            objectCacheHitCount[ idxCounter ] = 0;
            checkCount[ idxCounter ] = 0;
        }
        noStateUpdateDrawCount = 0;
        stateUpdateDrawCount = 0;
        pipelineCacheFailConstState = 0;
        pipelineCacheFailConstStateShaderState = 0;
        pipelineCacheFailConstStateVertexInputState = 0;
        pipelineCacheFailConstStateInputAssemblyState = 0;
        pipelineCacheFailConstStateTessellationState = 0;
        pipelineCacheFailConstStateViewportState = 0;
        pipelineCacheFailConstStateRasterizationState = 0;
        pipelineCacheFailConstStateDepthStencilState = 0;
        pipelineCacheFailConstStateBlendState = 0;
        pipelineCacheFailVertexBufferState = 0;
        pipelineCacheFailVertexAttributeState = 0;
        pipelineCacheFailBlendTargetState = 0;
        pipelineCacheFailRenderTargetState0 = 0;
        pipelineCacheFailRenderTargetState1 = 0;
        pipelineCacheFailRenderTargetState2 = 0;
        pipelineCacheFailDescriptorSetState = 0;
        pipelineCacheFailDescriptorSetStateSlotCount = 0;
        pipelineCacheFailDescriptorSetStateMaxIndex = 0;
        pipelineCacheFailDescriptorSetStateFlag = 0;
        pipelineCacheFailDescriptorSetStateType = 0;
        pipelineCacheFailDescriptorSetStateShaderStage = 0;
    }

    enum Counter
    {
        Counter_Pipeline = 0,
        Counter_Framebuffer,
        Counter_DescriptorSet,
        Counter_Last
    };

    int drawDispatchCount;
    int drawCount;
    int dispatchCount;
    int noStateUpdateDrawCount;
    int stateUpdateDrawCount;
    int objectFrameCreatedCount[ Counter_Last ];
    int objectTotalCreatedCount[ Counter_Last ];
    int objectCacheHitCount[ Counter_Last ];
    int checkCount[ Counter_Last ];
    int descriptorSetObjectNodeTotalCreatedCount;
    int pipelineCacheFailConstState;
    int pipelineCacheFailConstStateShaderState;
    int pipelineCacheFailConstStateVertexInputState;
    int pipelineCacheFailConstStateInputAssemblyState;
    int pipelineCacheFailConstStateTessellationState;
    int pipelineCacheFailConstStateViewportState;
    int pipelineCacheFailConstStateRasterizationState;
    int pipelineCacheFailConstStateDepthStencilState;
    int pipelineCacheFailConstStateBlendState;
    int pipelineCacheFailVertexBufferState;
    int pipelineCacheFailVertexAttributeState;
    int pipelineCacheFailBlendTargetState;
    int pipelineCacheFailRenderTargetState0;
    int pipelineCacheFailRenderTargetState1;
    int pipelineCacheFailRenderTargetState2;
    int pipelineCacheFailDescriptorSetState;
    int pipelineCacheFailDescriptorSetStateSlotCount;
    int pipelineCacheFailDescriptorSetStateMaxIndex;
    int pipelineCacheFailDescriptorSetStateFlag;
    int pipelineCacheFailDescriptorSetStateType;
    int pipelineCacheFailDescriptorSetStateShaderStage;

} g_DebugInfo;

#endif

namespace {


class DeviceInfo
{
    NN_DISALLOW_COPY( DeviceInfo );

public:
    DeviceInfo() NN_NOEXCEPT : m_IsInitialized( false ), m_pGfxDevice( nullptr ){}

    void Initialize( DeviceImpl< ApiVariationVk1 >* pGfxDevice ) NN_NOEXCEPT
    {
        if ( !m_IsInitialized )
        {
            m_pGfxDevice = pGfxDevice;
            m_IsInitialized = true;
        }
    }

    bool IsInitialized() const NN_NOEXCEPT
    {
        return m_IsInitialized;
    }

    DeviceImpl< ApiVariationVk1 >* GetGfxDevice() NN_NOEXCEPT
    {
        return m_pGfxDevice;
    }

private:
    bool m_IsInitialized;
    DeviceImpl< ApiVariationVk1 >* m_pGfxDevice;

} g_DeviceInfo;

DeviceInfo* GetGlobalDeviceInfo() NN_NOEXCEPT
{
    return g_DeviceInfo.IsInitialized() ? &g_DeviceInfo : NULL;
}

void SetGlobalDeviceInfo( DeviceImpl< ApiVariationVk1 >* pDevice ) NN_NOEXCEPT
{
    g_DeviceInfo.Initialize( pDevice );
}

}

VkManager::VkManager( CommandBufferImpl< ApiVariationVk1 >* pCommandBuffer ) NN_NOEXCEPT
{
    SetGlobalDeviceInfo( pCommandBuffer->ToData()->pGfxDevice );

    this->m_pGfxCommandBuffer = pCommandBuffer;
    this->m_CommandInfoStorageSize = 0;
    this->m_pCommandInfoArray = NULL;
    this->m_StateFlag.Clear();

    Reset();

    // Cacheモジュールの初期化
    this->m_GraphicsPipelineCache.SetHashFunction( GraphicsPipelineCache::CreateHashValue );
    this->m_GraphicsPipelineCache.SetTableIndexFunction( GraphicsPipelineCache::GetHashTableIndexFromHashValue );
    this->m_GraphicsPipelineCache.SetCompareFunction( GraphicsPipelineCache::Compare );

    this->m_ComputePipelineCache.SetHashFunction( ComputePipelineCache::CreateHashValue );
    this->m_ComputePipelineCache.SetTableIndexFunction( ComputePipelineCache::GetHashTableIndexFromHashValue );
    this->m_ComputePipelineCache.SetCompareFunction( ComputePipelineCache::Compare );

    this->m_FramebufferCache.SetHashFunction( FramebufferCache::CreateHashValue );
    this->m_FramebufferCache.SetTableIndexFunction( FramebufferCache::GetHashTableIndexFromHashValue );
    this->m_FramebufferCache.SetCompareFunction( FramebufferCache::Compare );

    this->m_DescriptorSetCache.SetHashFunction( DescriptorSetCache::CreateHashValue );
    this->m_DescriptorSetCache.SetTableIndexFunction( DescriptorSetCache::GetHashTableIndexFromHashValue );
    this->m_DescriptorSetCache.SetCompareFunction( DescriptorSetCache::Compare );

    #ifdef USE_DEBUG_COUNTER
    g_DebugInfo.Initialize();
    #endif
}

VkManager::~VkManager() NN_NOEXCEPT
{
    if ( this->m_pCommandInfoArray )
    {
        for ( int idxArray = 0; idxArray < this->m_CommandInfoStorageSize; ++idxArray )
        {
            nn::os::FinalizeLightEvent( &this->m_pCommandInfoArray[ idxArray ].lightEvent );
        }
        Vk::FreeDriverMemory( this->m_pCommandInfoArray );
    }
    ClearCache();
}

void VkManager::Reset() NN_NOEXCEPT
{
    // Resetにより、描画コマンド情報は破棄されますが、
    // Vulkanオブジェクトの情報は破棄されません
    if ( this->m_pCommandInfoArray )
    {
        for ( int idxArray = 0; idxArray < this->m_CommandInfoCount; ++idxArray )
        {
            nn::os::ClearLightEvent( &this->m_pCommandInfoArray[ idxArray ].lightEvent );
            this->m_pCommandInfoArray[ idxArray ].updatedState.Clear();
            this->m_pCommandInfoArray[ idxArray ].info.Clear();
            for ( int idxOffset = 0; idxOffset < GfxVkGpuState::maxDynamicDescriptorSlot; ++idxOffset )
            {
                this->m_pCommandInfoArray[ idxArray ].dynamicOffsets[ idxOffset ] = 0;
            }
        }
    }
    this->m_CommandInfoCount = 0;

    /*
    * キャッシュ内で参照されているVulkanオブジェクトが削除されている場合、
    * そのキャッシュ自体も破棄します。全てのCommandBufferのVkManagerのキャッシュから参照がなくなったときに
    * オブジェクトは破棄されます。
    */
    ClearCache();
}

void VkManager::Begin( int infoCount ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->m_CommandInfoCount == 0 );

    /*
     * ドローごとのコマンド情報を格納するpCommandInfoArrayの配列を確保します。
     * CommandBuffer::End時のドロー＆ディスパッチの回数分生成します。
     * 前回より回数が増加した場合のみ再確保します。
    */
    if ( this->m_CommandInfoStorageSize < infoCount )
    {
        if ( this->m_pCommandInfoArray != NULL )
        {
            for ( int idxArray = 0; idxArray < this->m_CommandInfoStorageSize; ++idxArray )
            {
                nn::os::FinalizeLightEvent( &this->m_pCommandInfoArray[ idxArray ].lightEvent );
            }
            Vk::FreeDriverMemory( this->m_pCommandInfoArray );
        }
        this->m_pCommandInfoArray = static_cast< GfxVkCommandInfo* >(
            Vk::AllocDriverMemory( sizeof( GfxVkCommandInfo ) * infoCount, 8 ) );
        memset( this->m_pCommandInfoArray, 0, sizeof( GfxVkCommandInfo ) * infoCount ); // 再確保されないときは、VkManager::Reset()でクリーンされます。
        for ( int idxArray = 0; idxArray < infoCount; ++idxArray )
        {
            nn::os::InitializeLightEvent( &this->m_pCommandInfoArray[ idxArray ].lightEvent, false, nn::os::EventClearMode_ManualClear );
        }
        this->m_CommandInfoStorageSize = infoCount;
    }
    this->m_CommandInfoCount = infoCount;

    /*
     *  DescriptorSetCacheは、リストの各ノードに、同じDescriptorSetLayoutを元にしたDescriptorSetが複数格納されており
     *  描画で使用される順番に割り当てていきます。各DescriptorSetは同一のCommandBuffer内でただ1回のみ使われます。
     *  vkCmdPushDescriptorSetが対応されたら不用になる処理です。
    */
    for ( int idxTable = 0; idxTable < DescriptorSetCacheHashSize;  ++idxTable )
    {
        DescriptorSetCacheList::HashNode& hashNode = this->m_DescriptorSetCache.GetHashTableNode( idxTable );
        for ( DescriptorSetCacheList::HashNodeIterator iterator = hashNode.Begin();
            !hashNode.IsEnd( iterator ); ++iterator )
        {
            ( *iterator ).nodeData.SetCurrentNodeIteratorBegin();
        }
    }

    this->m_pUniformBuffer = this->m_pGfxCommandBuffer->ToData()->pUniformMemory;
    this->m_UniformBufferBinding = -1;
    this->m_UniformBlockSize = 0;
    this->m_UniformBufferOffset = 0;

    #ifdef USE_DEBUG_COUNTER
    g_DebugInfo.Reset();
    g_DebugInfo.drawDispatchCount = infoCount;
    this->m_GraphicsPipelineCache.ClearCounter();
    this->m_ComputePipelineCache.ClearCounter();
    this->m_FramebufferCache.ClearCounter();
    this->m_DescriptorSetCache.ClearCounter();
    #endif
}

void VkManager::End() NN_NOEXCEPT
{
    VkResult result;
    VkDevice device = CastToVkDispatchableObject< VkDevice >(
        this->m_pGfxCommandBuffer->ToData()->pGfxDevice->ToData()->hDevice );

    VkMappedMemoryRange range;
    range.sType = VK_STRUCTURE_TYPE_MAPPED_MEMORY_RANGE;
    range.pNext = NULL;
    range.memory = CastToVkNonDispatchableObject< VkDeviceMemory >( this->m_pGfxCommandBuffer->ToData()->hUniformMemory );
    range.offset = 0;
    range.size = nn::util::align_up( this->m_UniformBufferOffset, static_cast< uint32_t >( this->m_pGfxCommandBuffer->ToData()->pGfxDevice->ToData()->nonCoherentAtomSize ) );

    NN_GFX_CALL_VK_FUNCTION( result = vkFlushMappedMemoryRanges( device, 1, &range ) );
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );

    // Update cached flag.
    if ( this->m_CommandInfoCount != 0 )
    {
        if ( !m_StateFlag.GetBit( StateFlag_Cached ) )
        {
            m_StateFlag.SetBit( StateFlag_Cached, true );
        }
    }
    else if ( m_StateFlag.GetBit( StateFlag_Cached ) )
    {
        m_StateFlag.SetBit( StateFlag_Cached, !m_GraphicsPipelineCache.IsEmpty() || !m_FramebufferCache.IsEmpty() );
    }

    #ifdef USE_DEBUG_COUNTER
    g_DebugInfo.objectFrameCreatedCount[ DebugInformation::Counter_Pipeline ] = this->m_GraphicsPipelineCache.GetMissCount() + this->m_ComputePipelineCache.GetMissCount();
    g_DebugInfo.objectFrameCreatedCount[ DebugInformation::Counter_Framebuffer ] = this->m_FramebufferCache.GetMissCount();
    g_DebugInfo.objectFrameCreatedCount[ DebugInformation::Counter_DescriptorSet ] = this->m_DescriptorSetCache.GetMissCount();
    g_DebugInfo.objectTotalCreatedCount[ DebugInformation::Counter_Pipeline ] += g_DebugInfo.objectFrameCreatedCount[ DebugInformation::Counter_Pipeline ];
    g_DebugInfo.objectTotalCreatedCount[ DebugInformation::Counter_Framebuffer ] += g_DebugInfo.objectFrameCreatedCount[ DebugInformation::Counter_Framebuffer ];
    g_DebugInfo.objectTotalCreatedCount[ DebugInformation::Counter_DescriptorSet ] += g_DebugInfo.objectFrameCreatedCount[ DebugInformation::Counter_DescriptorSet ];
    g_DebugInfo.objectCacheHitCount[ DebugInformation::Counter_Pipeline ] = this->m_GraphicsPipelineCache.GetHitCount() + this->m_ComputePipelineCache.GetHitCount();
    g_DebugInfo.objectCacheHitCount[ DebugInformation::Counter_Framebuffer ] = this->m_FramebufferCache.GetHitCount();
    g_DebugInfo.objectCacheHitCount[ DebugInformation::Counter_DescriptorSet ] = this->m_DescriptorSetCache.GetHitCount();
    #endif
}

void VkManager::ClearCache() NN_NOEXCEPT
{
    if ( m_DestroyingShaderList.IsEmpty() && m_DestroyingColorTargetViewList.IsEmpty()
        && m_DestroyingDepthTargetViewList.IsEmpty() )
    {
        return;
    }

    NN_GFX_CALL_NNOS_FUNCTION( nn::os::LockMutex( &this->m_pGfxCommandBuffer->ToData()->pGfxDevice->ToData()->resouceMutex ) );
    if ( !m_DestroyingShaderList.IsEmpty() )
    {
        this->m_GraphicsPipelineCache.RemoveIf( IsShaderObjectUsed, this );
        this->m_ComputePipelineCache.RemoveIf( IsShaderObjectUsed, this );

        DestroyShaderList();
    }

    if ( !m_DestroyingColorTargetViewList.IsEmpty() || !m_DestroyingDepthTargetViewList.IsEmpty() )
    {
        this->m_FramebufferCache.RemoveIf( IsImageViewObjectUsed, this );

        DestroyImageViewList( &this->m_DestroyingColorTargetViewList );
        DestroyImageViewList( &this->m_DestroyingDepthTargetViewList );
    }

    NN_GFX_CALL_NNOS_FUNCTION( nn::os::UnlockMutex( &this->m_pGfxCommandBuffer->ToData()->pGfxDevice->ToData()->resouceMutex ) );
}

bool VkManager::IsShaderObjectUsed( const GraphicsPipelineCacheList::Node& graphicsPipelineCache, void* pVkManager ) NN_NOEXCEPT
{
    GraphicsPipelineCacheList::Node* pCache = const_cast< GraphicsPipelineCacheList::Node* >( &graphicsPipelineCache );
    auto pPipelineState = pCache->nodeData.GetConstSizedPipelineState();

    List< DestroyingShaderObjectInfo* >* pDestroyList = &( static_cast< VkManager* >( pVkManager )->m_DestroyingShaderList );
    for ( auto iterator = pDestroyList->Begin(); !pDestroyList->IsEnd( iterator ); ++iterator )
    {
        for ( int idxStage = 0; idxStage < ShaderStage_End; ++idxStage )
        {
            VkShaderModule shader = CastToVkNonDispatchableObject< VkShaderModule >( pPipelineState->shaderState.shaderModule[ idxStage ] );
            if ( iterator->GetHandle().handle[ idxStage ] != 0
                && iterator->GetHandle().handle[ idxStage ] == shader )
            {
                return true;
            }
        }
    }

    return false;
}

bool VkManager::IsShaderObjectUsed( const ComputePipelineCacheList::Node& computePipelineCache, void* pVkManager ) NN_NOEXCEPT
{
    ComputePipelineCacheList::Node* pCache = const_cast< ComputePipelineCacheList::Node* >( &computePipelineCache );
    auto pPipelineState = pCache->nodeData.GetConstSizedPipelineState();

    List< DestroyingShaderObjectInfo* >* pDestroyList = &( static_cast< VkManager* >( pVkManager )->m_DestroyingShaderList );
    for ( auto iterator = pDestroyList->Begin(); !pDestroyList->IsEnd( iterator ); ++iterator )
    {
        VkShaderModule shader = CastToVkNonDispatchableObject< VkShaderModule >( pPipelineState->shaderState.shaderModule[ ShaderStage_Compute ] );
        if ( iterator->GetHandle().handle[ ShaderStage_Compute ] != 0
            && iterator->GetHandle().handle[ ShaderStage_Compute ] == shader )
        {
            return true;
        }
    }

    return false;
}

bool VkManager::IsImageViewObjectUsed( const FramebufferCacheList::Node& framebufferCache, void* pVkManager ) NN_NOEXCEPT
{
    FramebufferCacheList::Node* pCache = const_cast< FramebufferCacheList::Node* >( & framebufferCache );
    auto pState = pCache->nodeData.GetRenderTargetState();

    List< DestroyingImageViewObjectInfo* >* pDestroyList = &( static_cast< VkManager* >( pVkManager )->m_DestroyingColorTargetViewList );
    for ( auto iterator = pDestroyList->Begin(); !pDestroyList->IsEnd( iterator ); ++iterator )
    {
        for ( int idxColor = 0; idxColor < GfxVkGpuState::maxColorAttachments; ++idxColor )
        {
            VkImageView imageView = CastToVkNonDispatchableObject< VkImageView >( pState->colorTargetView[ idxColor ] );
            if ( imageView != 0 && imageView == iterator->GetHandle().imageViewHandle )
            {
                return true;
            }
        }
    }

    pDestroyList = &( static_cast< VkManager* >( pVkManager )->m_DestroyingDepthTargetViewList );
    for ( auto iterator = pDestroyList->Begin(); !pDestroyList->IsEnd( iterator ); ++iterator )
    {
        VkImageView imageView = CastToVkNonDispatchableObject< VkImageView >( pState->depthStencilTargetView );
        if ( imageView != 0 && imageView == iterator->GetHandle().imageViewHandle )
        {
            return true;
        }
    }

    return false;
}

void VkManager::DestroyShaderList() NN_NOEXCEPT
{
    auto pDestroyList = &this->m_DestroyingShaderList;
    for ( auto iterator = pDestroyList->Begin(); !pDestroyList->IsEnd( iterator ); ++iterator )
    {
        int count = iterator->GetReferenceCount().Decrement();
        if ( count == 0 )
        {
            Vk::DestroyShaderObject( this->m_pGfxCommandBuffer->ToData()->pGfxDevice, *iterator );
        }
    }

    pDestroyList->RemoveAll();
}

void VkManager::DestroyImageViewList( List< DestroyingImageViewObjectInfo* >* pDestroyList ) NN_NOEXCEPT
{
    for ( auto iterator = pDestroyList->Begin(); !pDestroyList->IsEnd( iterator ); ++iterator )
    {
        int imageViewReferenceCount = iterator->GetReferenceCount().Decrement();
        if ( imageViewReferenceCount == 0 )
        {
            int imageReferenceCount = iterator->GetHandle().pDestroyingImageObjectInfo->GetReferenceCount().Decrement();
            if ( imageReferenceCount == 0 )
            {
                Vk::DestroyImageObject( this->m_pGfxCommandBuffer->ToData()->pGfxDevice,
                    iterator->GetHandle().pDestroyingImageObjectInfo );
            }
            Vk::DestroyImageViewObject( this->m_pGfxCommandBuffer->ToData()->pGfxDevice, *iterator );
        }
    }

    pDestroyList->RemoveAll();
}

GfxVkCommandInfo* VkManager::GetCommandInfo( int infoIndex ) NN_NOEXCEPT
{
    if ( infoIndex < this->m_CommandInfoCount )
    {
        return &this->m_pCommandInfoArray[ infoIndex ];
    }
    else
    {
        return NULL;
    }
}

GfxVkCommandInfo* VkManager::GetUpdatedCommandInfo( int infoIndex ) NN_NOEXCEPT
{
    if ( infoIndex < this->m_CommandInfoCount )
    {
        GfxVkCommandInfo* pCommandInfo = &this->m_pCommandInfoArray[ infoIndex ];
        nn::os::WaitLightEvent( &pCommandInfo->lightEvent );
        return pCommandInfo;
    }
    else
    {
        return NULL;
    }
}

ImageLayoutStateCacheList::HashValue GfxVkImageLayoutState::CreateHashValue( const VkImage& imageHandle ) NN_NOEXCEPT
{
    return CastFromVkNonDispatchableObject( imageHandle );
}

int GfxVkImageLayoutState::GetHashTableIndexFromHashValue( ImageLayoutStateCacheList::HashValue hashValue, int hashTableSize ) NN_NOEXCEPT
{
    int index = 0;

    for ( int byte = 0; byte < 8; ++byte )
    {
        index += ( hashValue >> ( byte * 8 ) ) & 0xff;
    }

    return index % hashTableSize;
}

bool GfxVkImageLayoutState::Compare( const GfxVkImageLayoutState& cachedObject, const VkImage& imageHandle ) NN_NOEXCEPT
{
    return cachedObject.image == imageHandle;
}

template< bool isDraw >
GfxVkCommandInfo* VkManager::UpdateCommandInfo( const GfxVkGpuState& gpuState, int infoIndex, bool force ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( this->m_pCommandInfoArray );
    NN_SDK_ASSERT( infoIndex < this->m_CommandInfoCount );

    GfxVkCommandInfo* pCommandInfo = &this->m_pCommandInfoArray[ infoIndex ];

    NN_SDK_ASSERT( infoIndex != 0 || ValidateUpdatedState( gpuState, infoIndex ) );

    #ifdef USE_DEBUG_COUNTER
    if ( isDraw )
    {
        ++g_DebugInfo.drawCount;
    }
    else
    {
        ++g_DebugInfo.dispatchCount;
    }
    #endif

    if ( pCommandInfo->updatedState.GetMaskedBits( ~0 ) != 0 || force )
    {
        pCommandInfo->info.SetBit( GfxVkCommandInfo::CommandInfoFlag_IsStateUpdated, true );

        #ifdef USE_DEBUG_COUNTER
        ++g_DebugInfo.stateUpdateDrawCount;
        #endif

        PipelineCache* pPipelineCache;
        bool isFound = false;

        if ( isDraw )
        {
            pPipelineCache = this->m_GraphicsPipelineCache.GetNode( &isFound, gpuState );
        }
        else
        {
            pPipelineCache = this->m_ComputePipelineCache.GetNode( &isFound, gpuState );
        }

        DescriptorSetCache* pDescriptorSetCache;

        if ( !isFound )
        {
            // パイプラインキャッシュが見つからない場合、先にDescriptorSetキャッシュを探す or 生成して初期化します。
            // VkDescriptorSetLayoutはDescriptorSetCache側で生成し、VkPipelineLayoutの生成のためにPipelineCacheへ渡します。
            pDescriptorSetCache = this->m_DescriptorSetCache.GetNode( &isFound, gpuState );
            if ( !isFound )
            {
                pDescriptorSetCache->Initialize( gpuState );
            }
            VkDescriptorSetLayout descriptorSetLayout = pDescriptorSetCache->GetDescriptorSetLayout();

            // キャッシュから見つからなかった場合は新たに生成されたキャッシュノードであるため
            // ノードの初期化をします。
            pPipelineCache->Initialize( gpuState, descriptorSetLayout );

            pPipelineCache->SetDescriptorSetCache( pDescriptorSetCache );
        }
        else
        {
            // パイプラインキャッシュが見つかった場合は、対応するDescriptorSetキャッシュが既に利用可能なはずです。
            pDescriptorSetCache = pPipelineCache->GetDescriptorSetCache();
            NN_SDK_ASSERT_NOT_NULL( pDescriptorSetCache );
        }

        // DescriptorSetCacheのVkDescriptorPoolとVkDescriptorSetが不足している場合は
        // 新規に生成して追加します。
        if ( pDescriptorSetCache->IsCurrentNodeIteratorEnd() )
        {
            VkDescriptorSetLayout descriptorSetLayout = pDescriptorSetCache->GetDescriptorSetLayout();
            DescriptorSetCache::Node& descriptorSetCacheNode = pDescriptorSetCache->GetNodeList().EmplaceBack();
            descriptorSetCacheNode.Initialize( gpuState, descriptorSetLayout );
            pDescriptorSetCache->SetCurrentNodeIteratorBeforeEnd();
        }

        DescriptorSetCache::Node& descriptorSetNode = *( pDescriptorSetCache->GetCurrentNodeIterator() );
        // 次のDescriptorSetの利用のため、リストを進めておく
        pDescriptorSetCache->SetCurrentNodeIteratorNext();

        FramebufferCache* pFramebufferCache = NULL;

        if ( isDraw )
        {
            VkRenderPass hRenderPass = pPipelineCache->GetRenderPass();
            pFramebufferCache = this->m_FramebufferCache.GetNode( &isFound, gpuState );
            if ( !isFound )
            {
                pFramebufferCache->Initialize( gpuState, hRenderPass );
            }
        }

        pCommandInfo->pPipelineCache = pPipelineCache;
        pCommandInfo->pFramebufferCache = pFramebufferCache;
        pCommandInfo->pDescriptorSetCache = pDescriptorSetCache;
        pCommandInfo->pDescriptorSetCacheNode = &descriptorSetNode;
    }
    else
    {
        NN_SDK_ASSERT( infoIndex > 0 );
        #ifdef USE_DEBUG_COUNTER
        ++g_DebugInfo.noStateUpdateDrawCount;
        #endif

        // 前回の描画からステート変更がされていない場合は、同じVulkanオブジェクトを使用します。
        // ただし、通常は、前回の描画でバインドしたVulkanオブジェクトをそのまま使用します。
        // ここでコピーする前回の描画で使用したVulkanオブジェクトは、描画後にClearを呼び出す場合のように、
        // RenderPassの開始後に一時的にRenderPassを終了させたときに、RenderPassを再Beginする時に使用するだけでなく
        // 前後のPipelineオブジェクトを比較し、同じであれば再Bindを避けるために参照します。
        GfxVkCommandInfo* pPreviousCommandInfo = &this->m_pCommandInfoArray[ infoIndex - 1 ];

        pCommandInfo->pPipelineCache = pPreviousCommandInfo->pPipelineCache;
        pCommandInfo->pFramebufferCache = pPreviousCommandInfo->pFramebufferCache;
        pCommandInfo->pDescriptorSetCache = pPreviousCommandInfo->pDescriptorSetCache;
        pCommandInfo->pDescriptorSetCacheNode = pPreviousCommandInfo->pDescriptorSetCacheNode;
    }
    pCommandInfo->info.SetBit( GfxVkCommandInfo::CommandInfoFlag_IsDrawCommand, isDraw );

    nn::os::SignalLightEvent( &pCommandInfo->lightEvent );

    return pCommandInfo;
}

template GfxVkCommandInfo* VkManager::UpdateCommandInfo< true >( const GfxVkGpuState& gpuState, int infoIndex, bool force ) NN_NOEXCEPT;
template GfxVkCommandInfo* VkManager::UpdateCommandInfo< false >( const GfxVkGpuState& gpuState, int infoIndex, bool force ) NN_NOEXCEPT;

bool VkManager::ValidateUpdatedState( const GfxVkGpuState& gpuState, int infoIndex ) NN_NOEXCEPT
{
    const GfxVkCommandInfo* pCommandInfo = GetCommandInfo( infoIndex );

    bool isValid = pCommandInfo->updatedState.GetBit( GfxVkCommandInfo::UpdatedStateFlag_SetShader );
    if ( gpuState.constSizedPipelineState.shaderState.shaderModule[ ShaderStage_Compute ] == 0 )
    {
        isValid &= pCommandInfo->updatedState.GetBit( GfxVkCommandInfo::UpdatedStateFlag_SetRenderTarget );
        isValid &= pCommandInfo->updatedState.GetBit( GfxVkCommandInfo::UpdatedStateFlag_SetRasterizerState );
        isValid &= pCommandInfo->updatedState.GetBit( GfxVkCommandInfo::UpdatedStateFlag_SetBlendState );
        //isValid &= pCommandInfo->updatedState.GetBit( GfxVkCommandInfo::UpdatedStateFlag_SetDepthStencilState );
        //isValid &= pCommandInfo->updatedState.GetBit( GfxVkCommandInfo::UpdatedStateFlag_SetVertexState );
    }
    else
    {
        isValid &= pCommandInfo->updatedState.GetBit( GfxVkCommandInfo::UpdatedStateFlag_SetDescriptorPool );
    }
    if ( gpuState.constSizedPipelineState.shaderState.shaderModule[ ShaderStage_Hull ] != 0 )
    {
        isValid &= pCommandInfo->updatedState.GetBit( GfxVkCommandInfo::UpdatedStateFlag_SetTessellationState );
    }

    return isValid;
}

void VkManager::SetUniformBlockSize( uint32_t size ) NN_NOEXCEPT
{
    this->m_UniformBlockSize = size;
}

void VkManager::SetUniformBufferBinding( int slot ) NN_NOEXCEPT
{
    this->m_UniformBufferBinding = slot;
}

int32_t VkManager::GetUniformBufferBinding() NN_NOEXCEPT
{
    return this->m_UniformBufferBinding;
}

uint32_t VkManager::GetUniformBufferOffset() NN_NOEXCEPT
{
    return this->m_UniformBufferOffset;
}

void* VkManager::GetUniformBufferPtr( uint32_t offset ) NN_NOEXCEPT
{
    return reinterpret_cast< uint8_t* >( this->m_pUniformBuffer ) + offset;
}

void VkManager::AdvanceUniformBufferOffset() NN_NOEXCEPT
{
    uint32_t offset = nn::util::align_up( this->m_UniformBufferOffset + this->m_UniformBlockSize, uniformBufferAlignment );
    memcpy( GetUniformBufferPtr( offset ), GetUniformBufferPtr( this->m_UniformBufferOffset ), this->m_UniformBlockSize );
    this->m_UniformBufferOffset = offset;
}

void VkManager::RegisterDestroyingShaderList( DestroyingShaderObjectInfo* pDestroyingShaderObjectInfo ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pDestroyingShaderObjectInfo );
    DestroyingShaderObjectInfo*& pNewNode = this->m_DestroyingShaderList.EmplaceBack();
    pNewNode = pDestroyingShaderObjectInfo;
}

void VkManager::RegisterDestroyingColorTargetViewList( DestroyingImageViewObjectInfo* pDestroyingImageViewObjectInfo ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pDestroyingImageViewObjectInfo );
    DestroyingImageViewObjectInfo*& pNewNode = this->m_DestroyingColorTargetViewList.EmplaceBack();
    pNewNode = pDestroyingImageViewObjectInfo;
}

void VkManager::RegisterDestroyingDepthTargetViewList( DestroyingImageViewObjectInfo* pDestroyingImageViewObjectInfo ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pDestroyingImageViewObjectInfo );
    DestroyingImageViewObjectInfo*& pNewNode = this->m_DestroyingDepthTargetViewList.EmplaceBack();
    pNewNode = pDestroyingImageViewObjectInfo;
}

PipelineCache::PipelineCache() NN_NOEXCEPT
{
}

PipelineCache::~PipelineCache() NN_NOEXCEPT
{
}

GraphicsPipelineCache::GraphicsPipelineCache() NN_NOEXCEPT
{
}

GraphicsPipelineCache::~GraphicsPipelineCache() NN_NOEXCEPT
{
    Finalize();
}

void GraphicsPipelineCache::Initialize( const GfxVkGpuState& gpuState, VkDescriptorSetLayout descriptorSetLayout ) NN_NOEXCEPT
{
    memcpy( &this->m_ConstSizedPipelineState, &gpuState.constSizedPipelineState, sizeof( GfxVkGpuState::ConstSizedPipelineState ) );
    memcpy( &this->m_VariableSizedPipelineState, &gpuState.variableSizedPipelineState, sizeof( GfxVkGpuState::VariableSizedPipelineState ) );
    memcpy( &this->m_RenderTargetState, &gpuState.renderTargetState, sizeof( GfxVkGpuState::RenderTargetState ) );
    //memcpy( &this->descriptorSetLayoutState, &gpuState.descriptorSetLayoutState, sizeof( GfxVkGpuState::DescriptorSetLayoutState ) );
    this->m_DescriptorSetLayoutState.availableDescriptorCount = gpuState.descriptorSetLayoutState.availableDescriptorCount;
    this->m_DescriptorSetLayoutState.maxDescriptorIndex = gpuState.descriptorSetLayoutState.maxDescriptorIndex;
    this->m_DescriptorSetLayoutState.validSlotMask = gpuState.descriptorSetLayoutState.validSlotMask;
    memcpy( this->m_DescriptorSetLayoutState.slotState, gpuState.descriptorSetLayoutState.slotState,
        sizeof( GfxVkGpuState::DescriptorSetLayoutState::SlotState ) * ( gpuState.descriptorSetLayoutState.maxDescriptorIndex + 1 ) );

    DeviceInfo* pDeviceInfo = GetGlobalDeviceInfo();
    NN_SDK_ASSERT_NOT_NULL( pDeviceInfo );
    DeviceImpl< ApiVariationVk1 >* pGfxDevice = pDeviceInfo->GetGfxDevice();

    Vk::CreatePipelineLayout( &this->m_hPipelineLayout, descriptorSetLayout, pGfxDevice );

    Vk::CreateRenderPass( &this->m_hRenderPass, pGfxDevice, gpuState );

    Vk::CreateGraphicsPipeline( &this->m_hPipeline, pGfxDevice, gpuState, this->m_hPipelineLayout, this->m_hRenderPass );

    this->m_pDescriptorSetCache = NULL;
}

void GraphicsPipelineCache::Finalize() NN_NOEXCEPT
{
    DeviceInfo* pDeviceInfo = GetGlobalDeviceInfo();
    NN_SDK_ASSERT_NOT_NULL( pDeviceInfo );

    VkDevice device = CastToVkDispatchableObject< VkDevice >( pDeviceInfo->GetGfxDevice()->ToData()->hDevice );
    const VkAllocationCallbacks* pAllocator =
        static_cast< const VkAllocationCallbacks* >( pDeviceInfo->GetGfxDevice()->ToData()->pAllocationCallback.ptr );

    if ( this->m_hPipeline != VK_NULL_HANDLE )
    {
        NN_GFX_CALL_VK_FUNCTION( vkDestroyPipeline( device, this->m_hPipeline, pAllocator ) );
    }

    if ( this->m_hPipelineLayout != VK_NULL_HANDLE )
    {
        NN_GFX_CALL_VK_FUNCTION( vkDestroyPipelineLayout( device, this->m_hPipelineLayout, pAllocator ) );
    }

    if ( this->m_hRenderPass != VK_NULL_HANDLE )
    {
        NN_GFX_CALL_VK_FUNCTION( vkDestroyRenderPass( device, this->m_hRenderPass, pAllocator ) );
    }
}

GraphicsPipelineCacheList::HashValue GraphicsPipelineCache::CreateHashValue( const GfxVkGpuState& gpuState ) NN_NOEXCEPT
{
    GraphicsPipelineCacheList::HashValue hashValue = 0;

    // 一先ずシェーダモジュールのみでhashValueを作ります。
    for ( int idxShader = 0; idxShader < ShaderStage_End; ++idxShader )
    {
        if ( gpuState.constSizedPipelineState.shaderState.shaderModule[ idxShader ] != 0 )
        {
            hashValue += static_cast< GraphicsPipelineCacheList::HashValue >( gpuState.constSizedPipelineState.shaderState.shaderModule[ idxShader ] );
        }
    }
    // G3dDemo/Townを参考に以下追加
    for ( uint32_t idxBuffer = 0; idxBuffer < gpuState.constSizedPipelineState.vertexInputState.bufferStateCount; ++idxBuffer )
    {
        hashValue += static_cast< GraphicsPipelineCacheList::HashValue >( gpuState.variableSizedPipelineState.vertexBufferState[ idxBuffer ].stride );
    }
    for ( uint32_t idxAttribute = 0; idxAttribute < gpuState.constSizedPipelineState.vertexInputState.attributeStateCount; ++idxAttribute )
    {
        hashValue += static_cast< GraphicsPipelineCacheList::HashValue >( gpuState.variableSizedPipelineState.vertexAttributeState[ idxAttribute ].format );
    }
    for ( uint32_t idxBlendTarget = 0; idxBlendTarget < gpuState.constSizedPipelineState.blendState.blendTargetCount; ++idxBlendTarget )
    {
        hashValue += static_cast< GraphicsPipelineCacheList::HashValue >(
            gpuState.variableSizedPipelineState.blendTargetState[ idxBlendTarget ].srcRGB
            | ( gpuState.variableSizedPipelineState.blendTargetState[ idxBlendTarget ].dstRGB << 6 ) );
    }
    // porterhouseを参考に以下追加
    hashValue += static_cast< GraphicsPipelineCacheList::HashValue >( gpuState.constSizedPipelineState.depthStencilState.flag.GetMaskedBits( ~0 ) ) << 16;
    hashValue += static_cast< GraphicsPipelineCacheList::HashValue >( gpuState.constSizedPipelineState.depthStencilState.depthFunc ) << 32;
    hashValue += static_cast< GraphicsPipelineCacheList::HashValue >( gpuState.descriptorSetLayoutState.availableDescriptorCount ) << 36;

    return hashValue;
}

int GraphicsPipelineCache::GetHashTableIndexFromHashValue( GraphicsPipelineCacheList::HashValue hashValue, int hashTableSize ) NN_NOEXCEPT
{
    int index = 0;

    for ( int byte = 0; byte < 8; ++byte )
    {
        index += ( hashValue >> ( byte * 8 ) ) & 0xff;
    }

    return index % hashTableSize;
}

bool GraphicsPipelineCache::Compare( const GraphicsPipelineCache& cachedObject,
    const GfxVkGpuState& gpuState ) NN_NOEXCEPT
{
    #ifdef USE_DEBUG_COUNTER
    ++g_DebugInfo.checkCount[ DebugInformation::Counter_Pipeline ];
    #endif

    const GfxVkGpuState::ConstSizedPipelineState* pConstPipelineState0 = &cachedObject.m_ConstSizedPipelineState;
    const GfxVkGpuState::ConstSizedPipelineState* pConstPipelineState1 = &gpuState.constSizedPipelineState;
    const GfxVkGpuState::VariableSizedPipelineState* pVariablePipelineState0 = &cachedObject.m_VariableSizedPipelineState;
    const GfxVkGpuState::VariableSizedPipelineState* pVariablePipelineState1 = &gpuState.variableSizedPipelineState;
    const GfxVkGpuState::RenderTargetState* pRenderTargetState0 = &cachedObject.m_RenderTargetState;
    const GfxVkGpuState::RenderTargetState* pRenderTargetState1 = &gpuState.renderTargetState;
    const GfxVkGpuState::DescriptorSetLayoutState* pDescriptorSetLayoutState0 = &cachedObject.m_DescriptorSetLayoutState;
    const GfxVkGpuState::DescriptorSetLayoutState* pDescriptorSetLayoutState1 = &gpuState.descriptorSetLayoutState;

    // Compare constant sized state.
    if ( memcmp( pConstPipelineState0, pConstPipelineState1,
        sizeof( GfxVkGpuState::ConstSizedPipelineState ) ) != 0 )
    {
        #ifdef USE_DEBUG_COUNTER
        ++g_DebugInfo.pipelineCacheFailConstState;

        if ( memcmp( &pConstPipelineState0->shaderState, &pConstPipelineState1->shaderState,
            sizeof( GfxVkGpuState::ConstSizedPipelineState::ShaderState ) ) != 0 )
        {
            ++g_DebugInfo.pipelineCacheFailConstStateShaderState;
        }
        if ( memcmp( &pConstPipelineState0->vertexInputState, &pConstPipelineState1->vertexInputState,
            sizeof( GfxVkGpuState::ConstSizedPipelineState::VertexInputState ) ) != 0 )
        {
            ++g_DebugInfo.pipelineCacheFailConstStateVertexInputState;
        }
        if ( memcmp( &pConstPipelineState0->inputAssemblyState, &pConstPipelineState1->inputAssemblyState,
            sizeof( GfxVkGpuState::ConstSizedPipelineState::InputAssemblyState ) ) != 0 )
        {
            ++g_DebugInfo.pipelineCacheFailConstStateInputAssemblyState;
        }
        if ( memcmp( &pConstPipelineState0->tessellationState, &pConstPipelineState1->tessellationState,
            sizeof( GfxVkGpuState::ConstSizedPipelineState::TessellationState ) ) != 0 )
        {
            ++g_DebugInfo.pipelineCacheFailConstStateTessellationState;
        }
        if ( memcmp( &pConstPipelineState0->viewportState, &pConstPipelineState1->viewportState,
            sizeof( GfxVkGpuState::ConstSizedPipelineState::ViewportState ) ) != 0 )
        {
            ++g_DebugInfo.pipelineCacheFailConstStateViewportState;
        }
        if ( memcmp( &pConstPipelineState0->rasterizationState, &pConstPipelineState1->rasterizationState,
            sizeof( GfxVkGpuState::ConstSizedPipelineState::RasterizationState ) ) != 0 )
        {
            ++g_DebugInfo.pipelineCacheFailConstStateRasterizationState;
        }
        if ( memcmp( &pConstPipelineState0->depthStencilState, &pConstPipelineState1->depthStencilState,
            sizeof( GfxVkGpuState::ConstSizedPipelineState::DepthStencilState ) ) != 0 )
        {
            ++g_DebugInfo.pipelineCacheFailConstStateDepthStencilState;
        }
        if ( memcmp( &pConstPipelineState0->blendState, &pConstPipelineState1->blendState,
            sizeof( GfxVkGpuState::ConstSizedPipelineState::BlendState ) ) != 0 )
        {
            ++g_DebugInfo.pipelineCacheFailConstStateBlendState;
        }
        #endif
        return false;
    }

    // Compare variable sized state.
    if ( memcmp( &pVariablePipelineState0->vertexBufferState, &pVariablePipelineState1->vertexBufferState,
        sizeof( GfxVkGpuState::VariableSizedPipelineState::VertexBufferState )
        * pConstPipelineState0->vertexInputState.bufferStateCount ) != 0 )
    {
        #ifdef USE_DEBUG_COUNTER
        ++g_DebugInfo.pipelineCacheFailVertexBufferState;
        #endif
        return false;
    }
    if ( memcmp( &pVariablePipelineState0->vertexAttributeState, &pVariablePipelineState1->vertexAttributeState,
        sizeof( GfxVkGpuState::VariableSizedPipelineState::VertexAttributeState )
        * pConstPipelineState0->vertexInputState.attributeStateCount ) != 0 )
    {
        #ifdef USE_DEBUG_COUNTER
        ++g_DebugInfo.pipelineCacheFailVertexAttributeState;
        #endif
        return false;
    }
    if ( memcmp( &pVariablePipelineState0->blendTargetState, &pVariablePipelineState1->blendTargetState,
        sizeof( GfxVkGpuState::VariableSizedPipelineState::BlendTargetState )
        * pConstPipelineState0->blendState.blendTargetCount ) != 0 )
    {
        #ifdef USE_DEBUG_COUNTER
        ++g_DebugInfo.pipelineCacheFailBlendTargetState;
        #endif
        return false;
    }

    // Compare render target state
    if ( pRenderTargetState0->flag.GetMaskedBits( ~0 ) != pRenderTargetState1->flag.GetMaskedBits( ~0 )
        || pRenderTargetState0->colorAttachmentCount != pRenderTargetState1->colorAttachmentCount )
    {
        #ifdef USE_DEBUG_COUNTER
        ++g_DebugInfo.pipelineCacheFailRenderTargetState0;
        #endif
        return false;
    }
    if ( pRenderTargetState0->colorAttachmentCount > 0
        && memcmp( pRenderTargetState0->colorAttachment, pRenderTargetState1->colorAttachment,
        sizeof( GfxVkGpuState::RenderTargetState::AttachmentState ) * pRenderTargetState0->colorAttachmentCount ) != 0 )
    {
        #ifdef USE_DEBUG_COUNTER
        ++g_DebugInfo.pipelineCacheFailRenderTargetState1;
        #endif
        return false;
    }
    if ( pRenderTargetState0->flag.GetBit( GfxVkGpuState::RenderTargetState::Flag_DepthAttached )
        && memcmp( &pRenderTargetState0->depthStencilAttachment, &pRenderTargetState1->depthStencilAttachment,
        sizeof( GfxVkGpuState::RenderTargetState::AttachmentState ) ) != 0 )
    {
        #ifdef USE_DEBUG_COUNTER
        ++g_DebugInfo.pipelineCacheFailRenderTargetState2;
        #endif
        return false;
    }

    // Compare DescriptorSetLayout state
    if ( pDescriptorSetLayoutState0->availableDescriptorCount != pDescriptorSetLayoutState1->availableDescriptorCount
        || pDescriptorSetLayoutState0->maxDescriptorIndex != pDescriptorSetLayoutState1->maxDescriptorIndex
        || pDescriptorSetLayoutState0->validSlotMask != pDescriptorSetLayoutState1->validSlotMask
        || memcmp( pDescriptorSetLayoutState0->slotState, pDescriptorSetLayoutState1->slotState,
        sizeof( GfxVkGpuState::DescriptorSetLayoutState::SlotState ) * ( pDescriptorSetLayoutState0->maxDescriptorIndex + 1 ) ) != 0 )
    {
        #ifdef USE_DEBUG_COUNTER
        ++g_DebugInfo.pipelineCacheFailDescriptorSetState;
        if ( pDescriptorSetLayoutState0->availableDescriptorCount != pDescriptorSetLayoutState1->availableDescriptorCount )
        {
            ++g_DebugInfo.pipelineCacheFailDescriptorSetStateSlotCount;
        }
        if ( pDescriptorSetLayoutState0->maxDescriptorIndex != pDescriptorSetLayoutState1->maxDescriptorIndex )
        {
            ++g_DebugInfo.pipelineCacheFailDescriptorSetStateMaxIndex;
        }
        if ( pDescriptorSetLayoutState0->availableDescriptorCount == pDescriptorSetLayoutState1->availableDescriptorCount
            && pDescriptorSetLayoutState0->maxDescriptorIndex == pDescriptorSetLayoutState1->maxDescriptorIndex )
        {
            bool flagIsSame = true;
            bool typeIsSame = true;
            bool shaderStageIsSame = true;
            for ( uint32_t idx = 0; idx <= pDescriptorSetLayoutState0->maxDescriptorIndex; ++idx )
            {
                if ( pDescriptorSetLayoutState0->slotState[ idx ].flag.storage != pDescriptorSetLayoutState1->slotState[ idx ].flag.storage )
                {
                    flagIsSame = false;
                }
                if ( pDescriptorSetLayoutState0->slotState[ idx ].type != pDescriptorSetLayoutState1->slotState[ idx ].type )
                {
                    typeIsSame = false;
                }
                if ( pDescriptorSetLayoutState0->slotState[ idx ].shaderStage != pDescriptorSetLayoutState1->slotState[ idx ].shaderStage )
                {
                    shaderStageIsSame = false;
                }
            }
            if ( !flagIsSame )
            {
                ++g_DebugInfo.pipelineCacheFailDescriptorSetStateFlag;
            }
            if ( !typeIsSame )
            {
                ++g_DebugInfo.pipelineCacheFailDescriptorSetStateType;
            }
            if ( !shaderStageIsSame )
            {
                ++g_DebugInfo.pipelineCacheFailDescriptorSetStateShaderStage;
            }
        }
        #endif
        return false;
    }

    return true;
} // NOLINT(impl/function_size)

ComputePipelineCache::ComputePipelineCache() NN_NOEXCEPT
{
}

ComputePipelineCache::~ComputePipelineCache() NN_NOEXCEPT
{
    Finalize();
}

void ComputePipelineCache::Initialize( const GfxVkGpuState& gpuState, VkDescriptorSetLayout descriptorSetLayout ) NN_NOEXCEPT
{
    memcpy( &this->m_ConstSizedPipelineState, &gpuState.constSizedPipelineState, sizeof( GfxVkGpuState::ConstSizedPipelineState ) );
    memcpy( &this->m_VariableSizedPipelineState, &gpuState.variableSizedPipelineState, sizeof( GfxVkGpuState::VariableSizedPipelineState ) );
    //memcpy( &this->descriptorSetLayoutState, &gpuState.descriptorSetLayoutState, sizeof( GfxVkGpuState::DescriptorSetLayoutState ) );
    this->m_DescriptorSetLayoutState.availableDescriptorCount = gpuState.descriptorSetLayoutState.availableDescriptorCount;
    this->m_DescriptorSetLayoutState.maxDescriptorIndex = gpuState.descriptorSetLayoutState.maxDescriptorIndex;
    this->m_DescriptorSetLayoutState.validSlotMask = gpuState.descriptorSetLayoutState.validSlotMask;
    memcpy( this->m_DescriptorSetLayoutState.slotState, gpuState.descriptorSetLayoutState.slotState,
        sizeof( GfxVkGpuState::DescriptorSetLayoutState::SlotState ) * ( gpuState.descriptorSetLayoutState.maxDescriptorIndex + 1 ) );

    DeviceInfo* pDeviceInfo = GetGlobalDeviceInfo();
    NN_SDK_ASSERT_NOT_NULL( pDeviceInfo );
    Vk::CreatePipelineLayout( &this->m_hPipelineLayout, descriptorSetLayout, pDeviceInfo->GetGfxDevice() );

    Vk::CreateComputePipeline( &this->m_hPipeline, pDeviceInfo->GetGfxDevice(), gpuState, this->m_hPipelineLayout );

    this->m_pDescriptorSetCache = NULL;
}

void ComputePipelineCache::Finalize() NN_NOEXCEPT
{
    DeviceInfo* pDeviceInfo = GetGlobalDeviceInfo();
    NN_SDK_ASSERT_NOT_NULL( pDeviceInfo );

    VkDevice device = CastToVkDispatchableObject< VkDevice >( pDeviceInfo->GetGfxDevice()->ToData()->hDevice );
    const VkAllocationCallbacks* pAllocator =
        static_cast< const VkAllocationCallbacks* >( pDeviceInfo->GetGfxDevice()->ToData()->pAllocationCallback.ptr );

    if ( this->m_hPipeline != VK_NULL_HANDLE )
    {
        NN_GFX_CALL_VK_FUNCTION( vkDestroyPipeline( device, this->m_hPipeline, pAllocator ) );
    }

    if ( this->m_hPipelineLayout != VK_NULL_HANDLE )
    {
        NN_GFX_CALL_VK_FUNCTION( vkDestroyPipelineLayout( device, this->m_hPipelineLayout, pAllocator ) );
    }
}

ComputePipelineCacheList::HashValue ComputePipelineCache::CreateHashValue( const GfxVkGpuState& gpuState ) NN_NOEXCEPT
{
    ComputePipelineCacheList::HashValue hashValue = 0;

    for ( int idxShader = 0; idxShader < ShaderStage_End; ++idxShader )
    {
        if ( gpuState.constSizedPipelineState.shaderState.shaderModule[ idxShader ] != 0 )
        {
            hashValue += static_cast< ComputePipelineCacheList::HashValue >( gpuState.constSizedPipelineState.shaderState.shaderModule[ idxShader ] );
        }
    }

    hashValue += static_cast< ComputePipelineCacheList::HashValue >( gpuState.descriptorSetLayoutState.availableDescriptorCount ) << 36;

    return hashValue;
}

int ComputePipelineCache::GetHashTableIndexFromHashValue( ComputePipelineCacheList::HashValue hashValue, int hashTableSize ) NN_NOEXCEPT
{
    int index = 0;

    for ( int byte = 0; byte < 8; ++byte )
    {
        index += ( hashValue >> ( byte * 8 ) ) & 0xff;
    }

    return index % hashTableSize;
}

bool ComputePipelineCache::Compare( const ComputePipelineCache& cachedObject,
    const GfxVkGpuState& gpuState ) NN_NOEXCEPT
{
    #ifdef USE_DEBUG_COUNTER
    ++g_DebugInfo.checkCount[ DebugInformation::Counter_Pipeline ];
    #endif

    const GfxVkGpuState::ConstSizedPipelineState* pConstPipelineState0 = &cachedObject.m_ConstSizedPipelineState;
    const GfxVkGpuState::ConstSizedPipelineState* pConstPipelineState1 = &gpuState.constSizedPipelineState;
    const GfxVkGpuState::VariableSizedPipelineState* pVariablePipelineState0 = &cachedObject.m_VariableSizedPipelineState;
    const GfxVkGpuState::VariableSizedPipelineState* pVariablePipelineState1 = &gpuState.variableSizedPipelineState;
    const GfxVkGpuState::DescriptorSetLayoutState* pDescriptorSetLayoutState0 = &cachedObject.m_DescriptorSetLayoutState;
    const GfxVkGpuState::DescriptorSetLayoutState* pDescriptorSetLayoutState1 = &gpuState.descriptorSetLayoutState;

    // Compare constant sized state.
    if ( memcmp( pConstPipelineState0, pConstPipelineState1,
        sizeof( GfxVkGpuState::ConstSizedPipelineState ) ) != 0 )
    {
        return false;
    }

    // Compare variable sized state.
    if ( memcmp( &pVariablePipelineState0->vertexBufferState, &pVariablePipelineState1->vertexBufferState,
        sizeof( GfxVkGpuState::VariableSizedPipelineState::VertexBufferState )
        * pConstPipelineState0->vertexInputState.bufferStateCount ) != 0 )
    {
        return false;
    }
    if ( memcmp( &pVariablePipelineState0->vertexAttributeState, &pVariablePipelineState1->vertexAttributeState,
        sizeof( GfxVkGpuState::VariableSizedPipelineState::VertexAttributeState )
        * pConstPipelineState0->vertexInputState.attributeStateCount ) != 0 )
    {
        return false;
    }
    if ( memcmp( &pVariablePipelineState0->blendTargetState, &pVariablePipelineState1->blendTargetState,
        sizeof( GfxVkGpuState::VariableSizedPipelineState::BlendTargetState )
        * pConstPipelineState0->blendState.blendTargetCount ) != 0 )
    {
        return false;
    }

    // Compare DescriptorSetLayout state
    if ( pDescriptorSetLayoutState0->availableDescriptorCount != pDescriptorSetLayoutState1->availableDescriptorCount
        || pDescriptorSetLayoutState0->maxDescriptorIndex != pDescriptorSetLayoutState1->maxDescriptorIndex
        || pDescriptorSetLayoutState0->validSlotMask != pDescriptorSetLayoutState1->validSlotMask
        || memcmp( pDescriptorSetLayoutState0->slotState, pDescriptorSetLayoutState1->slotState,
        sizeof( GfxVkGpuState::DescriptorSetLayoutState::SlotState ) * ( pDescriptorSetLayoutState0->maxDescriptorIndex + 1 ) ) != 0 )
    {
        return false;
    }

    return true;
}

FramebufferCache::FramebufferCache() NN_NOEXCEPT
{
}

FramebufferCache::~FramebufferCache() NN_NOEXCEPT
{
    Finalize();
}

void FramebufferCache::Initialize( const GfxVkGpuState& gpuState, VkRenderPass renderPass ) NN_NOEXCEPT
{
    memcpy( &this->m_RenderTargetState, &gpuState.renderTargetState, sizeof( GfxVkGpuState::RenderTargetState ) );

    DeviceInfo* pDeviceInfo = GetGlobalDeviceInfo();
    NN_SDK_ASSERT_NOT_NULL( pDeviceInfo );

    DeviceImpl< ApiVariationVk1 >* pDevice = pDeviceInfo->GetGfxDevice();

    Vk::CreateFramebuffer( &this->m_hFramebuffer, pDevice, gpuState, renderPass );
}

void FramebufferCache::Finalize() NN_NOEXCEPT
{
    DeviceImpl< ApiVariationVk1 >* pDevice = GetGlobalDeviceInfo()->GetGfxDevice();
    VkDevice device = CastToVkDispatchableObject< VkDevice >( pDevice->ToData()->hDevice );
    const VkAllocationCallbacks* pAllocator =
        static_cast< const VkAllocationCallbacks* >( pDevice->ToData()->pAllocationCallback.ptr );

    if ( this->m_hFramebuffer != VK_NULL_HANDLE )
    {
        NN_GFX_CALL_VK_FUNCTION( vkDestroyFramebuffer( device, this->m_hFramebuffer, pAllocator ) );
    }
}

FramebufferCacheList::HashValue FramebufferCache::CreateHashValue( const GfxVkGpuState& gpuState ) NN_NOEXCEPT
{
    FramebufferCacheList::HashValue hashValue = 0;

    // Framebufferは、ImageViewが一致しないといけないため
    // ImageViewを使ったkeyにする
    for ( int idxAttachment = 0; idxAttachment < gpuState.renderTargetState.colorAttachmentCount; ++idxAttachment )
    {
        hashValue += static_cast< FramebufferCacheList::HashValue >( gpuState.renderTargetState.colorTargetView[ idxAttachment ] );
    }
    if ( gpuState.renderTargetState.flag.GetBit( GfxVkGpuState::RenderTargetState::Flag_DepthAttached ) )
    {
        hashValue += static_cast< FramebufferCacheList::HashValue >( gpuState.renderTargetState.depthStencilTargetView );
    }

    return hashValue;
}

int FramebufferCache::GetHashTableIndexFromHashValue( FramebufferCacheList::HashValue hashValue, int hashTableSize ) NN_NOEXCEPT
{
    int index = 0;

    for ( int byte = 0; byte < 8; ++byte )
    {
        index += ( hashValue >> ( byte * 8 ) ) & 0xff;
    }

    return index % hashTableSize;
}

bool FramebufferCache::Compare( const FramebufferCache& cachedObject, const GfxVkGpuState& gpuState ) NN_NOEXCEPT
{
    #ifdef USE_DEBUG_COUNTER
    ++g_DebugInfo.checkCount[ DebugInformation::Counter_Framebuffer ];
    #endif

    const GfxVkGpuState::RenderTargetState* pRenderTargetState0 = &cachedObject.m_RenderTargetState;
    const GfxVkGpuState::RenderTargetState* pRenderTargetState1 = &gpuState.renderTargetState;

    // Compare render target state
    if ( pRenderTargetState0->flag.GetMaskedBits( ~0 ) != pRenderTargetState1->flag.GetMaskedBits( ~0 )
        || pRenderTargetState0->colorAttachmentCount != pRenderTargetState1->colorAttachmentCount
        || memcmp( pRenderTargetState0->colorTargetView, pRenderTargetState1->colorTargetView,
        sizeof( detail::VkHandle ) * pRenderTargetState0->colorAttachmentCount ) != 0 )
    {
        return false;
    }
    if ( pRenderTargetState0->flag.GetBit( GfxVkGpuState::RenderTargetState::Flag_DepthAttached )
        && pRenderTargetState0->depthStencilTargetView != pRenderTargetState1->depthStencilTargetView )
    {
        return false;
    }

    return true;
}

DescriptorSetCache::DescriptorSetCache() NN_NOEXCEPT
{
}

DescriptorSetCache::~DescriptorSetCache() NN_NOEXCEPT
{
    Finalize();
}

void DescriptorSetCache::Initialize( const GfxVkGpuState& gpuState ) NN_NOEXCEPT
{
    this->m_CurrentNodeIterator = m_NodeList.Begin();

    //memcpy( &this->descriptorSetLayoutState, &gpuState.descriptorSetLayoutState, sizeof( GfxVkGpuState::DescriptorSetLayoutState ) );
    this->m_DescriptorSetLayoutState.availableDescriptorCount = gpuState.descriptorSetLayoutState.availableDescriptorCount;
    this->m_DescriptorSetLayoutState.maxDescriptorIndex = gpuState.descriptorSetLayoutState.maxDescriptorIndex;
    this->m_DescriptorSetLayoutState.validSlotMask = gpuState.descriptorSetLayoutState.validSlotMask;
    memcpy( this->m_DescriptorSetLayoutState.slotState, gpuState.descriptorSetLayoutState.slotState,
        sizeof( GfxVkGpuState::DescriptorSetLayoutState::SlotState ) * ( gpuState.descriptorSetLayoutState.maxDescriptorIndex + 1 ) );

    DeviceInfo* pDeviceInfo = GetGlobalDeviceInfo();
    NN_SDK_ASSERT_NOT_NULL( pDeviceInfo );
    Vk::CreateDescriptorSetLayout( &this->m_hDescriptorSetLayout, pDeviceInfo->GetGfxDevice(), gpuState );

    // ここでは、VkDescriptorPoolおよびVkDescriptorSetは生成しません。
}

void DescriptorSetCache::Finalize() NN_NOEXCEPT
{
    DeviceInfo* pDeviceInfo = GetGlobalDeviceInfo();
    NN_SDK_ASSERT_NOT_NULL( pDeviceInfo );

    VkDevice device = CastToVkDispatchableObject< VkDevice >( pDeviceInfo->GetGfxDevice()->ToData()->hDevice );
    const VkAllocationCallbacks* pAllocator =
        static_cast< const VkAllocationCallbacks* >( pDeviceInfo->GetGfxDevice()->ToData()->pAllocationCallback.ptr );

    if ( this->m_hDescriptorSetLayout != VK_NULL_HANDLE )
    {
        NN_GFX_CALL_VK_FUNCTION( vkDestroyDescriptorSetLayout( device, this->m_hDescriptorSetLayout, pAllocator ) );
    }
}

DescriptorSetCacheList::HashValue DescriptorSetCache::CreateHashValue( const GfxVkGpuState& gpuState ) NN_NOEXCEPT
{
    DescriptorSetCacheList::HashValue hashValue = 0;
    uint64_t validDescriptorMask = gpuState.descriptorSetLayoutState.validSlotMask;

    // DescriptorSetは、typeとShaderStageが一致すればよいため、BufferやImageViewは使用しない。
    for ( uint32_t idxSlot = 0; idxSlot <=
        gpuState.descriptorSetLayoutState.maxDescriptorIndex; ++idxSlot )
    {
        const GfxVkGpuState::DescriptorSetLayoutState::SlotState* pState =
            &gpuState.descriptorSetLayoutState.slotState[ idxSlot ];

        if ( !pState->flag.GetBit( GfxVkGpuState::DescriptorSetLayoutState::SlotState::Flag_IsSet )
            || !( validDescriptorMask & ( 1ULL << idxSlot ) ) )
        {
            continue;
        }

        hashValue += static_cast< DescriptorSetCacheList::HashValue >( ( pState->type << 16 ) | pState->shaderStage );
    }

    return hashValue;
}

int DescriptorSetCache::GetHashTableIndexFromHashValue( DescriptorSetCacheList::HashValue hashValue, int hashTableSize ) NN_NOEXCEPT
{
    int index = 0;

    for ( int byte = 0; byte < 8; ++byte )
    {
        index += ( hashValue >> ( byte * 8 ) ) & 0xff;
    }

    return index % hashTableSize;
}

bool DescriptorSetCache::Compare( const DescriptorSetCache& cachedObject, const GfxVkGpuState& gpuState ) NN_NOEXCEPT
{
    #ifdef USE_DEBUG_COUNTER
    ++g_DebugInfo.checkCount[ DebugInformation::Counter_DescriptorSet ];
    #endif

    const GfxVkGpuState::DescriptorSetLayoutState* pDescriptorSetLayoutState0 = &cachedObject.m_DescriptorSetLayoutState;
    const GfxVkGpuState::DescriptorSetLayoutState* pDescriptorSetLayoutState1 = &gpuState.descriptorSetLayoutState;

    // Compare DescriptorSetLayout state
    if ( pDescriptorSetLayoutState0->availableDescriptorCount != pDescriptorSetLayoutState1->availableDescriptorCount
        || pDescriptorSetLayoutState0->maxDescriptorIndex != pDescriptorSetLayoutState1->maxDescriptorIndex
        || pDescriptorSetLayoutState0->validSlotMask != pDescriptorSetLayoutState1->validSlotMask
        || memcmp( pDescriptorSetLayoutState0->slotState, pDescriptorSetLayoutState1->slotState,
        sizeof( GfxVkGpuState::DescriptorSetLayoutState::SlotState ) * ( pDescriptorSetLayoutState0->maxDescriptorIndex + 1 ) ) != 0 )
    {
        return false;
    }

    return true;
}

DescriptorSetCache::Node::Node() NN_NOEXCEPT
{
}

DescriptorSetCache::Node::~Node() NN_NOEXCEPT
{
    Finalize();
}

void DescriptorSetCache::Node::Initialize( const GfxVkGpuState& gpuState, VkDescriptorSetLayout descriptorSetLayout ) NN_NOEXCEPT
{
    // ここで、VkDescriptorPoolおよびVkDescriptorSetを生成します。
    Vk::CreateDescriptorSet( &this->m_hDescriptorPool, &this->m_hDescriptorSet,
        GetGlobalDeviceInfo()->GetGfxDevice(), gpuState, descriptorSetLayout );

    this->m_pWriteDescriptorSet = gpuState.descriptorSetLayoutState.availableDescriptorCount == 0 ? NULL :
        static_cast< VkWriteDescriptorSet* >( Vk::AllocDriverMemory(
        sizeof( VkWriteDescriptorSet ) * gpuState.descriptorSetLayoutState.availableDescriptorCount, 8 ) );

    uint64_t validDescriptorMask = gpuState.descriptorSetLayoutState.validSlotMask;
    uint32_t idxAvailableBinding = 0;
    for ( uint32_t idxSlotState = 0; idxSlotState <=
        gpuState.descriptorSetLayoutState.maxDescriptorIndex; ++idxSlotState )
    {
        const GfxVkGpuState::DescriptorSetLayoutState::SlotState* pSrcState =
            &gpuState.descriptorSetLayoutState.slotState[ idxSlotState ];

        if ( !pSrcState->flag.GetBit( GfxVkGpuState::DescriptorSetLayoutState::SlotState::Flag_IsSet )
            || !( validDescriptorMask & ( 1ULL << idxSlotState ) ) )
        {
            continue;
        }

        VkWriteDescriptorSet* pDstState = &this->m_pWriteDescriptorSet[ idxAvailableBinding ];

        pDstState->sType = VK_STRUCTURE_TYPE_WRITE_DESCRIPTOR_SET;
        pDstState->pNext = NULL;
        pDstState->dstBinding = idxSlotState;
        pDstState->dstArrayElement = 0;
        pDstState->descriptorCount = 1;
        pDstState->descriptorType = static_cast< VkDescriptorType >( pSrcState->type );

        ++idxAvailableBinding;
    }

    #ifdef USE_DEBUG_COUNTER
    ++g_DebugInfo.descriptorSetObjectNodeTotalCreatedCount;
    #endif
}

void DescriptorSetCache::Node::Finalize() NN_NOEXCEPT
{
    VkDevice device = CastToVkDispatchableObject< VkDevice >( GetGlobalDeviceInfo()->GetGfxDevice()->ToData()->hDevice );
    const VkAllocationCallbacks* pAllocator =
        static_cast< const VkAllocationCallbacks* >( GetGlobalDeviceInfo()->GetGfxDevice()->ToData()->pAllocationCallback.ptr );

    if ( this->m_hDescriptorSet != VK_NULL_HANDLE )
    {
        NN_GFX_CALL_VK_FUNCTION( vkFreeDescriptorSets( device, this->m_hDescriptorPool, 1, &this->m_hDescriptorSet ) );
    }
    if ( this->m_hDescriptorPool != VK_NULL_HANDLE )
    {
        NN_GFX_CALL_VK_FUNCTION( vkDestroyDescriptorPool( device, this->m_hDescriptorPool, pAllocator ) );
    }

    if ( this->m_pWriteDescriptorSet )
    {
        Vk::FreeDriverMemory( this->m_pWriteDescriptorSet );
    }
}


}
}
}
