﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <algorithm>
#include <memory>
#include <cstring>
#include <cstdlib>

#include <nn/nn_SdkLog.h>
#include <nn/nn_StaticAssert.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os/os_MutexApi.h>

#include <nn/util/util_BytePtr.h>
#include <nn/util/util_BitArray.h>

#include <nn/gfx/gfx_Common.h>

#include <nn/gfx/gfx_GpuAddress.h>
#include <nn/gfx/gfx_CommandBufferInfo.h>

#include <nn/gfx/detail/gfx_MemoryPool-api.vk.1.h>
#include <nn/gfx/detail/gfx_Device-api.vk.1.h>
#include <nn/gfx/detail/gfx_CommandBuffer-api.vk.1.h>
#include <nn/gfx/detail/gfx_Shader-api.vk.1.h>
#include <nn/gfx/detail/gfx_Texture-api.vk.1.h>
#include <nn/gfx/detail/gfx_Log.h>

#include "gfx_VkHelper.h"
#include "gfx_VkManager.h"
#include "gfx_CommonHelper.h"

namespace nn {
namespace gfx {
namespace detail {

namespace {

const struct ImageFormatAndVkFormat
{
    ImageFormat imageFormat;
    VkFormat vkFormat;
} VkTextureFormatList[] =
{
    // This is sorted by ImageFormat
    { ImageFormat_R8_Unorm, VK_FORMAT_R8_UNORM },
    { ImageFormat_R8_Snorm, VK_FORMAT_R8_SNORM },
    { ImageFormat_R8_Uint, VK_FORMAT_R8_UINT },
    { ImageFormat_R8_Sint, VK_FORMAT_R8_SINT },
    { ImageFormat_R4_G4_B4_A4_Unorm, VK_FORMAT_UNDEFINED },
    { ImageFormat_A4_B4_G4_R4_Unorm, VK_FORMAT_R4G4B4A4_UNORM_PACK16 },
    { ImageFormat_R5_G5_B5_A1_Unorm, VK_FORMAT_UNDEFINED },
    { ImageFormat_A1_B5_G5_R5_Unorm, VK_FORMAT_R5G5B5A1_UNORM_PACK16 },
    { ImageFormat_R5_G6_B5_Unorm, VK_FORMAT_B5G6R5_UNORM_PACK16 },
    { ImageFormat_B5_G6_R5_Unorm, VK_FORMAT_R5G6B5_UNORM_PACK16 },
    { ImageFormat_R8_G8_Unorm, VK_FORMAT_R8G8_UNORM },
    { ImageFormat_R8_G8_Snorm, VK_FORMAT_R8G8_SNORM },
    { ImageFormat_R8_G8_Uint, VK_FORMAT_R8G8_UINT },
    { ImageFormat_R8_G8_Sint, VK_FORMAT_R8G8_SINT },
    { ImageFormat_R16_Unorm, VK_FORMAT_R16_UNORM },
    { ImageFormat_R16_Snorm, VK_FORMAT_R16_SNORM },
    { ImageFormat_R16_Uint, VK_FORMAT_R16_UINT },
    { ImageFormat_R16_Sint, VK_FORMAT_R16_SINT },
    { ImageFormat_R16_Float, VK_FORMAT_R16_SFLOAT },
    { ImageFormat_D16_Unorm, VK_FORMAT_D16_UNORM },
    { ImageFormat_R8_G8_B8_A8_Unorm, VK_FORMAT_R8G8B8A8_UNORM },
    { ImageFormat_R8_G8_B8_A8_Snorm, VK_FORMAT_R8G8B8A8_SNORM },
    { ImageFormat_R8_G8_B8_A8_Uint, VK_FORMAT_R8G8B8A8_UINT },
    { ImageFormat_R8_G8_B8_A8_Sint, VK_FORMAT_R8G8B8A8_SINT },
    { ImageFormat_R8_G8_B8_A8_UnormSrgb, VK_FORMAT_R8G8B8A8_SRGB },
    { ImageFormat_B8_G8_R8_A8_Unorm, VK_FORMAT_B8G8R8A8_UNORM},
    { ImageFormat_B8_G8_R8_A8_Snorm, VK_FORMAT_B8G8R8A8_SNORM},
    { ImageFormat_B8_G8_R8_A8_Uint, VK_FORMAT_B8G8R8A8_UINT},
    { ImageFormat_B8_G8_R8_A8_Sint, VK_FORMAT_B8G8R8A8_SINT},
    { ImageFormat_B8_G8_R8_A8_UnormSrgb, VK_FORMAT_B8G8R8A8_SRGB},
    { ImageFormat_R9_G9_B9_E5_SharedExp, VK_FORMAT_E5B9G9R9_UFLOAT_PACK32 },
    { ImageFormat_R10_G10_B10_A2_Unorm, VK_FORMAT_A2B10G10R10_UNORM_PACK32 },
    { ImageFormat_R10_G10_B10_A2_Uint, VK_FORMAT_A2B10G10R10_UINT_PACK32 },
    { ImageFormat_R11_G11_B10_Float, VK_FORMAT_B10G11R11_UFLOAT_PACK32 },
    { ImageFormat_R16_G16_Unorm, VK_FORMAT_R16G16_UNORM },
    { ImageFormat_R16_G16_Snorm, VK_FORMAT_R16G16_SNORM },
    { ImageFormat_R16_G16_Uint, VK_FORMAT_R16G16_UINT },
    { ImageFormat_R16_G16_Sint, VK_FORMAT_R16G16_SINT },
    { ImageFormat_R16_G16_Float, VK_FORMAT_R16G16_SFLOAT },
    { ImageFormat_D24_Unorm_S8_Uint, VK_FORMAT_D24_UNORM_S8_UINT },
    { ImageFormat_R32_Uint, VK_FORMAT_R32_UINT },
    { ImageFormat_R32_Sint, VK_FORMAT_R32_SINT },
    { ImageFormat_R32_Float, VK_FORMAT_R32_SFLOAT },
    { ImageFormat_D32_Float, VK_FORMAT_D32_SFLOAT },
    { ImageFormat_R16_G16_B16_A16_Unorm, VK_FORMAT_R16G16B16A16_UNORM },
    { ImageFormat_R16_G16_B16_A16_Snorm, VK_FORMAT_R16G16B16A16_SNORM },
    { ImageFormat_R16_G16_B16_A16_Uint, VK_FORMAT_R16G16B16A16_UINT },
    { ImageFormat_R16_G16_B16_A16_Sint, VK_FORMAT_R16G16B16A16_SINT },
    { ImageFormat_R16_G16_B16_A16_Float, VK_FORMAT_R16G16B16A16_SFLOAT },
    { ImageFormat_D32_Float_S8_Uint_X24, VK_FORMAT_D32_SFLOAT_S8_UINT },
    { ImageFormat_R32_G32_Uint, VK_FORMAT_R32G32_UINT },
    { ImageFormat_R32_G32_Sint, VK_FORMAT_R32G32_SINT },
    { ImageFormat_R32_G32_Float, VK_FORMAT_R32G32_SFLOAT },
    { ImageFormat_R32_G32_B32_Uint, VK_FORMAT_R32G32B32_UINT },
    { ImageFormat_R32_G32_B32_Sint, VK_FORMAT_R32G32B32_SINT },
    { ImageFormat_R32_G32_B32_Float, VK_FORMAT_R32G32B32_SFLOAT },
    { ImageFormat_R32_G32_B32_A32_Uint, VK_FORMAT_R32G32B32A32_UINT },
    { ImageFormat_R32_G32_B32_A32_Sint, VK_FORMAT_R32G32B32A32_SINT },
    { ImageFormat_R32_G32_B32_A32_Float, VK_FORMAT_R32G32B32A32_SFLOAT },
    { ImageFormat_Bc1_Unorm, VK_FORMAT_BC1_RGBA_UNORM_BLOCK },
    { ImageFormat_Bc1_UnormSrgb, VK_FORMAT_BC1_RGBA_SRGB_BLOCK },
    { ImageFormat_Bc2_Unorm, VK_FORMAT_BC2_UNORM_BLOCK },
    { ImageFormat_Bc2_UnormSrgb, VK_FORMAT_BC2_SRGB_BLOCK },
    { ImageFormat_Bc3_Unorm, VK_FORMAT_BC3_UNORM_BLOCK },
    { ImageFormat_Bc3_UnormSrgb, VK_FORMAT_BC3_SRGB_BLOCK },
    { ImageFormat_Bc4_Unorm, VK_FORMAT_BC4_UNORM_BLOCK },
    { ImageFormat_Bc4_Snorm, VK_FORMAT_BC4_SNORM_BLOCK },
    { ImageFormat_Bc5_Unorm, VK_FORMAT_BC5_UNORM_BLOCK },
    { ImageFormat_Bc5_Snorm, VK_FORMAT_BC5_SNORM_BLOCK },
    { ImageFormat_Bc6_Float, VK_FORMAT_BC6H_SFLOAT_BLOCK },
    { ImageFormat_Bc6_Ufloat, VK_FORMAT_BC6H_UFLOAT_BLOCK },
    { ImageFormat_Bc7_Unorm, VK_FORMAT_BC7_UNORM_BLOCK },
    { ImageFormat_Bc7_UnormSrgb, VK_FORMAT_BC7_SRGB_BLOCK },

    { ImageFormat_Eac_R11_Unorm, VK_FORMAT_EAC_R11_UNORM_BLOCK },
    { ImageFormat_Eac_R11_Snorm, VK_FORMAT_EAC_R11_SNORM_BLOCK },
    { ImageFormat_Eac_R11_G11_Unorm, VK_FORMAT_EAC_R11G11_UNORM_BLOCK },
    { ImageFormat_Eac_R11_G11_Snorm, VK_FORMAT_EAC_R11G11_SNORM_BLOCK },

    { ImageFormat_Etc2_Unorm, VK_FORMAT_ETC2_R8G8B8_UNORM_BLOCK },
    { ImageFormat_Etc2_UnormSrgb, VK_FORMAT_ETC2_R8G8B8_SRGB_BLOCK },
    { ImageFormat_Etc2_Mask_Unorm, VK_FORMAT_ETC2_R8G8B8A1_UNORM_BLOCK },
    { ImageFormat_Etc2_Mask_UnormSrgb, VK_FORMAT_ETC2_R8G8B8A1_SRGB_BLOCK },
    { ImageFormat_Etc2_Alpha_Unorm, VK_FORMAT_ETC2_R8G8B8A8_UNORM_BLOCK },
    { ImageFormat_Etc2_Alpha_UnormSrgb, VK_FORMAT_ETC2_R8G8B8A8_SRGB_BLOCK },

    { ImageFormat_Pvrtc1_2Bpp_Unorm, VK_FORMAT_PVRTC1_2BPP_UNORM_BLOCK_IMG },
    { ImageFormat_Pvrtc1_2Bpp_UnormSrgb, VK_FORMAT_PVRTC1_2BPP_SRGB_BLOCK_IMG },
    { ImageFormat_Pvrtc1_4Bpp_Unorm, VK_FORMAT_PVRTC1_4BPP_UNORM_BLOCK_IMG },
    { ImageFormat_Pvrtc1_4Bpp_UnormSrgb, VK_FORMAT_PVRTC1_4BPP_SRGB_BLOCK_IMG },
    { ImageFormat_Pvrtc2_Alpha_2Bpp_Unorm, VK_FORMAT_PVRTC2_2BPP_UNORM_BLOCK_IMG },
    { ImageFormat_Pvrtc2_Alpha_2Bpp_UnormSrgb, VK_FORMAT_PVRTC2_2BPP_SRGB_BLOCK_IMG },
    { ImageFormat_Pvrtc2_Alpha_4Bpp_Unorm, VK_FORMAT_PVRTC2_4BPP_UNORM_BLOCK_IMG },
    { ImageFormat_Pvrtc2_Alpha_4Bpp_UnormSrgb, VK_FORMAT_PVRTC2_4BPP_SRGB_BLOCK_IMG },

    { ImageFormat_Astc_4x4_Unorm, VK_FORMAT_ASTC_4x4_UNORM_BLOCK },
    { ImageFormat_Astc_4x4_UnormSrgb, VK_FORMAT_ASTC_4x4_SRGB_BLOCK },
    { ImageFormat_Astc_5x4_Unorm, VK_FORMAT_ASTC_5x4_UNORM_BLOCK },
    { ImageFormat_Astc_5x4_UnormSrgb, VK_FORMAT_ASTC_5x4_SRGB_BLOCK },
    { ImageFormat_Astc_5x5_Unorm, VK_FORMAT_ASTC_5x5_UNORM_BLOCK },
    { ImageFormat_Astc_5x5_UnormSrgb, VK_FORMAT_ASTC_5x5_SRGB_BLOCK },
    { ImageFormat_Astc_6x5_Unorm, VK_FORMAT_ASTC_6x5_UNORM_BLOCK },
    { ImageFormat_Astc_6x5_UnormSrgb, VK_FORMAT_ASTC_6x5_SRGB_BLOCK },
    { ImageFormat_Astc_6x6_Unorm, VK_FORMAT_ASTC_6x6_UNORM_BLOCK },
    { ImageFormat_Astc_6x6_UnormSrgb, VK_FORMAT_ASTC_6x6_SRGB_BLOCK },
    { ImageFormat_Astc_8x5_Unorm, VK_FORMAT_ASTC_8x5_UNORM_BLOCK },
    { ImageFormat_Astc_8x5_UnormSrgb, VK_FORMAT_ASTC_8x5_SRGB_BLOCK },
    { ImageFormat_Astc_8x6_Unorm, VK_FORMAT_ASTC_8x6_UNORM_BLOCK },
    { ImageFormat_Astc_8x6_UnormSrgb, VK_FORMAT_ASTC_8x6_SRGB_BLOCK },
    { ImageFormat_Astc_8x8_Unorm, VK_FORMAT_ASTC_8x8_UNORM_BLOCK },
    { ImageFormat_Astc_8x8_UnormSrgb, VK_FORMAT_ASTC_8x8_SRGB_BLOCK },
    { ImageFormat_Astc_10x5_Unorm, VK_FORMAT_ASTC_10x5_UNORM_BLOCK },
    { ImageFormat_Astc_10x5_UnormSrgb, VK_FORMAT_ASTC_10x5_SRGB_BLOCK },
    { ImageFormat_Astc_10x6_Unorm, VK_FORMAT_ASTC_10x6_UNORM_BLOCK },
    { ImageFormat_Astc_10x6_UnormSrgb, VK_FORMAT_ASTC_10x6_SRGB_BLOCK },
    { ImageFormat_Astc_10x8_Unorm, VK_FORMAT_ASTC_10x8_UNORM_BLOCK },
    { ImageFormat_Astc_10x8_UnormSrgb, VK_FORMAT_ASTC_10x8_SRGB_BLOCK },
    { ImageFormat_Astc_10x10_Unorm, VK_FORMAT_ASTC_10x10_UNORM_BLOCK },
    { ImageFormat_Astc_10x10_UnormSrgb, VK_FORMAT_ASTC_10x10_SRGB_BLOCK },
    { ImageFormat_Astc_12x10_Unorm, VK_FORMAT_ASTC_12x10_UNORM_BLOCK },
    { ImageFormat_Astc_12x10_UnormSrgb, VK_FORMAT_ASTC_12x10_SRGB_BLOCK },
    { ImageFormat_Astc_12x12_Unorm, VK_FORMAT_ASTC_12x12_UNORM_BLOCK },
    { ImageFormat_Astc_12x12_UnormSrgb, VK_FORMAT_ASTC_12x12_SRGB_BLOCK },

    { ImageFormat_B5_G5_R5_A1_Unorm, VK_FORMAT_UNDEFINED }
};

const struct AttributeFormatAndVkFormat
{
    AttributeFormat attributeFormat;
    VkFormat vkFormat;
} VkAttributeFormatList[] =
{
    { AttributeFormat_4_4_Unorm, VK_FORMAT_R4G4_UNORM_PACK8 },
    { AttributeFormat_8_Unorm, VK_FORMAT_R8_UNORM },
    { AttributeFormat_8_Snorm, VK_FORMAT_R8_SNORM },
    { AttributeFormat_8_Uint, VK_FORMAT_R8_UINT },
    { AttributeFormat_8_Sint, VK_FORMAT_R8_SINT },
    { AttributeFormat_8_UintToFloat, VK_FORMAT_R8_USCALED },
    { AttributeFormat_8_SintToFloat, VK_FORMAT_R8_SSCALED },
    { AttributeFormat_8_8_Unorm, VK_FORMAT_R8G8_UNORM },
    { AttributeFormat_8_8_Snorm, VK_FORMAT_R8G8_SNORM },
    { AttributeFormat_8_8_Uint, VK_FORMAT_R8G8_UINT },
    { AttributeFormat_8_8_Sint, VK_FORMAT_R8G8_SINT },
    { AttributeFormat_8_8_UintToFloat, VK_FORMAT_R8G8_USCALED },
    { AttributeFormat_8_8_SintToFloat, VK_FORMAT_R8G8_SSCALED },
    { AttributeFormat_16_Unorm, VK_FORMAT_R16_UNORM },
    { AttributeFormat_16_Snorm, VK_FORMAT_R16_SNORM },
    { AttributeFormat_16_Uint, VK_FORMAT_R16_UINT },
    { AttributeFormat_16_Sint, VK_FORMAT_R16_SINT },
    { AttributeFormat_16_Float, VK_FORMAT_R16_SFLOAT },
    { AttributeFormat_16_UintToFloat, VK_FORMAT_R16_USCALED },
    { AttributeFormat_16_SintToFloat, VK_FORMAT_R16_SSCALED },
    { AttributeFormat_8_8_8_8_Unorm, VK_FORMAT_R8G8B8A8_UNORM },
    { AttributeFormat_8_8_8_8_Snorm, VK_FORMAT_R8G8B8A8_SNORM },
    { AttributeFormat_8_8_8_8_Uint, VK_FORMAT_R8G8B8A8_UINT },
    { AttributeFormat_8_8_8_8_Sint, VK_FORMAT_R8G8B8A8_SINT },
    { AttributeFormat_8_8_8_8_UintToFloat, VK_FORMAT_R8G8B8A8_USCALED },
    { AttributeFormat_8_8_8_8_SintToFloat, VK_FORMAT_R8G8B8A8_SSCALED },
    { AttributeFormat_10_10_10_2_Unorm, VK_FORMAT_A2B10G10R10_UNORM_PACK32 },
    { AttributeFormat_10_10_10_2_Snorm, VK_FORMAT_A2B10G10R10_SNORM_PACK32 },
    { AttributeFormat_10_10_10_2_Uint, VK_FORMAT_A2B10G10R10_UINT_PACK32 },
    { AttributeFormat_10_10_10_2_Sint, VK_FORMAT_A2B10G10R10_SINT_PACK32 },
    { AttributeFormat_16_16_Unorm, VK_FORMAT_R16G16_UNORM },
    { AttributeFormat_16_16_Snorm, VK_FORMAT_R16G16_SNORM },
    { AttributeFormat_16_16_Uint, VK_FORMAT_R16G16_UINT },
    { AttributeFormat_16_16_Sint, VK_FORMAT_R16G16_SINT },
    { AttributeFormat_16_16_Float, VK_FORMAT_R16G16_SFLOAT },
    { AttributeFormat_16_16_UintToFloat, VK_FORMAT_R16G16_USCALED },
    { AttributeFormat_16_16_SintToFloat, VK_FORMAT_R16G16_SSCALED },
    { AttributeFormat_32_Uint, VK_FORMAT_R32_UINT },
    { AttributeFormat_32_Sint, VK_FORMAT_R32_SINT },
    { AttributeFormat_32_Float, VK_FORMAT_R32_SFLOAT },
    { AttributeFormat_16_16_16_16_Unorm, VK_FORMAT_R16G16B16A16_UNORM },
    { AttributeFormat_16_16_16_16_Snorm, VK_FORMAT_R16G16B16A16_SNORM },
    { AttributeFormat_16_16_16_16_Uint, VK_FORMAT_R16G16B16A16_UINT },
    { AttributeFormat_16_16_16_16_Sint, VK_FORMAT_R16G16B16A16_SINT },
    { AttributeFormat_16_16_16_16_Float, VK_FORMAT_R16G16B16A16_SFLOAT },
    { AttributeFormat_16_16_16_16_UintToFloat, VK_FORMAT_R16G16B16A16_USCALED },
    { AttributeFormat_16_16_16_16_SintToFloat, VK_FORMAT_R16G16B16A16_SSCALED },
    { AttributeFormat_32_32_Uint, VK_FORMAT_R32G32_UINT },
    { AttributeFormat_32_32_Sint, VK_FORMAT_R32G32_SINT },
    { AttributeFormat_32_32_Float, VK_FORMAT_R32G32_SFLOAT },
    { AttributeFormat_32_32_32_Uint, VK_FORMAT_R32G32B32_UINT },
    { AttributeFormat_32_32_32_Sint, VK_FORMAT_R32G32B32_SINT },
    { AttributeFormat_32_32_32_Float, VK_FORMAT_R32G32B32_SFLOAT },
    { AttributeFormat_32_32_32_32_Uint, VK_FORMAT_R32G32B32A32_UINT },
    { AttributeFormat_32_32_32_32_Sint, VK_FORMAT_R32G32B32A32_SINT },
    { AttributeFormat_32_32_32_32_Float, VK_FORMAT_R32G32B32A32_SFLOAT },
};

const struct VkFormatAndImageFormat
{
    VkFormat vkFormat;
    ImageFormat imageFormat;
} VkFormatAndImageFormatList[] =
{
    // This is sorted by vkFormat.
    { VK_FORMAT_R4G4B4A4_UNORM_PACK16, ImageFormat_A4_B4_G4_R4_Unorm },
    { VK_FORMAT_R5G6B5_UNORM_PACK16, ImageFormat_B5_G6_R5_Unorm },
    { VK_FORMAT_B5G6R5_UNORM_PACK16, ImageFormat_R5_G6_B5_Unorm },
    { VK_FORMAT_R5G5B5A1_UNORM_PACK16, ImageFormat_A1_B5_G5_R5_Unorm },
    { VK_FORMAT_A1R5G5B5_UNORM_PACK16, ImageFormat_B5_G5_R5_A1_Unorm },
    { VK_FORMAT_R8_UNORM, ImageFormat_R8_Unorm },
    { VK_FORMAT_R8_SNORM, ImageFormat_R8_Snorm },
    { VK_FORMAT_R8_UINT, ImageFormat_R8_Uint },
    { VK_FORMAT_R8_SINT, ImageFormat_R8_Sint },
    { VK_FORMAT_R8G8_UNORM, ImageFormat_R8_G8_Unorm },
    { VK_FORMAT_R8G8_SNORM, ImageFormat_R8_G8_Snorm },
    { VK_FORMAT_R8G8_UINT, ImageFormat_R8_G8_Uint },
    { VK_FORMAT_R8G8_SINT, ImageFormat_R8_G8_Sint },
    { VK_FORMAT_R8G8B8A8_UNORM, ImageFormat_R8_G8_B8_A8_Unorm },
    { VK_FORMAT_R8G8B8A8_SNORM, ImageFormat_R8_G8_B8_A8_Snorm },
    { VK_FORMAT_R8G8B8A8_UINT, ImageFormat_R8_G8_B8_A8_Uint },
    { VK_FORMAT_R8G8B8A8_SINT, ImageFormat_R8_G8_B8_A8_Sint },
    { VK_FORMAT_R8G8B8A8_SRGB, ImageFormat_R8_G8_B8_A8_UnormSrgb },
    { VK_FORMAT_B8G8R8A8_UNORM, ImageFormat_B8_G8_R8_A8_Unorm},
    { VK_FORMAT_B8G8R8A8_SNORM, ImageFormat_B8_G8_R8_A8_Snorm},
    { VK_FORMAT_B8G8R8A8_UINT, ImageFormat_B8_G8_R8_A8_Uint},
    { VK_FORMAT_B8G8R8A8_SINT, ImageFormat_B8_G8_R8_A8_Sint},
    { VK_FORMAT_B8G8R8A8_SRGB, ImageFormat_B8_G8_R8_A8_UnormSrgb},
    { VK_FORMAT_A2B10G10R10_UNORM_PACK32, ImageFormat_R10_G10_B10_A2_Unorm },
    { VK_FORMAT_A2B10G10R10_UINT_PACK32, ImageFormat_R10_G10_B10_A2_Uint },
    { VK_FORMAT_R16_UNORM, ImageFormat_R16_Unorm },
    { VK_FORMAT_R16_SNORM, ImageFormat_R16_Snorm },
    { VK_FORMAT_R16_UINT, ImageFormat_R16_Uint },
    { VK_FORMAT_R16_SINT, ImageFormat_R16_Sint },
    { VK_FORMAT_R16_SFLOAT, ImageFormat_R16_Float },
    { VK_FORMAT_R16G16_UNORM, ImageFormat_R16_G16_Unorm },
    { VK_FORMAT_R16G16_SNORM, ImageFormat_R16_G16_Snorm },
    { VK_FORMAT_R16G16_UINT, ImageFormat_R16_G16_Uint },
    { VK_FORMAT_R16G16_SINT, ImageFormat_R16_G16_Sint },
    { VK_FORMAT_R16G16_SFLOAT, ImageFormat_R16_G16_Float },
    { VK_FORMAT_R16G16B16A16_UNORM, ImageFormat_R16_G16_B16_A16_Unorm },
    { VK_FORMAT_R16G16B16A16_SNORM, ImageFormat_R16_G16_B16_A16_Snorm },
    { VK_FORMAT_R16G16B16A16_UINT, ImageFormat_R16_G16_B16_A16_Uint },
    { VK_FORMAT_R16G16B16A16_SINT, ImageFormat_R16_G16_B16_A16_Sint },
    { VK_FORMAT_R16G16B16A16_SFLOAT, ImageFormat_R16_G16_B16_A16_Float },
    { VK_FORMAT_R32_UINT, ImageFormat_R32_Uint },
    { VK_FORMAT_R32_SINT, ImageFormat_R32_Sint },
    { VK_FORMAT_R32_SFLOAT, ImageFormat_R32_Float },
    { VK_FORMAT_R32G32_UINT, ImageFormat_R32_G32_Uint },
    { VK_FORMAT_R32G32_SINT, ImageFormat_R32_G32_Sint },
    { VK_FORMAT_R32G32_SFLOAT, ImageFormat_R32_G32_Float },
    { VK_FORMAT_R32G32B32_UINT, ImageFormat_R32_G32_B32_Uint },
    { VK_FORMAT_R32G32B32_SINT, ImageFormat_R32_G32_B32_Sint },
    { VK_FORMAT_R32G32B32_SFLOAT, ImageFormat_R32_G32_B32_Float },
    { VK_FORMAT_R32G32B32A32_UINT, ImageFormat_R32_G32_B32_A32_Uint },
    { VK_FORMAT_R32G32B32A32_SINT, ImageFormat_R32_G32_B32_A32_Sint },
    { VK_FORMAT_R32G32B32A32_SFLOAT, ImageFormat_R32_G32_B32_A32_Float },
    { VK_FORMAT_B10G11R11_UFLOAT_PACK32, ImageFormat_R11_G11_B10_Float },
    { VK_FORMAT_E5B9G9R9_UFLOAT_PACK32, ImageFormat_R9_G9_B9_E5_SharedExp },
    { VK_FORMAT_D16_UNORM, ImageFormat_D16_Unorm },
    { VK_FORMAT_X8_D24_UNORM_PACK32, ImageFormat_D24_Unorm_S8_Uint },
    { VK_FORMAT_D32_SFLOAT, ImageFormat_D32_Float },
    { VK_FORMAT_D24_UNORM_S8_UINT, ImageFormat_D24_Unorm_S8_Uint },
    { VK_FORMAT_D32_SFLOAT_S8_UINT, ImageFormat_D32_Float_S8_Uint_X24 },
    { VK_FORMAT_BC1_RGBA_UNORM_BLOCK, ImageFormat_Bc1_Unorm },
    { VK_FORMAT_BC1_RGBA_SRGB_BLOCK, ImageFormat_Bc1_UnormSrgb },
    { VK_FORMAT_BC2_UNORM_BLOCK, ImageFormat_Bc2_Unorm },
    { VK_FORMAT_BC2_SRGB_BLOCK, ImageFormat_Bc2_UnormSrgb },
    { VK_FORMAT_BC3_UNORM_BLOCK, ImageFormat_Bc3_Unorm },
    { VK_FORMAT_BC3_SRGB_BLOCK, ImageFormat_Bc3_UnormSrgb },
    { VK_FORMAT_BC4_UNORM_BLOCK, ImageFormat_Bc4_Unorm },
    { VK_FORMAT_BC4_SNORM_BLOCK, ImageFormat_Bc4_Snorm },
    { VK_FORMAT_BC5_UNORM_BLOCK, ImageFormat_Bc5_Unorm },
    { VK_FORMAT_BC5_SNORM_BLOCK, ImageFormat_Bc5_Snorm },
    { VK_FORMAT_BC6H_UFLOAT_BLOCK, ImageFormat_Bc6_Ufloat },
    { VK_FORMAT_BC6H_SFLOAT_BLOCK, ImageFormat_Bc6_Float },
    { VK_FORMAT_BC7_UNORM_BLOCK, ImageFormat_Bc7_Unorm },
    { VK_FORMAT_BC7_SRGB_BLOCK, ImageFormat_Bc7_UnormSrgb },
    { VK_FORMAT_ETC2_R8G8B8_UNORM_BLOCK, ImageFormat_Etc2_Unorm },
    { VK_FORMAT_ETC2_R8G8B8_SRGB_BLOCK, ImageFormat_Etc2_UnormSrgb },
    { VK_FORMAT_ETC2_R8G8B8A1_UNORM_BLOCK, ImageFormat_Etc2_Mask_Unorm },
    { VK_FORMAT_ETC2_R8G8B8A1_SRGB_BLOCK, ImageFormat_Etc2_Mask_UnormSrgb },
    { VK_FORMAT_ETC2_R8G8B8A8_UNORM_BLOCK, ImageFormat_Etc2_Alpha_Unorm },
    { VK_FORMAT_ETC2_R8G8B8A8_SRGB_BLOCK, ImageFormat_Etc2_Alpha_UnormSrgb },
    { VK_FORMAT_EAC_R11_UNORM_BLOCK, ImageFormat_Eac_R11_Unorm },
    { VK_FORMAT_EAC_R11_SNORM_BLOCK, ImageFormat_Eac_R11_Snorm },
    { VK_FORMAT_EAC_R11G11_UNORM_BLOCK, ImageFormat_Eac_R11_G11_Unorm },
    { VK_FORMAT_EAC_R11G11_SNORM_BLOCK, ImageFormat_Eac_R11_G11_Snorm },
    { VK_FORMAT_ASTC_4x4_UNORM_BLOCK, ImageFormat_Astc_4x4_Unorm },
    { VK_FORMAT_ASTC_4x4_SRGB_BLOCK, ImageFormat_Astc_4x4_UnormSrgb },
    { VK_FORMAT_ASTC_5x4_UNORM_BLOCK, ImageFormat_Astc_5x4_Unorm },
    { VK_FORMAT_ASTC_5x4_SRGB_BLOCK, ImageFormat_Astc_5x4_UnormSrgb },
    { VK_FORMAT_ASTC_5x5_UNORM_BLOCK, ImageFormat_Astc_5x5_Unorm },
    { VK_FORMAT_ASTC_5x5_SRGB_BLOCK, ImageFormat_Astc_5x5_UnormSrgb },
    { VK_FORMAT_ASTC_6x5_UNORM_BLOCK, ImageFormat_Astc_6x5_Unorm },
    { VK_FORMAT_ASTC_6x5_SRGB_BLOCK, ImageFormat_Astc_6x5_UnormSrgb },
    { VK_FORMAT_ASTC_6x6_UNORM_BLOCK, ImageFormat_Astc_6x6_Unorm },
    { VK_FORMAT_ASTC_6x6_SRGB_BLOCK, ImageFormat_Astc_6x6_UnormSrgb },
    { VK_FORMAT_ASTC_8x5_UNORM_BLOCK, ImageFormat_Astc_8x5_Unorm },
    { VK_FORMAT_ASTC_8x5_SRGB_BLOCK, ImageFormat_Astc_8x5_UnormSrgb },
    { VK_FORMAT_ASTC_8x6_UNORM_BLOCK, ImageFormat_Astc_8x6_Unorm },
    { VK_FORMAT_ASTC_8x6_SRGB_BLOCK, ImageFormat_Astc_8x6_UnormSrgb },
    { VK_FORMAT_ASTC_8x8_UNORM_BLOCK, ImageFormat_Astc_8x8_Unorm },
    { VK_FORMAT_ASTC_8x8_SRGB_BLOCK, ImageFormat_Astc_8x8_UnormSrgb },
    { VK_FORMAT_ASTC_10x5_UNORM_BLOCK, ImageFormat_Astc_10x5_Unorm },
    { VK_FORMAT_ASTC_10x5_SRGB_BLOCK, ImageFormat_Astc_10x5_UnormSrgb },
    { VK_FORMAT_ASTC_10x6_UNORM_BLOCK, ImageFormat_Astc_10x6_Unorm },
    { VK_FORMAT_ASTC_10x6_SRGB_BLOCK, ImageFormat_Astc_10x6_UnormSrgb },
    { VK_FORMAT_ASTC_10x8_UNORM_BLOCK, ImageFormat_Astc_10x8_Unorm },
    { VK_FORMAT_ASTC_10x8_SRGB_BLOCK, ImageFormat_Astc_10x8_UnormSrgb },
    { VK_FORMAT_ASTC_10x10_UNORM_BLOCK, ImageFormat_Astc_10x10_Unorm },
    { VK_FORMAT_ASTC_10x10_SRGB_BLOCK, ImageFormat_Astc_10x10_UnormSrgb },
    { VK_FORMAT_ASTC_12x10_UNORM_BLOCK, ImageFormat_Astc_12x10_Unorm },
    { VK_FORMAT_ASTC_12x10_SRGB_BLOCK, ImageFormat_Astc_12x10_UnormSrgb },
    { VK_FORMAT_ASTC_12x12_UNORM_BLOCK, ImageFormat_Astc_12x12_Unorm },
    { VK_FORMAT_ASTC_12x12_SRGB_BLOCK, ImageFormat_Astc_12x12_UnormSrgb },
    { VK_FORMAT_PVRTC1_2BPP_UNORM_BLOCK_IMG, ImageFormat_Pvrtc1_2Bpp_Unorm },
    { VK_FORMAT_PVRTC1_4BPP_UNORM_BLOCK_IMG, ImageFormat_Pvrtc1_4Bpp_Unorm },
    { VK_FORMAT_PVRTC2_2BPP_UNORM_BLOCK_IMG, ImageFormat_Pvrtc2_Alpha_2Bpp_Unorm },
    { VK_FORMAT_PVRTC2_4BPP_UNORM_BLOCK_IMG, ImageFormat_Pvrtc2_Alpha_4Bpp_Unorm },
    { VK_FORMAT_PVRTC1_2BPP_SRGB_BLOCK_IMG, ImageFormat_Pvrtc1_2Bpp_UnormSrgb },
    { VK_FORMAT_PVRTC1_4BPP_SRGB_BLOCK_IMG, ImageFormat_Pvrtc1_4Bpp_UnormSrgb },
    { VK_FORMAT_PVRTC2_2BPP_SRGB_BLOCK_IMG, ImageFormat_Pvrtc2_Alpha_2Bpp_UnormSrgb },
    { VK_FORMAT_PVRTC2_4BPP_SRGB_BLOCK_IMG, ImageFormat_Pvrtc2_Alpha_4Bpp_UnormSrgb }
};

const size_t MaxLayerProperties = 128;
VkLayerProperties g_LayerProperties[ MaxLayerProperties ];

const size_t MaxInstanceExtensionProperties = 128;
VkExtensionProperties g_InstanceExtensionProperties[ MaxInstanceExtensionProperties ];

const size_t MaxDeviceExtensionProperties = 128;
VkExtensionProperties g_DeviceExtensionProperties[ MaxDeviceExtensionProperties ];

uint32_t g_LayerCount = 0;
uint32_t g_InstanceExtensionCount = 0;
uint32_t g_DeviceExtensionCount = 0;

VkDynamicState g_ValidDynamicState[ GfxVkGpuState::dynamicStateItemCount ];
uint32_t g_ValidDynamicStateCount = 0;

double g_NanosecondsPerTick = 1.0;

bool g_LayoutManagementEnabled = true;

VkBool32 VKAPI_CALL DebugMessageCallback( VkDebugReportFlagsEXT flags, VkDebugReportObjectTypeEXT
    objectType, uint64_t object, size_t location, int32_t messageCode, const char* pLayerPrefix,
    const char* pMessage, void* pUserData )
{
    NN_UNUSED( pMessage );
    NN_UNUSED( pLayerPrefix );
    NN_UNUSED( messageCode );
    NN_UNUSED( location );
    NN_UNUSED( object );

    // Flags
    static const struct
    {
        VkDebugReportFlagsEXT flag;
        const char* string;
    } DebuCallbackFlags[] =
    {
        { VK_DEBUG_REPORT_INFORMATION_BIT_EXT, "Information" },
        { VK_DEBUG_REPORT_WARNING_BIT_EXT, "Warning" },
        { VK_DEBUG_REPORT_PERFORMANCE_WARNING_BIT_EXT, "Performance Warning" },
        { VK_DEBUG_REPORT_ERROR_BIT_EXT, "Error" },
        { VK_DEBUG_REPORT_DEBUG_BIT_EXT, "Debug" },
    };

    // ObjectType
    static const char* DebugCallbackObjectType[] =
    {
        "Unknown", // VK_DEBUG_REPORT_OBJECT_TYPE_UNKNOWN_EXT
        "Instance", // VK_DEBUG_REPORT_OBJECT_TYPE_INSTANCE_EXT
        "Physical Device", // VK_DEBUG_REPORT_OBJECT_TYPE_PHYSICAL_DEVICE_EXT
        "Device", // VK_DEBUG_REPORT_OBJECT_TYPE_DEVICE_EXT
        "Queue", // VK_DEBUG_REPORT_OBJECT_TYPE_QUEUE_EXT
        "Semaphore", // VK_DEBUG_REPORT_OBJECT_TYPE_SEMAPHORE_EXT
        "Command Buffer", // VK_DEBUG_REPORT_OBJECT_TYPE_COMMAND_BUFFER_EXT
        "Fence", // VK_DEBUG_REPORT_OBJECT_TYPE_FENCE_EXT
        "Device Memory", // VK_DEBUG_REPORT_OBJECT_TYPE_DEVICE_MEMORY_EXT
        "Buffer", // VK_DEBUG_REPORT_OBJECT_TYPE_BUFFER_EXT
        "Image", // VK_DEBUG_REPORT_OBJECT_TYPE_IMAGE_EXT
        "Event", // VK_DEBUG_REPORT_OBJECT_TYPE_EVENT_EXT
        "Query Pool", // VK_DEBUG_REPORT_OBJECT_TYPE_QUERY_POOL_EXT
        "Buffer View", // VK_DEBUG_REPORT_OBJECT_TYPE_BUFFER_VIEW_EXT
        "Image View", // VK_DEBUG_REPORT_OBJECT_TYPE_IMAGE_VIEW_EXT
        "Shader Module", // VK_DEBUG_REPORT_OBJECT_TYPE_SHADER_MODULE_EXT
        "Pipeline Cache", // VK_DEBUG_REPORT_OBJECT_TYPE_PIPELINE_CACHE_EXT
        "Pipeline Layout", // VK_DEBUG_REPORT_OBJECT_TYPE_PIPELINE_LAYOUT_EXT
        "Render Pass", // VK_DEBUG_REPORT_OBJECT_TYPE_RENDER_PASS_EXT
        "Pipeline", // VK_DEBUG_REPORT_OBJECT_TYPE_PIPELINE_EXT
        "Descriptor Set Layout", // VK_DEBUG_REPORT_OBJECT_TYPE_DESCRIPTOR_SET_LAYOUT_EXT
        "Sampler", // VK_DEBUG_REPORT_OBJECT_TYPE_SAMPLER_EXT
        "Descriptor Pool", // VK_DEBUG_REPORT_OBJECT_TYPE_DESCRIPTOR_POOL_EXT
        "Descripto Set", // VK_DEBUG_REPORT_OBJECT_TYPE_DESCRIPTOR_SET_EXT
        "Famebuffer", // VK_DEBUG_REPORT_OBJECT_TYPE_FRAMEBUFFER_EXT
        "CommandPool", // VK_DEBUG_REPORT_OBJECT_TYPE_COMMAND_POOL_EXT
        "Surface KHR", // VK_DEBUG_REPORT_OBJECT_TYPE_SURFACE_KHR_EXT
        "Swapchain KHR", // VK_DEBUG_REPORT_OBJECT_TYPE_SWAPCHAIN_KHR_EXT
        "Debug Repot", // VK_DEBUG_REPORT_OBJECT_TYPE_DEBUG_REPORT_EXT
        "Display KHR", // VK_DEBUG_REPORT_OBJECT_TYPE_DISPLAY_KHR_EXT
        "Display Mode KHR", // VK_DEBUG_REPORT_OBJECT_TYPE_DISPLAY_MODE_KHR_EXT
        "Object Talbe NVX", // VK_DEBUG_REPORT_OBJECT_TYPE_OBJECT_TABLE_NVX_EXT
        "Indirect Commands Layout NVX", // VK_DEBUG_REPORT_OBJECT_TYPE_INDIRECT_COMMANDS_LAYOUT_NVX_EXT
    };
    NN_STATIC_ASSERT( NN_ARRAY_SIZE( DebugCallbackObjectType ) <= VK_DEBUG_REPORT_OBJECT_TYPE_END_RANGE_EXT + 1 );
    NN_UNUSED( pUserData );

    const char* pObjectTypeString = ( objectType < VK_DEBUG_REPORT_OBJECT_TYPE_END_RANGE_EXT + 1 ) ?
        DebugCallbackObjectType[ objectType ] : "Undefined";
    NN_UNUSED( pObjectTypeString );

    if ( ( flags & VK_DEBUG_REPORT_ERROR_BIT_EXT ) != 0 )
    {
        NN_DETAIL_GFX_ERROR( "[Debug Callback] " );
        for ( int idxFlag = 0; idxFlag < NN_ARRAY_SIZE( DebuCallbackFlags ); ++idxFlag )
        {
            if ( 0 != ( flags & DebuCallbackFlags[ idxFlag ].flag ) )
            {
                NN_DETAIL_GFX_ERROR( "%s ", DebuCallbackFlags[ idxFlag ].string );
            }
        }

        NN_DETAIL_GFX_ERROR( "Object Type ( %s (%d) ) Object( %llx ) ( Location: %d ) ( Message Code: %d ) %s : %s\n",
            pObjectTypeString, objectType, object, location, messageCode, pLayerPrefix, pMessage );
    }
    else if ( ( flags & VK_DEBUG_REPORT_WARNING_BIT_EXT ) != 0 )
    {
        NN_DETAIL_GFX_WARN( "[Debug Callback] " );
        for ( int idxFlag = 0; idxFlag < NN_ARRAY_SIZE( DebuCallbackFlags ); ++idxFlag )
        {
            if ( 0 != ( flags & DebuCallbackFlags[ idxFlag ].flag ) )
            {
                NN_DETAIL_GFX_WARN( "%s ", DebuCallbackFlags[ idxFlag ].string );
            }
        }

        NN_DETAIL_GFX_WARN( "Object Type ( %s (%d) ) Object( %llx ) ( Location: %d ) ( Message Code: %d ) %s : %s\n",
            pObjectTypeString, objectType, object, location, messageCode, pLayerPrefix, pMessage );
    }
    else
    {
        NN_DETAIL_GFX_INFO( "[Debug Callback] " );
        for ( int idxFlag = 0; idxFlag < NN_ARRAY_SIZE( DebuCallbackFlags ); ++idxFlag )
        {
            if ( 0 != ( flags & DebuCallbackFlags[ idxFlag ].flag ) )
            {
                NN_DETAIL_GFX_INFO( "%s ", DebuCallbackFlags[ idxFlag ].string );
            }
        }

        NN_DETAIL_GFX_INFO( "Object Type ( %s (%d) ) Object( %llx ) ( Location: %d ) ( Message Code: %d ) %s : %s\n",
            pObjectTypeString, objectType, object, location, messageCode, pLayerPrefix, pMessage );
    }

    // Proceed as though validation passed.
    return VK_FALSE;
}//NOLINT(impl/function_size)

// TODO: 複数のDeviceオブジェクトをサポートするには
// 以下の変数を経由せずに、アロケータにアクセスできるようにする必要があります。
VkAllocator* g_pVkAllocator = NULL;

void* VKAPI_CALL GfxVkAllocationFunction( void* pUserData, size_t size, size_t alignment,
    VkSystemAllocationScope allocationScope )
{
    VkAllocator* pVkAllocator
        = static_cast< VkAllocator* >( pUserData );

    return pVkAllocator->pAllocate( pVkAllocator->pUserData, size, alignment,
        static_cast< AllocationScope >( allocationScope ) );
}

void* VKAPI_CALL GfxVkReallocationFunction( void* pUserData, void* pOriginal, size_t size, size_t alignment,
    VkSystemAllocationScope allocationScope )
{
    VkAllocator* pVkAllocator
        = static_cast< VkAllocator* >( pUserData );

    return pVkAllocator->pReallocate( pVkAllocator->pUserData, pOriginal, size, alignment,
        static_cast< AllocationScope >( allocationScope ) );
}

void VKAPI_CALL GfxVkFreeFunction( void* pUserData, void* pMemory )
{
    VkAllocator* pVkAllocator
        = static_cast< VkAllocator* >( pUserData );

    pVkAllocator->pFree( pVkAllocator->pUserData, pMemory );
}

void VKAPI_CALL GfxVkInternalAllocationNotification( void* pUserData, size_t size,
    VkInternalAllocationType allocationType, VkSystemAllocationScope allocationScope )
{
    VkAllocator* pVkAllocator
        = static_cast< VkAllocator* >( pUserData );

    pVkAllocator->pNotifyInternalAllocation( pVkAllocator->pUserData, size,
        static_cast< InternalAllocationType >( allocationType ),
        static_cast< AllocationScope >( allocationScope ) );
}

void VKAPI_CALL GfxVkInternalFreeNotification( void* pUserData, size_t size,
    VkInternalAllocationType allocationType, VkSystemAllocationScope allocationScope )
{
    VkAllocator* pVkAllocator
        = static_cast< VkAllocator* >( pUserData );

    pVkAllocator->pNotifyInternalFree( pVkAllocator->pUserData, size,
        static_cast< InternalAllocationType >( allocationType ),
        static_cast< AllocationScope >( allocationScope ) );
}

}

VkImageLayout Vk::LayoutColorAttachment = VK_IMAGE_LAYOUT_COLOR_ATTACHMENT_OPTIMAL;
VkImageLayout Vk::LayoutDepthStencilAttachment = VK_IMAGE_LAYOUT_DEPTH_STENCIL_ATTACHMENT_OPTIMAL;
VkImageLayout Vk::LayoutCopyImageSrc = VK_IMAGE_LAYOUT_TRANSFER_SRC_OPTIMAL;
VkImageLayout Vk::LayoutCopyImageDst = VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL;
VkImageLayout Vk::LayoutBlitImageSrc = VK_IMAGE_LAYOUT_TRANSFER_SRC_OPTIMAL;
VkImageLayout Vk::LayoutBlitImageDst = VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL;
VkImageLayout Vk::LayoutResolveImageSrc = VK_IMAGE_LAYOUT_TRANSFER_SRC_OPTIMAL;
VkImageLayout Vk::LayoutResolveImageDst = VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL;
VkImageLayout Vk::LayoutClearImage = VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL;
VkImageLayout Vk::LayoutTextureRead = VK_IMAGE_LAYOUT_SHADER_READ_ONLY_OPTIMAL;
VkImageLayout Vk::LayoutTextureDepthStencilRead = VK_IMAGE_LAYOUT_DEPTH_STENCIL_READ_ONLY_OPTIMAL;

void Vk::GetDeviceFeatures( nn::util::BitArray* pOutBits, const VkPhysicalDeviceFeatures& features ) NN_NOEXCEPT
{
    pOutBits->reset();

    pOutBits->set( VkDeviceFeature_RobustBufferAccess, features.robustBufferAccess == VK_TRUE );
    pOutBits->set( VkDeviceFeature_FullDrawIndexUint32, features.fullDrawIndexUint32 == VK_TRUE );
    pOutBits->set( VkDeviceFeature_ImageCubeArray, features.imageCubeArray == VK_TRUE );
    pOutBits->set( VkDeviceFeature_IndependentBlend, features.independentBlend == VK_TRUE );
    pOutBits->set( VkDeviceFeature_GeometryShader, features.geometryShader == VK_TRUE );
    pOutBits->set( VkDeviceFeature_TessellationShader, features.tessellationShader == VK_TRUE );
    pOutBits->set( VkDeviceFeature_SampleRateShading, features.sampleRateShading == VK_TRUE );
    pOutBits->set( VkDeviceFeature_DualSrcBlend, features.dualSrcBlend == VK_TRUE );
    pOutBits->set( VkDeviceFeature_LogicOp, features.logicOp == VK_TRUE );
    pOutBits->set( VkDeviceFeature_MultiDrawIndirect, features.multiDrawIndirect == VK_TRUE );
    pOutBits->set( VkDeviceFeature_DrawIndirectFirstInstance, features.drawIndirectFirstInstance == VK_TRUE );
    pOutBits->set( VkDeviceFeature_DepthClamp, features.depthClamp == VK_TRUE );
    pOutBits->set( VkDeviceFeature_DepthBiasClamp, features.depthBiasClamp == VK_TRUE );
    pOutBits->set( VkDeviceFeature_FillModeNonSolid, features.fillModeNonSolid == VK_TRUE );
    pOutBits->set( VkDeviceFeature_DepthBounds, features.depthBounds == VK_TRUE );
    pOutBits->set( VkDeviceFeature_WideLines, features.wideLines == VK_TRUE );
    pOutBits->set( VkDeviceFeature_LargePoints, features.largePoints == VK_TRUE );
    pOutBits->set( VkDeviceFeature_AlphaToOne, features.alphaToOne == VK_TRUE );
    pOutBits->set( VkDeviceFeature_MultiViewport, features.multiViewport == VK_TRUE );
    pOutBits->set( VkDeviceFeature_SamplerAnisotropy, features.samplerAnisotropy == VK_TRUE );
    pOutBits->set( VkDeviceFeature_TextureCompressionETC2, features.textureCompressionETC2 == VK_TRUE );
    pOutBits->set( VkDeviceFeature_TextureCompressionASTC_LDR, features.textureCompressionASTC_LDR == VK_TRUE );
    pOutBits->set( VkDeviceFeature_TextureCompressionBC, features.textureCompressionBC == VK_TRUE );
    pOutBits->set( VkDeviceFeature_OcclusionQueryPrecise, features.occlusionQueryPrecise == VK_TRUE );
    pOutBits->set( VkDeviceFeature_PipelineStatisticsQuery, features.pipelineStatisticsQuery == VK_TRUE );
    pOutBits->set( VkDeviceFeature_VertexPipelineStoresAndAtomics, features.vertexPipelineStoresAndAtomics == VK_TRUE );
    pOutBits->set( VkDeviceFeature_FragmentStoresAndAtomics, features.fragmentStoresAndAtomics == VK_TRUE );
    pOutBits->set( VkDeviceFeature_ShaderTessellationAndGeometryPointSize, features.shaderTessellationAndGeometryPointSize == VK_TRUE );
    pOutBits->set( VkDeviceFeature_ShaderImageGatherExtended, features.shaderImageGatherExtended == VK_TRUE );
    pOutBits->set( VkDeviceFeature_ShaderStorageImageExtendedFormats, features.shaderStorageImageExtendedFormats == VK_TRUE );
    pOutBits->set( VkDeviceFeature_ShaderStorageImageMultisample, features.shaderStorageImageMultisample == VK_TRUE );
    pOutBits->set( VkDeviceFeature_ShaderStorageImageReadWithoutFormat, features.shaderStorageImageReadWithoutFormat == VK_TRUE );
    pOutBits->set( VkDeviceFeature_ShaderStorageImageWriteWithoutFormat, features.shaderStorageImageWriteWithoutFormat == VK_TRUE );
    pOutBits->set( VkDeviceFeature_ShaderUniformBufferArrayDynamicIndexing, features.shaderUniformBufferArrayDynamicIndexing == VK_TRUE );
    pOutBits->set( VkDeviceFeature_ShaderSampledImageArrayDynamicIndexing, features.shaderSampledImageArrayDynamicIndexing == VK_TRUE );
    pOutBits->set( VkDeviceFeature_ShaderStorageBufferArrayDynamicIndexing, features.shaderStorageBufferArrayDynamicIndexing == VK_TRUE );
    pOutBits->set( VkDeviceFeature_ShaderStorageImageArrayDynamicIndexing, features.shaderStorageImageArrayDynamicIndexing == VK_TRUE );
    pOutBits->set( VkDeviceFeature_ShaderClipDistance, features.shaderClipDistance == VK_TRUE );
    pOutBits->set( VkDeviceFeature_ShaderCullDistance, features.shaderCullDistance == VK_TRUE );
    pOutBits->set( VkDeviceFeature_ShaderFloat64, features.shaderFloat64 == VK_TRUE );
    pOutBits->set( VkDeviceFeature_ShaderInt64, features.shaderInt64 == VK_TRUE );
    pOutBits->set( VkDeviceFeature_ShaderInt16, features.shaderInt16 == VK_TRUE );
    pOutBits->set( VkDeviceFeature_ShaderResourceResidency, features.shaderResourceResidency == VK_TRUE );
    pOutBits->set( VkDeviceFeature_ShaderResourceMinLod, features.shaderResourceMinLod == VK_TRUE );
    pOutBits->set( VkDeviceFeature_SparseBinding, features.sparseBinding == VK_TRUE );
    pOutBits->set( VkDeviceFeature_SparseResidencyBuffer, features.sparseResidencyBuffer == VK_TRUE );
    pOutBits->set( VkDeviceFeature_SparseResidencyImage2D, features.sparseResidencyImage2D == VK_TRUE );
    pOutBits->set( VkDeviceFeature_SparseResidencyImage3D, features.sparseResidencyImage3D == VK_TRUE );
    pOutBits->set( VkDeviceFeature_SparseResidency2Samples, features.sparseResidency2Samples == VK_TRUE );
    pOutBits->set( VkDeviceFeature_SparseResidency4Samples, features.sparseResidency4Samples == VK_TRUE );
    pOutBits->set( VkDeviceFeature_SparseResidency8Samples, features.sparseResidency8Samples == VK_TRUE );
    pOutBits->set( VkDeviceFeature_SparseResidency16Samples, features.sparseResidency16Samples == VK_TRUE );
    pOutBits->set( VkDeviceFeature_SparseResidencyAliased, features.sparseResidencyAliased == VK_TRUE );
    pOutBits->set( VkDeviceFeature_VariableMultisampleRate, features.variableMultisampleRate == VK_TRUE );
    pOutBits->set( VkDeviceFeature_InheritedQueries, features.inheritedQueries == VK_TRUE );
}

void Vk::SetupGlobalDynamicState( const VkPhysicalDeviceFeatures& features ) NN_NOEXCEPT
{
    g_ValidDynamicState[ g_ValidDynamicStateCount++ ] = VK_DYNAMIC_STATE_VIEWPORT;
    g_ValidDynamicState[ g_ValidDynamicStateCount++ ] = VK_DYNAMIC_STATE_SCISSOR;
    g_ValidDynamicState[ g_ValidDynamicStateCount++ ] = VK_DYNAMIC_STATE_DEPTH_BIAS;
    g_ValidDynamicState[ g_ValidDynamicStateCount++ ] = VK_DYNAMIC_STATE_BLEND_CONSTANTS;
    g_ValidDynamicState[ g_ValidDynamicStateCount++ ] = VK_DYNAMIC_STATE_STENCIL_COMPARE_MASK;
    g_ValidDynamicState[ g_ValidDynamicStateCount++ ] = VK_DYNAMIC_STATE_STENCIL_WRITE_MASK;
    g_ValidDynamicState[ g_ValidDynamicStateCount++ ] = VK_DYNAMIC_STATE_STENCIL_REFERENCE;

    if ( features.wideLines == VK_TRUE )
    {
        g_ValidDynamicState[ g_ValidDynamicStateCount++ ] = VK_DYNAMIC_STATE_LINE_WIDTH;
    }
    if ( features.depthBounds == VK_TRUE )
    {
        g_ValidDynamicState[ g_ValidDynamicStateCount++ ] = VK_DYNAMIC_STATE_DEPTH_BOUNDS;
    }
}

void Vk::GetDeviceExtensions( nn::util::BitPack32* pOutBits, int extensionCount, const char** ppExtensionNames ) NN_NOEXCEPT
{
    pOutBits->Clear();

    for ( int idxExtension = 0; idxExtension < extensionCount; ++idxExtension )
    {
        if ( !strcmp( ppExtensionNames[ idxExtension ], VK_KHR_MAINTENANCE1_EXTENSION_NAME ) )
        {
            pOutBits->SetBit( VkDeviceExtension_KhrMaintenance1, true );
            continue;
        }
        else if ( !strcmp( ppExtensionNames[ idxExtension ], VK_KHR_MAINTENANCE2_EXTENSION_NAME ) )
        {
            pOutBits->SetBit( VkDeviceExtension_KhrMaintenance2, true );
            continue;
        }
        else if ( !strcmp( ppExtensionNames[ idxExtension ], VK_NV_GLSL_SHADER_EXTENSION_NAME ) )
        {
            pOutBits->SetBit( VkDeviceExtension_NvGlslShader, true );
            continue;
        }
        else if ( !strcmp( ppExtensionNames[ idxExtension ], VK_NV_DEDICATED_ALLOCATION_EXTENSION_NAME ) )
        {
            pOutBits->SetBit( VkDeviceExtension_NvDedicatedAllocation, true );
            continue;
        }
    }
}

void Vk::SetTicksPerNanosecond( double ticksPerNanosecond ) NN_NOEXCEPT
{
    g_NanosecondsPerTick = ticksPerNanosecond;
}

VkFormat Vk::GetImageFormat( ImageFormat format ) NN_NOEXCEPT
{
    struct Comp
    {
        bool operator()( const ImageFormatAndVkFormat& lhs, ImageFormat rhs ) const
        {
            return lhs.imageFormat < rhs;
        }
    };
    const ImageFormatAndVkFormat* pEnd = VkTextureFormatList + NN_GFX_ARRAY_LENGTH( VkTextureFormatList );
    const ImageFormatAndVkFormat* pFound = std::lower_bound( VkTextureFormatList, pEnd, format, Comp() );
    if ( pFound == pEnd || format < pFound->imageFormat )
    {
        return VK_FORMAT_UNDEFINED;
    }
    else
    {
        return pFound->vkFormat;
    }
}

VkFormat Vk::GetAttributeFormat( AttributeFormat format ) NN_NOEXCEPT
{
    struct Comp
    {
        bool operator()( const AttributeFormatAndVkFormat& lhs, AttributeFormat rhs ) const
        {
            return lhs.attributeFormat < rhs;
        }
    };
    const AttributeFormatAndVkFormat* pEnd = VkAttributeFormatList + NN_GFX_ARRAY_LENGTH( VkAttributeFormatList );
    const AttributeFormatAndVkFormat* pFound = std::lower_bound( VkAttributeFormatList, pEnd, format, Comp() );
    if ( pFound == pEnd || format < pFound->attributeFormat )
    {
        return VK_FORMAT_UNDEFINED;
    }
    else
    {
        return pFound->vkFormat;
    }
}

VkImageViewType Vk::GetImageTarget( ImageDimension dimension ) NN_NOEXCEPT
{
    static const VkImageViewType s_TargetTable[] =
    {
        VK_IMAGE_VIEW_TYPE_1D,
        VK_IMAGE_VIEW_TYPE_2D,
        VK_IMAGE_VIEW_TYPE_3D,
        VK_IMAGE_VIEW_TYPE_CUBE,
        VK_IMAGE_VIEW_TYPE_1D_ARRAY,
        VK_IMAGE_VIEW_TYPE_2D_ARRAY,
        VK_IMAGE_VIEW_TYPE_2D,   // ImageDimension_2dMultisample
        VK_IMAGE_VIEW_TYPE_2D_ARRAY,   // ImageDimension_2dMultisampleArray
        VK_IMAGE_VIEW_TYPE_CUBE_ARRAY,
    };

    return s_TargetTable[ dimension ];
}

VkImageType Vk::GetImageType( ImageStorageDimension dimension ) NN_NOEXCEPT
{
    static const VkImageType s_TargetTable[] =
    {
        static_cast< VkImageType >( -1 ),
        VK_IMAGE_TYPE_1D,
        VK_IMAGE_TYPE_2D,
        VK_IMAGE_TYPE_3D,
    };

    return s_TargetTable[ dimension ];
}

VkCompareOp Vk::GetCompareOperation( ComparisonFunction compare ) NN_NOEXCEPT
{
    static const VkCompareOp DepthFunction[] =
    {
        VK_COMPARE_OP_NEVER,
        VK_COMPARE_OP_LESS,
        VK_COMPARE_OP_EQUAL,
        VK_COMPARE_OP_LESS_OR_EQUAL,
        VK_COMPARE_OP_GREATER,
        VK_COMPARE_OP_NOT_EQUAL,
        VK_COMPARE_OP_GREATER_OR_EQUAL,
        VK_COMPARE_OP_ALWAYS,
    };
    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( DepthFunction ) == ComparisonFunction_End );
    NN_SDK_ASSERT( compare < ComparisonFunction_End );

    return DepthFunction[ compare ];
}

VkStencilOp Vk::GetStencilOperation( StencilOperation operation ) NN_NOEXCEPT
{
    static const VkStencilOp s_StencilOperation[] =
    {
        VK_STENCIL_OP_KEEP,
        VK_STENCIL_OP_ZERO,
        VK_STENCIL_OP_REPLACE,
        VK_STENCIL_OP_INCREMENT_AND_CLAMP,
        VK_STENCIL_OP_DECREMENT_AND_CLAMP,
        VK_STENCIL_OP_INVERT,
        VK_STENCIL_OP_INCREMENT_AND_WRAP,
        VK_STENCIL_OP_DECREMENT_AND_WRAP,
    };

    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( s_StencilOperation ) == StencilOperation_End );
    NN_SDK_ASSERT( operation < StencilOperation_End );

    return s_StencilOperation[ operation ];
}

VkBlendOp Vk::GetBlendEquation( BlendFunction function ) NN_NOEXCEPT
{
    static const VkBlendOp s_BlendEquation[] =
    {
        VK_BLEND_OP_ADD,
        VK_BLEND_OP_SUBTRACT,
        VK_BLEND_OP_REVERSE_SUBTRACT,
        VK_BLEND_OP_MIN,
        VK_BLEND_OP_MAX,
    };
    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( s_BlendEquation ) == BlendFunction_End );
    NN_SDK_ASSERT( function < BlendFunction_End );

    return s_BlendEquation[ function ];
}

VkBlendFactor Vk::GetBlendFunction( BlendFactor factor ) NN_NOEXCEPT
{
    static const VkBlendFactor s_BlendFunction[] =
    {
        VK_BLEND_FACTOR_ZERO,
        VK_BLEND_FACTOR_ONE,
        VK_BLEND_FACTOR_SRC_COLOR,
        VK_BLEND_FACTOR_ONE_MINUS_SRC_COLOR,
        VK_BLEND_FACTOR_DST_COLOR,
        VK_BLEND_FACTOR_ONE_MINUS_DST_COLOR,
        VK_BLEND_FACTOR_SRC_ALPHA,
        VK_BLEND_FACTOR_ONE_MINUS_SRC_ALPHA,
        VK_BLEND_FACTOR_DST_ALPHA,
        VK_BLEND_FACTOR_ONE_MINUS_DST_ALPHA,
        VK_BLEND_FACTOR_CONSTANT_COLOR,
        VK_BLEND_FACTOR_ONE_MINUS_CONSTANT_COLOR,
        VK_BLEND_FACTOR_CONSTANT_ALPHA,
        VK_BLEND_FACTOR_ONE_MINUS_CONSTANT_ALPHA,
        VK_BLEND_FACTOR_SRC_ALPHA_SATURATE,
        VK_BLEND_FACTOR_SRC1_COLOR,
        VK_BLEND_FACTOR_ONE_MINUS_SRC1_COLOR,
        VK_BLEND_FACTOR_SRC1_ALPHA,
        VK_BLEND_FACTOR_ONE_MINUS_SRC1_ALPHA,
    };
    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( s_BlendFunction ) == BlendFactor_End );
    NN_SDK_ASSERT( factor < BlendFactor_End );

    return s_BlendFunction[ factor ];
}

VkLogicOp Vk::GetLogicOperation( LogicOperation operation ) NN_NOEXCEPT
{
    static const VkLogicOp s_LogicOperation[] =
    {
        VK_LOGIC_OP_CLEAR,
        VK_LOGIC_OP_AND,
        VK_LOGIC_OP_AND_REVERSE,
        VK_LOGIC_OP_COPY,
        VK_LOGIC_OP_AND_INVERTED,
        VK_LOGIC_OP_NO_OP,
        VK_LOGIC_OP_XOR,
        VK_LOGIC_OP_OR,
        VK_LOGIC_OP_NOR,
        VK_LOGIC_OP_EQUIVALENT,
        VK_LOGIC_OP_INVERT,
        VK_LOGIC_OP_OR_REVERSE,
        VK_LOGIC_OP_COPY_INVERTED,
        VK_LOGIC_OP_OR_INVERTED,
        VK_LOGIC_OP_NAND,
        VK_LOGIC_OP_SET,
    };
    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( s_LogicOperation ) == LogicOperation_End );
    NN_SDK_ASSERT( operation < LogicOperation_End );

    return s_LogicOperation[ operation ];
}

VkFrontFace Vk::GetFrontFace( FrontFace face ) NN_NOEXCEPT
{
    static const VkFrontFace s_FrontFace[] =
    {
        VK_FRONT_FACE_COUNTER_CLOCKWISE,
        VK_FRONT_FACE_CLOCKWISE,
    };
    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( s_FrontFace ) == FrontFace_End );
    NN_SDK_ASSERT( face < FrontFace_End );

    return s_FrontFace[ face ];
}

VkPolygonMode Vk::GetFillMode( FillMode mode ) NN_NOEXCEPT
{
    static const VkPolygonMode s_FillMode[] =
    {
        VK_POLYGON_MODE_POINT,
        VK_POLYGON_MODE_LINE,
        VK_POLYGON_MODE_FILL,
    };
    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( s_FillMode ) == FillMode_End );
    NN_SDK_ASSERT( mode < FillMode_End );

    return s_FillMode[ mode ];
}

VkPrimitiveTopology Vk::GetDrawPrimitive( PrimitiveTopology topology ) NN_NOEXCEPT
{
    static const VkPrimitiveTopology s_DrawPrimitive[] =
    {
        VK_PRIMITIVE_TOPOLOGY_POINT_LIST,
        VK_PRIMITIVE_TOPOLOGY_LINE_LIST,
        VK_PRIMITIVE_TOPOLOGY_LINE_STRIP,
        VK_PRIMITIVE_TOPOLOGY_TRIANGLE_LIST,
        VK_PRIMITIVE_TOPOLOGY_TRIANGLE_STRIP,
        VK_PRIMITIVE_TOPOLOGY_LINE_LIST_WITH_ADJACENCY,
        VK_PRIMITIVE_TOPOLOGY_LINE_STRIP_WITH_ADJACENCY,
        VK_PRIMITIVE_TOPOLOGY_TRIANGLE_LIST_WITH_ADJACENCY,
        VK_PRIMITIVE_TOPOLOGY_TRIANGLE_STRIP_WITH_ADJACENCY,
        VK_PRIMITIVE_TOPOLOGY_PATCH_LIST
    };
    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( s_DrawPrimitive ) == PrimitiveTopology_End );
    NN_SDK_ASSERT( topology < PrimitiveTopology_End );

    return s_DrawPrimitive[ topology ];
}

VkFilter Vk::GetMinFilter( FilterMode filterMode ) NN_NOEXCEPT
{
    switch ( ( filterMode & FilterModeBit_MinFilterMask ) >> FilterModeBit_MinFilterShift )
    {
    case FilterModeBit_Point:
        return VK_FILTER_NEAREST;
    case FilterModeBit_Linear:
        return VK_FILTER_LINEAR;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

VkFilter Vk::GetMagFilter( FilterMode filterMode ) NN_NOEXCEPT
{
    switch ( ( filterMode & FilterModeBit_MagFilterMask ) >> FilterModeBit_MagFilterShift )
    {
    case FilterModeBit_Point:
        return VK_FILTER_NEAREST;
    case FilterModeBit_Linear:
        return VK_FILTER_LINEAR;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

VkSamplerMipmapMode Vk::GetMipmapMode( FilterMode filterMode ) NN_NOEXCEPT
{
    switch ( ( filterMode & FilterModeBit_MipFilterMask ) >> FilterModeBit_MipFilterShift )
    {
    case FilterModeBit_Point:
        return VK_SAMPLER_MIPMAP_MODE_NEAREST;
    case FilterModeBit_Linear:
        return VK_SAMPLER_MIPMAP_MODE_LINEAR;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

bool Vk::IsMipMapUsed( FilterMode filterMode ) NN_NOEXCEPT
{
    switch ( ( filterMode & FilterModeBit_MipFilterMask ) >> FilterModeBit_MipFilterShift )
    {
    case FilterModeBit_Point:
    case FilterModeBit_Linear:
        return true;
    default:
        return false;
    }
}

VkSamplerAddressMode Vk::GetWrapMode( TextureAddressMode wrapMode ) NN_NOEXCEPT
{
    static const VkSamplerAddressMode WrapMode[] =
    {
        VK_SAMPLER_ADDRESS_MODE_REPEAT,
        VK_SAMPLER_ADDRESS_MODE_MIRRORED_REPEAT,
        VK_SAMPLER_ADDRESS_MODE_CLAMP_TO_EDGE,
        VK_SAMPLER_ADDRESS_MODE_CLAMP_TO_BORDER,
        VK_SAMPLER_ADDRESS_MODE_MIRROR_CLAMP_TO_EDGE,
    };
    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( WrapMode ) == TextureAddressMode_End );
    NN_SDK_ASSERT( wrapMode < TextureAddressMode_End );

    return WrapMode[ wrapMode ];
}

VkShaderStageFlagBits Vk::GetShaderStage( ShaderStage stage ) NN_NOEXCEPT
{
    static const VkShaderStageFlagBits s_Stage[] =
    {
        VK_SHADER_STAGE_VERTEX_BIT,
        VK_SHADER_STAGE_TESSELLATION_CONTROL_BIT,    /* hull shader */
        VK_SHADER_STAGE_TESSELLATION_EVALUATION_BIT, /* domain shader */
        VK_SHADER_STAGE_GEOMETRY_BIT,
        VK_SHADER_STAGE_FRAGMENT_BIT,
        VK_SHADER_STAGE_COMPUTE_BIT
    };
    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( s_Stage ) == ShaderStage_End );
    NN_SDK_ASSERT( stage < ShaderStage_End );

    return s_Stage[ stage ];
}

int Vk::GetShaderStageBits( int shaderStageBits ) NN_NOEXCEPT
{
    static const VkShaderStageFlagBits s_ShaderStageBitTable[] =
    {
        VK_SHADER_STAGE_VERTEX_BIT,
        VK_SHADER_STAGE_TESSELLATION_CONTROL_BIT,
        VK_SHADER_STAGE_TESSELLATION_EVALUATION_BIT,
        VK_SHADER_STAGE_GEOMETRY_BIT,
        VK_SHADER_STAGE_FRAGMENT_BIT,
        VK_SHADER_STAGE_COMPUTE_BIT
    };
    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( s_ShaderStageBitTable ) == ShaderStage_End );

    int ret = 0;
    for ( int idxStage = 0, mask = 0x01; idxStage < ShaderStage_End; ++idxStage, mask <<= 1 )
    {
        if ( shaderStageBits & mask )
        {
            ret |= s_ShaderStageBitTable[ idxStage ];
        }
    }
    return ret;
}

VkCompareOp Vk::GetRComparisonFunction( ComparisonFunction compare ) NN_NOEXCEPT
{
    static const VkCompareOp s_ComparisonFunction[] =
    {
        VK_COMPARE_OP_NEVER,
        VK_COMPARE_OP_LESS,
        VK_COMPARE_OP_EQUAL,
        VK_COMPARE_OP_LESS_OR_EQUAL,
        VK_COMPARE_OP_GREATER,
        VK_COMPARE_OP_NOT_EQUAL,
        VK_COMPARE_OP_GREATER_OR_EQUAL,
        VK_COMPARE_OP_ALWAYS,
    };
    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( s_ComparisonFunction ) == ComparisonFunction_End );
    NN_SDK_ASSERT( compare < ComparisonFunction_End );

    return s_ComparisonFunction[ compare ];
}

VkIndexType Vk::GetIndexFormat( IndexFormat indexFormat ) NN_NOEXCEPT
{
    static const VkIndexType s_IndexTable[] =
    {
        static_cast< VkIndexType >( -1 ),
        VK_INDEX_TYPE_UINT16,
        VK_INDEX_TYPE_UINT32,
    };
    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( s_IndexTable ) == IndexFormat_End );
    NN_SDK_ASSERT( indexFormat < IndexFormat_End );
    NN_SDK_ASSERT( s_IndexTable[ indexFormat ] != static_cast< VkIndexType >( -1 ) );

    return s_IndexTable[ indexFormat ];
}

VkCullModeFlags Vk::GetCullMode( CullMode mode ) NN_NOEXCEPT
{
    static const VkCullModeFlags s_Face[] =
    {
        VK_CULL_MODE_NONE,
        VK_CULL_MODE_FRONT_BIT,
        VK_CULL_MODE_BACK_BIT,
    };
    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( s_Face ) == CullMode_End );
    NN_SDK_ASSERT( mode < CullMode_End );

    return s_Face[ mode ];
}

int Vk::GetMemoryPoolFlags( int memoryPoolProperty ) NN_NOEXCEPT
{
    // vk と同じ定義値
    return memoryPoolProperty;
}

VkDeviceMemory* Vk::GetBufferAddress( const GpuAddress gpuAddress ) NN_NOEXCEPT
{
    return reinterpret_cast< VkDeviceMemory * >( gpuAddress.ToData()->value );
}

void Vk::SetupScanBufferTextureInfo( TextureInfo* pOutInfo, const SwapChainInfo& info ) NN_NOEXCEPT
{
    pOutInfo->SetImageStorageDimension( nn::gfx::ImageStorageDimension_2d );
    pOutInfo->SetTileMode( nn::gfx::TileMode_Optimal );
    pOutInfo->SetSwizzle( 0 );
    pOutInfo->SetMipCount( 1 );
    pOutInfo->SetMultiSampleCount( 1 );
    pOutInfo->SetImageFormat( info.GetFormat() );
    pOutInfo->SetGpuAccessFlags( nn::gfx::GpuAccess_ColorBuffer | nn::gfx::GpuAccess_ScanBuffer );
    pOutInfo->SetWidth( info.GetWidth() );
    pOutInfo->SetHeight( info.GetHeight() );
    pOutInfo->SetDepth( 1 );
    pOutInfo->SetArrayLength( 1 );
}

void Vk::GetImageFormatProperty( ImageFormatProperty* pOutImageFormatProperty,
    DeviceImpl< ApiVariationVk1 >* pDevice, VkFormat vkFormat ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pOutImageFormatProperty );

    VkPhysicalDevice physicalDevice = CastToVkDispatchableObject< VkPhysicalDevice >(
        pDevice->ToData()->hPhysicalDevice );

    VkFormatProperties formatProperties;
    NN_GFX_CALL_VK_FUNCTION( vkGetPhysicalDeviceFormatProperties(
        physicalDevice, vkFormat, &formatProperties ) );

    pOutImageFormatProperty->propertyFlags = 0;

    if ( formatProperties.linearTilingFeatures & VK_FORMAT_FEATURE_SAMPLED_IMAGE_BIT ||
        formatProperties.optimalTilingFeatures & VK_FORMAT_FEATURE_SAMPLED_IMAGE_BIT )
    {
        pOutImageFormatProperty->propertyFlags |= ImageFormatPropertyFlag_Texture;
    }

    if ( formatProperties.linearTilingFeatures & VK_FORMAT_FEATURE_COLOR_ATTACHMENT_BIT ||
        formatProperties.optimalTilingFeatures & VK_FORMAT_FEATURE_COLOR_ATTACHMENT_BIT )
    {
        pOutImageFormatProperty->propertyFlags |= ImageFormatPropertyFlag_ColorTarget;
    }

    if ( formatProperties.linearTilingFeatures & VK_FORMAT_FEATURE_STORAGE_IMAGE_BIT ||
        formatProperties.optimalTilingFeatures & VK_FORMAT_FEATURE_STORAGE_IMAGE_BIT )
    {
        pOutImageFormatProperty->propertyFlags |= ImageFormatPropertyFlag_Image;
    }
}

ImageFormat Vk::GetGfxImageFormat( VkFormat vkFormat ) NN_NOEXCEPT
{
    struct Comp
    {
        bool operator()( const VkFormatAndImageFormat& lhs, VkFormat rhs ) const
        {
            return lhs.vkFormat < rhs;
        }
    };
    const VkFormatAndImageFormat* pEnd = VkFormatAndImageFormatList + NN_GFX_ARRAY_LENGTH( VkFormatAndImageFormatList );
    const VkFormatAndImageFormat* pFound = std::lower_bound( VkFormatAndImageFormatList, pEnd, vkFormat, Comp() );
    if ( pFound == pEnd || vkFormat < pFound->vkFormat )
    {
        return ImageFormat_Undefined;
    }
    else
    {
        return pFound->imageFormat;
    }

}

VkBool32 Vk::IsDepthFormat( VkFormat vkFormat ) NN_NOEXCEPT
{
    VkBool32 isDepth;

    switch ( vkFormat )
    {
    case VK_FORMAT_D16_UNORM:
    case VK_FORMAT_D16_UNORM_S8_UINT:
    case VK_FORMAT_D24_UNORM_S8_UINT:
    case VK_FORMAT_D32_SFLOAT:
    case VK_FORMAT_D32_SFLOAT_S8_UINT:
        {
            isDepth = VK_TRUE;
        }
        break;
    default:
        {
            isDepth = VK_FALSE;
        }
        break;
    }

    return isDepth;
}

VkBool32 Vk::IsDepthOnlyFormat( VkFormat vkFormat ) NN_NOEXCEPT
{
    VkBool32 isDepthOnly;

    switch ( vkFormat )
    {
    case VK_FORMAT_D16_UNORM:
    case VK_FORMAT_D32_SFLOAT:
        {
            isDepthOnly = VK_TRUE;
        }
        break;
    default:
        {
            isDepthOnly = VK_FALSE;
        }
        break;
    }

    return isDepthOnly;
}

VkBool32 Vk::IsStencilOnlyFormat( VkFormat vkFormat ) NN_NOEXCEPT
{
    VkBool32 isStencilOnly;

    switch ( vkFormat )
    {
    case VK_FORMAT_S8_UINT:
        {
            isStencilOnly = VK_TRUE;
        }
        break;
    default:
        {
            isStencilOnly = VK_FALSE;
        }
        break;
    }

    return isStencilOnly;
}

VkImageAspectFlags Vk::GetImageAspect( ImageFormat imageFormat ) NN_NOEXCEPT
{
    VkFormat vkFormat = GetImageFormat( imageFormat );
    VkImageAspectFlags flags = 0;

    switch ( vkFormat )
    {
    case VK_FORMAT_D16_UNORM:
    case VK_FORMAT_D32_SFLOAT:
        {
            flags = VK_IMAGE_ASPECT_DEPTH_BIT;
        }
        break;
    case VK_FORMAT_D16_UNORM_S8_UINT:
    case VK_FORMAT_D24_UNORM_S8_UINT:
    case VK_FORMAT_D32_SFLOAT_S8_UINT:
        {
           flags = VK_IMAGE_ASPECT_DEPTH_BIT | VK_IMAGE_ASPECT_STENCIL_BIT;
        }
        break;
    default:
        {
            flags = VK_IMAGE_ASPECT_COLOR_BIT;
        }
        break;
    }
    return flags;
}

VkBool32 Vk::IsColor( VkImageAspectFlags flags ) NN_NOEXCEPT
{
    return ( flags & VK_IMAGE_ASPECT_COLOR_BIT ) == VK_IMAGE_ASPECT_COLOR_BIT;
}

VkBool32 Vk::IsDepth( VkImageAspectFlags flags ) NN_NOEXCEPT
{
    return ( flags & VK_IMAGE_ASPECT_DEPTH_BIT ) == VK_IMAGE_ASPECT_DEPTH_BIT;
}

VkBool32 Vk::IsStencil( VkImageAspectFlags flags ) NN_NOEXCEPT
{
    return ( flags & VK_IMAGE_ASPECT_STENCIL_BIT ) == VK_IMAGE_ASPECT_STENCIL_BIT;
}

VkBufferUsageFlags Vk::GetBufferUsageFlags( int gpuAccessFlags ) NN_NOEXCEPT
{
    // FIXME - This routine is incomplete
    static const VkBufferUsageFlags BufferUsageBitTable[] =
    {
        VK_BUFFER_USAGE_TRANSFER_SRC_BIT,  // GpuAccess_Read = 0x01,
        VK_BUFFER_USAGE_TRANSFER_DST_BIT,  // GpuAccess_Write = 0x02,
        VK_BUFFER_USAGE_VERTEX_BUFFER_BIT,  // GpuAccess_VertexBuffer = 0x04,
        VK_BUFFER_USAGE_INDEX_BUFFER_BIT,  // GpuAccess_IndexBuffer = 0x08,
        VK_BUFFER_USAGE_UNIFORM_BUFFER_BIT,  // GpuAccess_ConstantBuffer = 0x10,
        VK_BUFFER_USAGE_STORAGE_TEXEL_BUFFER_BIT,  // GpuAccess_Texture = 0x20,
        VK_BUFFER_USAGE_STORAGE_BUFFER_BIT,  // GpuAccess_UnorderedAccessBuffer = 0x40,
        VK_BUFFER_USAGE_STORAGE_BUFFER_BIT,  // GpuAccess_ColorBuffer = 0x80,
        VK_BUFFER_USAGE_STORAGE_BUFFER_BIT,  // GpuAccess_DepthStencil = 0x0100,
        VK_BUFFER_USAGE_INDIRECT_BUFFER_BIT,  // GpuAccess_IndirectBuffer = 0x0200,
        VK_BUFFER_USAGE_STORAGE_BUFFER_BIT,  // GpuAccess_ScanBuffer = 0x0400,
        VK_BUFFER_USAGE_TRANSFER_SRC_BIT | VK_BUFFER_USAGE_TRANSFER_DST_BIT,  // GpuAccess_QueryBuffer = 0x0800,
        0,  // GpuAccess_Descriptor = 0x1000,
        0,  // GpuAccess_ShaderCode = 0x2000,
        0,  // GpuAccess_Image = 0x4000
    };
    NN_SDK_ASSERT( gpuAccessFlags < GpuAccess_Image );

    VkBufferUsageFlags ret = 0;
    uint32_t idx, mask;
    for ( idx = 0, mask = GpuAccess_Read; mask <= GpuAccess_Image; ++idx, mask <<= 1 )
    {
        if ( gpuAccessFlags & mask )
        {
            if ( mask == GpuAccess_Texture )
            {
                if ( gpuAccessFlags & GpuAccess_Write )
                {
                    ret |= VK_BUFFER_USAGE_STORAGE_TEXEL_BUFFER_BIT;
                }
                else
                {
                    ret |= VK_BUFFER_USAGE_UNIFORM_TEXEL_BUFFER_BIT;
                }
            }
            else
            {
                ret |= BufferUsageBitTable[ idx ];
            }
        }
    }
    return ret;

}

VkImageUsageFlags Vk::GetImageUsageFlags( int gpuAccessFlags ) NN_NOEXCEPT
{
    // FIXME - This routine is incomplete
    static const VkImageUsageFlags s_ImageUsageBitTable[] =
    {
        VK_IMAGE_USAGE_TRANSFER_SRC_BIT,  // GpuAccess_Read = 0x01,
        VK_IMAGE_USAGE_TRANSFER_DST_BIT,  // GpuAccess_Write = 0x02,
        VK_IMAGE_USAGE_INPUT_ATTACHMENT_BIT,  // GpuAccess_VertexBuffer = 0x04,
        VK_IMAGE_USAGE_INPUT_ATTACHMENT_BIT,  // GpuAccess_IndexBuffer = 0x08,
        VK_IMAGE_USAGE_INPUT_ATTACHMENT_BIT,  // GpuAccess_ConstantBuffer = 0x10,
        VK_IMAGE_USAGE_SAMPLED_BIT,  // GpuAccess_Texture = 0x20,
        VK_IMAGE_USAGE_STORAGE_BIT,  // GpuAccess_UnorderedAccessBuffer = 0x40,
        VK_IMAGE_USAGE_COLOR_ATTACHMENT_BIT,  // GpuAccess_ColorBuffer = 0x80,
        VK_IMAGE_USAGE_DEPTH_STENCIL_ATTACHMENT_BIT,  // GpuAccess_DepthStencil = 0x0100,
        VK_IMAGE_USAGE_INPUT_ATTACHMENT_BIT,  // GpuAccess_IndirectBuffer = 0x0200,
        VK_IMAGE_USAGE_COLOR_ATTACHMENT_BIT,  // GpuAccess_ScanBuffer = 0x0400,
        VK_IMAGE_USAGE_INPUT_ATTACHMENT_BIT,  // GpuAccess_QueryBuffer = 0x0800,
        VK_IMAGE_USAGE_INPUT_ATTACHMENT_BIT,  // GpuAccess_Descriptor = 0x1000,
        VK_IMAGE_USAGE_INPUT_ATTACHMENT_BIT,  // GpuAccess_ShaderCode = 0x2000,
        VK_IMAGE_USAGE_INPUT_ATTACHMENT_BIT,  // GpuAccess_Image = 0x4000
    };
    NN_SDK_ASSERT( gpuAccessFlags < ( GpuAccess_Image << 1 ) );

    VkBufferUsageFlags ret = 0;
    uint32_t idx, mask;
    for ( idx = 0, mask = GpuAccess_Read; mask <= GpuAccess_Image; ++idx, mask <<= 1 )
    {
        if ( gpuAccessFlags & mask )
        {
            ret |= s_ImageUsageBitTable[ idx ];
        }
    }

    return ret;
}

VkFormatFeatureFlags Vk::GetFormatFeatureFlagsFromGpuFlags( int gpuAccessFlags ) NN_NOEXCEPT
{
    VkFormatFeatureFlags ret = 0;

    // FIXME: We need to add support for VK_KHR_maintenance1
    if ( ( GpuAccess_Read & gpuAccessFlags ) == GpuAccess_Read )
    {
        if ( !( GpuAccess_DepthStencil & gpuAccessFlags ) )
        {
            ret |= VK_FORMAT_FEATURE_BLIT_SRC_BIT;
        }
    }

    if ( ( GpuAccess_Write & gpuAccessFlags ) == GpuAccess_Write )
    {
        if ( !( GpuAccess_DepthStencil & gpuAccessFlags ) )
        {
            ret |= VK_FORMAT_FEATURE_BLIT_DST_BIT;
        }
    }

    if ( ( GpuAccess_VertexBuffer & gpuAccessFlags ) == GpuAccess_VertexBuffer )
    {
        ret |= VK_FORMAT_FEATURE_VERTEX_BUFFER_BIT;
    }

    if ( ( GpuAccess_Texture & gpuAccessFlags ) == GpuAccess_Texture )
    {
        ret |= VK_FORMAT_FEATURE_SAMPLED_IMAGE_BIT;
    }

    if ( ( GpuAccess_ColorBuffer & gpuAccessFlags ) == GpuAccess_ColorBuffer )
    {
        //ret |= VK_FORMAT_FEATURE_COLOR_ATTACHMENT_BLEND_BIT | VK_FORMAT_FEATURE_COLOR_ATTACHMENT_BIT;
        // BlendができないFormatもあるため、必須フラグからは外します。
        ret |= VK_FORMAT_FEATURE_COLOR_ATTACHMENT_BIT;
    }

    if ( ( GpuAccess_DepthStencil & gpuAccessFlags ) == GpuAccess_DepthStencil )
    {
        ret |= VK_FORMAT_FEATURE_DEPTH_STENCIL_ATTACHMENT_BIT;
    }

    if ( ( GpuAccess_ScanBuffer & gpuAccessFlags ) == GpuAccess_ScanBuffer )
    {
        ret |= VK_FORMAT_FEATURE_BLIT_SRC_BIT | VK_FORMAT_FEATURE_BLIT_DST_BIT;
    }

    if ( ( GpuAccess_Image & gpuAccessFlags ) == GpuAccess_Image )
    {
        ret |= VK_FORMAT_FEATURE_STORAGE_IMAGE_BIT;
        // 下記featureが未サポートのformatでImageを使う場合があるため、とりあえず無効にしています。
        //ret |= VK_FORMAT_FEATURE_STORAGE_IMAGE_ATOMIC_BIT; // FIXME: We don't always need this...
    }

    return ret;
}

VkAccessFlags Vk::GetBufferStateAccessFlags( int bufferState ) NN_NOEXCEPT
{
    static const VkAccessFlags s_AccessFlagBitTable[] =
    {
        VK_ACCESS_TRANSFER_READ_BIT |
            VK_ACCESS_TRANSFER_WRITE_BIT, // BufferState_DataTransfer
        VK_ACCESS_TRANSFER_READ_BIT, // BufferState_CopySource
        VK_ACCESS_TRANSFER_WRITE_BIT, // BufferState_CopyDestination
        VK_ACCESS_VERTEX_ATTRIBUTE_READ_BIT, // BufferState_VertexBuffer
        VK_ACCESS_INDEX_READ_BIT, // BufferState_IndexBuffer
        VK_ACCESS_UNIFORM_READ_BIT, // BufferState_ConstantBuffer
        VK_ACCESS_SHADER_READ_BIT | VK_ACCESS_SHADER_WRITE_BIT, // BufferState_UnorderedAccessBuffer
        VK_ACCESS_INDIRECT_COMMAND_READ_BIT, // BufferState_IndirectArgument
        VK_ACCESS_HOST_READ_BIT, // BufferState_QueryBuffer
    };
    NN_SDK_ASSERT( bufferState <= BufferState_QueryBuffer );

    VkAccessFlags flags = 0;
    uint32_t idxTable, mask;
    for ( idxTable = 0, mask = BufferState_DataTransfer; mask <= BufferState_QueryBuffer; ++idxTable, mask <<= 1 )
    {
        if ( ( bufferState & 1 ) == 1 )
        {
            flags |= s_AccessFlagBitTable[ idxTable ];
        }
        bufferState >>= 1;
    }

    return flags;
}

VkAccessFlags Vk::GetGpuAccessFlags( int gpuAccessFlags ) NN_NOEXCEPT
{
    static const VkAccessFlags s_AccessFlagBitTable[] =
    {
        VK_ACCESS_MEMORY_READ_BIT, // GpuAccess_Read
        VK_ACCESS_MEMORY_WRITE_BIT, // GpuAccess_Write
        VK_ACCESS_VERTEX_ATTRIBUTE_READ_BIT, // GpuAccess_VertexBuffer
        VK_ACCESS_INDEX_READ_BIT, // GpuAccess_IndexBuffer
        VK_ACCESS_UNIFORM_READ_BIT, // GpuAccess_ConstantBuffer
        VK_ACCESS_SHADER_READ_BIT, // GpuAccess_Texture
        VK_ACCESS_SHADER_READ_BIT | VK_ACCESS_SHADER_WRITE_BIT, // GpuAccess_UnorderedAccessBuffer
        VK_ACCESS_COLOR_ATTACHMENT_READ_BIT | VK_ACCESS_COLOR_ATTACHMENT_WRITE_BIT,// GpuAccess_ColorBuffer
        VK_ACCESS_DEPTH_STENCIL_ATTACHMENT_READ_BIT | VK_ACCESS_DEPTH_STENCIL_ATTACHMENT_WRITE_BIT, // GpuAccess_DepthStencil
        VK_ACCESS_INDIRECT_COMMAND_READ_BIT, // GpuAccess_IndirectBuffer
        VK_ACCESS_MEMORY_READ_BIT | VK_ACCESS_MEMORY_WRITE_BIT, // GpuAccess_ScanBuffer
        0, // GpuAccess_QueryBuffer
        0, // GpuAccess_Descriptor
        VK_ACCESS_SHADER_READ_BIT, // GpuAccess_ShaderCode
        VK_ACCESS_SHADER_READ_BIT | VK_ACCESS_SHADER_WRITE_BIT, // GpuAccess_Image
    };
    NN_SDK_ASSERT( gpuAccessFlags <= GpuAccess_Image );

    VkAccessFlags flags = 0;
    uint32_t idxTable, mask;
    for ( idxTable = 0, mask = GpuAccess_Read; mask <= GpuAccess_Image; ++idxTable, mask <<= 1 )
    {
        if ( ( gpuAccessFlags & 1 ) == 1 )
        {
            flags |= s_AccessFlagBitTable[ idxTable ];
        }
        gpuAccessFlags >>= 1;
    }

    return flags;
}

VkAccessFlags Vk::GetTextureStateAccessFlags( int textureState ) NN_NOEXCEPT
{
    static const VkAccessFlags s_AccessFlagBitTable[] =
    {
        VK_ACCESS_TRANSFER_READ_BIT |
            VK_ACCESS_TRANSFER_WRITE_BIT,  // TextureState_DataTransfer = 0x01,
        VK_ACCESS_TRANSFER_READ_BIT,  // TextureState_CopySource = 0x02,
        VK_ACCESS_TRANSFER_WRITE_BIT,  // TextureState_CopyDestination = 0x04,
        VK_ACCESS_SHADER_READ_BIT,  // TextureState_ShaderRead = 0x08,
        VK_ACCESS_SHADER_WRITE_BIT,  // TextureState_ShaderWrite = 0x10,
        VK_ACCESS_COLOR_ATTACHMENT_READ_BIT |
            VK_ACCESS_COLOR_ATTACHMENT_WRITE_BIT,  // TextureState_ColorTarget = 0x20,
        VK_ACCESS_DEPTH_STENCIL_ATTACHMENT_READ_BIT,  // TextureState_DepthRead = 0x40,
        VK_ACCESS_DEPTH_STENCIL_ATTACHMENT_WRITE_BIT,  // TextureState_DepthWrite = 0x80,
        VK_ACCESS_TRANSFER_WRITE_BIT,  // TextureState_Clear = 0x0100,
        VK_ACCESS_TRANSFER_READ_BIT,  // TextureState_ResolveSource = 0x0200,
        VK_ACCESS_TRANSFER_WRITE_BIT,  // TextureState_ResolveDestination = 0x0400,
        VK_ACCESS_MEMORY_READ_BIT,  // TextureState_Scanbuffer = 0x800,
    };
    NN_SDK_ASSERT( textureState <= TextureState_Present );

    VkAccessFlags flags = 0;

    for ( uint32_t idxTable = 0, mask = TextureState_DataTransfer; mask <= TextureState_Present; ++idxTable, mask <<= 1 )
    {
        if ( ( textureState & 1 ) == 1 )
        {
            flags |= s_AccessFlagBitTable[ idxTable ];
        }
        textureState >>= 1;
    }

    return flags;
}

VkAccessFlags Vk::GetLayoutAccessFlags( VkImageLayout layout ) NN_NOEXCEPT
{
    VkAccessFlags flags = 0;

    switch ( layout )
    {
    case VK_IMAGE_LAYOUT_GENERAL:
        flags = VK_ACCESS_MEMORY_READ_BIT | VK_ACCESS_MEMORY_WRITE_BIT;
        break;
    case VK_IMAGE_LAYOUT_COLOR_ATTACHMENT_OPTIMAL:
        flags = VK_ACCESS_COLOR_ATTACHMENT_WRITE_BIT | VK_ACCESS_COLOR_ATTACHMENT_READ_BIT;
        break;
    case VK_IMAGE_LAYOUT_DEPTH_STENCIL_ATTACHMENT_OPTIMAL:
        flags = VK_ACCESS_DEPTH_STENCIL_ATTACHMENT_WRITE_BIT | VK_ACCESS_DEPTH_STENCIL_ATTACHMENT_READ_BIT;
        break;
    case VK_IMAGE_LAYOUT_DEPTH_STENCIL_READ_ONLY_OPTIMAL:
        flags = VK_ACCESS_DEPTH_STENCIL_ATTACHMENT_READ_BIT;
        break;
    case VK_IMAGE_LAYOUT_SHADER_READ_ONLY_OPTIMAL:
        flags = VK_ACCESS_SHADER_READ_BIT;
        break;
    case VK_IMAGE_LAYOUT_TRANSFER_SRC_OPTIMAL:
        flags = VK_ACCESS_TRANSFER_READ_BIT;
        break;
    case VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL:
        flags = VK_ACCESS_TRANSFER_WRITE_BIT;
        break;
    case VK_IMAGE_LAYOUT_PRESENT_SRC_KHR:
        flags = VK_ACCESS_MEMORY_READ_BIT;
        break;
    case VK_IMAGE_LAYOUT_UNDEFINED: break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    return flags;
}
VkImageLayout Vk::GetImageLayout( int textureState ) NN_NOEXCEPT
{
    static const int s_ImageLayoutTable[] =
    {
        VK_IMAGE_LAYOUT_PREINITIALIZED,  // TextureState_DataTransfer = 0x01,
        VK_IMAGE_LAYOUT_TRANSFER_SRC_OPTIMAL,  // TextureState_CopySource = 0x02,
        VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL,  // TextureState_CopyDestination = 0x04,
        VK_IMAGE_LAYOUT_SHADER_READ_ONLY_OPTIMAL,  // TextureState_ShaderRead = 0x08,
        VK_IMAGE_LAYOUT_COLOR_ATTACHMENT_OPTIMAL,  // TextureState_ShaderWrite = 0x10,
        VK_IMAGE_LAYOUT_COLOR_ATTACHMENT_OPTIMAL,  // TextureState_ColorTarget = 0x20,
        VK_IMAGE_LAYOUT_DEPTH_STENCIL_READ_ONLY_OPTIMAL,  // TextureState_DepthRead = 0x40,
        VK_IMAGE_LAYOUT_DEPTH_STENCIL_ATTACHMENT_OPTIMAL,  // TextureState_DepthWrite = 0x80,
        VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL,  // TextureState_Clear = 0x0100,
        VK_IMAGE_LAYOUT_TRANSFER_SRC_OPTIMAL,  // TextureState_ResolveSource = 0x0200,
        VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL,  // TextureState_ResolveDestination = 0x0400,
        VK_IMAGE_LAYOUT_PRESENT_SRC_KHR,  // TextureState_Present = 0x800,
    };
    NN_SDK_ASSERT( textureState <= TextureState_Present );

    VkImageLayout layout = VK_IMAGE_LAYOUT_UNDEFINED;

    if ( Vk::IsLayoutManagementEnabled() )
    {
        if ( ( textureState & TextureState_DepthWrite ) && ( textureState & TextureState_DepthRead ) )
        {
            // Turn off DepthRead since DepthWrite is already set
            textureState &= ~( TextureState_DepthRead );
        }

        for ( uint32_t idxTable = 0, mask = TextureState_DataTransfer; mask <= TextureState_Present; ++idxTable, mask <<= 1 )
        {
            if ( ( textureState & 1 ) == 1 )
            {
                layout = static_cast< VkImageLayout >( s_ImageLayoutTable[ idxTable ] );
                break;
            }
            textureState >>= 1;
        }

        return layout;
    }
    else
    {
        if ( textureState & TextureState_Present )
        {
            return VK_IMAGE_LAYOUT_PRESENT_SRC_KHR;
        }
        else if ( textureState & TextureState_DataTransfer )
        {
            return VK_IMAGE_LAYOUT_PREINITIALIZED;
        }
        else if ( textureState == TextureState_Undefined )
        {
            return VK_IMAGE_LAYOUT_UNDEFINED;
        }

        return VK_IMAGE_LAYOUT_GENERAL;
    }
}

VkComponentSwizzle Vk::GetComponentSwizzle( nn::gfx::ChannelMapping mapping ) NN_NOEXCEPT
{
    static const VkComponentSwizzle s_ComponentSwizzleTable[] =
    {
        VK_COMPONENT_SWIZZLE_ZERO,  // ChannelMapping_Zero,
        VK_COMPONENT_SWIZZLE_ONE,   // ChannelMapping_One,
        VK_COMPONENT_SWIZZLE_R,     // ChannelMapping_Red,
        VK_COMPONENT_SWIZZLE_G,     // ChannelMapping_Green,
        VK_COMPONENT_SWIZZLE_B,     // ChannelMapping_Blue,
        VK_COMPONENT_SWIZZLE_A,     // ChannelMapping_Alpha,
    };
    NN_SDK_ASSERT( mapping < ChannelMapping_End );

    return s_ComponentSwizzleTable[ mapping ];
}

VkPipelineStageFlags Vk::GetPipelineStageBits( int pipelineStageBits, bool isDepth ) NN_NOEXCEPT
{
    static const VkPipelineStageFlags s_PipelineStageFlagBitTable[] =
    {
        VK_PIPELINE_STAGE_VERTEX_INPUT_BIT,  // PipelineStageBit_VertexInput = 0x01,
        VK_PIPELINE_STAGE_VERTEX_SHADER_BIT,  // PipelineStageBit_VertexShader = 0x02,
        VK_PIPELINE_STAGE_TESSELLATION_CONTROL_SHADER_BIT,  // PipelineStageBit_HullShader = 0x04,
        VK_PIPELINE_STAGE_TESSELLATION_EVALUATION_SHADER_BIT,  // PipelineStageBit_DomainShader = 0x08,
        VK_PIPELINE_STAGE_GEOMETRY_SHADER_BIT,  // PipelineStageBit_GeometryShader = 0x10,
        VK_PIPELINE_STAGE_FRAGMENT_SHADER_BIT,  // PipelineStageBit_PixelShader = 0x20,
        0,  // PipelineStageBit_RenderTarget = 0x40,
        VK_PIPELINE_STAGE_COMPUTE_SHADER_BIT,  // PipelineStageBit_ComputeShader = 0x80,
    };
    NN_SDK_ASSERT( pipelineStageBits < ( PipelineStageBit_ComputeShader << 1 ) );

    VkAccessFlags flags = 0;

    if ( ( pipelineStageBits & PipelineStageBit_RenderTarget ) != 0 )
    {
        if ( isDepth )
        {
            flags |= VK_PIPELINE_STAGE_EARLY_FRAGMENT_TESTS_BIT | VK_PIPELINE_STAGE_LATE_FRAGMENT_TESTS_BIT;
        }
        else
        {
            flags |= VK_PIPELINE_STAGE_COLOR_ATTACHMENT_OUTPUT_BIT;
        }
    }

    uint32_t idxTable, mask;
    for ( idxTable = 0, mask = PipelineStageBit_VertexInput; mask <= PipelineStageBit_ComputeShader; ++idxTable, mask <<= 1 )
    {
        if ( ( pipelineStageBits & 1 ) == 1 )
        {
            flags |= s_PipelineStageFlagBitTable[ idxTable ];
        }
        pipelineStageBits >>= 1;
    }

    if ( flags == 0 )
    {
        flags = VK_PIPELINE_STAGE_TRANSFER_BIT;
    }

    return flags;
}

VkPipelineBindPoint Vk::GetPipelineBindPoint( PipelineType pipelineType ) NN_NOEXCEPT
{
    VkPipelineBindPoint pipelineBindPoints[] = {
        VK_PIPELINE_BIND_POINT_GRAPHICS, // PipelineType_Graphics, //!< グラフィックスパイプラインです
        VK_PIPELINE_BIND_POINT_COMPUTE, // PipelineType_Compute, //!< 演算パイプラインです
    };
    NN_SDK_ASSERT( pipelineType < PipelineType_End );

    return pipelineBindPoints[ pipelineType ];
}

VkResult Vk::SelectMemoryType( uint32_t* pOutMemoryTypeIndex,
    uint32_t memoryTypeCount, const uint32_t* pMemoryPropertyFlags,
    uint32_t typeBits, uint32_t requirementsMask ) NN_NOEXCEPT
{
    VkResult result = VK_ERROR_FORMAT_NOT_SUPPORTED;

    for ( uint32_t idxMemoryType = 0; idxMemoryType < memoryTypeCount; ++idxMemoryType )
    {
        if ( ( ( typeBits >> idxMemoryType ) & 1 ) == 1 )
        {
            // Select a memory type for the device that matches the requirements
            if ( ( pMemoryPropertyFlags[ idxMemoryType ] & requirementsMask )
                == requirementsMask )
            {
                *pOutMemoryTypeIndex = idxMemoryType;
                result = VK_SUCCESS;
                break;
            }
        }
    }

    return result;
}

void Vk::EnumerateLayers() NN_NOEXCEPT
{
    VkResult result;

    do
    {
        result = vkEnumerateInstanceLayerProperties( &g_LayerCount, NULL );
        NN_SDK_ASSERT_EQUAL( VK_SUCCESS, result );
        if ( g_LayerCount == 0 )
        {
            break;
        }
        NN_SDK_ASSERT_LESS_EQUAL( g_LayerCount, MaxLayerProperties );

        result = vkEnumerateInstanceLayerProperties( &g_LayerCount, &g_LayerProperties[ 0 ] );
    }
    while ( result == VK_INCOMPLETE );
}

void Vk::EnumerateInstanceExtensions() NN_NOEXCEPT
{
    VkResult result;

    VkExtensionProperties* pExtensions = &g_InstanceExtensionProperties[ 0 ];
    VkLayerProperties* pLayers = &g_LayerProperties[ 0 ];

    {
        uint32_t extensionCount = 0;
        result = vkEnumerateInstanceExtensionProperties( NULL, &extensionCount, NULL );
        NN_SDK_ASSERT_EQUAL( VK_SUCCESS, result );

        if ( extensionCount != 0 )
        {
            g_InstanceExtensionCount += extensionCount;
            NN_SDK_ASSERT_LESS_EQUAL( g_InstanceExtensionCount, MaxInstanceExtensionProperties );

            result = vkEnumerateInstanceExtensionProperties( NULL, &extensionCount, pExtensions );
            pExtensions += extensionCount;
        }
    }
    for ( uint32_t idxLayer = 0; idxLayer < g_LayerCount; ++idxLayer )
    {
        uint32_t extensionCount = 0;
        do
        {
            result = vkEnumerateInstanceExtensionProperties( pLayers[ idxLayer ].layerName, &extensionCount, NULL );
            NN_SDK_ASSERT_EQUAL( VK_SUCCESS, result );

            if ( extensionCount == 0 )
            {
                break;
            }

            g_InstanceExtensionCount += extensionCount;
            NN_SDK_ASSERT_LESS_EQUAL( g_InstanceExtensionCount, MaxInstanceExtensionProperties );

            result = vkEnumerateInstanceExtensionProperties( pLayers[ idxLayer ].layerName, &extensionCount, pExtensions );
            pExtensions += extensionCount;
        }
        while ( result == VK_INCOMPLETE );
    }
}

void Vk::EnumerateDeviceExtensions( DeviceImpl< ApiVariationVk1 >* pDevice ) NN_NOEXCEPT
{
    VkResult result;

    VkExtensionProperties* pExtensions = &g_DeviceExtensionProperties[ 0 ];
    VkLayerProperties* pLayers = &g_LayerProperties[ 0 ];

    VkPhysicalDevice physicalDevice = CastToVkDispatchableObject< VkPhysicalDevice >( pDevice->ToData()->hPhysicalDevice );

    {
        uint32_t extensionCount = 0;
        result = vkEnumerateDeviceExtensionProperties( physicalDevice, NULL, &extensionCount, NULL );
        NN_SDK_ASSERT_EQUAL( VK_SUCCESS, result );

        if ( extensionCount != 0 )
        {
            g_DeviceExtensionCount += extensionCount;
            NN_SDK_ASSERT_LESS_EQUAL( g_DeviceExtensionCount, MaxDeviceExtensionProperties );

            result = vkEnumerateDeviceExtensionProperties( physicalDevice, NULL, &extensionCount, pExtensions );
            pExtensions += extensionCount;
        }
    }
    for ( uint32_t idxLayer = 0; idxLayer < g_LayerCount; ++idxLayer )
    {
        uint32_t extensionCount = 0;
        do
        {
            result = vkEnumerateDeviceExtensionProperties( physicalDevice, pLayers[ idxLayer ].layerName, &extensionCount, NULL );
            NN_SDK_ASSERT_EQUAL( VK_SUCCESS, result );

            if ( extensionCount == 0 )
            {
                break;
            }

            g_DeviceExtensionCount += extensionCount;
            NN_SDK_ASSERT_LESS_EQUAL( g_DeviceExtensionCount, MaxDeviceExtensionProperties );

            result = vkEnumerateDeviceExtensionProperties( physicalDevice, pLayers[ idxLayer ].layerName, &extensionCount, pExtensions );
            pExtensions += extensionCount;
        }
        while ( result == VK_INCOMPLETE );
    }
}

void Vk::CreateDebugCallback( DeviceImpl< ApiVariationVk1 >* pDevice ) NN_NOEXCEPT
{
    VkResult result;
    VkDebugReportCallbackCreateInfoEXT createInfo = { };
    createInfo.sType = VK_STRUCTURE_TYPE_DEBUG_REPORT_CALLBACK_CREATE_INFO_EXT;
    createInfo.flags = VK_DEBUG_REPORT_WARNING_BIT_EXT | VK_DEBUG_REPORT_PERFORMANCE_WARNING_BIT_EXT | VK_DEBUG_REPORT_ERROR_BIT_EXT;
    // Use code below for more detail information.
    //createInfo.flags |= VK_DEBUG_REPORT_DEBUG_BIT_EXT | VK_DEBUG_REPORT_INFORMATION_BIT_EXT;
    createInfo.pfnCallback = DebugMessageCallback;
    createInfo.pNext = NULL;
    createInfo.pUserData = NULL;

    VkDebugReportCallbackEXT debugCallbackEXT;

    PFN_vkCreateDebugReportCallbackEXT pVkCreateDebugReportCallbackEXT;
    NN_GFX_CALL_VK_FUNCTION( pVkCreateDebugReportCallbackEXT = ( PFN_vkCreateDebugReportCallbackEXT )vkGetInstanceProcAddr(
        CastToVkDispatchableObject< VkInstance >( pDevice->ToData()->hInstance ), "vkCreateDebugReportCallbackEXT" ) );

    NN_GFX_CALL_VK_FUNCTION( result = pVkCreateDebugReportCallbackEXT(
        CastToVkDispatchableObject< VkInstance >( pDevice->ToData()->hInstance ),
        &createInfo, pDevice->ToData()->pAllocationCallback, &debugCallbackEXT ) );
    NN_SDK_ASSERT_EQUAL( VK_SUCCESS, result );
    NN_UNUSED( result );
    pDevice->ToData()->hDebugCallback = CastFromVkNonDispatchableObject< VkDebugReportCallbackEXT >(
        debugCallbackEXT );
}

void Vk::DestroyDebugCallback( DeviceImpl< ApiVariationVk1 >* pDevice ) NN_NOEXCEPT
{
    PFN_vkDestroyDebugReportCallbackEXT pVkDestroyDebugReportCallbackEXT;
    NN_GFX_CALL_VK_FUNCTION( pVkDestroyDebugReportCallbackEXT = ( PFN_vkDestroyDebugReportCallbackEXT )vkGetInstanceProcAddr(
        CastToVkDispatchableObject< VkInstance >( pDevice->ToData()->hInstance ), "vkDestroyDebugReportCallbackEXT" ) );

    NN_GFX_CALL_VK_FUNCTION( pVkDestroyDebugReportCallbackEXT(
        CastToVkDispatchableObject< VkInstance >( pDevice->ToData()->hInstance ),
        CastToVkNonDispatchableObject< VkDebugReportCallbackEXT >(
            pDevice->ToData()->hDebugCallback ), pDevice->ToData()->pAllocationCallback ) );
}

bool Vk::QueryExtension( const char* pExtensionName ) NN_NOEXCEPT
{
    for ( size_t idxExtension = 0; idxExtension < g_InstanceExtensionCount; ++idxExtension )
    {
        if ( !strcmp( pExtensionName, g_InstanceExtensionProperties[ idxExtension ].extensionName ) )
        {
            return true;
        }
    }

    for ( size_t idxExtension = 0; idxExtension < g_DeviceExtensionCount; ++idxExtension )
    {
        if ( !strcmp( pExtensionName, g_DeviceExtensionProperties[ idxExtension ].extensionName ) )
        {
            return true;
        }
    }

    return false;
}

bool Vk::QueryLayer( const char* pLayerName ) NN_NOEXCEPT
{
    for ( size_t idxLayer = 0; idxLayer < g_LayerCount; ++idxLayer )
    {
        if ( !strcmp( pLayerName, g_LayerProperties[ idxLayer ].layerName ) )
        {
            return true;
        }
    }

    return false;
}

bool Vk::VerifyLayers( const char** pLayerNames, size_t layerNameSize ) NN_NOEXCEPT
{
    for ( size_t idxLayerName = 0; idxLayerName < layerNameSize; ++idxLayerName )
    {
        if ( !QueryLayer( pLayerNames[ idxLayerName ] ) )
        {
            NN_DETAIL_GFX_INFO( "Failed to find layer: %s\n", pLayerNames[ idxLayerName ] );
            return false;
        }
    }

    return true;
}

bool Vk::IsLayoutManagementEnabled() NN_NOEXCEPT
{
    return g_LayoutManagementEnabled;
}

void Vk::CreateInstance( DeviceImpl< ApiVariationVk1 >* pDevice,
    int layerCount, const char** pLayerNames,
    int extensionCount, const char** pExtensionNames ) NN_NOEXCEPT
{
    VkResult result;

    VkInstanceCreateInfo createInfo = {};
    VkApplicationInfo myAppInfo = {};

    NN_SDK_ASSERT( layerCount == 0 || Vk::VerifyLayers( pLayerNames, layerCount ) );

    myAppInfo.sType = VK_STRUCTURE_TYPE_APPLICATION_INFO;
    myAppInfo.pNext = NULL;
    myAppInfo.pApplicationName = "gfx on Vulkan Application";
    myAppInfo.applicationVersion = 1;
    myAppInfo.pEngineName = "gfx on Vulkan Application";
    myAppInfo.apiVersion = VK_MAKE_VERSION( 1, 0, 0 );

    createInfo.sType = VK_STRUCTURE_TYPE_INSTANCE_CREATE_INFO;
    createInfo.pNext = NULL;
    createInfo.flags = 0;
    createInfo.pApplicationInfo = &myAppInfo;
    createInfo.enabledLayerCount = layerCount;
    createInfo.ppEnabledLayerNames = pLayerNames;
    createInfo.enabledExtensionCount = extensionCount;
    createInfo.ppEnabledExtensionNames = pExtensionNames;

    VkInstance vkInstance;
    const VkAllocationCallbacks* pAllocator =
        static_cast< const VkAllocationCallbacks* >( pDevice->ToData()->pAllocationCallback.ptr );
    NN_GFX_CALL_VK_FUNCTION( result = vkCreateInstance( &createInfo, pAllocator, &vkInstance ) );
    NN_SDK_ASSERT_EQUAL( VK_SUCCESS, result );
    NN_UNUSED( result );

    pDevice->ToData()->hInstance = CastFromVkDispatchableObject< VkInstance >( vkInstance );
}

void Vk::DestroyInstance( DeviceImpl< ApiVariationVk1 >* pDevice ) NN_NOEXCEPT
{
    VkInstance instance = CastToVkDispatchableObject< VkInstance >( pDevice->ToData()->hInstance );
    const VkAllocationCallbacks* pAllocator =
        static_cast< const VkAllocationCallbacks* >( pDevice->ToData()->pAllocationCallback.ptr );
    NN_GFX_CALL_VK_FUNCTION( vkDestroyInstance( instance, pAllocator ) );
}

void Vk::InitializePhysicalDevice( DeviceImpl< ApiVariationVk1 >* pDevice ) NN_NOEXCEPT
{
    VkResult result;
    uint32_t deviceCount = 0;

    NN_GFX_CALL_VK_FUNCTION( result = vkEnumeratePhysicalDevices(
        CastToVkDispatchableObject< VkInstance >( pDevice->ToData()->hInstance ), &deviceCount, NULL ) );
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );

    NN_SDK_ASSERT_GREATER_EQUAL( deviceCount, 1U );

    // FIXME: For now just grab the first device (most systems)
    deviceCount = 1;
    VkPhysicalDevice physicalDevice;
    NN_GFX_CALL_VK_FUNCTION( result = vkEnumeratePhysicalDevices(
        CastToVkDispatchableObject< VkInstance >( pDevice->ToData()->hInstance ), &deviceCount, &physicalDevice ) );
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
    NN_UNUSED( result );
    pDevice->ToData()->hPhysicalDevice = CastFromVkDispatchableObject< VkPhysicalDevice >( physicalDevice );

    VkPhysicalDeviceProperties physicalDeviceProperties;
    NN_GFX_CALL_VK_FUNCTION( vkGetPhysicalDeviceProperties(
        CastToVkDispatchableObject< VkPhysicalDevice >( pDevice->ToData()->hPhysicalDevice ),
        &physicalDeviceProperties ) );
    pDevice->ToData()->minUniformBufferOffsetAlignment = physicalDeviceProperties.limits.minUniformBufferOffsetAlignment;
    pDevice->ToData()->minStorageBufferOffsetAlignment = physicalDeviceProperties.limits.minStorageBufferOffsetAlignment;
    pDevice->ToData()->minTexelBufferOffsetAlignment = physicalDeviceProperties.limits.minTexelBufferOffsetAlignment;
    pDevice->ToData()->nonCoherentAtomSize = physicalDeviceProperties.limits.nonCoherentAtomSize;
    pDevice->ToData()->timestampComputeAndGraphics = physicalDeviceProperties.limits.timestampComputeAndGraphics;
    pDevice->ToData()->vendorId = physicalDeviceProperties.vendorID;
    pDevice->ToData()->timestampPeriod = physicalDeviceProperties.limits.timestampPeriod;

    VkPhysicalDeviceMemoryProperties physicalDeviceMemoryProperties;
    NN_GFX_CALL_VK_FUNCTION( vkGetPhysicalDeviceMemoryProperties(
        CastToVkDispatchableObject< VkPhysicalDevice >( pDevice->ToData()->hPhysicalDevice ),
        &physicalDeviceMemoryProperties ) );
    pDevice->ToData()->memoryTypeCount = physicalDeviceMemoryProperties.memoryTypeCount;
    for ( uint32_t idxMemoryType = 0; idxMemoryType < physicalDeviceMemoryProperties.memoryTypeCount; ++idxMemoryType )
    {
        pDevice->ToData()->memoryPropertyFlags[ idxMemoryType ] =
            physicalDeviceMemoryProperties.memoryTypes[ idxMemoryType ].propertyFlags;
    }

    Vk::SetTicksPerNanosecond( pDevice->ToData()->timestampPeriod );

    // Optimization for Nvidia HW. It is unecessary to perform layout transitions
    g_LayoutManagementEnabled = pDevice->ToData()->vendorId != 0x10DE;
    if ( !g_LayoutManagementEnabled )
    {
        LayoutColorAttachment = VK_IMAGE_LAYOUT_GENERAL;
        LayoutDepthStencilAttachment = VK_IMAGE_LAYOUT_GENERAL;
        LayoutCopyImageSrc = VK_IMAGE_LAYOUT_GENERAL;
        LayoutCopyImageDst = VK_IMAGE_LAYOUT_GENERAL;
        LayoutBlitImageSrc = VK_IMAGE_LAYOUT_GENERAL;
        LayoutBlitImageDst = VK_IMAGE_LAYOUT_GENERAL;
        LayoutResolveImageSrc = VK_IMAGE_LAYOUT_GENERAL;
        LayoutResolveImageDst = VK_IMAGE_LAYOUT_GENERAL;
        LayoutTextureRead = VK_IMAGE_LAYOUT_GENERAL;
        LayoutTextureDepthStencilRead = VK_IMAGE_LAYOUT_GENERAL;
        // This is always transfer dst optimal to suppress validation warnings.
        // LayoutClearImage = VK_IMAGE_LAYOUT_GENERAL;
    }

    uint32_t queueFamilyPropertyCount = 0;
    NN_GFX_CALL_VK_FUNCTION( vkGetPhysicalDeviceQueueFamilyProperties(
        CastToVkDispatchableObject< VkPhysicalDevice >( pDevice->ToData()->hPhysicalDevice ),
        &queueFamilyPropertyCount, NULL ) );
    NN_SDK_ASSERT_LESS_EQUAL( queueFamilyPropertyCount,
        sizeof( pDevice->ToData()->queueFamilyProperties ) / sizeof( VkQueueFamilyProperties ) );

    VkQueueFamilyProperties* pQueueFamilyProperties =
        reinterpret_cast< VkQueueFamilyProperties* >( pDevice->ToData()->queueFamilyProperties );
    NN_GFX_CALL_VK_FUNCTION( vkGetPhysicalDeviceQueueFamilyProperties(
        CastToVkDispatchableObject< VkPhysicalDevice >( pDevice->ToData()->hPhysicalDevice ),
        &queueFamilyPropertyCount, pQueueFamilyProperties ) );

    pDevice->ToData()->queueFamilyPropertyCount = queueFamilyPropertyCount;
}

void Vk::CreateLogicalDevice( DeviceImpl< ApiVariationVk1 >* pDevice,
    int layerCount, const char** pLayerNames,
    int extensionCount, const char** pExtensionNames ) NN_NOEXCEPT
{
    VkResult result;

    VkDeviceQueueCreateInfo queueInfo = {};
    float priority = 0.0;
    queueInfo.sType = VK_STRUCTURE_TYPE_DEVICE_QUEUE_CREATE_INFO;
    queueInfo.pNext = NULL;
    queueInfo.flags = 0;
    queueInfo.queueFamilyIndex = ~0U;
    queueInfo.queueCount = 1;
    queueInfo.pQueuePriorities = &priority;

    VkQueueFamilyProperties* pQueueProperties = reinterpret_cast< VkQueueFamilyProperties* >(
        &pDevice->ToData()->queueFamilyProperties[ 0 ] );
    for ( uint32_t idxProperty = 0; idxProperty < pDevice->ToData()->queueFamilyPropertyCount; ++idxProperty )
    {
        if ( ( pQueueProperties[ idxProperty ].queueFlags
            & ( VK_QUEUE_GRAPHICS_BIT | VK_QUEUE_COMPUTE_BIT | VK_QUEUE_TRANSFER_BIT ) )
            == ( VK_QUEUE_GRAPHICS_BIT | VK_QUEUE_COMPUTE_BIT | VK_QUEUE_TRANSFER_BIT ) )
        {
            queueInfo.queueFamilyIndex = idxProperty;
            break;
        }
    }
    pDevice->ToData()->queueFamilyIndex = queueInfo.queueFamilyIndex;
    NN_SDK_ASSERT( queueInfo.queueFamilyIndex != ~0U );

    VkPhysicalDevice physicalDevice = CastToVkDispatchableObject< VkPhysicalDevice >( pDevice->ToData()->hPhysicalDevice );
    VkPhysicalDeviceFeatures features = {};
    NN_GFX_CALL_VK_FUNCTION( vkGetPhysicalDeviceFeatures( physicalDevice, &features ) );
    features.robustBufferAccess = VK_FALSE;

    size_t workMemorySize = sizeof( pDevice->ToData()->supportedFeaturesBitArrayWorkMemory );
    NN_SDK_ASSERT( workMemorySize >= nn::util::BitArray::CalculateWorkMemorySize( VkDeviceFeature_End ) );
    pDevice->ToData()->supportedFeaturesBitArray.ResetWorkMemory(
        pDevice->ToData()->supportedFeaturesBitArrayWorkMemory, workMemorySize, VkDeviceFeature_End );
    Vk::GetDeviceFeatures( &pDevice->ToData()->supportedFeaturesBitArray, features );

    Vk::SetupGlobalDynamicState( features );

    const char* availableExtensionNames[ MaxInstanceExtensionProperties + MaxDeviceExtensionProperties ];
    int availableExtensionCount = 0;
    for ( int idxExtension = 0; idxExtension < extensionCount; ++idxExtension )
    {
        if ( QueryExtension( pExtensionNames[ idxExtension ] ) )
        {
            availableExtensionNames[ availableExtensionCount++ ] = pExtensionNames[ idxExtension ];
        }
    }
    GetDeviceExtensions( &pDevice->ToData()->availableExtensions, availableExtensionCount, availableExtensionNames );

    VkDeviceCreateInfo createInfo = {};
    createInfo.sType = VK_STRUCTURE_TYPE_DEVICE_CREATE_INFO;
    createInfo.pNext = NULL;
    createInfo.flags = 0;
    createInfo.queueCreateInfoCount = 1;
    createInfo.pQueueCreateInfos = &queueInfo;
    createInfo.enabledLayerCount = layerCount;
    createInfo.ppEnabledLayerNames = pLayerNames;
    createInfo.enabledExtensionCount = availableExtensionCount;
    createInfo.ppEnabledExtensionNames = availableExtensionNames;
    createInfo.pEnabledFeatures = &features;

    VkDevice vkDevice;
    NN_GFX_CALL_VK_FUNCTION( result = vkCreateDevice( physicalDevice, &createInfo,
        pDevice->ToData()->pAllocationCallback, &vkDevice ) );
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
    NN_UNUSED( result );
    pDevice->ToData()->hDevice = CastFromVkDispatchableObject< VkDevice >( vkDevice );
}

void Vk::DestroyLogicalDevice( DeviceImpl< ApiVariationVk1 >* pDevice ) NN_NOEXCEPT
{
    VkDevice device = CastToVkDispatchableObject< VkDevice >( pDevice->ToData()->hDevice );
    NN_GFX_CALL_VK_FUNCTION( vkDeviceWaitIdle( device ) );
    NN_GFX_CALL_VK_FUNCTION( vkDestroyDevice( device, pDevice->ToData()->pAllocationCallback ) );

    pDevice->ToData()->supportedFeaturesBitArray.ResetWorkMemory();
}

VkBuffer Vk::GetVkBufferFromGpuAddress( const GpuAddress& address ) NN_NOEXCEPT
{
#ifdef NN_BUILD_CONFIG_ADDRESS_32
    return static_cast< VkBuffer >( address.ToData()->impl );
#else
    return reinterpret_cast< VkBuffer >( address.ToData()->impl );
#endif
}

uint64_t Vk::GetVkOffsetFromGpuAddress( const GpuAddress& address ) NN_NOEXCEPT
{
    return address.ToData()->value;
}

size_t Vk::CopyCpuMemoryToTexture( void* pOutMem, VkDevice device, VkImage image,
    const nn::gfx::ImageFormat& imageFormat, uint32_t arrayLevel, uint32_t mipLevel,
    uint32_t width, uint32_t height, uint32_t depth, const void* pSourceMem ) NN_NOEXCEPT
{
    VkImageSubresource subResource = { 0 };
    VkSubresourceLayout layout = { 0 };
    subResource.aspectMask = VK_IMAGE_ASPECT_COLOR_BIT;
    subResource.arrayLayer = arrayLevel;
    subResource.mipLevel = mipLevel;

    NN_GFX_CALL_VK_FUNCTION( vkGetImageSubresourceLayout( device, image, &subResource, &layout ) );

    size_t copySize = CalculateImageSize( GetChannelFormat( imageFormat ), width, height, depth );
    nn::util::BytePtr destAddr( pOutMem, static_cast< ptrdiff_t >( layout.offset ) );

    NN_SDK_ASSERT_LESS_EQUAL( copySize, layout.size );
    memcpy( destAddr.Get(), pSourceMem, copySize );

    return copySize;
}

size_t Vk::InitializeOptimalTexture( DeviceImpl< ApiVariationVk1 >* pDevice, VkImage image,
    const nn::gfx::ImageFormat& imageFormat, uint32_t arrayLevel, uint32_t mipLevel, uint32_t width,
    uint32_t height, uint32_t depth, const void* pSourceMem ) NN_NOEXCEPT
{
    // FIXME: Nothing about this seems kosher but we need it temporarily
    VkResult result;
    VkDevice device = CastToVkDispatchableObject< VkDevice >( pDevice->ToData()->hDevice );
    VkQueueFamilyProperties* pQueueProperties = reinterpret_cast< VkQueueFamilyProperties* >( pDevice->ToData()->queueFamilyProperties );

    VkCommandPool tmpCBPool;
    {
        VkCommandPoolCreateInfo info;
        info.sType = VK_STRUCTURE_TYPE_COMMAND_POOL_CREATE_INFO;
        info.pNext = NULL;
        info.queueFamilyIndex = 0;
        info.flags = VK_COMMAND_POOL_CREATE_TRANSIENT_BIT;
        result = NN_GFX_CALL_VK_FUNCTION( vkCreateCommandPool( device, &info, pDevice->ToData()->pAllocationCallback, &tmpCBPool ) );
        NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
    }

    VkCommandBuffer tmpCB;
    {
        VkCommandBufferAllocateInfo info;
        info.sType = VK_STRUCTURE_TYPE_COMMAND_BUFFER_ALLOCATE_INFO;
        info.pNext = NULL;
        info.commandBufferCount = 1;
        info.commandPool = tmpCBPool;
        info.level = VK_COMMAND_BUFFER_LEVEL_PRIMARY;
        NN_GFX_CALL_VK_FUNCTION( result = vkAllocateCommandBuffers( device, &info, &tmpCB ) );
        NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
    }

    VkBuffer tmpBuf;
    VkDeviceMemory tmpBufMem;
    bool isCompressedFormat = IsCompressedFormat( GetChannelFormat( imageFormat ) );
    size_t imageSize = CalculateImageSize( GetChannelFormat( imageFormat ),
        nn::util::align_up( width, pQueueProperties->minImageTransferGranularity.width ),
        nn::util::align_up( height, pQueueProperties->minImageTransferGranularity.height ),
        nn::util::align_up( depth, pQueueProperties->minImageTransferGranularity.depth ) );
    {
        VkBufferCreateInfo info;
        info.sType = VK_STRUCTURE_TYPE_BUFFER_CREATE_INFO;
        info.pNext = NULL;
        info.flags = 0;
        info.queueFamilyIndexCount = 0;
        info.pQueueFamilyIndices = NULL;
        info.usage = VK_BUFFER_USAGE_TRANSFER_SRC_BIT;
        info.sharingMode = VK_SHARING_MODE_EXCLUSIVE;
        info.size = imageSize;
        NN_GFX_CALL_VK_FUNCTION( result = vkCreateBuffer( device, &info, pDevice->ToData()->pAllocationCallback, &tmpBuf ) );
        NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );

        // Required size may be larger than our buffer
        VkMemoryRequirements memReqs;
        NN_GFX_CALL_VK_FUNCTION( vkGetBufferMemoryRequirements( device, tmpBuf, &memReqs ) );
        VkMemoryAllocateInfo allocInfo;
        allocInfo.sType = VK_STRUCTURE_TYPE_MEMORY_ALLOCATE_INFO;
        allocInfo.pNext = NULL;
        allocInfo.memoryTypeIndex = 0;  // FIXME this is hardcoded to the 1st VkPhysicalDevice heap
        allocInfo.allocationSize = memReqs.size;

        uint32_t memFlags = VK_MEMORY_PROPERTY_HOST_VISIBLE_BIT | VK_MEMORY_PROPERTY_HOST_COHERENT_BIT;

        result = Vk::SelectMemoryType( &allocInfo.memoryTypeIndex, pDevice->ToData()->memoryTypeCount, pDevice->ToData()->memoryPropertyFlags, memReqs.memoryTypeBits, memFlags );
        NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );

        NN_GFX_CALL_VK_FUNCTION( result = vkAllocateMemory( device, &allocInfo, pDevice->ToData()->pAllocationCallback, &tmpBufMem ) );
        NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );

        NN_GFX_CALL_VK_FUNCTION( result = vkBindBufferMemory( device, tmpBuf, tmpBufMem, 0 ) );
        NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );

        void* pData = NULL;
        NN_GFX_CALL_VK_FUNCTION( result = vkMapMemory( device, tmpBufMem, 0, imageSize, 0, &pData ) );
        NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );

        memcpy( pData, pSourceMem, imageSize );

        NN_GFX_CALL_VK_FUNCTION( vkUnmapMemory( device, tmpBufMem ) );
    }
    VkBufferImageCopy region = { 0 };
    region.bufferOffset = 0;
    if ( isCompressedFormat )
    {
        int blockWidth = GetBlockWidth( GetChannelFormat( imageFormat ) );
        int blockHeight = GetBlockHeight ( GetChannelFormat( imageFormat ) );
        region.bufferRowLength = width % blockWidth ? width + blockWidth - ( width % blockWidth ) : width;
        region.bufferImageHeight = height % blockHeight ? height + blockHeight - ( height % blockHeight ) : height;
    }
    else
    {
        region.bufferRowLength = width;
        region.bufferImageHeight = height;
    }
    region.imageExtent.width = width;
    region.imageExtent.height = height;
    region.imageExtent.depth = depth;
    region.imageSubresource.aspectMask = VK_IMAGE_ASPECT_COLOR_BIT;
    region.imageSubresource.baseArrayLayer = arrayLevel;
    region.imageSubresource.layerCount = 1;
    region.imageSubresource.mipLevel = mipLevel;
    {
        VkCommandBufferBeginInfo info;
        info.sType = VK_STRUCTURE_TYPE_COMMAND_BUFFER_BEGIN_INFO;
        info.pNext = NULL;
        info.pInheritanceInfo = NULL;
        info.flags = 0;
        NN_GFX_CALL_VK_FUNCTION( vkBeginCommandBuffer( tmpCB, &info ) );
    }
    {
        VkImageMemoryBarrier imageBarrier;
        imageBarrier.sType = VK_STRUCTURE_TYPE_IMAGE_MEMORY_BARRIER;
        imageBarrier.pNext = NULL;
        imageBarrier.srcAccessMask = 0;
        imageBarrier.dstAccessMask = VK_ACCESS_TRANSFER_WRITE_BIT;
        imageBarrier.oldLayout = VK_IMAGE_LAYOUT_UNDEFINED;
        imageBarrier.newLayout = VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL;
        imageBarrier.srcQueueFamilyIndex = 0;
        imageBarrier.dstQueueFamilyIndex = 0;
        imageBarrier.image = image;
        imageBarrier.subresourceRange.aspectMask = VK_IMAGE_ASPECT_COLOR_BIT;
        imageBarrier.subresourceRange.baseArrayLayer = arrayLevel;
        imageBarrier.subresourceRange.baseMipLevel = mipLevel;
        imageBarrier.subresourceRange.layerCount = 1;
        imageBarrier.subresourceRange.levelCount = 1;
        NN_GFX_CALL_VK_FUNCTION( vkCmdPipelineBarrier( tmpCB, VK_PIPELINE_STAGE_ALL_COMMANDS_BIT, VK_PIPELINE_STAGE_ALL_COMMANDS_BIT,
            0, 0, NULL, 0, NULL, 1, &imageBarrier ) );
    }

    NN_GFX_CALL_VK_FUNCTION( vkCmdCopyBufferToImage( tmpCB, tmpBuf, image, VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL, 1, &region ) );

    if ( !Vk::IsLayoutManagementEnabled() )
    {
        VkImageMemoryBarrier imageBarrier;
        imageBarrier.sType = VK_STRUCTURE_TYPE_IMAGE_MEMORY_BARRIER;
        imageBarrier.pNext = NULL;
        imageBarrier.srcAccessMask = VK_ACCESS_TRANSFER_WRITE_BIT;
        imageBarrier.dstAccessMask = VK_ACCESS_TRANSFER_WRITE_BIT;
        imageBarrier.oldLayout = VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL;
        imageBarrier.newLayout = VK_IMAGE_LAYOUT_GENERAL;
        imageBarrier.srcQueueFamilyIndex = 0;
        imageBarrier.dstQueueFamilyIndex = 0;
        imageBarrier.image = image;
        imageBarrier.subresourceRange.aspectMask = VK_IMAGE_ASPECT_COLOR_BIT;
        imageBarrier.subresourceRange.baseArrayLayer = arrayLevel;
        imageBarrier.subresourceRange.baseMipLevel = mipLevel;
        imageBarrier.subresourceRange.layerCount = 1;
        imageBarrier.subresourceRange.levelCount = 1;
        NN_GFX_CALL_VK_FUNCTION( vkCmdPipelineBarrier( tmpCB, VK_PIPELINE_STAGE_ALL_COMMANDS_BIT, VK_PIPELINE_STAGE_ALL_COMMANDS_BIT,
            0, 0, NULL, 0, NULL, 1, &imageBarrier ) );
    }

    NN_GFX_CALL_VK_FUNCTION( result = vkEndCommandBuffer( tmpCB ) );
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );

    {
        VkQueue queue;
        NN_GFX_CALL_VK_FUNCTION( vkGetDeviceQueue( device, 0, 0, &queue ) );
        VkSubmitInfo info;
        info.sType = VK_STRUCTURE_TYPE_SUBMIT_INFO;
        info.pNext = NULL;
        info.commandBufferCount = 1;
        info.pCommandBuffers = &tmpCB;
        info.pSignalSemaphores = NULL;
        info.pWaitDstStageMask = NULL;
        info.pWaitSemaphores = NULL;
        info.signalSemaphoreCount = 0;
        info.waitSemaphoreCount = 0;

        NN_GFX_CALL_NNOS_FUNCTION( nn::os::LockMutex( &pDevice->ToData()->mutex ) );

        NN_GFX_CALL_VK_FUNCTION( result = vkQueueSubmit( queue, 1, &info, NULL ) );
        NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
        NN_GFX_CALL_VK_FUNCTION( result = vkQueueWaitIdle( queue ) );
        NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );

        NN_GFX_CALL_NNOS_FUNCTION( nn::os::UnlockMutex( &pDevice->ToData()->mutex ) );
    }

    NN_GFX_CALL_VK_FUNCTION( vkDestroyBuffer( device, tmpBuf, pDevice->ToData()->pAllocationCallback ) );
    NN_GFX_CALL_VK_FUNCTION( vkFreeMemory( device, tmpBufMem, pDevice->ToData()->pAllocationCallback ) );
    NN_GFX_CALL_VK_FUNCTION( vkFreeCommandBuffers( device, tmpCBPool, 1, &tmpCB ) );
    NN_GFX_CALL_VK_FUNCTION( vkDestroyCommandPool( device, tmpCBPool, pDevice->ToData()->pAllocationCallback ) );

    return imageSize;
}//NOLINT(impl/function_size)

void Vk::TransitionTextureLayout( DeviceImpl< ApiVariationVk1 >* pDevice, VkImage image,
    const nn::gfx::ImageFormat& imageFormat, VkImageLayout srcLayout, VkAccessFlags srcAccess,
    VkImageLayout dstLayout, VkAccessFlags dstAccess ) NN_NOEXCEPT
{
    // FIXME: Nothing about this seems kosher but we need it temporarily
    VkResult result;
    VkDevice device = CastToVkDispatchableObject< VkDevice >( pDevice->ToData()->hDevice );

    VkCommandPool tmpCBPool;
    {
        VkCommandPoolCreateInfo info;
        info.sType = VK_STRUCTURE_TYPE_COMMAND_POOL_CREATE_INFO;
        info.pNext = NULL;
        info.queueFamilyIndex = 0;
        info.flags = VK_COMMAND_POOL_CREATE_TRANSIENT_BIT;
        result = NN_GFX_CALL_VK_FUNCTION( vkCreateCommandPool( device, &info, pDevice->ToData()->pAllocationCallback, &tmpCBPool ) );
        NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
    }

    VkCommandBuffer tmpCB;
    {
        VkCommandBufferAllocateInfo info;
        info.sType = VK_STRUCTURE_TYPE_COMMAND_BUFFER_ALLOCATE_INFO;
        info.pNext = NULL;
        info.commandBufferCount = 1;
        info.commandPool = tmpCBPool;
        info.level = VK_COMMAND_BUFFER_LEVEL_PRIMARY;
        NN_GFX_CALL_VK_FUNCTION( result = vkAllocateCommandBuffers( device, &info, &tmpCB ) );
        NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
    }

    VkImageAspectFlags aspectMask = 0;

    switch ( imageFormat )
    {
    case ImageFormat_D16_Unorm:
    case ImageFormat_D32_Float:
        aspectMask = VK_IMAGE_ASPECT_DEPTH_BIT;
        break;
    case ImageFormat_D24_Unorm_S8_Uint:
    case ImageFormat_D32_Float_S8_Uint_X24:
        aspectMask = VK_IMAGE_ASPECT_DEPTH_BIT | VK_IMAGE_ASPECT_STENCIL_BIT;
        break;
    default:
        aspectMask = VK_IMAGE_ASPECT_COLOR_BIT;
        break;
    }

    {
        VkCommandBufferBeginInfo info;
        info.sType = VK_STRUCTURE_TYPE_COMMAND_BUFFER_BEGIN_INFO;
        info.pNext = NULL;
        info.pInheritanceInfo = NULL;
        info.flags = 0;
        NN_GFX_CALL_VK_FUNCTION( vkBeginCommandBuffer( tmpCB, &info ) );
    }
    {
        VkImageMemoryBarrier imageBarrier;
        imageBarrier.sType = VK_STRUCTURE_TYPE_IMAGE_MEMORY_BARRIER;
        imageBarrier.pNext = NULL;
        imageBarrier.srcAccessMask = srcAccess;
        imageBarrier.dstAccessMask = dstAccess;
        imageBarrier.oldLayout = srcLayout;
        imageBarrier.newLayout = dstLayout;
        imageBarrier.srcQueueFamilyIndex = 0;
        imageBarrier.dstQueueFamilyIndex = 0;
        imageBarrier.image = image;
        imageBarrier.subresourceRange.aspectMask = aspectMask;
        imageBarrier.subresourceRange.baseArrayLayer = 0;
        imageBarrier.subresourceRange.baseMipLevel = 0;
        imageBarrier.subresourceRange.layerCount = VK_REMAINING_ARRAY_LAYERS;
        imageBarrier.subresourceRange.levelCount = VK_REMAINING_MIP_LEVELS;
        NN_GFX_CALL_VK_FUNCTION( vkCmdPipelineBarrier( tmpCB, VK_PIPELINE_STAGE_ALL_COMMANDS_BIT, VK_PIPELINE_STAGE_ALL_COMMANDS_BIT,
            0, 0, NULL, 0, NULL, 1, &imageBarrier ) );
    }

    NN_GFX_CALL_VK_FUNCTION( result = vkEndCommandBuffer( tmpCB ) );
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );

    {
        VkQueue queue;
        NN_GFX_CALL_VK_FUNCTION( vkGetDeviceQueue( device, 0, 0, &queue ) );
        VkSubmitInfo info;
        info.sType = VK_STRUCTURE_TYPE_SUBMIT_INFO;
        info.pNext = NULL;
        info.commandBufferCount = 1;
        info.pCommandBuffers = &tmpCB;
        info.pSignalSemaphores = NULL;
        info.pWaitDstStageMask = NULL;
        info.pWaitSemaphores = NULL;
        info.signalSemaphoreCount = 0;
        info.waitSemaphoreCount = 0;

        NN_GFX_CALL_NNOS_FUNCTION( nn::os::LockMutex( &pDevice->ToData()->mutex ) );

        NN_GFX_CALL_VK_FUNCTION( result = vkQueueSubmit( queue, 1, &info, NULL ) );
        NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
        NN_GFX_CALL_VK_FUNCTION( result = vkQueueWaitIdle( queue ) );
        NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );

        NN_GFX_CALL_NNOS_FUNCTION( nn::os::UnlockMutex( &pDevice->ToData()->mutex ) );
    }

    NN_GFX_CALL_VK_FUNCTION( vkFreeCommandBuffers( device, tmpCBPool, 1, &tmpCB ) );
    NN_GFX_CALL_VK_FUNCTION( vkDestroyCommandPool( device, tmpCBPool, pDevice->ToData()->pAllocationCallback ) );
}//NOLINT(impl/function_size)

void Vk::ClearColorTexture( DeviceImpl< ApiVariationVk1 >* pDevice, VkImage image, const ClearColorValue& clearColor ) NN_NOEXCEPT
{
    VkResult result;
    VkDevice device = CastToVkDispatchableObject< VkDevice >( pDevice->ToData()->hDevice );

    VkCommandPool tmpCBPool;
    {
        VkCommandPoolCreateInfo info;
        info.sType = VK_STRUCTURE_TYPE_COMMAND_POOL_CREATE_INFO;
        info.pNext = NULL;
        info.queueFamilyIndex = 0;
        info.flags = VK_COMMAND_POOL_CREATE_TRANSIENT_BIT;
        result = NN_GFX_CALL_VK_FUNCTION( vkCreateCommandPool( device, &info, pDevice->ToData()->pAllocationCallback, &tmpCBPool ) );
        NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
    }

    VkCommandBuffer tmpCB;
    {
        VkCommandBufferAllocateInfo info;
        info.sType = VK_STRUCTURE_TYPE_COMMAND_BUFFER_ALLOCATE_INFO;
        info.pNext = NULL;
        info.commandBufferCount = 1;
        info.commandPool = tmpCBPool;
        info.level = VK_COMMAND_BUFFER_LEVEL_PRIMARY;
        NN_GFX_CALL_VK_FUNCTION( result = vkAllocateCommandBuffers( device, &info, &tmpCB ) );
        NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
    }

    {
        VkCommandBufferBeginInfo info;
        info.sType = VK_STRUCTURE_TYPE_COMMAND_BUFFER_BEGIN_INFO;
        info.pNext = NULL;
        info.pInheritanceInfo = NULL;
        info.flags = 0;
        NN_GFX_CALL_VK_FUNCTION( vkBeginCommandBuffer( tmpCB, &info ) );
    }
    {
        VkImageMemoryBarrier imageBarrier;
        imageBarrier.sType = VK_STRUCTURE_TYPE_IMAGE_MEMORY_BARRIER;
        imageBarrier.pNext = NULL;
        imageBarrier.srcAccessMask = 0;
        imageBarrier.dstAccessMask = Vk::GetLayoutAccessFlags( VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL );
        imageBarrier.oldLayout = VK_IMAGE_LAYOUT_UNDEFINED;
        imageBarrier.newLayout = VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL;
        imageBarrier.srcQueueFamilyIndex = 0;
        imageBarrier.dstQueueFamilyIndex = 0;
        imageBarrier.image = image;
        imageBarrier.subresourceRange.aspectMask = VK_IMAGE_ASPECT_COLOR_BIT;
        imageBarrier.subresourceRange.baseArrayLayer = 0;
        imageBarrier.subresourceRange.baseMipLevel = 0;
        imageBarrier.subresourceRange.layerCount = VK_REMAINING_ARRAY_LAYERS;
        imageBarrier.subresourceRange.levelCount = VK_REMAINING_MIP_LEVELS;
        NN_GFX_CALL_VK_FUNCTION( vkCmdPipelineBarrier( tmpCB, VK_PIPELINE_STAGE_ALL_COMMANDS_BIT, VK_PIPELINE_STAGE_ALL_COMMANDS_BIT,
            0, 0, NULL, 0, NULL, 1, &imageBarrier ) );
    }
    {
        VkClearColorValue color;
        for ( int idx = 0; idx < 4; ++idx )
        {
            color.float32[ idx ] = clearColor.valueFloat[ idx ];
        }

        VkImageSubresourceRange range;
        range.aspectMask = VK_IMAGE_ASPECT_COLOR_BIT;
        range.baseArrayLayer = 0;
        range.baseMipLevel = 0;
        range.layerCount = VK_REMAINING_ARRAY_LAYERS;
        range.levelCount = VK_REMAINING_MIP_LEVELS;

        NN_GFX_CALL_VK_FUNCTION( vkCmdClearColorImage( tmpCB, image, VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL, &color, 1, &range ) );
    }

    NN_GFX_CALL_VK_FUNCTION( result = vkEndCommandBuffer( tmpCB ) );
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );

    {
        VkQueue queue;
        NN_GFX_CALL_VK_FUNCTION( vkGetDeviceQueue( device, 0, 0, &queue ) );
        VkSubmitInfo info;
        info.sType = VK_STRUCTURE_TYPE_SUBMIT_INFO;
        info.pNext = NULL;
        info.commandBufferCount = 1;
        info.pCommandBuffers = &tmpCB;
        info.pSignalSemaphores = NULL;
        info.pWaitDstStageMask = NULL;
        info.pWaitSemaphores = NULL;
        info.signalSemaphoreCount = 0;
        info.waitSemaphoreCount = 0;

        NN_GFX_CALL_NNOS_FUNCTION( nn::os::LockMutex( &pDevice->ToData()->mutex ) );

        NN_GFX_CALL_VK_FUNCTION( result = vkQueueSubmit( queue, 1, &info, NULL ) );
        NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
        NN_GFX_CALL_VK_FUNCTION( result = vkQueueWaitIdle( queue ) );
        NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );

        NN_GFX_CALL_NNOS_FUNCTION( nn::os::UnlockMutex( &pDevice->ToData()->mutex ) );
    }

    NN_GFX_CALL_VK_FUNCTION( vkFreeCommandBuffers( device, tmpCBPool, 1, &tmpCB ) );
    NN_GFX_CALL_VK_FUNCTION( vkDestroyCommandPool( device, tmpCBPool, pDevice->ToData()->pAllocationCallback ) );
}//NOLINT(impl/function_size)

nn::TimeSpan Vk::ToTimeSpan( int64_t timestampValue ) NN_NOEXCEPT
{
    return nn::TimeSpan::FromNanoSeconds(
        static_cast< int64_t >( g_NanosecondsPerTick * timestampValue ) );
}

void Vk::UpdateAllocator( DeviceImpl< ApiVariationVk1 >* pDevice ) NN_NOEXCEPT
{
    g_pVkAllocator = static_cast< VkAllocator* >( pDevice->ToData()->pUserVkAllocator.ptr );

    if ( pDevice->ToData()->pUserVkAllocator.ptr == NULL )
    {
        pDevice->ToData()->pAllocationCallback.ptr = NULL;
    }
    else
    {
        VkAllocationCallbacks* pAllocationCallbacks = reinterpret_cast< VkAllocationCallbacks* >(
            pDevice->ToData()->allocationCallback );
        VkAllocator* pUserAllocator = pDevice->ToData()->pUserVkAllocator;

        pAllocationCallbacks->pUserData = pUserAllocator;
        pAllocationCallbacks->pfnAllocation = GfxVkAllocationFunction;
        pAllocationCallbacks->pfnReallocation = GfxVkReallocationFunction;
        pAllocationCallbacks->pfnFree = GfxVkFreeFunction;
        if ( pUserAllocator->pNotifyInternalAllocation != NULL
            && pUserAllocator->pNotifyInternalFree != NULL )
        {
            pAllocationCallbacks->pfnInternalAllocation = GfxVkInternalAllocationNotification;
            pAllocationCallbacks->pfnInternalFree = GfxVkInternalFreeNotification;
        }
        else
        {
            pAllocationCallbacks->pfnInternalAllocation = NULL;
            pAllocationCallbacks->pfnInternalFree = NULL;
        }
        pDevice->ToData()->pAllocationCallback.ptr =
            reinterpret_cast< void* >( pDevice->ToData()->allocationCallback );
    }
}

void* Vk::AllocDriverMemory( size_t size, size_t alignment ) NN_NOEXCEPT
{
    if ( g_pVkAllocator )
    {
        return g_pVkAllocator->pAllocate( g_pVkAllocator->pUserData, size, alignment,
            AllocationScope_GfxManage );
    }
    #if defined(NN_BUILD_CONFIG_OS_WIN)
    return ::_aligned_malloc( size, alignment );
    #else
    return std::aligned_alloc( alignment, size );
    #endif
}

void Vk::FreeDriverMemory( void* pMemory ) NN_NOEXCEPT
{
    if ( g_pVkAllocator )
    {
        g_pVkAllocator->pFree( g_pVkAllocator->pUserData, pMemory );
    }
    else
    {
        #if defined(NN_BUILD_CONFIG_OS_WIN)
        ::_aligned_free( pMemory );
        #else
        std::free( pMemory );
        #endif
    }
}

void Vk::CreatePipelineCache( DeviceImpl< ApiVariationVk1 >* pDevice ) NN_NOEXCEPT
{
    VkPipelineCacheCreateInfo info = {};
    info.sType = VK_STRUCTURE_TYPE_PIPELINE_CACHE_CREATE_INFO;

    VkResult result;
    VkPipelineCache pipelineCache;
    VkDevice device = CastToVkDispatchableObject< VkDevice >( pDevice->ToData()->hDevice );
    const VkAllocationCallbacks* pAllocator =
        static_cast< const VkAllocationCallbacks* >( pDevice->ToData()->pAllocationCallback.ptr );
    NN_GFX_CALL_VK_FUNCTION( result = vkCreatePipelineCache(
        device, &info, pAllocator, &pipelineCache ) );
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
    NN_UNUSED( result );

    pDevice->ToData()->hPipelineCache = CastFromVkNonDispatchableObject< VkPipelineCache >( pipelineCache );
}

void Vk::DestroyPipelineCache( DeviceImpl< ApiVariationVk1 >* pDevice ) NN_NOEXCEPT
{
    VkDevice device = CastToVkDispatchableObject< VkDevice >( pDevice->ToData()->hDevice );
    VkPipelineCache pipelineCache = CastToVkNonDispatchableObject< VkPipelineCache >( pDevice->ToData()->hPipelineCache );
    const VkAllocationCallbacks* pAllocator =
        static_cast< const VkAllocationCallbacks* >( pDevice->ToData()->pAllocationCallback.ptr );
    NN_GFX_CALL_VK_FUNCTION( vkDestroyPipelineCache( device, pipelineCache, pAllocator ) );
}

void Vk::CreateCommandBuffer( CommandBufferImpl< ApiVariationVk1 >* pCommandBuffer ) NN_NOEXCEPT
{
    DeviceImpl< ApiVariationVk1 >* pDevice = pCommandBuffer->ToData()->pGfxDevice;

    VkCommandPoolCreateInfo poolInfo;
    poolInfo.sType = VK_STRUCTURE_TYPE_COMMAND_POOL_CREATE_INFO;
    poolInfo.pNext = NULL;
    poolInfo.flags = VK_COMMAND_POOL_CREATE_RESET_COMMAND_BUFFER_BIT;
    poolInfo.queueFamilyIndex = pDevice->ToData()->queueFamilyIndex;

    VkCommandBufferAllocateInfo allocInfo;
    allocInfo.sType = VK_STRUCTURE_TYPE_COMMAND_BUFFER_ALLOCATE_INFO;
    allocInfo.pNext = NULL;
    allocInfo.level = VK_COMMAND_BUFFER_LEVEL_PRIMARY;
    allocInfo.commandBufferCount = 1;

    VkResult result;
    VkDevice device = CastToVkDispatchableObject< VkDevice >( pDevice->ToData()->hDevice );
    const VkAllocationCallbacks* pAllocator =
        static_cast< const VkAllocationCallbacks* >( pDevice->ToData()->pAllocationCallback.ptr );

    for ( int idxCommandBuffer = 0; idxCommandBuffer < pCommandBuffer->ToData()->VkCommandThreadCount; ++idxCommandBuffer )
    {
        for ( int idxUsage = 0; idxUsage < CommandBufferImpl< ApiVariationVk1 >::DataType::CommandBufferUsage_End; ++idxUsage )
        {
            VkCommandPool commandPool;
            NN_GFX_CALL_VK_FUNCTION( result = vkCreateCommandPool( device, &poolInfo, pAllocator, &commandPool ) );
            NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
            pCommandBuffer->ToData()->vkCommandBufferHandles[ idxCommandBuffer ].hCommandPool[ idxUsage ] =
                CastFromVkNonDispatchableObject< VkCommandPool >( commandPool );

            allocInfo.commandPool = commandPool;
            VkCommandBuffer commandBuffer;
            NN_GFX_CALL_VK_FUNCTION( result = vkAllocateCommandBuffers( device, &allocInfo, &commandBuffer ) );
            NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
            NN_UNUSED( result );
            pCommandBuffer->ToData()->vkCommandBufferHandles[ idxCommandBuffer ].hCommandBuffer[ idxUsage ] =
                CastFromVkDispatchableObject< VkCommandBuffer >( commandBuffer );
        }
    }

    {
        const uint32_t uniformBufferSize = pCommandBuffer->ToData()->uniformBufferSize = 512 * 1024;

        VkBufferCreateInfo bufferInfo;
        bufferInfo.sType = VK_STRUCTURE_TYPE_BUFFER_CREATE_INFO;
        bufferInfo.pNext = NULL;
        bufferInfo.flags = 0;
        bufferInfo.size = uniformBufferSize;
        bufferInfo.usage = VK_BUFFER_USAGE_UNIFORM_BUFFER_BIT;
        bufferInfo.sharingMode = VK_SHARING_MODE_EXCLUSIVE;
        bufferInfo.queueFamilyIndexCount = 0;
        bufferInfo.pQueueFamilyIndices = NULL;

        VkBuffer buffer;
        NN_GFX_CALL_VK_FUNCTION( result = vkCreateBuffer( device, &bufferInfo, pAllocator, &buffer ) );
        NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
        pCommandBuffer->ToData()->hUniformBuffer = CastFromVkNonDispatchableObject< VkBuffer >( buffer );

        VkMemoryRequirements memReqs;
        vkGetBufferMemoryRequirements( device, buffer, &memReqs );

        uint32_t memFlags = VK_MEMORY_PROPERTY_HOST_VISIBLE_BIT;

        VkMemoryAllocateInfo bufferAllocInfo;
        bufferAllocInfo.sType = VK_STRUCTURE_TYPE_MEMORY_ALLOCATE_INFO;
        bufferAllocInfo.pNext = NULL;
        bufferAllocInfo.allocationSize = memReqs.size;
        Vk::SelectMemoryType( &bufferAllocInfo.memoryTypeIndex, pDevice->ToData()->memoryTypeCount,
            pDevice->ToData()->memoryPropertyFlags, memReqs.memoryTypeBits, memFlags );

        VkDeviceMemory bufferMemory;
        NN_GFX_CALL_VK_FUNCTION( result = vkAllocateMemory( device, &bufferAllocInfo, pAllocator, &bufferMemory ) );
        NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
        pCommandBuffer->ToData()->hUniformMemory = CastFromVkNonDispatchableObject< VkDeviceMemory >( bufferMemory );

        NN_GFX_CALL_VK_FUNCTION( result = vkBindBufferMemory( device, buffer, bufferMemory, 0 ) );
        NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );

        NN_GFX_CALL_VK_FUNCTION( result = vkMapMemory( device, bufferMemory, 0, uniformBufferSize, 0, &pCommandBuffer->ToData()->pUniformMemory ) );
        NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
    }
}

void Vk::DestroyCommandBuffer( DeviceImpl< ApiVariationVk1 >* pDevice,
    CommandBufferImpl< ApiVariationVk1 >* pCommandBuffer ) NN_NOEXCEPT
{
    VkDevice device = CastToVkDispatchableObject< VkDevice >( pDevice->ToData()->hDevice );
    const VkAllocationCallbacks* pAllocator =
        static_cast< const VkAllocationCallbacks* >( pDevice->ToData()->pAllocationCallback.ptr );

    for ( int idxCommandBuffer = 0; idxCommandBuffer < pCommandBuffer->ToData()->VkCommandThreadCount; ++idxCommandBuffer )
    {
        for ( int idxUsage = 0; idxUsage < CommandBufferImpl< ApiVariationVk1 >::DataType::CommandBufferUsage_End; ++idxUsage )
        {
            VkCommandPool commandPool = CastToVkNonDispatchableObject< VkCommandPool >(
                pCommandBuffer->ToData()->vkCommandBufferHandles[ idxCommandBuffer ].hCommandPool[ idxUsage ] );
            VkCommandBuffer commandBuffer = CastToVkDispatchableObject< VkCommandBuffer >(
                pCommandBuffer->ToData()->vkCommandBufferHandles[ idxCommandBuffer ].hCommandBuffer[ idxUsage ] );

            NN_GFX_CALL_VK_FUNCTION( vkFreeCommandBuffers( device, commandPool, 1, &commandBuffer ) );
            NN_GFX_CALL_VK_FUNCTION( vkDestroyCommandPool( device, commandPool, pAllocator ) );
        }
    }

    VkBuffer uniformBuffer = CastToVkNonDispatchableObject< VkBuffer >( pCommandBuffer->ToData()->hUniformBuffer );
    VkDeviceMemory uniformMemory = CastToVkNonDispatchableObject< VkDeviceMemory >( pCommandBuffer->ToData()->hUniformMemory );

    NN_GFX_CALL_VK_FUNCTION( vkDestroyBuffer( device, uniformBuffer, pAllocator ) );
    NN_GFX_CALL_VK_FUNCTION( vkFreeMemory( device, uniformMemory, pAllocator ) );
}

void Vk::BeginCommandBuffer( VkCommandBuffer commandBuffer ) NN_NOEXCEPT
{
    VkCommandBufferBeginInfo info;
    info.sType = VK_STRUCTURE_TYPE_COMMAND_BUFFER_BEGIN_INFO;
    info.pNext = NULL;
    info.flags = 0;
    info.pInheritanceInfo = NULL;

    NN_GFX_CALL_VK_FUNCTION( vkBeginCommandBuffer( commandBuffer, &info ) );
}

void Vk::EndCommandBuffer( VkCommandBuffer commandBuffer ) NN_NOEXCEPT
{
    NN_GFX_CALL_VK_FUNCTION( vkEndCommandBuffer( commandBuffer ) );
}

void Vk::EndRenderPass( VkCommandBuffer commandBuffer ) NN_NOEXCEPT
{
    NN_GFX_CALL_VK_FUNCTION( vkCmdEndRenderPass( commandBuffer ) );
}

void Vk::CreateRenderPass( VkRenderPass* pOutRenderPass, DeviceImpl< ApiVariationVk1 >* pDevice,
    VkFormat format, int samples, bool isDepth ) NN_NOEXCEPT
{
    VkAttachmentDescription attachment;
    VkAttachmentReference attachmentReference;

    attachment.flags = 0;
    attachment.format = format;
    attachment.samples = static_cast< VkSampleCountFlagBits >( samples );
    attachment.loadOp = VK_ATTACHMENT_LOAD_OP_CLEAR;
    attachment.storeOp = VK_ATTACHMENT_STORE_OP_STORE;
    attachment.stencilLoadOp = isDepth ? VK_ATTACHMENT_LOAD_OP_CLEAR : VK_ATTACHMENT_LOAD_OP_DONT_CARE;
    attachment.stencilStoreOp = isDepth ? VK_ATTACHMENT_STORE_OP_STORE : VK_ATTACHMENT_STORE_OP_DONT_CARE;
    attachment.initialLayout = VK_IMAGE_LAYOUT_UNDEFINED;
    attachment.finalLayout = isDepth ? Vk::LayoutDepthStencilAttachment
        : Vk::LayoutColorAttachment;

    attachmentReference.attachment = 0;
    attachmentReference.layout = isDepth ? Vk::LayoutDepthStencilAttachment
        : Vk::LayoutColorAttachment;

    VkSubpassDescription subpass;
    subpass.flags = 0;
    subpass.pipelineBindPoint = VK_PIPELINE_BIND_POINT_GRAPHICS;
    subpass.inputAttachmentCount = 0;
    subpass.pInputAttachments = NULL;
    subpass.colorAttachmentCount = isDepth ? 0 : 1;
    subpass.pColorAttachments = isDepth ? NULL : &attachmentReference;
    subpass.pResolveAttachments = NULL;
    subpass.pDepthStencilAttachment = isDepth ? &attachmentReference : NULL;
    subpass.preserveAttachmentCount = 0;
    subpass.pPreserveAttachments = NULL;

    VkRenderPassCreateInfo info;
    info.sType = VK_STRUCTURE_TYPE_RENDER_PASS_CREATE_INFO;
    info.pNext = NULL;
    info.flags = 0;
    info.attachmentCount = 1;
    info.pAttachments = &attachment;
    info.subpassCount = 1;
    info.pSubpasses = &subpass;
    info.dependencyCount = 0;
    info.pDependencies = NULL;

    VkResult result;
    VkDevice device = CastToVkDispatchableObject< VkDevice >( pDevice->ToData()->hDevice );
    const VkAllocationCallbacks* pAllocator =
        static_cast< const VkAllocationCallbacks* >( pDevice->ToData()->pAllocationCallback.ptr );
    NN_GFX_CALL_VK_FUNCTION( result = vkCreateRenderPass( device, &info, pAllocator, pOutRenderPass ) );
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
    NN_UNUSED( result );
}

void Vk::CreateFramebuffer( VkFramebuffer* pOutFramebuffer, DeviceImpl< ApiVariationVk1 >* pDevice,
    VkRenderPass renderPass, VkImageView imageView,
    uint32_t width, uint32_t height, uint32_t layers ) NN_NOEXCEPT
{
    VkFramebufferCreateInfo info;
    info.sType = VK_STRUCTURE_TYPE_FRAMEBUFFER_CREATE_INFO;
    info.pNext = NULL;
    info.flags = 0;
    info.renderPass = renderPass;
    info.attachmentCount = 1;
    info.pAttachments = &imageView;
    info.width = width;
    info.height = height;
    info.layers = layers;

    VkResult result;
    VkDevice device = CastToVkDispatchableObject< VkDevice >( pDevice->ToData()->hDevice );
    const VkAllocationCallbacks* pAllocator =
        static_cast< const VkAllocationCallbacks* >( pDevice->ToData()->pAllocationCallback.ptr );
    NN_GFX_CALL_VK_FUNCTION( result = vkCreateFramebuffer(
        device, &info, pAllocator, pOutFramebuffer ) );
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
    NN_UNUSED( result );
}

void Vk::CreateQueryPool( VkQueryPool* pOutQueryPool, DeviceImpl< ApiVariationVk1 >* pDevice,
    QueryTarget queryTarget, uint32_t queryCount ) NN_NOEXCEPT
{
    VkQueryPoolCreateInfo queryInfo;
    queryInfo.sType = VK_STRUCTURE_TYPE_QUERY_POOL_CREATE_INFO;
    queryInfo.pNext = NULL;
    queryInfo.flags = 0;
    queryInfo.queryType = GetQueryType( queryTarget );
    queryInfo.queryCount = queryCount;
    queryInfo.pipelineStatistics = GetQueryPipelineStatisticFlag( queryTarget );

    if ( queryInfo.queryType == VK_QUERY_TYPE_PIPELINE_STATISTICS
        && !pDevice->ToData()->supportedFeaturesBitArray.test( VkDeviceFeature_PipelineStatisticsQuery ) )
    {
        *pOutQueryPool = NULL;
    }
    else
    {
        VkResult result;
        VkDevice device = CastToVkDispatchableObject< VkDevice >( pDevice->ToData()->hDevice );
        const VkAllocationCallbacks* pAllocator =
            static_cast< const VkAllocationCallbacks* >( pDevice->ToData()->pAllocationCallback.ptr );
        NN_GFX_CALL_VK_FUNCTION( result = vkCreateQueryPool(
            device, &queryInfo, pAllocator, pOutQueryPool ) );
        NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
        NN_UNUSED( result );
    }
}

void Vk::CreateGraphicsPipelineStatisticsQueriesPool( VkQueryPool* pOutQueryPool,
    DeviceImpl< ApiVariationVk1 >* pDevice, uint32_t queryCount ) NN_NOEXCEPT
{
    VkQueryPoolCreateInfo queryInfo;
    queryInfo.sType = VK_STRUCTURE_TYPE_QUERY_POOL_CREATE_INFO;
    queryInfo.pNext = NULL;
    queryInfo.flags = 0;
    queryInfo.queryType = VK_QUERY_TYPE_PIPELINE_STATISTICS;
    queryInfo.queryCount = queryCount;
    queryInfo.pipelineStatistics = VK_QUERY_PIPELINE_STATISTIC_INPUT_ASSEMBLY_VERTICES_BIT |
        VK_QUERY_PIPELINE_STATISTIC_INPUT_ASSEMBLY_PRIMITIVES_BIT |
        VK_QUERY_PIPELINE_STATISTIC_VERTEX_SHADER_INVOCATIONS_BIT |
        VK_QUERY_PIPELINE_STATISTIC_GEOMETRY_SHADER_INVOCATIONS_BIT |
        VK_QUERY_PIPELINE_STATISTIC_GEOMETRY_SHADER_PRIMITIVES_BIT |
        VK_QUERY_PIPELINE_STATISTIC_CLIPPING_INVOCATIONS_BIT |
        VK_QUERY_PIPELINE_STATISTIC_CLIPPING_PRIMITIVES_BIT |
        VK_QUERY_PIPELINE_STATISTIC_FRAGMENT_SHADER_INVOCATIONS_BIT |
        VK_QUERY_PIPELINE_STATISTIC_TESSELLATION_CONTROL_SHADER_PATCHES_BIT |
        VK_QUERY_PIPELINE_STATISTIC_TESSELLATION_EVALUATION_SHADER_INVOCATIONS_BIT;

    if ( !pDevice->ToData()->supportedFeaturesBitArray.test( VkDeviceFeature_PipelineStatisticsQuery ) )
    {
        *pOutQueryPool = NULL;
    }
    else
    {
        VkResult result;
        VkDevice device = CastToVkDispatchableObject< VkDevice >( pDevice->ToData()->hDevice );
        const VkAllocationCallbacks* pAllocator =
            static_cast< const VkAllocationCallbacks* >( pDevice->ToData()->pAllocationCallback.ptr );
        NN_GFX_CALL_VK_FUNCTION( result = vkCreateQueryPool(
            device, &queryInfo, pAllocator, pOutQueryPool ) );
        NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
        NN_UNUSED( result );
    }
}

void Vk::DestroyQueryPool( DeviceImpl< ApiVariationVk1 >* pDevice,
    VkQueryPool queryPool ) NN_NOEXCEPT
{
    VkDevice device = CastToVkDispatchableObject< VkDevice >( pDevice->ToData()->hDevice );
    const VkAllocationCallbacks* pAllocator =
        static_cast< const VkAllocationCallbacks* >( pDevice->ToData()->pAllocationCallback.ptr );
    NN_GFX_CALL_VK_FUNCTION( vkDestroyQueryPool( device, queryPool, pAllocator ) );
}

VkQueryType Vk::GetQueryType( QueryTarget queryTarget ) NN_NOEXCEPT
{
    VkQueryType vkQueryType;

    switch ( queryTarget )
    {
    case QueryTarget_Timestamp:
        {
            vkQueryType = VK_QUERY_TYPE_TIMESTAMP;
        }
        break;
    case QueryTarget_SamplesPassed:
        {
            vkQueryType = VK_QUERY_TYPE_OCCLUSION;
        }
        break;
    case QueryTarget_InputVertices:
    case QueryTarget_InputPrimitives:
    case QueryTarget_VertexShaderInvocations:
    case QueryTarget_GeometryShaderInvocations:
    case QueryTarget_GeometryShaderPrimitives:
    case QueryTarget_ClippingInputPrimitives:
    case QueryTarget_ClippingOutputPrimitives:
    case QueryTarget_PixelShaderInvocations:
    case QueryTarget_HullShaderInvocations:
    case QueryTarget_DomainShaderInvocations:
    case QueryTarget_ComputeShaderInvocations:
        {
            vkQueryType = VK_QUERY_TYPE_PIPELINE_STATISTICS;
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    return vkQueryType;
}

VkQueryPipelineStatisticFlags Vk::GetQueryPipelineStatisticFlag( QueryTarget queryTarget ) NN_NOEXCEPT
{
    VkQueryPipelineStatisticFlags pipelineStatisticFlags = 0;

    switch ( queryTarget )
    {
    case QueryTarget_Timestamp:
    case QueryTarget_SamplesPassed:
        break;
    case QueryTarget_InputVertices:
        {
            pipelineStatisticFlags = VK_QUERY_PIPELINE_STATISTIC_INPUT_ASSEMBLY_VERTICES_BIT;
        }
        break;
    case QueryTarget_InputPrimitives:
        {
            pipelineStatisticFlags = VK_QUERY_PIPELINE_STATISTIC_INPUT_ASSEMBLY_PRIMITIVES_BIT;;
        }
        break;
    case QueryTarget_VertexShaderInvocations:
        {
            pipelineStatisticFlags = VK_QUERY_PIPELINE_STATISTIC_VERTEX_SHADER_INVOCATIONS_BIT;
        }
        break;
    case QueryTarget_GeometryShaderInvocations:
        {
            pipelineStatisticFlags = VK_QUERY_PIPELINE_STATISTIC_GEOMETRY_SHADER_INVOCATIONS_BIT;
        }
        break;
    case QueryTarget_GeometryShaderPrimitives:
        {
            pipelineStatisticFlags = VK_QUERY_PIPELINE_STATISTIC_GEOMETRY_SHADER_PRIMITIVES_BIT;
        }
        break;
    case QueryTarget_ClippingInputPrimitives:
        {
            pipelineStatisticFlags = VK_QUERY_PIPELINE_STATISTIC_CLIPPING_INVOCATIONS_BIT;
        }
        break;
    case QueryTarget_ClippingOutputPrimitives:
        {
            pipelineStatisticFlags = VK_QUERY_PIPELINE_STATISTIC_CLIPPING_PRIMITIVES_BIT;
        }
        break;
    case QueryTarget_PixelShaderInvocations:
        {
            pipelineStatisticFlags = VK_QUERY_PIPELINE_STATISTIC_FRAGMENT_SHADER_INVOCATIONS_BIT;
        }
        break;
    case QueryTarget_HullShaderInvocations:
        {
            pipelineStatisticFlags = VK_QUERY_PIPELINE_STATISTIC_TESSELLATION_CONTROL_SHADER_PATCHES_BIT;
        }
        break;
    case QueryTarget_DomainShaderInvocations:
        {
            pipelineStatisticFlags = VK_QUERY_PIPELINE_STATISTIC_TESSELLATION_EVALUATION_SHADER_INVOCATIONS_BIT;
        }
        break;
    case QueryTarget_ComputeShaderInvocations:
        {
            pipelineStatisticFlags = VK_QUERY_PIPELINE_STATISTIC_COMPUTE_SHADER_INVOCATIONS_BIT;
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    return pipelineStatisticFlags;
}

void Vk::CreateComputePipeline(
    VkPipeline* pOutPipeline,
    DeviceImpl< ApiVariationVk1 >* pDevice, const GfxVkGpuState& gpuState,
    VkPipelineLayout pipelineLayout ) NN_NOEXCEPT
{
    VkComputePipelineCreateInfo createInfo = {};
    createInfo.sType = VK_STRUCTURE_TYPE_COMPUTE_PIPELINE_CREATE_INFO;
    createInfo.stage.sType = VK_STRUCTURE_TYPE_PIPELINE_SHADER_STAGE_CREATE_INFO;
    createInfo.stage.stage = VK_SHADER_STAGE_COMPUTE_BIT;
    createInfo.stage.module = CastToVkNonDispatchableObject< VkShaderModule >(
        gpuState.constSizedPipelineState.shaderState.shaderModule[ ShaderStage_Compute ] );
    createInfo.stage.pName = "main";
    createInfo.layout = pipelineLayout;
    createInfo.basePipelineHandle = VK_NULL_HANDLE;
    createInfo.basePipelineIndex = -1;

    VkResult result;
    VkDevice device = CastToVkDispatchableObject< VkDevice >( pDevice->ToData()->hDevice );
    VkPipelineCache pipelineCache =
        CastToVkNonDispatchableObject< VkPipelineCache >( pDevice->ToData()->hPipelineCache );
    const VkAllocationCallbacks* pAllocator =
        static_cast< const VkAllocationCallbacks* >( pDevice->ToData()->pAllocationCallback.ptr );
    NN_GFX_CALL_VK_FUNCTION( result = vkCreateComputePipelines(
        device, pipelineCache, 1, &createInfo, pAllocator, pOutPipeline ) );
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
    NN_UNUSED( result );
}

void Vk::CreateGraphicsPipeline(
    VkPipeline* pOutPipeline,
    const DeviceImpl< ApiVariationVk1 >* pDevice, const GfxVkGpuState& gpuState,
    VkPipelineLayout pipelineLayout, VkRenderPass renderPass ) NN_NOEXCEPT
{
    int shaderStageCount = 0;
    for ( auto idxShaderStage = 0; idxShaderStage < ShaderStage_End; ++idxShaderStage )
    {
        if ( gpuState.constSizedPipelineState.shaderState.shaderModule[ idxShaderStage ] != 0 )
        {
            ++shaderStageCount;
        }
    }
    int vertexInputBindingCount = gpuState.constSizedPipelineState.vertexInputState.bufferStateCount;
    int vertexInputAttributeCount = gpuState.constSizedPipelineState.vertexInputState.attributeStateCount;
    int blendAttachmentCount = gpuState.constSizedPipelineState.blendState.blendTargetCount;
    size_t temporalStateSize = shaderStageCount * sizeof( VkPipelineShaderStageCreateInfo )
        + vertexInputBindingCount * sizeof( VkVertexInputBindingDescription )
        + vertexInputAttributeCount * sizeof( VkVertexInputAttributeDescription )
        + blendAttachmentCount * sizeof( VkPipelineColorBlendAttachmentState );
    void* pTemporalData = Vk::AllocDriverMemory( temporalStateSize, 8 );
    NN_SDK_ASSERT_NOT_NULL( pTemporalData );
    nn::util::BytePtr ptr( pTemporalData );
    VkPipelineShaderStageCreateInfo* pStages =
        ptr.Get< VkPipelineShaderStageCreateInfo >();
    ptr.Advance( sizeof( VkPipelineShaderStageCreateInfo ) * shaderStageCount );
    VkVertexInputBindingDescription* pVertexInputBinding =
        ptr.Get< VkVertexInputBindingDescription >();
    ptr.Advance( sizeof( VkVertexInputBindingDescription ) * vertexInputBindingCount );
    VkVertexInputAttributeDescription* pVertexInputAttribute =
        ptr.Get< VkVertexInputAttributeDescription >();
    ptr.Advance( sizeof( VkVertexInputAttributeDescription ) * vertexInputAttributeCount );
    VkPipelineColorBlendAttachmentState* pColorBlendAttachmentState =
        ptr.Get< VkPipelineColorBlendAttachmentState >();

    SetupPipelineShaderStageCreateInfo( pStages, gpuState );

    VkPipelineVertexInputStateCreateInfo vertexInputState;
    SetupPipelineVertexInputStateCreateInfo(
        &vertexInputState, pVertexInputBinding, gpuState.constSizedPipelineState.vertexInputState.bufferStateCount,
        pVertexInputAttribute, gpuState.constSizedPipelineState.vertexInputState.attributeStateCount, gpuState );

    VkPipelineInputAssemblyStateCreateInfo inputAssemblyState;
    SetupPipelineInputAssemblyStateCreateInfo(
        &inputAssemblyState, gpuState );

    VkPipelineTessellationStateCreateInfo tessellationState;
    SetupPipelineTessellationStateCreateInfo(
        &tessellationState, gpuState );

    VkPipelineViewportStateCreateInfo viewportState;
    SetupPipelineViewportStateCreateInfo(
        &viewportState, gpuState );

    VkPipelineRasterizationStateCreateInfo rasterizationState;
    SetupPipelineRasterizationStateCreateInfo(
        &rasterizationState, gpuState );

    VkPipelineMultisampleStateCreateInfo multisampleState;
    VkSampleMask sampleMask;
    SetupPipelineMultisampleStateCreateInfo(
        &multisampleState, &sampleMask, gpuState );

    VkPipelineDepthStencilStateCreateInfo depthStencilState;
    SetupPipelineDepthStencilStateCreateInfo( pDevice,
        &depthStencilState, gpuState );

    VkPipelineColorBlendStateCreateInfo colorBlendState;
    SetupPipelineColorBlendStateCreateInfo(
        &colorBlendState, pColorBlendAttachmentState,
        /*gpuState.staticState.blendState.blendTargetCount*/
        gpuState.renderTargetState.colorAttachmentCount, gpuState );
    // SetRenderTargetsのcolorAttachmentCountとSetBlendStateのBlendTargetCountが異なる場合
    // SetRenderTargetsの設定値を優先します。（Vulkanとしては、Pipelineの設定はRenderPassの設定に合わせる必要があります。）

    VkPipelineDynamicStateCreateInfo dynamicState;
    SetupPipelineDynamicStateCreateInfo( &dynamicState );

    VkGraphicsPipelineCreateInfo info;
    info.sType = VK_STRUCTURE_TYPE_GRAPHICS_PIPELINE_CREATE_INFO;
    info.pNext = NULL;
    info.flags = 0;
    info.stageCount = shaderStageCount;
    info.pStages = pStages;
    info.pVertexInputState = &vertexInputState;
    info.pInputAssemblyState = &inputAssemblyState;
    info.pTessellationState = &tessellationState;
    info.pViewportState = &viewportState;
    info.pRasterizationState = &rasterizationState;
    info.pMultisampleState = &multisampleState;
    info.pDepthStencilState = &depthStencilState;
    info.pColorBlendState = &colorBlendState;
    info.pDynamicState = &dynamicState;
    info.layout = pipelineLayout;
    info.renderPass = renderPass;
    info.subpass = 0;
    info.basePipelineHandle = VK_NULL_HANDLE;
    info.basePipelineIndex = -1;

    VkResult result;
    VkDevice device = CastToVkDispatchableObject< VkDevice >( pDevice->ToData()->hDevice );
    const VkAllocationCallbacks* pAllocator =
        static_cast< const VkAllocationCallbacks* >( pDevice->ToData()->pAllocationCallback.ptr );
    VkPipelineCache pipelineCache =
        CastToVkNonDispatchableObject< VkPipelineCache >( pDevice->ToData()->hPipelineCache );
    NN_GFX_CALL_VK_FUNCTION( result = vkCreateGraphicsPipelines(
        device, pipelineCache, 1, &info, pAllocator, pOutPipeline ) );
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
    NN_UNUSED( result );

    Vk::FreeDriverMemory( pTemporalData );
}

void Vk::CreateRenderPass( VkRenderPass* pOutRenderPass,
    DeviceImpl< ApiVariationVk1 >* pDevice, const GfxVkGpuState& gpuState ) NN_NOEXCEPT
{
    int colorAttachmentCount = gpuState.renderTargetState.colorAttachmentCount;
    bool isDepthAttached = gpuState.renderTargetState.flag.GetBit( GfxVkGpuState::RenderTargetState::Flag_DepthAttached );
    int maxAttachmentCount = colorAttachmentCount + ( isDepthAttached ? 1 : 0 );
    int depthAttachmentIndex = colorAttachmentCount;

    size_t temporalStateSize = ( sizeof( VkAttachmentDescription ) + sizeof( VkAttachmentReference ) )
        * maxAttachmentCount;
    void* pTemporalData = Vk::AllocDriverMemory( temporalStateSize, 8 );
    NN_SDK_ASSERT_NOT_NULL( pTemporalData );
    nn::util::BytePtr ptr( pTemporalData );
    VkAttachmentDescription* pAttachments = ptr.Get< VkAttachmentDescription >();
    ptr.Advance( sizeof( VkAttachmentDescription ) * maxAttachmentCount );
    VkAttachmentReference* pAttachmentReferences = ptr.Get< VkAttachmentReference >();

    int colorReferenceCount = 0;
    int attachmentCount = 0;
    for ( int idxAttachment = 0; idxAttachment < colorAttachmentCount; ++idxAttachment )
    {
        bool isColorInUse = gpuState.renderTargetState.colorTargetView[ idxAttachment ] != VK_NULL_HANDLE &&
            gpuState.variableSizedPipelineState.blendTargetState[ idxAttachment ].colorMask != 0;
        if ( isColorInUse )
        {
            VkAttachmentDescription* pDstState = &pAttachments[ attachmentCount ];
            const GfxVkGpuState::RenderTargetState::AttachmentState* pSrcState =
                &gpuState.renderTargetState.colorAttachment[ idxAttachment ];
            pDstState->flags = VK_ATTACHMENT_DESCRIPTION_MAY_ALIAS_BIT;
            pDstState->format = static_cast< VkFormat >( pSrcState->format );
            pDstState->samples = static_cast< VkSampleCountFlagBits >( pSrcState->samples );
            pDstState->loadOp = static_cast< VkAttachmentLoadOp >( pSrcState->loadOp );
            pDstState->storeOp = VK_ATTACHMENT_STORE_OP_STORE;
            pDstState->stencilLoadOp = VK_ATTACHMENT_LOAD_OP_DONT_CARE;
            pDstState->stencilStoreOp = VK_ATTACHMENT_STORE_OP_DONT_CARE;
            pDstState->initialLayout = ( pSrcState->loadOp == VK_ATTACHMENT_LOAD_OP_LOAD ) ?
                Vk::LayoutColorAttachment : VK_IMAGE_LAYOUT_UNDEFINED;
            pDstState->finalLayout = Vk::LayoutColorAttachment;

            pAttachmentReferences[ colorReferenceCount ].attachment = attachmentCount;
            pAttachmentReferences[ colorReferenceCount ].layout = Vk::LayoutColorAttachment;
            attachmentCount++;
        }
        else
        {
            pAttachmentReferences[ colorReferenceCount ].attachment = VK_ATTACHMENT_UNUSED;
            pAttachmentReferences[ colorReferenceCount ].layout = Vk::LayoutColorAttachment;
        }
        colorReferenceCount++;
    }
    depthAttachmentIndex = colorReferenceCount;
    if ( isDepthAttached )
    {
        const GfxVkGpuState::RenderTargetState::AttachmentState* pSrcState =
            &gpuState.renderTargetState.depthStencilAttachment;
        auto& depthStencilState = gpuState.constSizedPipelineState.depthStencilState;
        bool isLoadOp = pSrcState->loadOp == VK_ATTACHMENT_LOAD_OP_LOAD;
        bool isDepthWriteEnabled = depthStencilState.flag.GetBit( GfxVkGpuState::ConstSizedPipelineState::DepthStencilState::Flag_DepthWriteEnable );
        VkImageLayout depthLayout = (pSrcState->loadOp == VK_ATTACHMENT_LOAD_OP_CLEAR || isDepthWriteEnabled) ? Vk::LayoutDepthStencilAttachment : Vk::LayoutTextureDepthStencilRead;

        VkAttachmentDescription* pDstState = &pAttachments[ attachmentCount ];
        pDstState->flags = VK_ATTACHMENT_DESCRIPTION_MAY_ALIAS_BIT;
        pDstState->format = static_cast< VkFormat >( pSrcState->format );
        pDstState->samples = static_cast< VkSampleCountFlagBits >( pSrcState->samples );
        pDstState->loadOp = static_cast< VkAttachmentLoadOp >( pSrcState->loadOp );
        pDstState->storeOp = VK_ATTACHMENT_STORE_OP_STORE;
        pDstState->stencilLoadOp = static_cast< VkAttachmentLoadOp >( pSrcState->loadOp );
        pDstState->stencilStoreOp = VK_ATTACHMENT_STORE_OP_STORE;
        pDstState->initialLayout = isLoadOp ? depthLayout : VK_IMAGE_LAYOUT_UNDEFINED;
        pDstState->finalLayout = depthLayout;

        pAttachmentReferences[ depthAttachmentIndex ].attachment = attachmentCount;
        pAttachmentReferences[ depthAttachmentIndex ].layout = depthLayout;
        attachmentCount++;
    }

    VkSubpassDescription subpass;
    subpass.flags = 0;
    subpass.pipelineBindPoint = VK_PIPELINE_BIND_POINT_GRAPHICS;
    subpass.inputAttachmentCount = 0;
    subpass.pInputAttachments = NULL;
    subpass.colorAttachmentCount = colorReferenceCount;
    subpass.pColorAttachments = pAttachmentReferences;
    subpass.pResolveAttachments = NULL;
    subpass.pDepthStencilAttachment = isDepthAttached ? &pAttachmentReferences[ depthAttachmentIndex ] : NULL;
    subpass.preserveAttachmentCount = 0;
    subpass.pPreserveAttachments = NULL;

    VkRenderPassCreateInfo info;
    info.sType = VK_STRUCTURE_TYPE_RENDER_PASS_CREATE_INFO;
    info.pNext = NULL;
    info.flags = 0;
    info.attachmentCount = attachmentCount;
    info.pAttachments = pAttachments;
    info.subpassCount = 1;
    info.pSubpasses = &subpass;
    info.dependencyCount = 0;
    info.pDependencies = NULL;

    VkResult result;
    VkDevice device = CastToVkDispatchableObject< VkDevice >( pDevice->ToData()->hDevice );
    const VkAllocationCallbacks* pAllocator =
        static_cast< const VkAllocationCallbacks* >( pDevice->ToData()->pAllocationCallback.ptr );
    NN_GFX_CALL_VK_FUNCTION( result = vkCreateRenderPass( device, &info, pAllocator, pOutRenderPass ) );
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
    NN_UNUSED( result );

    Vk::FreeDriverMemory( pTemporalData );
}

void Vk::CreateDescriptorSetLayout( VkDescriptorSetLayout* pOutDescriptorSetLayout,
    DeviceImpl< ApiVariationVk1 >* pDevice, const GfxVkGpuState& gpuState ) NN_NOEXCEPT
{
    /**
    * descriptor set layout and pipeline layout must be created even if no descriptor binding.
    * vkCreateGraphicsPipeline occurs exception with VK_NULL_HANDLE for pipelineLayout,
    * or with valid pipelineLayout created with 0 for setLayoutCount.
    */
    VkDescriptorSetLayoutBinding* pBinding = NULL;

    uint64_t validSlotMask = gpuState.descriptorSetLayoutState.validSlotMask;
    uint32_t idxAvailableBinding = 0;
    if ( gpuState.descriptorSetLayoutState.availableDescriptorCount != 0 )
    {
        size_t temporalStateSize = sizeof( VkDescriptorSetLayoutBinding )
            * gpuState.descriptorSetLayoutState.availableDescriptorCount;
        pBinding = static_cast< VkDescriptorSetLayoutBinding* >( Vk::AllocDriverMemory( temporalStateSize, 8 ) );

        for ( uint32_t idxSlotState = 0; idxSlotState <=
            gpuState.descriptorSetLayoutState.maxDescriptorIndex; ++idxSlotState )
        {
            const GfxVkGpuState::DescriptorSetLayoutState::SlotState* pSrcState =
                &gpuState.descriptorSetLayoutState.slotState[ idxSlotState ];

            if ( !pSrcState->flag.GetBit( GfxVkGpuState::DescriptorSetLayoutState::SlotState::Flag_IsSet )
                || !( validSlotMask & ( 1ULL << idxSlotState ) ) )
            {
                continue;
            }

            VkDescriptorSetLayoutBinding* pDstState = &pBinding[ idxAvailableBinding ];
            pDstState->binding = idxSlotState;
            pDstState->descriptorType = static_cast< VkDescriptorType >( pSrcState->type );
            pDstState->descriptorCount = 1;
            pDstState->stageFlags = static_cast< VkShaderStageFlags >( pSrcState->shaderStage );
            pDstState->pImmutableSamplers = NULL;

            ++idxAvailableBinding;
        }
    }

    VkDescriptorSetLayoutCreateInfo descriptorSetLayoutInfo;
    descriptorSetLayoutInfo.sType = VK_STRUCTURE_TYPE_DESCRIPTOR_SET_LAYOUT_CREATE_INFO;
    descriptorSetLayoutInfo.pNext = NULL;
    descriptorSetLayoutInfo.flags = 0;
    descriptorSetLayoutInfo.bindingCount = idxAvailableBinding;
    descriptorSetLayoutInfo.pBindings = pBinding;

    VkResult result;
    VkDevice device = CastToVkDispatchableObject< VkDevice >( pDevice->ToData()->hDevice );
    const VkAllocationCallbacks* pAllocator =
        static_cast< const VkAllocationCallbacks* >( pDevice->ToData()->pAllocationCallback.ptr );

    NN_GFX_CALL_VK_FUNCTION( result = vkCreateDescriptorSetLayout(
        device, &descriptorSetLayoutInfo, pAllocator, pOutDescriptorSetLayout ) );
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );

    if ( pBinding != NULL )
    {
        Vk::FreeDriverMemory( pBinding );
    }
}

void Vk::CreatePipelineLayout( VkPipelineLayout* pOutPipelineLayout,
    VkDescriptorSetLayout descriptorSetLayout,
    DeviceImpl< ApiVariationVk1 >* pDevice ) NN_NOEXCEPT
{
    VkPipelineLayoutCreateInfo pipelineLayoutInfo;
    pipelineLayoutInfo.sType = VK_STRUCTURE_TYPE_PIPELINE_LAYOUT_CREATE_INFO;
    pipelineLayoutInfo.pNext = NULL;
    pipelineLayoutInfo.flags = 0;
    pipelineLayoutInfo.setLayoutCount = 1;
    pipelineLayoutInfo.pSetLayouts = &descriptorSetLayout;
    pipelineLayoutInfo.pushConstantRangeCount = 0;
    pipelineLayoutInfo.pPushConstantRanges = NULL;

    VkResult result;
    VkDevice device = CastToVkDispatchableObject< VkDevice >( pDevice->ToData()->hDevice );
    const VkAllocationCallbacks* pAllocator =
        static_cast< const VkAllocationCallbacks* >( pDevice->ToData()->pAllocationCallback.ptr );
    NN_GFX_CALL_VK_FUNCTION( result = vkCreatePipelineLayout(
        device, &pipelineLayoutInfo, pAllocator, pOutPipelineLayout ) );
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
    NN_UNUSED( result );
}

void Vk::CreateDescriptorSet( VkDescriptorPool* pOutDescriptorPool,
    VkDescriptorSet* pOutDescriptorSet,
    DeviceImpl< ApiVariationVk1 >* pDevice, const GfxVkGpuState& gpuState,
    VkDescriptorSetLayout descriptorSetLayout ) NN_NOEXCEPT
{
    int totalDescriptorSlot = gpuState.descriptorSetLayoutState.availableDescriptorCount;

    if ( totalDescriptorSlot == 0 )
    {
        *pOutDescriptorPool = VK_NULL_HANDLE;
        *pOutDescriptorSet = VK_NULL_HANDLE;
        return;
    }

    size_t temporalStateSize = sizeof( VkDescriptorPoolSize ) * totalDescriptorSlot;
    void* pTemporalData = Vk::AllocDriverMemory( temporalStateSize, 8 );
    NN_SDK_ASSERT( pTemporalData );
    nn::util::BytePtr ptr( pTemporalData );
    VkDescriptorPoolSize* pDescriptorPoolSize = ptr.Get< VkDescriptorPoolSize >();

    uint64_t validSlotMask = gpuState.descriptorSetLayoutState.validSlotMask;
    uint32_t idxAvailableBinding = 0;
    for ( uint32_t idxSlotState = 0; idxSlotState <=
        gpuState.descriptorSetLayoutState.maxDescriptorIndex; ++idxSlotState )
    {
        const GfxVkGpuState::DescriptorSetLayoutState::SlotState* pSrcState =
            &gpuState.descriptorSetLayoutState.slotState[ idxSlotState ];

        if ( !pSrcState->flag.GetBit( GfxVkGpuState::DescriptorSetLayoutState::SlotState::Flag_IsSet )
             || !( validSlotMask & ( 1ULL << idxSlotState ) ) )
        {
            continue;
        }

        {
            VkDescriptorPoolSize* pDstState = &pDescriptorPoolSize[ idxAvailableBinding ];

            pDstState->type = static_cast< VkDescriptorType >( pSrcState->type );
            pDstState->descriptorCount = 1;
        }

        ++idxAvailableBinding;
    }

    VkDescriptorPoolCreateInfo descriptorPoolInfo;
    descriptorPoolInfo.sType = VK_STRUCTURE_TYPE_DESCRIPTOR_POOL_CREATE_INFO;
    descriptorPoolInfo.pNext = NULL;
    descriptorPoolInfo.flags = VK_DESCRIPTOR_POOL_CREATE_FREE_DESCRIPTOR_SET_BIT;
    descriptorPoolInfo.maxSets = 1;
    descriptorPoolInfo.poolSizeCount = idxAvailableBinding;
    descriptorPoolInfo.pPoolSizes = pDescriptorPoolSize;

    VkResult result;
    VkDevice device = CastToVkDispatchableObject< VkDevice >( pDevice->ToData()->hDevice );
    const VkAllocationCallbacks* pAllocator =
        static_cast< const VkAllocationCallbacks* >( pDevice->ToData()->pAllocationCallback.ptr );
    NN_GFX_CALL_VK_FUNCTION( result = vkCreateDescriptorPool(
        device, &descriptorPoolInfo, pAllocator, pOutDescriptorPool ) );
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );

    VkDescriptorSetAllocateInfo allocateInfo;
    allocateInfo.sType = VK_STRUCTURE_TYPE_DESCRIPTOR_SET_ALLOCATE_INFO;
    allocateInfo.pNext = NULL;
    allocateInfo.descriptorPool = *pOutDescriptorPool;
    allocateInfo.descriptorSetCount = 1;
    allocateInfo.pSetLayouts = &descriptorSetLayout;
    NN_GFX_CALL_VK_FUNCTION( result = vkAllocateDescriptorSets(
        device, &allocateInfo, pOutDescriptorSet ) );
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
    NN_UNUSED( result );

    Vk::FreeDriverMemory( pTemporalData );
} //NOLINT(impl/function_size)

void Vk::CreateFramebuffer( VkFramebuffer* pOutFramebuffer,
    DeviceImpl< ApiVariationVk1 >* pDevice, const GfxVkGpuState& gpuState,
    VkRenderPass renderPass ) NN_NOEXCEPT
{
    int colorAttachmentCount = gpuState.renderTargetState.colorAttachmentCount;
    bool isDepthAttached = gpuState.renderTargetState.flag.GetBit(
        GfxVkGpuState::RenderTargetState::Flag_DepthAttached ) ? true : false;

    VkImageView pImageView[ GfxVkGpuState::maxColorAttachments + 1 ];

    int attachmentCount = 0;
    for ( int idxAttachment = 0; idxAttachment < colorAttachmentCount; ++idxAttachment )
    {
        bool isColorInUse = gpuState.renderTargetState.colorTargetView[ idxAttachment ] != VK_NULL_HANDLE &&
            gpuState.variableSizedPipelineState.blendTargetState[ idxAttachment ].colorMask != 0;
        if ( isColorInUse )
        {
            pImageView[ attachmentCount++ ] = CastToVkNonDispatchableObject< VkImageView >(
                gpuState.renderTargetState.colorTargetView[ idxAttachment ] );
        }
    }
    if ( isDepthAttached )
    {
        pImageView[ attachmentCount++ ] = CastToVkNonDispatchableObject< VkImageView >(
            gpuState.renderTargetState.depthStencilTargetView );
    }

    VkFramebufferCreateInfo info;
    info.sType = VK_STRUCTURE_TYPE_FRAMEBUFFER_CREATE_INFO;
    info.pNext = NULL;
    info.flags = 0;
    info.renderPass = renderPass;
    info.attachmentCount = attachmentCount;
    info.pAttachments = pImageView;
    info.width = gpuState.renderTargetState.width;
    info.height = gpuState.renderTargetState.height;
    info.layers = gpuState.renderTargetState.layers;

    VkResult result;
    VkDevice device = CastToVkDispatchableObject< VkDevice >( pDevice->ToData()->hDevice );
    const VkAllocationCallbacks* pAllocator =
        static_cast< const VkAllocationCallbacks* >( pDevice->ToData()->pAllocationCallback.ptr );
    NN_GFX_CALL_VK_FUNCTION( result = vkCreateFramebuffer(
        device, &info, pAllocator, pOutFramebuffer ) );
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
    NN_UNUSED( result );
}

void Vk::SetupPipelineShaderStageCreateInfo(
    VkPipelineShaderStageCreateInfo* pOutStages, const GfxVkGpuState& gpuState ) NN_NOEXCEPT
{
    int currentStage = 0;
    for ( int idxStage = 0; idxStage < ShaderStage_End; ++idxStage )
    {
        if ( gpuState.constSizedPipelineState.shaderState.shaderModule[ idxStage ] == 0 )
            continue;

        static const VkShaderStageFlagBits stageBit[ ShaderStage_End ] =
        {
            VK_SHADER_STAGE_VERTEX_BIT,
            VK_SHADER_STAGE_TESSELLATION_CONTROL_BIT,
            VK_SHADER_STAGE_TESSELLATION_EVALUATION_BIT,
            VK_SHADER_STAGE_GEOMETRY_BIT,
            VK_SHADER_STAGE_FRAGMENT_BIT,
            VK_SHADER_STAGE_COMPUTE_BIT
        };
        static const char* EntrypointName = "main";

        VkPipelineShaderStageCreateInfo* pInfo = &pOutStages[ currentStage ];
        pInfo->sType = VK_STRUCTURE_TYPE_PIPELINE_SHADER_STAGE_CREATE_INFO;
        pInfo->pNext = NULL;
        pInfo->flags = 0;
        pInfo->stage = stageBit[ idxStage ];
        pInfo->module = CastToVkNonDispatchableObject< VkShaderModule >(
            gpuState.constSizedPipelineState.shaderState.shaderModule[ idxStage ] );
        pInfo->pName = EntrypointName;
        pInfo->pSpecializationInfo = NULL;

        ++currentStage;
    }
}

void Vk::SetupPipelineVertexInputStateCreateInfo(
    VkPipelineVertexInputStateCreateInfo* pOutVertexInputState,
    VkVertexInputBindingDescription pOutVertexInputBinding[], int vertexInputBindingCount,
    VkVertexInputAttributeDescription pOutVertexInputAttribute[], int vertexInputAttributeCount,
    const GfxVkGpuState& gpuState ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL( gpuState.constSizedPipelineState.vertexInputState.bufferStateCount,
        static_cast< uint32_t >( vertexInputBindingCount ) );
    NN_SDK_ASSERT_EQUAL( gpuState.constSizedPipelineState.vertexInputState.attributeStateCount,
        static_cast< uint32_t >( vertexInputAttributeCount ) );

    pOutVertexInputState->sType = VK_STRUCTURE_TYPE_PIPELINE_VERTEX_INPUT_STATE_CREATE_INFO;
    pOutVertexInputState->pNext = NULL;
    pOutVertexInputState->flags = 0;

    for ( int idxBuffer = 0; idxBuffer < vertexInputBindingCount; ++idxBuffer)
    {
        VkVertexInputBindingDescription* pVkBindingDescription = &pOutVertexInputBinding[ idxBuffer ];
        const GfxVkGpuState::VariableSizedPipelineState::VertexBufferState* pBufferState =
            &gpuState.variableSizedPipelineState.vertexBufferState[ idxBuffer ];
        pVkBindingDescription->binding = idxBuffer;
        pVkBindingDescription->stride = pBufferState->stride;
        pVkBindingDescription->inputRate = pBufferState->divisor == 0
            ? VK_VERTEX_INPUT_RATE_VERTEX : VK_VERTEX_INPUT_RATE_INSTANCE;
    }
    pOutVertexInputState->vertexBindingDescriptionCount = vertexInputBindingCount;
    pOutVertexInputState->pVertexBindingDescriptions = pOutVertexInputBinding;

    for ( int idxAttribute = 0; idxAttribute < vertexInputAttributeCount; ++idxAttribute )
    {
        VkVertexInputAttributeDescription* pVkAttributeDescription = &pOutVertexInputAttribute[ idxAttribute ];
        const GfxVkGpuState::VariableSizedPipelineState::VertexAttributeState* pAttributeState =
            &gpuState.variableSizedPipelineState.vertexAttributeState[ idxAttribute ];
        pVkAttributeDescription->location = pAttributeState->location;
        pVkAttributeDescription->binding = pAttributeState->bindingIndex;
        pVkAttributeDescription->format = static_cast< VkFormat >( pAttributeState->format );
        pVkAttributeDescription->offset = pAttributeState->offset;
    }
    pOutVertexInputState->vertexAttributeDescriptionCount = vertexInputAttributeCount;
    pOutVertexInputState->pVertexAttributeDescriptions = pOutVertexInputAttribute;
}

void Vk::SetupPipelineInputAssemblyStateCreateInfo(
    VkPipelineInputAssemblyStateCreateInfo* pOutInputAssemblyState,
    const GfxVkGpuState& gpuState ) NN_NOEXCEPT
{
    pOutInputAssemblyState->sType = VK_STRUCTURE_TYPE_PIPELINE_INPUT_ASSEMBLY_STATE_CREATE_INFO;
    pOutInputAssemblyState->pNext = NULL;
    pOutInputAssemblyState->flags = 0;
    pOutInputAssemblyState->topology = static_cast< VkPrimitiveTopology >(
        gpuState.constSizedPipelineState.inputAssemblyState.primitiveTopology );
    pOutInputAssemblyState->primitiveRestartEnable = VK_FALSE;
}

void Vk::SetupPipelineTessellationStateCreateInfo(
    VkPipelineTessellationStateCreateInfo* pOutTessellationState,
    const GfxVkGpuState& gpuState ) NN_NOEXCEPT
{
    pOutTessellationState->sType = VK_STRUCTURE_TYPE_PIPELINE_TESSELLATION_STATE_CREATE_INFO;
    pOutTessellationState->pNext = NULL;
    pOutTessellationState->flags = 0;
    pOutTessellationState->patchControlPoints = gpuState.constSizedPipelineState.tessellationState.patchControlPoints;
}

void Vk::SetupPipelineViewportStateCreateInfo(
    VkPipelineViewportStateCreateInfo* pOutViewportState,
    const GfxVkGpuState& gpuState ) NN_NOEXCEPT
{
    pOutViewportState->sType = VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_STATE_CREATE_INFO;
    pOutViewportState->pNext = NULL;
    pOutViewportState->flags = 0;
    pOutViewportState->viewportCount = gpuState.constSizedPipelineState.viewportState.viewportCount;
    pOutViewportState->scissorCount = gpuState.constSizedPipelineState.viewportState.viewportCount;
    pOutViewportState->pViewports = NULL; // dynamic state
    pOutViewportState->pScissors = NULL; // dynamic state
}

void Vk::SetupPipelineRasterizationStateCreateInfo(
    VkPipelineRasterizationStateCreateInfo* pOutRasterizationState,
    const GfxVkGpuState& gpuState ) NN_NOEXCEPT
{
    pOutRasterizationState->sType = VK_STRUCTURE_TYPE_PIPELINE_RASTERIZATION_STATE_CREATE_INFO;
    pOutRasterizationState->pNext = NULL;
    pOutRasterizationState->flags = 0;

    const GfxVkGpuState::ConstSizedPipelineState::RasterizationState* pSrcState =
        &gpuState.constSizedPipelineState.rasterizationState;
    pOutRasterizationState->depthClampEnable = pSrcState->flag.GetBit(
        GfxVkGpuState::ConstSizedPipelineState::RasterizationState::Flag_DepthClampEnable ) ? VK_TRUE : VK_FALSE;
    pOutRasterizationState->rasterizerDiscardEnable = pSrcState->flag.GetBit(
        GfxVkGpuState::ConstSizedPipelineState::RasterizationState::Flag_RasterDisable ) ? VK_TRUE : VK_FALSE;
    pOutRasterizationState->polygonMode = static_cast< VkPolygonMode >( pSrcState->polygonMode );
    pOutRasterizationState->cullMode = static_cast< VkCullModeFlags >( pSrcState->cullFace );
    pOutRasterizationState->frontFace = static_cast< VkFrontFace >( pSrcState->frontFace );
    pOutRasterizationState->depthBiasEnable = pSrcState->flag.GetBit(
        GfxVkGpuState::ConstSizedPipelineState::RasterizationState::Flag_PolygonOffsetEnable ) ? VK_TRUE : VK_FALSE;
    pOutRasterizationState->depthBiasConstantFactor = 0.0; // dynamic state
    pOutRasterizationState->depthBiasClamp = 0.0; // dynamic state
    pOutRasterizationState->depthBiasSlopeFactor = 0.0; // dynamic state
    pOutRasterizationState->lineWidth = 1.0; // In case dynamic state is disabled
}

void Vk::SetupPipelineMultisampleStateCreateInfo(
    VkPipelineMultisampleStateCreateInfo* pOutMultisampleState,
    VkSampleMask* pSampleMask, const GfxVkGpuState& gpuState ) NN_NOEXCEPT
{
    pOutMultisampleState->sType = VK_STRUCTURE_TYPE_PIPELINE_MULTISAMPLE_STATE_CREATE_INFO;
    pOutMultisampleState->pNext = NULL;
    pOutMultisampleState->flags = 0;
    pOutMultisampleState->rasterizationSamples = static_cast< VkSampleCountFlagBits >(
        gpuState.constSizedPipelineState.rasterizationState.sampleCount );
    pOutMultisampleState->sampleShadingEnable = VK_FALSE;
    pOutMultisampleState->minSampleShading = 0.0;
    pOutMultisampleState->alphaToCoverageEnable = gpuState.constSizedPipelineState.rasterizationState.flag.GetBit(
        GfxVkGpuState::ConstSizedPipelineState::RasterizationState::Flag_AlphaToCoverageEnable ) ? VK_TRUE : VK_FALSE;
    pOutMultisampleState->alphaToOneEnable = gpuState.constSizedPipelineState.rasterizationState.flag.GetBit(
        GfxVkGpuState::ConstSizedPipelineState::RasterizationState::Flag_AlphaToOneEnable ) ? VK_TRUE : VK_FALSE;
    *pSampleMask = gpuState.constSizedPipelineState.rasterizationState.sampleMask;
    pOutMultisampleState->pSampleMask = pSampleMask;
}

void Vk::SetupPipelineDepthStencilStateCreateInfo(
    const DeviceImpl< ApiVariationVk1 >* pDevice,
    VkPipelineDepthStencilStateCreateInfo* pOutDepthStencilState,
    const GfxVkGpuState& gpuState ) NN_NOEXCEPT
{
    pOutDepthStencilState->sType = VK_STRUCTURE_TYPE_PIPELINE_DEPTH_STENCIL_STATE_CREATE_INFO;
    pOutDepthStencilState->pNext = NULL;
    pOutDepthStencilState->flags = 0;
    pOutDepthStencilState->depthTestEnable = gpuState.constSizedPipelineState.depthStencilState.flag.GetBit(
        GfxVkGpuState::ConstSizedPipelineState::DepthStencilState::Flag_DepthTestEnable ) ? VK_TRUE : VK_FALSE;
    pOutDepthStencilState->depthWriteEnable = gpuState.constSizedPipelineState.depthStencilState.flag.GetBit(
        GfxVkGpuState::ConstSizedPipelineState::DepthStencilState::Flag_DepthWriteEnable ) ? VK_TRUE : VK_FALSE;
    pOutDepthStencilState->depthCompareOp = static_cast< VkCompareOp >(
        gpuState.constSizedPipelineState.depthStencilState.depthFunc );
    pOutDepthStencilState->depthBoundsTestEnable = gpuState.constSizedPipelineState.depthStencilState.flag.GetBit(
        GfxVkGpuState::ConstSizedPipelineState::DepthStencilState::Flag_DepthBoundsTestEnable ) &&
        pDevice->ToData()->availableExtensions.GetBit( VkDeviceFeature_DepthBounds ) ? VK_TRUE : VK_FALSE;
    pOutDepthStencilState->stencilTestEnable = gpuState.constSizedPipelineState.depthStencilState.flag.GetBit(
        GfxVkGpuState::ConstSizedPipelineState::DepthStencilState::Flag_StencilTestEnable ) ? VK_TRUE : VK_FALSE;
    // front face
    pOutDepthStencilState->front.failOp = static_cast< VkStencilOp >(
        gpuState.constSizedPipelineState.depthStencilState.frontStencil.sfail );
    pOutDepthStencilState->front.passOp = static_cast< VkStencilOp >(
        gpuState.constSizedPipelineState.depthStencilState.frontStencil.dppass );
    pOutDepthStencilState->front.depthFailOp = static_cast< VkStencilOp >(
        gpuState.constSizedPipelineState.depthStencilState.frontStencil.dpfail );
    pOutDepthStencilState->front.compareOp = static_cast< VkCompareOp >(
        gpuState.constSizedPipelineState.depthStencilState.frontStencil.func );
    pOutDepthStencilState->front.compareMask = 0; // dynamic state
    pOutDepthStencilState->front.writeMask = 0; // dynamic state
    pOutDepthStencilState->front.reference = 0; // dynamic state
    // back face
    pOutDepthStencilState->back.failOp = static_cast< VkStencilOp >(
        gpuState.constSizedPipelineState.depthStencilState.backStencil.sfail );
    pOutDepthStencilState->back.passOp = static_cast< VkStencilOp >(
        gpuState.constSizedPipelineState.depthStencilState.backStencil.dppass );
    pOutDepthStencilState->back.depthFailOp = static_cast< VkStencilOp >(
        gpuState.constSizedPipelineState.depthStencilState.backStencil.dpfail );
    pOutDepthStencilState->back.compareOp = static_cast< VkCompareOp >(
        gpuState.constSizedPipelineState.depthStencilState.backStencil.func );
    pOutDepthStencilState->back.compareMask = 0; // dynamic state
    pOutDepthStencilState->back.writeMask = 0; // dynamic state
    pOutDepthStencilState->back.reference = 0; // dynamic state
    pOutDepthStencilState->minDepthBounds = 0.0; // dynamic state
    pOutDepthStencilState->maxDepthBounds = 1.0; // dynamic state
}

void Vk::SetupPipelineColorBlendStateCreateInfo(
    VkPipelineColorBlendStateCreateInfo* pOutColorBlendState,
    VkPipelineColorBlendAttachmentState pColorBlendAttachmentState[], int colorBlendAttachmentCount,
    const GfxVkGpuState& gpuState ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_LESS_EQUAL( static_cast< uint32_t >( colorBlendAttachmentCount ),
        gpuState.constSizedPipelineState.blendState.blendTargetCount );

    pOutColorBlendState->sType = VK_STRUCTURE_TYPE_PIPELINE_COLOR_BLEND_STATE_CREATE_INFO;
    pOutColorBlendState->pNext = NULL;
    pOutColorBlendState->flags = 0;
    pOutColorBlendState->logicOpEnable = gpuState.constSizedPipelineState.blendState.flag.GetBit(
        GfxVkGpuState::ConstSizedPipelineState::BlendState::Flag_LogicOpEnable ) ? VK_TRUE : VK_FALSE;
    pOutColorBlendState->logicOp = static_cast< VkLogicOp >(
        gpuState.constSizedPipelineState.blendState.logicOp );
    pOutColorBlendState->attachmentCount = colorBlendAttachmentCount;

    for ( int idxAttachment = 0; idxAttachment < colorBlendAttachmentCount; ++idxAttachment )
    {
        VkPipelineColorBlendAttachmentState* pDstState = &pColorBlendAttachmentState[ idxAttachment ];
        const GfxVkGpuState::VariableSizedPipelineState::BlendTargetState* pSrcState =
            &gpuState.variableSizedPipelineState.blendTargetState[ idxAttachment ];
        pDstState->blendEnable = pSrcState->flag.GetBit(
            GfxVkGpuState::VariableSizedPipelineState::BlendTargetState::Flag_BlendEnable ) ? VK_TRUE : VK_FALSE;
        pDstState->srcColorBlendFactor = static_cast< VkBlendFactor >( pSrcState->srcRGB );
        pDstState->dstColorBlendFactor = static_cast< VkBlendFactor >( pSrcState->dstRGB );
        pDstState->colorBlendOp = static_cast< VkBlendOp >( pSrcState->modeRGB );
        pDstState->srcAlphaBlendFactor = static_cast< VkBlendFactor >( pSrcState->srcAlpha );
        pDstState->dstAlphaBlendFactor = static_cast< VkBlendFactor >( pSrcState->dstAlpha );
        pDstState->alphaBlendOp = static_cast< VkBlendOp >( pSrcState->modeAlpha );
        pDstState->colorWriteMask = static_cast< VkColorComponentFlags >( pSrcState->colorMask );
    }
    pOutColorBlendState->pAttachments = pColorBlendAttachmentState;
    pOutColorBlendState->blendConstants[ 0 ] = pOutColorBlendState->blendConstants[ 1 ] =
        pOutColorBlendState->blendConstants[ 2 ] = pOutColorBlendState->blendConstants[ 3 ] = 0.0; // dynamic state
}

void Vk::SetupPipelineDynamicStateCreateInfo(
    VkPipelineDynamicStateCreateInfo* pOutDynamicState ) NN_NOEXCEPT
{
    pOutDynamicState->sType = VK_STRUCTURE_TYPE_PIPELINE_DYNAMIC_STATE_CREATE_INFO;
    pOutDynamicState->pNext = NULL;
    pOutDynamicState->flags = 0;
    pOutDynamicState->dynamicStateCount = g_ValidDynamicStateCount;
    pOutDynamicState->pDynamicStates = g_ValidDynamicState;
}

void Vk::CreateBufferMemoryHeap( DeviceImpl< ApiVariationVk1 >* pDevice ) NN_NOEXCEPT
{
    pDevice->ToData()->pBufferHeap = Vk::AllocDriverMemory( sizeof( BufferMemoryHeap ), 8 );
    new( pDevice->ToData()->pBufferHeap.ptr ) BufferMemoryHeap( pDevice );
}

void Vk::DestroyBufferMemoryHeap( DeviceImpl< ApiVariationVk1 >* pDevice ) NN_NOEXCEPT
{
    static_cast< BufferMemoryHeap* >( pDevice->ToData()->pBufferHeap )->~BufferMemoryHeap();
    Vk::FreeDriverMemory( pDevice->ToData()->pBufferHeap.ptr );
    pDevice->ToData()->pBufferHeap = NULL;
}

void Vk::CreateVkManagerList( DeviceImpl< ApiVariationVk1 >* pDevice ) NN_NOEXCEPT
{
    pDevice->ToData()->pVkManagerList = Vk::AllocDriverMemory( sizeof( List< VkManager* > ), 8 );
    NN_SDK_ASSERT( pDevice->ToData()->pVkManagerList.ptr != 0 );
    new( pDevice->ToData()->pVkManagerList.ptr ) List< VkManager* >;
}

void Vk::DestroyVkManagerList( DeviceImpl< ApiVariationVk1 >* pDevice ) NN_NOEXCEPT
{
    static_cast< List< VkManager* >* >( pDevice->ToData()->pVkManagerList )->~List< VkManager* >();
    Vk::FreeDriverMemory( pDevice->ToData()->pVkManagerList );
    pDevice->ToData()->pVkManagerList = NULL;
}

void Vk::AddToVkManagerList( DeviceImpl< ApiVariationVk1 >* pDevice, VkManager* pVkManager ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pVkManager );

    NN_GFX_CALL_NNOS_FUNCTION( nn::os::LockMutex( &pDevice->ToData()->resouceMutex ) );

    List< VkManager* >* pList = static_cast< List< VkManager* >* >( pDevice->ToData()->pVkManagerList );
    VkManager*& pNewNodeData = pList->EmplaceBack();
    pNewNodeData = pVkManager;

    NN_GFX_CALL_NNOS_FUNCTION( nn::os::UnlockMutex( &pDevice->ToData()->resouceMutex ) );
}

bool Vk::IsVkManagerNodeIdentical( VkManager* const& pNode, void* pArgument ) NN_NOEXCEPT
{
    VkManager* pRemoveNode = static_cast< VkManager* >( pArgument );
    return pNode == pRemoveNode;
}

void Vk::RemoveFromVkManagerList( DeviceImpl< ApiVariationVk1 >* pDevice, VkManager* pVkManager ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pVkManager );

    NN_GFX_CALL_NNOS_FUNCTION( nn::os::LockMutex( &pDevice->ToData()->resouceMutex ) );

    List< VkManager* >* pList = static_cast< List< VkManager* >* >( pDevice->ToData()->pVkManagerList );
    pList->RemoveIf( IsVkManagerNodeIdentical, pVkManager );

    NN_GFX_CALL_NNOS_FUNCTION( nn::os::UnlockMutex( &pDevice->ToData()->resouceMutex ) );
}


void Vk::DestroyShaderObject( DeviceImpl< ApiVariationVk1 >* pDevice, DestroyingShaderObjectInfo* pDestroyingShaderObjectInfo ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( pDestroyingShaderObjectInfo );
    NN_SDK_ASSERT( pDestroyingShaderObjectInfo->GetReferenceCount().IsZero() );

    DestroyingShaderObjectHandle& shaderObject = pDestroyingShaderObjectInfo->GetHandle();
    for ( int idxStage = 0; idxStage < ShaderStage_End; ++idxStage )
    {
        VkShaderModule vkShaderModule = shaderObject.handle[ idxStage ];
        if ( vkShaderModule )
        {
            NN_GFX_CALL_VK_FUNCTION( vkDestroyShaderModule(
                CastToVkDispatchableObject< VkDevice >( pDevice->ToData()->hDevice ),
                vkShaderModule, pDevice->ToData()->pAllocationCallback ) );
        }
    }
    Vk::FreeDriverMemory( pDestroyingShaderObjectInfo );
}


void Vk::DestroyImageObject( DeviceImpl< ApiVariationVk1 >* pDevice, DestroyingImageObjectInfo* pDestroyingImageObjectInfo ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( pDestroyingImageObjectInfo );
    NN_SDK_ASSERT( pDestroyingImageObjectInfo->GetReferenceCount().IsZero() );

    DestroyingImageObjectHandle& imageObjectHandle = pDestroyingImageObjectInfo->GetHandle();

    // preallocated image like swapchain image cannot be destroyed.
    if ( imageObjectHandle.memory != 0 )
    {
        VkDevice device = CastToVkDispatchableObject< VkDevice >( pDevice->ToData()->hDevice );
        NN_GFX_CALL_VK_FUNCTION( vkDestroyImage( device, imageObjectHandle.image, pDevice->ToData()->pAllocationCallback ) );
        NN_GFX_CALL_VK_FUNCTION( vkFreeMemory( device, imageObjectHandle.memory, pDevice->ToData()->pAllocationCallback ) );
    }

    Vk::FreeDriverMemory( pDestroyingImageObjectInfo );
}

void Vk::DestroyImageViewObject( DeviceImpl< ApiVariationVk1 >* pDevice, DestroyingImageViewObjectInfo* pDestroyingImageViewObjectInfo ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( pDestroyingImageViewObjectInfo );
    NN_SDK_ASSERT( pDestroyingImageViewObjectInfo->GetReferenceCount().IsZero() );

    DestroyingImageViewObjectHandle& imageViewObjectHandle = pDestroyingImageViewObjectInfo->GetHandle();

    VkDevice device = CastToVkDispatchableObject< VkDevice >( pDevice->ToData()->hDevice );

    NN_GFX_CALL_VK_FUNCTION( vkDestroyImageView( device, imageViewObjectHandle.imageViewHandle, pDevice->ToData()->pAllocationCallback ) );

    Vk::FreeDriverMemory( pDestroyingImageViewObjectInfo );
}

void Vk::RegisterDestroyingShaderObject( DeviceImpl< ApiVariationVk1 >* pDevice, ShaderImpl< ApiVariationVk1 >* pShader ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pDevice );
    NN_SDK_ASSERT_NOT_NULL( pShader );

    DestroyingShaderObjectHandle shaderObjectHandle;
    for ( int idxStage = 0; idxStage < ShaderStage_End; ++idxStage )
    {
        shaderObjectHandle.handle[ idxStage ] = CastToVkNonDispatchableObject< VkShaderModule >( pShader->ToData()->hShaderModule[ idxStage ] );
    }
    DestroyingShaderObjectInfo* pDestroyingShaderObjectInfo = static_cast< DestroyingShaderObjectInfo* >(
        Vk::AllocDriverMemory( sizeof( DestroyingShaderObjectInfo ), 8 ) );
    NN_SDK_ASSERT_NOT_NULL( pDestroyingShaderObjectInfo );
    new( pDestroyingShaderObjectInfo ) DestroyingShaderObjectInfo( shaderObjectHandle );

    NN_GFX_CALL_NNOS_FUNCTION( nn::os::LockMutex( &pDevice->ToData()->resouceMutex ) );

    List< VkManager* >* pVkManagerList = static_cast< List< VkManager* >* >( pDevice->ToData()->pVkManagerList );
    for ( auto iterator = pVkManagerList->Begin(); !pVkManagerList->IsEnd( iterator ); ++iterator )
    {
        // Register the destroying shader object only to active VkManager.
        if ( iterator->GetCommandBuffer()->ToData()->state == CommandBufferImpl< ApiVariationVk1 >::DataType::State_Begun
            || iterator->IsVulkanObjectCached() )
        {
            pDestroyingShaderObjectInfo->GetReferenceCount().Increment();
            iterator->RegisterDestroyingShaderList( pDestroyingShaderObjectInfo );
        }
    }

    if ( pDestroyingShaderObjectInfo->GetReferenceCount().IsZero() )
    {
        // If VkManager is not created or has no cached object, destroy it immediately.
        DestroyShaderObject( pDevice, pDestroyingShaderObjectInfo );
    }

    NN_GFX_CALL_NNOS_FUNCTION( nn::os::UnlockMutex( &pDevice->ToData()->resouceMutex ) );
}

void Vk::RegisterDestroyingImageObject( DeviceImpl<ApiVariationVk1>* pDevice, TextureImpl<ApiVariationVk1>* pTexture ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pDevice );
    NN_SDK_ASSERT_NOT_NULL( pTexture );

    DestroyingImageObjectInfo* pDestroyingImageObjectInfo = static_cast< DestroyingImageObjectInfo* >(
        Vk::AllocDriverMemory( sizeof( DestroyingImageObjectInfo ), 8 ) );
    NN_SDK_ASSERT_NOT_NULL( pDestroyingImageObjectInfo );
    DestroyingImageObjectHandle imageObjectHandle;
    imageObjectHandle.image = CastToVkNonDispatchableObject< VkImage >( pTexture->ToData()->hImage );
    imageObjectHandle.memory = CastToVkNonDispatchableObject< VkDeviceMemory >( pTexture->ToData()->hMemory );
    new( pDestroyingImageObjectInfo ) DestroyingImageObjectInfo( imageObjectHandle );

    DestroyingImageViewObjectHandle imageViewObjectHandle;
    imageViewObjectHandle.pDestroyingImageObjectInfo = pDestroyingImageObjectInfo;

    TextureManageData* pTextureManageData = static_cast< TextureManageData* >( pTexture->ToData()->pManager );

    NN_GFX_CALL_NNOS_FUNCTION( nn::os::LockMutex( &pDevice->ToData()->resouceMutex ) );

    List< VkManager* >* pVkManagerList = static_cast< List< VkManager* >* >( pDevice->ToData()->pVkManagerList );
    auto pColorTargetViewList = pTextureManageData->GetColorTargetView().GetNodeList();
    auto pDepthTargetViewList = pTextureManageData->GetDepthStencilView().GetNodeList();

    if ( pVkManagerList->IsEmpty() || ( pColorTargetViewList->IsEmpty() &&  pDepthTargetViewList->IsEmpty() ) )
    {
        // Destroy object immediately if no VkManger to register is created or no ImageView to be registered.
        NN_GFX_CALL_NNOS_FUNCTION( nn::os::UnlockMutex( &pDevice->ToData()->resouceMutex ) );
        pTextureManageData->GetColorTargetView().DestroyAllImageView( pDevice );
        pTextureManageData->GetDepthStencilView().DestroyAllImageView( pDevice );
        DestroyImageObject( pDevice, pDestroyingImageObjectInfo );
        return;
    }

    for ( auto colorTargetViewIterator = pColorTargetViewList->Begin();
        !pColorTargetViewList->IsEnd( colorTargetViewIterator ); ++colorTargetViewIterator )
    {
        DestroyingImageViewObjectInfo* pDestroyingImageViewObjectInfo = static_cast< DestroyingImageViewObjectInfo* >(
            Vk::AllocDriverMemory( sizeof( DestroyingImageViewObjectInfo ), 8 ) );
        NN_SDK_ASSERT_NOT_NULL( pDestroyingImageViewObjectInfo );

        imageViewObjectHandle.imageViewHandle = ( *colorTargetViewIterator ).imageView;
        new( pDestroyingImageViewObjectInfo ) DestroyingImageViewObjectInfo( imageViewObjectHandle );
        pDestroyingImageObjectInfo->GetReferenceCount().Increment();

        for ( auto vkManagerIterator = pVkManagerList->Begin(); !pVkManagerList->IsEnd( vkManagerIterator ); ++vkManagerIterator )
        {
            // Register the destroying image view object only to active VkManager.
            if ( vkManagerIterator->GetCommandBuffer()->ToData()->state == CommandBufferImpl< ApiVariationVk1 >::DataType::State_Begun
                || vkManagerIterator->IsVulkanObjectCached() )
            {
                pDestroyingImageViewObjectInfo->GetReferenceCount().Increment();
                vkManagerIterator->RegisterDestroyingColorTargetViewList( pDestroyingImageViewObjectInfo );
            }
        }
        if ( pDestroyingImageViewObjectInfo->GetReferenceCount().IsZero() )
        {
            // If all VkManager has no cached object, destroy image view object immediately.
            DestroyImageViewObject( pDevice, pDestroyingImageViewObjectInfo );
            pDestroyingImageObjectInfo->GetReferenceCount().Decrement();
        }
    }

    for ( auto depthTargetViewIterator = pDepthTargetViewList->Begin();
        !pDepthTargetViewList->IsEnd( depthTargetViewIterator ); ++depthTargetViewIterator )
    {
        DestroyingImageViewObjectInfo* pDestroyingImageViewObjectInfo = static_cast< DestroyingImageViewObjectInfo* >(
            Vk::AllocDriverMemory( sizeof( DestroyingImageViewObjectInfo ), 8 ) );
        NN_SDK_ASSERT_NOT_NULL( pDestroyingImageViewObjectInfo );

        imageViewObjectHandle.imageViewHandle = ( *depthTargetViewIterator ).imageView;
        new ( pDestroyingImageViewObjectInfo ) DestroyingImageViewObjectInfo( imageViewObjectHandle );
        pDestroyingImageObjectInfo->GetReferenceCount().Increment();

        for ( auto vkManagerIterator = pVkManagerList->Begin(); !pVkManagerList->IsEnd( vkManagerIterator ); ++vkManagerIterator )
        {
            // Register the destroying image view object only to active VkManager.
            if ( vkManagerIterator->GetCommandBuffer()->ToData()->state == CommandBufferImpl< ApiVariationVk1 >::DataType::State_Begun
                || vkManagerIterator->IsVulkanObjectCached() )
            {
                pDestroyingImageViewObjectInfo->GetReferenceCount().Increment();
                vkManagerIterator->RegisterDestroyingDepthTargetViewList( pDestroyingImageViewObjectInfo );
            }
        }
        if ( pDestroyingImageViewObjectInfo->GetReferenceCount().IsZero() )
        {
            // If all VkManager has no cached object, destroy image view object immediately.
            DestroyImageViewObject( pDevice, pDestroyingImageViewObjectInfo );
            pDestroyingImageObjectInfo->GetReferenceCount().Decrement();
        }
    }

    if ( pDestroyingImageObjectInfo->GetReferenceCount().IsZero() )
    {
        DestroyImageObject( pDevice, pDestroyingImageObjectInfo );
    }

    NN_GFX_CALL_NNOS_FUNCTION( nn::os::UnlockMutex( &pDevice->ToData()->resouceMutex ) );
}

}
}
}
