﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <vulkan/vulkan.h>
#include <cstring>

#include <nn/nn_Common.h>

#include <nn/nn_SdkAssert.h>
#include <nn/util/util_BytePtr.h>

#include <nn/gfx/detail/gfx_Common-api.vk.h>
#include <nn/gfx/detail/gfx_CommandBuffer-api.vk.1.h>

#include "gfx_VkHelper.h"
#include "gfx_VkManager.h"

namespace nn {
namespace gfx {
namespace detail {

class CommandContext
{
    NN_DISALLOW_COPY( CommandContext );

public:

    enum ContextType
    {
        ContextType_StateUpdater = 0,    //!< 独自コマンドのステート更新コマンドを参照し、描画時に必要なVulkanオブジェクトを生成します
        ContextType_VulkanCaller         //!< VulkanのAPIを発行して、Vulkanコマンドバッファにコマンドを蓄積します
    };


    CommandContext() NN_NOEXCEPT {}
    virtual ~CommandContext() NN_NOEXCEPT {}

    void Initialize( CommandBufferImpl< ApiVariationVk1 >* pGfxCommandBuffer, VkCommandBuffer hVkDrawCommandBuffer,
        VkCommandBuffer hVkPrepCommandBuffer, VkCommandBuffer hVkPostCommandBuffer,
        int firstCommand, int lastCommand, int threadCommandCount, bool isUseQuery, ContextType contextType ) NN_NOEXCEPT
    {
        this->m_IsInsideRenderPass = false;
        this->m_IsUseQuery = isUseQuery;
        this->m_CurrentCommand = 0;
        this->m_FirstCommand = firstCommand;
        this->m_LastCommand = lastCommand;
        this->m_ThreadCommandCount = threadCommandCount;
        this->m_ContextType = contextType;
        this->m_pGfxCommandBuffer = pGfxCommandBuffer;
        this->m_VkDrawCommandBuffer = hVkDrawCommandBuffer;
        this->m_VkPrepCommandBuffer = hVkPrepCommandBuffer;
        this->m_VkPostCommandBuffer = hVkPostCommandBuffer;
        memset( &this->m_GpuState, 0, sizeof( GfxVkGpuState ) );
    }

    void SetLayoutManager( ImageLayoutStateCacheList* pCache, nn::lmem::HeapHandle frameHeap ) NN_NOEXCEPT
    {
        this->m_FrameHeap = frameHeap;
        this->m_pLocalImageLayoutStateList = pCache;
    }

    bool IsContextTypeStateUpdater() const NN_NOEXCEPT
    {
        return m_ContextType == ContextType_StateUpdater;
    }

    bool IsContextTypeVulkanCaller() const NN_NOEXCEPT
    {
        return m_ContextType == ContextType_VulkanCaller;
    }

    GfxVkGpuState* GetGpuState() NN_NOEXCEPT
    {
        return &this->m_GpuState;
    }

    VkManager* GetVkManager() const NN_NOEXCEPT
    {
        return static_cast< VkManager* >( this->m_pGfxCommandBuffer->ToData()->pGfxVkManager );
    }

    GfxVkImageLayoutState* GetImageLayoutState( TextureManageData* pManager ) NN_NOEXCEPT
    {
        if ( Vk::IsLayoutManagementEnabled() )
        {
            VkImage imageHandle = pManager->GetImageHandle();
            uint32_t layerCount = pManager->GetLayerCount();
            uint32_t mipCount = pManager->GetMipCount();
            bool isFound = false;
            GfxVkImageLayoutState* pLayoutState = m_pLocalImageLayoutStateList->GetNode( &isFound, imageHandle );
            NN_SDK_ASSERT_NOT_NULL( pLayoutState );

            if ( !isFound )
            {
                pLayoutState->SetImageHandle( imageHandle );
                size_t requiredSize = pLayoutState->GetLayoutInfoSize( layerCount, mipCount );
                pLayoutState->SetLayoutInfo( layerCount, mipCount, pManager,
                    nn::lmem::AllocateFromFrameHeap( m_FrameHeap, requiredSize, 4 ) );
            }
            return pLayoutState;
        }
        else
        {
            return NULL;
        }
    }

    bool IsCurrentAssignedFirstCommand() const NN_NOEXCEPT
    {
        return m_FirstCommand == m_CurrentCommand;
    }

    bool IsCurrentAssignedCommand() const NN_NOEXCEPT
    {
        return m_FirstCommand <= m_CurrentCommand && m_CurrentCommand <= m_LastCommand;
    }

    bool IsCurrentCommandFinished() const NN_NOEXCEPT
    {
        return m_CurrentCommand > m_LastCommand;
    }

    bool IsCurrentStartCommand() const NN_NOEXCEPT
    {
        return m_ThreadCommandCount == 0 || ( m_CurrentCommand % m_ThreadCommandCount ) == 0;
    }

    CommandBufferImpl< ApiVariationVk1 >* GetGfxCommandBuffer() const NN_NOEXCEPT
    {
        return m_pGfxCommandBuffer;
    }

    VkCommandBuffer GetVkDrawCommandBuffer() const NN_NOEXCEPT
    {
        return m_VkDrawCommandBuffer;
    }

    VkCommandBuffer GetVkPrepCommandBuffer() const NN_NOEXCEPT
    {
        return m_VkPrepCommandBuffer;
    }

    VkCommandBuffer GetVkPostCommandBuffer() const NN_NOEXCEPT
    {
        return m_VkPostCommandBuffer;
    }

    void IncrementCurrentCommand() NN_NOEXCEPT
    {
        ++this->m_CurrentCommand;
    }

    int GetCurrentCommand() const NN_NOEXCEPT
    {
        return m_CurrentCommand;
    }

    void SetInsideRenderPass( bool isInside ) NN_NOEXCEPT
    {
        m_IsInsideRenderPass = isInside;
    }

    bool IsInsideRenderPass() const NN_NOEXCEPT
    {
        return m_IsInsideRenderPass;
    }

    bool IsUseQuery() const NN_NOEXCEPT
    {
        return m_IsUseQuery;
    }

    void SetUseQuery( bool useQuery ) NN_NOEXCEPT
    {
        m_IsUseQuery = useQuery;
    }

private:
    bool                                    m_IsInsideRenderPass;   //!< RenderPassの内部を処理中の場合はtrueです
    bool                                    m_IsUseQuery;           //!< Queryが利用可能な場合はtrueです
    ImageLayoutStateCacheList*              m_pLocalImageLayoutStateList; //!< Stores state of images used by this CommandContext.
    nn::lmem::HeapHandle                    m_FrameHeap;     //!< Memory used to track individual layer state.
    int                                     m_CurrentCommand;       //!< 現在処理中のコマンド番号です
    int                                     m_FirstCommand;         //!< 1つのVulkanCallerContextが受け持つコマンドの最初の番号です
    int                                     m_LastCommand;          //!< 1つのVulkanCallerContextが受け持つコマンドの最後の番号です
    int                                     m_ThreadCommandCount;   //!< 1つのVulkanCallerContextが受け持つコマンドの個数です（端数は最後のVulkanCallerContextが受け持ちます）
    ContextType                             m_ContextType;          //!< コンテキストタイプです
    VkCommandBuffer                         m_VkDrawCommandBuffer;  //!< Vulkan描画用コマンドバッファです
    VkCommandBuffer                         m_VkPrepCommandBuffer;  //!< Vulkan事前処理用コマンドバッファです
    VkCommandBuffer                         m_VkPostCommandBuffer;  //!< Vulkan事後処理用コマンドバッファです
    CommandBufferImpl< ApiVariationVk1 >*   m_pGfxCommandBuffer;    //!< Gfxコマンドバッファです
    GfxVkGpuState                           m_GpuState;             //!< GPUステートのスナップショットです
};

typedef void ( *VkCommandProcType )(
    CommandContext* pCommandContext, const void* pParam );

template< typename TParam >
void VkCommandProc(
    CommandContext* pCommandContext, const void* pParam ) NN_NOEXCEPT;

struct VkCommand
{
    union
    {
        VkCommandProcType pCommandProc;
        uint64_t commandProcPtr;
    };
    uint32_t paramSize;
    uint8_t param[ 1 ];
};

inline const void* ParseVkCommand( CommandContext* pCommandContext, const VkCommand& command ) NN_NOEXCEPT
{
    command.pCommandProc( pCommandContext, command.param );
    return command.param + command.paramSize;
}

inline const void* ParseVkCommand( CommandContext* pCommandContext, const void* pCommand ) NN_NOEXCEPT
{
    return ParseVkCommand( pCommandContext, *static_cast< const VkCommand* >( pCommand ) );
}

inline void ParseVkCommands( CommandContext* pCommandContext, const void* pCommands ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pCommands );
    while( *static_cast< const uintptr_t* >( pCommands ) )
    {
        pCommands = ParseVkCommand( pCommandContext, pCommands );
    }
}

inline void ParseVkCommandList( CommandContext* pCommandContext, const void* pCommandList ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pCommandContext );
    NN_SDK_ASSERT_NOT_NULL( pCommandList );

    for( ; ; )
    {
        if( *static_cast< const void* const* >( pCommandList ) == NULL )
        {
            pCommandList = nn::util::ConstBytePtr( pCommandList, sizeof( uint64_t ) ).Get();
            if( *static_cast< const void* const* >( pCommandList ) == NULL )
            {
                break;
            }
            pCommandList = *static_cast< const void* const* >( pCommandList );
            continue;
        }
        ParseVkCommands( pCommandContext,
            *static_cast< const void* const* >( pCommandList ) );
        pCommandList = nn::util::ConstBytePtr( pCommandList, sizeof( uint64_t ) ).Get();
    }
}

struct VkDispatchParam
{
    uint32_t groupCountX;
    uint32_t groupCountY;
    uint32_t groupCountZ;
};

struct VkDispatchIndirectParam
{
    uint64_t offset;
    VkHandle buffer;
};

struct VkDrawParam
{
    VkEnum topology;
    uint32_t vertexCount;
    uint32_t instanceCount;
    uint32_t firstVertex;
    uint32_t firstInstance;
};

struct VkDrawIndexedParam
{
    VkEnum topology;
    uint32_t indexCount;
    uint32_t instanceCount;
    int32_t vertexOffset;
    uint32_t firstInstance;
    VkEnum indexType;
    uint64_t indexOffset;
    VkHandle indexBuffer;
};

struct VkDrawIndirectParam
{
    VkEnum topology;
    uint64_t offset;
    VkHandle buffer;
};

struct VkDrawIndexedIndirectParam
{
    VkEnum topology;
    VkEnum indexType;
    uint64_t indexOffset;
    uint64_t indirectOffset;
    VkHandle indexBuffer;
    VkHandle indirectBuffer;
};

struct VkSetRenderTargetsParam
{
    int32_t colorAttachmentCount;
    uint32_t width;
    uint32_t height;
    uint32_t layers;

    struct AttachmentParam
    {
        TextureManageData* pManager;
        VkHandle imageView;
        VkEnum format;
        int32_t samples;
        uint32_t layer;
        uint32_t mip;
        uint32_t layerCount;
    };
    AttachmentParam depthAttachment;
    AttachmentParam colorAttachment[1];
};

struct VkSetVertexBufferParam
{
    int32_t bufferIndex;
    uint64_t offset;
    VkHandle buffer;
};

struct VkCopyBufferParam
{
    VkHandle dstBuffer;
    VkHandle srcBuffer;
    uint64_t dstOffset;
    uint64_t srcOffset;
    uint64_t size;
};

struct VkCopyImageParam
{
    VkHandle srcImage;
    TextureManageData* pSrcManager;
    uint32_t width;
    uint32_t height;
    uint32_t depth;
    uint32_t layerCount;
    int32_t srcX;
    int32_t srcY;
    int32_t srcZ;
    uint32_t srcMipLevel;
    uint32_t srcBaseLayer;
    uint32_t srcAspect;
    VkHandle dstImage;
    TextureManageData* pDstManager;
    int32_t dstX;
    int32_t dstY;
    int32_t dstZ;
    uint32_t dstMipLevel;
    uint32_t dstBaseLayer;
    uint32_t dstAspect;
};

struct VkBufferImageCopyParam
{
    VkHandle buffer;
    uint64_t bufferOffset;
    uint32_t bufferRowLength;
    uint32_t bufferImageHeight;

    VkHandle image;
    TextureManageData* pManager;
    int32_t imageOffsetX;
    int32_t imageOffsetY;
    int32_t imageOffsetZ;
    uint32_t imageWidth;
    uint32_t imageHeight;
    uint32_t imageDepth;
    uint32_t imageMipLevel;
    uint32_t imageBaseLayer;
    uint32_t imageAspect;
    uint32_t imageLayerCount;
};

struct VkCopyBufferToImageParam
{
    VkBufferImageCopyParam param;
};

struct VkCopyImageToBufferParam
{
    VkBufferImageCopyParam param;
};

struct VkBlitImageParam
{
    VkHandle srcImage;
    TextureManageData* pSrcManager;
    int32_t srcOffsetX;
    int32_t srcOffsetY;
    int32_t srcOffsetZ;
    uint32_t srcWidth;
    uint32_t srcHeight;
    uint32_t srcDepth;
    uint32_t srcMipLevel;
    uint32_t srcBaseLayer;
    uint32_t srcAspect;
    uint32_t srcLayerCount;
    VkHandle dstImage;
    TextureManageData* pDstManager;
    int32_t dstOffsetX;
    int32_t dstOffsetY;
    int32_t dstOffsetZ;
    uint32_t dstWidth;
    uint32_t dstHeight;
    uint32_t dstDepth;
    uint32_t dstMipLevel;
    uint32_t dstBaseLayer;
    uint32_t dstAspect;
    uint32_t dstLayerCount;
    VkEnum filter;
};

struct VkClearBufferParam
{
    VkHandle buffer;
    uint64_t offset;
    uint64_t size;
    uint32_t data;
};

struct VkClearColorParam
{
    VkHandle image;
    VkHandle imageView;
    TextureManageData* pManager;
    uint32_t width;
    uint32_t height;
    float color[ 4 ];
    uint32_t baseMipLevel;
    uint32_t mipLevelCount;
    uint32_t baseLayer;
    uint32_t layerCount;
};

struct VkClearDepthStencilParam
{
    VkHandle image;
    VkHandle imageView;
    TextureManageData* pManager;
    uint32_t width;
    uint32_t height;
    float depth;
    uint32_t stencil;
    uint32_t baseMipLevel;
    uint32_t mipLevelCount;
    uint32_t baseLayer;
    uint32_t layerCount;
    uint32_t aspectMask;
};

struct VkResolveParam
{
    uint32_t width;
    uint32_t height;
    uint32_t depth;
    uint32_t layerCount;
    VkHandle srcImage;
    TextureManageData* pSrcManager;
    uint32_t srcBaseLayer;
    VkHandle dstImage;
    TextureManageData* pDstManager;
    uint32_t dstMipLevel;
    uint32_t dstBaseLayer;
};

struct VkDiscardColorTargetParam
{
    uint32_t index;
};

struct VkDiscardDepthStencilTargetParam
{
    uint32_t aspectMask;
};

struct VkMemoryBarrierParam
{
    uint32_t srcAccessMask;
    uint32_t dstAccessMask;
};

struct VkCallCommandListParam
{
    union
    {
        const void* pCommandList;
        uint64_t commandListPtr;
    };
};

struct VkSetBufferStateTransitionParam
{
    VkHandle buffer;
    uint32_t srcAccessMask;
    uint32_t dstAccessMask;
    uint32_t srcStageFlag;
    uint32_t dstStageFlag;
};

struct VkSetTextureStateTransitionParam
{
    VkHandle image;
    TextureManageData* pManager;
    uint32_t imageBaseMipLevel;
    uint32_t imageMipCount;
    uint32_t imageBaseLayer;
    uint32_t imageLayerCount;
    uint32_t imageAspect;
    uint32_t srcAccessMask;
    uint32_t dstAccessMask;
    uint32_t srcStageFlag;
    uint32_t dstStageFlag;
    VkEnum oldLayout;
    VkEnum newLayout;
};

struct VkSetDepthBoundsParam
{
    float minDepthBounds;
    float maxDepthBounds;
};

struct VkSetLineWidthParam
{
    float lineWidth;
};

struct VkSetViewportsParam
{
    int32_t first;
    int32_t count;
    // float[ count * 6 ]
};

struct VkSetScissorsParam
{
    int32_t first;
    int32_t count;
    // int[ count * 4 ]
};

struct VkSetBufferParam
{
    VkHandle buffer;
    VkEnum descriptorType;
    int32_t slot;
    VkEnum shaderStage;
    uint64_t offset;
    uint64_t size;
};

struct VkSetTextureAndSamplerParam
{
    VkHandle texture;
    TextureManageData* pManager;
    VkHandle sampler;
    VkEnum descriptorType;
    int32_t slot;
    VkEnum shaderStage;
    VkEnum imageLayout;
    uint32_t baseLayer;
    uint32_t layerCount;
    uint32_t baseMip;
    uint32_t mipCount;
};

struct VkSetSeparateShaderParam
{
    VkHandle shaderModule[ ShaderStage_End ];
    uint64_t validMask[ ShaderStage_End ];
    int32_t stageBits;
};

struct VkBeginQueryParam
{
    VkHandle query;
    uint32_t flags;
    uint32_t queryTarget;
};

struct VkEndQueryParam
{
    VkHandle query;
    VkHandle buffer;
    uint64_t offset;
    uint64_t stride;
    uint32_t queryTarget;
};

struct VkWriteTimestampParam
{
    VkHandle query;
    VkHandle buffer;
    uint64_t offset;
};

struct VkBeginFastQueryParam
{
    VkHandle query;
    uint32_t flags;
    uint32_t queryIndex;
};

struct VkEndFastQueryParam
{
    VkHandle query;
    VkHandle buffer;
    uint32_t queryIndex;
    uint64_t offset;
    uint64_t stride;
};

struct VkWriteFastTimestampParam
{
    VkHandle query;
    VkHandle buffer;
    uint32_t queryIndex;
    uint64_t offset;
};

struct VkCallbackParam
{
    union
    {
        void ( *pCallback )( VkCommandBuffer, const void* );
        uint64_t callbackPtr;
    };
    union
    {
        const void* pParam;
        uint64_t paramPtr;
    };
};

struct VkPolygonOffsetParam
{
    float slopeScaledDepthBias;
    float depthBias;
    float depthBiasClamp;
};

struct VkPolygonOffsetEnableParam
{
    int32_t enable;
};

struct VkDepthClampParam
{
    int32_t enable;
};

struct VkActivateUniformBlockParam
{
    int32_t slot;
    int32_t stageBits;
    uint32_t size;
};

struct VkSetUniformParam
{
    uint32_t offset;
    uint32_t size;
    // uint8_t data[ 1 ];
};

}
}
}
