﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>

#include <nn/util/util_BytePtr.h>

#include <nn/gfx/gfx_TextureInfo.h>

#include <nn/gfx/detail/gfx_Texture-api.gl.4.h>
#include <nn/gfx/detail/gfx_MemoryPool-api.gl.4.h>

#include "gfx_CommonHelper.h"
#include "gfx_GlHelper.h"

namespace nn {
namespace gfx {
namespace detail {

namespace {

size_t CalculateMipDataOffsetsImpl( ptrdiff_t* pMipOffsets, const TextureInfo& info ) NN_NOEXCEPT
{
    ImageStorageDimension imageStorageDimension = info.GetImageStorageDimension();
    NN_SDK_ASSERT_NOT_EQUAL( imageStorageDimension, ImageStorageDimension_Undefined );
    int arrayLength = info.GetArrayLength();

    GLsizei width = info.GetWidth();
    GLsizei height = ( imageStorageDimension == ImageStorageDimension_1d
        && arrayLength > 1 ) ? arrayLength : info.GetHeight();
    GLsizei depth = imageStorageDimension== ImageStorageDimension_3d ? info.GetDepth() : arrayLength;

    GLsizei minHeight = 1;
    GLsizei minDepth = 1;
    if( arrayLength > 1 )
    {
        if( imageStorageDimension == ImageStorageDimension_1d )
        {
            minHeight = height;
        }
        else if( imageStorageDimension == ImageStorageDimension_2d )
        {
            minDepth = depth;
        }
    }

    ChannelFormat channelFormat = GetChannelFormat( info.GetImageFormat() );
    size_t size = 0;
    for( int mipLevel = 0; mipLevel < static_cast< int >( info.GetMipCount() ); ++mipLevel )
    {
        if( pMipOffsets )
        {
            pMipOffsets[ mipLevel ] = size;
        }
        size += CalculateImageSize( channelFormat,
            std::max NN_PREVENT_MACRO_FUNC ( width >> mipLevel, 1 ),
            std::max NN_PREVENT_MACRO_FUNC ( height >> mipLevel, minHeight ),
            std::max NN_PREVENT_MACRO_FUNC( depth >> mipLevel, minDepth ) );
    }
    return size;
}

}

typedef ApiVariationGl4 Target;

size_t TextureImpl< Target >::CalculateMipDataAlignment( DeviceImpl< Target >*, const InfoType& info ) NN_NOEXCEPT
{
    NN_UNUSED( info );

    return 1;
}

size_t TextureImpl< Target >::CalculateMipDataSize( DeviceImpl< Target >*, const InfoType& info ) NN_NOEXCEPT
{
    return CalculateMipDataOffsetsImpl( NULL, info );
}

void TextureImpl< Target >::CalculateMipDataOffsets(
    ptrdiff_t* pMipOffsets, DeviceImpl< Target >*, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pMipOffsets );

    CalculateMipDataOffsetsImpl( pMipOffsets, info );
}

size_t TextureImpl< Target >::GetRowPitch( DeviceImpl< Target >*, const InfoType& info ) NN_NOEXCEPT
{
    return CalculateRowSize( info.GetWidth(), GetChannelFormat( info.GetImageFormat() ) );
}

TextureImpl< Target >::TextureImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

TextureImpl< Target >::~TextureImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized || this->flags.GetBit( Flag_Shared ) );
}

void TextureImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info,
    MemoryPoolImpl< Target >* pMemoryPool, ptrdiff_t memoryPoolOffset, size_t memoryPoolSize ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_SDK_REQUIRES( !pMemoryPool || memoryPoolOffset >= 0 );
    NN_SDK_ASSERT( !pMemoryPool || pMemoryPool->ToData()->pMemory );
    NN_SDK_REQUIRES( !pMemoryPool || memoryPoolSize >= CalculateMipDataSize( pDevice, info ) );
    NN_UNUSED( memoryPoolSize );

    GlDeviceActivator activator( pDevice );

    ImageStorageDimension imageStorageDimension = info.GetImageStorageDimension();
    NN_SDK_ASSERT_NOT_EQUAL( imageStorageDimension, ImageStorageDimension_Undefined );
    int arrayLength = info.GetArrayLength();

    GLenum textureTarget = Gl::GetTextureTarget( GetImageDimension(
        imageStorageDimension, arrayLength > 1, info.GetMultisampleCount() > 1 ) );
    GlTextureFormat format = Gl::GetTextureFormat( info.GetImageFormat() );

    NN_SDK_ASSERT( !IsValid( this->hTexture ) );
    // NN_GFX_CALL_GL_FUNCTION( ::glCreateTextures( textureTarget, 1, &hTexture ) );
    NN_GFX_CALL_GL_FUNCTION( ::glGenTextures( 1, &this->hTexture ) );
    NN_GFX_CALL_GL_FUNCTION( ::glBindTexture( textureTarget, this->hTexture ) );
    NN_GFX_GL_ASSERT();
    NN_SDK_ASSERT( IsValid( this->hTexture ) );

    GLsizei width = info.GetWidth();
    GLsizei height = textureTarget == GL_TEXTURE_1D_ARRAY ? arrayLength : info.GetHeight();
    GLsizei depth = ( textureTarget == GL_TEXTURE_2D_ARRAY ||
        textureTarget == GL_TEXTURE_2D_MULTISAMPLE_ARRAY ) ? arrayLength : info.GetDepth();

    switch( textureTarget )
    {
    case GL_TEXTURE_1D:
        {
            NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glTextureStorage1D )(
                this->hTexture, textureTarget, info.GetMipCount(), format.imageFormat, width ) );
        }
        break;
    case GL_TEXTURE_1D_ARRAY:
    case GL_TEXTURE_2D:
        {
            NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glTextureStorage2D )(
                this->hTexture, textureTarget, info.GetMipCount(), format.imageFormat, width, height ) );
        }
        break;
    case GL_TEXTURE_2D_ARRAY:
    case GL_TEXTURE_3D:
        {
            NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glTextureStorage3D )(
                this->hTexture, textureTarget, info.GetMipCount(), format.imageFormat, width, height, depth ) );
        }
        break;
    case GL_TEXTURE_2D_MULTISAMPLE:
        {
            NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glTextureStorage2DMultisample )(
                this->hTexture, textureTarget, info.GetMultisampleCount(),
                format.imageFormat, width, height, GL_TRUE ) );
        }
        break;
    case GL_TEXTURE_2D_MULTISAMPLE_ARRAY:
        {
            NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glTextureStorage3DMultisample )(
                this->hTexture, textureTarget, info.GetMultisampleCount(),
                format.imageFormat, width, height, depth, GL_TRUE ) );
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    NN_GFX_GL_ASSERT();

    NN_GFX_CALL_GL_FUNCTION( ::glPixelStorei( GL_UNPACK_ROW_LENGTH, 0 ) );
    NN_GFX_CALL_GL_FUNCTION( ::glPixelStorei( GL_UNPACK_IMAGE_HEIGHT, 0 ) );
    if( pMemoryPool && textureTarget != GL_TEXTURE_2D_MULTISAMPLE
        && textureTarget != GL_TEXTURE_2D_MULTISAMPLE_ARRAY )
    {
        MemoryPoolImpl< Target >::DataType& memoryPool = pMemoryPool->ToData();
        nn::util::BytePtr pData( memoryPool.pMemory, memoryPoolOffset );
        GLsizei minHeight = textureTarget == GL_TEXTURE_1D_ARRAY ? height : 1;
        GLsizei minDepth = textureTarget == GL_TEXTURE_2D_ARRAY ? depth : 1;
        for( int mipLevel = 0; mipLevel < static_cast< int >( info.GetMipCount() ); ++mipLevel )
        {
            size_t size = Gl::CopyCpuMemoryToTexture( this->hTexture, textureTarget, mipLevel,
                info.GetImageFormat(), std::max NN_PREVENT_MACRO_FUNC ( width >> mipLevel, 1 ),
                std::max NN_PREVENT_MACRO_FUNC ( height >> mipLevel, minHeight ),
                std::max NN_PREVENT_MACRO_FUNC( depth >> mipLevel, minDepth ), pData.Get() );
            pData.Advance( static_cast< ptrdiff_t >( size ) );
            NN_GFX_GL_ASSERT();
        }
        NN_SDK_ASSERT( nn::util::BytePtr( memoryPool.pMemory.ptr ).Distance(
            pData.Get() ) <= static_cast< ptrdiff_t >( memoryPool.memorySize ) );
    }
    NN_GFX_GL_ASSERT();

    this->target = textureTarget;
    this->imageFormat = info.GetImageFormat();

    this->flags.SetBit( Flag_Shared, false );
    this->state = State_Initialized;
}

void TextureImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_ASSERT( !this->flags.GetBit( Flag_Shared ) );

    GlDeviceActivator activator( pDevice );

    if( this->hTexture )
    {
        NN_GFX_CALL_GL_FUNCTION( ::glDeleteTextures( 1, &this->hTexture ) );
        NN_GFX_GL_ASSERT();
        this->hTexture = GlInvalidHandle;
    }

    this->state = State_NotInitialized;
}

TextureViewImpl< Target >::TextureViewImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

TextureViewImpl< Target >::~TextureViewImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized || this->flags.GetBit( Flag_Shared ) );
}

void TextureViewImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );

    GlDeviceActivator activator( pDevice );

    NN_SDK_ASSERT( !IsValid( this->hTexture ) );
    NN_GFX_CALL_GL_FUNCTION( ::glGenTextures( 1, &this->hTexture ) );
    NN_GFX_GL_ASSERT();
    NN_SDK_ASSERT( IsValid( this->hTexture ) );

    const TextureImpl< Target >* pOriginalTexture = info.GetTexturePtr();
    NN_SDK_ASSERT_NOT_NULL( pOriginalTexture );
    NN_SDK_ASSERT( IsInitialized( *pOriginalTexture ) );

    GLenum textureTarget = Gl::GetTextureTarget( info.GetImageDimension() );
    GlTextureFormat format = Gl::GetTextureFormat( info.GetImageFormat() );

    NN_GFX_CALL_GL_FUNCTION( ::glTextureView(
        this->hTexture, textureTarget, pOriginalTexture->ToData()->hTexture,
        format.imageFormat, info.GetSubresourceRange().GetMipRange().GetMinMipLevel(),
        info.GetSubresourceRange().GetMipRange().GetMipCount(),
        info.GetSubresourceRange().GetArrayRange().GetBaseArrayIndex(),
        info.GetSubresourceRange().GetArrayRange().GetArrayLength() ) );
    NN_GFX_GL_ASSERT();

    GLint channelMappingTable[] =
    {
        GL_ZERO,
        GL_ONE,
        GL_RED,
        GL_GREEN,
        GL_BLUE,
        GL_ALPHA
    };
    GLint swizzleMask[] =
    {
        channelMappingTable[ info.GetChannelMapping( ColorChannel_Red ) ],
        channelMappingTable[ info.GetChannelMapping( ColorChannel_Green ) ],
        channelMappingTable[ info.GetChannelMapping( ColorChannel_Blue ) ],
        channelMappingTable[ info.GetChannelMapping( ColorChannel_Alpha ) ]
    };
    NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glTextureParameteriv )(
        this->hTexture, textureTarget, GL_TEXTURE_SWIZZLE_RGBA, swizzleMask ) );
    NN_GFX_GL_ASSERT();

    if( GetTypeFormat( info.GetImageFormat() ) == TypeFormat_DepthStencil )
    {
        GLenum depthStencilTextureModeTable[] =
        {
            GL_DEPTH_COMPONENT,
            GL_STENCIL_INDEX
        };

        NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glTextureParameteri )(
            this->hTexture, textureTarget, GL_DEPTH_STENCIL_TEXTURE_MODE,
            depthStencilTextureModeTable[ info.GetDepthStencilTextureMode() ] ) );
    }

    this->target = textureTarget;

    NN_GFX_GL_ASSERT();

    this->flags.SetBit( Flag_Shared, false );
    this->state = State_Initialized;
}

void TextureViewImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_ASSERT( !this->flags.GetBit( Flag_Shared ) );

    GlDeviceActivator activator( pDevice );

    if( this->hTexture )
    {
        NN_GFX_CALL_GL_FUNCTION( ::glDeleteTextures( 1, &this->hTexture ) );
        NN_GFX_GL_ASSERT();
        this->hTexture = GlInvalidHandle;
    }

    this->state = State_NotInitialized;
}

ColorTargetViewImpl< Target >::ColorTargetViewImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

ColorTargetViewImpl< Target >::~ColorTargetViewImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized || this->flags.GetBit( Flag_Shared ) );
}

void ColorTargetViewImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );

    GlDeviceActivator activator( pDevice );

    NN_SDK_ASSERT( !IsValid( this->hTexture ) );
    NN_GFX_CALL_GL_FUNCTION( ::glGenTextures( 1, &this->hTexture ) );
    NN_GFX_GL_ASSERT();
    NN_SDK_ASSERT( IsValid( this->hTexture ) );

    const TextureImpl< Target >* pOriginalTexture = info.GetTexturePtr();
    NN_SDK_ASSERT_NOT_NULL( pOriginalTexture );
    NN_SDK_ASSERT( IsInitialized( *pOriginalTexture ) );

    GLenum textureTarget = Gl::GetTextureTarget( info.GetImageDimension() );
    GLint internalFormat = Gl::GetTextureFormat( info.GetImageFormat() ).imageFormat;
    int baseArrayIndex = info.GetArrayRange().GetBaseArrayIndex();
    int arrayLength = info.GetArrayRange().GetArrayLength();
    this->layer = -1;
    if( textureTarget == GL_TEXTURE_3D )
    {
        this->layer = info.GetArrayRange().GetBaseArrayIndex();
        baseArrayIndex = 0;
        arrayLength = 1;
    }
    NN_GFX_CALL_GL_FUNCTION( ::glTextureView( this->hTexture, textureTarget,
        pOriginalTexture->ToData()->hTexture, internalFormat, info.GetMipLevel(),
        1, baseArrayIndex, arrayLength ) );

    this->target = textureTarget;

    NN_GFX_GL_ASSERT();

    this->flags.SetBit( Flag_Shared, false );
    this->state = State_Initialized;
}

void ColorTargetViewImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_ASSERT( !this->flags.GetBit( Flag_Shared ) );

    GlDeviceActivator activator( pDevice );

    if( this->hTexture )
    {
        NN_GFX_CALL_GL_FUNCTION( ::glDeleteTextures( 1, &this->hTexture ) );
        NN_GFX_GL_ASSERT();
        this->hTexture = GlInvalidHandle;
    }

    this->state = State_NotInitialized;
}

DepthStencilViewImpl< Target >::DepthStencilViewImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

DepthStencilViewImpl< Target >::~DepthStencilViewImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized || this->flags.GetBit( Flag_Shared ) );
}

void DepthStencilViewImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );

    GlDeviceActivator activator( pDevice );

    NN_SDK_ASSERT( !IsValid( this->hTexture ) );
    NN_GFX_CALL_GL_FUNCTION( ::glGenTextures( 1, &this->hTexture ) );
    NN_GFX_GL_ASSERT();
    NN_SDK_ASSERT( IsValid( this->hTexture ) );

    const TextureImpl< Target >* pOriginalTexture = info.GetTexturePtr();
    NN_SDK_ASSERT_NOT_NULL( pOriginalTexture );
    NN_SDK_ASSERT( IsInitialized( *pOriginalTexture ) );

    GLenum textureTarget = Gl::GetTextureTarget( info.GetImageDimension() );
    GLint format;
    NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glGetTextureLevelParameteriv )(
        pOriginalTexture->ToData()->hTexture, pOriginalTexture->ToData()->target,
        info.GetMipLevel(), GL_TEXTURE_INTERNAL_FORMAT, &format ) );

    NN_GFX_CALL_GL_FUNCTION( ::glTextureView( this->hTexture, textureTarget,
        pOriginalTexture->ToData()->hTexture, format, info.GetMipLevel(),
        1, info.GetArrayRange().GetBaseArrayIndex(), info.GetArrayRange().GetArrayLength() ) );

    this->target = textureTarget;
    this->internalFormat = format;

    NN_GFX_GL_ASSERT();

    this->flags.SetBit( Flag_Shared, false );
    this->state = State_Initialized;
}

void DepthStencilViewImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_ASSERT( !this->flags.GetBit( Flag_Shared ) );

    GlDeviceActivator activator( pDevice );

    if( this->hTexture )
    {
        NN_GFX_CALL_GL_FUNCTION( ::glDeleteTextures( 1, &this->hTexture ) );
        NN_GFX_GL_ASSERT();
        this->hTexture = GlInvalidHandle;
    }

    this->state = State_NotInitialized;
}

template<>
void GetImageFormatProperty< Target >( ImageFormatProperty* pOutImageFormatProperty,
    DeviceImpl< Target >* pDevice, ImageFormat imageFormat ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pOutImageFormatProperty );

    GlDeviceActivator activator( pDevice );

    GlTextureFormat glFormat = Gl::GetTextureFormat( imageFormat );

    Bit32 propertyFlags = 0;

    GLint isTextureSupported;
    NN_GFX_CALL_GL_FUNCTION( ::glGetInternalformativ( GL_TEXTURE_2D,
        glFormat.imageFormat, GL_INTERNALFORMAT_SUPPORTED, sizeof( GLint ), &isTextureSupported ) );
    propertyFlags |= isTextureSupported ? ImageFormatPropertyFlag_Texture : 0;

    GLint isRenderBufferSupported;
    NN_GFX_CALL_GL_FUNCTION( ::glGetInternalformativ( GL_TEXTURE_2D,
        glFormat.imageFormat, GL_COLOR_RENDERABLE, sizeof( GLint ), &isRenderBufferSupported ) );
    propertyFlags |= isRenderBufferSupported == GL_TRUE ? ImageFormatPropertyFlag_ColorTarget : 0;
    NN_GFX_GL_ASSERT();

    GLint isImageLoadSupported;
    NN_GFX_CALL_GL_FUNCTION( ::glGetInternalformativ( GL_TEXTURE_2D,
        glFormat.imageFormat, GL_SHADER_IMAGE_LOAD, sizeof( GLint ), &isImageLoadSupported ) );
    GLint isImageStoreSupported;
    NN_GFX_CALL_GL_FUNCTION( ::glGetInternalformativ( GL_TEXTURE_2D,
        glFormat.imageFormat, GL_SHADER_IMAGE_STORE, sizeof( GLint ), &isImageStoreSupported ) );
    GLint isImageAtomicSupported;
    NN_GFX_CALL_GL_FUNCTION( ::glGetInternalformativ( GL_TEXTURE_2D,
        glFormat.imageFormat, GL_SHADER_IMAGE_ATOMIC, sizeof( GLint ), &isImageAtomicSupported ) );
    if( isImageLoadSupported != GL_NONE || isImageStoreSupported != GL_NONE || isImageAtomicSupported != GL_NONE )
    {
        propertyFlags |= ImageFormatPropertyFlag_Image;
    }

    pOutImageFormatProperty->propertyFlags = propertyFlags;
}

}
}
}
