﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/gfx/gfx_SyncInfo.h>

#include <nn/gfx/detail/gfx_Sync-api.vk.1.h>
#include <nn/gfx/detail/gfx_Misc.h>
#include <nn/gfx/detail/gfx_Device-api.vk.1.h>
#include <nn/gfx/detail/gfx_Queue-api.vk.1.h>

#include "gfx_VkHelper.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationVk1 Target;

FenceImpl< Target >::FenceImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

FenceImpl< Target >::~FenceImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void FenceImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_UNUSED( info );

    this->pGfxDevice = pDevice;
    VkResult result;
    VkFenceCreateInfo fenceInfo;
    VkFence vkFence;
    VkDevice vkDevice = CastToVkDispatchableObject< VkDevice >( pDevice->ToData()->hDevice );
    fenceInfo.sType = VK_STRUCTURE_TYPE_FENCE_CREATE_INFO;
    fenceInfo.pNext = NULL;
    fenceInfo.flags = 0;
    NN_GFX_CALL_VK_FUNCTION( result = vkCreateFence( vkDevice, &fenceInfo,
        pDevice->ToData()->pAllocationCallback, &vkFence ) );
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
    NN_UNUSED( result );
    this->hFence = CastFromVkNonDispatchableObject< VkFence >( vkFence );

    this->state = State_Initialized;
}

void FenceImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( IsInitialized( *this ) );

    VkDevice vkDevice = CastToVkDispatchableObject< VkDevice >( pDevice->ToData()->hDevice );
    VkFence vkFence = CastToVkNonDispatchableObject< VkFence >( this->hFence );
    NN_GFX_CALL_VK_FUNCTION( vkDestroyFence(
        vkDevice, vkFence, pDevice->ToData()->pAllocationCallback ) );

    this->hFence = 0;
    this->state = State_NotInitialized;
}

bool FenceImpl< Target >::IsSignaled() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsInitialized( *this ) );

    VkDevice vkDevice = CastToVkDispatchableObject< VkDevice >( this->pGfxDevice->ToData()->hDevice );
    VkFence vkFence = CastToVkNonDispatchableObject< VkFence >( this->hFence );
    VkResult result;
    NN_GFX_CALL_VK_FUNCTION( result = vkGetFenceStatus( vkDevice, vkFence ) );

    return result == VK_SUCCESS ? true : false;
}

SyncResult FenceImpl< Target >::Sync( TimeSpan timeout ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsInitialized( *this ) );

    SyncResult result;
    VkFence vkFence = CastToVkNonDispatchableObject< VkFence >( this->hFence );
    uint64_t waitTime = timeout.GetNanoSeconds();
    VkDevice vkDevice = CastToVkDispatchableObject< VkDevice >( this->pGfxDevice->ToData()->hDevice );
    VkResult vkResult = vkWaitForFences( vkDevice, 1, &vkFence, VK_TRUE, waitTime );
    if ( vkResult == VK_SUCCESS )
    {
        result = SyncResult_Success;
    }
    else
    {
        result = SyncResult_TimeoutExpired;
    }
    vkResetFences( vkDevice, 1, &vkFence );

    return result;
}

SemaphoreImpl< Target >::SemaphoreImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

SemaphoreImpl< Target >::~SemaphoreImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void SemaphoreImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_UNUSED( info );

    this->pGfxDevice = pDevice;

    VkSemaphoreCreateInfo semaphoreInfo;
    semaphoreInfo.sType = VK_STRUCTURE_TYPE_SEMAPHORE_CREATE_INFO;
    semaphoreInfo.pNext = NULL;
    semaphoreInfo.flags = 0;

    VkSemaphore vkSemaphore;
    VkResult result;
    NN_GFX_CALL_VK_FUNCTION( result = vkCreateSemaphore(
        CastToVkDispatchableObject< VkDevice >( pDevice->ToData()->hDevice ),
        &semaphoreInfo, pDevice->ToData()->pAllocationCallback, &vkSemaphore ) );
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
    this->hSemaphore = CastFromVkNonDispatchableObject< VkSemaphore >( vkSemaphore );

    this->state = State_Initialized;
}

void SemaphoreImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsInitialized( *this ) );

    VkSemaphore vkSemaphore = CastToVkNonDispatchableObject< VkSemaphore >( this->hSemaphore );

    NN_GFX_CALL_VK_FUNCTION( vkDestroySemaphore(
        CastToVkDispatchableObject< VkDevice >( pDevice->ToData()->hDevice ),
        vkSemaphore, pDevice->ToData()->pAllocationCallback ) );

    this->state = State_NotInitialized;
}

}
}
}
