﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/gfx/gfx_SyncInfo.h>

#include <nn/gfx/detail/gfx_Sync-api.nvn.8.h>
#include <nn/gfx/detail/gfx_Device-api.nvn.8.h>
#include <nn/gfx/detail/gfx_Misc.h>

#include "gfx_NvnHelper.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationNvn8 Target;

FenceImpl< Target >::FenceImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

FenceImpl< Target >::~FenceImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void FenceImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_STATIC_ASSERT( sizeof( this->nvnSync ) == sizeof( NVNsync ) );
    NN_UNUSED( info );

    this->pNnDevice = pDevice;

    this->pNvnSync = &this->nvnSync[ 0 ];
    NVNboolean isSyncOK = NN_GFX_CALL_NVN_FUNCTION(
        nvnSyncInitialize( this->pNvnSync, pDevice->ToData()->pNvnDevice ) );
    NN_SDK_ASSERT( isSyncOK );
    NN_UNUSED( isSyncOK );

    this->state = State_Initialized;
}

void FenceImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_UNUSED( pDevice );
    NN_SDK_REQUIRES( IsInitialized( *this ) );
    NN_UNUSED( pDevice );

    // SwapChain::AcquireNextScanBuffer により this->pNvnSync は別のところを指しているかもしれない
    NVNsync* pSync = reinterpret_cast< NVNsync* >( this->ToData()->nvnSync );
    NN_GFX_CALL_NVN_FUNCTION( nvnSyncFinalize( pSync ) );
    this->pNvnSync = NULL;

    this->state = State_NotInitialized;
}

bool FenceImpl< Target >::IsSignaled() const NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsInitialized( *this ) );

    return NN_GFX_CALL_NVN_FUNCTION( nvnSyncWait( this->pNvnSync, 0 ) )
        == NVN_SYNC_WAIT_RESULT_ALREADY_SIGNALED;
}

SyncResult FenceImpl< Target >::Sync( TimeSpan timeout ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsInitialized( *this ) );

    SyncResult result;
    uint64_t nvnTimeout = timeout.GetNanoSeconds();

    NVNsyncWaitResult nvnResult = NN_GFX_CALL_NVN_FUNCTION(
        nvnSyncWait( this->pNvnSync, nvnTimeout ) );
    switch( nvnResult )
    {
        case NVN_SYNC_WAIT_RESULT_ALREADY_SIGNALED:
        case NVN_SYNC_WAIT_RESULT_CONDITION_SATISFIED:
        {
            result = SyncResult_Success;
        }
        break;

        case NVN_SYNC_WAIT_RESULT_TIMEOUT_EXPIRED:
        {
            result = SyncResult_TimeoutExpired;
        }
        break;

        default: NN_UNEXPECTED_DEFAULT;
    }

    return result;
}

SemaphoreImpl< Target >::SemaphoreImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

SemaphoreImpl< Target >::~SemaphoreImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void SemaphoreImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_STATIC_ASSERT( sizeof( this->nvnSync ) == sizeof( NVNsync ) );
    NN_UNUSED( info );

    this->pNnDevice = pDevice;

    this->pNvnSync = &this->nvnSync[ 0 ];
    NVNboolean isSemaphoreOK = NN_GFX_CALL_NVN_FUNCTION(
        nvnSyncInitialize( this->pNvnSync, pDevice->ToData()->pNvnDevice ) );
    NN_SDK_ASSERT( isSemaphoreOK );
    NN_UNUSED( isSemaphoreOK );

    this->state = State_Initialized;
}

void SemaphoreImpl< Target >::Finalize( DeviceImpl< Target >* ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsInitialized( *this ) );

    NN_GFX_CALL_NVN_FUNCTION( nvnSyncFinalize( this->pNvnSync ) );
    this->pNvnSync = NULL;

    this->state = State_NotInitialized;
}

}
}
}
