﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>

#include <nn/nn_Macro.h>

#include <nn/vi/vi_Layer.h>

#include <nn/gfx/gfx_SwapChainInfo.h>
#include <nn/gfx/gfx_DataAccessorConverter.h>
#include <nn/gfx/gfx_Texture.h>

#include <nn/gfx/detail/gfx_SwapChain-api.nvn.8.h>
#include <nn/gfx/detail/gfx_Texture-api.nvn.8.h>
#include <nn/gfx/detail/gfx_Device-api.nvn.8.h>
#include <nn/gfx/detail/gfx_Sync-api.nvn.8.h>

#include <nn/util/util_BitUtil.h> // For align_up

#include "gfx_CommonHelper.h"
#include "gfx_NvnHelper.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationNvn8 Target;

size_t SwapChainImpl< Target >::GetScanBufferAlignment(
    DeviceImpl< Target >* pNnDevice, const InfoType& info ) NN_NOEXCEPT
{
    TextureInfo textureInfo;
    Nvn::SetupScanBufferTextureInfo( &textureInfo, info );

    return TextureImpl< Target >::CalculateMipDataAlignment( pNnDevice, textureInfo );
}

size_t SwapChainImpl< Target >::CalculateScanBufferSize(
    DeviceImpl< Target > *pNnDevice, const InfoType& info) NN_NOEXCEPT
{
    TextureInfo textureInfo;
    Nvn::SetupScanBufferTextureInfo( &textureInfo, info );
    size_t sizePerTexture = TextureImpl< Target >::CalculateMipDataSize(pNnDevice, textureInfo);

    // Since we need to create multiple textures align the sizePerTexture to the alignment size
    size_t alignment = GetScanBufferAlignment( pNnDevice, info );
    sizePerTexture = nn::util::align_up( sizePerTexture, alignment );

    return sizePerTexture * info.GetBufferCount();
}

SwapChainImpl< Target >::SwapChainImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

SwapChainImpl< Target >::~SwapChainImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void SwapChainImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info,
    MemoryPoolImpl< Target >* pMemoryPool, ptrdiff_t memoryPoolOffset, size_t memoryPoolSize ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_SDK_ASSERT_LESS_EQUAL( info.GetBufferCount(), static_cast<int>( NN_ARRAY_SIZE( this->scanBufferTextures ) ) );
    NN_SDK_ASSERT_ALIGNED( memoryPoolOffset, GetScanBufferAlignment( pDevice, info ) );

    this->pNnDevice = pDevice;
    this->totalScanBuffers = info.GetBufferCount();
    this->currentScanBufferIndex = info.GetBufferCount() > 1 ? Nvn::GetFirstScanBufferIndex() : 0; // FIXME: Prior to the first swap we can't update this using nvnQueueAcquireTexture so we have to start at 1.
                                //        This is what the first call to nvnQueuAcquireTexture will return
    this->flag.SetBit( Flag_IsFirstPresentation, true );

    NVNtexture* pNvnTextures[ NvnMaxScanBufferCount ];
    TextureInfo textureInfo;
    Nvn::SetupScanBufferTextureInfo( &textureInfo, info );

    size_t scanBufferAlignment = TextureImpl< Target >::CalculateMipDataAlignment( pDevice, textureInfo );
    size_t scanBufferSize = TextureImpl< Target >::CalculateMipDataSize( pDevice, textureInfo );

    // Align the scanBufferSize to the alignment
    scanBufferSize = nn::util::align_up( scanBufferSize, scanBufferAlignment );

    NN_SDK_ASSERT( scanBufferSize * info.GetBufferCount() <= memoryPoolSize );
    NN_UNUSED( memoryPoolSize );

    // Setup the scan buffer Textures and ColorTargetViews
    for ( int index = 0; index < info.GetBufferCount(); index++ )
    {
        TextureImpl< Target >* pScanBuffer = nn::gfx::DataToAccessor( this->scanBufferTextures[ index ] );
        ColorTargetViewImpl< Target >* pScanBufferView = nn::gfx::DataToAccessor( this->scanBufferViews[ index ] );

        pScanBuffer->Initialize( pDevice, textureInfo, pMemoryPool, memoryPoolOffset, scanBufferSize );

        pNvnTextures[ index ] = pScanBuffer->ToData()->pNvnTexture;

        ColorTargetViewInfo colorTargetViewInfo;
        colorTargetViewInfo.SetImageDimension( nn::gfx::ImageDimension_2d );
        colorTargetViewInfo.SetMipLevel( 0 );
        colorTargetViewInfo.SetImageFormat( info.GetFormat() );
        colorTargetViewInfo.EditArrayRange().SetBaseArrayIndex( 0 );
        colorTargetViewInfo.EditArrayRange().SetArrayLength( 1 );
        colorTargetViewInfo.SetTexturePtr( static_cast< TTexture< Target >* >( pScanBuffer ) );
        pScanBufferView->Initialize( pDevice, colorTargetViewInfo );

        memoryPoolOffset += scanBufferSize;
    }

    // Setup the NVNwindow
    this->pNvnWindow = &this->nvnWindow;

    nn::vi::NativeWindowHandle hNativeWindow = NULL;
    nn::vi::Layer* pLayer = info.GetLayer();
    NN_UNUSED( pLayer );
    NN_SDK_ASSERT_NOT_NULL( pLayer );
    nn::Result viResult;
    NN_GFX_CALL_NNVI_FUNCTION( viResult = nn::vi::GetNativeWindow( &hNativeWindow, pLayer ) );
    NN_SDK_ASSERT( viResult.IsSuccess() );
    NN_UNUSED( viResult );

    NVNwindowBuilder windowBuilder;
    NN_GFX_CALL_NVN_FUNCTION( nvnWindowBuilderSetDevice(
        &windowBuilder, pDevice->ToData()->pNvnDevice ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnWindowBuilderSetDefaults( &windowBuilder ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnWindowBuilderSetNativeWindow( &windowBuilder, hNativeWindow ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnWindowBuilderSetTextures(
        &windowBuilder, info.GetBufferCount(), pNvnTextures ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnWindowBuilderSetPresentInterval( &windowBuilder, 1 ) );
    NVNboolean isOK = NN_GFX_CALL_NVN_FUNCTION(
        nvnWindowInitialize( this->pNvnWindow, &windowBuilder ) );
    NN_SDK_ASSERT( isOK );
    NN_UNUSED( isOK );

    this->state = State_Initialized;
}

void SwapChainImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( IsInitialized( *this ) );

    // Finalize the window handle before the internal views
    NN_GFX_CALL_NVN_FUNCTION( nvnWindowFinalize( this->pNvnWindow ) );
    this->pNvnWindow = NULL;

    // Setup the scan buffer Textures and ColorTargetViews
    for ( int index = 0; index < static_cast< int >( this->totalScanBuffers ); index++ )
    {
        TextureImpl< Target >* pScanBuffer = nn::gfx::DataToAccessor( this->scanBufferTextures[ index ] );
        ColorTargetViewImpl< Target >* pScanBufferView = nn::gfx::DataToAccessor( this->scanBufferViews[ index ] );

        pScanBufferView->Finalize( pDevice );
        pScanBuffer->Finalize( pDevice );
    }

    this->state = State_NotInitialized;
}

int SwapChainImpl< Target >::GetScanBufferViews(
    TColorTargetView< Target >** ppOutScanBufferArray, int maxScanBufferCount ) NN_NOEXCEPT
{
    if( ppOutScanBufferArray )
    {
        int count = std::min NN_PREVENT_MACRO_FUNC( maxScanBufferCount,
            static_cast< int >( this->totalScanBuffers ) );
        for( int idxScanBuffer = 0; idxScanBuffer < count; ++idxScanBuffer )
        {
            ppOutScanBufferArray[ idxScanBuffer ] =
                nn::gfx::DataToAccessor( this->scanBufferViews[ idxScanBuffer ] );
        }
        return count;
    }
    else
    {
        return this->totalScanBuffers;
    }
}

int SwapChainImpl< Target >::GetScanBuffers(
    TTexture< Target >** ppOutScanBufferTextureArray, int maxScanBufferTextureCount ) NN_NOEXCEPT
{
    if( ppOutScanBufferTextureArray )
    {
        int count = std::min NN_PREVENT_MACRO_FUNC( maxScanBufferTextureCount,
            static_cast< int >( this->totalScanBuffers ) );
        for( int idxScanBuffer = 0; idxScanBuffer < count; ++idxScanBuffer )
        {
            ppOutScanBufferTextureArray[ idxScanBuffer ] =
                nn::gfx::DataToAccessor( this->scanBufferTextures[ idxScanBuffer ] );
        }
        return count;
    }
    else
    {
        return this->totalScanBuffers;
    }
}

AcquireScanBufferResult SwapChainImpl< Target >::AcquireNextScanBufferIndex(
    int* pOutScanBufferIndex,  SemaphoreImpl< Target >* pSemaphore,
    FenceImpl< Target >* pFence ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsInitialized( *this ) );
    NN_SDK_REQUIRES_NOT_NULL( pOutScanBufferIndex );
    NN_SDK_REQUIRES_NOT_NULL( pSemaphore );
    NN_SDK_REQUIRES( IsInitialized( *pSemaphore ) );
    NN_SDK_REQUIRES( pFence == NULL || IsInitialized( *pFence ) );

    NVNwindowAcquireTextureResult result;
    NN_GFX_CALL_NVN_FUNCTION( result = nvnWindowAcquireTexture(
        this->pNvnWindow, pSemaphore->ToData()->pNvnSync, pOutScanBufferIndex ) );
    this->currentScanBufferIndex = *pOutScanBufferIndex;
    if( pFence )
    {
        // 共有している
        pFence->ToData()->pNvnSync = pSemaphore->ToData()->pNvnSync;
    }

    this->flag.SetBit( Flag_Acquired, true ); // TODO: 削除

    return result == NVN_WINDOW_ACQUIRE_TEXTURE_RESULT_SUCCESS ?
        AcquireScanBufferResult_Success : AcquireScanBufferResult_Failed;
}

int SwapChainImpl< Target >::AcquireNextScanBufferIndex() NN_NOEXCEPT
{
    return this->currentScanBufferIndex;
}

ColorTargetViewImpl< Target >* SwapChainImpl< Target >::AcquireNextScanBufferView() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    return nn::gfx::DataToAccessor( this->scanBufferViews[ this->currentScanBufferIndex ] );
}

TextureImpl< Target >* SwapChainImpl< Target >::AcquireNextScanBuffer() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    return nn::gfx::DataToAccessor( this->scanBufferTextures[ this->currentScanBufferIndex ] );
}

}
}
}
