﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>

#include <nn/nn_Macro.h>

#include <nn/gfx/gfx_SwapChainInfo.h>
#include <nn/gfx/gfx_DataAccessorConverter.h>
#include <nn/gfx/gfx_Texture.h>

#include <nn/gfx/detail/gfx_SwapChain-api.gl.4.h>
#include <nn/gfx/detail/gfx_Device-api.gl.4.h>
#include <nn/gfx/detail/gfx_Texture-api.gl.4.h>
#include <nn/gfx/detail/gfx_MemoryPool-api.gl.4.h>
#include <nn/gfx/detail/gfx_Sync-api.gl.4.h>

#include "gfx_CommonHelper.h"
#include "gfx_GlHelper.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationGl4 Target;

namespace {

void SetupScanbufferTextureInfo( TextureInfo* pOutInfo, const SwapChainInfo& info ) NN_NOEXCEPT
{
    pOutInfo->SetImageStorageDimension( ImageStorageDimension_2d );
    pOutInfo->SetTileMode( TileMode_Optimal );
    pOutInfo->SetSwizzle( 0 );
    pOutInfo->SetMipCount( 1 );
    pOutInfo->SetMultiSampleCount( 1 );
    pOutInfo->SetImageFormat( info.GetFormat() );
    pOutInfo->SetGpuAccessFlags( GpuAccess_ColorBuffer );
    pOutInfo->SetWidth( info.GetWidth() );
    pOutInfo->SetHeight( info.GetHeight() );
    pOutInfo->SetDepth( 0 );
    pOutInfo->SetArrayLength( 1 );
}

}

void InitializeSwapChainCommonImpl( SwapChainImpl< Target >* pThis, DeviceImpl< Target >* pDevice,
    const SwapChainInfo& info, MemoryPoolImpl< Target >* pMemoryPool,
    ptrdiff_t memoryPoolOffset, size_t memoryPoolSize ) NN_NOEXCEPT
{
    SwapChainImpl< Target >::DataType& data = pThis->ToData();

    TextureInfo textureInfo;
    SetupScanbufferTextureInfo( &textureInfo, info );

    ColorTargetViewInfo colorTargetViewInfo;
    colorTargetViewInfo.SetImageDimension( nn::gfx::ImageDimension_2d );
    colorTargetViewInfo.SetMipLevel( 0 );
    colorTargetViewInfo.SetImageFormat( info.GetFormat() );
    colorTargetViewInfo.EditArrayRange().SetBaseArrayIndex( 0 );
    colorTargetViewInfo.EditArrayRange().SetArrayLength( 1 );

    data.scanBufferCount = static_cast< int8_t >( info.GetBufferCount() );
    for( int idxScanBuffer = 0; idxScanBuffer < data.scanBufferCount; ++idxScanBuffer )
    {
        TextureImpl< Target >* pScanBuffer =
            nn::gfx::DataToAccessor( data.scanBuffers[ idxScanBuffer ] );
        ColorTargetViewImpl< Target >* pScanBufferView =
            nn::gfx::DataToAccessor( data.scanBufferViews[ idxScanBuffer ] );
        pScanBuffer->Initialize( pDevice, textureInfo, pMemoryPool, memoryPoolOffset, memoryPoolSize );
        colorTargetViewInfo.SetTexturePtr( static_cast< TTexture< Target >* >( pScanBuffer ) );
        pScanBufferView->Initialize( pDevice, colorTargetViewInfo );
    }

    data.currentScanBufferIndex = 0;
}

void FinalizeSwapChainCommonImpl( SwapChainImpl< Target >* pThis,
    DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    SwapChainImpl< Target >::DataType& data = pThis->ToData();

    for( int idxScanBuffer = 0; idxScanBuffer < data.scanBufferCount; ++idxScanBuffer )
    {
        TextureImpl< Target >* pScanBuffer =
            nn::gfx::DataToAccessor( data.scanBuffers[ idxScanBuffer ] );
        ColorTargetViewImpl< Target >* pScanBufferView =
            nn::gfx::DataToAccessor( data.scanBufferViews[ idxScanBuffer ] );
        NN_SDK_ASSERT( IsInitialized( *pScanBuffer ) );
        NN_SDK_ASSERT( IsInitialized( *pScanBufferView ) );
        pScanBuffer->Finalize( pDevice );
        pScanBufferView->Finalize( pDevice );
    }
}

size_t SwapChainImpl< Target >::GetScanBufferAlignment( DeviceImpl< Target >*, const SwapChainInfo& ) NN_NOEXCEPT
{
    return 1;
}

size_t SwapChainImpl< Target >::CalculateScanBufferSize(
    DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    TextureInfo textureInfo;
    SetupScanbufferTextureInfo( &textureInfo, info );

    return TextureImpl< Target >::CalculateMipDataSize( pDevice, textureInfo ) * info.GetBufferCount();
}

SwapChainImpl< Target >::SwapChainImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

SwapChainImpl< Target >::~SwapChainImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

int SwapChainImpl< Target >::GetScanBufferViews(
    TColorTargetView< Target >** ppOutScanBufferArray, int maxScanBufferCount ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    if( ppOutScanBufferArray )
    {
        int count = std::min NN_PREVENT_MACRO_FUNC( maxScanBufferCount,
            static_cast< int >( this->scanBufferCount ) );
        for( int idxScanBuffer = 0; idxScanBuffer < count; ++idxScanBuffer )
        {
            ppOutScanBufferArray[ idxScanBuffer ] =
                nn::gfx::DataToAccessor( this->scanBufferViews[ idxScanBuffer ] );
        }
        return count;
    }
    else
    {
        return this->scanBufferCount;
    }
}

int SwapChainImpl< Target >::GetScanBuffers(
    TTexture< Target >** ppOutScanBufferTextureArray, int maxScanBufferTextureCount ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    if( ppOutScanBufferTextureArray )
    {
        int count = std::min NN_PREVENT_MACRO_FUNC( maxScanBufferTextureCount,
            static_cast< int >( this->scanBufferCount ) );
        for( int idxScanBuffer = 0; idxScanBuffer < count; ++idxScanBuffer )
        {
            ppOutScanBufferTextureArray[ idxScanBuffer ] =
                nn::gfx::DataToAccessor( this->scanBuffers[ idxScanBuffer ] );
        }
        return count;
    }
    else
    {
        return this->scanBufferCount;
    }
}

AcquireScanBufferResult SwapChainImpl< Target >::AcquireNextScanBufferIndex(
    int* pOutScanBufferIndex, SemaphoreImpl< Target >* pSemaphore,
    FenceImpl< Target >* pFence ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsInitialized( *this ) );
    NN_SDK_REQUIRES_NOT_NULL( pOutScanBufferIndex );
    NN_SDK_REQUIRES_NOT_NULL( pSemaphore );
    NN_SDK_REQUIRES( IsInitialized( *pSemaphore ) );
    NN_SDK_REQUIRES( pFence == NULL || IsInitialized( *pFence ) );

    // 擬似的
    pSemaphore->ToData()->flags.SetBit( SemaphoreImpl< Target >::DataType::Flag_Signaled, true );
    if( pFence )
    {
        pFence->ToData()->flags.SetBit( FenceImpl< Target >::DataType::Flag_Signaled, true );
    }

    *pOutScanBufferIndex = this->currentScanBufferIndex;

    this->flag.SetBit( Flag_Acquired, true );

    return AcquireScanBufferResult_Success;
}

int SwapChainImpl< Target >::AcquireNextScanBufferIndex() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    this->flag.SetBit( Flag_Acquired, true );
    return this->currentScanBufferIndex;
}

ColorTargetViewImpl< Target >* SwapChainImpl< Target >::AcquireNextScanBufferView() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    this->flag.SetBit( Flag_Acquired, true );
    return nn::gfx::DataToAccessor( this->scanBufferViews[ this->currentScanBufferIndex ] );
}

TextureImpl< Target >* SwapChainImpl< Target >::AcquireNextScanBuffer() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    this->flag.SetBit( Flag_Acquired, true );
    return nn::gfx::DataToAccessor( this->scanBuffers[ this->currentScanBufferIndex ] );
}

}
}
}
