﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_StaticAssert.h>

#include <nn/util/util_BitPack.h>
#include <nn/util/util_BytePtr.h>

#include <nn/gfx/gfx_StateInfo.h>

#include <nn/gfx/detail/gfx_State-api.vk.1.h>
#include <nn/gfx/detail/gfx_Shader-api.vk.1.h>
#include <nn/gfx/detail/gfx_Misc.h>

#include "gfx_CommonHelper.h"
#include "gfx_VkHelper.h"
#include "gfx_VkCommand.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationVk1 Target;


namespace {

void InitializeStencilState( DepthStencilStateImpl< Target >::DataType::StencilState* pDst,
    const StencilStateInfo& src ) NN_NOEXCEPT
{
    // all definisions are matching between gfx and vulkan.
    pDst->sfail = src.GetStencilFailOperation();
    pDst->dpfail = src.GetDepthFailOperation();
    pDst->dppass = src.GetDepthPassOperation();
    pDst->func = src.GetComparisonFunction();
    pDst->ref = static_cast< uint32_t >( src.GetStencilRef() );
}

}RasterizerStateImpl< Target >::RasterizerStateImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

RasterizerStateImpl< Target >::~RasterizerStateImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void RasterizerStateImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( state == State_NotInitialized );
    NN_UNUSED( pDevice );

    static const VkPolygonMode PolygonModeTable[] =
    {
        VK_POLYGON_MODE_POINT,
        VK_POLYGON_MODE_LINE,
        VK_POLYGON_MODE_FILL
    };
    this->polygonMode = PolygonModeTable[ info.GetFillMode() ];

    static const VkFrontFace FrontFaceTable[] =
    {
        VK_FRONT_FACE_COUNTER_CLOCKWISE,
        VK_FRONT_FACE_CLOCKWISE
    };
    this->frontFace = FrontFaceTable[ info.GetFrontFace() ];

    static const VkCullModeFlagBits CullModeTable[] =
    {
        VK_CULL_MODE_NONE,
        VK_CULL_MODE_FRONT_BIT,
        VK_CULL_MODE_BACK_BIT
    };
    this->cullFace = CullModeTable[ info.GetCullMode() ];

    this->flag.SetBit( Flag_RasterDisable, !info.IsRasterEnabled() );
    this->flag.SetBit( Flag_MultisampleEnable, info.IsMultisampleEnabled() );
    this->flag.SetBit( Flag_DepthClampEnable, !info.IsDepthClipEnabled() );
    this->flag.SetBit( Flag_AlphaToCoverageEnable, info.GetMultisampleStateInfo().IsAlphaToCoverageEnabled() );

    this->flag.SetBit( Flag_PolygonOffsetEnable, info.GetDepthBias() != 0 || info.GetSlopeScaledDepthBias() != 0.0f );
    this->slopeScaledDepthBias = info.GetSlopeScaledDepthBias();
    this->depthBias = static_cast< float >( info.GetDepthBias() );
    this->depthBiasClamp = info.GetDepthBiasClamp();
    this->sampleMask = info.GetMultisampleStateInfo().GetSampleMask();
    this->sampleCount = info.GetMultisampleStateInfo().GetSampleCount() == 0 ? 1 : info.GetMultisampleStateInfo().GetSampleCount();

    this->flag.SetBit( Flag_ConservativeRasterEnable,
        info.GetConservativeRasterizationMode() == ConservativeRasterizationMode_Enable );

    this->state = State_Initialized;
}

void RasterizerStateImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_UNUSED( pDevice );

    this->state = State_NotInitialized;
}

size_t BlendStateImpl< Target >::GetRequiredMemorySize( const InfoType& info ) NN_NOEXCEPT
{
    return sizeof( BlendTargetState ) * info.GetBlendTargetCount();
}

BlendStateImpl< Target >::BlendStateImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

BlendStateImpl< Target >::~BlendStateImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void BlendStateImpl< Target >::SetMemory( void* pMemory, size_t size ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pMemory || size == 0 );

    this->pTargetArray = static_cast< BlendTargetState* >( pMemory );
    this->memorySize = static_cast< uint32_t >( size );
}

void* BlendStateImpl< Target >::GetMemory() NN_NOEXCEPT
{
    return this->pTargetArray;
}

const void* BlendStateImpl< Target >::GetMemory() const NN_NOEXCEPT
{
    return this->pTargetArray;
}

void BlendStateImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_UNUSED( pDevice );

    NN_SDK_REQUIRES( this->memorySize >= GetRequiredMemorySize( info ) );

    this->flag = info.ToData()->flag;

    // all definisions are matching between gfx::LogicOperation and VkLogicOp.
    this->logicOp = info.GetLogicOperation();

    this->blendColor[ 0 ] = info.GetBlendConstant( ColorChannel_Red );
    this->blendColor[ 1 ] = info.GetBlendConstant( ColorChannel_Green );
    this->blendColor[ 2 ] = info.GetBlendConstant( ColorChannel_Blue );
    this->blendColor[ 3 ] = info.GetBlendConstant( ColorChannel_Alpha );

    this->blendTargetCount = info.GetBlendTargetCount();
    const BlendTargetStateInfo* pTargetStates = info.GetBlendTargetStateInfoArray();
    NN_SDK_ASSERT( pTargetStates || this->blendTargetCount < 1 );
    NN_SDK_ASSERT( !info.IsIndependentBlendEnabled()
        || pDevice->ToData()->supportedFeaturesBitArray.test( VkDeviceFeature_IndependentBlend ) );
    int targetCount = info.IsIndependentBlendEnabled() ?
        this->blendTargetCount : ( this->blendTargetCount > 0 ? 1 : 0 );
    for( int idxTarget = 0; idxTarget < targetCount; ++idxTarget )
    {
        const BlendTargetStateInfo& src = pTargetStates[ idxTarget ];
        BlendTargetState& dst = this->pTargetArray.ptr[ idxTarget ];

        // all definisions are matching between gfx and vulkan.
        dst.colorMask = static_cast< Bit8 >( src.GetChannelMask() );
        dst.srcRGB = src.GetSourceColorBlendFactor();
        dst.dstRGB = src.GetDestinationColorBlendFactor();
        dst.srcAlpha = src.GetSourceAlphaBlendFactor();
        dst.dstAlpha = src.GetDestinationAlphaBlendFactor();
        dst.modeRGB = src.GetColorBlendFunction();
        dst.modeAlpha = src.GetAlphaBlendFunction();
        dst.flag = src.ToData()->flag;
    }

    this->state = State_Initialized;
}

void BlendStateImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_UNUSED( pDevice );

    this->state = State_NotInitialized;
}

DepthStencilStateImpl< Target >::DepthStencilStateImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

DepthStencilStateImpl< Target >::~DepthStencilStateImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void DepthStencilStateImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_UNUSED( pDevice );

    this->flag = info.ToData()->flag;

    // all definisions are matching between gfx and vulkan.
    this->depthFunc = info.GetDepthComparisonFunction();

    InitializeStencilState( &this->frontStencil, info.GetFrontStencilStateInfo() );
    InitializeStencilState( &this->backStencil, info.GetBackStencilStateInfo() );
    this->stencilWriteMask = info.GetStencilWriteMask();
    this->stencilReadMask = info.GetStencilReadMask();

    this->state = State_Initialized;
}

void DepthStencilStateImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_UNUSED( pDevice );

    this->state = State_NotInitialized;
}

size_t VertexStateImpl< Target >::GetRequiredMemorySize( const InfoType& info ) NN_NOEXCEPT
{
    return info.GetVertexAttributeCount() * sizeof( AttributeState )
        + info.GetVertexBufferCount() * sizeof( BufferState );
}

VertexStateImpl< Target >::VertexStateImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

VertexStateImpl< Target >::~VertexStateImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void VertexStateImpl< Target >::SetMemory( void* value, size_t size ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( value || size == 0 );

    this->pWorkMemory = value;
    this->memorySize = static_cast< uint32_t >( size );
}

void* VertexStateImpl< Target >::GetMemory() NN_NOEXCEPT
{
    return this->pWorkMemory;
}

const void* VertexStateImpl< Target >::GetMemory() const NN_NOEXCEPT
{
    return this->pWorkMemory;
}

void VertexStateImpl< Target >::Initialize( DeviceImpl< Target >* pDevice,
    const InfoType& info, const ShaderImpl< Target >* pVertexShader ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_SDK_REQUIRES( pVertexShader == NULL || IsInitialized( *pVertexShader ) );
    NN_SDK_REQUIRES( this->memorySize >= GetRequiredMemorySize( info ) );
    NN_UNUSED( pDevice );
    NN_UNUSED( pVertexShader );

    this->attributeCount = info.GetVertexAttributeCount();
    this->bufferCount = info.GetVertexBufferCount();

    nn::util::BytePtr ptr( this->pWorkMemory.ptr );
    AttributeState* pAttributeArray = ptr.Get< AttributeState >();
    BufferState* pBufferArray = ptr.Advance( sizeof( AttributeState ) * this->attributeCount ).Get< BufferState >();

    const VertexAttributeStateInfo* pAttributes = info.GetVertexAttributeStateInfoArray();
    NN_SDK_ASSERT( pAttributes || this->attributeCount < 1 );
    for( int idxAttribute = 0; idxAttribute < this->attributeCount; ++idxAttribute )
    {
        const VertexAttributeStateInfo& src = pAttributes[ idxAttribute ];
        AttributeState& dst = pAttributeArray[ idxAttribute ];
        if ( ( dst.location = static_cast< uint32_t >( src.GetShaderSlot() ) ) == -1
            && src.GetNamePtr() != NULL && pVertexShader != NULL
            && pVertexShader->ToData()->pReflection.ptr != NULL )
        {
            dst.location = pVertexShader->GetInterfaceSlot(
                ShaderStage_Vertex, ShaderInterfaceType_Input, src.GetNamePtr() );
        }
        dst.format = Vk::GetAttributeFormat( static_cast< AttributeFormat >( src.GetFormat() ) );
        dst.offset = static_cast< uint32_t >( src.GetOffset() );
        dst.bindingIndex = src.GetBufferIndex();
    }

    NN_SDK_ASSERT( pBufferArray || this->bufferCount < 1 );
    for( int idxBuffer = 0; idxBuffer < this->bufferCount; ++idxBuffer )
    {
        pBufferArray[ idxBuffer ].divisor = static_cast< uint32_t >(
            info.GetVertexBufferStateInfoArray()[ idxBuffer ].GetDivisor() );
        pBufferArray[ idxBuffer ].stride = static_cast< uint32_t >(
            info.GetVertexBufferStateInfoArray()[ idxBuffer ].GetStride() );
    }

    this->state = State_Initialized;

}

void VertexStateImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_UNUSED( pDevice );

    this->state = State_NotInitialized;
}

TessellationStateImpl< Target >::TessellationStateImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

TessellationStateImpl< Target >::~TessellationStateImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void TessellationStateImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_UNUSED( pDevice );

    this->patchControlPointCount = info.GetPatchControlPointCount();

    this->state = State_Initialized;
}

void TessellationStateImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_UNUSED( pDevice );

    this->state = State_NotInitialized;
}

size_t ViewportScissorStateImpl< Target >::GetRequiredMemorySize( const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( info.GetViewportCount() > 0 );
    NN_SDK_ASSERT( !info.IsScissorEnabled() || info.GetViewportCount() == info.GetScissorCount() );
    int extraViewportCount = info.GetViewportCount() - 1;
    return sizeof( float ) * 6 * extraViewportCount
        + ( info.IsScissorEnabled() ? sizeof( int32_t ) * 4 * extraViewportCount : 0 );
}

ViewportScissorStateImpl< Target >::ViewportScissorStateImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

ViewportScissorStateImpl< Target >::~ViewportScissorStateImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void ViewportScissorStateImpl< Target >::SetMemory( void* value, size_t size ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( value || size == 0 );

    this->pWorkMemory = value;
    this->memorySize = static_cast< uint32_t >( size );
}

void* ViewportScissorStateImpl< Target >::GetMemory() NN_NOEXCEPT
{
    return this->pWorkMemory;
}

const void* ViewportScissorStateImpl< Target >::GetMemory() const NN_NOEXCEPT
{
    return this->pWorkMemory;
}

void ViewportScissorStateImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_UNUSED( pDevice );

    NN_SDK_ASSERT( info.GetViewportCount() > 0 );
    NN_SDK_ASSERT( !info.IsScissorEnabled() || info.GetViewportCount() == info.GetScissorCount() );

    NN_SDK_REQUIRES( info.GetViewportCount() < 2 || this->pWorkMemory );
    NN_SDK_REQUIRES( info.GetViewportCount() < 2 || this->memorySize >= GetRequiredMemorySize( info ) );

    this->flag = info.ToData()->flag;

    this->viewportCount = info.GetViewportCount();

    const ViewportStateInfo* pViewports = info.GetViewportStateInfoArray();
    const ScissorStateInfo* pScissor = info.GetScissorStateInfoArray();
    NN_SDK_ASSERT_NOT_NULL( pViewports );
    NN_SDK_ASSERT( !info.IsScissorEnabled() || pScissor );

    this->viewport[ 0 ] = pViewports[ 0 ].GetOriginX();
    this->viewport[ 1 ] = pViewports[ 0 ].GetOriginY();
    this->viewport[ 2 ] = pViewports[ 0 ].GetWidth();
    this->viewport[ 3 ] = pViewports[ 0 ].GetHeight();
    this->viewport[ 4 ] = pViewports[ 0 ].GetMinDepth();
    this->viewport[ 5 ] = pViewports[ 0 ].GetMaxDepth();
    if( info.IsScissorEnabled() )
    {
        this->scissor[ 0 ] = pScissor[ 0 ].GetOriginX();
        this->scissor[ 1 ] = pScissor[ 0 ].GetOriginY();
        this->scissor[ 2 ] = pScissor[ 0 ].GetWidth();
        this->scissor[ 3 ] = pScissor[ 0 ].GetHeight();
    }

    int extraViewportCount = info.GetViewportCount() - 1;

    nn::util::BytePtr ptr( this->pWorkMemory.ptr );
    float* viewportArray = ptr.Get< float >();
    int32_t* scissorArray = ptr.Advance( sizeof( float ) * 6 * extraViewportCount ).Get< int32_t >();

    for( int idx = 1; idx < this->viewportCount; ++idx )
    {
        int idxExtra = idx - 1;
        const ViewportStateInfo& viewportInfo = pViewports[ idx ];
        viewportArray[ idxExtra * 6 + 0 ] = viewportInfo.GetOriginX();
        viewportArray[ idxExtra * 6 + 1 ] = viewportInfo.GetOriginY();
        viewportArray[ idxExtra * 6 + 2 ] = viewportInfo.GetWidth();
        viewportArray[ idxExtra * 6 + 3 ] = viewportInfo.GetHeight();
        viewportArray[ idxExtra * 6 + 4 ] = viewportInfo.GetMinDepth();
        viewportArray[ idxExtra * 6 + 5 ] = viewportInfo.GetMaxDepth();
    }

    if( info.IsScissorEnabled() )
    {
        for( int idx = 1; idx < this->viewportCount; ++idx )
        {
            int idxExtra = idx - 1;
            const ScissorStateInfo& scissorInfo = pScissor[ idx ];
            scissorArray[ idxExtra * 4 + 0 ] = scissorInfo.GetOriginX();
            scissorArray[ idxExtra * 4 + 1 ] = scissorInfo.GetOriginY();
            scissorArray[ idxExtra * 4 + 2 ] = scissorInfo.GetWidth();
            scissorArray[ idxExtra * 4 + 3 ] = scissorInfo.GetHeight();
        }
    }

    this->state = State_Initialized;

}

void ViewportScissorStateImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_UNUSED( pDevice );

    this->state = State_NotInitialized;
}

}
}
}
