﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_BytePtr.h>

#include <nn/gfx/gfx_StateInfo.h>
#include <nn/gfx/detail/gfx_State-api.gx.2.h>
#include <nn/gfx/detail/gfx_Shader-api.gx.2.h>
#include <nn/gfx/detail/gfx_Misc.h>

#include "gfx_CommonHelper.h"
#include "gfx_GxHelper.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationGx2 Target;

namespace {

struct StencilState
{
    detail::GxEnum func;
    detail::GxEnum zPass;
    detail::GxEnum zFail;
    detail::GxEnum sFail;
    detail::GxEnum preMask;
    detail::GxEnum writeMask;
    detail::GxEnum ref;
};

void InitializeStencilState( StencilState* pDst,
    const StencilStateInfo& src, nn::Bit8 readMask, nn::Bit8 writeMask ) NN_NOEXCEPT
{
    pDst->sFail = Gx::GetStencilFunction( src.GetStencilFailOperation() );
    pDst->zFail = Gx::GetStencilFunction( src.GetDepthFailOperation() );
    pDst->zPass = Gx::GetStencilFunction( src.GetDepthPassOperation() );

    pDst->func = Gx::GetCompareFunction( src.GetComparisonFunction() );
    pDst->ref = src.GetStencilRef();
    pDst->preMask = readMask;
    pDst->writeMask = writeMask;
}

}

RasterizerStateImpl< Target >::RasterizerStateImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

RasterizerStateImpl< Target >::~RasterizerStateImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void RasterizerStateImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( state == State_NotInitialized );
    NN_SDK_ASSERT( pDevice );
    NN_UNUSED( pDevice );

    int depthOffset = info.GetDepthBias();
    float depthSlopeScale = info.GetSlopeScaledDepthBias();
    float depthClamp = info.GetDepthBiasClamp();

    int sampleCount = info.GetMultisampleStateInfo().GetSampleCount();
    nn::Bit32 sampleMask = info.GetMultisampleStateInfo().GetSampleMask();

    GX2PolygonMode polygonMode = Gx::GetPolygonMode( info.GetFillMode() );
    GX2Boolean polygonOffsetEnable = static_cast< GX2Boolean >( (polygonMode != GX2_POLYGON_MODE_TRIANGLE ) );

    NN_STATIC_ASSERT( sizeof( this->polygonControlRegister ) == sizeof( GX2PolygonControlReg ) );
    NN_STATIC_ASSERT( sizeof( this->polygonOffsetRegister ) == sizeof( GX2PolygonOffsetReg ) );
    NN_STATIC_ASSERT( sizeof( this->aaMaskRegister ) == sizeof( GX2AAMaskReg ) );

    NN_GFX_CALL_GX_FUNCTION( GX2InitPolygonControlReg(
        reinterpret_cast< GX2PolygonControlReg* >( &this->polygonControlRegister ),
        static_cast< GX2FrontFaceMode >( info.GetFrontFace() ), // FrontFace == GX2FrontFaceMode
        static_cast< GX2Boolean >( (info.GetCullMode() == CullMode_Front) ),
        static_cast< GX2Boolean >( (info.GetCullMode() == CullMode_Back) ),
        static_cast< GX2Boolean >( (info.GetFillMode() != FillMode_Solid) ),
        static_cast< GX2PolygonMode >( polygonMode ),
        static_cast< GX2PolygonMode >( polygonMode ),
        static_cast< GX2Boolean >( polygonOffsetEnable ),
        static_cast< GX2Boolean >( polygonOffsetEnable ),
        static_cast< GX2Boolean >( polygonOffsetEnable ) ) );

    NN_GFX_CALL_GX_FUNCTION( GX2InitPolygonOffsetReg(
        reinterpret_cast< GX2PolygonOffsetReg* >( &this->polygonOffsetRegister ),
        static_cast< f32 >( depthOffset ),
        static_cast< f32 >( depthSlopeScale ),
        static_cast< f32 >( depthOffset ),
        static_cast< f32 >( depthSlopeScale ),
        static_cast< f32 >( depthClamp ) ) );

    // Setup multisampling mask/enable
    if( info.IsMultisampleEnabled() )
    {
        NN_GFX_CALL_GX_FUNCTION( GX2InitAAMaskReg(
            reinterpret_cast< GX2AAMaskReg* >( &this->aaMaskRegister ),
            sampleMask & 0xFF, ( sampleMask >> 8 ) & 0xFF,
            ( sampleMask >> 16 ) & 0xFF, ( sampleMask >> 24 ) & 0xFF ) );

        this->aaModeRegister = Gx::GetAaMode( sampleCount );
    }
    else
    {
        // Currently GX2 doesn't have a feature to turn off MSAA so we just
        // use 1X rendering.
        this->aaModeRegister = GX2_AA_MODE_1X;
        NN_GFX_CALL_GX_FUNCTION( GX2InitAAMaskReg(
            reinterpret_cast< GX2AAMaskReg* >( &this->aaMaskRegister ),
            0xFF, 0xFF, 0xFF, 0xFF ) );
    }

    // Set Raster/ZClip flags
    this->rasterizerEnable = info.IsRasterEnabled();
    this->zClipEnable = info.IsDepthClipEnabled();

    this->state = State_Initialized;
}

void RasterizerStateImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_ASSERT( pDevice );
    NN_UNUSED( pDevice );
    this->state = State_NotInitialized;
}

BlendStateImpl< Target >::BlendStateImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

BlendStateImpl< Target >::~BlendStateImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

size_t BlendStateImpl< Target >::GetRequiredMemorySize( const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( info.GetBlendTargetCount() <= GX2_MAX_RENDER_TARGETS );
    return sizeof( BlendTargetState ) * info.GetBlendTargetCount();
}

void BlendStateImpl< Target >::SetMemory( void* pMemory, size_t size ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( pMemory || size == 0 );

    this->pTargetArray = static_cast< BlendTargetState* >( pMemory );
    this->memorySize = static_cast< uint32_t >( size );
}

void* BlendStateImpl< Target >::GetMemory() NN_NOEXCEPT
{
    return this->pTargetArray;
}

const void* BlendStateImpl< Target >::GetMemory() const NN_NOEXCEPT
{
    return this->pTargetArray;
}

void BlendStateImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_SDK_ASSERT( pDevice );
    NN_UNUSED( pDevice );

    NN_SDK_REQUIRES( this->memorySize >= GetRequiredMemorySize( info ) );

    GX2LogicOp logicOperation = info.IsLogicOperationEnabled() ?
        Gx::GetLogicalOperation( info.GetLogicOperation() ) : GX2_LOGIC_OP_COPY;
    GX2ChannelMask targetColorMask[ GX2_MAX_RENDER_TARGETS ] =
    {
        GX2_CHANNEL_MASK_NONE,
        GX2_CHANNEL_MASK_NONE,
        GX2_CHANNEL_MASK_NONE,
        GX2_CHANNEL_MASK_NONE,
        GX2_CHANNEL_MASK_NONE,
        GX2_CHANNEL_MASK_NONE,
        GX2_CHANNEL_MASK_NONE,
        GX2_CHANNEL_MASK_NONE,
    };
    int blendMask = 0;
    int combinedChannelMask = 0;

    this->blendTargetCount = info.GetBlendTargetCount();
    NN_SDK_ASSERT( this->blendTargetCount <= GX2_MAX_RENDER_TARGETS );

    const BlendTargetStateInfo* pTargetStates = info.GetBlendTargetStateInfoArray();
    for( int idxTarget = 0; idxTarget < this->blendTargetCount; idxTarget++ )
    {
        const BlendTargetStateInfo& src = pTargetStates[ info.IsIndependentBlendEnabled() ? idxTarget : 0 ];

        blendMask |= src.IsBlendEnabled() ? (1 << idxTarget) : 0;

        // This conversion is safe because ChannelMask_* and GX2_CHANNEL_MASK_* use the same
        // bitfield scheme.
        targetColorMask[ idxTarget ] = static_cast< GX2ChannelMask >( src.GetChannelMask() );
        combinedChannelMask |= targetColorMask[ idxTarget ];

        NN_GFX_CALL_GX_FUNCTION( GX2InitBlendControlReg(
            reinterpret_cast< GX2BlendControlReg* >( &this->pTargetArray[ idxTarget ].blendControlRegister ),
            static_cast< GX2RenderTarget >( idxTarget + GX2_RENDER_TARGET_0 ),
            Gx::GetBlendFunction( src.GetSourceColorBlendFactor() ),
            Gx::GetBlendFunction( src.GetDestinationColorBlendFactor() ),
            Gx::GetBlendCombineMode( src.GetColorBlendFunction() ),
            GX2_TRUE, // Always use separate blend function for alpha
            Gx::GetBlendFunction( src.GetSourceAlphaBlendFactor() ),
            Gx::GetBlendFunction( src.GetDestinationAlphaBlendFactor() ),
            Gx::GetBlendCombineMode( src.GetAlphaBlendFunction() ) ) );
    }

    NN_GFX_CALL_GX_FUNCTION( GX2InitColorControlReg(
        reinterpret_cast< GX2ColorControlReg* >( &this->colorControlRegister ),
        logicOperation,
        blendMask,
        static_cast< GX2Boolean >( info.IsDualSourceBlendEnabled() ),
        static_cast< GX2Boolean >( !!combinedChannelMask ) ) );

    NN_GFX_CALL_GX_FUNCTION( GX2InitBlendConstantColorReg(
        reinterpret_cast< GX2BlendConstantColorReg* >( &this->blendConstantColorRegister ),
        info.GetBlendConstant( ColorChannel_Red ),
        info.GetBlendConstant( ColorChannel_Green ),
        info.GetBlendConstant( ColorChannel_Blue ),
        info.GetBlendConstant( ColorChannel_Alpha ) ) );

    NN_GFX_CALL_GX_FUNCTION( GX2InitAlphaToMaskReg(
        reinterpret_cast< GX2AlphaToMaskReg* >( &this->alphaToMaskRegister ),
        static_cast< GX2Boolean >( info.IsAlphaToCoverageEnabled() ),
        GX2_ALPHA_TO_MASK_0 ) );

    NN_GFX_CALL_GX_FUNCTION( GX2InitTargetChannelMasksReg(
        reinterpret_cast< GX2TargetChannelMaskReg* >( &this->targetChannelMaskRegister ),
        targetColorMask[0], targetColorMask[1], targetColorMask[2], targetColorMask[3],
        targetColorMask[4], targetColorMask[5], targetColorMask[6], targetColorMask[7] ) );

    this->state = State_Initialized;
}

void BlendStateImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_ASSERT( pDevice );
    NN_UNUSED( pDevice );

    this->state = State_NotInitialized;
}

DepthStencilStateImpl< Target >::DepthStencilStateImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

DepthStencilStateImpl< Target >::~DepthStencilStateImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void DepthStencilStateImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_SDK_ASSERT( pDevice );
    NN_UNUSED( pDevice );
    StencilState frontStencil;
    StencilState backStencil;
    GxEnum depthFunc;

    depthFunc = Gx::GetCompareFunction( info.GetDepthComparisonFunction() );

    InitializeStencilState( &frontStencil, info.GetFrontStencilStateInfo(),
        info.GetStencilReadMask(), info.GetStencilWriteMask() );
    InitializeStencilState( &backStencil, info.GetBackStencilStateInfo(),
        info.GetStencilReadMask(), info.GetStencilWriteMask() );

    NN_STATIC_ASSERT( sizeof( this->depthStencilControlRegister ) == sizeof( GX2DepthStencilControlReg ) );

    NN_GFX_CALL_GX_FUNCTION( GX2InitDepthStencilControlReg(
        reinterpret_cast< GX2DepthStencilControlReg* >( &depthStencilControlRegister ),
        static_cast< GX2Boolean >( info.IsDepthTestEnabled() ),
        static_cast< GX2Boolean >( info.IsDepthWriteEnabled() ),
        static_cast< GX2CompareFunction >( depthFunc ),
        static_cast< GX2Boolean >( info.IsStencilTestEnabled() ),
        static_cast< GX2Boolean >( info.IsStencilTestEnabled() ),
        static_cast< GX2CompareFunction >( frontStencil.func),
        static_cast< GX2StencilFunction >( frontStencil.zPass ),
        static_cast< GX2StencilFunction >( frontStencil.zFail ),
        static_cast< GX2StencilFunction >( frontStencil.sFail ),
        static_cast< GX2CompareFunction >( backStencil.func),
        static_cast< GX2StencilFunction >( backStencil.zPass ),
        static_cast< GX2StencilFunction >( backStencil.zFail ),
        static_cast< GX2StencilFunction >( backStencil.sFail ) ) );

    NN_STATIC_ASSERT( sizeof( this->stencilMaskRegister ) == sizeof( GX2StencilMaskReg ) );

    NN_GFX_CALL_GX_FUNCTION( GX2InitStencilMaskReg(
        reinterpret_cast< GX2StencilMaskReg* >( &this->stencilMaskRegister ),
        frontStencil.preMask,
        frontStencil.writeMask,
        frontStencil.ref,
        backStencil.preMask,
        backStencil.writeMask,
        backStencil.ref ) );

    this->state = State_Initialized;
}

void DepthStencilStateImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_ASSERT( pDevice );
    NN_UNUSED( pDevice );

    this->state = State_NotInitialized;
}

VertexStateImpl< Target >::VertexStateImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

VertexStateImpl< Target >::~VertexStateImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

size_t VertexStateImpl< Target >::GetRequiredMemorySize( const InfoType& info ) NN_NOEXCEPT
{
    // Currently we do not support tessellation.

    // Need to add padding so we can align internally
    return GX2_SHADER_ALIGNMENT +
        NN_GFX_CALL_GX_FUNCTION( GX2CalcFetchShaderSizeEx( info.GetVertexAttributeCount(),
        GX2_FETCH_SHADER_TESSELATION_NONE, GX2_TESSELLATION_MODE_DISCRETE ) );
}

void VertexStateImpl< Target >::SetMemory( void* pMemory, size_t size ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( pMemory || size == 0 );

    this->pMemory = pMemory;

    // Guarantee alignment on GX2_SHADER_ALIGNMENT boundary
    this->fetchShaderData = reinterpret_cast< void* >( ( reinterpret_cast< ptrdiff_t >(
        pMemory ) + GX2_SHADER_ALIGNMENT - 1 ) & ( ~( GX2_SHADER_ALIGNMENT - 1 ) ) );
    this->memorySize = size;
}

void* VertexStateImpl< Target >::GetMemory( ) NN_NOEXCEPT
{
    return this->pMemory;
}

void VertexStateImpl< Target >::Initialize( DeviceImpl< Target >* pDevice,
    const InfoType& info, const ShaderImpl< Target >* pVertexShader ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_SDK_REQUIRES( this->memorySize >= GetRequiredMemorySize( info ) );
    NN_STATIC_ASSERT( sizeof( this->fetchShader ) == sizeof( GX2FetchShader ) );

    NN_SDK_ASSERT( pDevice );
    NN_UNUSED( pDevice );
    NN_SDK_ASSERT( info.GetVertexAttributeCount() <= 2 * GX2_MAX_ATTRIB_BUFFERS );
    NN_SDK_ASSERT( info.GetVertexBufferCount() <= GX2_MAX_ATTRIB_BUFFERS );

    GX2AttribStream attribs[ 2 * GX2_MAX_ATTRIB_BUFFERS ];
    const VertexAttributeStateInfo* pState = info.GetVertexAttributeStateInfoArray();
    const VertexBufferStateInfo* pBufferState = info.GetVertexBufferStateInfoArray();

    for( int idx = 0; idx < info.GetVertexAttributeCount(); idx++ )
    {
        GX2AttribFormat attributeFormat = Gx::GetAttributeFormat( pState[ idx ].GetFormat() );
        int bufferIdx = pState[ idx ].GetBufferIndex();
        int offset = pState[ idx ].GetOffset();
        int slot = pState[ idx ].GetShaderSlot();

        if( slot < 0 && pState[ idx ].GetNamePtr() && pVertexShader )
        {
            slot = pVertexShader->GetInterfaceSlot( ShaderStage_Vertex,
                ShaderInterfaceType_Input, pState[ idx ].GetNamePtr() );
        }

        NN_GFX_CALL_GX_FUNCTION( GX2InitAttribStream( &attribs[ idx ],
            slot, bufferIdx, offset, attributeFormat ) );

        // Set the ALU divisor based on the buffer id.
        NN_SDK_ASSERT( bufferIdx < info.GetVertexBufferCount() );
        attribs[ idx ].aluDivisor = pBufferState[ bufferIdx ].GetDivisor();

        // Mark the ID as an instance ID
        if( attribs[ idx ].aluDivisor != 0 )
        {
            attribs[ idx ].indexType = GX2_ATTRIB_INDEX_INSTANCE_ID;
        }
    }

    // Currently we do not support tessellation.
    NN_GFX_CALL_GX_FUNCTION( GX2InitFetchShaderEx(
        reinterpret_cast< GX2FetchShader* >( &this->fetchShader ),
        this->fetchShaderData,
        info.GetVertexAttributeCount(),
        attribs,
        GX2_FETCH_SHADER_TESSELATION_NONE,
        GX2_TESSELLATION_MODE_DISCRETE ) );

    this->state = State_Initialized;
}

void VertexStateImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_UNUSED( pDevice );

    this->state = State_NotInitialized;
}

TessellationStateImpl< Target >::TessellationStateImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

TessellationStateImpl< Target >::~TessellationStateImpl() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
}

void TessellationStateImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_UNUSED( pDevice );

    this->state = State_Initialized;
}

void TessellationStateImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_UNUSED( pDevice );

    this->state = State_NotInitialized;
}

size_t ViewportScissorStateImpl< Target >::GetRequiredMemorySize( const InfoType& ) NN_NOEXCEPT
{
    return 0;
}

void ViewportScissorStateImpl< Target >::SetMemory( void*, size_t ) NN_NOEXCEPT
{
}

void* ViewportScissorStateImpl< Target >::GetMemory() NN_NOEXCEPT
{
    return NULL;
}

const void* ViewportScissorStateImpl< Target >::GetMemory() const NN_NOEXCEPT
{
    return NULL;
}

ViewportScissorStateImpl< Target >::ViewportScissorStateImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

ViewportScissorStateImpl< Target >::~ViewportScissorStateImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void ViewportScissorStateImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_STATIC_ASSERT( sizeof( this->viewportRegister ) == sizeof( GX2ViewportReg ) );
    NN_STATIC_ASSERT( sizeof( this->scissorRegister ) == sizeof( GX2ScissorReg ) );

    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_SDK_ASSERT( pDevice );
    NN_UNUSED( pDevice );

    NN_SDK_ASSERT( info.GetViewportCount() == 1 ); // As of 2.12.13, Cafe doesn't support multiple viewports

    NN_SDK_ASSERT( info.GetViewportStateInfoArray() );
    const ViewportStateInfo& viewport = *info.GetViewportStateInfoArray();
    NN_GFX_CALL_GX_FUNCTION( GX2InitViewportReg(
        static_cast< GX2ViewportReg* >( static_cast< void* >( this->viewportRegister ) ),
        viewport.GetOriginX(),
        viewport.GetOriginY(),
        viewport.GetWidth(),
        viewport.GetHeight(),
        viewport.GetMinDepth(),
        viewport.GetMaxDepth() ) );

    if( info.IsScissorEnabled() )
    {
        NN_SDK_ASSERT( info.GetScissorStateInfoArray() );
        const ScissorStateInfo& scissor = *info.GetScissorStateInfoArray();
        NN_GFX_CALL_GX_FUNCTION( GX2InitScissorReg(
            static_cast< GX2ScissorReg* >( static_cast< void* >( this->scissorRegister ) ),
            scissor.GetOriginX(),
            scissor.GetOriginY(),
            scissor.GetWidth(),
            scissor.GetHeight() ) );
    }
    else
    {
        // GX2 doesn't support turning scissoring off, so just use the maximums.
        NN_GFX_CALL_GX_FUNCTION( GX2InitScissorReg( static_cast< GX2ScissorReg* >( static_cast< void* >(
            this->scissorRegister ) ), 0, 0, GX2_MAX_VIEWPORT_SIZE, GX2_MAX_VIEWPORT_SIZE ) );
    }

    this->state = State_Initialized;
}

void ViewportScissorStateImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_UNUSED( pDevice );

    this->state = State_NotInitialized;
}

}
}
}
