﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_StaticAssert.h>

#include <nn/util/util_BitPack.h>
#include <nn/util/util_BytePtr.h>

#include <nn/gfx/gfx_StateInfo.h>

#include <nn/gfx/detail/gfx_State-api.gl.4.h>
#include <nn/gfx/detail/gfx_Shader-api.gl.4.h>
#include <nn/gfx/detail/gfx_Misc.h>

#include "gfx_CommonHelper.h"
#include "gfx_GlHelper.h"
#include "gfx_GlCommand.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationGl4 Target;

namespace {

void InitializeStencilState( DepthStencilStateImpl< Target >::DataType::StencilState* pDst,
    const StencilStateInfo& src, int readMask, int writeMask ) NN_NOEXCEPT
{
    static const GLenum s_StencilOpTable[] =
    {
        GL_KEEP,
        GL_ZERO,
        GL_REPLACE,
        GL_INCR,
        GL_DECR,
        GL_INVERT,
        GL_INCR_WRAP,
        GL_DECR_WRAP
    };
    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( s_StencilOpTable ) == StencilOperation_End );

    pDst->sfail = s_StencilOpTable[ src.GetStencilFailOperation() ];
    pDst->dpfail = s_StencilOpTable[ src.GetDepthFailOperation() ];
    pDst->dppass = s_StencilOpTable[ src.GetDepthPassOperation() ];
    pDst->func = src.GetComparisonFunction() + ( GL_NEVER - ComparisonFunction_Never );
    pDst->ref = src.GetStencilRef();
    pDst->mask = readMask & writeMask; // 要検討
}

}

RasterizerStateImpl< Target >::RasterizerStateImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

RasterizerStateImpl< Target >::~RasterizerStateImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void RasterizerStateImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( state == State_NotInitialized );
    NN_UNUSED( pDevice );

    static const GLenum s_PolygonModeTable[] =
    {
        GL_POINT,
        GL_LINE,
        GL_FILL
    };
    this->polygonMode = s_PolygonModeTable[ info.GetFillMode() ];

    static const GLenum s_FrontFaceTable[] =
    {
        GL_CCW,
        GL_CW
    };
    this->frontFace = s_FrontFaceTable[ info.GetFrontFace() ];

    static const GLenum s_CullModeTable[] =
    {
        GL_NONE,
        GL_FRONT,
        GL_BACK
    };
    this->flag.SetBit( Flag_CullEnable, info.GetCullMode() != CullMode_None );
    this->cullFace = s_CullModeTable[ info.GetCullMode() ];

    this->flag.SetBit( Flag_RasterDisable, !info.IsRasterEnabled() );
    this->flag.SetBit( Flag_MultisampleEnable, info.IsMultisampleEnabled() );
    this->flag.SetBit( Flag_DepthClampEnable, !info.IsDepthClipEnabled() );

    this->flag.SetBit( Flag_PolygonOffsetEnable, info.GetDepthBias() != 0 || info.GetSlopeScaledDepthBias() != 0.0f );
    this->slopeScaledDepthBias = info.GetSlopeScaledDepthBias();
    this->depthBias = static_cast< float >( info.GetDepthBias() );
    this->depthBiasClamp = info.GetDepthBiasClamp();

    this->sampleMask = info.GetMultisampleStateInfo().GetSampleMask();

    this->flag.SetBit( Flag_ConservativeRasterEnable,
        info.GetConservativeRasterizationMode() == ConservativeRasterizationMode_Enable );

    this->state = State_Initialized;
}

void RasterizerStateImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_UNUSED( pDevice );

    this->state = State_NotInitialized;
}

size_t BlendStateImpl< Target >::GetRequiredMemorySize( const InfoType& info ) NN_NOEXCEPT
{
    return sizeof( BlendTargetState ) * info.GetBlendTargetCount();
}

BlendStateImpl< Target >::BlendStateImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

BlendStateImpl< Target >::~BlendStateImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void BlendStateImpl< Target >::SetMemory( void* pMemory, size_t size ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pMemory || size == 0 );

    this->pTargetArray = static_cast< BlendTargetState* >( pMemory );
    this->memorySize = static_cast< uint32_t >( size );
}

void* BlendStateImpl< Target >::GetMemory() NN_NOEXCEPT
{
    return this->pTargetArray;
}

const void* BlendStateImpl< Target >::GetMemory() const NN_NOEXCEPT
{
    return this->pTargetArray;
}

void BlendStateImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_UNUSED( pDevice );

    NN_SDK_REQUIRES( this->memorySize >= GetRequiredMemorySize( info ) );

    this->flag = info.ToData()->flag;

    this->logicOp = info.GetLogicOperation() + ( GL_CLEAR - LogicOperation_Clear );

    this->blendColor[ 0 ] = info.GetBlendConstant( ColorChannel_Red );
    this->blendColor[ 1 ] = info.GetBlendConstant( ColorChannel_Green );
    this->blendColor[ 2 ] = info.GetBlendConstant( ColorChannel_Blue );
    this->blendColor[ 3 ] = info.GetBlendConstant( ColorChannel_Alpha );

    static const GLenum s_BlendFactorTable[] =
    {
        GL_ZERO,
        GL_ONE,
        GL_SRC_COLOR,
        GL_ONE_MINUS_SRC_COLOR,
        GL_DST_COLOR,
        GL_ONE_MINUS_DST_COLOR,
        GL_SRC_ALPHA,
        GL_ONE_MINUS_SRC_ALPHA,
        GL_DST_ALPHA,
        GL_ONE_MINUS_DST_ALPHA,
        GL_CONSTANT_COLOR,
        GL_ONE_MINUS_CONSTANT_COLOR,
        GL_CONSTANT_ALPHA,
        GL_ONE_MINUS_CONSTANT_ALPHA,
        GL_SRC_ALPHA_SATURATE,
        GL_SRC1_COLOR,
        GL_ONE_MINUS_SRC1_COLOR,
        GL_SRC1_ALPHA,
        GL_ONE_MINUS_SRC1_ALPHA
    };
    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( s_BlendFactorTable ) == BlendFactor_End );

    static const GLenum s_BlendFuncTable[] =
    {
        GL_FUNC_ADD,
        GL_FUNC_SUBTRACT,
        GL_FUNC_REVERSE_SUBTRACT,
        GL_MIN,
        GL_MAX
    };

    this->blendTargetCount = info.GetBlendTargetCount();
    const BlendTargetStateInfo* pTargetStates = info.GetBlendTargetStateInfoArray();
    NN_SDK_ASSERT( pTargetStates || this->blendTargetCount < 1 );
    int targetCount = info.IsIndependentBlendEnabled() ?
        this->blendTargetCount : ( this->blendTargetCount > 0 ? 1 : 0 );
    for( int idxTarget = 0; idxTarget < targetCount; ++idxTarget )
    {
        const BlendTargetStateInfo& src = pTargetStates[ idxTarget ];
        BlendTargetState& dst = this->pTargetArray.ptr[ idxTarget ];
        dst.colorMask = static_cast< Bit8 >( src.GetChannelMask() );
        dst.srcRGB = s_BlendFactorTable[ src.GetSourceColorBlendFactor() ];
        dst.dstRGB = s_BlendFactorTable[ src.GetDestinationColorBlendFactor() ];
        dst.srcAlpha = s_BlendFactorTable[ src.GetSourceAlphaBlendFactor() ];
        dst.dstAlpha = s_BlendFactorTable[ src.GetDestinationAlphaBlendFactor() ];
        dst.modeRGB = s_BlendFuncTable[ src.GetColorBlendFunction() ];
        dst.modeAlpha = s_BlendFuncTable[ src.GetAlphaBlendFunction() ];
        dst.flag = src.ToData()->flag;
    }

    this->state = State_Initialized;
}

void BlendStateImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_UNUSED( pDevice );

    this->state = State_NotInitialized;
}

DepthStencilStateImpl< Target >::DepthStencilStateImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

DepthStencilStateImpl< Target >::~DepthStencilStateImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void DepthStencilStateImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_UNUSED( pDevice );

    this->flag = info.ToData()->flag;

    this->depthFunc = info.GetDepthComparisonFunction() + ( GL_NEVER - ComparisonFunction_Never );

    InitializeStencilState( &this->frontStencil, info.GetFrontStencilStateInfo(),
        info.GetStencilReadMask(), info.GetStencilWriteMask() );
    InitializeStencilState( &this->backStencil, info.GetBackStencilStateInfo(),
        info.GetStencilReadMask(), info.GetStencilWriteMask() );

    this->state = State_Initialized;
}

void DepthStencilStateImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_UNUSED( pDevice );

    this->state = State_NotInitialized;
}

size_t VertexStateImpl< Target >::GetRequiredMemorySize( const InfoType& info ) NN_NOEXCEPT
{
    return info.GetVertexAttributeCount() * sizeof( AttributeState )
        + info.GetVertexBufferCount() * sizeof( uint32_t );
}

VertexStateImpl< Target >::VertexStateImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

VertexStateImpl< Target >::~VertexStateImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void VertexStateImpl< Target >::SetMemory( void* value, size_t size ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( value || size == 0 );

    this->pWorkMemory = value;
    this->memorySize = static_cast< uint32_t >( size );
}

void* VertexStateImpl< Target >::GetMemory() NN_NOEXCEPT
{
    return this->pWorkMemory;
}

const void* VertexStateImpl< Target >::GetMemory() const NN_NOEXCEPT
{
    return this->pWorkMemory;
}

void VertexStateImpl< Target >::Initialize( DeviceImpl< Target >* pDevice,
    const InfoType& info, const ShaderImpl< Target >* pVertexShader ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_SDK_REQUIRES( pVertexShader == NULL || IsInitialized( *pVertexShader ) );
    NN_SDK_REQUIRES( this->memorySize >= GetRequiredMemorySize( info ) );

    NN_SDK_ASSERT( pVertexShader == NULL || IsValid( pVertexShader->ToData()->hVertexProgram ) );

    GlDeviceActivator activator( pDevice );

    this->attributeCount = info.GetVertexAttributeCount();
    this->bufferCount = info.GetVertexBufferCount();

    nn::util::BytePtr ptr( this->pWorkMemory.ptr );
    AttributeState* pAttributeArray = ptr.Get< AttributeState >();
    uint32_t* pDivisorArray = ptr.Advance( sizeof( AttributeState ) * this->attributeCount ).Get< uint32_t >();

    const VertexAttributeStateInfo* pAttributes = info.GetVertexAttributeStateInfoArray();
    NN_SDK_ASSERT( pAttributes || this->attributeCount < 1 );
    for( int idxAttribute = 0; idxAttribute < this->attributeCount; ++idxAttribute )
    {
        const VertexAttributeStateInfo& src = pAttributes[ idxAttribute ];
        AttributeState& dst = pAttributeArray[ idxAttribute ];
        dst.slot = src.GetShaderSlot();
        AttributeFormat attributeFormat = static_cast< AttributeFormat >( src.GetFormat() );
        GlAttributeFormat format = Gl::GetAttributeFormat( attributeFormat );
        dst.typeFormat = GetTypeFormat( attributeFormat );
        dst.size = format.size;
        dst.type = format.type;
        dst.normalized = format.normalized;
        dst.offset = static_cast< uint32_t >( src.GetOffset() );
        dst.bindingIndex = src.GetBufferIndex();
        if( pVertexShader && src.GetNamePtr() )
        {
            dst.slot = NN_GFX_CALL_GL_FUNCTION( ::glGetProgramResourceLocation(
                pVertexShader->ToData()->hVertexProgram, GL_PROGRAM_INPUT, src.GetNamePtr() ) );
        }
    }

    NN_SDK_ASSERT( pDivisorArray || this->bufferCount < 1 );
    for( int idxBuffer = 0; idxBuffer < this->bufferCount; ++idxBuffer )
    {
        pDivisorArray[ idxBuffer ] = static_cast< uint32_t >(
            info.GetVertexBufferStateInfoArray()[ idxBuffer ].GetDivisor() );
    }

    this->state = State_Initialized;
}

void VertexStateImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_UNUSED( pDevice );

    this->state = State_NotInitialized;
}

TessellationStateImpl< Target >::TessellationStateImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

TessellationStateImpl< Target >::~TessellationStateImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void TessellationStateImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_UNUSED( pDevice );

    this->patchControlPointCount = info.GetPatchControlPointCount();

    this->state = State_Initialized;
}

void TessellationStateImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_UNUSED( pDevice );

    this->state = State_NotInitialized;
}

size_t ViewportScissorStateImpl< Target >::GetRequiredMemorySize( const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( info.GetViewportCount() > 0 );
    NN_SDK_ASSERT( !info.IsScissorEnabled() || info.GetViewportCount() == info.GetScissorCount() );
    int extraViewportCount = info.GetViewportCount() - 1;
    return sizeof( float ) * 4 * extraViewportCount + sizeof( double ) * 2 * extraViewportCount +
        ( info.IsScissorEnabled() ? sizeof( int32_t ) * 4 * extraViewportCount : 0 );
}

ViewportScissorStateImpl< Target >::ViewportScissorStateImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

ViewportScissorStateImpl< Target >::~ViewportScissorStateImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void ViewportScissorStateImpl< Target >::SetMemory( void* value, size_t size ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( value || size == 0 );

    this->pWorkMemory = value;
    this->memorySize = static_cast< uint32_t >( size );
}

void* ViewportScissorStateImpl< Target >::GetMemory() NN_NOEXCEPT
{
    return this->pWorkMemory;
}

const void* ViewportScissorStateImpl< Target >::GetMemory() const NN_NOEXCEPT
{
    return this->pWorkMemory;
}

void ViewportScissorStateImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_UNUSED( pDevice );

    NN_SDK_ASSERT( info.GetViewportCount() > 0 );
    NN_SDK_ASSERT( !info.IsScissorEnabled() || info.GetViewportCount() == info.GetScissorCount() );

    NN_SDK_REQUIRES( info.GetViewportCount() < 2 || this->pWorkMemory );
    NN_SDK_REQUIRES( info.GetViewportCount() < 2 || this->memorySize >= GetRequiredMemorySize( info ) );

    this->flag = info.ToData()->flag;

    this->viewportCount = info.GetViewportCount();

    const ViewportStateInfo* pViewports = info.GetViewportStateInfoArray();
    const ScissorStateInfo* pScissor = info.GetScissorStateInfoArray();
    NN_SDK_ASSERT_NOT_NULL( pViewports );
    NN_SDK_ASSERT( !info.IsScissorEnabled() || pScissor );

    this->viewport[ 0 ] = pViewports[ 0 ].GetOriginX();
    this->viewport[ 1 ] = pViewports[ 0 ].GetOriginY();
    this->viewport[ 2 ] = pViewports[ 0 ].GetWidth();
    this->viewport[ 3 ] = pViewports[ 0 ].GetHeight();
    this->depthRange[ 0 ] = pViewports[ 0 ].GetMinDepth();
    this->depthRange[ 1 ] = pViewports[ 0 ].GetMaxDepth();
    if( info.IsScissorEnabled() )
    {
        this->scissor[ 0 ] = pScissor[ 0 ].GetOriginX();
        this->scissor[ 1 ] = pScissor[ 0 ].GetOriginY();
        this->scissor[ 2 ] = pScissor[ 0 ].GetWidth();
        this->scissor[ 3 ] = pScissor[ 0 ].GetHeight();
    }

    int extraViewportCount = info.GetViewportCount() - 1;

    nn::util::BytePtr ptr( this->pWorkMemory.ptr );
    float* viewportArray = ptr.Get< float >();
    double* depthRangeArray = ptr.Advance( sizeof( float ) * 4 * extraViewportCount ).Get< double >();
    int32_t* scissorArray = ptr.Advance( sizeof( double ) * 2 * extraViewportCount ).Get< int32_t >();

    for( int idx = 1; idx < this->viewportCount; ++idx )
    {
        int idxExtra = idx - 1;
        const ViewportStateInfo& viewportInfo = pViewports[ idx ];
        viewportArray[ idxExtra * 4 + 0 ] = viewportInfo.GetOriginX();
        viewportArray[ idxExtra * 4 + 1 ] = viewportInfo.GetOriginY();
        viewportArray[ idxExtra * 4 + 2 ] = viewportInfo.GetWidth();
        viewportArray[ idxExtra * 4 + 3 ] = viewportInfo.GetHeight();
        depthRangeArray[ idxExtra * 2 + 0 ] = viewportInfo.GetMinDepth();
        depthRangeArray[ idxExtra * 2 + 1 ] = viewportInfo.GetMaxDepth();
    }

    if( info.IsScissorEnabled() )
    {
        for( int idx = 1; idx < this->viewportCount; ++idx )
        {
            int idxExtra = idx - 1;
            const ScissorStateInfo& scissorInfo = pScissor[ idx ];
            scissorArray[ idxExtra * 4 + 0 ] = scissorInfo.GetOriginX();
            scissorArray[ idxExtra * 4 + 1 ] = scissorInfo.GetOriginY();
            scissorArray[ idxExtra * 4 + 2 ] = scissorInfo.GetWidth();
            scissorArray[ idxExtra * 4 + 3 ] = scissorInfo.GetHeight();
        }
    }

    this->state = State_Initialized;
}

void ViewportScissorStateImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_UNUSED( pDevice );

    this->state = State_NotInitialized;
}

}
}
}
