﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkLog.h>

#include <nn/gfx/gfx_SamplerInfo.h>

#include <nn/gfx/detail/gfx_Sampler-api.vk.1.h>

#include "gfx_VkHelper.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationVk1 Target;

SamplerImpl< Target >::SamplerImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

SamplerImpl< Target >::~SamplerImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized || this->flags.GetBit( Flag_Shared ) );
}

void SamplerImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_SDK_ASSERT_NOT_NULL( pDevice );

    const VkBorderColor BorderColors[] =
    {
        VK_BORDER_COLOR_FLOAT_OPAQUE_WHITE, // TextureBorderColorType_White
        VK_BORDER_COLOR_FLOAT_TRANSPARENT_BLACK, // TextureBorderColorType_TransparentBlack
        VK_BORDER_COLOR_FLOAT_OPAQUE_BLACK, // TextureBorderColorType_OpaqueBlack
    };
    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( BorderColors ) == TextureBorderColorType_End );

    NN_SDK_REQUIRES( info.GetMaxAnisotropy() == 1
        || pDevice->ToData()->supportedFeaturesBitArray.test( VkDeviceFeature_SamplerAnisotropy ) );

    VkDevice vkDevice = CastToVkDispatchableObject< VkDevice >( pDevice->ToData()->hDevice );
    VkSamplerCreateInfo createInfo;
    createInfo.sType = VK_STRUCTURE_TYPE_SAMPLER_CREATE_INFO;
    createInfo.pNext = NULL;
    createInfo.flags = 0;
    createInfo.magFilter = Vk::GetMagFilter( info.GetFilterMode() );
    createInfo.minFilter = Vk::GetMinFilter( info.GetFilterMode() );
    createInfo.addressModeU = Vk::GetWrapMode( info.GetAddressU() );
    createInfo.addressModeV = Vk::GetWrapMode( info.GetAddressV() );
    createInfo.addressModeW = Vk::GetWrapMode( info.GetAddressW() );
    createInfo.mipLodBias = info.GetLodBias();
    createInfo.anisotropyEnable = info.GetMaxAnisotropy() != 1.0 ? VK_TRUE : VK_FALSE;
    createInfo.maxAnisotropy = static_cast< float >( info.GetMaxAnisotropy() );
    if ( info.GetFilterMode() & FilterModeBit_Comparison )
    {
        createInfo.compareEnable = VK_TRUE;
        createInfo.compareOp = Vk::GetCompareOperation( info.GetComparisonFunction() );
    }
    else
    {
        createInfo.compareEnable = VK_FALSE;
        createInfo.compareOp = VK_COMPARE_OP_ALWAYS;
    }
    if ( Vk::IsMipMapUsed( info.GetFilterMode() ) )
    {
        createInfo.minLod = info.GetMinLod();
        createInfo.maxLod = info.GetMaxLod();
        createInfo.mipmapMode = Vk::GetMipmapMode( info.GetFilterMode() );
    }
    else
    {
        createInfo.minLod = 0.f;
        createInfo.maxLod = 0.25f;
        createInfo.mipmapMode = VK_SAMPLER_MIPMAP_MODE_NEAREST;
    }
    createInfo.borderColor = BorderColors[ info.GetBorderColorType() ];
    createInfo.unnormalizedCoordinates = VK_FALSE;

    VkResult result;
    VkSampler sampler;

    NN_GFX_CALL_VK_FUNCTION( result = vkCreateSampler( vkDevice, &createInfo,
        pDevice->ToData()->pAllocationCallback, &sampler ) );
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
    NN_UNUSED( result );
    this->ToData()->hSampler = CastFromVkNonDispatchableObject< VkSampler >( sampler );

    this->flags.SetBit( Flag_Shared, false );
    this->state = State_Initialized;
    memset( &this->createInfoCopy, 0, sizeof( createInfoCopy ) );
    this->createInfoCopy.magFilter = static_cast< int32_t >( createInfo.magFilter );
    this->createInfoCopy.minFilter = static_cast< int32_t >( createInfo.minFilter );
    this->createInfoCopy.mipmapMode = static_cast< int32_t >( createInfo.mipmapMode );
    this->createInfoCopy.addressModeU = static_cast< int32_t >( createInfo.addressModeU );
    this->createInfoCopy.addressModeV = static_cast< int32_t >( createInfo.addressModeV );
    this->createInfoCopy.addressModeW = static_cast< int32_t >( createInfo.addressModeW );
    this->createInfoCopy.mipLoadBias = createInfo.mipLodBias;
    this->createInfoCopy.maxAnisotropy = createInfo.maxAnisotropy;
    this->createInfoCopy.compareOp = static_cast< int32_t >( createInfo.compareOp );
    this->createInfoCopy.minLod = createInfo.minLod;
    this->createInfoCopy.maxLod = createInfo.maxLod;
    this->createInfoCopy.borderColor = static_cast< int32_t >( createInfo.borderColor );
    this->createInfoCopy.flags.SetBit( CreateInfoCopy::Flag::Flag_AnisotropyEnable, createInfo.anisotropyEnable != 0 );
    this->createInfoCopy.flags.SetBit( CreateInfoCopy::Flag::Flag_CompareEnable, createInfo.compareEnable != 0 );
    this->createInfoCopy.flags.SetBit( CreateInfoCopy::Flag::Flag_UnnormalizedCoordinates, createInfo.unnormalizedCoordinates != 0 );
}

void SamplerImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsInitialized( *this ) );
    NN_SDK_ASSERT( !this->flags.GetBit( Flag_Shared ) );
    VkDevice vkDevice = CastToVkDispatchableObject< VkDevice >( pDevice->ToData()->hDevice );
    VkSampler vkSampler = CastToVkNonDispatchableObject< VkSampler >( this->hSampler );

    if ( vkSampler )
    {
        NN_GFX_CALL_VK_FUNCTION( vkDestroySampler( vkDevice, vkSampler,
            pDevice->ToData()->pAllocationCallback ) );
    }

    this->state = State_NotInitialized;
}

}
}
}
