﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/gfx/gfx_SamplerInfo.h>

#include <nn/gfx/detail/gfx_Sampler-api.gl.4.h>

#include "gfx_GlHelper.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationGl4 Target;

SamplerImpl< Target >::SamplerImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

SamplerImpl< Target >::~SamplerImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized || this->flags.GetBit( Flag_Shared ) );
}

void SamplerImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );

    GlDeviceActivator activator( pDevice );

    NN_SDK_ASSERT( !IsValid( this->hSampler ) );
    NN_GFX_CALL_GL_FUNCTION( ::glGenSamplers( 1, &this->hSampler ) );
    NN_GFX_GL_ASSERT();
    NN_SDK_ASSERT( IsValid( this->hSampler ) );

    static const GLint s_FilterModeTable[] =
    {
        0,
        GL_NEAREST,
        GL_LINEAR,
        0,
        0,
        GL_NEAREST_MIPMAP_NEAREST,
        GL_LINEAR_MIPMAP_NEAREST,
        0,
        0,
        GL_NEAREST_MIPMAP_LINEAR,
        GL_LINEAR_MIPMAP_LINEAR
    };

    NN_GFX_CALL_GL_FUNCTION( ::glSamplerParameteri( this->hSampler, GL_TEXTURE_MIN_FILTER,
        s_FilterModeTable[ ( ( info.GetFilterMode() & FilterModeBit_MipFilterMask ) << 2 ) |
        ( ( info.GetFilterMode() & FilterModeBit_MinFilterMask ) >> FilterModeBit_MinFilterShift ) ] ) );
    NN_GFX_CALL_GL_FUNCTION( ::glSamplerParameteri( this->hSampler, GL_TEXTURE_MAG_FILTER,
        s_FilterModeTable[ ( info.GetFilterMode() & FilterModeBit_MagFilterMask ) >> FilterModeBit_MagFilterShift ] ) );
    NN_GFX_CALL_GL_FUNCTION( ::glSamplerParameterf( this->hSampler,
        GL_TEXTURE_MAX_ANISOTROPY_EXT, static_cast< GLfloat >( info.GetMaxAnisotropy() ) ) );

    static const GLint s_AddressModeTable[] =
    {
        GL_REPEAT,
        GL_MIRRORED_REPEAT,
        GL_CLAMP_TO_EDGE,
        GL_CLAMP_TO_BORDER,
        GL_MIRROR_CLAMP_TO_EDGE
    };

    NN_GFX_CALL_GL_FUNCTION( ::glSamplerParameteri(
        this->hSampler, GL_TEXTURE_WRAP_S, s_AddressModeTable[ info.GetAddressU() ] ) );
    NN_GFX_CALL_GL_FUNCTION( ::glSamplerParameteri(
        this->hSampler, GL_TEXTURE_WRAP_T, s_AddressModeTable[ info.GetAddressV() ] ) );
    NN_GFX_CALL_GL_FUNCTION( ::glSamplerParameteri(
        this->hSampler, GL_TEXTURE_WRAP_R, s_AddressModeTable[ info.GetAddressW() ] ) );

    if( info.GetFilterMode() & 0x80 )
    {
        NN_GFX_CALL_GL_FUNCTION( ::glSamplerParameteri( this->hSampler,
            GL_TEXTURE_COMPARE_MODE, GL_COMPARE_REF_TO_TEXTURE ) );
        NN_GFX_CALL_GL_FUNCTION( ::glSamplerParameteri( this->hSampler,
            GL_TEXTURE_COMPARE_FUNC, info.GetComparisonFunction() +
            ( GL_NEVER - ComparisonFunction_Never ) ) );
    }
    else
    {
        NN_GFX_CALL_GL_FUNCTION( ::glSamplerParameteri( this->hSampler,
            GL_TEXTURE_COMPARE_MODE, GL_NONE ) );
    }

    static const GLfloat s_BorderColorTable[][ 4 ] =
    {
        { 1.0f, 1.0f, 1.0f, 1.0f },
        { 0.0f, 0.0f, 0.0f, 0.0f },
        { 0.0f, 0.0f, 0.0f, 1.0f }
    };
    NN_GFX_CALL_GL_FUNCTION( ::glSamplerParameterfv( this->hSampler,
        GL_TEXTURE_BORDER_COLOR, s_BorderColorTable[ info.GetBorderColorType() ] ) );

    NN_GFX_CALL_GL_FUNCTION( ::glSamplerParameterf( this->hSampler,
        GL_TEXTURE_MIN_LOD, info.GetMinLod() ) );
    NN_GFX_CALL_GL_FUNCTION( ::glSamplerParameterf( this->hSampler,
        GL_TEXTURE_MAX_LOD, info.GetMaxLod() ) );
    NN_GFX_CALL_GL_FUNCTION( ::glSamplerParameterf( this->hSampler,
        GL_TEXTURE_LOD_BIAS, info.GetLodBias() ) );

    NN_GFX_GL_ASSERT();

    this->flags.SetBit( Flag_Shared, false );
    this->state = State_Initialized;
}

void SamplerImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_ASSERT( !this->flags.GetBit( Flag_Shared ) );

    GlDeviceActivator activator( pDevice );

    if( this->hSampler )
    {
        NN_GFX_CALL_GL_FUNCTION( ::glDeleteSamplers( 1, &this->hSampler ) );
        NN_GFX_GL_ASSERT();
        this->hSampler = GlInvalidHandle;
    }

    this->state = State_NotInitialized;
}

}
}
}
