﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/gfx/gfx_Variation-api.nvn.h>
#include <nn/gfx/gfx_ResShader.h>
#include <nn/gfx/gfx_Device.h>
#include <nn/gfx/gfx_MemoryPoolInfo.h>
#include <nn/gfx/gfx_BufferInfo.h>
#include <nn/gfx/gfx_ResShaderData-api.nvn.h>

#include <nn/gfx/detail/gfx_ResShaderImpl.h>
#include <nn/gfx/detail/gfx_MemoryPool-api.nvn.8.h>
#include <nn/gfx/detail/gfx_Buffer-api.nvn.8.h>
#include <nn/gfx/detail/gfx_Device-api.nvn.8.h>

#include "gfx_CommonHelper.h"
#include "gfx_NvnHelper.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationNvn8 Target;

template<>
size_t ResShaderProgramImpl::NvnGetRecommendedScrachMemorySize< Target >(
    const ResShaderProgram* pThis, DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pThis );
    NN_SDK_ASSERT_NOT_NULL( pDevice );

    static detail::Ptr< const void > ShaderInfoData::* s_pStageCodes[] =
    {
        &ShaderInfoData::pVertexShaderCode,
        &ShaderInfoData::pHullShaderCode,
        &ShaderInfoData::pDomainShaderCode,
        &ShaderInfoData::pGeometryShaderCode,
        &ShaderInfoData::pPixelShaderCode,
        &ShaderInfoData::pComputeShaderCode
    };

    const ResShaderProgram::value_type& data = pThis->ToData();
    uint32_t maxSizeRecommended = 0;
    uint32_t maxSizePerWarp = 0;
    for( int idxStage = 0; idxStage < ShaderStage_End; ++idxStage )
    {
        if( const NvnShaderCode* pNvnShaderData = static_cast<
            const NvnShaderCode* >( data.info.*s_pStageCodes[ idxStage ] ) )
        {
            maxSizeRecommended = std::max NN_PREVENT_MACRO_FUNC(
                maxSizeRecommended, pNvnShaderData->scratchMemoryRecommended );
            maxSizePerWarp = std::max NN_PREVENT_MACRO_FUNC (
                maxSizePerWarp, pNvnShaderData->scratchMemoryPerWarp );
        }
    }

    uint32_t maxSize = 0;
    if( maxSizeRecommended > 0 )
    {
        maxSize = maxSizeRecommended;
    }
    else if( maxSizePerWarp > 0 )
    {
        int scaleFactor;
        NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger( pDevice->ToData()->pNvnDevice,
            NVN_DEVICE_INFO_SHADER_SCRATCH_MEMORY_SCALE_FACTOR_RECOMMENDED, &scaleFactor ) );
        maxSize = maxSizePerWarp * scaleFactor;
    }
    else
    {
        return 0;
    }

    int granularity;
    NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger( pDevice->ToData()->pNvnDevice,
        NVN_DEVICE_INFO_SHADER_SCRATCH_MEMORY_GRANULARITY, &granularity ) );
    return ( maxSize + granularity - 1 ) / granularity * granularity;
}

template<>
void ResShaderContainerImpl::Initialize< Target >( ResShaderContainer* pThis, DeviceImpl< Target >* pDevice,
    MemoryPoolImpl< Target >* pMemoryPool, ptrdiff_t memoryPoolOffset, size_t memoryPoolSize ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pThis );
    NN_SDK_ASSERT_NOT_NULL( pDevice );
    NN_UNUSED( memoryPoolSize );

    ResShaderContainer::value_type& data = pThis->ToData();
    NvnShaderPool* pShaderPool = static_cast< NvnShaderPool* >( data.pShaderBinaryPool.Get() );
    MemoryPoolInfo& memoryPoolInfo = DataToAccessor( pShaderPool->memoryPoolInfo );

    NN_SDK_ASSERT( CheckBinaryTarget( data, Target::Type::value, Target::Version::value ) );

    int majorVersion = static_cast< uint16_t >( data.lowLevelCompilerVersion >> 16 );
    int minorVersion = static_cast< uint16_t >( data.lowLevelCompilerVersion >> 0 );
    int minMajorVersion;
    int minMinorVersion;
    int maxMajorVersion;
    int maxMinorVersion;
    NN_SDK_ASSERT( (
        NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger( pDevice->ToData()->pNvnDevice,
            NVN_DEVICE_INFO_GLSLC_MIN_SUPPORTED_GPU_CODE_MAJOR_VERSION, &minMajorVersion ) ),
        NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger( pDevice->ToData()->pNvnDevice,
            NVN_DEVICE_INFO_GLSLC_MIN_SUPPORTED_GPU_CODE_MINOR_VERSION, &minMinorVersion ) ),
        NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger( pDevice->ToData()->pNvnDevice,
            NVN_DEVICE_INFO_GLSLC_MAX_SUPPORTED_GPU_CODE_MAJOR_VERSION, &maxMajorVersion ) ),
        NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger( pDevice->ToData()->pNvnDevice,
            NVN_DEVICE_INFO_GLSLC_MAX_SUPPORTED_GPU_CODE_MINOR_VERSION, &maxMinorVersion ) ),
        ( majorVersion >= minMajorVersion && minorVersion >= minMinorVersion &&
        majorVersion <= maxMajorVersion && minorVersion <= maxMinorVersion ) ||
        majorVersion == 0 && minorVersion == 0 ),
        "Glslc shader version mismatch:\n  Binary: %d.%d\n  Supported: %d.%d - %d.%d\n",
        majorVersion, minorVersion, minMajorVersion, minMinorVersion, maxMajorVersion, maxMinorVersion );
    NN_UNUSED( majorVersion );
    NN_UNUSED( minorVersion );
    NN_UNUSED( minMajorVersion );
    NN_UNUSED( minMinorVersion );
    NN_UNUSED( maxMajorVersion );
    NN_UNUSED( maxMinorVersion );

    NVNbufferAddress headAddress;
    if( pMemoryPool )
    {
        // 外部メモリープールの場合
        NN_SDK_ASSERT( static_cast< ptrdiff_t >( memoryPoolSize ) >= nn::util::BytePtr( pThis ).Distance(
            pShaderPool->memoryPoolInfo.pMemory ) + static_cast< ptrdiff_t >( pShaderPool->memoryPoolInfo.memorySize ) );
        pShaderPool->pCurrentMemoryPool.Set( pMemoryPool );
        ptrdiff_t offset = memoryPoolOffset + nn::util::BytePtr(
            pThis ).Distance( pShaderPool->memoryPoolInfo.pMemory );
        headAddress = NN_GFX_CALL_NVN_FUNCTION(
            nvnMemoryPoolGetBufferAddress( pMemoryPool->ToData()->pNvnMemoryPool ) );
        NN_SDK_ASSERT( nn::util::is_aligned( offset, 256 ) );
        headAddress += offset;
    }
    else
    {
        MemoryPoolImpl< Target >* pInternalMemoryPool = static_cast<
            MemoryPoolImpl< Target >* >( pShaderPool->pMemoryPool.Get() );
        NN_SDK_ASSERT_NOT_NULL( pInternalMemoryPool );
        pInternalMemoryPool->Initialize( pDevice, memoryPoolInfo );
        pShaderPool->pCurrentMemoryPool.Set( pInternalMemoryPool );
        headAddress = NN_GFX_CALL_NVN_FUNCTION(
            nvnMemoryPoolGetBufferAddress( pInternalMemoryPool->ToData()->pNvnMemoryPool ) );
    }

    // 各プログラムの data のバッファーアドレスを埋める
    for( int idxVariation = 0, variationCount = pThis->GetShaderVariationCount();
        idxVariation < variationCount; ++idxVariation )
    {
        ResShaderVariation* pResShaderVariation = pThis->GetResShaderVariation( idxVariation );
        NN_SDK_ASSERT_NOT_NULL( pResShaderVariation );
        if( ResShaderProgram* pResShaderProgram =
            pResShaderVariation->GetResShaderProgram( ShaderCodeType_Binary ) )
        {
            ShaderInfo* pShaderInfo = pResShaderProgram->GetShaderInfo();
            for( int idxStage = 0; idxStage < ShaderStage_End; ++idxStage )
            {
                if( NvnShaderCode* pNvnShaderCode = const_cast< NvnShaderCode* >(
                    static_cast< const NvnShaderCode* >( pShaderInfo->GetShaderCodePtr(
                    static_cast< ShaderStage >( idxStage ) ) ) ) )
                {
                    pNvnShaderCode->dataAddress = headAddress + nn::util::BytePtr(
                        memoryPoolInfo.GetPoolMemory() ).Distance( pNvnShaderCode->pData.Get() );
                }
            }
        }
    }

    pThis->ToData().targetCodeType = static_cast< uint8_t >( ShaderCodeType_Binary );
}

template<>
void ResShaderContainerImpl::Finalize< Target >(
    ResShaderContainer* pThis, DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pThis );
    NN_SDK_ASSERT_NOT_NULL( pDevice );

    ResShaderContainer::value_type& data = pThis->ToData();
    NvnShaderPool* pShaderPool = static_cast< NvnShaderPool* >( data.pShaderBinaryPool.Get() );

    MemoryPoolImpl< Target >* pCurrentMemoryPool = static_cast<
        MemoryPoolImpl< Target >* >( pShaderPool->pCurrentMemoryPool.Get() );
    NN_SDK_ASSERT_NOT_NULL( pCurrentMemoryPool );
    if( pCurrentMemoryPool == pShaderPool->pMemoryPool.Get() )
    {
        pCurrentMemoryPool->Finalize( pDevice );
    }
    pShaderPool->pCurrentMemoryPool.Set( NULL );
}

}

template<>
size_t NvnGetMaxRecommendedScratchMemorySize( TDevice< ApiVariationNvn8 >* pDevice,
    const ResShaderFile* const* ppResShaderFileArray, int shaderFileCount ) NN_NOEXCEPT
{
    // 効率のため ResShaderProgram::NvnGetRecommendedScratchMemorySize() を使わずに自分で計算している

    NN_SDK_REQUIRES_NOT_NULL( pDevice );

    static detail::Ptr< const void > ShaderInfoData::* s_pStageCodes[] =
    {
        &ShaderInfoData::pVertexShaderCode,
        &ShaderInfoData::pHullShaderCode,
        &ShaderInfoData::pDomainShaderCode,
        &ShaderInfoData::pGeometryShaderCode,
        &ShaderInfoData::pPixelShaderCode,
        &ShaderInfoData::pComputeShaderCode
    };

    uint32_t maxSizeRecommended = 0;
    uint32_t maxSizePerWarp = 0;
    for( int idxResShaderFile = 0; idxResShaderFile < shaderFileCount; ++idxResShaderFile )
    {
        NN_SDK_ASSERT_NOT_NULL( ppResShaderFileArray[ idxResShaderFile ] );
        if( const ResShaderContainer* pResShaderContainer =
            ppResShaderFileArray[ idxResShaderFile ]->GetShaderContainer() )
        {
            for( int idxVariation = 0, variationCount = pResShaderContainer->GetShaderVariationCount();
                idxVariation < variationCount; ++idxVariation )
            {
                const ResShaderVariation* pResShaderVariation =
                    pResShaderContainer->GetResShaderVariation( idxVariation );
                NN_SDK_ASSERT_NOT_NULL( pResShaderVariation );
                // 現在はバイナリのみ
                if( const ResShaderProgram* pResShaderProgram =
                    pResShaderVariation->GetResShaderProgram( ShaderCodeType_Binary ) )
                {
                    for( int idxStage = 0; idxStage < ShaderStage_End; ++idxStage )
                    {
                        if( const NvnShaderCode* pNvnShaderData = static_cast<
                            const NvnShaderCode* >( pResShaderProgram->ToData().info.*s_pStageCodes[ idxStage ] ) )
                        {
                            maxSizeRecommended = std::max NN_PREVENT_MACRO_FUNC (
                                maxSizeRecommended, pNvnShaderData->scratchMemoryRecommended );
                            maxSizePerWarp = std::max NN_PREVENT_MACRO_FUNC (
                                maxSizePerWarp, pNvnShaderData->scratchMemoryPerWarp );
                        }
                    }
                }
            }
        }
    }

    uint32_t maxSize = 0;
    if( maxSizeRecommended > 0 )
    {
        maxSize = maxSizeRecommended;
    }
    else if( maxSizePerWarp > 0 )
    {
        int scaleFactor;
        NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger( pDevice->ToData()->pNvnDevice,
            NVN_DEVICE_INFO_SHADER_SCRATCH_MEMORY_SCALE_FACTOR_RECOMMENDED, &scaleFactor ) );
        maxSize = maxSizePerWarp * scaleFactor;
    }
    else
    {
        return 0;
    }

    int granularity;
    NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger( pDevice->ToData()->pNvnDevice,
        NVN_DEVICE_INFO_SHADER_SCRATCH_MEMORY_GRANULARITY, &granularity ) );
    return ( maxSize + granularity - 1 ) / granularity * granularity;
}

}
}
