﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>

#if !defined( NN_BUILD_CONFIG_OS_WIN32 )
    #error
#endif

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Windows.h>

#include <nn/os/os_MutexApi.h>

#include <nn/hws/hws_Window.h>

#include <nn/gfx/gfx_QueueInfo.h>
#include <nn/gfx/gfx_DataAccessorConverter.h>

#include <nn/gfx/detail/gfx_Misc.h>
#include <nn/gfx/detail/gfx_Queue-api.gl.4.h>
#include <nn/gfx/detail/gfx_Texture-api.gl.4.h>
#include <nn/gfx/detail/gfx_Device-api.gl.4-os.win32.h>
#include <nn/gfx/detail/gfx_SwapChain-api.gl.4-os.win32.h>

#include "gfx_CommonHelper.h"
#include "gfx_GlHelper.h"
#include "gfx_WglHelper.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationGl4 Target;

template< bool IsDstBackBuffer, bool IsSrcSwapChain >
void CopyToScanbufferImpl( QueueImpl< Target >* pThis,
    SwapChainImpl< Target >* pSwapChain, const ColorTargetViewImpl< Target >* pSrc ) NN_NOEXCEPT;
void InitializeQueueCommonImpl( QueueImpl< Target >* pThis,
    DeviceImpl< Target >* pDevice, const QueueInfo& info ) NN_NOEXCEPT;
void FinalizeQueueCommonImpl( QueueImpl< Target >* pThis, DeviceImpl< Target >* pDevice ) NN_NOEXCEPT;

void QueueImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_UNUSED( info );

    this->hWnd = nn::hws::FindWindow( "GfxQueueSystemDisplay" ).Get();
    NN_SDK_ASSERT_NOT_NULL( this->hWnd );

    this->hDc = ::GetDC( this->hWnd );
    NN_SDK_ASSERT_NOT_NULL( this->hDc );

    NN_SDK_ASSERT( this->renderingContext.hGlRc == NULL );
    this->renderingContext.hGlRc = Wgl::CreateGlRc( static_cast< HDC >( this->hDc ),
        static_cast< HGLRC >( pDevice->ToData()->renderingContext.hGlRc ),
        GlRequiredVersionMajor, GlRequiredVersionMinor,
        pDevice->ToData()->debugMode != DebugMode_Disable );
    NN_SDK_ASSERT_NOT_NULL( this->renderingContext.hGlRc );

    InitializeQueueCommonImpl( this, pDevice, info );

    this->state = State_Initialized;
}

void QueueImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_Initialized );

    FinalizeQueueCommonImpl( this, pDevice );

    if( this->renderingContext.hGlRc )
    {
        ::wglMakeCurrent( NULL, NULL );
        Wgl::DeleteGlRc( static_cast< HGLRC >( this->renderingContext.hGlRc ) );
        this->renderingContext.hGlRc = NULL;
        NN_GFX_CALL_NNOS_FUNCTION( nn::os::FinalizeMutex( &this->renderingContext.mutex ) );
    }

    if( this->hDc )
    {
        ::ReleaseDC( this->hWnd, static_cast<HDC>( this->hDc ) );
        this->hDc = NULL;
    }

    this->state = State_NotInitialized;
}

void QueueImpl< Target >::Present( SwapChainImpl< Target >* pSwapChain,
    int presentInterval ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pSwapChain );
    NN_SDK_REQUIRES( IsInitialized( *pSwapChain ) );
    NN_SDK_REQUIRES( IsInitialized( *this ) );

    SwapChainImpl< Target >::DataType& swapChain = pSwapChain->ToData();

    if( swapChain.flag.GetBit( SwapChainImpl< Target >::DataType::Flag_Acquired ) )
    {
        CopyToScanbufferImpl< true, true >( this, pSwapChain,
            nn::gfx::DataToAccessor( swapChain.scanBufferViews[ swapChain.currentScanBufferIndex ] ) );
        Flush();
    }

    if( !::IsWindowVisible( static_cast< HWND >( swapChain.hWnd ) ) )
    {
        ::ShowWindow( static_cast< HWND >( swapChain.hWnd ), SW_SHOW );
    }

    GlDeviceActivator activator( pSwapChain, &this->renderingContext );

    GLint tmpFboDraw;
    NN_GFX_CALL_GL_FUNCTION( ::glGetIntegerv( GL_DRAW_FRAMEBUFFER_BINDING, &tmpFboDraw ) );
    NN_GFX_CALL_GL_FUNCTION( ::glBindFramebuffer( GL_DRAW_FRAMEBUFFER, 0 ) );
    if( presentInterval != Wgl::GetSwapInterval() )
    {
        Wgl::SetSwapInterval( presentInterval );
    }
    NN_GFX_CALL_GL_FUNCTION( ::glBindFramebuffer( GL_DRAW_FRAMEBUFFER, tmpFboDraw ) );
    NN_GFX_GL_ASSERT();

    ::SwapBuffers( static_cast< HDC >( swapChain.hDc ) );

    if( ++swapChain.currentScanBufferIndex >= swapChain.scanBufferCount )
    {
        swapChain.currentScanBufferIndex = 0;
    }

    swapChain.flag.SetBit( SwapChainImpl< Target >::DataType::Flag_Acquired, false );
    swapChain.flag.SetBit( SwapChainImpl< Target >::DataType::Flag_Copied, false );
}

}
}
}
