﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>

#if !defined( NN_BUILD_CONFIG_OS_HORIZON )
    #error
#endif

#include <nn/nn_SdkAssert.h>

#include <nn/os/os_MutexApi.h>

#include <nn/gfx/gfx_QueueInfo.h>
#include <nn/gfx/gfx_DataAccessorConverter.h>

#include <nn/gfx/detail/gfx_Misc.h>
#include <nn/gfx/detail/gfx_Queue-api.gl.4.h>
#include <nn/gfx/detail/gfx_Texture-api.gl.4.h>
#include <nn/gfx/detail/gfx_Device-api.gl.4-os.horizon.h>
#include <nn/gfx/detail/gfx_SwapChain-api.gl.4-os.horizon.h>

#include "gfx_CommonHelper.h"
#include "gfx_GlHelper.h"
#include "gfx_EglHelper.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationGl4 Target;

template< bool IsDstBackBuffer, bool IsSrcSwapChain >
void CopyToScanbufferImpl( QueueImpl< Target >* pThis,
    SwapChainImpl< Target >* pSwapChain, const ColorTargetViewImpl< Target >* pSrc ) NN_NOEXCEPT;
void InitializeQueueCommonImpl( QueueImpl< Target >* pThis,
    DeviceImpl< Target >* pDevice, const QueueInfo& info ) NN_NOEXCEPT;
void FinalizeQueueCommonImpl( QueueImpl< Target >* pThis, DeviceImpl< Target >* pDevice ) NN_NOEXCEPT;

void QueueImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_UNUSED( info );

    NN_GFX_CALL_EGL_FUNCTION_WITH_CHECK( this->hDisplay = eglGetDisplay( EGL_DEFAULT_DISPLAY ) );
    NN_SDK_ASSERT_NOT_EQUAL( this->hDisplay, EGL_NO_DISPLAY );

    EGLConfig hConfig = Egl::GetConfig( hDisplay );

    NN_GFX_CALL_EGL_FUNCTION_WITH_CHECK_RESULT( eglBindAPI( EGL_OPENGL_API ) );

    NN_SDK_ASSERT_EQUAL( this->renderingContext.hGlRc, EGL_NO_CONTEXT );
    this->renderingContext.hGlRc = Egl::CreateGlRc( hDisplay, hConfig,
        pDevice->ToData()->renderingContext.hGlRc, GlRequiredVersionMajor, GlRequiredVersionMinor );
    NN_SDK_ASSERT_NOT_EQUAL( this->renderingContext.hGlRc, EGL_NO_CONTEXT );

    const EGLint pbufferAttribs[] =
    {
        EGL_NONE
    };
    NN_GFX_CALL_EGL_FUNCTION_WITH_CHECK( this->hSurface =
        eglCreatePbufferSurface( hDisplay, hConfig, pbufferAttribs ) );
    NN_SDK_ASSERT_NOT_EQUAL( this->hSurface, EGL_NO_SURFACE );

    InitializeQueueCommonImpl( this, pDevice, info );

    this->state = State_Initialized;
}

void QueueImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_Initialized );

    FinalizeQueueCommonImpl( this, pDevice );

    EGLDisplay hDisplay = pDevice->ToData()->hDisplay;
    if( this->hSurface != EGL_NO_SURFACE )
    {
        NN_GFX_CALL_EGL_FUNCTION_WITH_CHECK_RESULT(
            eglDestroySurface( hDisplay, this->hSurface ) );
        this->hSurface = EGL_NO_SURFACE;
    }
    if( this->renderingContext.hGlRc )
    {
        Egl::MakeCurrent( hDisplay, EGL_NO_SURFACE, EGL_NO_SURFACE, EGL_NO_CONTEXT );
        NN_GFX_CALL_EGL_FUNCTION_WITH_CHECK(
            eglDestroyContext( hDisplay, this->renderingContext.hGlRc ) );
        this->renderingContext.hGlRc = EGL_NO_CONTEXT;
        NN_GFX_CALL_NNOS_FUNCTION( nn::os::FinalizeMutex( &this->renderingContext.mutex ) );
    }

    this->hDisplay = EGL_NO_DISPLAY;

    this->state = State_NotInitialized;
}

void QueueImpl< Target >::Present( SwapChainImpl< Target >* pSwapChain,
    int presentInterval ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pSwapChain );
    NN_SDK_REQUIRES( IsInitialized( *pSwapChain ) );
    NN_SDK_REQUIRES( IsInitialized( *this ) );

    SwapChainImpl< Target >::DataType& swapChain = pSwapChain->ToData();

    if( swapChain.flag.GetBit( SwapChainImpl< Target >::DataType::Flag_Acquired ) )
    {
        CopyToScanbufferImpl< true, true >( this, pSwapChain,
            nn::gfx::DataToAccessor( swapChain.scanBufferViews[ swapChain.currentScanBufferIndex ] ) );
        Flush();
    }

    GlDeviceActivator activator( pSwapChain, &this->renderingContext );

    GLint tmpFboDraw;
    NN_GFX_CALL_GL_FUNCTION( ::glGetIntegerv( GL_DRAW_FRAMEBUFFER_BINDING, &tmpFboDraw ) );
    NN_GFX_CALL_GL_FUNCTION( ::glBindFramebuffer( GL_DRAW_FRAMEBUFFER, 0 ) );

    NN_GFX_CALL_EGL_FUNCTION_WITH_CHECK_RESULT( eglSwapInterval(
        swapChain.pGfxDevice->ToData()->hDisplay, presentInterval ) );

    NN_GFX_CALL_GL_FUNCTION( ::glBindFramebuffer( GL_DRAW_FRAMEBUFFER, tmpFboDraw ) );
    NN_GFX_GL_ASSERT();

    NN_GFX_CALL_EGL_FUNCTION_WITH_CHECK_RESULT( eglSwapBuffers(
        swapChain.pGfxDevice->ToData()->hDisplay, swapChain.hSurface ) );

    if( ++swapChain.currentScanBufferIndex >= swapChain.scanBufferCount )
    {
        swapChain.currentScanBufferIndex = 0;
    }

    swapChain.flag.SetBit( SwapChainImpl< Target >::DataType::Flag_Acquired, false );
    swapChain.flag.SetBit( SwapChainImpl< Target >::DataType::Flag_Copied, false );
}

}
}
}
