﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>

#if !defined( NN_BUILD_CONFIG_OS_WIN32 )
    #error
#endif

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Windows.h>

#include <nn/util/util_BytePtr.h>

#include <nn/gfx/detail/gfx_SwapChain-api.nvn.8.h>
#include <nn/gfx/detail/gfx_MemoryPool-api.nvn.8.h>
#include <nn/gfx/detail/gfx_Log.h>

#include "gfx_NvnHelper.h"

namespace nn {
namespace gfx {
namespace detail {

namespace {

const char* GetGlslcDllName() NN_NOEXCEPT
{
#if defined( _M_IX86 )
    return "NvnGlslc32.dll";
#else
    return "NvnGlslc.dll";
#endif
}

template< typename T >
T GetFunction( void* hDll, const char* pFunctionName ) NN_NOEXCEPT
{
    FARPROC pFunction = GetProcAddress( static_cast< HMODULE >( hDll ), pFunctionName );
    NN_SDK_ASSERT_NOT_NULL( pFunction );
    return reinterpret_cast< T >( pFunction );
}

}

nn::TimeSpan Nvn::ToTimeSpan( int64_t timestampValue ) NN_NOEXCEPT
{
    return nn::TimeSpan::FromNanoSeconds( timestampValue );
}

int Nvn::GetFirstScanBufferIndex() NN_NOEXCEPT
{
    return 1;
}

void Nvn::SetPackagedTextureDataImpl( NVNtextureBuilder* pTextureBuilder, const TextureInfo& info,
    MemoryPoolImpl< ApiVariationNvn8 >* pMemoryPool, ptrdiff_t memoryPoolOffset, size_t memoryPoolSize ) NN_NOEXCEPT
{
    if( info.GetTileMode() == TileMode_Optimal && info.GetMultisampleCount() < 2
        && ( info.GetGpuAccessFlags() & GpuAccess_Texture ) )
    {
        NVNmemoryPool* pNvnMemoryPool = pMemoryPool->ToData()->pNvnMemoryPool;
        NN_SDK_ASSERT_NOT_NULL( pNvnMemoryPool );
        int memoryPoolFlags = int();
        NN_GFX_CALL_NVN_FUNCTION( memoryPoolFlags = nvnMemoryPoolGetFlags( pNvnMemoryPool ) );
        void* pPoolMemory = pMemoryPool->ToData()->pMemory;
        if( ( memoryPoolFlags & NVN_MEMORY_POOL_FLAGS_CPU_CACHED_BIT ) &&
            !( memoryPoolFlags & NVN_MEMORY_POOL_FLAGS_VIRTUAL_BIT ) )
        {
            // プールの内容を反映する
            NN_GFX_CALL_NVN_FUNCTION( pPoolMemory = nvnMemoryPoolMap( pNvnMemoryPool ) );
            NN_GFX_CALL_NVN_FUNCTION( nvnMemoryPoolInvalidateMappedRange(
                pNvnMemoryPool, memoryPoolOffset, memoryPoolSize ) );
        }
        if( pPoolMemory )
        {
            NN_GFX_CALL_NVN_FUNCTION( nvnTextureBuilderSetPackagedTextureData(
                pTextureBuilder, nn::util::BytePtr( pPoolMemory, memoryPoolOffset ).Get() ) );
        }
    }
}

void Nvn::SetTextureViewFormat( NVNtextureView* pTextureView, NVNformat value,
    const NVNtexture* pTexture ) NN_NOEXCEPT
{
    NVNformat originalTextureFormat = nvnTextureGetFormat( pTexture );
    if( originalTextureFormat < NVN_FORMAT_RGBA_ASTC_4x4 ||
        originalTextureFormat > NVN_FORMAT_RGBA_ASTC_12x12_SRGB )
    {
        NN_GFX_CALL_NVN_FUNCTION( nvnTextureViewSetFormat( pTextureView, value ) );
    }
}

bool GlslcDll::Initialize() NN_NOEXCEPT
{
    if( ( m_hDll = LoadLibraryA( GetGlslcDllName() ) ) == NULL )
    {
        NN_DETAIL_GFX_ERROR( "%s was not found\n", GetGlslcDllName() );
        return false;
    }

    GlslcCompilePreSpecialized = GetFunction<
        GlslcCompilePreSpecializedType >( m_hDll, "glslcCompilePreSpecialized" );
    GlslcCompileSpecialized = GetFunction< GlslcCompileSpecializedType >( m_hDll, "glslcCompileSpecialized" );
    GlslcInitialize = GetFunction< GlslcInitializeType >( m_hDll, "glslcInitialize" );
    GlslcFinalize = GetFunction< GlslcFinalizeType >( m_hDll, "glslcFinalize" );
    GlslcCompile = GetFunction< GlslcCompileType >( m_hDll, "glslcCompile" );
    GlslcGetVersion = GetFunction< GlslcGetVersionType >( m_hDll, "glslcGetVersion" );

    return true;
}

void GlslcDll::Finalize() NN_NOEXCEPT
{
    if( m_hDll != NULL )
    {
        FreeLibrary( static_cast< HMODULE >( m_hDll ) );
        m_hDll = NULL;
    }
}

bool GlslcDll::IsInitialized() const NN_NOEXCEPT
{
    return m_hDll != NULL;
}

bool IsThinBinaryAvailable() NN_NOEXCEPT
{
    return false;
}

}
}
}
