﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <memory>

#include <nn/nn_SdkLog.h>
#include <nn/util/util_BytePtr.h>

#include <nn/gfx/gfx_Common.h>

#include <nn/gfx/detail/gfx_Log.h>
#include <nn/gfx/detail/gfx_Device-api.gl.4.h>

#include "gfx_GlHelper.h"
#include "gfx_CommonHelper.h"

namespace nn {
namespace gfx {
namespace detail {

namespace {

const struct ImageFormatAndGlFormat
{
    ImageFormat imageFormat;
    GlTextureFormat glFormat;
} GlTextureFormatList[] =
{
    { ImageFormat_R8_Unorm, { GL_R8, GL_RED, GL_UNSIGNED_BYTE } },
    { ImageFormat_R8_Snorm, { GL_R8_SNORM, GL_RED, GL_BYTE } },
    { ImageFormat_R8_Uint, { GL_R8UI, GL_RED_INTEGER, GL_UNSIGNED_BYTE } },
    { ImageFormat_R8_Sint, { GL_R8I, GL_RED_INTEGER, GL_BYTE } },
    { ImageFormat_R4_G4_B4_A4_Unorm, { GL_RGBA4, GL_RGBA, GL_UNSIGNED_SHORT_4_4_4_4_REV } },
    { ImageFormat_A4_B4_G4_R4_Unorm, { GL_RGBA4, GL_RGBA, GL_UNSIGNED_SHORT_4_4_4_4 } },
    { ImageFormat_R5_G5_B5_A1_Unorm, { GL_RGB5_A1, GL_RGBA, GL_UNSIGNED_SHORT_1_5_5_5_REV } },
    { ImageFormat_A1_B5_G5_R5_Unorm, { GL_RGB5_A1, GL_RGBA, GL_UNSIGNED_SHORT_5_5_5_1 } },
    { ImageFormat_R5_G6_B5_Unorm, { GL_RGB8, GL_RGB, GL_UNSIGNED_SHORT_5_6_5_REV } },
    { ImageFormat_B5_G6_R5_Unorm, { GL_RGB8, GL_RGB, GL_UNSIGNED_SHORT_5_6_5 } },
    { ImageFormat_R8_G8_Unorm, { GL_RG8, GL_RG, GL_UNSIGNED_BYTE } },
    { ImageFormat_R8_G8_Snorm, { GL_RG8_SNORM, GL_RG, GL_BYTE } },
    { ImageFormat_R8_G8_Uint, { GL_RG8UI, GL_RG_INTEGER, GL_UNSIGNED_BYTE } },
    { ImageFormat_R8_G8_Sint, { GL_RG8I, GL_RG_INTEGER, GL_BYTE } },
    { ImageFormat_R16_Unorm, { GL_R16, GL_RED, GL_UNSIGNED_SHORT } },
    { ImageFormat_R16_Snorm, { GL_R16_SNORM, GL_RED, GL_SHORT } },
    { ImageFormat_R16_Uint, { GL_R16UI, GL_RED_INTEGER, GL_UNSIGNED_SHORT } },
    { ImageFormat_R16_Sint, { GL_R16I, GL_RED_INTEGER, GL_SHORT } },
    { ImageFormat_R16_Float, { GL_R16F, GL_RED, GL_HALF_FLOAT } },
    { ImageFormat_D16_Unorm, { GL_DEPTH_COMPONENT16, GL_DEPTH_COMPONENT, GL_UNSIGNED_SHORT } },
    { ImageFormat_R8_G8_B8_A8_Unorm, { GL_RGBA8, GL_RGBA, GL_UNSIGNED_BYTE } },
    { ImageFormat_R8_G8_B8_A8_Snorm, { GL_RGBA8_SNORM, GL_RGBA, GL_BYTE } },
    { ImageFormat_R8_G8_B8_A8_Uint, { GL_RGBA8UI, GL_RGBA_INTEGER, GL_UNSIGNED_BYTE } },
    { ImageFormat_R8_G8_B8_A8_Sint, { GL_RGBA8I, GL_RGBA_INTEGER, GL_BYTE } },
    { ImageFormat_R8_G8_B8_A8_UnormSrgb, { GL_SRGB8_ALPHA8, GL_RGBA, GL_UNSIGNED_BYTE } },
    { ImageFormat_B8_G8_R8_A8_Unorm, { GL_RGBA8, GL_BGRA, GL_UNSIGNED_BYTE } },
    { ImageFormat_B8_G8_R8_A8_Snorm, { GL_RGBA8_SNORM, GL_BGRA, GL_BYTE } },
    { ImageFormat_B8_G8_R8_A8_Uint, { GL_RGBA8UI, GL_BGRA_INTEGER, GL_UNSIGNED_BYTE } },
    { ImageFormat_B8_G8_R8_A8_Sint, { GL_RGBA8I, GL_BGRA_INTEGER, GL_BYTE } },
    { ImageFormat_B8_G8_R8_A8_UnormSrgb, { GL_SRGB8_ALPHA8, GL_BGRA, GL_UNSIGNED_BYTE } },
    { ImageFormat_R9_G9_B9_E5_SharedExp, { GL_RGB9_E5, GL_RGB, GL_UNSIGNED_INT_5_9_9_9_REV } },
    { ImageFormat_R10_G10_B10_A2_Unorm, { GL_RGB10_A2, GL_RGBA, GL_UNSIGNED_INT_2_10_10_10_REV } },
    { ImageFormat_R10_G10_B10_A2_Uint, { GL_RGB10_A2UI, GL_RGBA_INTEGER, GL_UNSIGNED_INT_2_10_10_10_REV } },
    { ImageFormat_R11_G11_B10_Float, { GL_R11F_G11F_B10F, GL_RGB, GL_UNSIGNED_INT_10F_11F_11F_REV } },
    //ImageFormat_B10_G11_R11_Float
    { ImageFormat_R16_G16_Unorm, { GL_RG16, GL_RG, GL_UNSIGNED_SHORT } },
    { ImageFormat_R16_G16_Snorm, { GL_RG16_SNORM, GL_RG, GL_SHORT } },
    { ImageFormat_R16_G16_Uint, { GL_RG16UI, GL_RG_INTEGER, GL_UNSIGNED_SHORT } },
    { ImageFormat_R16_G16_Sint, { GL_RG16I, GL_RG_INTEGER, GL_SHORT } },
    { ImageFormat_R16_G16_Float, { GL_RG16F, GL_RG, GL_HALF_FLOAT } },
    { ImageFormat_D24_Unorm_S8_Uint, { GL_DEPTH24_STENCIL8, GL_DEPTH_STENCIL, GL_UNSIGNED_INT_24_8 } },
    { ImageFormat_R32_Uint, { GL_R32UI, GL_RED_INTEGER, GL_UNSIGNED_INT } },
    { ImageFormat_R32_Sint, { GL_R32I, GL_RED_INTEGER, GL_INT } },
    { ImageFormat_R32_Float, { GL_R32F, GL_RED, GL_FLOAT } },
    { ImageFormat_D32_Float, { GL_DEPTH_COMPONENT32F, GL_DEPTH_COMPONENT, GL_FLOAT } },
    { ImageFormat_R16_G16_B16_A16_Unorm, { GL_RGBA16, GL_RGBA, GL_UNSIGNED_SHORT } },
    { ImageFormat_R16_G16_B16_A16_Snorm, { GL_RGBA16_SNORM, GL_RGBA, GL_SHORT } },
    { ImageFormat_R16_G16_B16_A16_Uint, { GL_RGBA16UI, GL_RGBA_INTEGER, GL_UNSIGNED_SHORT } },
    { ImageFormat_R16_G16_B16_A16_Sint, { GL_RGBA16I, GL_RGBA_INTEGER, GL_SHORT } },
    { ImageFormat_R16_G16_B16_A16_Float, { GL_RGBA32F, GL_RGBA, GL_HALF_FLOAT } },
    { ImageFormat_D32_Float_S8_Uint_X24, { GL_DEPTH32F_STENCIL8, GL_DEPTH_STENCIL, GL_FLOAT_32_UNSIGNED_INT_24_8_REV } },
    { ImageFormat_R32_G32_Uint, { GL_RG32UI, GL_RG_INTEGER, GL_UNSIGNED_INT } },
    { ImageFormat_R32_G32_Sint, { GL_RG32I, GL_RG_INTEGER, GL_INT } },
    { ImageFormat_R32_G32_Float, { GL_RG32F, GL_RG, GL_FLOAT } },
    { ImageFormat_R32_G32_B32_Uint, { GL_RGB32UI, GL_RGB_INTEGER, GL_UNSIGNED_INT } },
    { ImageFormat_R32_G32_B32_Sint, { GL_RGB32I, GL_RGB_INTEGER, GL_INT } },
    { ImageFormat_R32_G32_B32_Float, { GL_RGB32F, GL_RGB, GL_FLOAT } },
    { ImageFormat_R32_G32_B32_A32_Uint, { GL_RGBA32UI, GL_RGBA_INTEGER, GL_UNSIGNED_INT } },
    { ImageFormat_R32_G32_B32_A32_Sint, { GL_RGBA32I, GL_RGBA_INTEGER, GL_INT } },
    { ImageFormat_R32_G32_B32_A32_Float, { GL_RGBA32F, GL_RGBA, GL_FLOAT } },
    { ImageFormat_Bc1_Unorm, { GL_COMPRESSED_RGBA_S3TC_DXT1_EXT, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Bc1_UnormSrgb, { GL_COMPRESSED_SRGB_ALPHA_S3TC_DXT1_EXT, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Bc2_Unorm, { GL_COMPRESSED_RGBA_S3TC_DXT3_EXT, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Bc2_UnormSrgb, { GL_COMPRESSED_SRGB_ALPHA_S3TC_DXT3_EXT, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Bc3_Unorm, { GL_COMPRESSED_RGBA_S3TC_DXT5_EXT, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Bc3_UnormSrgb, { GL_COMPRESSED_SRGB_ALPHA_S3TC_DXT5_EXT, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Bc4_Unorm, { GL_COMPRESSED_RED_RGTC1, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Bc4_Snorm, { GL_COMPRESSED_SIGNED_RED_RGTC1, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Bc5_Unorm, { GL_COMPRESSED_RG_RGTC2, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Bc5_Snorm, { GL_COMPRESSED_SIGNED_RG_RGTC2, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Bc6_Float, { GL_COMPRESSED_RGB_BPTC_SIGNED_FLOAT, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Bc6_Ufloat, { GL_COMPRESSED_RGB_BPTC_UNSIGNED_FLOAT, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Bc7_Unorm, { GL_COMPRESSED_RGBA_BPTC_UNORM, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Bc7_UnormSrgb, { GL_COMPRESSED_SRGB_ALPHA_BPTC_UNORM, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Eac_R11_Unorm, { GL_COMPRESSED_R11_EAC, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Eac_R11_Snorm, { GL_COMPRESSED_SIGNED_R11_EAC, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Eac_R11_G11_Unorm, { GL_COMPRESSED_RG11_EAC, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Eac_R11_G11_Snorm, { GL_COMPRESSED_SIGNED_RG11_EAC, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Etc1_Unorm, { GL_COMPRESSED_RGB8_ETC2, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Etc2_Unorm, { GL_COMPRESSED_RGB8_ETC2, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Etc2_UnormSrgb, { GL_COMPRESSED_SRGB8_ETC2, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Etc2_Mask_Unorm, { GL_COMPRESSED_RGB8_PUNCHTHROUGH_ALPHA1_ETC2, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Etc2_Mask_UnormSrgb, { GL_COMPRESSED_SRGB8_PUNCHTHROUGH_ALPHA1_ETC2, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Etc2_Alpha_Unorm, { GL_COMPRESSED_RGBA8_ETC2_EAC, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Etc2_Alpha_UnormSrgb, { GL_COMPRESSED_SRGB8_ALPHA8_ETC2_EAC, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    //{ ImageFormat_Pvrtc1_2bpp_Unorm, { GL_COMPRESSED_RGB_PVRTC_2BPPV1_IMG, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    //{ ImageFormat_Pvrtc1_2bpp_UnormSrgb, { GL_COMPRESSED_SRGB_PVRTC_2BPPV1_EXT, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    //{ ImageFormat_Pvrtc1_4bpp_Unorm, { GL_COMPRESSED_RGB_PVRTC_4BPPV1_IMG, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    //{ ImageFormat_Pvrtc1_4bpp_UnormSrgb, { GL_COMPRESSED_SRGB_PVRTC_4BPPV1_EXT, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    //{ ImageFormat_Pvrtc1_Alpha_2bpp_Unorm, { GL_COMPRESSED_RGBA_PVRTC_2BPPV1_IMG, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    //{ ImageFormat_Pvrtc1_Alpha_2bpp_UnormSrgb, { GL_COMPRESSED_SRGB_ALPHA_PVRTC_2BPPV1_EXT, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    //{ ImageFormat_Pvrtc1_Alpha_4bpp_Unorm, { GL_COMPRESSED_RGBA_PVRTC_4BPPV1_IMG, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    //{ ImageFormat_Pvrtc1_Alpha_4bpp_UnormSrgb, { GL_COMPRESSED_SRGB_ALPHA_PVRTC_4BPPV1_EXT, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    //{ ImageFormat_Pvrtc2_Alpha_2bpp_Unorm, { GL_COMPRESSED_RGBA_PVRTC_2BPPV2_IMG, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    //ImageFormat_Pvrtc2_Alpha_2bpp_UnormSrgb
    //{ ImageFormat_Pvrtc2_Alpha_4bpp_Unorm, { GL_COMPRESSED_RGBA_PVRTC_4BPPV2_IMG, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    //ImageFormat_Pvrtc2_Alpha_4bpp_UnormSrgb
    { ImageFormat_Astc_4x4_Unorm, { GL_COMPRESSED_RGBA_ASTC_4x4_KHR, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Astc_4x4_UnormSrgb, { GL_COMPRESSED_SRGB8_ALPHA8_ASTC_4x4_KHR, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Astc_5x4_Unorm, { GL_COMPRESSED_RGBA_ASTC_5x4_KHR, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Astc_5x4_UnormSrgb, { GL_COMPRESSED_SRGB8_ALPHA8_ASTC_5x4_KHR, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Astc_5x5_Unorm, { GL_COMPRESSED_RGBA_ASTC_5x5_KHR, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Astc_5x5_UnormSrgb, { GL_COMPRESSED_SRGB8_ALPHA8_ASTC_5x5_KHR, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Astc_6x5_Unorm, { GL_COMPRESSED_RGBA_ASTC_6x5_KHR, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Astc_6x5_UnormSrgb, { GL_COMPRESSED_SRGB8_ALPHA8_ASTC_6x5_KHR, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Astc_6x6_Unorm, { GL_COMPRESSED_RGBA_ASTC_6x6_KHR, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Astc_6x6_UnormSrgb, { GL_COMPRESSED_SRGB8_ALPHA8_ASTC_6x6_KHR, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Astc_8x5_Unorm, { GL_COMPRESSED_RGBA_ASTC_8x5_KHR, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Astc_8x5_UnormSrgb, { GL_COMPRESSED_SRGB8_ALPHA8_ASTC_8x5_KHR, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Astc_8x6_Unorm, { GL_COMPRESSED_RGBA_ASTC_8x6_KHR, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Astc_8x6_UnormSrgb, { GL_COMPRESSED_SRGB8_ALPHA8_ASTC_8x6_KHR, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Astc_8x8_Unorm, { GL_COMPRESSED_RGBA_ASTC_8x8_KHR, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Astc_8x8_UnormSrgb, { GL_COMPRESSED_SRGB8_ALPHA8_ASTC_8x8_KHR, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Astc_10x5_Unorm, { GL_COMPRESSED_RGBA_ASTC_10x5_KHR, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Astc_10x5_UnormSrgb, { GL_COMPRESSED_SRGB8_ALPHA8_ASTC_10x5_KHR, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Astc_10x6_Unorm, { GL_COMPRESSED_RGBA_ASTC_10x6_KHR, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Astc_10x6_UnormSrgb, { GL_COMPRESSED_SRGB8_ALPHA8_ASTC_10x6_KHR, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Astc_10x8_Unorm, { GL_COMPRESSED_RGBA_ASTC_10x8_KHR, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Astc_10x8_UnormSrgb, { GL_COMPRESSED_SRGB8_ALPHA8_ASTC_10x8_KHR, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Astc_10x10_Unorm, { GL_COMPRESSED_RGBA_ASTC_10x10_KHR, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Astc_10x10_UnormSrgb, { GL_COMPRESSED_SRGB8_ALPHA8_ASTC_10x10_KHR, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Astc_12x10_Unorm, { GL_COMPRESSED_RGBA_ASTC_12x10_KHR, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Astc_12x10_UnormSrgb, { GL_COMPRESSED_SRGB8_ALPHA8_ASTC_12x10_KHR, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Astc_12x12_Unorm, { GL_COMPRESSED_RGBA_ASTC_12x12_KHR, GL_INVALID_ENUM, GL_INVALID_ENUM } },
    { ImageFormat_Astc_12x12_UnormSrgb, { GL_COMPRESSED_SRGB8_ALPHA8_ASTC_12x12_KHR, GL_INVALID_ENUM, GL_INVALID_ENUM } }

    //ImageFormat_B5_G5_R5_A1_Unorm
};

template< typename T >
struct FreeDeleter
{
    void operator()( T* ptr )
    {
        if( ptr )
        {
            free( ptr );
        }
    }
};

}

#define NN_GFX_CASE_STRING(value) case value: return #value

const char* Gl::ErrorToString( GLenum error ) NN_NOEXCEPT
{
    switch( error )
    {
        NN_GFX_CASE_STRING( GL_NO_ERROR );
        NN_GFX_CASE_STRING( GL_INVALID_ENUM );
        NN_GFX_CASE_STRING( GL_INVALID_VALUE );
        NN_GFX_CASE_STRING( GL_INVALID_OPERATION );
        NN_GFX_CASE_STRING( GL_STACK_OVERFLOW );
        NN_GFX_CASE_STRING( GL_STACK_UNDERFLOW );
        NN_GFX_CASE_STRING( GL_OUT_OF_MEMORY );
        NN_GFX_CASE_STRING( GL_INVALID_FRAMEBUFFER_OPERATION );
    default:
        return "Unkown Error";
    }
}

const char* Gl::FramebufferStatusToString( GLenum status ) NN_NOEXCEPT
{
    switch( status )
    {
        NN_GFX_CASE_STRING( GL_FRAMEBUFFER_COMPLETE );
        NN_GFX_CASE_STRING( GL_FRAMEBUFFER_UNDEFINED );
        NN_GFX_CASE_STRING( GL_FRAMEBUFFER_INCOMPLETE_ATTACHMENT );
        NN_GFX_CASE_STRING( GL_FRAMEBUFFER_INCOMPLETE_MISSING_ATTACHMENT );
        NN_GFX_CASE_STRING( GL_FRAMEBUFFER_INCOMPLETE_DRAW_BUFFER );
        NN_GFX_CASE_STRING( GL_FRAMEBUFFER_INCOMPLETE_READ_BUFFER );
        NN_GFX_CASE_STRING( GL_FRAMEBUFFER_UNSUPPORTED );
        NN_GFX_CASE_STRING( GL_FRAMEBUFFER_INCOMPLETE_MULTISAMPLE );
        NN_GFX_CASE_STRING( GL_FRAMEBUFFER_INCOMPLETE_LAYER_TARGETS );
    default:
        return "Unkonwn Status";
    }
}

bool Gl::CheckFramebufferStatus( GlHandle hFramebuffer ) NN_NOEXCEPT
{
    GLenum status = NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA(
        ::glCheckNamedFramebufferStatus )( hFramebuffer, GL_FRAMEBUFFER ) );
    if( status != GL_FRAMEBUFFER_COMPLETE )
    {
        NN_DETAIL_GFX_ERROR( FramebufferStatusToString( status ) );
        return false;
    }

    return true;
}

bool Gl::CheckShaderStatus( GlHandle hShader ) NN_NOEXCEPT
{
    GLint status = 0;
    NN_GFX_CALL_GL_FUNCTION( ::glGetShaderiv( hShader, GL_COMPILE_STATUS, &status ) );
    if( !status )
    {
        GLint infoLength = 0;
        NN_GFX_CALL_GL_FUNCTION( ::glGetShaderiv( hShader, GL_INFO_LOG_LENGTH, &infoLength ) );
        if( infoLength > 1 )
        {
            // 動的なログ出力のため一時的にメモリーを確保
            std::unique_ptr< GLchar[], FreeDeleter< GLchar > > info(
                static_cast< GLchar* >( malloc( infoLength ) ) );
            if( info.get() )
            {
                NN_GFX_CALL_GL_FUNCTION( ::glGetShaderInfoLog(
                    hShader, infoLength, &infoLength, info.get() ) );
                NN_DETAIL_GFX_ERROR( info.get() );
            }
        }
        return false;
    }

    return true;
}

bool Gl::CheckProgramStatus( GlHandle hProgram ) NN_NOEXCEPT
{
    GLint status = 0;
    NN_GFX_CALL_GL_FUNCTION( ::glGetProgramiv( hProgram, GL_LINK_STATUS, &status ) );
    NN_GFX_GL_ASSERT();
    if( !status )
    {
        GLint infoLength = 0;
        NN_GFX_CALL_GL_FUNCTION( ::glGetProgramiv( hProgram, GL_INFO_LOG_LENGTH, &infoLength ) );
        if( infoLength > 1 )
        {
            // 動的なログ出力のため一時的にメモリーを確保
            std::unique_ptr< GLchar[], FreeDeleter< GLchar > > info(
                static_cast< GLchar* >( malloc( infoLength ) ) );
            if( info.get() )
            {
                NN_GFX_CALL_GL_FUNCTION( ::glGetProgramInfoLog(
                    hProgram, infoLength, &infoLength, info.get() ) );
                NN_DETAIL_GFX_ERROR( info.get() );
            }
        }
        return false;
    }

    return true;
}

GLenum Gl::GetTextureTarget( ImageDimension dimension ) NN_NOEXCEPT
{
    static const GLenum s_TargetTable[] =
    {
        GL_TEXTURE_1D,
        GL_TEXTURE_2D,
        GL_TEXTURE_3D,
        GL_TEXTURE_CUBE_MAP,
        GL_TEXTURE_1D_ARRAY,
        GL_TEXTURE_2D_ARRAY,
        GL_TEXTURE_2D_MULTISAMPLE,
        GL_TEXTURE_2D_MULTISAMPLE_ARRAY,
        GL_TEXTURE_CUBE_MAP_ARRAY,
        GL_TEXTURE_BUFFER
    };

    return s_TargetTable[ dimension ];
}

bool Gl::IsLayeredTarget( GLenum target ) NN_NOEXCEPT
{
    switch( target )
    {
    case GL_TEXTURE_3D:
    case GL_TEXTURE_1D_ARRAY:
    case GL_TEXTURE_2D_ARRAY:
    case GL_TEXTURE_2D_MULTISAMPLE_ARRAY:
    case GL_TEXTURE_CUBE_MAP_ARRAY:
        return true;
    default:
        return false;
    }
}

GlTextureFormat Gl::GetTextureFormat( ImageFormat format ) NN_NOEXCEPT
{
    struct Comp
    {
        bool operator()( const ImageFormatAndGlFormat& lhs, ImageFormat rhs ) const
        {
            return lhs.imageFormat < rhs;
        }
    };
    const ImageFormatAndGlFormat* pEnd = GlTextureFormatList + NN_GFX_ARRAY_LENGTH( GlTextureFormatList );
    const ImageFormatAndGlFormat* pFound = std::lower_bound( GlTextureFormatList, pEnd, format, Comp() );
    if( pFound == pEnd || format < pFound->imageFormat )
    {
        GlTextureFormat ret = { GL_INVALID_ENUM, GL_INVALID_ENUM, GL_INVALID_ENUM };
        return ret;
    }
    else
    {
        return pFound->glFormat;
    }
}

size_t Gl::TextureSubImage( GlHandle texture, GLenum target, GLint mipLevel, bool isCompressed,
    GLenum imageFormat, GLenum pixelFormat, GLenum pixelType, GLint xOffset, GLint yOffset,
    GLint zOffset, GLsizei width, GLsizei height, GLsizei depth, GLsizei imageSize, const void* pMemory ) NN_NOEXCEPT
{
    NN_GFX_CALL_GL_FUNCTION( ::glPixelStorei( GL_UNPACK_ALIGNMENT, 1 ) );
    if( isCompressed )
    {
        switch( target )
        {
        case GL_TEXTURE_1D:
            {
                NN_GFX_CALL_GL_FUNCTION( ::glCompressedTextureSubImage1D( texture,
                    mipLevel, xOffset, width, imageFormat, imageSize, pMemory ) );

            }
            break;
        case GL_TEXTURE_2D: NN_FALL_THROUGH;
        case GL_TEXTURE_1D_ARRAY:
            {
                NN_GFX_CALL_GL_FUNCTION( ::glCompressedTextureSubImage2D( texture,
                    mipLevel, xOffset, yOffset, width, height, imageFormat, imageSize, pMemory ) );
            }
            break;
        case GL_TEXTURE_CUBE_MAP:
        case GL_TEXTURE_CUBE_MAP_ARRAY:
        case GL_TEXTURE_3D:
        case GL_TEXTURE_2D_ARRAY:
            {
                NN_GFX_CALL_GL_FUNCTION( ::glCompressedTextureSubImage3D( texture, mipLevel,
                    xOffset, yOffset, zOffset, width, height, depth, imageFormat, imageSize, pMemory ) );
            }
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
    }
    else
    {
        switch( target )
        {
        case GL_TEXTURE_1D:
            {
                NN_GFX_CALL_GL_FUNCTION( ::glTextureSubImage1D( texture,
                    mipLevel, xOffset, width, pixelFormat, pixelType, pMemory ) );
            }
            break;
        case GL_TEXTURE_2D: NN_FALL_THROUGH;
        case GL_TEXTURE_1D_ARRAY:
            {
                NN_GFX_CALL_GL_FUNCTION( ::glTextureSubImage2D( texture,
                    mipLevel, xOffset, yOffset, width, height, pixelFormat, pixelType, pMemory ) );
            }
            break;
        case GL_TEXTURE_3D:
        case GL_TEXTURE_2D_ARRAY:
        case GL_TEXTURE_CUBE_MAP_ARRAY:
        case GL_TEXTURE_CUBE_MAP:
            {
                NN_GFX_CALL_GL_FUNCTION( ::glTextureSubImage3D( texture, mipLevel,
                    xOffset, yOffset, zOffset, width, height, depth, pixelFormat, pixelType, pMemory ) );
            }
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
    }
    NN_GFX_GL_ASSERT();

    return imageSize;
}

size_t Gl::CopyCpuMemoryToTexture( GlHandle texture, GLenum target, GLint mipLevel,
        ImageFormat imageFormat, GLsizei width, GLsizei height, GLsizei depth, const void* pMemory ) NN_NOEXCEPT
{
    GlTextureFormat format = GetTextureFormat( imageFormat );
    ChannelFormat channelFormat = GetChannelFormat( imageFormat );
    GLsizei imageSize = static_cast< GLsizei >( CalculateImageSize( channelFormat, width, height, depth ) );
    bool isCompressed = IsCompressedFormat( channelFormat );

    NN_GFX_CALL_GL_FUNCTION( ::glBindBuffer( GL_PIXEL_UNPACK_BUFFER, GlInvalidHandle ) );
    NN_GFX_GL_ASSERT();

    return TextureSubImage( texture, target, mipLevel, isCompressed, format.imageFormat,
        format.pixelFormat, format.pixelType, 0, 0, 0, width, height, depth, imageSize, pMemory );
}

void Gl::FramebufferTexture( GlHandle hFbo, GLenum attachment,
    GlHandle hTexture, GLint mipLevel, GLint layer ) NN_NOEXCEPT
{
    if( layer >= 0 )
    {
        NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glNamedFramebufferTextureLayer )(
            hFbo, attachment, hTexture, mipLevel, layer ) );
    }
    else
    {
        NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glNamedFramebufferTexture )(
            hFbo, attachment, hTexture, mipLevel ) );
    }
}

GlAttributeFormat Gl::GetAttributeFormat( AttributeFormat format ) NN_NOEXCEPT
{
    GlAttributeFormat ret = {};

    ChannelFormat channel = static_cast< ChannelFormat >( format >> TypeFormat_Bits );
    TypeFormat type = static_cast< TypeFormat >( format & ( ( 0x01 << TypeFormat_Bits ) - 1 ) );

    ret.size = GetChannelCount( channel );

    ret.normalized = type == TypeFormat_Unorm || type == TypeFormat_Snorm ? GL_TRUE : GL_FALSE;

    int bytePerChannel = GetBytePerPixel( channel ) / ret.size;
    NN_SDK_ASSERT_RANGE( bytePerChannel, 0, 5 );

    switch( type )
    {
    case TypeFormat_Unorm: NN_FALL_THROUGH;
    case TypeFormat_Uint: NN_FALL_THROUGH;
    case TypeFormat_UintToFloat:
        {
            switch ( format )
            {
            case AttributeFormat_10_10_10_2_Unorm:
            case AttributeFormat_10_10_10_2_Uint:
                {
                    // AttributeFormat_10_10_10_2 implies that R10 is the least-significant bit
                    ret.type = GL_UNSIGNED_INT_2_10_10_10_REV;
                }
                break;

            default:
                {
                    ret.type = bytePerChannel <= 1 ? GL_UNSIGNED_BYTE :
                        bytePerChannel <= 2 ? GL_UNSIGNED_SHORT : GL_UNSIGNED_INT;
                }
                break;
            }
        }
        break;
    case TypeFormat_Snorm: NN_FALL_THROUGH;
    case TypeFormat_Sint: NN_FALL_THROUGH;
    case TypeFormat_SintToFloat:
        {
            switch ( format )
            {
            case AttributeFormat_10_10_10_2_Snorm:
            case AttributeFormat_10_10_10_2_Sint:
                {
                    // AttributeFormat_10_10_10_2 implies that R10 is the least-significant bit
                    ret.type = GL_INT_2_10_10_10_REV;
                }
                break;

            default:
                {
                    ret.type = bytePerChannel <= 1 ? GL_BYTE :
                        bytePerChannel <= 2 ? GL_SHORT : GL_INT;
                }
                break;
            }
        }
        break;
    case TypeFormat_Float:
        {
            ret.type = bytePerChannel <= 2 ? GL_HALF_FLOAT : GL_FLOAT;
        }
        break;
    default:
        {
            ret.type = GL_INVALID_ENUM;
        }
        break;
    }

    switch( format )
    {
    case AttributeFormat_10_10_10_2_Unorm: NN_FALL_THROUGH;
    case AttributeFormat_10_10_10_2_Uint:
        {
            ret.type = GL_UNSIGNED_INT_2_10_10_10_REV;
        }
        break;
    case AttributeFormat_10_10_10_2_Snorm: NN_FALL_THROUGH;
    case AttributeFormat_10_10_10_2_Sint:
        {
            ret.type = GL_INT_2_10_10_10_REV;
        }
        break;
    default:
        break;
    }

    return ret;
}

bool Gl::IsSamplerType( GLenum type ) NN_NOEXCEPT
{
    switch( type )
    {
    case GL_SAMPLER_1D:
    case GL_SAMPLER_2D:
    case GL_SAMPLER_3D:
    case GL_SAMPLER_CUBE:
    case GL_SAMPLER_1D_SHADOW:
    case GL_SAMPLER_2D_SHADOW:
    case GL_SAMPLER_1D_ARRAY:
    case GL_SAMPLER_2D_ARRAY:
    case GL_SAMPLER_CUBE_MAP_ARRAY:
    case GL_SAMPLER_1D_ARRAY_SHADOW:
    case GL_SAMPLER_2D_ARRAY_SHADOW:
    case GL_SAMPLER_2D_MULTISAMPLE:
    case GL_SAMPLER_2D_MULTISAMPLE_ARRAY:
    case GL_SAMPLER_CUBE_SHADOW:
    case GL_SAMPLER_CUBE_MAP_ARRAY_SHADOW:
    case GL_SAMPLER_BUFFER:
    case GL_SAMPLER_2D_RECT:
    case GL_SAMPLER_2D_RECT_SHADOW:
    case GL_INT_SAMPLER_1D:
    case GL_INT_SAMPLER_2D:
    case GL_INT_SAMPLER_3D:
    case GL_INT_SAMPLER_CUBE:
    case GL_INT_SAMPLER_1D_ARRAY:
    case GL_INT_SAMPLER_2D_ARRAY:
    case GL_INT_SAMPLER_CUBE_MAP_ARRAY:
    case GL_INT_SAMPLER_2D_MULTISAMPLE:
    case GL_INT_SAMPLER_2D_MULTISAMPLE_ARRAY:
    case GL_INT_SAMPLER_BUFFER:
    case GL_INT_SAMPLER_2D_RECT:
    case GL_UNSIGNED_INT_SAMPLER_1D:
    case GL_UNSIGNED_INT_SAMPLER_2D:
    case GL_UNSIGNED_INT_SAMPLER_3D:
    case GL_UNSIGNED_INT_SAMPLER_CUBE:
    case GL_UNSIGNED_INT_SAMPLER_1D_ARRAY:
    case GL_UNSIGNED_INT_SAMPLER_2D_ARRAY:
    case GL_UNSIGNED_INT_SAMPLER_CUBE_MAP_ARRAY:
    case GL_UNSIGNED_INT_SAMPLER_2D_MULTISAMPLE:
    case GL_UNSIGNED_INT_SAMPLER_2D_MULTISAMPLE_ARRAY:
    case GL_UNSIGNED_INT_SAMPLER_BUFFER:
    case GL_UNSIGNED_INT_SAMPLER_2D_RECT:
        return true;
    default:
        return false;
    }
}

bool Gl::IsImageType( GLenum type ) NN_NOEXCEPT
{
    switch( type )
    {
    case GL_IMAGE_1D:
    case GL_IMAGE_2D:
    case GL_IMAGE_3D:
    case GL_IMAGE_CUBE:
    case GL_IMAGE_1D_ARRAY:
    case GL_IMAGE_2D_ARRAY:
    case GL_IMAGE_2D_MULTISAMPLE:
    case GL_IMAGE_2D_MULTISAMPLE_ARRAY:
    case GL_IMAGE_CUBE_MAP_ARRAY:
    case GL_IMAGE_BUFFER:
    case GL_IMAGE_2D_RECT:
    case GL_INT_IMAGE_1D:
    case GL_INT_IMAGE_2D:
    case GL_INT_IMAGE_3D:
    case GL_INT_IMAGE_CUBE:
    case GL_INT_IMAGE_1D_ARRAY:
    case GL_INT_IMAGE_2D_ARRAY:
    case GL_INT_IMAGE_2D_MULTISAMPLE:
    case GL_INT_IMAGE_2D_MULTISAMPLE_ARRAY:
    case GL_INT_IMAGE_CUBE_MAP_ARRAY:
    case GL_INT_IMAGE_BUFFER:
    case GL_INT_IMAGE_2D_RECT:
    case GL_UNSIGNED_INT_IMAGE_1D:
    case GL_UNSIGNED_INT_IMAGE_2D:
    case GL_UNSIGNED_INT_IMAGE_3D:
    case GL_UNSIGNED_INT_IMAGE_CUBE:
    case GL_UNSIGNED_INT_IMAGE_1D_ARRAY:
    case GL_UNSIGNED_INT_IMAGE_2D_ARRAY:
    case GL_UNSIGNED_INT_IMAGE_2D_MULTISAMPLE:
    case GL_UNSIGNED_INT_IMAGE_2D_MULTISAMPLE_ARRAY:
    case GL_UNSIGNED_INT_IMAGE_CUBE_MAP_ARRAY:
    case GL_UNSIGNED_INT_IMAGE_BUFFER:
    case GL_UNSIGNED_INT_IMAGE_2D_RECT:
        return true;
    default:
        return false;
    }
}

bool Gl::IsArrayTarget( GLenum target ) NN_NOEXCEPT
{
    return target == GL_TEXTURE_1D_ARRAY || target == GL_TEXTURE_2D_ARRAY
        || target == GL_TEXTURE_CUBE_MAP_ARRAY;
}

bool Gl::IsCompatibleBinaryFormat( GLenum binaryFormat ) NN_NOEXCEPT
{
    GLint formatCount = 0;
    NN_GFX_CALL_GL_FUNCTION( ::glGetIntegerv(
        GL_NUM_PROGRAM_BINARY_FORMATS, &formatCount ) );
    if( formatCount < 1 )
    {
        return false;
    }

    // 動的な数のフォーマット取得のため一時的にメモリーを確保
    std::unique_ptr< GLint[], FreeDeleter< GLint > > formats(
        static_cast< GLint* >( malloc( formatCount * sizeof( GLint ) ) ) );
    NN_GFX_CALL_GL_FUNCTION( ::glGetIntegerv(
        GL_PROGRAM_BINARY_FORMATS, formats.get() ) );
    for( int idxFormat = 0; idxFormat < formatCount; ++idxFormat )
    {
        if( static_cast< GLenum >( formats[ idxFormat ] ) == binaryFormat )
        {
            return true;
        }
    }

    return false;
}

GLenum Gl::GetQueryTarget( QueryTarget target ) NN_NOEXCEPT
{
    static const GLenum s_QueryTargetTable[] =
    {
        GL_TIMESTAMP,
        GL_SAMPLES_PASSED,
        GL_INVALID_ENUM,
        GL_INVALID_ENUM,
        GL_INVALID_ENUM,
        GL_INVALID_ENUM,
        GL_INVALID_ENUM,
        GL_INVALID_ENUM,
        GL_INVALID_ENUM,
        GL_INVALID_ENUM,
        GL_INVALID_ENUM,
        GL_INVALID_ENUM,
        GL_INVALID_ENUM
    };
    NN_STATIC_ASSERT( QueryTarget_End == NN_GFX_ARRAY_LENGTH( s_QueryTargetTable ) );

    static const GLenum s_QueryTargetTableEx[] =
    {
        GL_TIMESTAMP,
        GL_SAMPLES_PASSED,
        GL_VERTICES_SUBMITTED_ARB,
        GL_PRIMITIVES_SUBMITTED_ARB,
        GL_VERTEX_SHADER_INVOCATIONS_ARB,
        GL_GEOMETRY_SHADER_INVOCATIONS,
        GL_GEOMETRY_SHADER_PRIMITIVES_EMITTED_ARB,
        GL_CLIPPING_INPUT_PRIMITIVES_ARB,
        GL_CLIPPING_OUTPUT_PRIMITIVES_ARB,
        GL_FRAGMENT_SHADER_INVOCATIONS_ARB,
        GL_TESS_CONTROL_SHADER_PATCHES_ARB,
        GL_TESS_EVALUATION_SHADER_INVOCATIONS_ARB,
        GL_COMPUTE_SHADER_INVOCATIONS_ARB
    };
    NN_STATIC_ASSERT( QueryTarget_End == NN_GFX_ARRAY_LENGTH( s_QueryTargetTableEx ) );

    return GLEW_ARB_pipeline_statistics_query ? s_QueryTargetTableEx[ target ] : s_QueryTargetTable[ target ];
}

bool Gl::CheckRequiredVersion(GLint requiredVersionMajor, GLint requiredVersionMinor) NN_NOEXCEPT
{
    NN_UNUSED(requiredVersionMajor);
    NN_UNUSED(requiredVersionMinor);

    // GL バージョンチェック
    GLint glVersionMajor = 0, glVersionMinor = 0;
    NN_GFX_CALL_GL_FUNCTION(::glGetIntegerv(GL_MAJOR_VERSION, &glVersionMajor));
    NN_GFX_CALL_GL_FUNCTION(::glGetIntegerv(GL_MINOR_VERSION, &glVersionMinor));

    return glVersionMajor > requiredVersionMajor || (glVersionMajor == requiredVersionMajor && glVersionMinor >= requiredVersionMinor);
}

void Gl::InitializeDeviceCommonImpl(
    DeviceImpl< ApiVariationGl4 >* pDevice, const DeviceInfo& ) NN_NOEXCEPT
{
    DeviceImpl< ApiVariationGl4 >::DataType& data = pDevice->ToData();

    GlDeviceActivator activator( pDevice );

    glewExperimental = GL_TRUE;
    GLenum glewResult = glewInit();
    NN_UNUSED( glewResult );
    NN_SDK_ASSERT( glewResult == GLEW_OK, "glewInit: %s\n", ::glewGetErrorString( glewResult ) );

    // GLEW のバージョンによって、成功しても GL_INVALID_ENUM が返る可能性がある
    NN_GFX_CALL_GL_FUNCTION( glGetError() );

    // GL バージョンチェック
    NN_SDK_ASSERT( Gl::CheckRequiredVersion( GlRequiredVersionMajor, GlRequiredVersionMinor ) );

    // Get the maximum color attachments supported by the implementation
    GLint totalAttachments = -1;
    NN_GFX_CALL_GL_FUNCTION( ::glGetIntegerv( GL_MAX_COLOR_ATTACHMENTS, &totalAttachments ) );
    data.maxColorAttachments = totalAttachments;

    // バッファーの要求アライメント
    GLint constantBufferAlignment = -1;
    NN_GFX_CALL_GL_FUNCTION( ::glGetIntegerv(
        GL_UNIFORM_BUFFER_OFFSET_ALIGNMENT, &constantBufferAlignment ) );
    data.alignmentConstantBuffer = static_cast< uint32_t >( constantBufferAlignment );
    GLint unorderedAccessBufferAlignment = -1;
    NN_GFX_CALL_GL_FUNCTION( ::glGetIntegerv(
        GL_SHADER_STORAGE_BUFFER_OFFSET_ALIGNMENT, &unorderedAccessBufferAlignment ) );
    data.alignmentUnorderedAccessBuffer = static_cast< uint32_t >( unorderedAccessBufferAlignment );

    NN_GFX_GL_ASSERT();
}

}
}
}
