﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_BitPack.h>
#include <nn/util/util_BytePtr.h>

#include <nn/gfx/gfx_StateData-api.gl.4.h>

#include <nn/gfx/detail/gfx_Misc.h>

#include "gfx_GlCommand.h"
#include "gfx_GlHelper.h"

namespace nn {
namespace gfx {
namespace detail {

template<>
void GlCommandProc< GlDispatchParam >( const void* pParam, const GlCommandContext* ) NN_NOEXCEPT
{
    const GlDispatchParam& param = *static_cast< const GlDispatchParam* >( pParam );
    NN_GFX_CALL_GL_FUNCTION( ::glDispatchCompute(
        param.groupCountX, param.groupCountY, param.groupCountZ ) );
    NN_GFX_GL_ASSERT();
}

template<>
void GlCommandProc< GlDrawParam >( const void* pParam, const GlCommandContext* ) NN_NOEXCEPT
{
    const GlDrawParam& param = *static_cast< const GlDrawParam* >( pParam );
    NN_GFX_CALL_GL_FUNCTION( ::glDrawArrays( param.mode, param.offset, param.count ) );
    NN_GFX_GL_ASSERT();
}

template<>
void GlCommandProc< GlDrawInstancedParam >( const void* pParam, const GlCommandContext* ) NN_NOEXCEPT
{
    const GlDrawInstancedParam& param = *static_cast< const GlDrawInstancedParam* >( pParam );
    NN_GFX_CALL_GL_FUNCTION( ::glDrawArraysInstancedBaseInstance( param.drawParam.mode,
        param.drawParam.offset, param.drawParam.count, param.primCount, param.baseInstance ) );
    NN_GFX_GL_ASSERT();
}

template<>
void GlCommandProc< GlDrawIndexedParam >( const void* pParam, const GlCommandContext* pContext ) NN_NOEXCEPT
{
    const GlDrawIndexedParam& param = *static_cast< const GlDrawIndexedParam* >( pParam );
    NN_GFX_CALL_GL_FUNCTION( ::glBindVertexArray( pContext->hVao ) );
    NN_GFX_CALL_GL_FUNCTION( ::glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, param.hIndexBuffer ) );
    NN_GFX_CALL_GL_FUNCTION( ::glDrawElementsBaseVertex( param.mode, param.count,
        param.type, reinterpret_cast< void* >( static_cast< intptr_t >( param.offset ) ), param.baseVertex ) );
    NN_GFX_GL_ASSERT();
}

template<>
void GlCommandProc< GlDrawIndexedInstancedParam >( const void* pParam, const GlCommandContext* pContext ) NN_NOEXCEPT
{
    const GlDrawIndexedInstancedParam& param = *static_cast< const GlDrawIndexedInstancedParam* >( pParam );
    NN_GFX_CALL_GL_FUNCTION( ::glBindVertexArray( pContext->hVao ) );
    NN_GFX_CALL_GL_FUNCTION( ::glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, param.drawIndexedParam.hIndexBuffer ) );
    NN_GFX_CALL_GL_FUNCTION( ::glDrawElementsInstancedBaseVertexBaseInstance(
        param.drawIndexedParam.mode, param.drawIndexedParam.count, param.drawIndexedParam.type,
        reinterpret_cast< void* >( static_cast< intptr_t >( param.drawIndexedParam.offset ) ),
        param.primCount, param.drawIndexedParam.baseVertex, param.baseInstance ) );
    NN_GFX_GL_ASSERT();
}

template<>
void GlCommandProc< GlDispatchIndirectParam >( const void* pParam, const GlCommandContext* ) NN_NOEXCEPT
{
    const GlDispatchIndirectParam& param = *static_cast< const GlDispatchIndirectParam* >( pParam );
    NN_GFX_CALL_GL_FUNCTION( ::glBindBuffer( GL_DISPATCH_INDIRECT_BUFFER, param.hIndirectBuffer ) );
    NN_GFX_CALL_GL_FUNCTION( ::glDispatchComputeIndirect( param.offset ) );
    NN_GFX_GL_ASSERT();
}

template<>
void GlCommandProc< GlDrawIndirectParam >( const void* pParam, const GlCommandContext* ) NN_NOEXCEPT
{
    const GlDrawIndirectParam& param = *static_cast< const GlDrawIndirectParam* >( pParam );
    NN_GFX_CALL_GL_FUNCTION( ::glBindBuffer( GL_DRAW_INDIRECT_BUFFER, param.hIndirectBuffer ) );
    NN_GFX_CALL_GL_FUNCTION( ::glDrawArraysIndirect( param.mode,
        reinterpret_cast< void* >( static_cast< uintptr_t >( param.offset ) ) ) );
    NN_GFX_GL_ASSERT();
}

template<>
void GlCommandProc< GlDrawIndexedIndirectParam >( const void* pParam, const GlCommandContext* pContext ) NN_NOEXCEPT
{
    const GlDrawIndexedIndirectParam& param = *static_cast< const GlDrawIndexedIndirectParam* >( pParam );
    NN_GFX_CALL_GL_FUNCTION( ::glBindVertexArray( pContext->hVao ) );
    NN_GFX_CALL_GL_FUNCTION( ::glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, param.hIndexBuffer ) );
    NN_GFX_CALL_GL_FUNCTION( ::glBindBuffer( GL_DRAW_INDIRECT_BUFFER, param.hIndirectBuffer ) );
    NN_GFX_CALL_GL_FUNCTION( ::glDrawElementsIndirect( param.mode, param.type,
        reinterpret_cast< void* >( static_cast< uintptr_t >( param.offset ) ) ) );
    NN_GFX_GL_ASSERT();
}

template<>
void GlCommandProc< GlCopyBufferParam >( const void* pParam, const GlCommandContext* ) NN_NOEXCEPT
{
    const GlCopyBufferParam& param = *static_cast< const GlCopyBufferParam* >( pParam );
    NN_GFX_CALL_GL_FUNCTION( ::glNamedCopyBufferSubDataEXT( param.hSrcBuffer,
        param.hDstBuffer, param.srcOffset, param.dstOffset, param.size ) );
    NN_GFX_GL_ASSERT();
}

template<>
void GlCommandProc< GlCopyImageParam >( const void* pParam, const GlCommandContext* ) NN_NOEXCEPT
{
    const GlCopyImageParam& param = *static_cast< const GlCopyImageParam* >( pParam );
    NN_GFX_CALL_GL_FUNCTION( ::glCopyImageSubData( param.hSrcTexture, param.srcTarget,
        param.srcLevel, param.srcX, param.srcY, param.srcZ, param.hDstTexture, param.dstTarget,
        param.dstLevel, param.dstX, param.dstY, param.dstZ, param.width, param.height, param.depth ) );
    NN_GFX_GL_ASSERT();
}

template<>
void GlCommandProc< GlCopyBufferToImageParam >( const void* pParam, const GlCommandContext* ) NN_NOEXCEPT
{
    const GlBufferImageCopyParam& param = static_cast< const GlCopyBufferToImageParam* >( pParam )->param;
    NN_GFX_CALL_GL_FUNCTION( ::glBindBuffer( GL_PIXEL_UNPACK_BUFFER, param.hBuffer ) );

    NN_GFX_CALL_GL_FUNCTION( ::glPixelStorei( GL_UNPACK_ROW_LENGTH, param.rowLength ) );
    NN_GFX_CALL_GL_FUNCTION( ::glPixelStorei( GL_UNPACK_IMAGE_HEIGHT, param.imageHeight ) );

    NN_GFX_GL_ASSERT();

    Gl::TextureSubImage( param.hTexture, param.target, param.level, param.isCompressed ? true : false,
        param.internalFormat, param.pixelFormat, param.pixelType, param.xOffset, param.yOffset,
        param.zOffset, param.width, param.height, param.depth, param.imageSize,
        reinterpret_cast< const void* >( static_cast< intptr_t >( param.bufferOffset ) ) );
}

template<>
void GlCommandProc< GlCopyImageToBufferParam >( const void* pParam, const GlCommandContext* ) NN_NOEXCEPT
{
    const GlBufferImageCopyParam& param = static_cast< const GlCopyImageToBufferParam* >( pParam )->param;

    GLint64 size;
    NN_GFX_CALL_GL_FUNCTION( ::glGetNamedBufferParameteri64v(
        param.hBuffer, GL_BUFFER_SIZE, &size ) );
    NN_GFX_CALL_GL_FUNCTION( ::glBindBuffer( GL_PIXEL_PACK_BUFFER, param.hBuffer ) );
    NN_GFX_CALL_GL_FUNCTION( ::glPixelStorei( GL_PACK_ROW_LENGTH, param.rowLength ) );
    NN_GFX_CALL_GL_FUNCTION( ::glPixelStorei( GL_PACK_IMAGE_HEIGHT, param.imageHeight ) );
    NN_GFX_GL_ASSERT();

    if( param.isCompressed )
    {
        NN_GFX_CALL_GL_FUNCTION( ::glGetCompressedTextureSubImage(
            param.hTexture, param.level, param.xOffset, param.yOffset, param.zOffset,
            param.width, param.height, param.depth, static_cast< GLsizei >( size ),
            reinterpret_cast< void* >( static_cast< intptr_t >( param.bufferOffset ) ) ) );
    }
    else
    {
        NN_GFX_CALL_GL_FUNCTION( ::glGetTextureSubImage(
            param.hTexture, param.level, param.xOffset, param.yOffset, param.zOffset, param.width,
            param.height, param.depth, param.pixelFormat,  param.pixelType, static_cast< GLsizei >( size ),
            reinterpret_cast< void* >( static_cast< intptr_t >( param.bufferOffset ) ) ) );
    }
    NN_GFX_GL_ASSERT();
}

template<>
void GlCommandProc< GlBlitImageParam >( const void* pParam, const GlCommandContext* pContext ) NN_NOEXCEPT
{
    const GlBlitImageParam& param = *static_cast< const GlBlitImageParam* >( pParam );

    Gl::SetEnable( GL_FRAMEBUFFER_SRGB, true );
    GLboolean backScissorTest = NN_GFX_CALL_GL_FUNCTION( ::glIsEnabled( GL_SCISSOR_TEST ) );
    if( backScissorTest == GL_TRUE )
    {
        NN_GFX_CALL_GL_FUNCTION( ::glDisable( GL_SCISSOR_TEST ) );
    }

    GLint tmpFboDraw;
    NN_GFX_CALL_GL_FUNCTION( ::glGetIntegerv( GL_DRAW_FRAMEBUFFER_BINDING, &tmpFboDraw ) );
    NN_GFX_CALL_GL_FUNCTION( ::glBindFramebuffer( GL_DRAW_FRAMEBUFFER, pContext->hTmpFbo[ 0 ] ) );
    NN_GFX_CALL_GL_FUNCTION( ::glBindFramebuffer( GL_READ_FRAMEBUFFER, pContext->hTmpFbo[ 1 ] ) );
    GLenum attachment = GL_COLOR_ATTACHMENT0;
    NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA(
        ::glFramebufferDrawBuffers )( pContext->hTmpFbo[ 0 ], 1, &attachment ) );

    for( int arrayIndex = 0; arrayIndex < param.arrayCount; ++arrayIndex )
    {
        int srcLayer = param.srcStartArrayIndex + arrayIndex;
        int dstLayer = param.dstStartArrayIndex + arrayIndex;
        Gl::FramebufferTexture( pContext->hTmpFbo[ 1 ],
            GL_COLOR_ATTACHMENT0, param.hSrcTexture, param.srcMipLevel, srcLayer );
        Gl::FramebufferTexture( pContext->hTmpFbo[ 0 ],
            GL_COLOR_ATTACHMENT0, param.hDstTexture, param.dstMipLevel, dstLayer );
        NN_GFX_CALL_GL_FUNCTION( ::glBlitFramebuffer( param.srcX0, param.srcY0,
            param.srcX1, param.srcY1, param.dstX0, param.dstY0,
            param.dstX1, param.dstY1, param.mask, param.filter ) );
        NN_GFX_GL_ASSERT();
    }

    NN_GFX_CALL_GL_FUNCTION( ::glBindFramebuffer( GL_FRAMEBUFFER, tmpFboDraw ) );

    if( backScissorTest == GL_TRUE )
    {
        NN_GFX_CALL_GL_FUNCTION( ::glEnable( GL_SCISSOR_TEST ) );
    }

    NN_GFX_GL_ASSERT();
}

template<>
void GlCommandProc< GlSetRenderTargetsParam >( const void* pParam, const GlCommandContext* pContext ) NN_NOEXCEPT
{
    const GlSetRenderTargetsParam& param = *static_cast< const GlSetRenderTargetsParam* >( pParam );

    NN_GFX_CALL_GL_FUNCTION( ::glBindFramebuffer( GL_FRAMEBUFFER, pContext->hFbo ) );

    static const GLenum s_DrawBufferTable[] =
    {
        GL_COLOR_ATTACHMENT0,
        GL_COLOR_ATTACHMENT1,
        GL_COLOR_ATTACHMENT2,
        GL_COLOR_ATTACHMENT3,
        GL_COLOR_ATTACHMENT4,
        GL_COLOR_ATTACHMENT5,
        GL_COLOR_ATTACHMENT6,
        GL_COLOR_ATTACHMENT7,
        GL_COLOR_ATTACHMENT8,
        GL_COLOR_ATTACHMENT9,
        GL_COLOR_ATTACHMENT10,
        GL_COLOR_ATTACHMENT11,
        GL_COLOR_ATTACHMENT12,
        GL_COLOR_ATTACHMENT13,
        GL_COLOR_ATTACHMENT14,
        GL_COLOR_ATTACHMENT15,
    };
    NN_SDK_ASSERT( NN_GFX_ARRAY_LENGTH( s_DrawBufferTable ) >= param.maxColorAttachments );

    NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glFramebufferDrawBuffers )(
        pContext->hFbo, param.colorTargetCount, s_DrawBufferTable ) );

    // Disable previous textures not getting overwritten
    for ( int idxTarget = param.colorTargetCount; idxTarget < param.maxColorAttachments; ++idxTarget )
    {
        NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glNamedFramebufferTexture )(
            pContext->hFbo, GL_COLOR_ATTACHMENT0 + idxTarget, GlInvalidHandle, 0 ) );
    }
    NN_GFX_GL_ASSERT();

    // Attach the new color attachments
    for( int idxTarget = 0; idxTarget < param.colorTargetCount; ++idxTarget )
    {
        const GlSetRenderTargetsParam::RenderTarget& target = param.colorTargets[ idxTarget ];
        if( IsValid( target.hColorTarget ) )
        {
            if( target.layer >= 0 )
            {
                NN_GFX_CALL_GL_FUNCTION( glFramebufferTexture3D( GL_FRAMEBUFFER,
                    GL_COLOR_ATTACHMENT0 + idxTarget, GL_TEXTURE_3D, target.hColorTarget, 0, target.layer ) );
            }
            else
            {
                NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glNamedFramebufferTexture )(
                    pContext->hFbo, GL_COLOR_ATTACHMENT0 + idxTarget, target.hColorTarget, 0 ) );
            }
        }
    }
    NN_GFX_GL_ASSERT();

    // Since GL attaches the depth and stencil components separately it's
    // necessary to detach both first.
    NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glNamedFramebufferTexture )(
        pContext->hFbo, GL_DEPTH_ATTACHMENT, GlInvalidHandle, 0 ) );
    NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glNamedFramebufferTexture )(
        pContext->hFbo, GL_STENCIL_ATTACHMENT, GlInvalidHandle, 0 ) );

    if( IsValid( param.hDepthStencil ) )
    {
        NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glNamedFramebufferTexture )(
            pContext->hFbo, param.depthStencilAttachment, param.hDepthStencil, 0 ) );
    }

    NN_GFX_CALL_GL_FUNCTION( ::glEnable( GL_FRAMEBUFFER_SRGB ) );

    NN_GFX_GL_ASSERT();

    NN_SDK_ASSERT( Gl::CheckFramebufferStatus( pContext->hFbo ) );
}

template<>
void GlCommandProc< GlSetVertexBufferParam >( const void* pParam, const GlCommandContext* pContext ) NN_NOEXCEPT
{
    const GlSetVertexBufferParam& param = *static_cast< const GlSetVertexBufferParam* >( pParam );
    NN_GFX_CALL_GL_FUNCTION( ::glBindVertexArray( pContext->hVao ) );
    NN_GFX_CALL_GL_FUNCTION( ::glBindVertexBuffer( param.bufferIndex,
        param.hBuffer, param.offset, param.stride ) );
    NN_GFX_GL_ASSERT();
}

template<>
void GlCommandProc< GlSetShaderParam >( const void* pParam, const GlCommandContext* ) NN_NOEXCEPT
{
    const GlSetShaderParam& param = *static_cast< const GlSetShaderParam* >( pParam );
    NN_GFX_CALL_GL_FUNCTION( ::glBindProgramPipeline( GlInvalidHandle ) );
    NN_GFX_CALL_GL_FUNCTION( ::glUseProgram( param.hProgram ) );
    NN_GFX_GL_ASSERT();
}

template<>
void GlCommandProc< GlSetSeparateShaderParam >( const void* pParam, const GlCommandContext* pContext ) NN_NOEXCEPT
{
    const GlSetSeparateShaderParam& param = *static_cast< const GlSetSeparateShaderParam* >( pParam );

    static GLenum s_GlStageBitTable[] =
    {
        GL_VERTEX_SHADER_BIT,
        GL_TESS_CONTROL_SHADER_BIT,
        GL_TESS_EVALUATION_SHADER_BIT,
        GL_GEOMETRY_SHADER_BIT,
        GL_FRAGMENT_SHADER_BIT,
        GL_COMPUTE_SHADER_BIT
    };

    static int s_StageBitTable[] =
    {
        ShaderStageBit_Vertex,
        ShaderStageBit_Hull,
        ShaderStageBit_Domain,
        ShaderStageBit_Geometry,
        ShaderStageBit_Pixel,
        ShaderStageBit_Compute
    };

    static GlHandle GlSetSeparateShaderParam::* const s_pShaderProgram[] =
    {
        &GlSetSeparateShaderParam::hVertexProgram,
        &GlSetSeparateShaderParam::hHullProgram,
        &GlSetSeparateShaderParam::hDomainProgram,
        &GlSetSeparateShaderParam::hGeometryProgram,
        &GlSetSeparateShaderParam::hPixelProgram,
        &GlSetSeparateShaderParam::hComputeProgram
    };

    NN_GFX_CALL_GL_FUNCTION( ::glUseProgram( GlInvalidHandle ) );
    NN_GFX_CALL_GL_FUNCTION( ::glBindProgramPipeline( pContext->hPipeline ) );
    for( int idxStage = 0; idxStage < ShaderStage_End; ++idxStage )
    {
        if( param.stageBits & s_StageBitTable[ idxStage ] )
        {
            NN_GFX_CALL_GL_FUNCTION( ::glUseProgramStages( pContext->hPipeline,
                s_GlStageBitTable[ idxStage ], param.*s_pShaderProgram[ idxStage ] ) );
        }
    }
    NN_GFX_GL_ASSERT();
}

template<>
void GlCommandProc< GlSetSamplerParam >( const void* pParam, const GlCommandContext* ) NN_NOEXCEPT
{
    const GlSetSamplerParam& param = *static_cast< const GlSetSamplerParam* >( pParam );
    NN_GFX_CALL_GL_FUNCTION( ::glBindSampler( param.unit, param.hSampler ) );
    NN_GFX_GL_ASSERT();
}

template<>
void GlCommandProc< GlSetTextureParam >( const void* pParam, const GlCommandContext* ) NN_NOEXCEPT
{
    const GlSetTextureParam& param = *static_cast< const GlSetTextureParam* >( pParam );
    //NN_GFX_CALL_GL_FUNCTION( ::glBindTextureUnit( param.unit, param.target, param.hTexture ) );
    NN_GFX_CALL_GL_FUNCTION( ::glActiveTexture( GL_TEXTURE0 + param.unit ) );
    NN_GFX_CALL_GL_FUNCTION( ::glBindTexture( param.target, param.hTexture ) );
    NN_GFX_GL_ASSERT();
}

template<>
void GlCommandProc< GlSetTextureAndSamplerParam >( const void* pParam, const GlCommandContext* ) NN_NOEXCEPT
{
    const GlSetTextureAndSamplerParam& param = *static_cast< const GlSetTextureAndSamplerParam* >( pParam );
    NN_GFX_CALL_GL_FUNCTION( ::glActiveTexture( GL_TEXTURE0 + param.unit ) );
    NN_GFX_CALL_GL_FUNCTION( ::glBindTexture( param.target, param.hTexture ) );
    NN_GFX_CALL_GL_FUNCTION( ::glBindSampler( param.unit, param.hSampler ) );
    NN_GFX_GL_ASSERT();
}

template<>
void GlCommandProc< GlSetImageParam >( const void* pParam, const GlCommandContext* ) NN_NOEXCEPT
{
    const GlSetImageParam& param = *static_cast<const GlSetImageParam*>( pParam );
    GLint format;
    NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glGetTextureLevelParameteriv )(
        param.hTexture, param.target, 0, GL_TEXTURE_INTERNAL_FORMAT, &format ) );
    GLboolean isLayered = Gl::IsLayeredTarget( param.target ) ? GL_TRUE : GL_FALSE;
    NN_GFX_CALL_GL_FUNCTION( ::glBindImageTexture( param.unit,
        param.hTexture, 0, isLayered, 0, GL_READ_WRITE, format ) );
    NN_GFX_GL_ASSERT();
}

template<>
void GlCommandProc< GlSetBufferParam >( const void* pParam, const GlCommandContext* ) NN_NOEXCEPT
{
    const GlSetBufferParam& param = *static_cast< const GlSetBufferParam* >( pParam );
    NN_GFX_CALL_GL_FUNCTION( ::glBindBufferRange( param.target,
        param.index, param.hBuffer, param.offset, param.size ) );
    NN_GFX_GL_ASSERT();
}

template<>
void GlCommandProc< GlClearBufferParam >( const void* pParam, const GlCommandContext* ) NN_NOEXCEPT
{
    const GlClearBufferParam& param = *static_cast< const GlClearBufferParam* >( pParam );
    NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glClearNamedBufferSubData )(
        param.hBuffer, GL_RGBA8, param.offset, param.size, GL_RGBA, GL_UNSIGNED_BYTE, &param.value ) );
    NN_GFX_GL_ASSERT();
}

template<>
void GlCommandProc< GlClearColorParam >( const void* pParam, const GlCommandContext* pContext ) NN_NOEXCEPT
{
    const GlClearColorParam& param = *static_cast< const GlClearColorParam* >( pParam );

    GLint tmpFboDraw;
    NN_GFX_CALL_GL_FUNCTION( ::glGetIntegerv( GL_DRAW_FRAMEBUFFER_BINDING, &tmpFboDraw ) );

    NN_GFX_CALL_GL_FUNCTION( ::glBindFramebuffer( GL_DRAW_FRAMEBUFFER, pContext->hTmpFbo[ 0 ] ) );

    GLenum attachment = GL_COLOR_ATTACHMENT0;
    NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glFramebufferDrawBuffers )(
        pContext->hTmpFbo[ 0 ], 1, &attachment ) );

    GLboolean backWriteMask[ 4 ];
    NN_GFX_CALL_GL_FUNCTION( ::glGetBooleanv( GL_COLOR_WRITEMASK, backWriteMask ) );
    NN_GFX_CALL_GL_FUNCTION( ::glColorMaski( 0, GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE ) );
    GLboolean backScissorTest = NN_GFX_CALL_GL_FUNCTION( ::glIsEnabled( GL_SCISSOR_TEST ) );
    if( backScissorTest == GL_TRUE )
    {
        NN_GFX_CALL_GL_FUNCTION( ::glDisable( GL_SCISSOR_TEST ) );
    }

    NN_GFX_CALL_GL_FUNCTION( ::glEnable( GL_FRAMEBUFFER_SRGB ) );

    GLint type;
    NN_GFX_CALL_GL_FUNCTION( ::glGetTextureLevelParameteriv(
        param.hTexture, 0, GL_TEXTURE_RED_TYPE, &type ) );

    for( int layer = param.startLayer; layer < param.endLayer; ++layer )
    {
        Gl::FramebufferTexture( pContext->hTmpFbo[ 0 ], attachment, param.hTexture, 0, layer );
        NN_SDK_ASSERT( Gl::CheckFramebufferStatus( pContext->hTmpFbo[ 0 ] ) );

        switch( type )
        {
        case GL_INT:
        {
            NN_GFX_CALL_GL_FUNCTION( ::glClearBufferiv( GL_COLOR, 0, param.colori ) );
        }
        break;
        case GL_UNSIGNED_INT:
        {
            NN_GFX_CALL_GL_FUNCTION( ::glClearBufferuiv( GL_COLOR, 0, param.coloru ) );
        }
        break;
        default:
        {
            NN_GFX_CALL_GL_FUNCTION( ::glClearBufferfv( GL_COLOR, 0, param.color ) ); // TODO: Named に置き換え
        }
        break;
        }
    }

    // Detach the texture from the FBO
    Gl::FramebufferTexture( pContext->hTmpFbo[ 0 ], attachment, GlInvalidHandle, 0, 0 );

    NN_GFX_CALL_GL_FUNCTION( ::glColorMaski( 0, backWriteMask[ 0 ],
        backWriteMask[ 1 ], backWriteMask[ 2 ], backWriteMask[ 3 ] ) );
    if( backScissorTest == GL_TRUE )
    {
        NN_GFX_CALL_GL_FUNCTION( ::glEnable( GL_SCISSOR_TEST ) );
    }

    NN_GFX_CALL_GL_FUNCTION( ::glBindFramebuffer( GL_FRAMEBUFFER, tmpFboDraw ) );

    NN_GFX_GL_ASSERT();
}

template<>
void GlCommandProc< GlClearDepthStencilParam >( const void* pParam, const GlCommandContext* pContext ) NN_NOEXCEPT
{
    const GlClearDepthStencilParam& param = *static_cast< const GlClearDepthStencilParam* >( pParam );

    GLint tmpFboDraw;
    NN_GFX_CALL_GL_FUNCTION( ::glGetIntegerv( GL_DRAW_FRAMEBUFFER_BINDING, &tmpFboDraw ) );

    NN_GFX_CALL_GL_FUNCTION( ::glBindFramebuffer( GL_DRAW_FRAMEBUFFER, pContext->hTmpFbo[ 0 ] ) );

    GLboolean backDepthWriteMask = GL_TRUE;
    GLint backStencilWriteMask = 0;
    GLint backStencilBackWriteMask = 0;
    GLboolean backScissorTest = NN_GFX_CALL_GL_FUNCTION( ::glIsEnabled( GL_SCISSOR_TEST ) );

    if( backScissorTest == GL_TRUE )
    {
        NN_GFX_CALL_GL_FUNCTION( ::glDisable( GL_SCISSOR_TEST ) );
    }

    if( param.clearMode & DepthStencilClearMode_Depth )
    {
        NN_GFX_CALL_GL_FUNCTION( ::glGetBooleanv( GL_DEPTH_WRITEMASK, &backDepthWriteMask ) );
        if( backDepthWriteMask != GL_TRUE )
        {
            NN_GFX_CALL_GL_FUNCTION( ::glDepthMask( GL_TRUE ) );
        }
    }

    if( param.clearMode & DepthStencilClearMode_Stencil )
    {
        NN_GFX_CALL_GL_FUNCTION( ::glGetIntegerv( GL_STENCIL_WRITEMASK, &backStencilWriteMask ) );
        NN_GFX_CALL_GL_FUNCTION( ::glGetIntegerv( GL_STENCIL_BACK_WRITEMASK, &backStencilBackWriteMask ) );
        NN_GFX_CALL_GL_FUNCTION( ::glStencilMask( static_cast< GLuint >( ~0 ) ) );
    }

    for( int layer = param.startLayer; layer < param.endLayer; ++layer )
    {
        Gl::FramebufferTexture( pContext->hTmpFbo[ 0 ], param.attachment, param.hTexture, 0, layer );
        NN_SDK_ASSERT( Gl::CheckFramebufferStatus( pContext->hTmpFbo[ 0 ] ) );
        switch( param.clearMode )
        {
        case DepthStencilClearMode_Depth:
            {
                NN_GFX_CALL_GL_FUNCTION( ::glClearBufferfv( GL_DEPTH, 0, &param.depth ) );
            }
            break;
        case DepthStencilClearMode_Stencil:
            {
                NN_GFX_CALL_GL_FUNCTION( ::glClearBufferiv( GL_STENCIL, 0, &param.stencil ) );
            }
            break;
        case DepthStencilClearMode_DepthStencil:
            {
                NN_GFX_CALL_GL_FUNCTION( ::glClearBufferfi( GL_DEPTH_STENCIL, 0, param.depth, param.stencil ) );
            }
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    // Detach the texture from the FBO
    Gl::FramebufferTexture( pContext->hTmpFbo[ 0 ], param.attachment, GlInvalidHandle, 0, 0 );

    if( ( param.clearMode & DepthStencilClearMode_Depth ) && backDepthWriteMask != GL_TRUE )
    {
        NN_GFX_CALL_GL_FUNCTION( ::glDepthMask( backDepthWriteMask ) );
    }
    if( param.clearMode & DepthStencilClearMode_Stencil )
    {
        NN_GFX_CALL_GL_FUNCTION( ::glStencilMaskSeparate( GL_FRONT, backStencilWriteMask ) );
        NN_GFX_CALL_GL_FUNCTION( ::glStencilMaskSeparate( GL_BACK, backStencilBackWriteMask ) );
    }

    if( backScissorTest == GL_TRUE )
    {
        NN_GFX_CALL_GL_FUNCTION( ::glEnable( GL_SCISSOR_TEST ) );
    }

    NN_GFX_CALL_GL_FUNCTION( ::glBindFramebuffer( GL_FRAMEBUFFER, tmpFboDraw ) );

    NN_GFX_GL_ASSERT();
}

template<>
void GlCommandProc< GlResolveParam >( const void* pParam, const GlCommandContext* pContext ) NN_NOEXCEPT
{
    const GlResolveParam& param = *static_cast< const GlResolveParam* >( pParam );

    Gl::SetEnable( GL_FRAMEBUFFER_SRGB, true );
    GLboolean backScissorTest = NN_GFX_CALL_GL_FUNCTION( ::glIsEnabled( GL_SCISSOR_TEST ) );
    if( backScissorTest == GL_TRUE )
    {
        NN_GFX_CALL_GL_FUNCTION( ::glDisable( GL_SCISSOR_TEST ) );
    }

    GLint tmpFboDraw;
    NN_GFX_CALL_GL_FUNCTION( ::glGetIntegerv( GL_DRAW_FRAMEBUFFER_BINDING, &tmpFboDraw ) );

    NN_GFX_CALL_GL_FUNCTION( ::glBindFramebuffer( GL_DRAW_FRAMEBUFFER, pContext->hTmpFbo[ 0 ] ) );
    NN_GFX_CALL_GL_FUNCTION( ::glBindFramebuffer( GL_READ_FRAMEBUFFER, pContext->hTmpFbo[ 1 ] ) );
    GLenum attachment = GL_COLOR_ATTACHMENT0;
    NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA(
        ::glFramebufferDrawBuffers )( pContext->hTmpFbo[ 0 ], 1, &attachment ) );

    for( int layer = param.startLayer; layer < param.endLayer; ++layer )
    {
        int dstLayer = param.dstStartArrayIndex + layer - param.startLayer;
        Gl::FramebufferTexture( pContext->hTmpFbo[ 0 ], attachment, param.hDstTexture, param.dstMipLevel, dstLayer );
        Gl::FramebufferTexture( pContext->hTmpFbo[ 1 ], GL_COLOR_ATTACHMENT0, param.hSrcTexture, 0, layer );
        NN_GFX_CALL_GL_FUNCTION( ::glBlitFramebuffer( 0, 0, param.srcWidth, param.srcHeight, 0, 0,
            param.dstWidth, param.dstHeight, GL_COLOR_BUFFER_BIT, GL_NEAREST ) );
    }

    NN_GFX_CALL_GL_FUNCTION( ::glBindFramebuffer( GL_FRAMEBUFFER, tmpFboDraw ) );

    if( backScissorTest == GL_TRUE )
    {
        NN_GFX_CALL_GL_FUNCTION( ::glEnable( GL_SCISSOR_TEST ) );
    }

    NN_GFX_GL_ASSERT();
}

template<>
void GlCommandProc< GlMemoryBarrierParam >( const void* pParam, const GlCommandContext* ) NN_NOEXCEPT
{
    const GlMemoryBarrierParam& param = *static_cast< const GlMemoryBarrierParam* >( pParam );

    NN_GFX_CALL_GL_FUNCTION( ::glMemoryBarrier( param.barriers ) );

    NN_GFX_GL_ASSERT();
}

template<>
void GlCommandProc< GlBeginQueryParam >( const void* pParam, const GlCommandContext* pContext ) NN_NOEXCEPT
{
    const GlBeginQueryParam& param = *static_cast< const GlBeginQueryParam* >( pParam );

    NN_GFX_CALL_GL_FUNCTION( ::glBeginQuery( param.target, pContext->hQuery[ param.idxQueryObject ] ) );

    NN_GFX_GL_ASSERT();
}

template<>
void GlCommandProc< GlEndQueryParam >( const void* pParam, const GlCommandContext* pContext ) NN_NOEXCEPT
{
    const GlEndQueryParam& param = *static_cast< const GlEndQueryParam* >( pParam );

    NN_GFX_CALL_GL_FUNCTION( ::glBindBuffer( GL_QUERY_BUFFER, param.hBuffer ) );
    NN_GFX_CALL_GL_FUNCTION( ::glEndQuery( param.target ) );
    NN_GFX_CALL_GL_FUNCTION( ::glGetQueryObjecti64v( pContext->hQuery[ param.idxQueryObject ], GL_QUERY_RESULT,
        reinterpret_cast< GLint64* >( static_cast< uintptr_t >( param.bufferOffset ) ) ) );

    NN_GFX_GL_ASSERT();
}

template<>
void GlCommandProc< GlWriteTimestampParam >(
    const void* pParam, const GlCommandContext* pContext ) NN_NOEXCEPT
{
    const GlWriteTimestampParam& param = *static_cast< const GlWriteTimestampParam* >( pParam );

    NN_GFX_CALL_GL_FUNCTION( ::glBindBuffer( GL_QUERY_BUFFER, param.hBuffer ) );
    NN_GFX_CALL_GL_FUNCTION( ::glQueryCounter( pContext->hQuery[ 0 ], GL_TIMESTAMP ) );
    NN_GFX_CALL_GL_FUNCTION( ::glGetQueryObjecti64v( pContext->hQuery[ 0 ], GL_QUERY_RESULT,
        reinterpret_cast< GLint64* >( static_cast< uintptr_t >( param.bufferOffset ) ) ) );

    NN_GFX_GL_ASSERT();
}

template<>
void GlCommandProc< GlSetDepthBoundsParam >(
    const void* pParam, const GlCommandContext* ) NN_NOEXCEPT
{
    const GlSetDepthBoundsParam& param = *static_cast< const GlSetDepthBoundsParam* >( pParam );

    if( GLEW_EXT_depth_bounds_test )
    {
        NN_GFX_CALL_GL_FUNCTION( ::glDepthBoundsEXT( param.minDepthBounds, param.maxDepthBounds ) );
    }
}

template<>
void GlCommandProc< GlSetLineWidthParam >(
    const void* pParam, const GlCommandContext* ) NN_NOEXCEPT
{
    const GlSetLineWidthParam& param = *static_cast< const GlSetLineWidthParam* >( pParam );

    NN_GFX_CALL_GL_FUNCTION( ::glLineWidth( param.lineWidth ) );
}

template<>
void GlCommandProc< GlSetViewportsParam >(
    const void* pParam, const GlCommandContext* ) NN_NOEXCEPT
{
    const GlSetViewportsParam& param = *static_cast< const GlSetViewportsParam* >( pParam );

    const GLfloat* pViewports = nn::util::ConstBytePtr(
        &param, sizeof( GlSetViewportsParam ) ).Get< GLfloat >();
    NN_GFX_CALL_GL_FUNCTION( ::glViewportArrayv( param.first, param.count, pViewports ) );

    const GLdouble* pDepthRanges = nn::util::ConstBytePtr( &param,
        sizeof( GlSetViewportsParam ) + sizeof( float ) * 4 * param.count ).Get< GLdouble >();
    NN_GFX_CALL_GL_FUNCTION( ::glDepthRangeArrayv( param.first, param.count, pDepthRanges ) );

    NN_GFX_GL_ASSERT();
}

template<>
void GlCommandProc< GlSetScissorsParam >(
    const void* pParam, const GlCommandContext* ) NN_NOEXCEPT
{
    const GlSetScissorsParam& param = *static_cast< const GlSetScissorsParam* >( pParam );

    const GLint* pScissors = nn::util::ConstBytePtr(
        &param, sizeof( GlSetScissorsParam ) ).Get< GLint >();
    NN_GFX_CALL_GL_FUNCTION( ::glScissorArrayv( param.first, param.count, pScissors ) );

    NN_GFX_GL_ASSERT();
}

template<>
void GlCommandProc< GlBufferSubDataParam >(
    const void* pParam, const GlCommandContext* ) NN_NOEXCEPT
{
    const GlBufferSubDataParam& param = *static_cast<const GlBufferSubDataParam*>( pParam );

    const void* pData = nn::util::ConstBytePtr( pParam, sizeof( GlBufferSubDataParam ) ).Get();
    NN_GFX_CALL_GL_FUNCTION( ::glNamedBufferSubData( param.hBuffer, param.offset, param.size, pData ) );

    NN_GFX_GL_ASSERT();
}

template<>
void GlCommandProc< RasterizerStateImplData< ApiVariationGl4 > >(
    const void* pParam, const GlCommandContext* ) NN_NOEXCEPT
{
    const RasterizerStateImplData< ApiVariationGl4 >& param =
        *static_cast< const RasterizerStateImplData< ApiVariationGl4 >* >( pParam );

    NN_GFX_CALL_GL_FUNCTION( Gl::PolygonMode( GL_FRONT_AND_BACK, param.polygonMode ) );
    NN_GFX_CALL_GL_FUNCTION( ::glFrontFace( param.frontFace ) );
    Gl::SetEnable( GL_CULL_FACE, param.flag.GetBit(
        RasterizerStateImplData< ApiVariationGl4 >::Flag_CullEnable ) );
    if( param.flag.GetBit( RasterizerStateImplData< ApiVariationGl4 >::Flag_CullEnable ) )
    {
        NN_GFX_CALL_GL_FUNCTION( ::glCullFace( param.cullFace ) );
    }
    Gl::SetEnable( GL_RASTERIZER_DISCARD, param.flag.GetBit(
        RasterizerStateImplData< ApiVariationGl4 >::Flag_RasterDisable ) );
    Gl::SetEnable( GL_MULTISAMPLE, param.flag.GetBit(
        RasterizerStateImplData< ApiVariationGl4 >::Flag_MultisampleEnable ) );
    Gl::SetEnable( GL_DEPTH_CLAMP, param.flag.GetBit(
        RasterizerStateImplData< ApiVariationGl4 >::Flag_DepthClampEnable ) );

    if( param.flag.GetBit( RasterizerStateImplData< ApiVariationGl4 >::Flag_PolygonOffsetEnable ) )
    {
        NN_GFX_CALL_GL_FUNCTION( ::glEnable( GL_POLYGON_OFFSET_POINT ) );
        NN_GFX_CALL_GL_FUNCTION( ::glEnable( GL_POLYGON_OFFSET_LINE ) );
        NN_GFX_CALL_GL_FUNCTION( ::glEnable( GL_POLYGON_OFFSET_FILL ) );
        if( GLEW_EXT_polygon_offset_clamp )
        {
            NN_GFX_CALL_GL_FUNCTION( ::glPolygonOffsetClampEXT(
                param.slopeScaledDepthBias, param.depthBias, param.depthBiasClamp ) );
        }
        else
        {
            NN_GFX_CALL_GL_FUNCTION( ::glPolygonOffset( param.slopeScaledDepthBias, param.depthBias ) );
        }
    }
    else
    {
        NN_GFX_CALL_GL_FUNCTION( ::glDisable( GL_POLYGON_OFFSET_POINT ) );
        NN_GFX_CALL_GL_FUNCTION( ::glDisable( GL_POLYGON_OFFSET_LINE ) );
        NN_GFX_CALL_GL_FUNCTION( ::glDisable( GL_POLYGON_OFFSET_FILL ) );
    }

    NN_GFX_CALL_GL_FUNCTION( ::glEnable( GL_SAMPLE_MASK ) );
    NN_GFX_CALL_GL_FUNCTION( ::glSampleMaski( 0, param.sampleMask ) );

    NN_GFX_CALL_GL_FUNCTION( ::glEnable( GL_TEXTURE_CUBE_MAP_SEAMLESS ) );

    if( GLEW_NV_conservative_raster )
    {
        Gl::SetEnable( GL_CONSERVATIVE_RASTERIZATION_NV, param.flag.GetBit(
            RasterizerStateImplData< ApiVariationGl4 >::Flag_ConservativeRasterEnable ) );
    }

    NN_GFX_GL_ASSERT();
}

template<>
void GlCommandProc< BlendStateImplData< ApiVariationGl4 > >( const void* pParam, const GlCommandContext* ) NN_NOEXCEPT
{
    const BlendStateImplData< ApiVariationGl4 >& param =
        *static_cast< const BlendStateImplData< ApiVariationGl4 >* >( pParam );

    NN_GFX_CALL_GL_FUNCTION( Gl::LogicOp( param.logicOp ) );
    Gl::SetEnable( GL_SAMPLE_ALPHA_TO_COVERAGE, param.flag.GetBit(
        BlendStateImplData< ApiVariationGl4 >::Flag_AlphaToCoverageEnable ) );
    Gl::SetEnable( GL_COLOR_LOGIC_OP, param.flag.GetBit(
        BlendStateImplData< ApiVariationGl4 >::Flag_LogicOpEnable ) );
    NN_GFX_CALL_GL_FUNCTION( ::glBlendColor( param.blendColor[ 0 ],
        param.blendColor[ 1 ], param.blendColor[ 2 ], param.blendColor[ 3 ] ) );

    const BlendStateImplData< ApiVariationGl4 >::BlendTargetState* pTargets =
        reinterpret_cast< const BlendStateImplData< ApiVariationGl4 >::BlendTargetState* >( &param.pTargetArray );
    if( param.flag.GetBit( BlendStateImplData< ApiVariationGl4 >::Flag_IndependentBlendEnable ) )
    {
        for( int idxTarget = 0; idxTarget < param.blendTargetCount; ++idxTarget )
        {
            const BlendStateImplData< ApiVariationGl4 >::BlendTargetState& target = pTargets[ idxTarget ];
            NN_GFX_CALL_GL_FUNCTION( ::glColorMaski( idxTarget,
                ( target.colorMask & ChannelMask_Red ) ? GL_TRUE : GL_FALSE,
                ( target.colorMask & ChannelMask_Green ) ? GL_TRUE : GL_FALSE,
                ( target.colorMask & ChannelMask_Blue ) ? GL_TRUE : GL_FALSE,
                ( target.colorMask & ChannelMask_Alpha ) ? GL_TRUE : GL_FALSE ) );
            NN_GFX_CALL_GL_FUNCTION( ::glBlendFuncSeparatei( idxTarget,
                target.srcRGB, target.dstRGB, target.srcAlpha, target.dstAlpha ) );
            NN_GFX_CALL_GL_FUNCTION( ::glBlendEquationSeparatei( idxTarget,
                target.modeRGB, target.modeAlpha ) );
            if( target.flag.GetBit( BlendStateImplData< ApiVariationGl4 >::BlendTargetState::Flag_BlendEnable ) )
            {
                NN_GFX_CALL_GL_FUNCTION( ::glEnablei( GL_BLEND, idxTarget ) );
            }
            else
            {
                NN_GFX_CALL_GL_FUNCTION( ::glDisablei( GL_BLEND, idxTarget ) );
            }
        }
    }
    else
    {
        const BlendStateImplData< ApiVariationGl4 >::BlendTargetState& target = pTargets[ 0 ];
        NN_GFX_CALL_GL_FUNCTION( ::glColorMask(
            ( target.colorMask & ChannelMask_Red ) ? GL_TRUE : GL_FALSE,
            ( target.colorMask & ChannelMask_Green ) ? GL_TRUE : GL_FALSE,
            ( target.colorMask & ChannelMask_Blue ) ? GL_TRUE : GL_FALSE,
            ( target.colorMask & ChannelMask_Alpha ) ? GL_TRUE : GL_FALSE ) );
        NN_GFX_CALL_GL_FUNCTION( ::glBlendFuncSeparate(
            target.srcRGB, target.dstRGB, target.srcAlpha, target.dstAlpha ) );
        NN_GFX_CALL_GL_FUNCTION( ::glBlendEquationSeparate(
            target.modeRGB, target.modeAlpha ) );
        Gl::SetEnable( GL_BLEND, target.flag.GetBit(
            BlendStateImplData< ApiVariationGl4 >::BlendTargetState::Flag_BlendEnable ) );
    }

    NN_GFX_GL_ASSERT();
}

template<>
void GlCommandProc< DepthStencilStateImplData< ApiVariationGl4 > >( const void* pParam, const GlCommandContext* ) NN_NOEXCEPT
{
    const DepthStencilStateImplData< ApiVariationGl4 >& param =
        *static_cast< const DepthStencilStateImplData< ApiVariationGl4 >* >( pParam );

    NN_GFX_CALL_GL_FUNCTION( ::glDepthFunc( param.depthFunc ) );
    Gl::SetEnable( GL_DEPTH_TEST, param.flag.GetBit(
        DepthStencilStateImplData< ApiVariationGl4 >::Flag_DepthTestEnable ) );
    NN_GFX_CALL_GL_FUNCTION( ::glDepthMask( param.flag.GetBit(
        DepthStencilStateImplData< ApiVariationGl4 >::Flag_DepthWriteEnable ) ? GL_TRUE : GL_FALSE ) );
    Gl::SetEnable( GL_STENCIL_TEST, param.flag.GetBit(
        DepthStencilStateImplData< ApiVariationGl4 >::Flag_StencilTestEnable ) );

    if( GLEW_EXT_depth_bounds_test && param.flag.GetBit(
        DepthStencilStateImplData< ApiVariationGl4 >::Flag_DepthBoundsTestEnable ) )
    {
        NN_GFX_CALL_GL_FUNCTION( ::glEnable( GL_DEPTH_BOUNDS_TEST_EXT ) );
    }

    struct SetStencilStateFunctor
    {
        void operator ()( GLenum face, const DepthStencilStateImplData<
            ApiVariationGl4 >::StencilState& stencilState )
        {
            NN_GFX_CALL_GL_FUNCTION( ::glStencilOpSeparate( face,
                stencilState.sfail, stencilState.dpfail, stencilState.dppass ) );
            NN_GFX_CALL_GL_FUNCTION( ::glStencilFuncSeparate( face,
                stencilState.func, stencilState.ref, stencilState.mask ) );
        }
    } SetStencilState;

    SetStencilState( GL_FRONT, param.frontStencil );
    SetStencilState( GL_BACK, param.backStencil );

    NN_GFX_GL_ASSERT();
}

template<>
void GlCommandProc< VertexStateImplData< ApiVariationGl4 > >(
    const void* pParam, const GlCommandContext* pContext ) NN_NOEXCEPT
{
    const VertexStateImplData< ApiVariationGl4 >& param =
        *static_cast< const VertexStateImplData< ApiVariationGl4 >* >( pParam );

    NN_GFX_CALL_GL_FUNCTION( ::glBindVertexArray( pContext->hVao ) );

    nn::util::ConstBytePtr ptr( &param.pWorkMemory );
    const VertexStateImplData< ApiVariationGl4 >::AttributeState* pAttributes =
        ptr.Get< VertexStateImplData< ApiVariationGl4 >::AttributeState >();
    const uint32_t* pDivisors = ptr.Advance( sizeof( VertexStateImplData<
        ApiVariationGl4 >::AttributeState ) * param.attributeCount ).Get< uint32_t >();

    int maxAttribs = 0;
    NN_GFX_CALL_GL_FUNCTION( ::glGetIntegerv( GL_MAX_VERTEX_ATTRIBS, &maxAttribs ) );
    for( int idxAttribute = 0; idxAttribute < maxAttribs; ++idxAttribute )
    {
        NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glDisableVertexArrayAttrib )( pContext->hVao, idxAttribute ) );
    }

    for( int idxAttribute = 0; idxAttribute < param.attributeCount; ++idxAttribute )
    {
        const VertexStateImplData< ApiVariationGl4 >::AttributeState& attributeState = pAttributes[ idxAttribute ];
        if( attributeState.slot >= 0 )
        {
            NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glEnableVertexArrayAttrib )(
                pContext->hVao, attributeState.slot ) );

            // TODO: double 対応
            switch( attributeState.typeFormat )
            {
            case TypeFormat_Uint:
            case TypeFormat_Sint:
                {
                    if ( attributeState.type == GL_UNSIGNED_INT_2_10_10_10_REV ||
                        attributeState.type == GL_INT_2_10_10_10_REV )
                    {
                        NN_GFX_CALL_GL_FUNCTION( ::glVertexAttribFormat( attributeState.slot,
                            attributeState.size, attributeState.type, attributeState.normalized, attributeState.offset ) );
                    }
                    else
                    {
                        NN_GFX_CALL_GL_FUNCTION( ::glVertexAttribIFormat( attributeState.slot,
                            attributeState.size, attributeState.type, attributeState.offset ) );
                    }
                }
                break;
            default:
                {
                    NN_GFX_CALL_GL_FUNCTION( ::glVertexAttribFormat( attributeState.slot,
                        attributeState.size, attributeState.type, attributeState.normalized, attributeState.offset ) );
                }
                break;
            }

            NN_GFX_CALL_GL_FUNCTION( ::glVertexAttribBinding( attributeState.slot, attributeState.bindingIndex ) );

            NN_GFX_GL_ASSERT();
        }
    }

    for( int idxDivisor = 0; idxDivisor < param.bufferCount; ++idxDivisor )
    {
        NN_GFX_CALL_GL_FUNCTION( ::glVertexBindingDivisor( idxDivisor, pDivisors[ idxDivisor ] ) );
    }
    NN_GFX_GL_ASSERT();
}

template<>
void GlCommandProc< TessellationStateImplData< ApiVariationGl4 > >( const void* pParam, const GlCommandContext* ) NN_NOEXCEPT
{
    const TessellationStateImplData< ApiVariationGl4 >& param =
        *static_cast< const TessellationStateImplData< ApiVariationGl4 >* >( pParam );

    NN_GFX_CALL_GL_FUNCTION( ::glPatchParameteri( GL_PATCH_VERTICES, param.patchControlPointCount ) );

    NN_GFX_GL_ASSERT();
}

template<>
void GlCommandProc< ViewportScissorStateImplData< ApiVariationGl4 > >( const void* pParam, const GlCommandContext* ) NN_NOEXCEPT
{
    const ViewportScissorStateImplData< ApiVariationGl4 >& param =
        *static_cast< const ViewportScissorStateImplData< ApiVariationGl4 >* >( pParam );

    NN_GFX_CALL_GL_FUNCTION( ::glViewportIndexedfv( 0, param.viewport ) );
    NN_GFX_CALL_GL_FUNCTION( ::glDepthRangeIndexed( 0, param.depthRange[ 0 ], param.depthRange[ 1 ] ) );
    if( param.flag.GetBit( ViewportScissorStateImplData< ApiVariationGl4 >::Flag_ScissorEnable ) )
    {
        NN_GFX_CALL_GL_FUNCTION( ::glEnable( GL_SCISSOR_TEST ) );
        NN_GFX_CALL_GL_FUNCTION( ::glScissorIndexedv( 0, param.scissor ) );
    }
    else
    {
        NN_GFX_CALL_GL_FUNCTION( ::glDisable( GL_SCISSOR_TEST ) );
    }
    NN_GFX_GL_ASSERT();

    if( param.viewportCount > 1 )
    {
        nn::util::ConstBytePtr ptr( &param.pWorkMemory );
        int extraViewportCount = param.viewportCount - 1;
        const float* viewportArray = ptr.Get< float >();
        const double* depthRangeArray = ptr.Advance( sizeof( float ) * 4 * extraViewportCount ).Get< double >();
        const int32_t* scissorArray = ptr.Advance( sizeof( double ) * 2 * extraViewportCount ).Get< int32_t >();

        NN_GFX_CALL_GL_FUNCTION( ::glViewportArrayv( 1, extraViewportCount, viewportArray ) );
        NN_GFX_CALL_GL_FUNCTION( ::glDepthRangeArrayv( 1, extraViewportCount, depthRangeArray ) );

        if( param.flag.GetBit( ViewportScissorStateImplData< ApiVariationGl4 >::Flag_ScissorEnable ) )
        {
            NN_GFX_CALL_GL_FUNCTION( ::glScissorArrayv( 0, param.viewportCount, scissorArray ) );
        }
    }
    NN_GFX_GL_ASSERT();
}

template<>
void GlCommandProc< GlCallCommandListParam >( const void* pParam, const GlCommandContext* pContext ) NN_NOEXCEPT
{
    const GlCallCommandListParam& param = *static_cast< const GlCallCommandListParam* >( pParam );
    ExecuteGlCommandList( param.pCommandList, pContext );
}

template<>
void GlCommandProc< GlCallbackParam >( const void* pParam, const GlCommandContext* ) NN_NOEXCEPT
{
    const GlCallbackParam& param = *static_cast< const GlCallbackParam* >( pParam );
    return param.pCallback( param.pParam );
}

}
}
}
