﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/util/util_BytePtr.h>

#include <nn/gfx/gfx_DescriptorPoolInfo.h>
#include <nn/gfx/gfx_GpuAddress.h>
#include <nn/gfx/gfx_DescriptorSlot.h>

#include <nn/gfx/detail/gfx_Device-api.nvn.8.h>
#include <nn/gfx/detail/gfx_DescriptorPool-api.nvn.8.h>
#include <nn/gfx/detail/gfx_Buffer-api.nvn.8.h>
#include <nn/gfx/detail/gfx_Texture-api.nvn.8.h>
#include <nn/gfx/detail/gfx_Sampler-api.nvn.8.h>
#include <nn/gfx/detail/gfx_MemoryPool-api.nvn.8.h>

#include "gfx_NvnHelper.h"

namespace nn {
namespace gfx {
namespace detail {

namespace {

    static const size_t g_DescriptorSlotIncrementSize[] =
    {
        sizeof( NVNbufferAddress ) + sizeof( size_t ), // DescriptorPoolType_BufferView
        1, // DescriptorPoolType_TextureView: This is just an index so the slot increment is just 1
        1  // DescriptorPoolType_Sampler: This is just an index so the slot increment is just 1
    };

}

typedef ApiVariationNvn8 Target;

size_t DescriptorPoolImpl< Target >::CalculateDescriptorPoolSize(
    DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pDevice );
    NN_SDK_ASSERT( IsInitialized( *pDevice ) );

    switch ( info.GetDescriptorPoolType() )
    {
    case DescriptorPoolType_TextureView:
        {
            // Query the number of reserved texture descriptors
            int reservedTextures = 0;
            NN_GFX_CALL_NVN_FUNCTION(
                nvnDeviceGetInteger( pDevice->ToData()->pNvnDevice,
                    NVN_DEVICE_INFO_RESERVED_TEXTURE_DESCRIPTORS, &reservedTextures ) );
            NN_SDK_ASSERT( info.GetSlotCount() >= reservedTextures );
            NN_UNUSED( reservedTextures );

            // Query the texture descriptor size per slot
            int textureDescriptorSize = 0;
            NN_GFX_CALL_NVN_FUNCTION(
                nvnDeviceGetInteger( pDevice->ToData()->pNvnDevice,
                NVN_DEVICE_INFO_TEXTURE_DESCRIPTOR_SIZE, &textureDescriptorSize ) );

            return ( info.GetSlotCount() ) * textureDescriptorSize;
        }

    case DescriptorPoolType_Sampler:
        {
            // Query the number of reserved sampler descriptors
            int reservedSamplers = 0;
            NN_GFX_CALL_NVN_FUNCTION(
                nvnDeviceGetInteger( pDevice->ToData()->pNvnDevice,
                    NVN_DEVICE_INFO_RESERVED_SAMPLER_DESCRIPTORS, &reservedSamplers ) );
            NN_SDK_ASSERT( info.GetSlotCount() >= reservedSamplers );
            NN_UNUSED( reservedSamplers );

            // Query the sampler descriptor size per slot
            int samplerDescriptorSize = 0;
            NN_GFX_CALL_NVN_FUNCTION(
                nvnDeviceGetInteger( pDevice->ToData()->pNvnDevice,
                NVN_DEVICE_INFO_SAMPLER_DESCRIPTOR_SIZE, &samplerDescriptorSize ) );

            return ( info.GetSlotCount() ) * samplerDescriptorSize;
        }

    case DescriptorPoolType_BufferView:
        {
            return g_DescriptorSlotIncrementSize[ DescriptorPoolType_BufferView ] * info.GetSlotCount();
        }

    default: NN_UNEXPECTED_DEFAULT;
    }
}

size_t DescriptorPoolImpl< Target >::GetDescriptorPoolAlignment(
    DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    switch ( info.GetDescriptorPoolType() )
    {
    case DescriptorPoolType_TextureView:
        {
            // Query the texture descriptor size per slot
            int textureDescriptorSize = 0;
            NN_GFX_CALL_NVN_FUNCTION(
                nvnDeviceGetInteger( pDevice->ToData()->pNvnDevice,
                NVN_DEVICE_INFO_TEXTURE_DESCRIPTOR_SIZE, &textureDescriptorSize ) );

            // The documentation states that the alignment should be NVN_DEVICE_INFO_TEXTURE_DESCRIPTOR_SIZE
            return textureDescriptorSize;
        }

    case DescriptorPoolType_Sampler:
        {
            // Query the sampler descriptor size per slot
            int samplerDescriptorSize = 0;
            NN_GFX_CALL_NVN_FUNCTION(
                nvnDeviceGetInteger( pDevice->ToData()->pNvnDevice,
                NVN_DEVICE_INFO_SAMPLER_DESCRIPTOR_SIZE, &samplerDescriptorSize ) );

            // The documentation states that the alignment should be NVN_DEVICE_INFO_SAMPLER_DESCRIPTOR_SIZE
            return samplerDescriptorSize;
        }

    case DescriptorPoolType_BufferView:
        // DescriptorPoolType_BufferView's size isn't a power of 2 so we use uint32_t
        return sizeof( uint32_t );

    default: NN_UNEXPECTED_DEFAULT;
    }
}

ptrdiff_t DescriptorPoolImpl< Target >::GetDescriptorSlotIncrementSize(
    DeviceImpl< Target >* pDevice, DescriptorPoolType type ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pDevice );
    NN_SDK_ASSERT( IsInitialized( *pDevice ) );
    NN_SDK_ASSERT( type < DescriptorPoolType_End );
    NN_UNUSED( pDevice );

    return g_DescriptorSlotIncrementSize[ type ];
}

DescriptorPoolImpl< Target >::DescriptorPoolImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

DescriptorPoolImpl< Target >::~DescriptorPoolImpl() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
}

void DescriptorPoolImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info,
    MemoryPoolImpl< Target >* pMemoryPool, ptrdiff_t memoryPoolOffset, size_t memoryPoolSize ) NN_NOEXCEPT
{
    NN_STATIC_ASSERT( sizeof( this->nvnDescriptorPool ) >= sizeof( NVNtexturePool ) );
    NN_STATIC_ASSERT( sizeof( this->nvnDescriptorPool ) >= sizeof( NVNsamplerPool ) );

    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_SDK_REQUIRES( memoryPoolSize >= CalculateDescriptorPoolSize( pDevice, info ) );
    NN_SDK_REQUIRES_NOT_NULL( pMemoryPool );
    NN_SDK_REQUIRES( IsInitialized( *pMemoryPool ) );
    NN_SDK_ASSERT_ALIGNED( memoryPoolOffset, GetDescriptorPoolAlignment( pDevice, info ) );
    //NN_SDK_REQUIRES( pMemoryPool->ToData()->memorySize >= memoryPoolOffset + memoryPoolSize );
    NN_UNUSED( pDevice );
    NN_UNUSED( memoryPoolSize );

    NN_SDK_ASSERT( !this->pDescriptorPool );

    this->descriptorPoolType = static_cast< Bit8 >( info.GetDescriptorPoolType() );
    this->slotCount = static_cast< uint32_t >( info.GetSlotCount() );

    if ( this->descriptorPoolType == DescriptorPoolType_BufferView )
    {
        NN_SDK_ASSERT_NOT_NULL( pMemoryPool->ToData()->pMemory.ptr );
        this->pDescriptorPool = nn::util::BytePtr( pMemoryPool->ToData()->pMemory, memoryPoolOffset ).Get();
    }
    else
    {
        this->pDescriptorPool = this->nvnDescriptorPool;

        int reservedEntries = 0;
        int maxEntries = 0;
        if ( DescriptorPoolType_TextureView == this->descriptorPoolType )
        {
            NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger( pDevice->ToData()->pNvnDevice,
                NVN_DEVICE_INFO_RESERVED_TEXTURE_DESCRIPTORS, &reservedEntries ) );
            NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger( pDevice->ToData()->pNvnDevice,
                NVN_DEVICE_INFO_MAX_TEXTURE_POOL_SIZE, &maxEntries ) );
            NN_SDK_ASSERT_RANGE( info.GetSlotCount(), reservedEntries, maxEntries + 1 );
            NN_GFX_CALL_NVN_FUNCTION( nvnTexturePoolInitialize(
                static_cast< NVNtexturePool* >( this->pDescriptorPool.ptr ),
                pMemoryPool->ToData()->pNvnMemoryPool, memoryPoolOffset, info.GetSlotCount() ) );
        }
        else
        {
            NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger( pDevice->ToData()->pNvnDevice,
                NVN_DEVICE_INFO_RESERVED_SAMPLER_DESCRIPTORS, &reservedEntries ) );
            NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger( pDevice->ToData()->pNvnDevice,
                NVN_DEVICE_INFO_MAX_SAMPLER_POOL_SIZE, &maxEntries ) );
            NN_SDK_ASSERT_RANGE( info.GetSlotCount(), reservedEntries, maxEntries + 1 );
            NN_GFX_CALL_NVN_FUNCTION( nvnSamplerPoolInitialize(
                static_cast< NVNsamplerPool* >( this->pDescriptorPool.ptr ),
                pMemoryPool->ToData()->pNvnMemoryPool, memoryPoolOffset, info.GetSlotCount() ) );
        }
        NN_UNUSED( maxEntries );

        this->reservedSlots = reservedEntries;

        NN_SDK_ASSERT( this->slotCount >= this->reservedSlots );
    }

    this->flags.SetBit( Flag_Shared, false );
    this->state = State_Initialized;
}

void DescriptorPoolImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_ASSERT( !this->flags.GetBit( Flag_Shared ) );
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_UNUSED( pDevice );

    if( this->descriptorPoolType == DescriptorPoolType_TextureView )
    {
        NN_GFX_CALL_NVN_FUNCTION( nvnTexturePoolFinalize( this->pDescriptorPool ) );
    }
    else if( this->descriptorPoolType == DescriptorPoolType_Sampler )
    {
        NN_GFX_CALL_NVN_FUNCTION( nvnSamplerPoolFinalize( this->pDescriptorPool ) );
    }

    this->pDescriptorPool = NULL;

    this->state = State_NotInitialized;
}

void DescriptorPoolImpl< Target >::BeginUpdate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    this->state = State_Begun;
}

void DescriptorPoolImpl< Target >::EndUpdate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    this->state = State_Initialized;
}

void DescriptorPoolImpl< Target >::SetBufferView( int indexSlot,
    const GpuAddress& gpuAddress, size_t size ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES_RANGE( indexSlot, 0, static_cast< int >( this->slotCount ) );
    NN_SDK_REQUIRES( this->descriptorPoolType == DescriptorPoolType_BufferView );
    NN_SDK_ASSERT_NOT_EQUAL( Nvn::GetBufferAddress( gpuAddress ), 0 );
    nn::util::BytePtr pSlot( this->pDescriptorPool,
        g_DescriptorSlotIncrementSize[ descriptorPoolType ] * indexSlot );

    *pSlot.Get< NVNbufferAddress >() = Nvn::GetBufferAddress( gpuAddress );
    *pSlot.Advance( sizeof( NVNbufferAddress ) ).Get< size_t >() = size;
}

void DescriptorPoolImpl< Target >::SetSampler( int indexSlot,
    const SamplerImpl< Target >* pSampler ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES_RANGE( indexSlot, this->reservedSlots, static_cast< int >( this->slotCount ) );
    NN_SDK_REQUIRES( this->descriptorPoolType == DescriptorPoolType_Sampler );
    NN_SDK_REQUIRES_NOT_NULL( pSampler );
    NN_SDK_REQUIRES( IsInitialized( *pSampler ) );
    NN_SDK_ASSERT_NOT_NULL( pSampler->ToData()->pNvnSampler.ptr );

    NN_GFX_CALL_NVN_FUNCTION( nvnSamplerPoolRegisterSampler(
        this->pDescriptorPool, indexSlot, pSampler->ToData()->pNvnSampler ) );
}

void DescriptorPoolImpl< Target >::SetTextureView( int indexSlot,
    const TextureViewImpl< Target >* pTextureView ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES_RANGE( indexSlot, this->reservedSlots, static_cast< int >( this->slotCount ) );
    NN_SDK_REQUIRES( this->descriptorPoolType == DescriptorPoolType_TextureView );
    NN_SDK_REQUIRES_NOT_NULL( pTextureView );
    NN_SDK_REQUIRES( IsInitialized( *pTextureView ) );
    NN_SDK_ASSERT_NOT_NULL( pTextureView->ToData()->pNvnTexture.ptr );

    NN_GFX_CALL_NVN_FUNCTION( nvnTexturePoolRegisterTexture( this->pDescriptorPool,
        indexSlot, pTextureView->ToData()->pNvnTexture, pTextureView->ToData()->pNvnTextureView ) );
}

void DescriptorPoolImpl< Target >::SetImage( int indexSlot,
    const TextureViewImpl< Target >* pImage ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES_RANGE( indexSlot, this->reservedSlots, static_cast< int >( this->slotCount ) );
    NN_SDK_REQUIRES( this->descriptorPoolType == DescriptorPoolType_TextureView );
    NN_SDK_REQUIRES_NOT_NULL( pImage );
    NN_SDK_REQUIRES( IsInitialized( *pImage ) );
    NN_SDK_ASSERT_NOT_NULL( pImage->ToData()->pNvnTexture.ptr );

    NN_GFX_CALL_NVN_FUNCTION( nvnTexturePoolRegisterImage( this->pDescriptorPool,
        indexSlot, pImage->ToData()->pNvnTexture, pImage->ToData()->pNvnTextureView ) );
}

void DescriptorPoolImpl< Target >::SetBufferTextureView( int indexSlot,
    const BufferTextureViewImpl< Target >* pBufferTexture ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_Begun );
    NN_SDK_REQUIRES_RANGE( indexSlot, this->reservedSlots, static_cast<int>( this->slotCount ) );
    NN_SDK_REQUIRES( this->descriptorPoolType == DescriptorPoolType_TextureView );
    NN_SDK_REQUIRES_NOT_NULL( pBufferTexture );
    NN_SDK_REQUIRES( IsInitialized( *pBufferTexture ) );
    NN_SDK_ASSERT_NOT_NULL( pBufferTexture->ToData()->pNvnTexture.ptr );

    NN_GFX_CALL_NVN_FUNCTION( nvnTexturePoolRegisterTexture( this->pDescriptorPool,
        indexSlot, pBufferTexture->ToData()->pNvnTexture, NULL ) );
}

void DescriptorPoolImpl< Target >::SetBufferImage( int indexSlot,
    const BufferTextureViewImpl< Target >* pBufferImage ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_Begun );
    NN_SDK_REQUIRES_RANGE( indexSlot, this->reservedSlots, static_cast<int>( this->slotCount ) );
    NN_SDK_REQUIRES( this->descriptorPoolType == DescriptorPoolType_TextureView );
    NN_SDK_REQUIRES_NOT_NULL( pBufferImage );
    NN_SDK_REQUIRES( IsInitialized( *pBufferImage ) );
    NN_SDK_ASSERT_NOT_NULL( pBufferImage->ToData()->pNvnTexture.ptr );

    NN_GFX_CALL_NVN_FUNCTION( nvnTexturePoolRegisterImage( this->pDescriptorPool,
        indexSlot, pBufferImage->ToData()->pNvnTexture, NULL ) );
}

void DescriptorPoolImpl< Target >::GetDescriptorSlot(
    DescriptorSlot* pOutDescriptorSlot, int indexSlot ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state != State_NotInitialized );
    NN_SDK_REQUIRES_RANGE( indexSlot, this->reservedSlots, static_cast< int >( this->slotCount ) );

    switch ( this->descriptorPoolType )
    {
    case DescriptorPoolType_Sampler:
    case DescriptorPoolType_TextureView:
        {
            pOutDescriptorSlot->ToData()->value = indexSlot;
        }
        break;

    case DescriptorPoolType_BufferView:
        {
            pOutDescriptorSlot->ToData()->value = reinterpret_cast< uintptr_t >(
                nn::util::BytePtr( this->pDescriptorPool.ptr, indexSlot *
                g_DescriptorSlotIncrementSize[ DescriptorPoolType_BufferView ] ).Get() );
        }
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }
}

int DescriptorPoolImpl< Target >::GetDescriptorSlotIndex(
    const DescriptorSlot& descriptorSlot ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state != State_NotInitialized );
    NN_SDK_REQUIRES( descriptorSlot.IsValid() );

    if( this->descriptorPoolType == DescriptorPoolType_BufferView )
    {
        return static_cast< int >( nn::util::ConstBytePtr( this->pDescriptorPool.ptr ).Distance(
            reinterpret_cast< const void* >( static_cast< uintptr_t >( descriptorSlot.ToData()->value ) ) ) ) /
            static_cast< int >( g_DescriptorSlotIncrementSize[ DescriptorPoolType_BufferView ] );
    }

    return static_cast< int >( descriptorSlot.ToData()->value );
}

}
}
}
