﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_BytePtr.h>

#include <nn/gfx/gfx_DescriptorPoolInfo.h>
#include <nn/gfx/gfx_GpuAddress.h>
#include <nn/gfx/gfx_DescriptorSlot.h>

#include <nn/gfx/detail/gfx_DescriptorPool-api.gx.2.h>
#include <nn/gfx/detail/gfx_Sampler-api.gx.2.h>
#include <nn/gfx/detail/gfx_Texture-api.gx.2.h>
#include <nn/gfx/detail/gfx_MemoryPool-api.gx.2.h>

#include "gfx_GxHelper.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationGx2 Target;

namespace {

static const size_t g_DescriptorSlotIncrementSize[] =
{
    sizeof( void* ) + sizeof( uint32_t ),
    sizeof( void* ),
    sizeof( void* )
};

}

size_t DescriptorPoolImpl< Target >::CalculateDescriptorPoolSize(
    DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    return GetDescriptorSlotIncrementSize( pDevice, info.GetDescriptorPoolType() ) * info.GetSlotCount();
}

size_t DescriptorPoolImpl< Target >::GetDescriptorSlotIncrementSize(
    DeviceImpl< Target >*, DescriptorPoolType type ) NN_NOEXCEPT
{
    return g_DescriptorSlotIncrementSize[ type ];
};

size_t DescriptorPoolImpl< Target >::GetDescriptorPoolAlignment( DeviceImpl< Target >*, const InfoType& ) NN_NOEXCEPT
{
    // All of the data in the descriptor slot types only require 32bit alignment
    return sizeof( uint32_t );
}

DescriptorPoolImpl< Target >::DescriptorPoolImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

DescriptorPoolImpl< Target >::~DescriptorPoolImpl() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
}

void DescriptorPoolImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info,
    MemoryPoolImpl< Target >* pMemoryPool, ptrdiff_t memoryPoolOffset, size_t memoryPoolSize ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pDevice );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_SDK_REQUIRES( pMemoryPool );
    NN_SDK_ASSERT( pMemoryPool->ToData()->pMemory );
    NN_SDK_REQUIRES( pMemoryPool->ToData()->memorySize >= memoryPoolOffset + memoryPoolSize );
    NN_SDK_REQUIRES( memoryPoolSize >= CalculateDescriptorPoolSize( pDevice, info ) );
    NN_UNUSED( pDevice );
    NN_UNUSED( memoryPoolSize );

    NN_SDK_ASSERT( !this->pMemory );

    this->descriptorPoolType = static_cast< Bit8 >( info.GetDescriptorPoolType() );
    this->slotCount = static_cast< uint32_t >( info.GetSlotCount() );
    this->pMemory = nn::util::BytePtr( pMemoryPool->ToData()->pMemory, memoryPoolOffset ).Get();

    this->state = State_Initialized;
}

void DescriptorPoolImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_REQUIRES( pDevice );
    NN_UNUSED( pDevice );

    this->pMemory = NULL;

    this->state = State_NotInitialized;
}

void DescriptorPoolImpl< Target >::BeginUpdate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    this->state = State_Begun;
}

void DescriptorPoolImpl< Target >::EndUpdate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    this->state = State_Initialized;
}

void DescriptorPoolImpl< Target >::SetBufferView( int indexSlot,
    const GpuAddress& gpuAddress, size_t size ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES( indexSlot >= 0 && indexSlot < static_cast< int >( this->slotCount ) );
    NN_SDK_REQUIRES( this->descriptorPoolType == DescriptorPoolType_BufferView );

    nn::util::BytePtr pSlot( this->pMemory, g_DescriptorSlotIncrementSize[
        DescriptorPoolType_BufferView ] * indexSlot );
    *pSlot.Get< void* >() = reinterpret_cast< void* >( static_cast< uintptr_t >( gpuAddress.ToData()->value ) );
    *pSlot.Advance( sizeof( void* ) ).Get< uint32_t >() = static_cast< uint32_t >( size );
}

void DescriptorPoolImpl< Target >::SetSampler( int indexSlot,
    const SamplerImpl< Target >* pSampler ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES( indexSlot >= 0 && indexSlot < static_cast< int >( this->slotCount ) );
    NN_SDK_REQUIRES( this->descriptorPoolType == DescriptorPoolType_Sampler );
    NN_SDK_REQUIRES( pSampler );
    NN_SDK_REQUIRES( IsInitialized( *pSampler ) );
    static_cast< const void** >( this->pMemory )[ indexSlot ] = &pSampler->ToData()->gx2Sampler;
}

void DescriptorPoolImpl< Target >::SetTextureView( int indexSlot,
    const TextureViewImpl< Target >* pTextureView ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES( indexSlot >= 0 && indexSlot < static_cast< int >( this->slotCount ) );
    NN_SDK_REQUIRES( this->descriptorPoolType == DescriptorPoolType_TextureView );
    NN_SDK_REQUIRES( pTextureView );
    NN_SDK_REQUIRES( IsInitialized( *pTextureView ) );
    static_cast< const void** >( this->pMemory )[ indexSlot ] = &pTextureView->ToData()->gx2Texture;
}

void DescriptorPoolImpl< Target >::SetImage( int, const TextureViewImpl< Target >* ) NN_NOEXCEPT
{
    // TODO
}

void DescriptorPoolImpl< Target >::SetBufferTextureView( int,
    const BufferTextureViewImpl< Target >* ) NN_NOEXCEPT
{
    // TODO
}

void DescriptorPoolImpl< Target >::SetBufferImage( int,
    const BufferTextureViewImpl< Target >* ) NN_NOEXCEPT
{
    // TODO
}

void DescriptorPoolImpl< Target >::GetDescriptorSlot(
    DescriptorSlot* pOutDescriptorSlot, int indexSlot ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state != State_NotInitialized );
    NN_SDK_REQUIRES( indexSlot >= 0 && indexSlot < static_cast< int >( this->slotCount ) );

    pOutDescriptorSlot->ToData()->value = reinterpret_cast< uintptr_t >( nn::util::BytePtr( this->pMemory.ptr,
        indexSlot * g_DescriptorSlotIncrementSize[ this->descriptorPoolType ] ).Get() );
}

int DescriptorPoolImpl< Target >::GetDescriptorSlotIndex(
    const DescriptorSlot& descriptorSlot ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state != State_NotInitialized );

    return static_cast< int >( nn::util::ConstBytePtr( this->pMemory.ptr ).Distance(
        reinterpret_cast< const void* >( static_cast< uintptr_t >( descriptorSlot.ToData()->value ) ) ) ) /
        static_cast< int >( g_DescriptorSlotIncrementSize[ this->descriptorPoolType ] );
}

}
}
}
