﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <memory>
#include <unordered_map>

#include <nn/nn_SdkLog.h>
#include <nn/util/util_BytePtr.h>

#include <nn/gfx/gfx_Common.h>

#include <nn/gfx/detail/gfx_Log.h>
#include <nn/gfx/detail/gfx_Device-api.d3d.11.h>
#include <nn/gfx/detail/gfx_Buffer-api.d3d.11.h>

#include "gfx_D3dHelper.h"
#include "gfx_CommonHelper.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationD3d11 Target;

namespace {

const struct ImageFormatAndD3dFormat
{
    ImageFormat imageFormat;
    D3dTextureFormat d3dFormat;
} D3dTextureFormatList[] =
{
    { ImageFormat_R8_Unorm, { DXGI_FORMAT_R8_UNORM } },
    { ImageFormat_R8_Snorm, { DXGI_FORMAT_R8_SNORM } },
    { ImageFormat_R8_Uint, { DXGI_FORMAT_R8_UINT } },
    { ImageFormat_R8_Sint, { DXGI_FORMAT_R8_SINT } },
    //{ ImageFormat_R4_G4_B4_A4_Unorm, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_A4_B4_G4_R4_Unorm, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_R5_G5_B5_A1_Unorm, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_A1_B5_G5_R5_Unorm, { DXGI_FORMAT_UNKNOWN } },
    { ImageFormat_R5_G6_B5_Unorm, { DXGI_FORMAT_B5G6R5_UNORM } },
    //{ ImageFormat_B5_G6_R5_Unorm, { DXGI_FORMAT_UNKNOWN } },
    { ImageFormat_R8_G8_Unorm, { DXGI_FORMAT_R8G8_UNORM } },
    { ImageFormat_R8_G8_Snorm, { DXGI_FORMAT_R8G8_SNORM } },
    { ImageFormat_R8_G8_Uint, { DXGI_FORMAT_R8G8_UINT } },
    { ImageFormat_R8_G8_Sint, { DXGI_FORMAT_R8G8_SINT } },
    { ImageFormat_R16_Unorm, { DXGI_FORMAT_R16_UNORM } },
    { ImageFormat_R16_Snorm, { DXGI_FORMAT_R16_SNORM } },
    { ImageFormat_R16_Uint, { DXGI_FORMAT_R16_UINT } },
    { ImageFormat_R16_Sint, { DXGI_FORMAT_R16_SINT } },
    { ImageFormat_R16_Float, { DXGI_FORMAT_R16_FLOAT } },
    { ImageFormat_D16_Unorm, { DXGI_FORMAT_D16_UNORM } },
    { ImageFormat_R8_G8_B8_A8_Unorm, { DXGI_FORMAT_R8G8B8A8_UNORM } },
    { ImageFormat_R8_G8_B8_A8_Snorm, { DXGI_FORMAT_R8G8B8A8_SNORM } },
    { ImageFormat_R8_G8_B8_A8_Uint, { DXGI_FORMAT_R8G8B8A8_UINT } },
    { ImageFormat_R8_G8_B8_A8_Sint, { DXGI_FORMAT_R8G8B8A8_SINT } },
    { ImageFormat_R8_G8_B8_A8_UnormSrgb, { DXGI_FORMAT_R8G8B8A8_UNORM_SRGB } },
    { ImageFormat_B8_G8_R8_A8_Unorm, { DXGI_FORMAT_B8G8R8A8_UNORM } },
    //{ ImageFormat_B8_G8_R8_A8_Snorm, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_B8_G8_R8_A8_Uint, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_B8_G8_R8_A8_Sint, { DXGI_FORMAT_UNKNOWN } },
    { ImageFormat_B8_G8_R8_A8_UnormSrgb, { DXGI_FORMAT_B8G8R8A8_UNORM_SRGB } },
    { ImageFormat_R9_G9_B9_E5_SharedExp, { DXGI_FORMAT_R9G9B9E5_SHAREDEXP } },
    { ImageFormat_R10_G10_B10_A2_Unorm, { DXGI_FORMAT_R10G10B10A2_UNORM } },
    { ImageFormat_R10_G10_B10_A2_Uint, { DXGI_FORMAT_R10G10B10A2_UINT } },
    //{ ImageFormat_R11_G11_B10_Float, { DXGI_FORMAT_UNKNOWN } },
    { ImageFormat_B10_G11_R11_Float, { DXGI_FORMAT_R11G11B10_FLOAT } },
    { ImageFormat_R16_G16_Unorm, { DXGI_FORMAT_R16G16_UNORM } },
    { ImageFormat_R16_G16_Snorm, { DXGI_FORMAT_R16G16_SNORM } },
    { ImageFormat_R16_G16_Uint, { DXGI_FORMAT_R16G16_UINT } },
    { ImageFormat_R16_G16_Sint, { DXGI_FORMAT_R16G16_SINT } },
    { ImageFormat_R16_G16_Float, { DXGI_FORMAT_R16G16_FLOAT } },
    { ImageFormat_D24_Unorm_S8_Uint, { DXGI_FORMAT_D24_UNORM_S8_UINT } },
    { ImageFormat_R32_Uint, { DXGI_FORMAT_R32_UINT } },
    { ImageFormat_R32_Sint, { DXGI_FORMAT_R32_SINT } },
    { ImageFormat_R32_Float, { DXGI_FORMAT_R32_FLOAT } },
    { ImageFormat_D32_Float, { DXGI_FORMAT_D32_FLOAT } },
    { ImageFormat_R16_G16_B16_A16_Unorm, { DXGI_FORMAT_R16G16B16A16_UNORM } },
    { ImageFormat_R16_G16_B16_A16_Snorm, { DXGI_FORMAT_R16G16B16A16_SNORM } },
    { ImageFormat_R16_G16_B16_A16_Uint, { DXGI_FORMAT_R16G16B16A16_UINT } },
    { ImageFormat_R16_G16_B16_A16_Sint, { DXGI_FORMAT_R16G16B16A16_SINT } },
    { ImageFormat_R16_G16_B16_A16_Float, { DXGI_FORMAT_R16G16B16A16_FLOAT } },
    { ImageFormat_D32_Float_S8_Uint_X24, { DXGI_FORMAT_D32_FLOAT_S8X24_UINT } },
    { ImageFormat_R32_G32_Uint, { DXGI_FORMAT_R32G32_UINT } },
    { ImageFormat_R32_G32_Sint, { DXGI_FORMAT_R32G32_SINT } },
    { ImageFormat_R32_G32_Float, { DXGI_FORMAT_R32G32_FLOAT } },
    { ImageFormat_R32_G32_B32_Uint, { DXGI_FORMAT_R32G32B32_UINT } },
    { ImageFormat_R32_G32_B32_Sint, { DXGI_FORMAT_R32G32B32_SINT } },
    { ImageFormat_R32_G32_B32_Float, { DXGI_FORMAT_R32G32B32_FLOAT } },
    { ImageFormat_R32_G32_B32_A32_Uint, { DXGI_FORMAT_R32G32B32A32_UINT } },
    { ImageFormat_R32_G32_B32_A32_Sint, { DXGI_FORMAT_R32G32B32A32_SINT } },
    { ImageFormat_R32_G32_B32_A32_Float, { DXGI_FORMAT_R32G32B32A32_FLOAT } },
    { ImageFormat_Bc1_Unorm, { DXGI_FORMAT_BC1_UNORM } },
    { ImageFormat_Bc1_UnormSrgb, { DXGI_FORMAT_BC1_UNORM_SRGB } },
    { ImageFormat_Bc2_Unorm, { DXGI_FORMAT_BC2_UNORM } },
    { ImageFormat_Bc2_UnormSrgb, { DXGI_FORMAT_BC2_UNORM_SRGB } },
    { ImageFormat_Bc3_Unorm, { DXGI_FORMAT_BC3_UNORM } },
    { ImageFormat_Bc3_UnormSrgb, { DXGI_FORMAT_BC3_UNORM_SRGB } },
    { ImageFormat_Bc4_Unorm, { DXGI_FORMAT_BC4_UNORM } },
    { ImageFormat_Bc4_Snorm, { DXGI_FORMAT_BC4_SNORM } },
    { ImageFormat_Bc5_Unorm, { DXGI_FORMAT_BC5_UNORM } },
    { ImageFormat_Bc5_Snorm, { DXGI_FORMAT_BC5_SNORM } },
    { ImageFormat_Bc6_Float, { DXGI_FORMAT_BC6H_SF16 } },
    { ImageFormat_Bc6_Ufloat, { DXGI_FORMAT_BC6H_UF16 } },
    { ImageFormat_Bc7_Unorm, { DXGI_FORMAT_BC7_UNORM } },
    { ImageFormat_Bc7_UnormSrgb, { DXGI_FORMAT_BC7_UNORM_SRGB } }
    //{ ImageFormat_Eac_R11_Unorm, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Eac_R11_Snorm, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Eac_R11_G11_Unorm, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Eac_R11_G11_Snorm, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Etc1_Unorm, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Etc2_Unorm, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Etc2_UnormSrgb, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Etc2_Mask_Unorm, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Etc2_Mask_UnormSrgb, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Etc2_Alpha_Unorm, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Etc2_Alpha_UnormSrgb, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Pvrtc1_2bpp_Unorm, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Pvrtc1_2bpp_UnormSrgb, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Pvrtc1_4bpp_Unorm, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Pvrtc1_4bpp_UnormSrgb, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Pvrtc1_Alpha_2bpp_Unorm, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Pvrtc1_Alpha_2bpp_UnormSrgb, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Pvrtc1_Alpha_4bpp_Unorm, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Pvrtc1_Alpha_4bpp_UnormSrgb, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Pvrtc2_Alpha_2bpp_Unorm, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Pvrtc2_Alpha_2bpp_UnormSrgb, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Pvrtc2_Alpha_4bpp_Unorm, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Pvrtc2_Alpha_4bpp_UnormSrgb, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Astc_4x4_Unorm, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Astc_4x4_UnormSrgb, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Astc_5x4_Unorm, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Astc_5x4_UnormSrgb, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Astc_5x5_Unorm, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Astc_5x5_UnormSrgb, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Astc_6x5_Unorm, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Astc_6x5_UnormSrgb, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Astc_6x6_Unorm, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Astc_6x6_UnormSrgb, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Astc_8x5_Unorm, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Astc_8x5_UnormSrgb, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Astc_8x6_Unorm, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Astc_8x6_UnormSrgb, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Astc_8x8_Unorm, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Astc_8x8_UnormSrgb, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Astc_10x5_Unorm, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Astc_10x5_UnormSrgb, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Astc_10x6_Unorm, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Astc_10x6_UnormSrgb, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Astc_10x8_Unorm, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Astc_10x8_UnormSrgb, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Astc_10x10_Unorm, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Astc_10x10_UnormSrgb, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Astc_12x10_Unorm, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Astc_12x10_UnormSrgb, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Astc_12x12_Unorm, { DXGI_FORMAT_UNKNOWN } },
    //{ ImageFormat_Astc_12x12_UnormSrgb, { DXGI_FORMAT_UNKNOWN } },

    //{ ImageFormat_B5_G5_R5_A1_Unorm, { DXGI_FORMAT_UNKNOWN } }
};

const struct AttributeFormatAndD3dFormat
{
    AttributeFormat attributeFormat;
    D3dAttributeFormat d3dFormat;
} D3dAttributeFormatList[] =
{
    //{ AttributeFormat_Undefined, { DXGI_FORMAT_UNKNOWN } },
    { AttributeFormat_4_4_Unorm, { DXGI_FORMAT_AI44 } },
    { AttributeFormat_8_Unorm, { DXGI_FORMAT_R8_UNORM } },
    { AttributeFormat_8_Snorm, { DXGI_FORMAT_R8_SNORM } },
    { AttributeFormat_8_Uint, { DXGI_FORMAT_R8_UINT } },
    { AttributeFormat_8_Sint, { DXGI_FORMAT_R8_SINT } },
    //{ AttributeFormat_8_UintToFloat, { DXGI_FORMAT_UNKNOWN } },
    //{ AttributeFormat_8_SintToFloat, { DXGI_FORMAT_UNKNOWN } },
    { AttributeFormat_8_8_Unorm, { DXGI_FORMAT_R8G8_UNORM } },
    { AttributeFormat_8_8_Snorm, { DXGI_FORMAT_R8G8_SNORM } },
    { AttributeFormat_8_8_Uint, { DXGI_FORMAT_R8G8_UINT } },
    { AttributeFormat_8_8_Sint, { DXGI_FORMAT_R8G8_SINT } },
    //{ AttributeFormat_8_8_UintToFloat, { DXGI_FORMAT_UNKNOWN } },
    //{ AttributeFormat_8_8_SintToFloat, { DXGI_FORMAT_UNKNOWN } },
    { AttributeFormat_16_Unorm, { DXGI_FORMAT_R16_UNORM } },
    { AttributeFormat_16_Snorm, { DXGI_FORMAT_R16_SNORM } },
    { AttributeFormat_16_Uint, { DXGI_FORMAT_R16_UINT } },
    { AttributeFormat_16_Sint, { DXGI_FORMAT_R16_SINT } },
    { AttributeFormat_16_Float, { DXGI_FORMAT_R16_FLOAT } },
    //{ AttributeFormat_16_UintToFloat, { DXGI_FORMAT_UNKNOWN } },
    //{ AttributeFormat_16_SintToFloat, { DXGI_FORMAT_UNKNOWN } },
    { AttributeFormat_8_8_8_8_Unorm, { DXGI_FORMAT_R8G8B8A8_UNORM } },
    { AttributeFormat_8_8_8_8_Snorm, { DXGI_FORMAT_R8G8B8A8_SNORM } },
    { AttributeFormat_8_8_8_8_Uint, { DXGI_FORMAT_R8G8B8A8_UINT } },
    { AttributeFormat_8_8_8_8_Sint, { DXGI_FORMAT_R8G8B8A8_SINT } },
    //{ AttributeFormat_8_8_8_8_UintToFloat, { DXGI_FORMAT_UNKNOWN } },
    //{ AttributeFormat_8_8_8_8_SintToFloat, { DXGI_FORMAT_UNKNOWN } },
    { AttributeFormat_10_10_10_2_Unorm, { DXGI_FORMAT_R10G10B10A2_UNORM } },
    //{ AttributeFormat_10_10_10_2_Snorm, { DXGI_FORMAT_UNKNOWN } },
    { AttributeFormat_10_10_10_2_Uint, { DXGI_FORMAT_R10G10B10A2_UINT } },
    //{ AttributeFormat_10_10_10_2_Sint, { DXGI_FORMAT_UNKNOWN } },
    { AttributeFormat_16_16_Unorm, { DXGI_FORMAT_R16G16_UNORM } },
    { AttributeFormat_16_16_Snorm, { DXGI_FORMAT_R16G16_SNORM } },
    { AttributeFormat_16_16_Uint, { DXGI_FORMAT_R16G16_UINT } },
    { AttributeFormat_16_16_Sint, { DXGI_FORMAT_R16G16_SINT } },
    { AttributeFormat_16_16_Float, { DXGI_FORMAT_R16G16_FLOAT } },
    //{ AttributeFormat_16_16_UintToFloat, { DXGI_FORMAT_UNKNOWN } },
    //{ AttributeFormat_16_16_SintToFloat, { DXGI_FORMAT_UNKNOWN } },
    { AttributeFormat_32_Uint, { DXGI_FORMAT_R32_UINT } },
    { AttributeFormat_32_Sint, { DXGI_FORMAT_R32_SINT } },
    { AttributeFormat_32_Float, { DXGI_FORMAT_R32_FLOAT } },
    { AttributeFormat_16_16_16_16_Unorm, { DXGI_FORMAT_R16G16B16A16_UNORM } },
    { AttributeFormat_16_16_16_16_Snorm, { DXGI_FORMAT_R16G16B16A16_SNORM } },
    { AttributeFormat_16_16_16_16_Uint, { DXGI_FORMAT_R16G16B16A16_UINT } },
    { AttributeFormat_16_16_16_16_Sint, { DXGI_FORMAT_R16G16B16A16_SINT } },
    { AttributeFormat_16_16_16_16_Float, { DXGI_FORMAT_R16G16B16A16_FLOAT } },
    //{ AttributeFormat_16_16_16_16_UintToFloat, { DXGI_FORMAT_UNKNOWN } },
    //{ AttributeFormat_16_16_16_16_SintToFloat, { DXGI_FORMAT_UNKNOWN } },
    { AttributeFormat_32_32_Uint, { DXGI_FORMAT_R32G32_UINT } },
    { AttributeFormat_32_32_Sint, { DXGI_FORMAT_R32G32_SINT } },
    { AttributeFormat_32_32_Float, { DXGI_FORMAT_R32G32_FLOAT } },
    { AttributeFormat_32_32_32_Uint, { DXGI_FORMAT_R32G32B32_UINT } },
    { AttributeFormat_32_32_32_Sint, { DXGI_FORMAT_R32G32B32_SINT } },
    { AttributeFormat_32_32_32_Float, { DXGI_FORMAT_R32G32B32_FLOAT } },
    { AttributeFormat_32_32_32_32_Uint, { DXGI_FORMAT_R32G32B32A32_UINT } },
    { AttributeFormat_32_32_32_32_Sint, { DXGI_FORMAT_R32G32B32A32_SINT } },
    { AttributeFormat_32_32_32_32_Float, { DXGI_FORMAT_R32G32B32A32_FLOAT } }
};

const struct FilterModeAndD3dFilterMode
{
    FilterMode filterMode;
    D3D11_FILTER d3dFilterMode;
} D3dFilterModeList[] =
{
    { FilterMode_MinPoint_MagPoint_MipPoint, D3D11_FILTER_MIN_MAG_MIP_POINT },
    { FilterMode_MinPoint_MagLinear_MipLinear, D3D11_FILTER_MIN_POINT_MAG_MIP_LINEAR },
    { FilterMode_MinLinear_MagPoint_MipPoint, D3D11_FILTER_MIN_LINEAR_MAG_MIP_POINT },
    { FilterMode_MinLinear_MagPoint_MipLinear, D3D11_FILTER_MIN_LINEAR_MAG_POINT_MIP_LINEAR },
    { FilterMode_MinLinear_MagLinear_MipPoint, D3D11_FILTER_MIN_MAG_LINEAR_MIP_POINT },
    { FilterMode_MinLinear_MagLinear_MipLinear, D3D11_FILTER_MIN_MAG_MIP_LINEAR },
    { FilterMode_Anisotropic, D3D11_FILTER_ANISOTROPIC },
    { FilterMode_Comparison_MinPoint_MagPoint_MipPoint, D3D11_FILTER_COMPARISON_MIN_MAG_MIP_POINT },
    { FilterMode_Comparison_MinPoint_MagPoint_MipLinear, D3D11_FILTER_COMPARISON_MIN_MAG_POINT_MIP_LINEAR },
    { FilterMode_Comparison_MinPoint_MagLinear_MipPoint, D3D11_FILTER_COMPARISON_MIN_POINT_MAG_LINEAR_MIP_POINT },
    { FilterMode_Comparison_MinPoint_MagLinear_MipLinear, D3D11_FILTER_COMPARISON_MIN_POINT_MAG_MIP_LINEAR },
    { FilterMode_Comparison_MinLinear_MagPoint_MipPoint, D3D11_FILTER_COMPARISON_MIN_LINEAR_MAG_MIP_POINT },
    { FilterMode_Comparison_MinLinear_MagPoint_MipLinear, D3D11_FILTER_COMPARISON_MIN_LINEAR_MAG_POINT_MIP_LINEAR },
    { FilterMode_Comparison_MinLinear_MagLinear_MipPoint, D3D11_FILTER_COMPARISON_MIN_MAG_LINEAR_MIP_POINT },
    { FilterMode_Comparison_MinLinear_MagLinear_MipLinear, D3D11_FILTER_COMPARISON_MIN_MAG_MIP_LINEAR },
    { FilterMode_Comparison_Anisotropic, D3D11_FILTER_COMPARISON_ANISOTROPIC },
    { FilterMode_Minimum_MinPoint_MagPoint_MipPoint, D3D11_FILTER_MINIMUM_MIN_MAG_MIP_POINT },
    { FilterMode_Minimum_MinPoint_MagLinear_MipLinear, D3D11_FILTER_MINIMUM_MIN_POINT_MAG_MIP_LINEAR },
    { FilterMode_Minimum_MinLinear_MagPoint_MipPoint, D3D11_FILTER_MINIMUM_MIN_LINEAR_MAG_MIP_POINT },
    { FilterMode_Minimum_MinLinear_MagPoint_MipLinear, D3D11_FILTER_MINIMUM_MIN_LINEAR_MAG_POINT_MIP_LINEAR },
    { FilterMode_Minimum_MinLinear_MagLinear_MipPoint, D3D11_FILTER_MINIMUM_MIN_MAG_LINEAR_MIP_POINT },
    { FilterMode_Minimum_MinLinear_MagLinear_MipLinear, D3D11_FILTER_MINIMUM_MIN_MAG_MIP_LINEAR },
    { FilterMode_Minimum_Anisotropic, D3D11_FILTER_MINIMUM_ANISOTROPIC },
    { FilterMode_Maximum_MinPoint_MagPoint_MipPoint, D3D11_FILTER_MAXIMUM_MIN_MAG_MIP_POINT },
    { FilterMode_Maximum_MinPoint_MagLinear_MipLinear, D3D11_FILTER_MAXIMUM_MIN_POINT_MAG_MIP_LINEAR },
    { FilterMode_Maximum_MinLinear_MagPoint_MipPoint, D3D11_FILTER_MAXIMUM_MIN_LINEAR_MAG_MIP_POINT },
    { FilterMode_Maximum_MinLinear_MagPoint_MipLinear, D3D11_FILTER_MAXIMUM_MIN_LINEAR_MAG_POINT_MIP_LINEAR },
    { FilterMode_Maximum_MinLinear_MagLinear_MipPoint, D3D11_FILTER_MAXIMUM_MIN_MAG_LINEAR_MIP_POINT },
    { FilterMode_Maximum_MinLinear_MagLinear_MipLinear, D3D11_FILTER_MAXIMUM_MIN_MAG_MIP_LINEAR },
    { FilterMode_Maximum_Anisotropic, D3D11_FILTER_MAXIMUM_ANISOTROPIC }
};

const struct GpuAccessAndD3dBindFlag
{
    GpuAccess gpuAccess;
    D3D11_BIND_FLAG bindFlag;
} D3dBindFlag[] =
{
    //{ GpuAccess_Read, },
    //{ GpuAccess_Write, },
    { GpuAccess_VertexBuffer, D3D11_BIND_VERTEX_BUFFER },
    { GpuAccess_IndexBuffer, D3D11_BIND_INDEX_BUFFER },
    { GpuAccess_ConstantBuffer, D3D11_BIND_CONSTANT_BUFFER },
    { GpuAccess_Texture, D3D11_BIND_SHADER_RESOURCE },
    { GpuAccess_UnorderedAccessBuffer, D3D11_BIND_UNORDERED_ACCESS },
    { GpuAccess_ColorBuffer, D3D11_BIND_RENDER_TARGET },
    { GpuAccess_DepthStencil, D3D11_BIND_DEPTH_STENCIL }
    //{ GpuAccess_IndirectBuffer, },
    //{ GpuAccess_ScanBuffer, },
    //{ GpuAccess_QueryBuffer, },
    //{ GpuAccess_Descriptor, },
    //{ GpuAccess_ShaderCode, },
    //{ GpuAccess_Image, }
};

}

void D3d::CreateDevice( ID3D11Device** ppOutDevice, ID3D11DeviceContext** ppOutDeviceContext, bool debugMode ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( ppOutDevice );
    NN_SDK_ASSERT_NOT_NULL( ppOutDeviceContext );

    UINT createFlags = debugMode ? D3D11_CREATE_DEVICE_DEBUG : 0;
    D3D_FEATURE_LEVEL featureLevels[] =
    {
        D3D_FEATURE_LEVEL_11_1,
        D3D_FEATURE_LEVEL_11_0
    };
    D3D_FEATURE_LEVEL outFeatureLevel;

    HRESULT hResult = NN_GFX_CALL_D3D_FUNCTION( D3D11CreateDevice( NULL, D3D_DRIVER_TYPE_HARDWARE, NULL, createFlags, featureLevels,
        sizeof( featureLevels ) / sizeof( D3D_FEATURE_LEVEL ), D3D11_SDK_VERSION, ppOutDevice, &outFeatureLevel, ppOutDeviceContext ) );

    NN_SDK_ASSERT ( SUCCEEDED( hResult ) && outFeatureLevel != 0 );
    NN_UNUSED( hResult );

    return;
}

void D3d::DeleteDevice( ID3D11Device* pDevice, ID3D11DeviceContext* pDeviceContext ) NN_NOEXCEPT
{
    if( pDeviceContext )
    {
        NN_GFX_CALL_D3D_FUNCTION( pDeviceContext->ClearState() );
        NN_GFX_CALL_D3D_FUNCTION( pDeviceContext->Release() );
    }

    if( pDevice )
    {
        NN_GFX_CALL_D3D_FUNCTION( pDevice->Release() );
    }
}

void D3d::CreateSwapChain( IDXGISwapChain** ppSwapChain, int width, int height,
    ImageFormat imageFormat, int bufferCount, ID3D11Device* pDevice, HWND hWnd ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES_NOT_NULL( hWnd );
    NN_SDK_REQUIRES_NOT_NULL( ppSwapChain );

    IDXGIDevice* pDxgiDevice = NULL;
    HRESULT hResult = NN_GFX_CALL_D3D_FUNCTION( pDevice->QueryInterface( __uuidof( IDXGIDevice ), reinterpret_cast< void** >( &pDxgiDevice ) ) );
    NN_SDK_ASSERT( SUCCEEDED( hResult ) );

    IDXGIAdapter* pDxgiAdapter = NULL;
    hResult = NN_GFX_CALL_D3D_FUNCTION( pDxgiDevice->GetAdapter( &pDxgiAdapter ) );
    NN_SDK_ASSERT( SUCCEEDED( hResult ) );

    IDXGIFactory1* pDxgiFactory = NULL;
    hResult = NN_GFX_CALL_D3D_FUNCTION( pDxgiAdapter->GetParent( __uuidof( IDXGIFactory1 ), reinterpret_cast< void** >( &pDxgiFactory ) ) );
    NN_SDK_ASSERT( SUCCEEDED( hResult ) );

    NN_GFX_CALL_D3D_FUNCTION( pDxgiAdapter->Release() );
    NN_GFX_CALL_D3D_FUNCTION( pDxgiDevice->Release() );

    IDXGIFactory2* pDxgiFactory2 = NULL;
    hResult = NN_GFX_CALL_D3D_FUNCTION( pDxgiFactory->QueryInterface( __uuidof( IDXGIFactory2 ), reinterpret_cast< void** >( &pDxgiFactory2 ) ) );
    NN_SDK_ASSERT( SUCCEEDED( hResult ) );

    DXGI_SWAP_CHAIN_DESC1 desc;
    ZeroMemory( &desc, sizeof( desc ) );
    desc.Width = width;
    desc.Height = height;
    desc.Format = GetTextureFormat( imageFormat ).dxgiFormat;
    desc.SampleDesc.Count = 1;
    desc.SampleDesc.Quality = 0;
    desc.BufferUsage = DXGI_USAGE_RENDER_TARGET_OUTPUT;
    desc.BufferCount = bufferCount;

    IDXGISwapChain1* pSwapChain1 = NULL;
    hResult = NN_GFX_CALL_D3D_FUNCTION( pDxgiFactory2->CreateSwapChainForHwnd( pDevice, hWnd, &desc, NULL, NULL, &pSwapChain1 ) );
    NN_SDK_ASSERT( SUCCEEDED( hResult ) );

    hResult = NN_GFX_CALL_D3D_FUNCTION( pSwapChain1->QueryInterface( __uuidof( IDXGISwapChain ), reinterpret_cast< void** >( ppSwapChain ) ) );
    NN_SDK_ASSERT( SUCCEEDED( hResult ) );

    NN_GFX_CALL_D3D_FUNCTION( pSwapChain1->Release() );
    NN_GFX_CALL_D3D_FUNCTION( pDxgiFactory2->Release() );

    NN_GFX_CALL_D3D_FUNCTION( pDxgiFactory->MakeWindowAssociation( hWnd, DXGI_MWA_NO_ALT_ENTER ) );
    NN_GFX_CALL_D3D_FUNCTION( pDxgiFactory->Release() );
}

D3dTextureFormat D3d::GetTextureFormat( ImageFormat format ) NN_NOEXCEPT
{
    struct Comp
    {
        bool operator()( const ImageFormatAndD3dFormat& lhs, ImageFormat rhs ) const
        {
            return lhs.imageFormat < rhs;
        }
    };
    const ImageFormatAndD3dFormat* pEnd = D3dTextureFormatList + NN_GFX_ARRAY_LENGTH( D3dTextureFormatList );
    const ImageFormatAndD3dFormat* pFound = std::lower_bound( D3dTextureFormatList, pEnd, format, Comp() );

    NN_SDK_ASSERT( format == pFound->imageFormat );

    return pFound->d3dFormat;
}

D3dTextureFormat D3d::GetShaderResourceTextureFormat( ImageFormat format ) NN_NOEXCEPT
{
    D3dTextureFormat d3dFormat = GetTextureFormat( format );

    switch ( d3dFormat.dxgiFormat )
    {
    case DXGI_FORMAT_D32_FLOAT:
        {
            d3dFormat.dxgiFormat = DXGI_FORMAT_R32_TYPELESS;
        }
        break;
    case DXGI_FORMAT_D16_UNORM:
        {
            d3dFormat.dxgiFormat = DXGI_FORMAT_R16_TYPELESS;
        }
        break;
    case DXGI_FORMAT_D24_UNORM_S8_UINT:
        {
            d3dFormat.dxgiFormat = DXGI_FORMAT_R24G8_TYPELESS;
        }
        break;
    case DXGI_FORMAT_D32_FLOAT_S8X24_UINT:
        {
            d3dFormat.dxgiFormat = DXGI_FORMAT_R32G8X24_TYPELESS;
        }
        break;
    default:
        break;
    }

    return d3dFormat;
}

D3dTextureFormat D3d::GetShaderResourceTextureViewFormat( ImageFormat format, DepthStencilFetchMode depthStencilFetchMode ) NN_NOEXCEPT
{
    D3dTextureFormat d3dFormat = GetTextureFormat( format );

    switch ( d3dFormat.dxgiFormat )
    {
    case DXGI_FORMAT_D32_FLOAT:
        {
            d3dFormat.dxgiFormat = DXGI_FORMAT_R32_FLOAT;
        }
        break;
    case DXGI_FORMAT_D16_UNORM:
        {
            d3dFormat.dxgiFormat = DXGI_FORMAT_R16_UNORM;
        }
        break;
    case DXGI_FORMAT_D24_UNORM_S8_UINT:
        {
            if ( depthStencilFetchMode == DepthStencilFetchMode_DepthComponent )
            {
                d3dFormat.dxgiFormat = DXGI_FORMAT_R24_UNORM_X8_TYPELESS;
            }
            else if ( depthStencilFetchMode == DepthStencilFetchMode_StencilIndex )
            {
                d3dFormat.dxgiFormat = DXGI_FORMAT_X24_TYPELESS_G8_UINT;
            }
        }
        break;
    case DXGI_FORMAT_D32_FLOAT_S8X24_UINT:
        {
            if ( depthStencilFetchMode == DepthStencilFetchMode_DepthComponent )
            {
                d3dFormat.dxgiFormat = DXGI_FORMAT_R32_FLOAT_X8X24_TYPELESS;
            }
            else if ( depthStencilFetchMode == DepthStencilFetchMode_StencilIndex )
            {
                d3dFormat.dxgiFormat = DXGI_FORMAT_X32_TYPELESS_G8X24_UINT;
            }
        }
        break;
    default:
        break;
    }

    return d3dFormat;
}

int D3d::GetBindFlag( int gpuAccessFlags ) NN_NOEXCEPT
{
    int bindFlags = 0;

    for ( int flag = 0; flag < sizeof( D3dBindFlag ) / sizeof( GpuAccessAndD3dBindFlag ); ++flag )
    {
        if ( D3dBindFlag[ flag ].gpuAccess & static_cast< Bit32 >( gpuAccessFlags ) )
        {
            bindFlags |= static_cast< int >( D3dBindFlag[ flag ].bindFlag );
        }
    }

    return bindFlags;
}

size_t D3d::GetImageSize( ImageFormat imageFormat, int width, int height, int depth ) NN_NOEXCEPT
{
    ChannelFormat channelFormat = GetChannelFormat( imageFormat );

    return static_cast< size_t >( CalculateImageSize( channelFormat, width, height, depth ) );
}

D3D11_FILL_MODE D3d::GetFillMode( FillMode fillMode ) NN_NOEXCEPT
{
    D3D11_FILL_MODE d3dFillMode;

    switch ( fillMode )
    {
    case FillMode_Wireframe:
        {
            d3dFillMode = D3D11_FILL_WIREFRAME;
        }
        break;
    case FillMode_Solid:
        {
            d3dFillMode = D3D11_FILL_SOLID;
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    return d3dFillMode;
}

D3D11_CULL_MODE D3d::GetCullMode( CullMode cullMode ) NN_NOEXCEPT
{
    D3D11_CULL_MODE d3dCullMode;

    switch ( cullMode )
    {
    case CullMode_None:
        {
            d3dCullMode = D3D11_CULL_NONE;
        }
        break;
    case CullMode_Front:
        {
            d3dCullMode = D3D11_CULL_FRONT;
        }
        break;
    case CullMode_Back:
        {
            d3dCullMode = D3D11_CULL_BACK;
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    return d3dCullMode;
}

D3D11_BLEND D3d::GetBlendFactor( BlendFactor blendFactor ) NN_NOEXCEPT
{
    D3D11_BLEND d3dBlendFactor;

    switch ( blendFactor )
    {
    case BlendFactor_Zero:
        {
            d3dBlendFactor = D3D11_BLEND_ZERO;
        }
        break;
    case BlendFactor_One:
        {
            d3dBlendFactor = D3D11_BLEND_ONE;
        }
        break;
    case BlendFactor_SourceColor:
        {
            d3dBlendFactor = D3D11_BLEND_SRC_COLOR;
        }
        break;
    case BlendFactor_OneMinusSourceColor:
        {
            d3dBlendFactor = D3D11_BLEND_INV_SRC_COLOR;
        }
        break;
    case BlendFactor_DestinationColor:
        {
            d3dBlendFactor = D3D11_BLEND_DEST_COLOR;
        }
        break;
    case BlendFactor_OneMinusDestinationColor:
        {
            d3dBlendFactor = D3D11_BLEND_INV_DEST_COLOR;
        }
        break;
    case BlendFactor_SourceAlpha:
        {
            d3dBlendFactor = D3D11_BLEND_SRC_ALPHA;
        }
        break;
    case BlendFactor_OneMinusSourceAlpha:
        {
            d3dBlendFactor = D3D11_BLEND_INV_SRC_ALPHA;
        }
        break;
    case BlendFactor_DestinationAlpha:
        {
            d3dBlendFactor = D3D11_BLEND_DEST_ALPHA;
        }
        break;
    case BlendFactor_OneMinusDestinationAlpha:
        {
            d3dBlendFactor = D3D11_BLEND_INV_DEST_ALPHA;
        }
        break;
    case BlendFactor_ConstantColor:
        {
            d3dBlendFactor = D3D11_BLEND_BLEND_FACTOR;
        }
        break;
    case BlendFactor_OneMinusConstantColor:
        {
            d3dBlendFactor = D3D11_BLEND_INV_BLEND_FACTOR;
        }
        break;
    case BlendFactor_Source1Color:
        {
            d3dBlendFactor = D3D11_BLEND_SRC1_COLOR;
        }
        break;
    case BlendFactor_OneMinusSource1Color:
        {
            d3dBlendFactor = D3D11_BLEND_INV_SRC1_COLOR;
        }
        break;
    case BlendFactor_Source1Alpha:
        {
            d3dBlendFactor = D3D11_BLEND_SRC1_ALPHA;
        }
        break;
    case BlendFactor_OneMinusSource1Alpha:
        {
            d3dBlendFactor = D3D11_BLEND_INV_SRC1_ALPHA;
        }
        break;
    case BlendFactor_ConstantAlpha:
    case BlendFactor_OneMinusConstantAlpha:
    case BlendFactor_SourceAlphaSaturate:
        {
            // D3D11ではサポートされていません。
            NN_SDK_ASSERT( 0 );
            // ダミーで設定します。
            d3dBlendFactor = D3D11_BLEND_ONE;
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    return d3dBlendFactor;
}

D3D11_BLEND_OP D3d::GetBlendOperation( BlendFunction blendFunction ) NN_NOEXCEPT
{
    D3D11_BLEND_OP d3dBlendOperation;

    switch ( blendFunction )
    {
    case BlendFunction_Add:
        {
            d3dBlendOperation = D3D11_BLEND_OP_ADD;
        }
        break;
    case BlendFunction_Subtract:
        {
            d3dBlendOperation = D3D11_BLEND_OP_SUBTRACT;
        }
        break;
    case BlendFunction_ReverseSubtract:
        {
            d3dBlendOperation = D3D11_BLEND_OP_REV_SUBTRACT;
        }
        break;
    case BlendFunction_Min:
        {
            d3dBlendOperation = D3D11_BLEND_OP_MIN;
        }
        break;
    case BlendFunction_Max:
        {
            d3dBlendOperation = D3D11_BLEND_OP_MAX;
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    return d3dBlendOperation;
}

int D3d::GetChannelMask( int channelMask ) NN_NOEXCEPT
{
    int d3dChannelMask = 0;

    if ( channelMask & ChannelMask_Red )
    {
        d3dChannelMask |= D3D11_COLOR_WRITE_ENABLE_RED;
    }
    if ( channelMask & ChannelMask_Green )
    {
        d3dChannelMask |= D3D11_COLOR_WRITE_ENABLE_GREEN;
    }
    if ( channelMask & ChannelMask_Blue )
    {
        d3dChannelMask |= D3D11_COLOR_WRITE_ENABLE_BLUE;
    }
    if ( channelMask & ChannelMask_Alpha )
    {
        d3dChannelMask |= D3D11_COLOR_WRITE_ENABLE_ALPHA;
    }

    return d3dChannelMask;
}

D3D11_PRIMITIVE_TOPOLOGY D3d::GetPrimitiveTopology( PrimitiveTopology primitiveTopology, int patchCount ) NN_NOEXCEPT
{
    D3D11_PRIMITIVE_TOPOLOGY d3dPrimitiveTopology;

    switch ( primitiveTopology )
    {
    case PrimitiveTopology_PointList:
        {
            d3dPrimitiveTopology = D3D11_PRIMITIVE_TOPOLOGY_POINTLIST;
        }
        break;
    case PrimitiveTopology_LineList:
        {
            d3dPrimitiveTopology = D3D11_PRIMITIVE_TOPOLOGY_LINELIST;
        }
        break;
    case PrimitiveTopology_LineStrip:
        {
            d3dPrimitiveTopology = D3D11_PRIMITIVE_TOPOLOGY_LINESTRIP;
        }
        break;
    case PrimitiveTopology_TriangleList:
        {
            d3dPrimitiveTopology = D3D11_PRIMITIVE_TOPOLOGY_TRIANGLELIST;
        }
        break;
    case PrimitiveTopology_TriangleStrip:
        {
            d3dPrimitiveTopology = D3D11_PRIMITIVE_TOPOLOGY_TRIANGLESTRIP;
        }
        break;
    case PrimitiveTopology_LineListAdjacency:
        {
            d3dPrimitiveTopology = D3D11_PRIMITIVE_TOPOLOGY_LINELIST_ADJ;
        }
        break;
    case PrimitiveTopology_LineStripAdjacency:
        {
            d3dPrimitiveTopology = D3D11_PRIMITIVE_TOPOLOGY_LINESTRIP_ADJ;
        }
        break;
    case PrimitiveTopology_TriangleListAdjacency:
        {
            d3dPrimitiveTopology = D3D11_PRIMITIVE_TOPOLOGY_TRIANGLELIST_ADJ;
        }
        break;
    case PrimitiveTopology_TriangleStripAdjacency:
        {
            d3dPrimitiveTopology = D3D11_PRIMITIVE_TOPOLOGY_TRIANGLESTRIP_ADJ;
        }
        break;
    case PrimitiveTopology_PatchList:
        {
            d3dPrimitiveTopology = static_cast< D3D11_PRIMITIVE_TOPOLOGY >(
                D3D11_PRIMITIVE_TOPOLOGY_1_CONTROL_POINT_PATCHLIST + patchCount - 1 );
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    return d3dPrimitiveTopology;
}

D3D11_COMPARISON_FUNC D3d::GetComparisonFunction( ComparisonFunction comparisonFunction ) NN_NOEXCEPT
{
    D3D11_COMPARISON_FUNC d3dComparisonFunction;

    switch ( comparisonFunction )
    {
    case ComparisonFunction_Never:
        {
            d3dComparisonFunction = D3D11_COMPARISON_NEVER;
        }
        break;
    case ComparisonFunction_Less:
        {
            d3dComparisonFunction = D3D11_COMPARISON_LESS;
        }
        break;
    case ComparisonFunction_Equal:
        {
            d3dComparisonFunction = D3D11_COMPARISON_EQUAL;
        }
        break;
    case ComparisonFunction_LessEqual:
        {
            d3dComparisonFunction = D3D11_COMPARISON_LESS_EQUAL;
        }
        break;
    case ComparisonFunction_Greater:
        {
            d3dComparisonFunction = D3D11_COMPARISON_GREATER;
        }
        break;
    case ComparisonFunction_NotEqual:
        {
            d3dComparisonFunction = D3D11_COMPARISON_NOT_EQUAL;
        }
        break;
    case ComparisonFunction_GreaterEqual:
        {
            d3dComparisonFunction = D3D11_COMPARISON_GREATER_EQUAL;
        }
        break;
    case ComparisonFunction_Always:
        {
            d3dComparisonFunction = D3D11_COMPARISON_ALWAYS;
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    return d3dComparisonFunction;
}

D3D11_STENCIL_OP D3d::GetStencilOperation( StencilOperation stencilOperation ) NN_NOEXCEPT
{
    D3D11_STENCIL_OP d3dStencilOperation;

    switch ( stencilOperation )
    {
    case StencilOperation_Keep:
        {
            d3dStencilOperation = D3D11_STENCIL_OP_KEEP;
        }
        break;
    case StencilOperation_Zero:
        {
            d3dStencilOperation = D3D11_STENCIL_OP_ZERO;
        }
        break;
    case StencilOperation_Replace:
        {
            d3dStencilOperation = D3D11_STENCIL_OP_REPLACE;
        }
        break;
    case StencilOperation_IncrementClamp:
        {
            d3dStencilOperation = D3D11_STENCIL_OP_INCR_SAT;
        }
        break;
    case StencilOperation_DecrementClamp:
        {
            d3dStencilOperation = D3D11_STENCIL_OP_DECR_SAT;
        }
        break;
    case StencilOperation_Invert:
        {
            d3dStencilOperation = D3D11_STENCIL_OP_INVERT;
        }
        break;
    case StencilOperation_IncrementWrap:
        {
            d3dStencilOperation = D3D11_STENCIL_OP_INCR;
        }
        break;
    case StencilOperation_DecrementWrap:
        {
            d3dStencilOperation = D3D11_STENCIL_OP_DECR;
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    return d3dStencilOperation;
}

D3dAttributeFormat D3d::GetAttributeFormat( AttributeFormat format ) NN_NOEXCEPT
{
    struct Comp
    {
        bool operator()( const AttributeFormatAndD3dFormat& lhs, AttributeFormat rhs ) const
        {
            return lhs.attributeFormat < rhs;
        }
    };
    const AttributeFormatAndD3dFormat* pEnd = D3dAttributeFormatList + NN_GFX_ARRAY_LENGTH( D3dAttributeFormatList );
    const AttributeFormatAndD3dFormat* pFound = std::lower_bound( D3dAttributeFormatList, pEnd, format, Comp() );

    NN_SDK_ASSERT( format == pFound->attributeFormat );

    return pFound->d3dFormat;
}

D3dIndexFormat D3d::GetIndexFormat( IndexFormat format ) NN_NOEXCEPT
{
    D3dIndexFormat d3dIndexFormat = { DXGI_FORMAT_UNKNOWN };

    switch ( format )
    {
    case IndexFormat_Uint16:
        {
            d3dIndexFormat.dxgiFormat = DXGI_FORMAT_R16_UINT;
        }
        break;
    case IndexFormat_Uint32:
        {
            d3dIndexFormat.dxgiFormat = DXGI_FORMAT_R32_UINT;
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    return d3dIndexFormat;
}

int D3d::GetDepthStencilClearMode( DepthStencilClearMode clearMode ) NN_NOEXCEPT
{
    int clearFlags = 0;

    switch ( clearMode )
    {
    case DepthStencilClearMode_Depth:
        {
            clearFlags = D3D11_CLEAR_DEPTH;
        }
        break;
    case DepthStencilClearMode_Stencil:
        {
            clearFlags = D3D11_CLEAR_STENCIL;
        }
        break;
    case DepthStencilClearMode_DepthStencil:
        {
            clearFlags = D3D11_CLEAR_DEPTH | D3D11_CLEAR_STENCIL;
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    return clearFlags;
}

D3D11_FILTER D3d::GetFilterMode( FilterMode filterMode ) NN_NOEXCEPT
{
    struct Comp
    {
        bool operator()( const FilterModeAndD3dFilterMode& lhs, FilterMode rhs ) const
        {
            return lhs.filterMode < rhs;
        }
    };
    const FilterModeAndD3dFilterMode* pEnd = D3dFilterModeList + NN_GFX_ARRAY_LENGTH( D3dFilterModeList );
    const FilterModeAndD3dFilterMode* pFound = std::lower_bound( D3dFilterModeList, pEnd, filterMode, Comp() );

    NN_SDK_ASSERT( filterMode == pFound->filterMode );

    return pFound->d3dFilterMode;
}

D3D11_TEXTURE_ADDRESS_MODE D3d::GetTextureAddressMode( TextureAddressMode textureAddressMode ) NN_NOEXCEPT
{
    D3D11_TEXTURE_ADDRESS_MODE d3dTextureAddressMode;

    switch ( textureAddressMode )
    {
    case TextureAddressMode_Repeat:
        {
            d3dTextureAddressMode = D3D11_TEXTURE_ADDRESS_WRAP;
        }
        break;
    case TextureAddressMode_Mirror:
        {
            d3dTextureAddressMode = D3D11_TEXTURE_ADDRESS_MIRROR;
        }
        break;
    case TextureAddressMode_ClampToEdge:
        {
            d3dTextureAddressMode = D3D11_TEXTURE_ADDRESS_CLAMP;
        }
        break;
    case TextureAddressMode_ClampToBorder:
        {
            d3dTextureAddressMode = D3D11_TEXTURE_ADDRESS_BORDER;
        }
        break;
    case TextureAddressMode_MirrorClampToEdge:
        {
            d3dTextureAddressMode = D3D11_TEXTURE_ADDRESS_MIRROR_ONCE;
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    return d3dTextureAddressMode;
}

ImageDimension D3d::GetImageDimension( ID3D11Resource* pResource ) NN_NOEXCEPT
{
    ImageDimension ret;
    D3D11_RESOURCE_DIMENSION resourceType;

    pResource->GetType( &resourceType );

    switch ( resourceType )
    {
    case D3D11_RESOURCE_DIMENSION_TEXTURE1D:
        {
            ID3D11Texture1D* pTexture = static_cast< ID3D11Texture1D* >( pResource );
            D3D11_TEXTURE1D_DESC desc;
            pTexture->GetDesc( &desc );
            if ( desc.ArraySize > 1 )
            {
                ret = ImageDimension_1dArray;
            }
            else
            {
                ret = ImageDimension_1d;
            }
        }
        break;
    case D3D11_RESOURCE_DIMENSION_TEXTURE2D:
        {
            ID3D11Texture2D* pTexture = static_cast< ID3D11Texture2D* >( pResource );
            D3D11_TEXTURE2D_DESC desc;
            pTexture->GetDesc( &desc );
            if ( desc.ArraySize > 1 )
            {
                if ( desc.SampleDesc.Count > 1 )
                {
                    ret = ImageDimension_2dMultisampleArray;
                }
                else
                {
                    ret = ImageDimension_2dArray;
                }
            }
            else
            {
                if ( desc.SampleDesc.Count > 1 )
                {
                    ret = ImageDimension_2dMultisample;
                }
                else
                {
                    ret = ImageDimension_2d;
                }
            }
        }
        break;
    case D3D11_RESOURCE_DIMENSION_TEXTURE3D:
        {
            ret = ImageDimension_3d;
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    return ret;
}

int D3d::GetTextureMipLevels( ID3D11Resource* pResource ) NN_NOEXCEPT
{
    D3D11_RESOURCE_DIMENSION resourceType;
    int ret = 0;

    pResource->GetType( &resourceType );

    switch ( resourceType )
    {
    case D3D11_RESOURCE_DIMENSION_TEXTURE1D:
        {
            ID3D11Texture1D* pTexture = static_cast< ID3D11Texture1D* >( pResource );
            D3D11_TEXTURE1D_DESC desc;
            pTexture->GetDesc( &desc );
            ret = desc.MipLevels;
        }
        break;
    case D3D11_RESOURCE_DIMENSION_TEXTURE2D:
        {
            ID3D11Texture2D* pTexture = static_cast< ID3D11Texture2D* >( pResource );
            D3D11_TEXTURE2D_DESC desc;
            pTexture->GetDesc( &desc );
            ret = desc.MipLevels;
        }
        break;
    case D3D11_RESOURCE_DIMENSION_TEXTURE3D:
        {
            ID3D11Texture3D* pTexture = static_cast< ID3D11Texture3D* >( pResource );
            D3D11_TEXTURE3D_DESC desc;
            pTexture->GetDesc( &desc );
            ret = desc.MipLevels;
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    return ret;
}

void D3d::GetTextureSize( int* pWidth, int* pHeight, int* pDepth, ID3D11Resource* pResource ) NN_NOEXCEPT
{
    D3D11_RESOURCE_DIMENSION resourceType;

    pResource->GetType( &resourceType );

    int width = 1;
    int height = 1;
    int depth = 1;

    switch ( resourceType )
    {
    case D3D11_RESOURCE_DIMENSION_TEXTURE1D:
        {
            ID3D11Texture1D* pTexture = static_cast< ID3D11Texture1D* >( pResource );
            D3D11_TEXTURE1D_DESC desc;
            pTexture->GetDesc( &desc );
            width = desc.Width;
            if ( desc.ArraySize > 1 )
            {
                height = desc.ArraySize;
            }
        }
        break;
    case D3D11_RESOURCE_DIMENSION_TEXTURE2D:
        {
            ID3D11Texture2D* pTexture = static_cast< ID3D11Texture2D* >( pResource );
            D3D11_TEXTURE2D_DESC desc;
            pTexture->GetDesc( &desc );
            width = desc.Width;
            height = desc.Height;
            if ( desc.ArraySize > 1 )
            {
                depth = desc.ArraySize;
            }
        }
        break;
    case D3D11_RESOURCE_DIMENSION_TEXTURE3D:
        {
            ID3D11Texture3D* pTexture = static_cast< ID3D11Texture3D* >( pResource );
            D3D11_TEXTURE3D_DESC desc;
            pTexture->GetDesc( &desc );
            width = desc.Width;
            height = desc.Height;
            depth = desc.Depth;
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    if ( pWidth )
    {
        *pWidth = width;
    }
    if ( pHeight )
    {
        *pHeight = height;
    }
    if ( pDepth )
    {
        *pDepth = depth;
    }
}

ID3D11Buffer* D3d::GetBufferResource( uint64_t bufferAddressImpl ) NN_NOEXCEPT
{
    if ( bufferAddressImpl == 0 )
        return NULL;

    BufferImpl< Target >::DataType &data = reinterpret_cast< BufferImpl< Target >* >( bufferAddressImpl )->ToData();

    return static_cast< ID3D11Buffer* >( data.pGpuBuffer );
}

ID3D11UnorderedAccessView* D3d::GetBufferResourceView( uint64_t bufferAddressImpl ) NN_NOEXCEPT
{
    if ( bufferAddressImpl == 0 )
        return NULL;

    BufferImpl< Target >::DataType &data = reinterpret_cast< BufferImpl< Target >* >( bufferAddressImpl )->ToData();

    return static_cast< ID3D11UnorderedAccessView* >( data.pBufferView );
}

}
}
}
