﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <algorithm>

#include <nn/nn_SdkAssert.h>
#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/nn_SdkLog.h>

#include <nn/util/util_BytePtr.h>

#include <nn/gfx/gfx_CommandBufferInfo.h>
#include <nn/gfx/gfx_TextureInfo.h>
#include <nn/gfx/gfx_RootSignatureInfo.h>
#include <nn/gfx/gfx_GpuAddress.h>
#include <nn/gfx/gfx_DescriptorSlot.h>
#include <nn/gfx/gfx_CommandBuffer.h>
#include <nn/gfx/gfx_StateInfo.h>

#include <nn/gfx/detail/gfx_Misc.h>
#include <nn/gfx/detail/gfx_CommandBuffer-api.vk.1.h>
#include <nn/gfx/detail/gfx_State-api.vk.1.h>
#include <nn/gfx/detail/gfx_Texture-api.vk.1.h>
#include <nn/gfx/detail/gfx_Pipeline-api.vk.1.h>
#include <nn/gfx/detail/gfx_Device-api.vk.1.h>
#include <nn/gfx/detail/gfx_Buffer-api.vk.1.h>
#include <nn/gfx/detail/gfx_Sampler-api.vk.1.h>
#include <nn/gfx/detail/gfx_Shader-api.vk.1.h>
#include <nn/gfx/detail/gfx_DescriptorPool-api.vk.1.h>
#include <nn/gfx/detail/gfx_RootSignature-api.vk.1.h>
#include <nn/gfx/detail/gfx_MemoryPool-api.vk.1.h>
#include <nn/gfx/detail/gfx_BufferLayout-api.vk.1.h>

#include "gfx_CommonHelper.h"
#include "gfx_VkHelper.h"
#include "gfx_VkManager.h"
#include "gfx_VkCommand.h"

NN_PRAGMA_PUSH_WARNINGS
NN_DISABLE_WARNING_DEPRECATED_DECLARATIONS

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationVk1 Target;

namespace {

template< typename T >
T* ToPtr( const DescriptorSlot& slot )
{
    return reinterpret_cast< T* >( slot.ToData()->value );
}

bool HasEnoughWorkingMemory( const CommandBufferImpl< Target >* pObj )
{
    NN_SDK_ASSERT_NOT_NULL( pObj );
    const CommandBufferImpl< Target >::DataType& obj = pObj->ToData();
    return obj.pHeadControlMemory && obj.pControlMemory && obj.controlMemorySize >=
        nn::util::ConstBytePtr( obj.pHeadControlMemory.ptr ).Distance( obj.pControlMemory ) + sizeof( uint64_t ) * 3;
}

bool HasEnoughCommandMemory( const CommandBufferImpl< Target >* pObj, uint32_t paramSize )
{
    NN_SDK_ASSERT_NOT_NULL( pObj );
    const CommandBufferImpl< Target >::DataType& obj = pObj->ToData();
    return obj.pHeadCommandMemory && obj.pCommandMemory &&
        obj.commandMemorySize >= nn::util::ConstBytePtr( obj.pHeadCommandMemory.ptr ).Distance(
            obj.pCommandMemory ) + offsetof( VkCommand, param ) + paramSize + sizeof( uint64_t );
}

void CheckWorkingMemory( CommandBufferImpl< Target >* pObj )
{
    NN_SDK_ASSERT_NOT_NULL( pObj );
    CommandBufferImpl< Target >::DataType& obj = pObj->ToData();
    if( !HasEnoughWorkingMemory( pObj ) )
    {
        Bit8 oldState = obj.state;
        obj.state = CommandBufferImpl< Target >::DataType::State_Callback;
        NN_SDK_ASSERT_NOT_NULL( obj.pOutOfControlMemoryCallback.ptr );
        OutOfMemoryEventArg arg;
        arg.minRequiredSize = obj.controlMemorySize - nn::util::ConstBytePtr(
            obj.pHeadControlMemory.ptr ).Distance( obj.pControlMemory ) + sizeof( uint64_t ) * 3;
        ( *reinterpret_cast< CommandBufferImpl< Target >::OutOfMemoryEventCallback >(
            obj.pOutOfControlMemoryCallback.ptr ) )( static_cast< TCommandBuffer< Target >* >( pObj ), arg );
        obj.state = oldState;
        NN_SDK_ASSERT( HasEnoughWorkingMemory( pObj ) );
    }
}

void CheckCommandMemory( CommandBufferImpl< Target >* pObj, uint32_t paramSize )
{
    NN_SDK_ASSERT_NOT_NULL( pObj );
    CommandBufferImpl< Target >::DataType& obj = pObj->ToData();
    if( !HasEnoughCommandMemory( pObj, paramSize ) )
    {
        Bit8 oldState = obj.state;
        obj.state = CommandBufferImpl< Target >::DataType::State_Callback;
        NN_SDK_ASSERT_NOT_NULL( obj.pOutOfCommandMemoryCallback.ptr );
        OutOfMemoryEventArg arg;
        arg.minRequiredSize = offsetof( VkCommand, param ) + paramSize + sizeof( uint64_t ) +
            ( ( obj.pHeadCommandMemory.ptr && obj.pCommandMemory.ptr ) ? obj.commandMemorySize -
                nn::util::ConstBytePtr( obj.pHeadCommandMemory.ptr ).Distance( obj.pCommandMemory ) : 0 );
        ( *reinterpret_cast< CommandBufferImpl< Target >::OutOfMemoryEventCallback >(
            obj.pOutOfCommandMemoryCallback.ptr ) )( static_cast< TCommandBuffer< Target >* >( pObj ), arg );
        obj.state = oldState;
        NN_SDK_ASSERT( HasEnoughCommandMemory( pObj, paramSize ) );
    }
}

template< typename TParam >
TParam* WriteVkCommand( CommandBufferImpl< Target >* pObj,
    uint32_t paramSize = sizeof( TParam ) ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pObj );
    NN_SDK_ASSERT( pObj->ToData()->state == CommandBufferImpl< Target >::DataType::State_Begun );

    uint32_t alignedParamSize = nn::util::align_up( paramSize, 8 );

    CheckCommandMemory( pObj, alignedParamSize );

    VkCommand* pCommand = static_cast< VkCommand* >( pObj->ToData()->pCommandMemory );
    pCommand->pCommandProc = VkCommandProc< TParam >;
    pCommand->paramSize = alignedParamSize;
    pObj->ToData()->pCommandMemory = pCommand->param + alignedParamSize;

    return static_cast< TParam* >( static_cast< void* >( pCommand->param ) );
}

template< VkEnum DescriptorType >
void SetBuffer( CommandBufferImpl< Target >* pObj, int slot, ShaderStage shaderStage,
    VkHandle handle, uint64_t offset, uint64_t size ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pObj );
    NN_SDK_ASSERT( slot >= 0 );
    NN_SDK_ASSERT( pObj->ToData()->state == CommandBufferImpl< Target >::DataType::State_Begun );

    NN_SDK_ASSERT( handle != 0 );

    VkSetBufferParam& param = *WriteVkCommand< VkSetBufferParam >( pObj );
    {
        param.buffer = handle;
        param.descriptorType = DescriptorType;
        param.slot = slot;
        param.shaderStage = Vk::GetShaderStage( shaderStage );
        param.offset = offset;
        param.size = size;
    };
}

void CommandThreadMain( void* pArg ) NN_NOEXCEPT;

class CommandThread
{
    NN_DISALLOW_COPY( CommandThread );

public:
    CommandThread() NN_NOEXCEPT
    {
        for ( int idxThread = 0; idxThread < ThreadCount; ++idxThread )
        {
            m_pThread[ idxThread ] = static_cast< nn::os::ThreadType* >( Vk::AllocDriverMemory( sizeof( nn::os::ThreadType ), 8 ) );
            m_pThreadArgument[ idxThread ] = static_cast< ThreadArgument* >( Vk::AllocDriverMemory( sizeof( ThreadArgument ), 8 ) );
            m_pThreadArgument[ idxThread ]->pStartEvent = static_cast< nn::os::EventType* >( Vk::AllocDriverMemory( sizeof( nn::os::EventType ), 8 ) );
            m_pThreadArgument[ idxThread ]->pEndEvent = static_cast< nn::os::EventType* >( Vk::AllocDriverMemory( sizeof( nn::os::EventType ), 8 ) );
            m_pThreadStack[ idxThread ] = static_cast< uint8_t* >( Vk::AllocDriverMemory( ThreadStackSize, nn::os::ThreadStackAlignment ) );

            if ( Vk::IsLayoutManagementEnabled() )
            {
                ImageLayoutStateCacheList* pCache = reinterpret_cast< ImageLayoutStateCacheList* >( Vk::AllocDriverMemory( sizeof( ImageLayoutStateCacheList ), 8 ) );
                new ( pCache ) ImageLayoutStateCacheList();
                size_t layoutCacheSize = 512 * 1024;
                void* pFrameHeapMem = Vk::AllocDriverMemory( layoutCacheSize, 8 );
                m_pThreadArgument[ idxThread ]->pFrameHeapMem = pFrameHeapMem;
                m_pThreadArgument[ idxThread ]->frameHeap = nn::lmem::CreateFrameHeap( pFrameHeapMem, layoutCacheSize, nn::lmem::CreationOption_NoOption );
                pCache->SetFrameHeap( m_pThreadArgument[ idxThread ]->frameHeap, true );

                m_pThreadArgument[ idxThread ]->pImageLayoutCache = pCache;
                m_pThreadArgument[ idxThread ]->pImageLayoutCache->SetHashFunction( GfxVkImageLayoutState::CreateHashValue );
                m_pThreadArgument[ idxThread ]->pImageLayoutCache->SetCompareFunction( GfxVkImageLayoutState::Compare );
                m_pThreadArgument[ idxThread ]->pImageLayoutCache->SetTableIndexFunction( GfxVkImageLayoutState::GetHashTableIndexFromHashValue );
            }
            nn::os::InitializeEvent( m_pThreadArgument[ idxThread ]->pStartEvent, false, nn::os::EventClearMode_AutoClear );
            nn::os::InitializeEvent( m_pThreadArgument[ idxThread ]->pEndEvent, false, nn::os::EventClearMode_AutoClear );
            m_pThreadArgument[ idxThread ]->isExit = false;

            nn::Result result;
            result = nn::os::CreateThread( m_pThread[ idxThread ], CommandThreadMain, m_pThreadArgument[ idxThread ],
                m_pThreadStack[ idxThread ], ThreadStackSize, nn::os::DefaultThreadPriority );
            NN_SDK_ASSERT( result.IsSuccess() );

            nn::os::SetThreadCoreMask( m_pThread[ idxThread ], idxThread, nn::Bit64( 1 ) << idxThread );

            nn::os::StartThread( m_pThread[ idxThread ] );
        }
        m_RunningThreadMask = 0;
    }

    virtual ~CommandThread() NN_NOEXCEPT
    {
        for ( int idxThread = 0; idxThread < ThreadCount; ++idxThread )
        {
            m_pThreadArgument[ idxThread ]->isExit = true;

            nn::os::SignalEvent( m_pThreadArgument[ idxThread ]->pStartEvent );

            nn::os::WaitThread( m_pThread[ idxThread ] );
            nn::os::DestroyThread( m_pThread[ idxThread ] );
        }

        for ( int idxThread = 0; idxThread < ThreadCount; ++idxThread )
        {
            nn::os::FinalizeEvent( m_pThreadArgument[ idxThread ]->pStartEvent );
            nn::os::FinalizeEvent( m_pThreadArgument[ idxThread ]->pEndEvent );

            if ( Vk::IsLayoutManagementEnabled() )
            {
                nn::lmem::DestroyFrameHeap( m_pThreadArgument[ idxThread ]->frameHeap );
                Vk::FreeDriverMemory( m_pThreadArgument[ idxThread ]->pFrameHeapMem );
                Vk::FreeDriverMemory( m_pThreadArgument[ idxThread ]->pImageLayoutCache );
            }
            Vk::FreeDriverMemory( m_pThreadArgument[ idxThread ]->pStartEvent );
            Vk::FreeDriverMemory( m_pThreadArgument[ idxThread ]->pEndEvent );
            Vk::FreeDriverMemory( m_pThreadArgument[ idxThread ] );
            Vk::FreeDriverMemory( m_pThreadStack[ idxThread ] );
            Vk::FreeDriverMemory( m_pThread[ idxThread ] );
        }
    }

    struct ThreadArgument
    {
        int firstCommand;
        int lastCommand;
        int threadCommandCount;
        CommandBufferImpl< ApiVariationVk1 >* pGfxCommandBuffer;
        VkCommandBuffer vkDrawCommandBuffer;
        VkCommandBuffer vkPrepCommandBuffer;
        VkCommandBuffer vkPostCommandBuffer;
        ImageLayoutStateCacheList* pImageLayoutCache;
        void* pFrameHeapMem;
        nn::lmem::HeapHandle frameHeap;
        nn::os::EventType* pStartEvent;
        nn::os::EventType* pEndEvent;
        CommandContext::ContextType contextType;
        bool isExit;
    };

    int GetThreadCount() const NN_NOEXCEPT
    {
        return ThreadCount;
    }

    ImageLayoutStateCacheList* GetImageLayoutCache(int idxThread) NN_NOEXCEPT
    {
        return m_pThreadArgument[ idxThread ]->pImageLayoutCache;
    }

    VkCommandBuffer GetPrepareCommandBuffer( int idxThread ) NN_NOEXCEPT
    {
        return m_pThreadArgument[ idxThread ]->vkPrepCommandBuffer;
    }

    void SetThreadArgument( int idxThread, int firstCommand, int lastCommand, int threadCommandCount,
        CommandBufferImpl< ApiVariationVk1 >* pGfxCommandBuffer, VkCommandBuffer vkDrawCommandBuffer,
        VkCommandBuffer vkPrepCommandBuffer, VkCommandBuffer vkPostCommandBuffer,
        CommandContext::ContextType contextType ) NN_NOEXCEPT
    {
        m_pThreadArgument[ idxThread ]->firstCommand = firstCommand;
        m_pThreadArgument[ idxThread ]->lastCommand = lastCommand;
        m_pThreadArgument[ idxThread ]->threadCommandCount = threadCommandCount;
        m_pThreadArgument[ idxThread ]->pGfxCommandBuffer = pGfxCommandBuffer;
        m_pThreadArgument[ idxThread ]->vkDrawCommandBuffer = vkDrawCommandBuffer;
        m_pThreadArgument[ idxThread ]->vkPrepCommandBuffer = vkPrepCommandBuffer;
        m_pThreadArgument[ idxThread ]->vkPostCommandBuffer = vkPostCommandBuffer;
        m_pThreadArgument[ idxThread ]->contextType = contextType;
    }

    void StartThread( int idxThread ) NN_NOEXCEPT
    {
        NN_SDK_ASSERT( ( m_RunningThreadMask & ( 1 << idxThread ) ) == 0 );
        nn::os::SignalEvent( m_pThreadArgument[ idxThread ]->pStartEvent );
        m_RunningThreadMask |= 1 << idxThread;
    }

    void StartAllThread() NN_NOEXCEPT
    {
        for ( int idxThread = 0; idxThread < ThreadCount; ++idxThread )
        {
            NN_SDK_ASSERT( ( m_RunningThreadMask & ( 1 << idxThread ) ) == 0 );
            StartThread( idxThread );
            m_RunningThreadMask |= 1 << idxThread;
        }
    }

    void WaitThread( int idxThread ) NN_NOEXCEPT
    {
        nn::os::WaitEvent( m_pThreadArgument[ idxThread ]->pEndEvent );
        m_RunningThreadMask &= ~( 1 << idxThread );
    }

    void WaitAllThread() NN_NOEXCEPT
    {
        for ( int idxThread = 0; idxThread < ThreadCount; ++idxThread )
        {
            if ( m_RunningThreadMask & ( 1 << idxThread ) )
            {
                WaitThread( idxThread );
                m_RunningThreadMask &= ~( 1 << idxThread );
            }
        }
    }

private:
    static const int ThreadCount = 3;
    static const int ThreadStackSize = 0x10000;

    nn::os::ThreadType* m_pThread[ ThreadCount ];
    ThreadArgument* m_pThreadArgument[ ThreadCount ];
    uint8_t* m_pThreadStack[ ThreadCount ];
    uint8_t m_RunningThreadMask;
};

void ParseCommandBuffer( CommandContext* pCommandContext ) NN_NOEXCEPT
{
    if ( pCommandContext->IsContextTypeVulkanCaller() )
    {
        Vk::BeginCommandBuffer( pCommandContext->GetVkDrawCommandBuffer() );
        Vk::BeginCommandBuffer( pCommandContext->GetVkPrepCommandBuffer() );
        Vk::BeginCommandBuffer( pCommandContext->GetVkPostCommandBuffer() );

        // Initialize line width here to avoid validation error.
        NN_GFX_CALL_VK_FUNCTION( vkCmdSetLineWidth( pCommandContext->GetVkDrawCommandBuffer(), 1.0f ) );
    }

    ParseVkCommandList( pCommandContext, pCommandContext->GetGfxCommandBuffer()->ToData()->pCommandList );

    if ( pCommandContext->IsContextTypeVulkanCaller() )
    {
        if ( pCommandContext->IsInsideRenderPass() )
        {
            Vk::EndRenderPass( pCommandContext->GetVkDrawCommandBuffer() );
        }
        Vk::EndCommandBuffer( pCommandContext->GetVkDrawCommandBuffer() );
        if ( !Vk::IsLayoutManagementEnabled() )
        {
            // More commands will be necessary before submission.
            Vk::EndCommandBuffer( pCommandContext->GetVkPrepCommandBuffer() );
        }
        Vk::EndCommandBuffer( pCommandContext->GetVkPostCommandBuffer() );
    }
}

void CommandThreadMain( void* pArg ) NN_NOEXCEPT
{
    CommandThread::ThreadArgument* pThreadArgument = static_cast< CommandThread::ThreadArgument* >( pArg );

    CommandContext* pCommandContext = static_cast< CommandContext* >( Vk::AllocDriverMemory( sizeof( CommandContext ), 8 ) );
    new( pCommandContext ) CommandContext();

    while ( !pThreadArgument->isExit )
    {
        nn::os::WaitEvent( pThreadArgument->pStartEvent );

        if ( pThreadArgument->isExit )
        {
            break;
        }

        pCommandContext->Initialize( pThreadArgument->pGfxCommandBuffer, pThreadArgument->vkDrawCommandBuffer,
            pThreadArgument->vkPrepCommandBuffer, pThreadArgument->vkPostCommandBuffer,
            pThreadArgument->firstCommand, pThreadArgument->lastCommand, pThreadArgument->threadCommandCount, false,
            pThreadArgument->contextType );

        if ( Vk::IsLayoutManagementEnabled() )
        {
            pThreadArgument->pImageLayoutCache->RemoveAll();
            pCommandContext->SetLayoutManager( pThreadArgument->pImageLayoutCache, pThreadArgument->frameHeap );
        }
        ParseCommandBuffer( pCommandContext );

        nn::os::SignalEvent( pThreadArgument->pEndEvent );
    }

    pCommandContext->~CommandContext();
    Vk::FreeDriverMemory( pCommandContext );
}

void StartRecordCommandBuffer( CommandBufferImpl< ApiVariationVk1 >* pGfxCommandBuffer ) NN_NOEXCEPT
{
    //#define DEBUG_RUN_THREAD_SEQUENTIAL

    #if defined( NN_SDK_BUILD_DEBUG ) || defined( NN_SDK_BUILD_DEVELOP )
    /*
     * Debug/Developビルドの場合は、スレッドを生成せずに、そのままカレントスレッドで独自コマンドの解析を行ないます。
     * この場合のみ、QueryのPipelineStatisticsが取得可能になります。
     * Relesaeビルドでは、QueryのPipelineStatisticsの結果は常に0が返ります。
    */
    #define DEBUG_RUN_WITHOUT_THREAD
    #endif

    #ifdef DEBUG_RUN_WITHOUT_THREAD

    /*
     * スレッドを生成せずに、現在のスレッドでそのまま実行します。
     */
    CommandContext* pCommandContext = static_cast< CommandContext* >( Vk::AllocDriverMemory( sizeof( CommandContext ), 8 ) );
    new( pCommandContext ) CommandContext();

    pCommandContext->Initialize( pGfxCommandBuffer, VK_NULL_HANDLE, VK_NULL_HANDLE, VK_NULL_HANDLE,
        0, pGfxCommandBuffer->ToData()->drawDispatchCount,
        pGfxCommandBuffer->ToData()->drawDispatchCount, true, CommandContext::ContextType_StateUpdater );

    ParseCommandBuffer( pCommandContext );

    pCommandContext->Initialize( pGfxCommandBuffer,
        CastToVkDispatchableObject< VkCommandBuffer >( pGfxCommandBuffer->ToData()->vkCommandBufferHandles[ 0 ].hCommandBuffer[
        CommandBufferImpl< ApiVariationVk1 >::DataType::CommandBufferUsage_Draw ] ),
        CastToVkDispatchableObject< VkCommandBuffer >( pGfxCommandBuffer->ToData()->vkCommandBufferHandles[ 0 ].hCommandBuffer[
        CommandBufferImpl< ApiVariationVk1 >::DataType::CommandBufferUsage_Prepare ] ),
        CastToVkDispatchableObject< VkCommandBuffer >( pGfxCommandBuffer->ToData()->vkCommandBufferHandles[ 0 ].hCommandBuffer[
        CommandBufferImpl< ApiVariationVk1 >::DataType::CommandBufferUsage_Post ] ),
        0, pGfxCommandBuffer->ToData()->drawDispatchCount, pGfxCommandBuffer->ToData()->drawDispatchCount, true,
        CommandContext::ContextType_VulkanCaller );

    ImageLayoutStateCacheList* pCache = nullptr;
    nn::lmem::HeapHandle frameHeap = nullptr;
    void* pFrameHeapMem = nullptr;
    if ( Vk::IsLayoutManagementEnabled() )
    {
        pCache = reinterpret_cast< ImageLayoutStateCacheList* >( Vk::AllocDriverMemory( sizeof( ImageLayoutStateCacheList ), 8 ) );
        new ( pCache ) ImageLayoutStateCacheList();
        size_t layoutCacheSize = 512 * 1024;
        pFrameHeapMem = Vk::AllocDriverMemory( layoutCacheSize, 8 );
        frameHeap = nn::lmem::CreateFrameHeap( pFrameHeapMem, layoutCacheSize, nn::lmem::CreationOption_NoOption );
        pCache->SetFrameHeap( frameHeap, true );

        pCache->SetHashFunction( GfxVkImageLayoutState::CreateHashValue );
        pCache->SetCompareFunction( GfxVkImageLayoutState::Compare );
        pCache->SetTableIndexFunction( GfxVkImageLayoutState::GetHashTableIndexFromHashValue );
        pCommandContext->SetLayoutManager( pCache, frameHeap );
    }

    ParseCommandBuffer( pCommandContext );

    pGfxCommandBuffer->ToData()->recordedCommandBufferCount = 1;

    if ( Vk::IsLayoutManagementEnabled() )
    {
        VkCommandBuffer commandBuffer = pCommandContext->GetVkPrepCommandBuffer();
        for ( int hashIdx = 0; hashIdx < ImageLayoutStateHashSize; hashIdx++ )
        {
            ImageLayoutStateCacheList::HashNode& hashNode = pCache->GetHashTableNode( hashIdx );
            for ( ImageLayoutStateCacheList::HashNodeIterator it = hashNode.Begin();
                !hashNode.IsEnd( it ); ++it )
            {
                ( *it ).nodeData.ApplyLayoutTransition( commandBuffer );
            }
        }
        Vk::EndCommandBuffer( commandBuffer );

        nn::lmem::DestroyFrameHeap( frameHeap );
        Vk::FreeDriverMemory( pFrameHeapMem );
        Vk::FreeDriverMemory( pCache );
    }

    pCommandContext->~CommandContext();
    Vk::FreeDriverMemory( pCommandContext );

    #else

    CommandThread* pCommandThread = static_cast< CommandThread* >( pGfxCommandBuffer->ToData()->pCommandThread );

    int threadCommandCount = pGfxCommandBuffer->ToData()->drawDispatchCount / ( pCommandThread->GetThreadCount() - 1 );
    bool vulkanCallerIsOneThread = threadCommandCount == 0;

    for ( int idxThread = 0; idxThread < pCommandThread->GetThreadCount(); ++idxThread )
    {
        if ( idxThread == 0 )
        {
            pCommandThread->SetThreadArgument( idxThread, 0, 0, threadCommandCount, pGfxCommandBuffer, VK_NULL_HANDLE,
                VK_NULL_HANDLE, VK_NULL_HANDLE, CommandContext::ContextType_StateUpdater );
        }
        else
        {
            int firstCommand = threadCommandCount * ( idxThread - 1 );
            int lastCommand = vulkanCallerIsOneThread || ( idxThread == ( pCommandThread->GetThreadCount() - 1 ) ) ?
                pGfxCommandBuffer->ToData()->drawDispatchCount : threadCommandCount * idxThread - 1;

            pCommandThread->SetThreadArgument( idxThread, firstCommand, lastCommand, threadCommandCount, pGfxCommandBuffer,
                CastToVkDispatchableObject< VkCommandBuffer >( pGfxCommandBuffer->ToData()->vkCommandBufferHandles[ idxThread - 1 ].hCommandBuffer[
                CommandBufferImpl< ApiVariationVk1 >::DataType::CommandBufferUsage_Draw ] ),
                CastToVkDispatchableObject< VkCommandBuffer >( pGfxCommandBuffer->ToData()->vkCommandBufferHandles[ idxThread - 1 ].hCommandBuffer[
                CommandBufferImpl< ApiVariationVk1 >::DataType::CommandBufferUsage_Prepare ] ),
                CastToVkDispatchableObject< VkCommandBuffer >( pGfxCommandBuffer->ToData()->vkCommandBufferHandles[ idxThread - 1 ].hCommandBuffer[
                CommandBufferImpl< ApiVariationVk1 >::DataType::CommandBufferUsage_Post ] ),
                CommandContext::ContextType_VulkanCaller );
        }

        pCommandThread->StartThread( idxThread );
        #ifdef DEBUG_RUN_THREAD_SEQUENTIAL
        pCommandThread->WaitThread( idxThread );
        #endif

        if ( vulkanCallerIsOneThread && idxThread == 1 )
        {
            break;
        }
    }

    pGfxCommandBuffer->ToData()->recordedCommandBufferCount = vulkanCallerIsOneThread ? 1 : pCommandThread->GetThreadCount() - 1;

    #ifndef DEBUG_RUN_THREAD_SEQUENTIAL
    pCommandThread->WaitAllThread();
    #endif

    if ( Vk::IsLayoutManagementEnabled() )
    {
        for ( int idxThread = 0; idxThread < pCommandThread->GetThreadCount(); ++idxThread )
        {
            ImageLayoutStateCacheList* pCache = pCommandThread->GetImageLayoutCache( idxThread );
            VkCommandBuffer commandBuffer = pCommandThread->GetPrepareCommandBuffer( idxThread );
            if ( commandBuffer != VK_NULL_HANDLE )
            {
                for ( int hashIdx = 0; hashIdx < ImageLayoutStateHashSize; hashIdx++ )
                {
                    ImageLayoutStateCacheList::HashNode& hashNode = pCache->GetHashTableNode( hashIdx );
                    for ( ImageLayoutStateCacheList::HashNodeIterator it = hashNode.Begin();
                        !hashNode.IsEnd( it ); ++it )
                    {
                        ( *it ).nodeData.ApplyLayoutTransition( commandBuffer );
                    }
                }
                Vk::EndCommandBuffer( commandBuffer );
            }

            if ( vulkanCallerIsOneThread && idxThread == 1 )
            {
                break;
            }
        }
    }
    #endif
} // NOLINT(impl/function_size)

}

size_t CommandBufferImpl< Target >::GetCommandMemoryAlignment( DeviceImpl< Target >* ) NN_NOEXCEPT
{
    return 1;
}

size_t CommandBufferImpl< Target >::GetControlMemoryAlignment( DeviceImpl< Target >* ) NN_NOEXCEPT
{
    return 1;
}

CommandBufferImpl< Target >::CommandBufferImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

CommandBufferImpl< Target >::~CommandBufferImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void CommandBufferImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );

    this->pGfxDevice = pDevice;
    this->commandBufferType = static_cast< Bit8 >( info.GetCommandBufferType() );
    this->queueCapability = static_cast< Bit16 >( info.GetQueueCapability() );
    this->pGfxVkManager = NULL;
    for ( int idxCommandBuffer = 0; idxCommandBuffer < this->VkCommandThreadCount; ++idxCommandBuffer )
    {
        for ( int idxUsage = 0; idxUsage < CommandBufferUsage_End; ++idxUsage )
        {
            this->vkCommandBufferHandles[ idxCommandBuffer ].hCommandPool[ idxUsage ] = 0;
            this->vkCommandBufferHandles[ idxCommandBuffer ].hCommandBuffer[ idxUsage ] = 0;
        }
    }
    this->isVulkanDedicatedApiModeEnabled = 0;

    if ( info.GetCommandBufferType() == CommandBufferType_Direct )
    {
        this->pGfxVkManager = Vk::AllocDriverMemory( sizeof( VkManager ), 8 );
        NN_SDK_ASSERT( this->pGfxVkManager.ptr != 0 );
        new( this->pGfxVkManager.ptr ) VkManager( this );
        Vk::AddToVkManagerList( pDevice, this->pGfxVkManager );

        this->pCommandThread = Vk::AllocDriverMemory( sizeof( CommandThread ), 8 );
        NN_SDK_ASSERT( this->pCommandThread.ptr != 0 );
        new ( this->pCommandThread.ptr ) CommandThread();

        Vk::CreateCommandBuffer( this );
    }

    Reset();

    this->pOutOfCommandMemoryCallback = NULL;
    this->pOutOfControlMemoryCallback = NULL;

    VkQueryPool queryPool;

    for ( int queryTarget = 0; queryTarget < QueryTarget_End; ++queryTarget )
    {
        Vk::CreateQueryPool( &queryPool, pDevice, static_cast< QueryTarget >( queryTarget ), 1 );

        this->hQueryPools[ queryTarget ] = CastFromVkNonDispatchableObject< VkQueryPool >( queryPool );
    }

    Vk::CreateGraphicsPipelineStatisticsQueriesPool( &queryPool, pDevice, 1 );
    this->hGraphicsPipelineStatisticsQueryPool = CastFromVkNonDispatchableObject< VkQueryPool >( queryPool );

    this->hFastGraphicsPipelineStatisticsQueryPool = 0;
    this->hFastTimestampQueryPool = 0;
    this->fastGraphicsPipelineStatisticsQueryCount = 0;
    this->fastTimestampQueryCount = 0;

    this->state = State_Initialized;
}

void CommandBufferImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_UNUSED( pDevice );

    Reset();

    if ( this->commandBufferType == CommandBufferType_Direct )
    {
        Vk::DestroyCommandBuffer( pDevice, this );
    }

    for ( int queryTarget = 0; queryTarget < QueryTarget_End; ++queryTarget )
    {
        VkQueryPool queryPool = CastToVkNonDispatchableObject< VkQueryPool >( this->hQueryPools[ queryTarget ] );

        Vk::DestroyQueryPool( pDevice, queryPool );
    }

    Vk::DestroyQueryPool( pDevice,
        CastToVkNonDispatchableObject< VkQueryPool >( this->hGraphicsPipelineStatisticsQueryPool ) );

    if ( this->hFastGraphicsPipelineStatisticsQueryPool )
    {
        Vk::DestroyQueryPool( pDevice,
            CastToVkNonDispatchableObject< VkQueryPool >( this->hFastGraphicsPipelineStatisticsQueryPool ) );
    }

    if ( this->hFastTimestampQueryPool )
    {
        Vk::DestroyQueryPool( pDevice,
            CastToVkNonDispatchableObject< VkQueryPool >( this->hFastTimestampQueryPool ) );
    }

    if ( this->pGfxVkManager )
    {
        Vk::RemoveFromVkManagerList( pDevice, this->pGfxVkManager );
        static_cast< VkManager* >( this->pGfxVkManager )->~VkManager();
        Vk::FreeDriverMemory( this->pGfxVkManager );
        this->pGfxVkManager = NULL;
    }
    if ( this->pCommandThread )
    {
        static_cast< CommandThread* >( this->pCommandThread )->~CommandThread();
        Vk::FreeDriverMemory( this->pCommandThread );
        this->pCommandThread = NULL;
    }

    this->pGfxDevice = NULL;
    this->state = State_NotInitialized;
}

void CommandBufferImpl< Target >::AddCommandMemory( MemoryPoolImpl< Target >* pMemoryPool,
    ptrdiff_t memoryPoolOffset, size_t memorySize ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pMemoryPool );
    NN_SDK_REQUIRES( IsInitialized( *pMemoryPool ) );
    NN_SDK_REQUIRES( memoryPoolOffset + memorySize <= pMemoryPool->ToData()->memorySize );
    NN_SDK_ASSERT( pMemoryPool->ToData()->pMemory );
    NN_SDK_REQUIRES( this->state != State_NotInitialized );

    void* pMemory = nn::util::BytePtr( pMemoryPool->ToData()->pMemory, memoryPoolOffset ).Get();
    if( this->pCommandMemory )
    {
        NN_SDK_ASSERT_NOT_NULL( this->pHeadCommandMemory.ptr );
        NN_SDK_ASSERT( static_cast< ptrdiff_t >( this->commandMemorySize ) >=
            nn::util::ConstBytePtr( this->pHeadCommandMemory.ptr ).Distance(
                nn::util::BytePtr( this->pCommandMemory.ptr, sizeof( uint64_t ) ).Get() ) );
        *static_cast< uint64_t* >( this->pCommandMemory ) = 0;
    }
    this->commandMemorySize = static_cast< uint32_t >( memorySize );
    this->pHeadCommandMemory = pMemory;
    this->pCommandMemory = pMemory;

    CheckWorkingMemory( this );

    *static_cast< void** >( this->pControlMemory ) = this->pCommandMemory;
    this->pControlMemory = nn::util::BytePtr( this->pControlMemory, sizeof( uint64_t ) ).Get();
}

void CommandBufferImpl< Target >::AddControlMemory( void* pMemory, size_t memorySize ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pMemory );
    NN_SDK_REQUIRES( this->state != State_NotInitialized );
    NN_SDK_ASSERT( memorySize >= sizeof( uint64_t ) * 3 );

    if( ( this->state == State_Begun || this->state == State_Callback ) &&
        this->pControlMemory && this->pCommandList )
    {
        NN_SDK_ASSERT_NOT_NULL( this->pHeadControlMemory.ptr );
        NN_SDK_ASSERT( static_cast< ptrdiff_t >( this->controlMemorySize ) >=
            nn::util::ConstBytePtr( this->pHeadControlMemory.ptr ).Distance(
                nn::util::ConstBytePtr( this->pControlMemory.ptr, sizeof( uint64_t ) * 2 ).Get() ) );
        *static_cast< uint64_t* >( this->pControlMemory ) = 0;
        *nn::util::BytePtr( this->pControlMemory, sizeof( uint64_t ) ).Get< void* >() = pMemory;
    }

    this->controlMemorySize = static_cast< uint32_t >( memorySize );
    this->pHeadControlMemory = pMemory;
    this->pControlMemory = pMemory;
}

void CommandBufferImpl< Target >::SetOutOfCommandMemoryEventCallback(
    OutOfMemoryEventCallback pCallback ) NN_NOEXCEPT
{
    this->pOutOfCommandMemoryCallback.ptr = reinterpret_cast< void (*)() >( pCallback );
}

void CommandBufferImpl< Target >::SetOutOfControlMemoryEventCallback(
    OutOfMemoryEventCallback pCallback ) NN_NOEXCEPT
{
    this->pOutOfControlMemoryCallback.ptr = reinterpret_cast< void (*)() >( pCallback );
}

void CommandBufferImpl< Target >::Reset() NN_NOEXCEPT
{
    this->commandMemorySize = 0;
    this->pHeadCommandMemory = NULL;
    this->pCommandMemory = NULL;
    this->controlMemorySize = sizeof( this->defaultControlMemory );
    this->pHeadControlMemory = this->defaultControlMemory;
    this->pControlMemory = this->defaultControlMemory;
    this->pCommandList = NULL;
    this->drawDispatchCount = 0;
    this->flags.Clear();

    this->graphicsPipelineStatisticsQueryIndex = 0;
    this->timestampQueryIndex = 0;

    if ( this->commandBufferType == CommandBufferType_Direct )
    {
        for ( int idxCommandBuffer = 0; idxCommandBuffer < this->VkCommandThreadCount; ++idxCommandBuffer )
        {
            for ( int idxUsage = 0; idxUsage < CommandBufferUsage_End; ++idxUsage )
            {
                VkResult result;
                VkCommandBuffer vkCommandBuffer = CastToVkDispatchableObject< VkCommandBuffer >(
                    this->vkCommandBufferHandles[ idxCommandBuffer ].hCommandBuffer[ idxUsage ] );
                NN_GFX_CALL_VK_FUNCTION( result = vkResetCommandBuffer(
                    vkCommandBuffer, VK_COMMAND_BUFFER_RESET_RELEASE_RESOURCES_BIT ) );
                NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
                NN_UNUSED( result );
            }
        }

        if ( this->pGfxVkManager )
        {
            static_cast< VkManager* >( this->pGfxVkManager )->Reset();
        }
    }
}

void CommandBufferImpl< Target >::Begin() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_REQUIRES_NOT_NULL( this->pGfxDevice.ptr );

    this->pCommandList = NULL;
    this->drawDispatchCount = 0;

    CheckWorkingMemory( this );

    if ( this->pCommandMemory && HasEnoughCommandMemory( this, 0 ) )
    {
        *static_cast< void** >( this->pControlMemory ) = this->pCommandMemory;
        this->pCommandList = this->pControlMemory.ptr;
        this->pControlMemory = nn::util::BytePtr( this->pControlMemory, sizeof( uint64_t ) ).Get();
    }
    else
    {
        this->pCommandMemory = NULL;
        this->pCommandList = this->pControlMemory.ptr;
    }

    if ( this->commandBufferType == CommandBufferType_Direct )
    {
        // Need implicit reset of VkManager. VkCommandBuffer held in VkManager is also reset implicitly in vkBeginCommandBuffer.
        if ( this->pGfxVkManager )
        {
            static_cast< VkManager* >( this->pGfxVkManager )->Reset();
        }
    }

    this->state = State_Begun;
}

void CommandBufferImpl< Target >::End() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );

    if( this->pCommandMemory )
    {
        *static_cast< uint64_t* >( this->pCommandMemory ) = 0;
        this->pCommandMemory = nn::util::BytePtr( this->pCommandMemory, sizeof( uint64_t ) ).Get();
        NN_SDK_ASSERT( static_cast< ptrdiff_t >( this->commandMemorySize ) >=
            nn::util::ConstBytePtr( this->pHeadCommandMemory.ptr ).Distance( this->pCommandMemory ) );
    }

    *static_cast< uint64_t* >( this->pControlMemory ) = 0;
    this->pControlMemory = nn::util::BytePtr( this->pControlMemory, sizeof( uint64_t ) ).Get();
    *static_cast< uint64_t* >( this->pControlMemory ) = 0;
    this->pControlMemory = nn::util::BytePtr( this->pControlMemory, sizeof( uint64_t ) ).Get();
    NN_SDK_ASSERT( static_cast< ptrdiff_t >( this->controlMemorySize ) >=
        nn::util::ConstBytePtr( this->pHeadControlMemory.ptr ).Distance( this->pControlMemory ) );

    if ( this->commandBufferType == CommandBufferType_Direct )
    {
        NN_SDK_REQUIRES( this->pGfxVkManager.ptr != 0 );

        static_cast< VkManager* >( this->pGfxVkManager )->Begin( this->drawDispatchCount );

        StartRecordCommandBuffer( this );

        static_cast< VkManager* >( this->pGfxVkManager )->End();
    }

    this->state = State_Initialized;
}

void CommandBufferImpl< Target >::Dispatch(
    int groupCountX, int groupCountY, int groupCountZ ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );

    VkDispatchParam& param = *WriteVkCommand< VkDispatchParam >( this );
    {
        param.groupCountX = static_cast< uint32_t >( groupCountX );
        param.groupCountY = static_cast< uint32_t >( groupCountY );
        param.groupCountZ = static_cast< uint32_t >( groupCountZ );
    }

    ++this->drawDispatchCount;
}

void CommandBufferImpl< Target >::Draw(
    PrimitiveTopology primitiveTopology, int vertexCount, int vertexOffset ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );

    VkDrawParam& param = *WriteVkCommand< VkDrawParam >( this );
    {
        param.topology = Vk::GetDrawPrimitive( primitiveTopology );
        param.vertexCount = vertexCount;
        param.instanceCount = 1;
        param.firstVertex = vertexOffset;
        param.firstInstance = 0;
    };

    ++this->drawDispatchCount;
}

void CommandBufferImpl< Target >::Draw( PrimitiveTopology primitiveTopology,
    int vertexCountPerInstance, int vertexOffset, int instanceCount, int baseInstance ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );

    VkDrawParam& param = *WriteVkCommand< VkDrawParam >( this );
    {
        param.topology = Vk::GetDrawPrimitive( primitiveTopology );
        param.vertexCount = vertexCountPerInstance;
        param.instanceCount = instanceCount;
        param.firstVertex = vertexOffset;
        param.firstInstance = baseInstance;
    };

    ++this->drawDispatchCount;
}

void CommandBufferImpl< Target >::DrawIndexed( PrimitiveTopology primitiveTopology,
    IndexFormat indexFormat, const GpuAddress& indexBufferAddress,
    int indexCount, int baseVertex ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( indexBufferAddress.ToData()->impl != 0 );

    VkDrawIndexedParam& param = *WriteVkCommand< VkDrawIndexedParam >( this );
    {
        param.topology = Vk::GetDrawPrimitive( primitiveTopology );
        param.indexCount = indexCount;
        param.instanceCount = 1;
        param.vertexOffset = baseVertex;
        param.firstInstance = 0;
        param.indexType = Vk::GetIndexFormat( indexFormat );
        param.indexOffset = indexBufferAddress.ToData()->value;
        param.indexBuffer = indexBufferAddress.ToData()->impl;
    }

    ++this->drawDispatchCount;
}

void CommandBufferImpl< Target >::DrawIndexed( PrimitiveTopology primitiveTopology, IndexFormat indexFormat,
    const GpuAddress& indexBufferAddress, int indexCountPerInstance, int baseVertex, int instanceCount, int baseInstance ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( indexBufferAddress.ToData()->impl != 0 );

    VkDrawIndexedParam& param = *WriteVkCommand< VkDrawIndexedParam >( this );
    {
        param.topology = Vk::GetDrawPrimitive( primitiveTopology );
        param.indexCount = indexCountPerInstance;
        param.instanceCount = instanceCount;
        param.vertexOffset = baseVertex;
        param.firstInstance = baseInstance;
        param.indexType = Vk::GetIndexFormat( indexFormat );
        param.indexOffset = indexBufferAddress.ToData()->value;
        param.indexBuffer = indexBufferAddress.ToData()->impl;
    }

    ++this->drawDispatchCount;
}

void CommandBufferImpl< Target >::DispatchIndirect( const GpuAddress& indirectBufferAddress ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( indirectBufferAddress.ToData()->impl != 0 );

    VkDispatchIndirectParam& param = *WriteVkCommand< VkDispatchIndirectParam >( this );
    {
        param.offset = indirectBufferAddress.ToData()->value;
        param.buffer = indirectBufferAddress.ToData()->impl;
    }

    ++this->drawDispatchCount;
}

void CommandBufferImpl< Target >::DrawIndirect(
    PrimitiveTopology primitiveTopology, const GpuAddress& indirectBufferAddress ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( indirectBufferAddress.ToData()->impl != 0 );

    VkDrawIndirectParam& param = *WriteVkCommand< VkDrawIndirectParam >( this );
    {
        param.topology = Vk::GetDrawPrimitive( primitiveTopology );
        param.offset = indirectBufferAddress.ToData()->value;
        param.buffer = indirectBufferAddress.ToData()->impl;
    }

    ++this->drawDispatchCount;
}

void CommandBufferImpl< Target >::DrawIndexedIndirect( PrimitiveTopology primitiveTopology,
    IndexFormat indexFormat, const GpuAddress& indexBufferAddress, const GpuAddress& indirectBufferAddress ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( indexBufferAddress.ToData()->impl != 0 );
    NN_SDK_ASSERT( indirectBufferAddress.ToData()->impl != 0 );

    VkDrawIndexedIndirectParam& param = *WriteVkCommand< VkDrawIndexedIndirectParam >( this );
    {
        param.topology = Vk::GetDrawPrimitive( primitiveTopology );
        param.indexType = Vk::GetIndexFormat( indexFormat );
        param.indexOffset = indexBufferAddress.ToData()->value;
        param.indexBuffer = indexBufferAddress.ToData()->impl;
        param.indirectOffset = indirectBufferAddress.ToData()->value;
        param.indirectBuffer = indirectBufferAddress.ToData()->impl;
    }

    ++this->drawDispatchCount;
}

void CommandBufferImpl< Target >::SetPipeline( const PipelineImpl< Target >* pPipeline ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pPipeline );
    NN_SDK_REQUIRES( IsInitialized( *pPipeline ) );
    NN_SDK_REQUIRES( this->state == State_Begun );

    const PipelineImpl< Target >::DataType& pipeline = pPipeline->ToData();

    if( pipeline.pipelineType == PipelineType_Graphics )
    {
        SetRasterizerState( nn::gfx::DataToAccessor( pipeline.rasterizerState ) );
        SetBlendState( nn::gfx::DataToAccessor( pipeline.blendState ) );
        SetDepthStencilState( nn::gfx::DataToAccessor( pipeline.depthStencilState ) );
        SetVertexState( nn::gfx::DataToAccessor( pipeline.vertexState ) );
        if( pipeline.flags.GetBit( PipelineImpl< Target >::DataType::Flag_HasTessellationState ) )
        {
            SetTessellationState( nn::gfx::DataToAccessor( pipeline.tessellationState ) );
        }
        SetShader( pipeline.pShader, ShaderStageBit_All );
    }
    else if ( pipeline.pipelineType == PipelineType_Compute )
    {
        SetShader( pipeline.pShader, ShaderStageBit_Compute );
    }
}

void CommandBufferImpl< Target >::SetRenderTargets( int colorTargetCount,
    const ColorTargetViewImpl< Target >* const * ppColorTargets,
    const DepthStencilViewImpl< Target >* pDepthStencil ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( colorTargetCount < 1 || ppColorTargets );
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES( colorTargetCount <= GfxVkGpuState::maxColorAttachments );

    VkSetRenderTargetsParam& param = *WriteVkCommand< VkSetRenderTargetsParam >( this,
        sizeof( VkSetRenderTargetsParam )
        + ( colorTargetCount - 1 ) * sizeof( VkSetRenderTargetsParam::AttachmentParam ) );
    {
        param.depthAttachment.imageView = 0;

        uint32_t width = 0;
        uint32_t height = 0;
        uint32_t layers = 0;

        for( int idxTarget = 0; idxTarget < colorTargetCount; ++idxTarget )
        {
            if( ppColorTargets[ idxTarget ] )
            {
                const ColorTargetViewImpl< Target >::DataType& data = ppColorTargets[ idxTarget ]->ToData();
                width = std::max NN_PREVENT_MACRO_FUNC ( data.width >> data.targetMipLevel, 1u );
                height = std::max NN_PREVENT_MACRO_FUNC ( data.height >> data.targetMipLevel, 1u );
                layers = data.targetArrayCount;
                break;
            }
        }

        if( pDepthStencil )
        {
            NN_SDK_REQUIRES( IsInitialized( *pDepthStencil ) );
            param.depthAttachment.imageView = pDepthStencil->ToData()->hImageView;
            TextureImpl< Target >* pGfxTexture = pDepthStencil->ToData()->pManager;
            param.depthAttachment.pManager = pGfxTexture->ToData()->pManager;
            param.depthAttachment.layer = pDepthStencil->ToData()->targetBaseArray;
            param.depthAttachment.layerCount = pDepthStencil->ToData()->targetArrayCount;
            param.depthAttachment.mip = pDepthStencil->ToData()->targetMipLevel;
            param.depthAttachment.format = pDepthStencil->ToData()->format;
            param.depthAttachment.samples = pDepthStencil->ToData()->samples;
            width = std::max NN_PREVENT_MACRO_FUNC (
                pDepthStencil->ToData()->width >> pDepthStencil->ToData()->targetMipLevel, 1u );
            height = std::max NN_PREVENT_MACRO_FUNC (
                pDepthStencil->ToData()->height >> pDepthStencil->ToData()->targetMipLevel, 1u );
            layers = pDepthStencil->ToData()->targetArrayCount;
        }
        else
        {
            param.depthAttachment.imageView = 0;
            param.depthAttachment.pManager = nullptr;
            param.depthAttachment.format = 0;
            param.depthAttachment.samples = 0;
            param.depthAttachment.layer = 0;
            param.depthAttachment.layerCount = 0;
            param.depthAttachment.mip = 0;
        }
        param.colorAttachmentCount = colorTargetCount;
        for( int idxTarget = 0; idxTarget < colorTargetCount; ++idxTarget )
        {
            if( ppColorTargets[ idxTarget ] )
            {
                NN_SDK_REQUIRES( IsInitialized( *( ppColorTargets[ idxTarget ] ) ) );
                param.colorAttachment[ idxTarget ].imageView = ppColorTargets[ idxTarget ]->ToData()->hImageView;
                TextureImpl< Target >* pGfxTexture = ppColorTargets[ idxTarget ]->ToData()->pManager;
                param.colorAttachment[ idxTarget ].pManager = pGfxTexture->ToData()->pManager;
                param.colorAttachment[ idxTarget ].format = ppColorTargets[ idxTarget ]->ToData()->format;
                param.colorAttachment[ idxTarget ].samples = ppColorTargets[ idxTarget ]->ToData()->samples;
                param.colorAttachment[ idxTarget ].layer = ppColorTargets[ idxTarget ]->ToData()->targetBaseArray;
                param.colorAttachment[ idxTarget ].mip = ppColorTargets[ idxTarget ]->ToData()->targetMipLevel;
                param.colorAttachment[ idxTarget ].layerCount = ppColorTargets[ idxTarget ]->ToData()->targetArrayCount;

                const ColorTargetViewImpl< Target >::DataType& data = ppColorTargets[ idxTarget ]->ToData();
                uint32_t levelWidth = std::max NN_PREVENT_MACRO_FUNC ( data.width >> data.targetMipLevel, 1u );
                uint32_t levelHeight = std::max NN_PREVENT_MACRO_FUNC ( data.height >> data.targetMipLevel, 1u );

                width = std::min NN_PREVENT_MACRO_FUNC ( levelWidth, width );
                height = std::min NN_PREVENT_MACRO_FUNC ( levelHeight, height );
                layers = std::min NN_PREVENT_MACRO_FUNC ( ppColorTargets[ idxTarget ]->ToData()->targetArrayCount, layers );
            }
            else
            {
                param.colorAttachment[ idxTarget ].imageView = VK_NULL_HANDLE;
                param.colorAttachment[ idxTarget ].pManager = nullptr;
                param.colorAttachment[ idxTarget ].format = 0;
                param.colorAttachment[ idxTarget ].samples = 0;
                param.colorAttachment[ idxTarget ].layer = 0;
                param.colorAttachment[ idxTarget ].mip = 0;
                param.colorAttachment[ idxTarget ].layerCount = 0;
            }
        }
        param.width = width;
        param.height = height;
        param.layers = layers;
    }
}

void CommandBufferImpl< Target >::SetVertexBuffer( int bufferIndex,
    const GpuAddress& vertexBufferAddress, ptrdiff_t stride, size_t size ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_UNUSED( stride );
    NN_UNUSED( size );

    NN_SDK_ASSERT( vertexBufferAddress.ToData()->impl != 0 );

    VkSetVertexBufferParam& param = *WriteVkCommand< VkSetVertexBufferParam >( this );
    {
        param.bufferIndex = bufferIndex;
        param.buffer = static_cast< VkHandle >( vertexBufferAddress.ToData()->impl );
        param.offset = vertexBufferAddress.ToData()->value;
    };
}

void CommandBufferImpl< Target >::SetViewportScissorState(
    const ViewportScissorStateImpl< Target >* pViewportScissor ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pViewportScissor );
    NN_SDK_REQUIRES( IsInitialized( *pViewportScissor ) );
    NN_SDK_REQUIRES( this->state == State_Begun );

    const ViewportScissorStateImpl< Target >::DataType& data = pViewportScissor->ToData();
    NN_SDK_ASSERT( data.viewportCount > 0 );
    int staticSize = offsetof( ViewportScissorStateImplData< Target >, pWorkMemory );
    int dynamicSize = ( sizeof( float ) * 6 + sizeof( int32_t ) * 4 ) * ( data.viewportCount - 1 );

    ViewportScissorStateImplData< Target >& param = *WriteVkCommand<
        ViewportScissorStateImplData< Target > >( this, staticSize + dynamicSize );
    {
        memcpy( &param, &data, staticSize );
        memcpy( &param.pWorkMemory, data.pWorkMemory, dynamicSize );
    }
}

void CommandBufferImpl< Target >::CopyBuffer( BufferImpl< Target >* pDstBuffer, ptrdiff_t dstOffset,
    const BufferImpl< Target >* pSrcBuffer, ptrdiff_t srcOffset, size_t size ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDstBuffer );
    NN_SDK_REQUIRES_NOT_NULL( pSrcBuffer );
    NN_SDK_REQUIRES( IsInitialized( *pDstBuffer ) );
    NN_SDK_REQUIRES( IsInitialized( *pSrcBuffer ) );
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( pDstBuffer->ToData()->hBuffer != 0 );
    NN_SDK_ASSERT( pSrcBuffer->ToData()->hBuffer != 0 );

    VkCopyBufferParam& param = *WriteVkCommand< VkCopyBufferParam >( this );
    {
        param.dstBuffer = pDstBuffer->ToData()->hBuffer;
        param.srcBuffer = pSrcBuffer->ToData()->hBuffer;
        param.dstOffset = static_cast< uint64_t >( dstOffset );
        param.srcOffset = static_cast< uint64_t >( srcOffset );
        param.size = static_cast< uint64_t >( size );
    }
}

void CommandBufferImpl< Target >::CopyImage( TextureImpl< Target >* pDstTexture,
    const TextureSubresource& dstSubresource, int dstOffsetU, int dstOffsetV, int dstOffsetW,
    const TextureImpl< Target >* pSrcTexture, const TextureCopyRegion& srcCopyRegion ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDstTexture );
    NN_SDK_REQUIRES_NOT_NULL( pSrcTexture );
    NN_SDK_REQUIRES( IsInitialized( *pDstTexture ) );
    NN_SDK_REQUIRES( IsInitialized( *pSrcTexture ) );
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( pDstTexture->ToData()->hImage != 0 );
    NN_SDK_ASSERT( pSrcTexture->ToData()->hImage != 0 );

    VkCopyImageParam& param = *WriteVkCommand< VkCopyImageParam >( this );
    {
        param.srcImage = pSrcTexture->ToData()->hImage;
        param.pSrcManager = pSrcTexture->ToData()->pManager;
        param.dstImage = pDstTexture->ToData()->hImage;
        param.pDstManager = pDstTexture->ToData()->pManager;
        param.width = srcCopyRegion.GetWidth();
        param.height = srcCopyRegion.GetHeight();
        param.depth = srcCopyRegion.GetDepth();
        param.layerCount = srcCopyRegion.GetArrayLength();
        param.srcX = srcCopyRegion.GetOffsetU();
        param.srcY = srcCopyRegion.GetOffsetV();
        param.srcZ = srcCopyRegion.GetOffsetW();
        param.srcMipLevel = srcCopyRegion.GetSubresource().GetMipLevel();
        param.srcBaseLayer = srcCopyRegion.GetSubresource().GetArrayIndex();
        param.srcAspect = Vk::GetImageAspect( pSrcTexture->ToData()->format );
        param.dstX = dstOffsetU;
        param.dstY = dstOffsetV;
        param.dstZ = dstOffsetW;
        param.dstMipLevel = dstSubresource.GetMipLevel();
        param.dstBaseLayer = dstSubresource.GetArrayIndex();
        param.dstAspect = Vk::GetImageAspect( pDstTexture->ToData()->format );
    }
}

void CommandBufferImpl< Target >::CopyBufferToImage( TextureImpl< Target >* pDstTexture,
     const BufferImpl< Target >* pSrcBuffer, const BufferTextureCopyRegion& region ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDstTexture );
    NN_SDK_REQUIRES_NOT_NULL( pSrcBuffer );
    NN_SDK_REQUIRES( IsInitialized( *pDstTexture ) );
    NN_SDK_REQUIRES( IsInitialized( *pSrcBuffer ) );
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( pDstTexture->ToData()->hImage != 0 );
    NN_SDK_ASSERT( pSrcBuffer->ToData()->hBuffer != 0 );

    VkBufferImageCopyParam& param = ( *WriteVkCommand< VkCopyBufferToImageParam >( this ) ).param;
    {
        param.buffer = pSrcBuffer->ToData()->hBuffer;
        param.bufferOffset = region.GetBufferOffset();
        param.bufferRowLength = region.GetBufferImageWidth();
        param.bufferImageHeight = region.GetBufferImageHeight();
        param.image = pDstTexture->ToData()->hImage;
        param.pManager = pDstTexture->ToData()->pManager;
        param.imageOffsetX = region.GetTextureCopyRegion().GetOffsetU();
        param.imageOffsetY = region.GetTextureCopyRegion().GetOffsetV();
        param.imageOffsetZ = region.GetTextureCopyRegion().GetOffsetW();
        param.imageWidth = region.GetTextureCopyRegion().GetWidth();
        param.imageHeight = region.GetTextureCopyRegion().GetHeight();
        param.imageDepth = region.GetTextureCopyRegion().GetDepth();
        param.imageMipLevel = region.GetTextureCopyRegion().GetSubresource().GetMipLevel();
        param.imageBaseLayer = region.GetTextureCopyRegion().GetSubresource().GetArrayIndex();
        param.imageAspect = Vk::GetImageAspect( pDstTexture->ToData()->format );
        param.imageLayerCount = region.GetTextureCopyRegion().GetArrayLength();
    }
}

void CommandBufferImpl< Target >::CopyImageToBuffer( BufferImpl< Target >* pDstBuffer,
    const TextureImpl< Target >* pSrcTexture, const BufferTextureCopyRegion& region ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDstBuffer );
    NN_SDK_REQUIRES_NOT_NULL( pSrcTexture );
    NN_SDK_REQUIRES( IsInitialized( *pDstBuffer ) );
    NN_SDK_REQUIRES( IsInitialized( *pSrcTexture ) );
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( pSrcTexture->ToData()->hImage != 0 );
    NN_SDK_ASSERT( pDstBuffer->ToData()->hBuffer != 0 );

    VkBufferImageCopyParam& param = ( *WriteVkCommand< VkCopyImageToBufferParam >( this ) ).param;
    {
        param.buffer = pDstBuffer->ToData()->hBuffer;
        param.bufferOffset = region.GetBufferOffset();
        param.bufferRowLength = region.GetBufferImageWidth();
        param.bufferImageHeight = region.GetBufferImageHeight();
        param.image = pSrcTexture->ToData()->hImage;
        param.pManager = pSrcTexture->ToData()->pManager;
        param.imageOffsetX = region.GetTextureCopyRegion().GetOffsetU();
        param.imageOffsetY = region.GetTextureCopyRegion().GetOffsetV();
        param.imageOffsetZ = region.GetTextureCopyRegion().GetOffsetW();
        param.imageWidth = region.GetTextureCopyRegion().GetWidth();
        param.imageHeight = region.GetTextureCopyRegion().GetHeight();
        param.imageDepth = region.GetTextureCopyRegion().GetDepth();
        param.imageMipLevel = region.GetTextureCopyRegion().GetSubresource().GetMipLevel();
        param.imageBaseLayer = region.GetTextureCopyRegion().GetSubresource().GetArrayIndex();
        param.imageAspect = Vk::GetImageAspect( pSrcTexture->ToData()->format );
        param.imageLayerCount = region.GetTextureCopyRegion().GetArrayLength();
    }
}

void CommandBufferImpl< Target >::CopyBufferToImage( TextureImpl< Target >* pDstTexture,
    const TextureCopyRegion& dstRegion, const BufferImpl< Target >* pSrcBuffer, ptrdiff_t srcOffset ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDstTexture );
    NN_SDK_REQUIRES_NOT_NULL( pSrcBuffer );
    NN_SDK_REQUIRES( IsInitialized( *pDstTexture ) );
    NN_SDK_REQUIRES( IsInitialized( *pSrcBuffer ) );
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( pDstTexture->ToData()->hImage != 0 );
    NN_SDK_ASSERT( pSrcBuffer->ToData()->hBuffer != 0 );

    VkBufferImageCopyParam& param = ( *WriteVkCommand< VkCopyBufferToImageParam >( this ) ).param;
    {
        param.buffer = pSrcBuffer->ToData()->hBuffer;
        param.bufferOffset = srcOffset;
        param.bufferRowLength = 0;
        param.bufferImageHeight = 0;
        param.image = pDstTexture->ToData()->hImage;
        param.pManager = pDstTexture->ToData()->pManager;
        param.imageOffsetX = dstRegion.GetOffsetU();
        param.imageOffsetY = dstRegion.GetOffsetV();
        param.imageOffsetZ = dstRegion.GetOffsetW();
        param.imageWidth = dstRegion.GetWidth();
        param.imageHeight = dstRegion.GetHeight();
        param.imageDepth = dstRegion.GetDepth();
        param.imageMipLevel = dstRegion.GetSubresource().GetMipLevel();
        param.imageBaseLayer = dstRegion.GetSubresource().GetArrayIndex();
        param.imageAspect = Vk::GetImageAspect( pDstTexture->ToData()->format );
        param.imageLayerCount = dstRegion.GetArrayLength();
    }
}

void CommandBufferImpl< Target >::CopyImageToBuffer( BufferImpl< Target >* pDstBuffer,
    ptrdiff_t dstOffset, const TextureImpl< Target >* pSrcTexture, const TextureCopyRegion& srcRegion ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDstBuffer );
    NN_SDK_REQUIRES_NOT_NULL( pSrcTexture );
    NN_SDK_REQUIRES( IsInitialized( *pDstBuffer ) );
    NN_SDK_REQUIRES( IsInitialized( *pSrcTexture ) );
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( pSrcTexture->ToData()->hImage != 0 );
    NN_SDK_ASSERT( pDstBuffer->ToData()->hBuffer != 0 );

    VkBufferImageCopyParam& param = ( *WriteVkCommand< VkCopyImageToBufferParam >( this ) ).param;
    {
        param.buffer = pDstBuffer->ToData()->hBuffer;
        param.bufferOffset = dstOffset;
        param.bufferRowLength = 0;
        param.bufferImageHeight = 0;
        param.image = pSrcTexture->ToData()->hImage;
        param.pManager = pSrcTexture->ToData()->pManager;
        param.imageOffsetX = srcRegion.GetOffsetU();
        param.imageOffsetY = srcRegion.GetOffsetV();
        param.imageOffsetZ = srcRegion.GetOffsetW();
        param.imageWidth = srcRegion.GetWidth();
        param.imageHeight = srcRegion.GetHeight();
        param.imageDepth = srcRegion.GetDepth();
        param.imageMipLevel = srcRegion.GetSubresource().GetMipLevel();
        param.imageBaseLayer = srcRegion.GetSubresource().GetArrayIndex();
        param.imageAspect = Vk::GetImageAspect( pSrcTexture->ToData()->format );
        param.imageLayerCount = srcRegion.GetArrayLength();
    }
}

void CommandBufferImpl< Target >::BlitImage( TextureImpl< Target >* pDstTexture,
    const TextureCopyRegion& dstRegion, const TextureImpl< Target >* pSrcTexture,
    const TextureCopyRegion& srcRegion, int copyFlags ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDstTexture );
    NN_SDK_REQUIRES_NOT_NULL( pSrcTexture );
    NN_SDK_REQUIRES( IsInitialized( *pDstTexture ) );
    NN_SDK_REQUIRES( IsInitialized( *pSrcTexture ) );
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( pDstTexture->ToData()->hImage != 0 );
    NN_SDK_ASSERT( pSrcTexture->ToData()->hImage != 0 );

    VkBlitImageParam& param = *WriteVkCommand< VkBlitImageParam >( this );
    {
        param.srcImage = pSrcTexture->ToData()->hImage;
        param.pSrcManager = pSrcTexture->ToData()->pManager;
        param.srcOffsetX = srcRegion.GetOffsetU();
        param.srcOffsetY = srcRegion.GetOffsetV();
        param.srcOffsetZ = srcRegion.GetOffsetW();
        param.srcWidth = srcRegion.GetWidth();
        param.srcHeight = srcRegion.GetHeight();
        param.srcDepth = srcRegion.GetDepth();
        param.srcMipLevel = srcRegion.GetSubresource().GetMipLevel();
        param.srcBaseLayer = srcRegion.GetSubresource().GetArrayIndex();
        param.srcAspect = Vk::GetImageAspect( pSrcTexture->ToData()->format );
        param.srcLayerCount = srcRegion.GetArrayLength();

        param.dstImage = pDstTexture->ToData()->hImage;
        param.pDstManager = pDstTexture->ToData()->pManager;
        param.dstOffsetX = dstRegion.GetOffsetU();
        param.dstOffsetY = dstRegion.GetOffsetV();
        param.dstOffsetZ = dstRegion.GetOffsetW();
        param.dstWidth = dstRegion.GetWidth();
        param.dstHeight = dstRegion.GetHeight();
        param.dstDepth = dstRegion.GetDepth();
        param.dstMipLevel = dstRegion.GetSubresource().GetMipLevel();
        param.dstBaseLayer = dstRegion.GetSubresource().GetArrayIndex();
        param.dstAspect = Vk::GetImageAspect( pDstTexture->ToData()->format );
        param.dstLayerCount = dstRegion.GetArrayLength();

        param.filter = ( copyFlags & ImageCopyFlag_LinearFilter ) ? VK_FILTER_LINEAR : VK_FILTER_NEAREST;
    }
}

void CommandBufferImpl< Target >::ClearBuffer( BufferImpl< Target >* pBuffer,
    ptrdiff_t offset, size_t size, uint32_t value ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pBuffer );
    NN_SDK_REQUIRES( IsInitialized( *pBuffer ) );
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( pBuffer->ToData()->hBuffer != 0 );

    VkClearBufferParam& param = *WriteVkCommand< VkClearBufferParam >( this );
    {
        param.buffer = pBuffer->ToData()->hBuffer;
        param.offset = static_cast< uint64_t >( offset );
        param.size = static_cast< uint64_t >( size );
        param.data = value;
    }
}

void CommandBufferImpl< Target >::ClearColor( ColorTargetViewImpl< Target >* pColorTarget,
    float red, float green, float blue, float alpha, const TextureArrayRange* pArrayRange ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pColorTarget );
    NN_SDK_REQUIRES( IsInitialized( *pColorTarget ) );
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( pColorTarget->ToData()->hImage != 0 );

    VkClearColorParam& param = *WriteVkCommand< VkClearColorParam >( this );
    {
        param.image = pColorTarget->ToData()->hImage;
        param.imageView = pColorTarget->ToData()->hImageView;
        TextureImpl< Target >* pGfxTexture = pColorTarget->ToData()->pManager;
        param.pManager = pGfxTexture->ToData()->pManager;
        param.width = std::max NN_PREVENT_MACRO_FUNC ( pColorTarget->ToData()->width >> pColorTarget->ToData()->targetMipLevel, 1u );
        param.height = std::max NN_PREVENT_MACRO_FUNC ( pColorTarget->ToData()->height >> pColorTarget->ToData()->targetMipLevel, 1u );
        param.color[ 0 ] = red;
        param.color[ 1 ] = green;
        param.color[ 2 ] = blue;
        param.color[ 3 ] = alpha;
        param.baseMipLevel = pColorTarget->ToData()->targetMipLevel;
        param.mipLevelCount = 1;
        param.baseLayer = pArrayRange != NULL ? pArrayRange->GetBaseArrayIndex()
            : pColorTarget->ToData()->targetBaseArray;
        param.layerCount = pArrayRange != NULL ? pArrayRange->GetArrayLength()
            : pColorTarget->ToData()->targetArrayCount;
    }
}

void CommandBufferImpl< Target >::ClearColorTarget( ColorTargetViewImpl< Target >* pColorTarget,
    const ClearColorValue& clearColor, const TextureArrayRange* pArrayRange ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pColorTarget );
    NN_SDK_REQUIRES( IsInitialized( *pColorTarget ) );
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( pColorTarget->ToData()->hImage != 0 );

    VkClearColorParam& param = *WriteVkCommand< VkClearColorParam >( this );
    {
        param.image = pColorTarget->ToData()->hImage;
        param.imageView = pColorTarget->ToData()->hImageView;
        TextureImpl< Target >* pGfxTexture = pColorTarget->ToData()->pManager;
        param.pManager = pGfxTexture->ToData()->pManager;
        param.width = std::max NN_PREVENT_MACRO_FUNC ( pColorTarget->ToData()->width >> pColorTarget->ToData()->targetMipLevel, 1u );
        param.height = std::max NN_PREVENT_MACRO_FUNC ( pColorTarget->ToData()->height >> pColorTarget->ToData()->targetMipLevel, 1u );
        for ( int idx = 0; idx < 4; ++idx )
        {
            param.color[ idx ] = clearColor.valueFloat[ idx ];
        }
        param.baseMipLevel = pColorTarget->ToData()->targetMipLevel;
        param.mipLevelCount = 1;
        param.baseLayer = pArrayRange != NULL ? pArrayRange->GetBaseArrayIndex()
            : pColorTarget->ToData()->targetBaseArray;
        param.layerCount = pArrayRange != NULL ? pArrayRange->GetArrayLength()
            : pColorTarget->ToData()->targetArrayCount;
    }
}

void CommandBufferImpl< Target >::ClearDepthStencil( DepthStencilViewImpl< Target >* pDepthStencil,
    float depth, int stencil, DepthStencilClearMode clearMode, const TextureArrayRange* pArrayRange ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDepthStencil );
    NN_SDK_REQUIRES( IsInitialized( *pDepthStencil ) );
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( pDepthStencil->ToData()->hImage != 0 );

    // aspect mask must be correct for validation.
    VkBool32 isDepthOnly = Vk::IsDepthOnlyFormat( static_cast< VkFormat >( pDepthStencil->ToData()->format ) );
    VkBool32 isStencilOnly = Vk::IsStencilOnlyFormat( static_cast< VkFormat >( pDepthStencil->ToData()->format ) );

    VkClearDepthStencilParam& param = *WriteVkCommand< VkClearDepthStencilParam >( this );
    {
        param.image = pDepthStencil->ToData()->hImage;
        param.imageView = pDepthStencil->ToData()->hImageView;
        TextureImpl< Target >* pGfxTexture = pDepthStencil->ToData()->pManager;
        param.pManager = pGfxTexture->ToData()->pManager;
        param.width = pDepthStencil->ToData()->width;
        param.height = pDepthStencil->ToData()->height;
        param.depth = depth;
        param.stencil = static_cast< uint32_t >( stencil );
        param.baseMipLevel = pDepthStencil->ToData()->targetMipLevel;
        param.mipLevelCount = 1;
        param.baseLayer = pArrayRange != NULL ? pArrayRange->GetBaseArrayIndex()
            : pDepthStencil->ToData()->targetBaseArray;
        param.layerCount = pArrayRange != NULL ? pArrayRange->GetArrayLength()
            : pDepthStencil->ToData()->targetArrayCount;
        param.aspectMask = ( !isStencilOnly && ( clearMode & DepthStencilClearMode_Depth ) ? VK_IMAGE_ASPECT_DEPTH_BIT : 0 )
            | ( !isDepthOnly && ( clearMode & DepthStencilClearMode_Stencil ) ? VK_IMAGE_ASPECT_STENCIL_BIT : 0 );
    }
}

void CommandBufferImpl< Target >::Resolve( TextureImpl< Target >* pDstTexture, int dstMipLevel, int dstStartArrayIndex,
    const ColorTargetViewImpl< Target >* pSrcColorTarget, const TextureArrayRange* pSrcArrayRange ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDstTexture );
    NN_SDK_REQUIRES_NOT_NULL( pSrcColorTarget );
    NN_SDK_REQUIRES( IsInitialized( *pDstTexture ) );
    NN_SDK_REQUIRES( IsInitialized( *pSrcColorTarget ) );
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( pDstTexture->ToData()->hImage != 0 );
    NN_SDK_ASSERT( pSrcColorTarget->ToData()->hImage != 0 );

    VkResolveParam& param = *WriteVkCommand< VkResolveParam >( this );
    {
        param.width = pDstTexture->ToData()->width;
        param.height = pDstTexture->ToData()->height;
        param.depth = pDstTexture->ToData()->depth;
        param.layerCount = pSrcArrayRange != NULL ? pSrcArrayRange->GetArrayLength()
            : pSrcColorTarget->ToData()->targetArrayCount;

        param.srcImage = pSrcColorTarget->ToData()->hImage;
        TextureImpl< Target >* pGfxTexture = pSrcColorTarget->ToData()->pManager;
        param.pSrcManager = pGfxTexture->ToData()->pManager;
        param.srcBaseLayer = pSrcArrayRange != NULL ? pSrcArrayRange->GetBaseArrayIndex()
            : pSrcColorTarget->ToData()->targetBaseArray;

        param.dstImage = pDstTexture->ToData()->hImage;
        param.pDstManager = pDstTexture->ToData()->pManager;
        param.dstMipLevel = dstMipLevel;
        param.dstBaseLayer = dstStartArrayIndex;
    }
}

void CommandBufferImpl< Target >::FlushMemory( int gpuAccessFlags ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );

    VkMemoryBarrierParam& param = *WriteVkCommand< VkMemoryBarrierParam >( this );
    {
        param.srcAccessMask = 0;
        param.dstAccessMask = 0;
        if ( gpuAccessFlags & GpuAccess_ColorBuffer )
        {
            param.srcAccessMask |= VK_ACCESS_COLOR_ATTACHMENT_READ_BIT
                | VK_ACCESS_COLOR_ATTACHMENT_WRITE_BIT;
        }
        if ( gpuAccessFlags & GpuAccess_Read )
        {
            param.dstAccessMask |= VK_ACCESS_MEMORY_READ_BIT;
        }
        if ( gpuAccessFlags & GpuAccess_Write )
        {
            param.srcAccessMask |= VK_ACCESS_MEMORY_WRITE_BIT;
        }
        if ( gpuAccessFlags & GpuAccess_VertexBuffer )
        {
            param.dstAccessMask |= VK_ACCESS_VERTEX_ATTRIBUTE_READ_BIT;
        }
        if ( gpuAccessFlags & GpuAccess_IndexBuffer )
        {
            param.dstAccessMask |= VK_ACCESS_INDEX_READ_BIT;
        }
        if ( gpuAccessFlags & GpuAccess_IndirectBuffer )
        {
            param.dstAccessMask |= VK_ACCESS_INDIRECT_COMMAND_READ_BIT;
        }
        if ( gpuAccessFlags & ( GpuAccess_ConstantBuffer | GpuAccess_UnorderedAccessBuffer ) )
        {
            param.dstAccessMask |= VK_ACCESS_UNIFORM_READ_BIT;
        }
        if ( gpuAccessFlags & GpuAccess_DepthStencil )
        {
            param.srcAccessMask |= VK_ACCESS_DEPTH_STENCIL_ATTACHMENT_READ_BIT
                | VK_ACCESS_DEPTH_STENCIL_ATTACHMENT_WRITE_BIT;
        }
        if ( gpuAccessFlags & GpuAccess_Image )
        {
            param.srcAccessMask |= VK_ACCESS_SHADER_WRITE_BIT;
        }
        if ( gpuAccessFlags & ( GpuAccess_ColorBuffer | GpuAccess_Image | GpuAccess_Texture
            | GpuAccess_DepthStencil | GpuAccess_UnorderedAccessBuffer ) )
        {
            param.dstAccessMask |= VK_ACCESS_SHADER_READ_BIT | VK_ACCESS_SHADER_WRITE_BIT
                | VK_ACCESS_INPUT_ATTACHMENT_READ_BIT;
        }
        // FIXME: Not sure how to handle GpuAccess_QueryBuffer
    }
}

void CommandBufferImpl< Target >::InvalidateMemory( int gpuAccessFlags ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );

    VkMemoryBarrierParam& param = *WriteVkCommand< VkMemoryBarrierParam >( this );
    {
        param.srcAccessMask = 0;
        param.dstAccessMask = 0;
        if ( gpuAccessFlags & GpuAccess_ColorBuffer )
        {
            param.srcAccessMask |= VK_ACCESS_COLOR_ATTACHMENT_READ_BIT
                | VK_ACCESS_COLOR_ATTACHMENT_WRITE_BIT;
        }
        if ( gpuAccessFlags & GpuAccess_Read )
        {
            param.dstAccessMask |= VK_ACCESS_MEMORY_READ_BIT;
        }
        if ( gpuAccessFlags & GpuAccess_Write )
        {
            param.srcAccessMask |= VK_ACCESS_MEMORY_WRITE_BIT;
        }
        if ( gpuAccessFlags & GpuAccess_VertexBuffer )
        {
            param.dstAccessMask |= VK_ACCESS_VERTEX_ATTRIBUTE_READ_BIT;
        }
        if ( gpuAccessFlags & GpuAccess_IndexBuffer )
        {
            param.dstAccessMask |= VK_ACCESS_INDEX_READ_BIT;
        }
        if ( gpuAccessFlags & GpuAccess_IndirectBuffer )
        {
            param.dstAccessMask |= VK_ACCESS_INDIRECT_COMMAND_READ_BIT;
        }
        if ( gpuAccessFlags & ( GpuAccess_ConstantBuffer | GpuAccess_UnorderedAccessBuffer ) )
        {
            param.dstAccessMask |= VK_ACCESS_UNIFORM_READ_BIT;
        }
        if ( gpuAccessFlags & GpuAccess_DepthStencil )
        {
            param.srcAccessMask |= VK_ACCESS_DEPTH_STENCIL_ATTACHMENT_READ_BIT
                | VK_ACCESS_DEPTH_STENCIL_ATTACHMENT_WRITE_BIT;
        }
        if ( gpuAccessFlags & GpuAccess_Image )
        {
            param.srcAccessMask |= VK_ACCESS_SHADER_WRITE_BIT;
        }
        if ( gpuAccessFlags & ( GpuAccess_ColorBuffer | GpuAccess_Image | GpuAccess_Texture
            | GpuAccess_DepthStencil | GpuAccess_UnorderedAccessBuffer ) )
        {
            param.dstAccessMask |= VK_ACCESS_SHADER_READ_BIT | VK_ACCESS_SHADER_WRITE_BIT
                | VK_ACCESS_INPUT_ATTACHMENT_READ_BIT;
        }
        // FIXME: Not sure how to handle GpuAccess_QueryBuffer
    }
}

void CommandBufferImpl< Target >::CallCommandBuffer( const CommandBufferImpl< Target >* pNestedCommandBuffer ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES( pNestedCommandBuffer );

    const CommandBufferImpl< Target >::DataType& nestedCommandBuffer = pNestedCommandBuffer->ToData();
    NN_SDK_REQUIRES( nestedCommandBuffer.commandBufferType == CommandBufferType_Nested );
    NN_SDK_REQUIRES( nestedCommandBuffer.state == State_Initialized );

    if( nestedCommandBuffer.pCommandList.ptr == NULL )
    {
        return;
    }

    VkCallCommandListParam& param = *WriteVkCommand< VkCallCommandListParam >( this );
    {
        param.pCommandList = nestedCommandBuffer.pCommandList;
    }

    this->drawDispatchCount += pNestedCommandBuffer->drawDispatchCount;
}

void CommandBufferImpl< Target >::CopyCommandBuffer( const CommandBufferImpl< Target >* pNestedCommandBuffer ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES( pNestedCommandBuffer );

    const CommandBufferImpl< Target >::DataType& nestedCommandBuffer = pNestedCommandBuffer->ToData();
    NN_SDK_REQUIRES( nestedCommandBuffer.commandBufferType == CommandBufferType_Nested );
    NN_SDK_REQUIRES( nestedCommandBuffer.state == State_Initialized );

    if( nestedCommandBuffer.pCommandList.ptr == NULL )
    {
        return;
    }

    const void* pCurrentCommandList = nestedCommandBuffer.pCommandList;
    for( ; ; )
    {
        if( *static_cast< const void* const* >( pCurrentCommandList ) == NULL )
        {
            pCurrentCommandList = nn::util::ConstBytePtr( pCurrentCommandList, sizeof( uint64_t ) ).Get();
            if( *static_cast< const void* const* >( pCurrentCommandList ) == NULL )
            {
                break;
            }
            pCurrentCommandList = *static_cast< const void* const* >( pCurrentCommandList );
        }
        const void* pCommandStart = *static_cast< const void* const* >( pCurrentCommandList );
        const void* pCurrentCommand = pCommandStart;
        while( *static_cast< const uintptr_t* >( pCurrentCommand ) )
        {
            const VkCommand* pCommand = static_cast< const VkCommand* >( pCurrentCommand );
            pCurrentCommand = pCommand->param + pCommand->paramSize;
        }
        size_t commandSize = nn::util::ConstBytePtr( pCommandStart ).Distance( pCurrentCommand );
        CheckCommandMemory( this, static_cast< uint32_t >( commandSize ) );
        memcpy( this->pCommandMemory, pCommandStart, commandSize );
        this->pCommandMemory = nn::util::BytePtr( this->pCommandMemory,
            static_cast< ptrdiff_t >( commandSize ) ).Get();

        pCurrentCommandList = nn::util::ConstBytePtr( pCurrentCommandList, sizeof( uint64_t ) ).Get();
    }

    this->drawDispatchCount += pNestedCommandBuffer->drawDispatchCount;
}

void CommandBufferImpl< Target >::SetBufferStateTransition(
    BufferImpl< Target >* pBuffer, int oldState, int oldStageBits,
    int newState, int newStageBits ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES_NOT_NULL( pBuffer );
    NN_SDK_REQUIRES( IsInitialized( *pBuffer ) );

    VkSetBufferStateTransitionParam& param = *WriteVkCommand< VkSetBufferStateTransitionParam >( this );
    {
        param.buffer = pBuffer->ToData()->hBuffer;
        param.srcAccessMask = Vk::GetBufferStateAccessFlags( oldState );
        param.dstAccessMask = Vk::GetBufferStateAccessFlags( newState );
        param.srcStageFlag = Vk::GetPipelineStageBits( oldStageBits, false );
        param.dstStageFlag = Vk::GetPipelineStageBits( newStageBits, false );
    }
}

void CommandBufferImpl< Target >::SetTextureStateTransition( TextureImpl< Target >* pTexture,
    const TextureSubresourceRange* pRange, int oldState, int oldStageBits,
    int newState, int newStageBits ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES_NOT_NULL( pTexture );
    NN_SDK_REQUIRES( IsInitialized( *pTexture ) );

    VkSetTextureStateTransitionParam& param = *WriteVkCommand< VkSetTextureStateTransitionParam >( this );
    {
        bool isDepth = Vk::IsDepthFormat( Vk::GetImageFormat( pTexture->ToData()->format ) ) == VK_TRUE;
        param.image = pTexture->ToData()->hImage;
        param.pManager = pTexture->ToData()->pManager;
        param.imageBaseMipLevel = pRange != NULL ? pRange->GetMipRange().GetMinMipLevel() : 0;
        param.imageMipCount = pRange != NULL ? pRange->GetMipRange().GetMipCount()
            : pTexture->ToData()->levelCount;
        param.imageBaseLayer = pRange != NULL ? pRange->GetArrayRange().GetBaseArrayIndex() : 0;
        param.imageLayerCount = pRange != NULL ? pRange->GetArrayRange().GetArrayLength()
            : pTexture->ToData()->layerCount;
        param.imageAspect = Vk::GetImageAspect( pTexture->ToData()->format );
        param.srcAccessMask = Vk::GetTextureStateAccessFlags( oldState );
        param.dstAccessMask = Vk::GetTextureStateAccessFlags( newState );

        // TODO: Should VK_PIPELINE_STAGE_TRANSFER_BIT become an official nn::gfx::PipelineStageBit?
        if ( 0 != ( oldState & ( TextureState_Clear | TextureState_CopySource | TextureState_CopyDestination ) ) )
        {
            param.srcStageFlag = VK_PIPELINE_STAGE_TRANSFER_BIT;
        }
        else
        {
            param.srcStageFlag = Vk::GetPipelineStageBits( oldStageBits, isDepth );
        }

        if ( 0 != ( newState & ( TextureState_Clear | TextureState_CopySource | TextureState_CopyDestination ) ) )
        {
            param.dstStageFlag = VK_PIPELINE_STAGE_TRANSFER_BIT;
        }
        else
        {
            param.dstStageFlag = Vk::GetPipelineStageBits( newStageBits, isDepth );
        }
        param.oldLayout = Vk::GetImageLayout( oldState );
        param.newLayout = Vk::GetImageLayout( newState );
    }
}

void CommandBufferImpl< Target >::SetDescriptorPool( const DescriptorPoolImpl< Target >* ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
}

void CommandBufferImpl< Target >::SetRootSignature( PipelineType,
    RootSignatureImpl< Target >* pRootSignatureValue ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES( !pRootSignatureValue || pRootSignatureValue->ToData()->pWorkMemory );
    this->pGfxRootSignature = pRootSignatureValue;
}

void CommandBufferImpl< Target >::SetRootBufferDescriptorTable( PipelineType pipelineType,
    int indexDescriptorTable, const DescriptorSlot& startBufferDescriptorSlot ) NN_NOEXCEPT
{
    return nn::gfx::detail::SetRootBufferDescriptorTable( this, pipelineType, indexDescriptorTable,
        startBufferDescriptorSlot, DescriptorPoolImpl< Target >::GetDescriptorSlotIncrementSize(
            this->pGfxDevice, DescriptorPoolType_BufferView ) );
}

void CommandBufferImpl< Target >::SetRootTextureAndSamplerDescriptorTable(
    PipelineType pipelineType, int indexDescriptorTable, const DescriptorSlot& startTextureDescriptorSlot,
    const DescriptorSlot& startSamplerDescriptorSlot ) NN_NOEXCEPT
{
    return nn::gfx::detail::SetRootTextureAndSamplerDescriptorTable( this, pipelineType,
        indexDescriptorTable, startTextureDescriptorSlot, startSamplerDescriptorSlot,
        DescriptorPoolImpl< Target >::GetDescriptorSlotIncrementSize( this->pGfxDevice, DescriptorPoolType_TextureView ),
        DescriptorPoolImpl< Target >::GetDescriptorSlotIncrementSize( this->pGfxDevice, DescriptorPoolType_Sampler ) );
}

void CommandBufferImpl< Target >::SetRootConstantBuffer( PipelineType pipelineType,
    int indexDynamicDescriptor, const GpuAddress& constantBufferAddress, size_t size ) NN_NOEXCEPT
{
    return nn::gfx::detail::SetRootConstantBuffer( this,
        pipelineType, indexDynamicDescriptor, constantBufferAddress, size );
}

void CommandBufferImpl< Target >::SetRootUnorderedAccessBuffer( PipelineType pipelineType,
    int indexDynamicDescriptor, const GpuAddress& unorderedAccessBufferAddress, size_t size ) NN_NOEXCEPT
{
    return nn::gfx::detail::SetRootUnorderedAccessBuffer( this,
        pipelineType, indexDynamicDescriptor, unorderedAccessBufferAddress, size );
}

void CommandBufferImpl< Target >::SetRootTextureAndSampler( PipelineType pipelineType, int indexDynamicDescriptor,
    const TextureViewImpl< Target >* pTextureView, const SamplerImpl< Target >* pSampler ) NN_NOEXCEPT
{
    return nn::gfx::detail::SetRootTextureAndSampler( this,
        pipelineType, indexDynamicDescriptor, pTextureView, pSampler );
}

void CommandBufferImpl< Target >::BeginQuery( QueryTarget target ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES_NOT_EQUAL( target, QueryTarget_Timestamp );
    NN_SDK_REQUIRES_LESS( target, QueryTarget_End );
    NN_SDK_REQUIRES( this->pGfxDevice->ToData()->supportedFeaturesBitArray.test( VkDeviceFeature_PipelineStatisticsQuery )
        || target == QueryTarget_SamplesPassed );

    VkQueryControlFlags queryControlFlags = this->pGfxDevice->ToData()->supportedFeaturesBitArray.test(
        VkDeviceFeature_OcclusionQueryPrecise ) ? VK_QUERY_CONTROL_PRECISE_BIT : 0;

    VkBeginQueryParam& param = *WriteVkCommand< VkBeginQueryParam >( this );
    {
        param.query = this->hQueryPools[ target ];
        param.flags = target == QueryTarget_SamplesPassed ? queryControlFlags : 0;
        param.queryTarget = static_cast< uint32_t >( target );
    }
}

void CommandBufferImpl< Target >::EndQuery( const GpuAddress& dstBufferAddress,
    QueryTarget target ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES_NOT_EQUAL( target, QueryTarget_Timestamp );
    NN_SDK_REQUIRES_LESS( target, QueryTarget_End );
    NN_SDK_REQUIRES( this->pGfxDevice->ToData()->supportedFeaturesBitArray.test( VkDeviceFeature_PipelineStatisticsQuery )
        || target == QueryTarget_SamplesPassed );
    NN_SDK_ASSERT( dstBufferAddress.ToData()->impl != 0 );

    VkEndQueryParam& param = *WriteVkCommand< VkEndQueryParam >( this );
    {
        param.query = this->hQueryPools[ target ];
        param.buffer = static_cast< VkHandle >( dstBufferAddress.ToData()->impl );
        param.offset = dstBufferAddress.ToData()->value;
        param.stride = sizeof( uint64_t );
        param.queryTarget = static_cast< uint32_t >( target );
    }
}

void CommandBufferImpl< Target >::Vk1BeginGraphicsPipelineStatisticsQueries() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES( this->pGfxDevice->ToData()->supportedFeaturesBitArray.test( VkDeviceFeature_PipelineStatisticsQuery ) );
    NN_SDK_ASSERT( this->fastGraphicsPipelineStatisticsQueryCount == 0 || this->fastGraphicsPipelineStatisticsQueryCount > this->graphicsPipelineStatisticsQueryIndex );

    if ( this->fastGraphicsPipelineStatisticsQueryCount == 0 )
    {
        VkBeginQueryParam& param = *WriteVkCommand< VkBeginQueryParam >( this );
        {
            param.query = this->hGraphicsPipelineStatisticsQueryPool;
            param.flags = 0;
            param.queryTarget = GfxVkGpuState::QueryState::QueryTargetFlag_GraphicsPipelineStatistics;
        }
    }
    else
    {
        VkBeginFastQueryParam& param = *WriteVkCommand< VkBeginFastQueryParam >( this );
        {
            param.query = this->hFastGraphicsPipelineStatisticsQueryPool;
            param.flags = 0;
            param.queryIndex = this->graphicsPipelineStatisticsQueryIndex;
        }
    }
}

void CommandBufferImpl< Target >::Vk1EndGraphicsPipelineStatisticsQueries( const GpuAddress& dstBufferAddress ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES( this->pGfxDevice->ToData()->supportedFeaturesBitArray.test( VkDeviceFeature_PipelineStatisticsQuery ) );
    NN_SDK_ASSERT( dstBufferAddress.ToData()->impl != 0 );
    NN_SDK_ASSERT( this->fastGraphicsPipelineStatisticsQueryCount == 0 || this->fastGraphicsPipelineStatisticsQueryCount > this->graphicsPipelineStatisticsQueryIndex );

    if ( this->fastGraphicsPipelineStatisticsQueryCount == 0 )
    {
        VkEndQueryParam& param = *WriteVkCommand< VkEndQueryParam >( this );
        {
            param.query = this->hGraphicsPipelineStatisticsQueryPool;
            param.buffer = static_cast< VkHandle >( dstBufferAddress.ToData()->impl );
            param.offset = dstBufferAddress.ToData()->value;
            param.stride = sizeof( GraphicsPipelineStatisticsQueriesBufferImplData< ApiVariationVk1 > );
            param.queryTarget = GfxVkGpuState::QueryState::QueryTargetFlag_GraphicsPipelineStatistics;
        }
    }
    else
    {
        VkEndFastQueryParam& param = *WriteVkCommand< VkEndFastQueryParam >( this );
        {
            param.query = this->hFastGraphicsPipelineStatisticsQueryPool;
            param.buffer = static_cast< VkHandle >( dstBufferAddress.ToData()->impl );
            param.queryIndex = graphicsPipelineStatisticsQueryIndex++;
            param.offset = dstBufferAddress.ToData()->value;
            param.stride = sizeof( GraphicsPipelineStatisticsQueriesBufferImplData< ApiVariationVk1 > );
        }
    }
}

void CommandBufferImpl< Target >::Vk1SetFastQueryMode( bool enabled, int queryCount ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_REQUIRES_NOT_NULL( this->pGfxDevice.ptr );
    NN_SDK_ASSERT( !enabled || queryCount > 0 );

    if ( enabled && this->hFastGraphicsPipelineStatisticsQueryPool != 0
        && this->fastGraphicsPipelineStatisticsQueryCount == queryCount )
    {
        return;
    }

    if ( this->hFastGraphicsPipelineStatisticsQueryPool != 0 )
    {
        VkQueryPool queryPool = CastToVkNonDispatchableObject< VkQueryPool >( this->hFastGraphicsPipelineStatisticsQueryPool );
        Vk::DestroyQueryPool( this->pGfxDevice, queryPool );
    }

    if ( enabled )
    {
        VkQueryPool queryPool;
        Vk::CreateGraphicsPipelineStatisticsQueriesPool( &queryPool, this->pGfxDevice, queryCount );
        this->hFastGraphicsPipelineStatisticsQueryPool = CastFromVkNonDispatchableObject< VkQueryPool >( queryPool );
        this->fastGraphicsPipelineStatisticsQueryCount = queryCount;
    }
    else
    {
        this->hFastGraphicsPipelineStatisticsQueryPool = 0;
        this->fastGraphicsPipelineStatisticsQueryCount = 0;
    }
}

void CommandBufferImpl< Target >::Vk1SetFastTimestampMode( bool enabled, int queryCount ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_REQUIRES_NOT_NULL( this->pGfxDevice.ptr );
    NN_SDK_ASSERT( !enabled || queryCount > 0 );

    if ( enabled && this->hFastTimestampQueryPool != 0
        && this->fastTimestampQueryCount == queryCount )
    {
        return;
    }

    if ( this->hFastTimestampQueryPool != 0 )
    {
        VkQueryPool queryPool = CastToVkNonDispatchableObject< VkQueryPool >( this->hFastTimestampQueryPool );
        Vk::DestroyQueryPool( this->pGfxDevice, queryPool );
    }

    if ( enabled )
    {
        VkQueryPool queryPool;
        Vk::CreateQueryPool( &queryPool, this->pGfxDevice, QueryTarget_Timestamp, queryCount );
        this->hFastTimestampQueryPool = CastFromVkNonDispatchableObject< VkQueryPool >( queryPool );
        this->fastTimestampQueryCount = queryCount;
    }
    else
    {
        this->hFastTimestampQueryPool = 0;
        this->fastTimestampQueryCount = 0;
    }
}

void CommandBufferImpl< Target >::WriteTimestamp( const GpuAddress& dstBufferAddress ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( dstBufferAddress.ToData()->impl != 0 );
    NN_SDK_ASSERT( this->fastTimestampQueryCount == 0 || this->fastTimestampQueryCount > this->timestampQueryIndex );

    if ( this->fastTimestampQueryCount == 0 )
    {
        VkWriteTimestampParam& param = *WriteVkCommand< VkWriteTimestampParam >( this );
        {
            param.query = this->hQueryPools[ QueryTarget_Timestamp ];
            param.buffer = static_cast< VkHandle >( dstBufferAddress.ToData()->impl );
            param.offset = dstBufferAddress.ToData()->value;
        }
    }
    else
    {
        VkWriteFastTimestampParam& param = *WriteVkCommand< VkWriteFastTimestampParam >( this );
        {
            param.query = this->hFastTimestampQueryPool;
            param.queryIndex = this->timestampQueryIndex++;
            param.buffer = static_cast< VkHandle >( dstBufferAddress.ToData()->impl );
            param.offset = dstBufferAddress.ToData()->value;
        }
    }

    this->flags.SetBit( Flag_TimestampUsed, true );
}

void CommandBufferImpl< Target >::SetDepthBounds( float minDepthBounds, float maxDepthBounds ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );

    VkSetDepthBoundsParam& param = *WriteVkCommand< VkSetDepthBoundsParam >( this );
    {
        param.minDepthBounds = minDepthBounds;
        param.maxDepthBounds = maxDepthBounds;
    }
}

void CommandBufferImpl< Target >::SetLineWidth( float lineWidth ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );

    VkSetLineWidthParam& param = *WriteVkCommand< VkSetLineWidthParam >( this );
    {
        param.lineWidth = lineWidth;
    }
}

void CommandBufferImpl< Target >::SetViewports( int firstViewport,
    int viewportCount, const ViewportStateInfo* pViewports ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );

    VkSetViewportsParam& param = *WriteVkCommand< VkSetViewportsParam >( this,
        sizeof( VkSetViewportsParam ) + sizeof( float ) * 6 * viewportCount );
    {
        param.first = firstViewport;
        param.count = viewportCount;
        float* pDstViewports = nn::util::BytePtr( &param, sizeof( VkSetViewportsParam ) ).Get< float >();
        for( int idxViewport = 0; idxViewport < viewportCount; ++idxViewport )
        {
            const ViewportStateInfo& srcViewport = pViewports[ idxViewport ];
            *pDstViewports++ = srcViewport.GetOriginX();
            *pDstViewports++ = srcViewport.GetOriginY();
            *pDstViewports++ = srcViewport.GetWidth();
            *pDstViewports++ = srcViewport.GetHeight();
            *pDstViewports++ = srcViewport.GetMinDepth();
            *pDstViewports++ = srcViewport.GetMaxDepth();
        }
    }
}

void CommandBufferImpl< Target >::SetScissors( int firstScissor,
    int scissorCount, const ScissorStateInfo* pScissors ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );

    VkSetScissorsParam& param = *WriteVkCommand< VkSetScissorsParam >( this,
        sizeof( VkSetScissorsParam ) + sizeof( int32_t ) * 4 * scissorCount );
    {
        param.first = firstScissor;
        param.count = scissorCount;
        int32_t* pDstScissors = nn::util::BytePtr( &param, sizeof( VkSetScissorsParam ) ).Get< int32_t >();
        for( int idxScissor = 0; idxScissor < scissorCount; ++idxScissor )
        {
            const ScissorStateInfo& srcScissor = pScissors[ idxScissor ];
            *pDstScissors++ = srcScissor.GetOriginX();
            *pDstScissors++ = srcScissor.GetOriginY();
            *pDstScissors++ = srcScissor.GetWidth();
            *pDstScissors++ = srcScissor.GetHeight();
        }
    }
}

void CommandBufferImpl< Target >::SetConstantBuffer( int slot,
    ShaderStage stage, const DescriptorSlot& constantBufferDescriptor ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    const uint64_t* pDescriptor = ToPtr< const uint64_t >( constantBufferDescriptor );
    SetBuffer< VK_DESCRIPTOR_TYPE_UNIFORM_BUFFER >(
        this, slot, stage, pDescriptor[ 0 ], pDescriptor[ 1 ], pDescriptor[ 2 ] );
}

void CommandBufferImpl< Target >::SetUnorderedAccessBuffer( int slot,
    ShaderStage stage, const DescriptorSlot& unorderedAccessBufferDescriptor ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    const uint64_t* pDescriptor = ToPtr< const uint64_t >( unorderedAccessBufferDescriptor );
    SetBuffer< VK_DESCRIPTOR_TYPE_STORAGE_BUFFER >(
        this, slot, stage, pDescriptor[ 0 ], pDescriptor[ 1 ], pDescriptor[ 2 ] );
}

void CommandBufferImpl< Target >::SetTextureAndSampler( int slot, ShaderStage stage,
    const DescriptorSlot& textureDescriptor, const DescriptorSlot& samplerDescriptor ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES( slot >= 0 );
    const Vk::TextureViewDescriptorSlot* pTextureView =
        nn::util::ConstBytePtr( reinterpret_cast< const void* >( textureDescriptor.ToData()->value ) ).Get<const Vk::TextureViewDescriptorSlot>();
    const Vk::SamplerDescriptorSlot* pSampler =
        nn::util::ConstBytePtr( reinterpret_cast< const void* >( samplerDescriptor.ToData()->value ) ).Get<const Vk::SamplerDescriptorSlot>();
    VkSetTextureAndSamplerParam& param = *WriteVkCommand< VkSetTextureAndSamplerParam >( this );
    {
        param.texture = pTextureView->imageView;
        param.pManager = pTextureView->pManager;
        param.sampler = pSampler->sampler;
        param.descriptorType = VK_DESCRIPTOR_TYPE_COMBINED_IMAGE_SAMPLER;
        param.slot = slot;
        param.shaderStage = Vk::GetShaderStage( stage );
        param.imageLayout = Vk::IsDepthFormat( static_cast< VkFormat >( pTextureView->format ) ) ? Vk::LayoutTextureDepthStencilRead : Vk::LayoutTextureRead;
        param.baseLayer = pTextureView->baseArrayLayer;
        param.layerCount = pTextureView->layerCount;
        param.baseMip = pTextureView->baseMipLevel;
        param.mipCount = pTextureView->mipCount;
    }
}

void CommandBufferImpl< Target >::SetTexture( int slot, ShaderStage stage,
    const DescriptorSlot& textureDescriptor ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES( slot >= 0 );
    const Vk::TextureViewDescriptorSlot* pTextureView =
        nn::util::ConstBytePtr( reinterpret_cast< const void* >( textureDescriptor.ToData()->value ) ).Get<const Vk::TextureViewDescriptorSlot>();
    VkSetTextureAndSamplerParam& param = *WriteVkCommand< VkSetTextureAndSamplerParam >( this );
    {
        param.texture = pTextureView->imageView;
        param.pManager = pTextureView->pManager;
        param.sampler = 0;
        param.slot = slot;
        param.shaderStage = Vk::GetShaderStage( stage );
        param.imageLayout = Vk::IsDepthFormat( static_cast< VkFormat >( pTextureView->format ) ) ? Vk::LayoutTextureDepthStencilRead : Vk::LayoutTextureRead;

        param.descriptorType = pTextureView->flags.GetBit( Vk::TextureViewDescriptorSlot::Flag_TexelBuffer ) ?
            VK_DESCRIPTOR_TYPE_UNIFORM_TEXEL_BUFFER : VK_DESCRIPTOR_TYPE_SAMPLED_IMAGE;
        param.baseLayer = pTextureView->baseArrayLayer;
        param.layerCount = pTextureView->layerCount;
        param.baseMip = pTextureView->baseMipLevel;
        param.mipCount = pTextureView->mipCount;
    }
}

void CommandBufferImpl< Target >::SetImage( int slot, ShaderStage stage,
    const DescriptorSlot& imageDescriptor ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES( slot >= 0 );
    const Vk::TextureViewDescriptorSlot* pTextureView =
        nn::util::ConstBytePtr( reinterpret_cast< const void* >( imageDescriptor.ToData()->value ) ).Get<const Vk::TextureViewDescriptorSlot>();
    VkSetTextureAndSamplerParam& param = *WriteVkCommand< VkSetTextureAndSamplerParam >( this );
    {
        param.texture = pTextureView->imageView;
        param.pManager = pTextureView->pManager;
        param.sampler = 0;
        param.slot = slot;
        param.shaderStage = Vk::GetShaderStage( stage );
        param.imageLayout = VK_IMAGE_LAYOUT_GENERAL;

        param.descriptorType = pTextureView->flags.GetBit( Vk::TextureViewDescriptorSlot::Flag_TexelBuffer ) ?
            VK_DESCRIPTOR_TYPE_STORAGE_TEXEL_BUFFER : VK_DESCRIPTOR_TYPE_STORAGE_IMAGE;
        param.baseLayer = pTextureView->baseArrayLayer;
        param.layerCount = pTextureView->layerCount;
        param.baseMip = pTextureView->baseMipLevel;
        param.mipCount = pTextureView->mipCount;
    }
}

void CommandBufferImpl< Target >::SetConstantBuffer( int slot, ShaderStage stage,
    const GpuAddress& constantBufferAddress, size_t size ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( constantBufferAddress.ToData()->impl != 0 );
    SetBuffer< VK_DESCRIPTOR_TYPE_UNIFORM_BUFFER >( this, slot, stage,
        static_cast< VkHandle >( constantBufferAddress.ToData()->impl ),
        static_cast< uint64_t >( constantBufferAddress.ToData()->value ),
        static_cast< uint64_t >( size ) );
}

void CommandBufferImpl< Target >::SetUnorderedAccessBuffer( int slot, ShaderStage stage,
    const GpuAddress& unorderedAccessBufferAddress, size_t size ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( unorderedAccessBufferAddress.ToData()->impl != 0 );
    SetBuffer< VK_DESCRIPTOR_TYPE_STORAGE_BUFFER >( this, slot, stage,
        static_cast< VkHandle >( unorderedAccessBufferAddress.ToData()->impl ),
        static_cast< uint64_t >( unorderedAccessBufferAddress.ToData()->value ),
        static_cast< uint64_t >( size ) );
}

void CommandBufferImpl< Target >::SetTextureAndSampler( int slot, ShaderStage stage,
    const TextureViewImpl< Target >* pTextureView, const SamplerImpl< Target >* pSampler ) NN_NOEXCEPT
{
    NN_UNUSED( stage );
    NN_SDK_REQUIRES_NOT_NULL( pTextureView );
    NN_SDK_REQUIRES( IsInitialized( *pTextureView ) );
    NN_SDK_ASSERT( pTextureView->ToData()->hImageView != 0 );
    NN_SDK_REQUIRES_NOT_NULL( pSampler );
    NN_SDK_REQUIRES( IsInitialized( *pSampler ) );
    NN_SDK_ASSERT( pSampler->ToData()->hSampler != 0 );
    NN_SDK_ASSERT( slot >= 0 );
    NN_SDK_ASSERT( this->state == State_Begun );

    VkSetTextureAndSamplerParam& param = *WriteVkCommand< VkSetTextureAndSamplerParam >( this );
    {
        param.texture = pTextureView->ToData()->hImageView;
        param.pManager = pTextureView->ToData()->pManager;
        param.sampler = pSampler->ToData()->hSampler;
        param.descriptorType = VK_DESCRIPTOR_TYPE_COMBINED_IMAGE_SAMPLER;
        param.slot = slot;
        param.shaderStage = Vk::GetShaderStage( stage );
        param.imageLayout = Vk::IsDepthFormat( static_cast< VkFormat >( pTextureView->ToData()->format ) ) ? Vk::LayoutTextureDepthStencilRead : Vk::LayoutTextureRead;
        param.baseLayer = pTextureView->ToData()->baseArrayLayer;
        param.layerCount = pTextureView->ToData()->layerCount;
        param.baseMip = pTextureView->ToData()->baseMipLevel;
        param.mipCount = pTextureView->ToData()->levelCount;
    }
}

void CommandBufferImpl< Target >::SetImage(
    int slot, ShaderStage stage, const TextureViewImpl< Target >* pImage ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pImage );
    NN_SDK_REQUIRES( IsInitialized( *pImage ) );
    NN_SDK_ASSERT( pImage->ToData()->hImageView != 0 );
    NN_SDK_ASSERT( slot >= 0 );
    NN_SDK_ASSERT( this->state == State_Begun );

    VkSetTextureAndSamplerParam& param = *WriteVkCommand< VkSetTextureAndSamplerParam >( this );
    {
        param.texture = pImage->ToData()->hImageView;
        param.pManager = pImage->ToData()->pManager;
        param.sampler = 0;
        param.slot = slot;
        param.shaderStage = Vk::GetShaderStage( stage );
        param.imageLayout = VK_IMAGE_LAYOUT_GENERAL;
        param.descriptorType = VK_DESCRIPTOR_TYPE_STORAGE_IMAGE;
    }
}

void CommandBufferImpl< Target >::SetShader( const ShaderImpl< Target >* pShader, int stageBits ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pShader == NULL || pShader->ToData()->state ==
        ShaderImpl< Target >::DataType::State_Initialized );
    NN_SDK_REQUIRES( this->state == State_Begun );

    VkSetSeparateShaderParam& param = *WriteVkCommand< VkSetSeparateShaderParam >( this );
    for ( int idxStage = 0; idxStage < ShaderStage_End; ++idxStage )
    {
        param.shaderModule[ idxStage ] =
            pShader != NULL ? pShader->ToData()->hShaderModule[ idxStage ] : 0;
        param.validMask[ idxStage ] =
            pShader != NULL ? pShader->ToData()->validSlotMask[ idxStage ] : 0;
    }
    param.stageBits = stageBits;
}

void CommandBufferImpl< Target >::SetRasterizerState(
    const RasterizerStateImpl< Target >* pRasterizerState ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pRasterizerState );
    NN_SDK_REQUIRES( IsInitialized( *pRasterizerState ) );
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_ASSERT( !( pRasterizerState->ToData()->flag.GetBit( RasterizerStateImplData< ApiVariationVk1 >::Flag_PolygonOffsetEnable )
        && this->isVulkanDedicatedApiModeEnabled ) );
    NN_SDK_ASSERT( !( pRasterizerState->ToData()->flag.GetBit( RasterizerStateImplData< ApiVariationVk1 >::Flag_DepthClampEnable )
        && this->isVulkanDedicatedApiModeEnabled ) );

    RasterizerStateImplData< Target >& param = *WriteVkCommand<
        RasterizerStateImplData< Target > >( this );
    param = pRasterizerState->ToData();
}

void CommandBufferImpl< Target >::SetBlendState(
    const BlendStateImpl< Target >* pBlendState ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pBlendState );
    NN_SDK_REQUIRES( IsInitialized( *pBlendState ) );
    NN_SDK_REQUIRES( this->state == State_Begun );

    int staticSize = offsetof( BlendStateImplData< Target >, pTargetArray );
    int dynamicSize = sizeof( BlendStateImplData< Target >::BlendTargetState ) * pBlendState->ToData()->blendTargetCount;

    const BlendStateImplData< Target >& src = pBlendState->ToData();
    BlendStateImplData< Target >& param = *WriteVkCommand<
        BlendStateImplData< Target > >( this, staticSize + dynamicSize );
    {
        memcpy( &param, &src, staticSize );
        memcpy( &param.pTargetArray, src.pTargetArray, dynamicSize );
    }
}

void CommandBufferImpl< Target >::SetDepthStencilState(
    const DepthStencilStateImpl< Target >* pDepthStencilState ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDepthStencilState );
    NN_SDK_REQUIRES( IsInitialized( *pDepthStencilState ) );
    NN_SDK_REQUIRES( this->state == State_Begun );

    DepthStencilStateImplData< Target >& param = *WriteVkCommand<
        DepthStencilStateImplData< Target > >( this );
    param = pDepthStencilState->ToData();
}

void CommandBufferImpl< Target >::SetVertexState(
    const VertexStateImpl< Target >* pVertexState ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pVertexState );
    NN_SDK_REQUIRES( IsInitialized( *pVertexState ) );
    NN_SDK_REQUIRES( this->state == State_Begun );

    int staticSize = offsetof( VertexStateImplData< Target >, pWorkMemory );
    int dynamicSize = sizeof( VertexStateImplData< Target >::AttributeState )
        * pVertexState->ToData()->attributeCount
        + sizeof( VertexStateImplData< Target >::BufferState ) * pVertexState->ToData()->bufferCount;

    const VertexStateImplData< Target >& src = pVertexState->ToData();
    VertexStateImplData< Target >& param = *WriteVkCommand<
        VertexStateImplData< Target > >( this, staticSize + dynamicSize );
    {
        memcpy( &param, &src, staticSize );
        memcpy( &param.pWorkMemory, src.pWorkMemory, dynamicSize );
    }
}

void CommandBufferImpl< Target >::SetTessellationState(
    const TessellationStateImpl< Target >* pTessellationState ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pTessellationState );
    NN_SDK_REQUIRES( IsInitialized( *pTessellationState ) );
    NN_SDK_REQUIRES( this->state == State_Begun );

    TessellationStateImplData< Target >& param = *WriteVkCommand<
        TessellationStateImplData< Target > >( this );
    param = pTessellationState->ToData();
}

void CommandBufferImpl< Target >::Vk1SetDynamicPolygonOffset( float slopeScaledDepthBias, float depthBias, float depthBiasClamp ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );

    VkPolygonOffsetParam& param = *WriteVkCommand< VkPolygonOffsetParam >( this );
    {
        param.slopeScaledDepthBias = slopeScaledDepthBias;
        param.depthBias = depthBias;
        param.depthBiasClamp = depthBiasClamp;
    }

    this->isVulkanDedicatedApiModeEnabled = 1;
}

void CommandBufferImpl< Target >::Vk1SetDynamicPolygonOffsetEnable( bool enable ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );

    VkPolygonOffsetEnableParam& param = *WriteVkCommand< VkPolygonOffsetEnableParam >( this );
    {
        param.enable = enable ? 1 : 0;
    }

    this->isVulkanDedicatedApiModeEnabled = 1;
}

void CommandBufferImpl< Target >::Vk1SetDynamicDepthClampEnable( bool enable ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );

    VkDepthClampParam& param = *WriteVkCommand< VkDepthClampParam >( this );
    {
        param.enable = enable ? 1 : 0;
    }

    this->isVulkanDedicatedApiModeEnabled = 1;
}

void CommandBufferImpl< Target >::Vk1ActivateUniformBlock( int slot, int stageBits, size_t size ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( slot >= 0 );
    NN_SDK_REQUIRES( slot < GfxVkGpuState::maxDynamicDescriptorSlot );
    NN_SDK_REQUIRES( stageBits != 0 );
    NN_SDK_REQUIRES( this->state == State_Begun );

    VkActivateUniformBlockParam& param = *WriteVkCommand< VkActivateUniformBlockParam >( this );
    param.slot = slot;
    param.stageBits = stageBits;
    param.size = static_cast< uint32_t >( size );
}

void CommandBufferImpl< Target >::Vk1SetUniform( const void *pValue, size_t size, size_t offset ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pValue );
    NN_SDK_REQUIRES( size > 0 );
    NN_SDK_REQUIRES( this->state == State_Begun );

    VkSetUniformParam& param = *WriteVkCommand< VkSetUniformParam >( this,
        sizeof( VkSetUniformParam ) + static_cast< uint32_t >( size ) );
    param.size = static_cast< uint32_t >( size );
    param.offset = static_cast< uint32_t >( offset );

    void* pDst = nn::util::BytePtr( &param, sizeof( VkSetUniformParam ) ).Get< void >();
    memcpy( pDst, pValue, size );
}

void CommandBufferImpl< Target >::Vk1SetUserCommandDynamic(
    Vk1UserCommandCallbackType pCallback, const void* pParam, size_t paramSize ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pCallback );
    NN_SDK_REQUIRES( this->state == State_Begun );

    VkCallbackParam& param = *WriteVkCommand< VkCallbackParam >( this,
        static_cast< uint32_t >( sizeof( VkCallbackParam ) + paramSize ) );
    {
        void* pDstParam = nn::util::BytePtr( &param, sizeof( VkCallbackParam ) ).Get();
        param.pCallback = pCallback;
        param.pParam = pDstParam;
        memcpy( pDstParam, pParam, paramSize );
    }
}

void CommandBufferImpl< Target >::Vk1DiscardColorTarget( int index ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES( index < GfxVkGpuState::maxColorAttachments );

    VkDiscardColorTargetParam& param = *WriteVkCommand< VkDiscardColorTargetParam >( this );
    param.index = static_cast< uint32_t >( index );
}

void CommandBufferImpl< Target >::Vk1DiscardDepthStencilTarget( bool depth, bool stencil ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );

    VkDiscardDepthStencilTargetParam& param = *WriteVkCommand< VkDiscardDepthStencilTargetParam >( this );
    param.aspectMask = ( depth ? VK_IMAGE_ASPECT_DEPTH_BIT : 0 ) | ( stencil ? VK_IMAGE_ASPECT_STENCIL_BIT : 0 );
}

}
}
}

NN_PRAGMA_POP_WARNINGS
