﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_BytePtr.h>

#include <nn/gfx/gfx_BufferInfo.h>
#include <nn/gfx/gfx_GpuAddress.h>

#include <nn/gfx/detail/gfx_Buffer-api.nvn.8.h>
#include <nn/gfx/detail/gfx_Device-api.nvn.8.h>
#include <nn/gfx/detail/gfx_MemoryPool-api.nvn.8.h>
#include <nn/gfx/detail/gfx_Misc.h>

#include "gfx_NvnHelper.h"

#include <cstring>
#include <algorithm>

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationNvn8 Target;

namespace {

void SetupTextureBuilder( NVNtextureBuilder* pBuilder,
    NVNdevice* pDevice, const BufferTextureViewInfo& info ) NN_NOEXCEPT
{
    const BufferImpl< Target >* pBuffer = info.GetBufferPtr();
    NN_SDK_ASSERT_NOT_NULL( pBuffer );
    NN_SDK_ASSERT( IsInitialized( *pBuffer ) );
    NVNbuffer* pNvnBuffer = pBuffer->ToData()->pNvnBuffer;
    NN_SDK_ASSERT_NOT_NULL( pNvnBuffer );
    NVNformat nvnFormat = Nvn::GetImageFormat( info.GetImageFormat() );

    NVNmemoryPool* pNvnMemoryPool = NULL;
    ptrdiff_t bufferOffset = ptrdiff_t();
    NN_GFX_CALL_NVN_FUNCTION( pNvnMemoryPool = nvnBufferGetMemoryPool( pNvnBuffer ) );
    NN_GFX_CALL_NVN_FUNCTION( bufferOffset = nvnBufferGetMemoryOffset( pNvnBuffer ) );

    NN_GFX_CALL_NVN_FUNCTION( nvnTextureBuilderSetDevice( pBuilder, pDevice ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureBuilderSetDefaults( pBuilder ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureBuilderSetWidth( pBuilder, static_cast< int >( info.GetSize() ) ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureBuilderSetHeight( pBuilder, 1 ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureBuilderSetDepth( pBuilder, 1 ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureBuilderSetFormat( pBuilder, nvnFormat ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureBuilderSetTarget( pBuilder, NVN_TEXTURE_TARGET_BUFFER ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureBuilderSetFlags( pBuilder, NVN_TEXTURE_FLAGS_IMAGE_BIT ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureBuilderSetStorage(
        pBuilder, pNvnMemoryPool, bufferOffset + info.GetOffset() ) );
}

}

size_t BufferImpl< Target >::GetBufferAlignment( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    int gpuAccessFlag = info.GetGpuAccessFlags();
    int alignment = 8;

    if( gpuAccessFlag & GpuAccess_ConstantBuffer )
    {
        int uniformBufferAlignment;
        NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger( pDevice->ToData()->pNvnDevice,
            NVN_DEVICE_INFO_UNIFORM_BUFFER_ALIGNMENT, &uniformBufferAlignment ) );
        alignment = std::max NN_PREVENT_MACRO_FUNC ( alignment, uniformBufferAlignment );
    }

    if( gpuAccessFlag & GpuAccess_IndirectBuffer )
    {
        int indirectBufferAlignment;
        NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger( pDevice->ToData()->pNvnDevice,
            NVN_DEVICE_INFO_INDIRECT_DRAW_ALIGNMENT, &indirectBufferAlignment ) );
        alignment = std::max NN_PREVENT_MACRO_FUNC ( alignment, indirectBufferAlignment );

        NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger( pDevice->ToData()->pNvnDevice,
            NVN_DEVICE_INFO_INDIRECT_DISPATCH_ALIGNMENT, &indirectBufferAlignment ) );
        alignment = std::max NN_PREVENT_MACRO_FUNC ( alignment, indirectBufferAlignment );
    }

    if( gpuAccessFlag & GpuAccess_QueryBuffer )
    {
        int counterAlignment;
        NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger( pDevice->ToData()->pNvnDevice,
            NVN_DEVICE_INFO_COUNTER_ALIGNMENT, &counterAlignment ) );
        alignment = std::max NN_PREVENT_MACRO_FUNC ( alignment, counterAlignment );
    }

    if ( gpuAccessFlag & GpuAccess_UnorderedAccessBuffer )
    {
        alignment = std::max NN_PREVENT_MACRO_FUNC ( alignment, 32 );
    }

    if ( gpuAccessFlag & ( GpuAccess_ColorBuffer | GpuAccess_DepthStencil | GpuAccess_Texture ) )
    {
        alignment = std::max NN_PREVENT_MACRO_FUNC ( alignment, 512 );
    }

    return alignment;
}

BufferImpl< Target >::BufferImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

BufferImpl< Target >::~BufferImpl() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized || this->flags.GetBit( Flag_Shared ) );
}

void BufferImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info,
    MemoryPoolImpl< Target >* pMemoryPool, ptrdiff_t memoryPoolOffset, size_t memoryPoolSize ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_SDK_REQUIRES( memoryPoolOffset >= 0 );
    NN_SDK_REQUIRES( memoryPoolSize >= info.GetSize() );
    NN_SDK_ASSERT_NOT_NULL( pMemoryPool );
    NN_SDK_ASSERT_ALIGNED( memoryPoolOffset, GetBufferAlignment( pDevice, info ) );
    NN_UNUSED( memoryPoolSize );
    NN_STATIC_ASSERT( sizeof( this->nvnBuffer ) == sizeof( NVNbuffer ) );

    NVNbufferBuilder builder;

    NN_GFX_CALL_NVN_FUNCTION( nvnBufferBuilderSetDefaults( &builder ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnBufferBuilderSetDevice( &builder, pDevice->ToData()->pNvnDevice ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnBufferBuilderSetStorage(
        &builder, pMemoryPool->ToData()->pNvnMemoryPool, memoryPoolOffset, info.GetSize() ) );

    // Create a convenience pointer to the NVNbuffer structure.
    this->pNvnBuffer = &this->nvnBuffer;

    NVNboolean isBufferOK = NN_GFX_CALL_NVN_FUNCTION(
        nvnBufferInitialize( this->pNvnBuffer, &builder ) );
    NN_SDK_ASSERT( isBufferOK );
    NN_UNUSED( isBufferOK );

    int nvnMemoryPoolFlags = static_cast< int >( NN_GFX_CALL_NVN_FUNCTION(
        nvnMemoryPoolGetFlags( pMemoryPool->ToData()->pNvnMemoryPool ) ) );

    this->flags.SetBit( Flag_CpuCached, ( nvnMemoryPoolFlags
        & NVN_MEMORY_POOL_FLAGS_CPU_CACHED_BIT ) != 0 );

    this->flags.SetBit( Flag_Shared, false );
    this->state = State_Initialized;
}

void BufferImpl< Target >::Finalize( DeviceImpl< Target >* ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsInitialized( *this ) );
    NN_SDK_ASSERT( !this->flags.GetBit( Flag_Shared ) );

    NN_GFX_CALL_NVN_FUNCTION( nvnBufferFinalize( this->pNvnBuffer ) );
    this->pNvnBuffer = NULL;

    this->state = State_NotInitialized;
}

void* BufferImpl< Target >::Map() const NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_Initialized );
    return NN_GFX_CALL_NVN_FUNCTION( nvnBufferMap( this->pNvnBuffer ) );
}

void BufferImpl< Target >::Unmap() const NN_NOEXCEPT
{
    // nvn に Unmap 関数なし
}

void BufferImpl< Target >::FlushMappedRange( ptrdiff_t offset, size_t flushSize ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_Initialized );
    if( this->flags.GetBit( Flag_CpuCached ) )
    {
        NN_GFX_CALL_NVN_FUNCTION( nvnBufferFlushMappedRange( this->pNvnBuffer, offset, flushSize ) );
    }
}

void BufferImpl< Target >::InvalidateMappedRange( ptrdiff_t offset, size_t invalidateSize ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_Initialized );
    if( this->flags.GetBit( Flag_CpuCached ) )
    {
        NN_GFX_CALL_NVN_FUNCTION( nvnBufferInvalidateMappedRange( this->pNvnBuffer, offset, invalidateSize ) );
    }
}

void BufferImpl< Target >::GetGpuAddress( GpuAddress* pOutGpuAddress ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsInitialized( *this ) );

    pOutGpuAddress->ToData()->value =
        NN_GFX_CALL_NVN_FUNCTION( nvnBufferGetAddress( this->pNvnBuffer ) );
    pOutGpuAddress->ToData()->impl = 0;
}

size_t BufferTextureViewImpl< Target >::GetOffsetAlignment(
    DeviceImpl< Target >* pDevice, const BufferTextureViewInfo& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( IsInitialized( *pDevice ) );
    NN_SDK_ASSERT_NOT_NULL( pDevice->ToData()->pNvnDevice.ptr );

    NVNtextureBuilder builder;
    SetupTextureBuilder( &builder, pDevice->ToData()->pNvnDevice, info );

    size_t alignment = size_t();
    NN_GFX_CALL_NVN_FUNCTION( alignment = nvnTextureBuilderGetStorageAlignment( &builder ) );

    return alignment;
}

BufferTextureViewImpl< Target >::BufferTextureViewImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

BufferTextureViewImpl< Target >::~BufferTextureViewImpl() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized || this->flags.GetBit( Flag_Shared ) );
}

void BufferTextureViewImpl< Target >::Initialize(
    DeviceImpl< Target >* pDevice, const BufferTextureViewInfo& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( IsInitialized( *pDevice ) );
    NN_SDK_ASSERT_NOT_NULL( pDevice->ToData()->pNvnDevice.ptr );
    NN_SDK_ASSERT( nn::util::is_aligned( info.GetOffset(), GetOffsetAlignment( pDevice, info ) ) );
    NN_STATIC_ASSERT( sizeof( this->nvnTexture ) == sizeof( NVNtexture ) );

    this->pNvnTexture = &this->nvnTexture;

    NVNtextureBuilder builder;
    SetupTextureBuilder( &builder, pDevice->ToData()->pNvnDevice, info );

    NVNboolean result = NVNboolean();
    NN_GFX_CALL_NVN_FUNCTION( result = nvnTextureInitialize( this->pNvnTexture, &builder ) );
    NN_SDK_ASSERT( result == NVN_TRUE );
    NN_UNUSED( result );

    this->flags.SetBit( Flag_Shared, false );

    this->state = State_Initialized;
}

void BufferTextureViewImpl< Target >::Finalize( DeviceImpl< Target >* ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsInitialized( *this ) );
    NN_SDK_ASSERT( !this->flags.GetBit( Flag_Shared ) );

    NN_GFX_CALL_NVN_FUNCTION( nvnTextureFinalize( this->pNvnTexture ) );
    this->pNvnTexture = NULL;

    this->state = State_NotInitialized;
}

}
}
}
