﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>

#include <nn/util/util_BytePtr.h>

#include <nn/gfx/gfx_Variation-api.gl.h>
#include <nn/gfx/gfx_BufferInfo.h>
#include <nn/gfx/gfx_GpuAddress.h>

#include <nn/gfx/detail/gfx_Misc.h>
#include <nn/gfx/detail/gfx_Buffer-api.gl.4.h>
#include <nn/gfx/detail/gfx_MemoryPool-api.gl.4.h>

#include "gfx_GlHelper.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationGl4 Target;

size_t BufferImpl< Target >::GetBufferAlignment(
    DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    GlDeviceActivator activator( pDevice );

    GLint ret = 1;
    GLint alignment;
    if( info.GetGpuAccessFlags() & GpuAccess_ConstantBuffer )
    {
        NN_GFX_CALL_GL_FUNCTION( ::glGetIntegerv(
            GL_UNIFORM_BUFFER_OFFSET_ALIGNMENT, &alignment ) );
        ret = std::max NN_PREVENT_MACRO_FUNC ( ret, alignment );
    }
    if( info.GetGpuAccessFlags() & GpuAccess_UnorderedAccessBuffer )
    {
        NN_GFX_CALL_GL_FUNCTION( ::glGetIntegerv(
            GL_SHADER_STORAGE_BUFFER_OFFSET_ALIGNMENT, &alignment ) );
        ret = std::max NN_PREVENT_MACRO_FUNC ( ret, alignment );
    }
    return ret;
}

BufferImpl< Target >::BufferImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

BufferImpl< Target >::~BufferImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized || this->flags.GetBit( Flag_Shared ) );
}

void BufferImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info,
    MemoryPoolImpl< Target >* pMemoryPool, ptrdiff_t memoryPoolOffset, size_t memoryPoolSize ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_SDK_REQUIRES( !pMemoryPool || memoryPoolOffset >= 0 );
    NN_SDK_ASSERT( !pMemoryPool || pMemoryPool->ToData()->pMemory );
    NN_SDK_REQUIRES( !pMemoryPool || memoryPoolSize >= info.GetSize() );
    NN_UNUSED( memoryPoolSize );

    GlDeviceActivator activator( pDevice );

    NN_SDK_ASSERT( !IsValid( this->hBuffer ) );
    NN_GFX_CALL_GL_FUNCTION( ::glGenBuffers( 1, &this->hBuffer ) );
    NN_GFX_GL_ASSERT();

    MemoryPoolImpl< Target >::DataType* pMemoryPoolData = pMemoryPool->ToData();
    NN_SDK_ASSERT( !pMemoryPoolData || pMemoryPoolData->memorySize >= memoryPoolOffset + info.GetSize() );
    int memoryPoolProperty = pMemoryPoolData ? pMemoryPoolData->memoryPoolProperty
        : ( MemoryPoolProperty_CpuUncached | MemoryPoolProperty_GpuCached );

    GLenum glFlags = 0;
    glFlags |= ( memoryPoolProperty & MemoryPoolProperty_CpuInvisible ) ? 0 :
        GL_MAP_READ_BIT | GL_MAP_WRITE_BIT | GL_MAP_PERSISTENT_BIT;
    glFlags |= ( memoryPoolProperty & MemoryPoolProperty_CpuUncached ) ? GL_MAP_COHERENT_BIT : 0;
    GLvoid* pData = pMemoryPoolData ? nn::util::BytePtr(
        pMemoryPoolData->pMemory, memoryPoolOffset ).Get() : NULL;
    GLenum glStorageFlags = glFlags;
    glStorageFlags |= ( memoryPoolProperty & MemoryPoolProperty_CpuInvisible ) ? 0 : GL_DYNAMIC_STORAGE_BIT;
    NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glNamedBufferStorage )(
        this->hBuffer, info.GetSize(), pData, glStorageFlags ) );
    this->pMapped = NULL;
    if( !( memoryPoolProperty & MemoryPoolProperty_CpuInvisible ) )
    {
        // GL のコンテキストセットは重いので初期化時に永続マッピングを行う
        glFlags |= ( memoryPoolProperty & MemoryPoolProperty_CpuCached ) ? GL_MAP_FLUSH_EXPLICIT_BIT : 0;
        this->pMapped = NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA(
            ::glMapNamedBufferRange )( this->hBuffer, 0, info.GetSize(), glFlags ) );
        NN_SDK_ASSERT_NOT_NULL( pMapped.ptr );
    }
    NN_GFX_GL_ASSERT();

    this->pGfxDevice = pDevice;
    this->flags.SetBit( Flag_Shared, false );
    this->flags.SetBit( Flag_FlushExplicit, ( memoryPoolProperty & MemoryPoolProperty_CpuCached ) != 0 );
    this->state = State_Initialized;
}

void BufferImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_ASSERT( !this->flags.GetBit( Flag_Shared ) );

    GlDeviceActivator activator( pDevice );

    GLint isMapped;
    NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glGetNamedBufferParameteriv )(
        this->hBuffer, GL_BUFFER_MAPPED, &isMapped ) );
    if( isMapped == GL_TRUE )
    {
        NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glUnmapNamedBuffer )( this->hBuffer ) );
    }
    if( this->hBuffer )
    {
        NN_GFX_CALL_GL_FUNCTION( ::glDeleteBuffers( 1, &this->hBuffer ) );
        NN_GFX_GL_ASSERT();
        this->hBuffer = GlInvalidHandle;
    }

    this->pGfxDevice = NULL;
    this->pMapped = NULL;

    this->state = State_NotInitialized;
}

void* BufferImpl< Target >::Map() const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( this->pMapped.ptr );

    return this->pMapped;
}

void BufferImpl< Target >::Unmap() const NN_NOEXCEPT
{
}

void BufferImpl< Target >::FlushMappedRange( ptrdiff_t offset, size_t size ) const NN_NOEXCEPT
{
    if( this->flags.GetBit( Flag_FlushExplicit ) )
    {
        GlDeviceActivator activator( this->pGfxDevice.ptr );
        NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA(
            ::glFlushMappedNamedBufferRange )( this->hBuffer, offset, size ) );
        NN_GFX_GL_ASSERT();
    }
}

void BufferImpl< Target >::InvalidateMappedRange( ptrdiff_t, size_t ) const NN_NOEXCEPT
{
}

void BufferImpl< Target >::GetGpuAddress( GpuAddress* pOutGpuAddress ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsInitialized( *this ) );

    pOutGpuAddress->ToData()->value = 0;
    pOutGpuAddress->ToData()->impl = this->hBuffer;
}

size_t BufferTextureViewImpl< Target >::GetOffsetAlignment(
    DeviceImpl< Target >* pDevice, const InfoType& ) NN_NOEXCEPT
{
    GlDeviceActivator activator( pDevice );

    GLint alignment = GLint();
    NN_GFX_CALL_GL_FUNCTION( ::glGetIntegerv(
        GL_TEXTURE_BUFFER_OFFSET_ALIGNMENT, &alignment ) );

    return alignment;
}

BufferTextureViewImpl< Target >::BufferTextureViewImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

BufferTextureViewImpl< Target >::~BufferTextureViewImpl() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized || this->flags.GetBit( Flag_Shared ) );
}

void BufferTextureViewImpl< Target >::Initialize(
    DeviceImpl< Target >* pDevice, const BufferTextureViewInfo& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );

    const BufferImpl< Target >* pBuffer = info.GetBufferPtr();
    NN_SDK_ASSERT_NOT_NULL( pBuffer );
    NN_SDK_ASSERT( IsInitialized( *pBuffer ) );
    NN_SDK_ASSERT( IsValid( pBuffer->ToData()->hBuffer ) );
    NN_SDK_ASSERT( nn::util::is_aligned( info.GetOffset(), GetOffsetAlignment( pDevice, info ) ) );

    GlDeviceActivator activator( pDevice );

    NN_SDK_ASSERT( !IsValid( this->hBufferTexture ) );
    NN_GFX_CALL_GL_FUNCTION( ::glGenTextures( 1, &this->hBufferTexture ) );
    NN_GFX_GL_ASSERT();

    GlTextureFormat format = Gl::GetTextureFormat( info.GetImageFormat() );

    NN_GFX_CALL_GL_FUNCTION( ::glBindTexture( GL_TEXTURE_BUFFER, this->hBufferTexture ) );
    NN_GFX_CALL_GL_FUNCTION( ::glTexBufferRange( GL_TEXTURE_BUFFER,
        format.imageFormat, pBuffer->ToData()->hBuffer, info.GetOffset(), info.GetSize() ) );
    NN_GFX_GL_ASSERT();

    this->flags.SetBit( Flag_Shared, false );

    this->state = State_Initialized;
}

void BufferTextureViewImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_ASSERT( !this->flags.GetBit( Flag_Shared ) );

    if( this->hBufferTexture )
    {
        GlDeviceActivator activator( pDevice );
        NN_GFX_CALL_GL_FUNCTION( ::glDeleteTextures( 1, &this->hBufferTexture ) );
        NN_GFX_GL_ASSERT();
        this->hBufferTexture = GlInvalidHandle;
    }

    this->state = State_NotInitialized;
}

}
}
}
